#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/cpumask.h>
#include <linux/delay.h>

#include <asm/delay.h>

#define ag7100_delay1s()    mdelay(1000);

#include "ar7100.h"
/*
 * Support for Ar7100 pci interrupt and core pci initialization
 */
/*
 * PCI interrupts.
 * roughly the interrupts flow is:
 *
 * - save flags
 * - CLI (disable all)
 * - IC->ack (mask out the source)
 * - EI (enable all, except the source that was masked of course)
 * - action (ISR)
 * - IC->enable (unmask the source)
 *
 * The reason we have a separate PCI IC is beacause of the following:
 * If we dont, then Throughout the "action" of a PCI slot, the
 * entire PCI "IP" on the cpu will remain disabled. Which means that we cant
 * prioritize between PCI interrupts. Normally this should be ok, if all PCI 
 * interrupts are considered equal. However, creating a PCI IC gives 
 * the flexibility to prioritize.
 */

static void
ar7100_pci_irq_enable(unsigned int irq)
{
    ar7100_reg_rmw_set(AR7100_PCI_INT_MASK, 
                       (1 << (irq - AR7100_PCI_IRQ_BASE)));
}

static void
ar7100_pci_irq_disable(unsigned int irq)
{
    ar7100_reg_rmw_clear(AR7100_PCI_INT_MASK, 
                       (1 << (irq - AR7100_PCI_IRQ_BASE)));
}

static unsigned int
ar7100_pci_irq_startup(unsigned int irq)
{
	ar7100_pci_irq_enable(irq);
	return 0;
}

static void
ar7100_pci_irq_shutdown(unsigned int irq)
{
	ar7100_pci_irq_disable(irq);
}

static void
ar7100_pci_irq_ack(unsigned int irq)
{
	ar7100_pci_irq_disable(irq);
}

static void
ar7100_pci_irq_end(unsigned int irq)
{
	if (!(irq_desc[irq].status & (IRQ_DISABLED | IRQ_INPROGRESS)))
		ar7100_pci_irq_enable(irq);
}

static void
ar7100_pci_irq_set_affinity(unsigned int irq, cpumask_t mask)
{
	/* 
     * Only 1 CPU; ignore affinity request 
     */
}

struct hw_interrupt_type ar7100_pci_irq_controller = {
	"AR7100 PCI",
	ar7100_pci_irq_startup,
	ar7100_pci_irq_shutdown,
	ar7100_pci_irq_enable,
	ar7100_pci_irq_disable,
	ar7100_pci_irq_ack,
	ar7100_pci_irq_end,
	ar7100_pci_irq_set_affinity,
};

void
ar7100_pci_irq_init(int irq_base)
{
	int i;

	for (i = irq_base; i < irq_base + AR7100_PCI_IRQ_COUNT; i++) {
		irq_desc[i].status = IRQ_DISABLED;
		irq_desc[i].action = NULL;
		irq_desc[i].depth = 1;
		irq_desc[i].handler = &ar7100_pci_irq_controller;
	}
}

/*
 * init the pci controller
 */

static struct resource ar7100_io_resource = {
	"PCI IO space",
	0x0000,
	0,
	IORESOURCE_IO
};

static struct resource ar7100_mem_resource = {
	"PCI memory space",
	AR7100_PCI_MEM_BASE,
	AR7100_PCI_MEM_BASE + AR7100_PCI_WINDOW - 1,
	IORESOURCE_MEM
};

extern struct pci_ops ar7100_pci_ops;

static struct pci_controller ar7100_pci_controller = {
	.pci_ops	    = &ar7100_pci_ops,
	.mem_resource	= &ar7100_mem_resource,
	.io_resource	= &ar7100_io_resource,
};


irqreturn_t 
ar7100_pci_core_intr(int cpl, void *dev_id, struct pt_regs *regs)
{
    printk("PCI error intr\n");
    ar7100_check_error(1);

    return IRQ_HANDLED;
}

/*
 * We want a 1:1 mapping between PCI and DDR for inbound and outbound.
 * The PCI<---AHB decoding works as follows:
 *
 * 8 registers in the DDR unit provide software configurable 32 bit offsets
 * for each of the eight 16MB PCI windows in the 128MB. The offsets will be 
 * added to any address in the 16MB segment before being sent to the PCI unit.
 *
 * Essentially  for any AHB address generated by the CPU,
 * 1. the MSB  four bits are stripped off, [31:28],
 * 2. Bit 27 is used to decide between the lower 128Mb (PCI) or the rest of 
 *    the AHB space
 * 3. Bits 26:24 are used to access one of the 8 window registers and are 
 *    masked off.
 * 4. If it is a PCI address, then the WINDOW offset in the WINDOW register 
 *    corresponding to the next 3 bits (bit 26:24) is ADDED to the address, 
 *    to generate the address to PCI unit.
 *
 *     eg. CPU address = 0x100000ff
 *         window 0 offset = 0x10000000
 *         This points to lowermost 16MB window in PCI space.
 *         So the resulting address would be 0x000000ff+0x10000000
 *         = 0x100000ff
 *
 *         eg2. CPU address = 0x120000ff
 *         WINDOW 2 offset = 0x12000000
 *         resulting address would be 0x000000ff+0x12000000
 *                         = 0x120000ff 
 *
 * There is no translation for inbound access (PCI device as a master)
 */ 
static int __init ar7100_pcibios_init(void)
{
    uint32_t cmd, val;

#ifndef CONFIG_PCI_INIT_IN_MONITOR

    /*
    ** Perform PCI reset and initialize the host interface
    */

    ar7100_reg_rmw_set(AR7100_RESET, 
                      (AR7100_RESET_PCI_BUS|AR7100_RESET_PCI_CORE));
    ag7100_delay1s();

    ar7100_reg_rmw_clear(AR7100_RESET, 
                      (AR7100_RESET_PCI_BUS|AR7100_RESET_PCI_CORE));
    ag7100_delay1s();

    ar7100_write_pci_window(0);
    ar7100_write_pci_window(1);
    ar7100_write_pci_window(2);
    ar7100_write_pci_window(3);
    ar7100_write_pci_window(4);
    ar7100_write_pci_window(5);
    ar7100_write_pci_window(6);
    ar7100_write_pci_window(7);

    ag7100_delay1s();


	cmd = PCI_COMMAND_MEMORY | PCI_COMMAND_MASTER | PCI_COMMAND_INVALIDATE |
          PCI_COMMAND_PARITY|PCI_COMMAND_SERR|PCI_COMMAND_FAST_BACK;

    ar7100_local_write_config(PCI_COMMAND, 4, cmd);

    /*
     * clear any lingering errors and register core error IRQ
     */
    ar7100_check_error(0);
#endif
	register_pci_controller(&ar7100_pci_controller);

    request_irq(AR7100_PCI_IRQ_CORE, ar7100_pci_core_intr, SA_INTERRUPT,
                "ar7100 pci core", NULL);

    return 0;
}

arch_initcall(ar7100_pcibios_init);
