/*
 * (C) Copyright 2000
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/* #define	DEBUG	*/

#include <common.h>
#include <watchdog.h>
#include <command.h>
#ifdef CONFIG_MODEM_SUPPORT
#include <malloc.h>		/* for free() prototype */
#endif

#ifdef CFG_HUSH_PARSER
#include <hush.h>
#endif

#include <net.h>
#include <post.h>
#include <ar7240_soc.h>
#include <asm-mips/addrspace.h>
#include <environment.h>

#define XMK_STR(x)	#x
#define MK_STR(x)	XMK_STR(x)

#ifdef CONFIG_SILENT_CONSOLE
DECLARE_GLOBAL_DATA_PTR;
#endif

#if defined(CONFIG_BOOT_RETRY_TIME) && defined(CONFIG_RESET_TO_RETRY)
extern int do_reset (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);		/* for do_reset() prototype */
#endif

extern int do_bootd (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[]);


#define MAX_DELAY_STOP_STR 32

static char * delete_char (char *buffer, char *p, int *colp, int *np, int plen);
static int parse_line (char *, char *[]);
#if defined(CONFIG_BOOTDELAY) && (CONFIG_BOOTDELAY >= 0)
static int abortboot(int);
#endif

#undef DEBUG_PARSER

char        console_buffer[CFG_CBSIZE];		/* console I/O buffer	*/

static char erase_seq[] = "\b \b";		/* erase sequence	*/
static char   tab_seq[] = "        ";		/* used to expand TABs	*/

#ifdef CONFIG_BOOT_RETRY_TIME
static uint64_t endtime = 0;  /* must be set, default is instant timeout */
static int      retry_time = -1; /* -1 so can call readline before main_loop */
#endif

#define	endtick(seconds) (get_ticks() + (uint64_t)(seconds) * get_tbclk())

#ifndef CONFIG_BOOT_RETRY_MIN
#define CONFIG_BOOT_RETRY_MIN CONFIG_BOOT_RETRY_TIME
#endif

#ifdef CONFIG_MODEM_SUPPORT
int do_mdm_init = 0;
extern void mdm_init(void); /* defined in board.c */
#endif

/***************************************************************************
 * Watch for 'delay' seconds for autoboot stop or autoboot delay string.
 * returns: 0 -  no key string, allow autoboot
 *          1 - got key string, abort
 */
#if defined(CONFIG_BOOTDELAY) && (CONFIG_BOOTDELAY >= 0)
# if defined(CONFIG_AUTOBOOT_KEYED)
static __inline__ int abortboot(int bootdelay)
{
	int abort = 0;
	uint64_t etime = endtick(bootdelay);
	struct
	{
		char* str;
		u_int len;
		int retry;
	}
	delaykey [] =
	{
		{ str: getenv ("bootdelaykey"),  retry: 1 },
		{ str: getenv ("bootdelaykey2"), retry: 1 },
		{ str: getenv ("bootstopkey"),   retry: 0 },
		{ str: getenv ("bootstopkey2"),  retry: 0 },
	};

	char presskey [MAX_DELAY_STOP_STR];
	u_int presskey_len = 0;
	u_int presskey_max = 0;
	u_int i;

#ifdef CONFIG_SILENT_CONSOLE
	if (gd->flags & GD_FLG_SILENT) {
		/* Restore serial console */
		console_assign (stdout, "serial");
		console_assign (stderr, "serial");
	}
#endif

#  ifdef CONFIG_AUTOBOOT_PROMPT
	printf (CONFIG_AUTOBOOT_PROMPT, bootdelay);
#  endif

#  ifdef CONFIG_AUTOBOOT_DELAY_STR
	if (delaykey[0].str == NULL)
		delaykey[0].str = CONFIG_AUTOBOOT_DELAY_STR;
#  endif
#  ifdef CONFIG_AUTOBOOT_DELAY_STR2
	if (delaykey[1].str == NULL)
		delaykey[1].str = CONFIG_AUTOBOOT_DELAY_STR2;
#  endif
#  ifdef CONFIG_AUTOBOOT_STOP_STR
	if (delaykey[2].str == NULL)
		delaykey[2].str = CONFIG_AUTOBOOT_STOP_STR;
#  endif
#  ifdef CONFIG_AUTOBOOT_STOP_STR2
	if (delaykey[3].str == NULL)
		delaykey[3].str = CONFIG_AUTOBOOT_STOP_STR2;
#  endif

	for (i = 0; i < sizeof(delaykey) / sizeof(delaykey[0]); i ++) {
		delaykey[i].len = delaykey[i].str == NULL ?
				    0 : strlen (delaykey[i].str);
		delaykey[i].len = delaykey[i].len > MAX_DELAY_STOP_STR ?
				    MAX_DELAY_STOP_STR : delaykey[i].len;

		presskey_max = presskey_max > delaykey[i].len ?
				    presskey_max : delaykey[i].len;

#  if DEBUG_BOOTKEYS
		printf("%s key:<%s>\n",
		       delaykey[i].retry ? "delay" : "stop",
		       delaykey[i].str ? delaykey[i].str : "NULL");
#  endif
	}

	/* In order to keep up with incoming data, check timeout only
	 * when catch up.
	 */
	while (!abort && get_ticks() <= etime) {
		for (i = 0; i < sizeof(delaykey) / sizeof(delaykey[0]); i ++) {
			if (delaykey[i].len > 0 &&
			    presskey_len >= delaykey[i].len &&
			    memcmp (presskey + presskey_len - delaykey[i].len,
				    delaykey[i].str,
				    delaykey[i].len) == 0) {
#  if DEBUG_BOOTKEYS
				printf("got %skey\n",
				       delaykey[i].retry ? "delay" : "stop");
#  endif

#  ifdef CONFIG_BOOT_RETRY_TIME
				/* don't retry auto boot */
				if (! delaykey[i].retry)
					retry_time = -1;
#  endif
				abort = 1;
			}
		}

		if (tstc()) {
			if (presskey_len < presskey_max) {
				presskey [presskey_len ++] = getc();
			}
			else {
				for (i = 0; i < presskey_max - 1; i ++)
					presskey [i] = presskey [i + 1];

				presskey [i] = getc();
			}
		}
	}
#  if DEBUG_BOOTKEYS
	if (!abort)
		puts ("key timeout\n");
#  endif

#ifdef CONFIG_SILENT_CONSOLE
	if (abort) {
		/* permanently enable normal console output */
		gd->flags &= ~(GD_FLG_SILENT);
	} else if (gd->flags & GD_FLG_SILENT) {
		/* Restore silent console */
		console_assign (stdout, "nulldev");
		console_assign (stderr, "nulldev");
	}
#endif

	return abort;
}

# else	/* !defined(CONFIG_AUTOBOOT_KEYED) */

#ifdef CONFIG_MENUKEY
static int menukey = 0;
#endif

static __inline__ int abortboot(int bootdelay)
{
	int abort = 0;

#ifdef CONFIG_SILENT_CONSOLE
	if (gd->flags & GD_FLG_SILENT) {
		/* Restore serial console */
		console_assign (stdout, "serial");
		console_assign (stderr, "serial");
	}
#endif

#ifdef CONFIG_MENUPROMPT
	printf(CONFIG_MENUPROMPT, bootdelay);
#else
	printf("Hit any key to stop autoboot: %2d ", bootdelay);
#endif

#if defined CONFIG_ZERO_BOOTDELAY_CHECK
	/*
	 * Check if key already pressed
	 * Don't check if bootdelay < 0
	 */
	if (bootdelay >= 0) {
		if (tstc()) {	/* we got a key press	*/
			(void) getc();  /* consume input	*/
			puts ("\b\b\b 0");
			abort = 1; 	/* don't auto boot	*/
		}
	}
#endif

	while ((bootdelay > 0) && (!abort)) {
		int i;

		--bootdelay;
		/* delay 100 * 10ms */
		for (i=0; !abort && i<100; ++i) {
			if (tstc()) {	/* we got a key press	*/
				abort  = 1;	/* don't auto boot	*/
				bootdelay = 0;	/* no more delay	*/
# ifdef CONFIG_MENUKEY
				menukey = getc();
# else
				(void) getc();  /* consume input	*/
# endif
				break;
			}
			udelay (10000);
		}

		printf ("\b\b\b%2d ", bootdelay);
	}

	putc ('\n');

#ifdef CONFIG_SILENT_CONSOLE
	if (abort) {
		/* permanently enable normal console output */
		gd->flags &= ~(GD_FLG_SILENT);
	} else if (gd->flags & GD_FLG_SILENT) {
		/* Restore silent console */
		console_assign (stdout, "nulldev");
		console_assign (stderr, "nulldev");
	}
#endif

	return abort;
}
# endif	/* CONFIG_AUTOBOOT_KEYED */
#endif	/* CONFIG_BOOTDELAY >= 0  */

/****************************************************************************/

void my_string_to_ip(char * src, uchar *dst)
{
	int i;
	char *e;
	for (i=0; i<4; ++i) {
		ulong val = src ? simple_strtoul(src, &e, 10) : 0;
		dst[i] = (val & 0xFF);
		if (src)
		{
			src = (*e) ? e+1 : e;
		}
	}
}
void my_ip_to_string (uchar * src, char *dst)
{
	sprintf (dst, "%d.%d.%d.%d", src[0], src[1], src[2], src[3]);
}
unsigned short reverse16(unsigned short num)
{
	return ((num & 0xff00) >> 8) + ((num & 0xff)<<8);
}
static mfg_t mfg_info;
static int read_mfg_info(void)
{
	uchar ip[16] = {0};
	mfg_t *pMfg = (mfg_t *)CONFIG_MFG_ADDR;
	if (pMfg->m_mfg_id != CONFIG_MFG_ID)	// no config.bin in flash, use default value
	{
		debug("MFG: use default config\n");
		strcpy(mfg_info.m_fw_name, DEFAULT_FIRMWARE_NAME);
		strcpy(mfg_info.m_mfg_name, DEFAULT_CONFIG_NAME);
		my_string_to_ip(MK_STR(CONFIG_IPADDR), mfg_info.m_local_ip);
		my_string_to_ip(MK_STR(CONFIG_NETMASK), mfg_info.m_subnet_mask);
		my_string_to_ip(MK_STR(CONFIG_SERVERIP), mfg_info.m_server_ip);
		mfg_info.m_domain = CONFIG_DOMAIN_ID;
		strcpy(mfg_info.m_model_name, DEFAULT_MODEL_NAME);
		memset(mfg_info.m_reserved, 0, sizeof(mfg_info.m_reserved));
	}
	else
	{
		mfg_info.m_mfg_id = reverse16(pMfg->m_mfg_id);
		mfg_info.m_mfg_len = reverse16(pMfg->m_mfg_len);
		memcpy(mfg_info.m_lan_addr, pMfg->m_lan_addr, 6);
		memcpy(mfg_info.m_wan_addr, pMfg->m_wan_addr, 6);
		memcpy(mfg_info.m_local_ip, pMfg->m_local_ip, 4);
		memcpy(mfg_info.m_subnet_mask, pMfg->m_subnet_mask, 4);
		memcpy(mfg_info.m_server_ip, pMfg->m_server_ip, 4);
		strncpy(mfg_info.m_fw_name, pMfg->m_fw_name, sizeof(mfg_info.m_fw_name));
		strncpy(mfg_info.m_mfg_name, pMfg->m_mfg_name, sizeof(mfg_info.m_mfg_name));
		mfg_info.m_domain = reverse16(pMfg->m_domain);
		strncpy(mfg_info.m_model_name, pMfg->m_model_name, sizeof(mfg_info.m_model_name));
		mfg_info.m_checksum = reverse16(pMfg->m_checksum);
	}
	// set as default value
	my_ip_to_string(mfg_info.m_server_ip, ip);
	setenv("serverip", ip);
	my_ip_to_string(mfg_info.m_local_ip, ip);
	setenv("ipaddr", ip);
	setenv("bootargs", CONFIG_BOOTARGS);
	setenv("mtdparts", MTDPARTS_DEFAULT);
	saveenv();
#ifdef DEBUG
	debug("id:\t%x\n", mfg_info.m_mfg_id);
	debug("len:\t%x\n", mfg_info.m_mfg_len);
	debug("fw:\t%s\n", mfg_info.m_fw_name);
	debug("mfg:\t%s\n", mfg_info.m_mfg_name);
	debug("model:\t%s\n", mfg_info.m_model_name);
	debug("local:\t%3d.%3d.%3d.%3d\n", mfg_info.m_local_ip[0], mfg_info.m_local_ip[1], mfg_info.m_local_ip[2], mfg_info.m_local_ip[3]);
	debug("mask:\t%3d.%3d.%3d.%3d\n", mfg_info.m_subnet_mask[0], mfg_info.m_subnet_mask[1], mfg_info.m_subnet_mask[2], mfg_info.m_subnet_mask[3]);
	debug("server:\t%3d.%3d.%3d.%3d\n", mfg_info.m_server_ip[0], mfg_info.m_server_ip[1], mfg_info.m_server_ip[2], mfg_info.m_server_ip[3]);
	debug("lan:\t%02x:%02x:%02x:%02x:%02x:%02x\n",
			mfg_info.m_lan_addr[0],mfg_info.m_lan_addr[1],mfg_info.m_lan_addr[2],
			mfg_info.m_lan_addr[3],mfg_info.m_lan_addr[4],mfg_info.m_lan_addr[5]);
	debug("wlan:\t %02x:%02x:%02x:%02x:%02x:%02x\n",
			mfg_info.m_lan_addr[0],mfg_info.m_wan_addr[1],mfg_info.m_wan_addr[2],
			mfg_info.m_wan_addr[3],mfg_info.m_wan_addr[4],mfg_info.m_wan_addr[5]);
	debug("domain:\t%x\n", mfg_info.m_domain);
#endif
}
static int valid_zip_image(u8* addr, u8 **Kimg, u8 **Rimg, u32 *KimgLen, u32 *RimgLen)
{
	u8 *buf = NULL;
	char line[128];
	memset(line,0,sizeof(line));
	buf = (u8*)addr;	
	if(memcmp(buf, "firm", 4))
	{
		printf("Invalid firmware image!\n");
		return 0;
	}
	debug("check the file........\n");
	buf += 16;
	if(memcmp(buf, "Kimg", 4))
	{
		printf("Error!!\n");
		return 0;
	}
	*KimgLen = ((buf[4])&0xff)|((buf[5]<<8)&0xff00)|((buf[6]<<16)&0xff0000)|((buf[7]<<24)&0xff000000);
	if (*KimgLen > CONFIG_UIMAGE_ADDR_END - CONFIG_UIMAGE_ADDR)
	{
		printf("*KimgLen=0x%x, uimage too large!\n", *KimgLen);
		return 0;
	}
	buf += 16;
	*Kimg = buf;
	buf += *KimgLen;
	debug("KimgLen=%08x,", *KimgLen);
	if (memcmp(buf, "Rimg", 4) != 0)
	{
		printf("Error!!\n");
		return 0;
	}
	*RimgLen = (buf[4]&0xff)|((buf[5]<<8)&0xff00)|((buf[6]<<16)&0xff0000)|((buf[7]<<24)&0xff000000);//*(u32 *)(buf+4);
	debug("RimgLen=%08x,",*RimgLen);
	if (*RimgLen > CONFIG_ROOTFS_ADDR_END - CONFIG_ROOTFS_ADDR)
	{
		printf("Error!!\n");
		return 0;
	}	
	buf+=16;
	*Rimg = buf;
	printf("firmware check pass\n");
	return 1;
}
static int valid_config_bin(u8 *addr)
{
	unsigned short mfg_id = *(unsigned short*)addr;
	if (mfg_id == CONFIG_MFG_ID)
	{
		printf("config check pass\n");
		return 1;
	}
	else
	{
		debug("Invalid config.bin!\n");
		return 0;
	}
}
static void write_flash(u8 *src, u8 *dst_start, u8 *dst_end, u32 len)
{
	u8 cmd_line[100] = {0};
	if (len == 0)
		len = dst_end - dst_start;
	sprintf(cmd_line, "erase %x %x;"
					  "cp.b %x %x %x", 
					  dst_start, dst_end,
					  src, dst_start, len);
	debug("cmd_line=%s\n", cmd_line);
	run_command(cmd_line, 0);
}
static void erase_flash(u8 *start, u8 *end)
{
	u8 cmd_line[64] = {0};
	sprintf(cmd_line, "erase %x %x", start, end);
	run_command(cmd_line, 0);
}
int transfer_file(char *filename, char *addr)
{
	u8 cmd_line[128] = {0};
	if (filename == NULL)
	{
		debug("%s: file name can NOT be NULL!", __FUNCTION__);
		return -1;
	}
	sprintf(cmd_line, "tftp %x %s", addr, filename);
	debug("cmd_line=%s\n", cmd_line);
	run_command(cmd_line, 0);
	if (NetState == NETLOOP_SUCCESS)
	{
		return 1;
	}else
	{
		printf("download %s fail!\n", filename);
		return 0;
	}
}

DECLARE_GLOBAL_DATA_PTR;
void load_ram_art(unsigned char *loadaddr)
{
	void (*theKernel) (int, char **, char **, int);
	char buf[256] = {0};
	char *ram_argv[3];
	int  flashSize_KB;
	u8	*entry;

	flashSize_KB = gd->bd->bi_flashsize/(1024);
	ram_argv[0] = "3";
	ram_argv[1] = "console=ttyS0,115200 rd_start=0x80260000 rd_size=0x800000 init=/sbin/init";
	sprintf(buf, "mtdparts=ar7240-nor0:256k(u-boot),64k(u-boot-env),%dk(reserved)", flashSize_KB-320);
	ram_argv[2] = buf;
	entry = ((loadaddr[3]&0xff)<<24) | ((loadaddr[2]&0xff)<<16)
			|((loadaddr[1]&0xff)<<8) | (loadaddr[0]&0xff);
	printf("\nRun RAM art, Entry is 0x%08x ......\n", entry);
	theKernel = (void (*)(int, char **, char **, int))ntohl(entry);
	theKernel(3, ram_argv, 0, flashSize_KB/1024);
}

void upgrade_firmware(int bDownloadArt)
{
	printf("RESET button is pressed.(%sdownload ART)\n", bDownloadArt ? "" : "not ");
	read_mfg_info();
	udelay(1000 * 1000 * 3);	//wait for eth up

	// art fw
	if (bDownloadArt && transfer_file(DEFAULT_ART_NAME, (char*)CONFIG_RAM_ART_ADDR))
	{
		load_ram_art( (char*)CONFIG_RAM_ART_ADDR);
		return;
	}
	// config file
	if (transfer_file(mfg_info.m_mfg_name, (char*)TMP_MEM_ADDR)
	  && valid_config_bin((u8*)TMP_MEM_ADDR))
	{
		u32 filesize = simple_strtol(getenv("filesize"), NULL, 16);
		write_flash((u8*)TMP_MEM_ADDR, (u8*)CONFIG_MFG_ADDR, (u8*)CONFIG_MFG_ADDR_END, filesize);
		printf("config download ok\n");
	}
	// normal fw
	{
		u8 *kimage, *rimage;
		u32 rimg_len, kimg_len;

		if (transfer_file(mfg_info.m_fw_name, (char*)TMP_MEM_ADDR)
		  && valid_zip_image((u8*)TMP_MEM_ADDR, &kimage, &rimage, &kimg_len, &rimg_len) )
		{
			write_flash(kimage, (u8*)CONFIG_UIMAGE_ADDR, (u8*)CONFIG_UIMAGE_ADDR_END, kimg_len-8);
			write_flash(rimage, (u8*)CONFIG_ROOTFS_ADDR, (u8*)CONFIG_ROOTFS_ADDR_END, rimg_len-8);
			erase_flash((u8*)CONFIG_CFG_ADDR, (u8*)CONFIG_CFG_ADDR_END);
			setenv("bootcmd", "bootm " MK_STR(CONFIG_UIMAGE_ADDR));
			printf("firmware download ok\n");
		}
#if 0	// download kernel file and fs file
		else
		{
			char rootfs[33], uimage[33];
			if ((rimage = getenv("rootfs")) == NULL)
				rimage = DEFAULT_RIMAGE_NAME;
			if ((kimage = getenv("uimage")) == NULL)
				kimage = DEFAULT_KIMAGE_NAME;
			strncpy(rootfs, rimage, sizeof(rootfs));
			strncpy(uimage, kimage, sizeof(uimage));
			if (transfer_file(rootfs, (char*)TMP_MEM_ADDR))
			{
				rimg_len = simple_strtol(getenv("filesize"), NULL, 16);
				write_flash((u8*)TMP_MEM_ADDR, (u8*)CONFIG_ROOTFS_ADDR, (u8*)CONFIG_ROOTFS_ADDR_END, rimg_len);
				erase_flash((u8*)CONFIG_CFG_ADDR, (u8*)CONFIG_CFG_ADDR_END);
				printf("rootfs download ok\n");
			}
			if (transfer_file(uimage, (char*)TMP_MEM_ADDR))
			{
				kimg_len = simple_strtol(getenv("filesize"), NULL, 16);
				write_flash((u8*)TMP_MEM_ADDR, (u8*)CONFIG_UIMAGE_ADDR, (u8*)CONFIG_UIMAGE_ADDR_END, kimg_len);
				setenv("bootcmd", "bootm " MK_STR(CONFIG_UIMAGE_ADDR));
				printf("kernel download ok\n");
			}
		}
#endif
	}
}

void zcom_pin_test(void)
{
	typedef struct strNum_t
	{
		char *name;
		u32  num;
	}strNum_t;
	const static strNum_t regs[] = {
			"OE",		AR7240_GPIO_OE,
			"IN",			AR7240_GPIO_IN,
			"OUT",		AR7240_GPIO_OUT,
			"SET",		AR7240_GPIO_SET,
			"CLR",		AR7240_GPIO_CLEAR,
			"INT_EN",	AR7240_GPIO_INT_ENABLE,
			"INT_TYPE", 	AR7240_GPIO_INT_TYPE,
			"INT_POL",	AR7240_GPIO_INT_POLARITY,
			"INT_PEN",	AR7240_GPIO_INT_PENDING,
			"INT_MASK", 	AR7240_GPIO_INT_MASK,
			"FUNC", 		AR7240_GPIO_FUNC
	};
	const u32 leds[] = {11, 13, 14, 15, 17};
	static u32 led_on = 0;
	u32 i;

	// show all reg's val
	for (i=0; i<sizeof(regs)/sizeof(regs[0]); i++)
	{
		printf("GPIO %-8s: %08x\n", regs[i].name, ar7240_reg_rd(regs[i].num));
	}

	// switch leds status
	led_on = !led_on;
	for (i=0; i<sizeof(leds)/sizeof(leds[0]); i++)
	{
		if (led_on)
		{
			ar7240_reg_rmw_clear(AR7240_GPIO_OUT, (1<<leds[i]));
		}
		else
		{
			ar7240_reg_rmw_set(AR7240_GPIO_OUT, (1<<leds[i]));
		}
	}

	// check btns status
	i = ar7240_reg_rd(AR7240_GPIO_IN);
	if ((~i) & (1<<GPIO_BTN_RESET))
		printf("RESET button is pressed.\n");
	if ((~i) & (1<<GPIO_BTN_MFG))
		printf("WPS button is pressed.\n");
}

void zcom_check_button(void)
{
	u32 val, bEnable = 0;
	char *pStr = NULL;

	val = ar7240_reg_rd(AR7240_GPIO_IN);
	pStr = getenv("mfg_enable");
	if (pStr != NULL && memcmp(pStr, "1", 2)==0)
					// || strcmp(pStr, "on")	 || strcmp(pStr, "y") || strcmp(pStr, "yes")))
	{
		bEnable = 1;
	}
	else
	{
		bEnable = 0;
	}
	if ((~val) & (1<<GPIO_BTN_MFG) && bEnable)
	{
		// download art fw
		if (transfer_file(DEFAULT_ART_NAME, (char*)CONFIG_RAM_ART_ADDR))
		{
			load_ram_art( (char*)CONFIG_RAM_ART_ADDR);
			return ;
		}
	}
	if ((~val) & (1<<GPIO_BTN_RESET)) 
	{
		// if mfg button Enable, we do not download art by reset button
		upgrade_firmware(!bEnable);
	}
}

#ifdef CONFIG_HW_WATCHDOG
void watchdog_reset(void)
{
	u32 val;
	val = ar7240_reg_rd(AR7240_GPIO_OE);
	val |= ((1<<GPIO_PIN_WATCHDOG));
	ar7240_reg_wr(AR7240_GPIO_OE, val);
	val = ar7240_reg_rd(AR7240_GPIO_OUT);
	val ^= (1<<GPIO_PIN_WATCHDOG);
	ar7240_reg_wr(AR7240_GPIO_OUT, val);
}
void watchdog_trigger(unsigned int hz)
{
	static u32	wdg_cnt = 0;
	wdg_cnt += hz;
	if (wdg_cnt > WATCHDOG_1S)	//about 1 sec
	{
		wdg_cnt = 0;
		watchdog_reset();
	}
}
#else
#ifdef CONFIG_WATCHDOG
void watchdog_reset(void)
{
	u32 val;

	// set watch dog action = 0b11 (full chip reset)
	val = ar7240_reg_rd(AR7240_WATCHDOG_TMR_CONTROL);
	ar7240_reg_wr(AR7240_WATCHDOG_TMR_CONTROL, val| 0x03);

	// reset watch dog timer register
	ar7240_reg_wr(AR7240_WATCHDOG_TMR, 0xffffffff);
}
void watchdog_trigger(unsigned int hz)
{
	static u32	wdg_cnt = 0;
	wdg_cnt += hz;
	if (wdg_cnt > WATCHDOG_1S)	//about 1 sec
	{
		wdg_cnt = 0;
		watchdog_reset();
	}
}
#endif
#endif

void main_loop (void)
{
#ifndef CFG_HUSH_PARSER
	static char lastcommand[CFG_CBSIZE] = { 0, };
	int len;
	int rc = 1;
	int flag;
#endif

#if defined(CONFIG_BOOTDELAY) && (CONFIG_BOOTDELAY >= 0)
	char *s;
	int bootdelay;
#endif
#ifdef CONFIG_PREBOOT
	char *p;
#endif
#ifdef CONFIG_BOOTCOUNT_LIMIT
	unsigned long bootcount = 0;
	unsigned long bootlimit = 0;
	char *bcs;
	char bcs_set[16];
#endif /* CONFIG_BOOTCOUNT_LIMIT */

#if defined(CONFIG_VFD) && defined(VFD_TEST_LOGO)
	ulong bmp = 0;		/* default bitmap */
	extern int trab_vfd (ulong bitmap);

#ifdef CONFIG_MODEM_SUPPORT
	if (do_mdm_init)
		bmp = 1;	/* alternate bitmap */
#endif
	trab_vfd (bmp);
#endif	/* CONFIG_VFD && VFD_TEST_LOGO */

#ifdef CONFIG_BOOTCOUNT_LIMIT
	bootcount = bootcount_load();
	bootcount++;
	bootcount_store (bootcount);
	sprintf (bcs_set, "%lu", bootcount);
	setenv ("bootcount", bcs_set);
	bcs = getenv ("bootlimit");
	bootlimit = bcs ? simple_strtoul (bcs, NULL, 10) : 0;
#endif /* CONFIG_BOOTCOUNT_LIMIT */

#ifdef CONFIG_MODEM_SUPPORT
	debug ("DEBUG: main_loop:   do_mdm_init=%d\n", do_mdm_init);
	if (do_mdm_init) {
		char *str = strdup(getenv("mdm_cmd"));
		setenv ("preboot", str);  /* set or delete definition */
		if (str != NULL)
			free (str);
		mdm_init(); /* wait for modem connection */
	}
#endif  /* CONFIG_MODEM_SUPPORT */

#ifdef CONFIG_VERSION_VARIABLE
	{
		extern char version_string[];

		setenv ("ver", version_string);  /* set version variable */
	}
#endif /* CONFIG_VERSION_VARIABLE */

#ifdef CFG_HUSH_PARSER
	u_boot_hush_start ();
#endif

#ifdef CONFIG_AUTO_COMPLETE
	install_auto_complete();
#endif

#ifdef CONFIG_PREBOOT
	if ((p = getenv ("preboot")) != NULL) {
# ifdef CONFIG_AUTOBOOT_KEYED
		int prev = disable_ctrlc(1);	/* disable Control C checking */
# endif

# ifndef CFG_HUSH_PARSER
		run_command (p, 0);
# else
		parse_string_outer(p, FLAG_PARSE_SEMICOLON |
				    FLAG_EXIT_FROM_LOOP);
# endif

# ifdef CONFIG_AUTOBOOT_KEYED
		disable_ctrlc(prev);	/* restore Control C checking */
# endif
	}
#endif /* CONFIG_PREBOOT */

	zcom_check_button();

#if defined(CONFIG_BOOTDELAY) && (CONFIG_BOOTDELAY >= 0)
	s = getenv ("bootdelay");
	bootdelay = s ? (int)simple_strtol(s, NULL, 10) : CONFIG_BOOTDELAY;

//	debug ("### main_loop entered: bootdelay=%d\n\n", bootdelay);

# ifdef CONFIG_BOOT_RETRY_TIME
	init_cmd_timeout ();
# endif	/* CONFIG_BOOT_RETRY_TIME */

#ifdef CONFIG_BOOTCOUNT_LIMIT
	if (bootlimit && (bootcount > bootlimit)) {
		printf ("Warning: Bootlimit (%u) exceeded. Using altbootcmd.\n",
		        (unsigned)bootlimit);
		s = getenv ("altbootcmd");
	}
	else
#endif /* CONFIG_BOOTCOUNT_LIMIT */
		s = getenv ("bootcmd");
       if (!s) {
#ifdef CONFIG_ROOTFS_FLASH
           /* XXX if rootfs is in flash, expect uImage to be in flash */
#ifdef CONFIG_AR7100
           setenv ("bootcmd", "bootm 0xbf200000");
#else
           setenv ("bootcmd", "bootm 0xbf450000");
#endif /* CONFIG_AR7100 */
#else
           setenv ("bootcmd", "tftpboot 0x8022c090 uImage; bootm 0x8022c090");
#endif
       }
		s = getenv ("bootcmd");

//	debug ("### main_loop: bootcmd=\"%s\"\n", s ? s : "<UNDEFINED>");

	if (bootdelay >= 0 && s && !abortboot (bootdelay)) {
# ifdef CONFIG_AUTOBOOT_KEYED
		int prev = disable_ctrlc(1);	/* disable Control C checking */
# endif

# ifndef CFG_HUSH_PARSER
		run_command (s, 0);
# else
		parse_string_outer(s, FLAG_PARSE_SEMICOLON |
				    FLAG_EXIT_FROM_LOOP);
# endif

# ifdef CONFIG_AUTOBOOT_KEYED
		disable_ctrlc(prev);	/* restore Control C checking */
# endif
	}

# ifdef CONFIG_MENUKEY
	if (menukey == CONFIG_MENUKEY) {
	    s = getenv("menucmd");
	    if (s) {
# ifndef CFG_HUSH_PARSER
		run_command (s, 0);
# else
		parse_string_outer(s, FLAG_PARSE_SEMICOLON |
				    FLAG_EXIT_FROM_LOOP);
# endif
	    }
	}
#endif /* CONFIG_MENUKEY */
#endif	/* CONFIG_BOOTDELAY */

#ifdef CONFIG_AMIGAONEG3SE
	{
	    extern void video_banner(void);
	    video_banner();
	}
#endif

	/*
	 * Main Loop for Monitor Command Processing
	 */
#ifdef CFG_HUSH_PARSER
	parse_file_outer();
	/* This point is never reached */
	for (;;);
#else
	for (;;) {
#ifdef CONFIG_BOOT_RETRY_TIME
		if (rc >= 0) {
			/* Saw enough of a valid command to
			 * restart the timeout.
			 */
			reset_cmd_timeout();
		}
#endif
		len = readline (CFG_PROMPT);

		flag = 0;	/* assume no special flags for now */
		if (len > 0)
			strcpy (lastcommand, console_buffer);
		else if (len == 0)
			flag |= CMD_FLAG_REPEAT;
#ifdef CONFIG_BOOT_RETRY_TIME
		else if (len == -2) {
			/* -2 means timed out, retry autoboot
			 */
			puts ("\nTimed out waiting for command\n");
# ifdef CONFIG_RESET_TO_RETRY
			/* Reinit board to run initialization code again */
			do_reset (NULL, 0, 0, NULL);
# else
			return;		/* retry autoboot */
# endif
		}
#endif

		if (len == -1)
			puts ("<INTERRUPT>\n");
		else
			rc = run_command (lastcommand, flag);

		if (rc <= 0) {
			/* invalid command or not repeatable, forget it */
			lastcommand[0] = 0;
		}
	}
#endif /*CFG_HUSH_PARSER*/
}

#ifdef CONFIG_BOOT_RETRY_TIME
/***************************************************************************
 * initialise command line timeout
 */
void init_cmd_timeout(void)
{
	char *s = getenv ("bootretry");

	if (s != NULL)
		retry_time = (int)simple_strtol(s, NULL, 10);
	else
		retry_time =  CONFIG_BOOT_RETRY_TIME;

	if (retry_time >= 0 && retry_time < CONFIG_BOOT_RETRY_MIN)
		retry_time = CONFIG_BOOT_RETRY_MIN;
}

/***************************************************************************
 * reset command line timeout to retry_time seconds
 */
void reset_cmd_timeout(void)
{
	endtime = endtick(retry_time);
}
#endif

/****************************************************************************/

/*
 * Prompt for input and read a line.
 * If  CONFIG_BOOT_RETRY_TIME is defined and retry_time >= 0,
 * time out when time goes past endtime (timebase time in ticks).
 * Return:	number of read characters
 *		-1 if break
 *		-2 if timed out
 */
int readline (const char *const prompt)
{
	char   *p = console_buffer;
	int	n = 0;				/* buffer index		*/
	int	plen = 0;			/* prompt length	*/
	int	col;				/* output column cnt	*/
	char	c;

	/* print prompt */
	if (prompt) {
		plen = strlen (prompt);
		puts (prompt);
	}
	col = plen;

	for (;;) {
#ifdef CONFIG_BOOT_RETRY_TIME
		while (!tstc()) {	/* while no incoming data */
			if (retry_time >= 0 && get_ticks() > endtime)
				return (-2);	/* timed out */
		}
#endif
		WATCHDOG_RESET();		/* Trigger watchdog, if needed */

#ifdef CONFIG_SHOW_ACTIVITY
		while (!tstc()) {
			extern void show_activity(int arg);
			show_activity(0);
		}
#endif
		c = getc();

		/*
		 * Special character handling
		 */
		switch (c) {
		case '\r':				/* Enter		*/
		case '\n':
			*p = '\0';
			puts ("\r\n");
			return (p - console_buffer);

		case '\0':				/* nul			*/
			continue;

		case 0x03:				/* ^C - break		*/
			console_buffer[0] = '\0';	/* discard input */
			return (-1);

		case 0x15:				/* ^U - erase line	*/
			while (col > plen) {
				puts (erase_seq);
				--col;
			}
			p = console_buffer;
			n = 0;
			continue;

		case 0x17:				/* ^W - erase word 	*/
			p=delete_char(console_buffer, p, &col, &n, plen);
			while ((n > 0) && (*p != ' ')) {
				p=delete_char(console_buffer, p, &col, &n, plen);
			}
			continue;

		case 0x08:				/* ^H  - backspace	*/
		case 0x7F:				/* DEL - backspace	*/
			p=delete_char(console_buffer, p, &col, &n, plen);
			continue;

		default:
			/*
			 * Must be a normal character then
			 */
			if (n < CFG_CBSIZE-2) {
				if (c == '\t') {	/* expand TABs		*/
#ifdef CONFIG_AUTO_COMPLETE
					/* if auto completion triggered just continue */
					*p = '\0';
					if (cmd_auto_complete(prompt, console_buffer, &n, &col)) {
						p = console_buffer + n;	/* reset */
						continue;
					}
#endif
					puts (tab_seq+(col&07));
					col += 8 - (col&07);
				} else {
					++col;		/* echo input		*/
					putc (c);
				}
				*p++ = c;
				++n;
			} else {			/* Buffer full		*/
				putc ('\a');
			}
		}
	}
}

/****************************************************************************/

static char * delete_char (char *buffer, char *p, int *colp, int *np, int plen)
{
	char *s;

	if (*np == 0) {
		return (p);
	}

	if (*(--p) == '\t') {			/* will retype the whole line	*/
		while (*colp > plen) {
			puts (erase_seq);
			(*colp)--;
		}
		for (s=buffer; s<p; ++s) {
			if (*s == '\t') {
				puts (tab_seq+((*colp) & 07));
				*colp += 8 - ((*colp) & 07);
			} else {
				++(*colp);
				putc (*s);
			}
		}
	} else {
		puts (erase_seq);
		(*colp)--;
	}
	(*np)--;
	return (p);
}

/****************************************************************************/

int parse_line (char *line, char *argv[])
{
	int nargs = 0;

#ifdef DEBUG_PARSER
	printf ("parse_line: \"%s\"\n", line);
#endif
	while (nargs < CFG_MAXARGS) {

		/* skip any white space */
		while ((*line == ' ') || (*line == '\t')) {
			++line;
		}

		if (*line == '\0') {	/* end of line, no more args	*/
			argv[nargs] = NULL;
#ifdef DEBUG_PARSER
		printf ("parse_line: nargs=%d\n", nargs);
#endif
			return (nargs);
		}

		argv[nargs++] = line;	/* begin of argument string	*/

		/* find end of string */
		while (*line && (*line != ' ') && (*line != '\t')) {
			++line;
		}

		if (*line == '\0') {	/* end of line, no more args	*/
			argv[nargs] = NULL;
#ifdef DEBUG_PARSER
		printf ("parse_line: nargs=%d\n", nargs);
#endif
			return (nargs);
		}

		*line++ = '\0';		/* terminate current arg	 */
	}

	printf ("** Too many args (max. %d) **\n", CFG_MAXARGS);

#ifdef DEBUG_PARSER
	printf ("parse_line: nargs=%d\n", nargs);
#endif
	return (nargs);
}

/****************************************************************************/

static void process_macros (const char *input, char *output)
{
	char c, prev;
	const char *varname_start = NULL;
	int inputcnt  = strlen (input);
	int outputcnt = CFG_CBSIZE;
	int state = 0;	/* 0 = waiting for '$'	*/
			/* 1 = waiting for '(' or '{' */
			/* 2 = waiting for ')' or '}' */
			/* 3 = waiting for '''  */
#ifdef DEBUG_PARSER
	char *output_start = output;

	printf ("[PROCESS_MACROS] INPUT len %d: \"%s\"\n", strlen(input), input);
#endif

	prev = '\0';			/* previous character	*/

	while (inputcnt && outputcnt) {
	    c = *input++;
	    inputcnt--;

	    if (state!=3) {
	    /* remove one level of escape characters */
	    if ((c == '\\') && (prev != '\\')) {
		if (inputcnt-- == 0)
			break;
		prev = c;
		c = *input++;
	    }
	    }

	    switch (state) {
	    case 0:			/* Waiting for (unescaped) $	*/
		if ((c == '\'') && (prev != '\\')) {
			state = 3;
			break;
		}
		if ((c == '$') && (prev != '\\')) {
			state++;
		} else {
			*(output++) = c;
			outputcnt--;
		}
		break;
	    case 1:			/* Waiting for (	*/
		if (c == '(' || c == '{') {
			state++;
			varname_start = input;
		} else {
			state = 0;
			*(output++) = '$';
			outputcnt--;

			if (outputcnt) {
				*(output++) = c;
				outputcnt--;
			}
		}
		break;
	    case 2:			/* Waiting for )	*/
		if (c == ')' || c == '}') {
			int i;
			char envname[CFG_CBSIZE], *envval;
			int envcnt = input-varname_start-1; /* Varname # of chars */

			/* Get the varname */
			for (i = 0; i < envcnt; i++) {
				envname[i] = varname_start[i];
			}
			envname[i] = 0;

			/* Get its value */
			envval = getenv (envname);

			/* Copy into the line if it exists */
			if (envval != NULL)
				while ((*envval) && outputcnt) {
					*(output++) = *(envval++);
					outputcnt--;
				}
			/* Look for another '$' */
			state = 0;
		}
		break;
	    case 3:			/* Waiting for '	*/
		if ((c == '\'') && (prev != '\\')) {
			state = 0;
		} else {
			*(output++) = c;
			outputcnt--;
		}
		break;
	    }
	    prev = c;
	}

	if (outputcnt)
		*output = 0;

#ifdef DEBUG_PARSER
	printf ("[PROCESS_MACROS] OUTPUT len %d: \"%s\"\n",
		strlen(output_start), output_start);
#endif
}

/****************************************************************************
 * returns:
 *	1  - command executed, repeatable
 *	0  - command executed but not repeatable, interrupted commands are
 *	     always considered not repeatable
 *	-1 - not executed (unrecognized, bootd recursion or too many args)
 *           (If cmd is NULL or "" or longer than CFG_CBSIZE-1 it is
 *           considered unrecognized)
 *
 * WARNING:
 *
 * We must create a temporary copy of the command since the command we get
 * may be the result from getenv(), which returns a pointer directly to
 * the environment data, which may change magicly when the command we run
 * creates or modifies environment variables (like "bootp" does).
 */

int run_command (const char *cmd, int flag)
{
	cmd_tbl_t *cmdtp;
	char cmdbuf[CFG_CBSIZE];	/* working copy of cmd		*/
	char *token;			/* start of token in cmdbuf	*/
	char *sep;			/* end of token (separator) in cmdbuf */
	char finaltoken[CFG_CBSIZE];
	char *str = cmdbuf;
	char *argv[CFG_MAXARGS + 1];	/* NULL terminated	*/
	int argc, inquotes;
	int repeatable = 1;
	int rc = 0;

#ifdef DEBUG_PARSER
	printf ("[RUN_COMMAND] cmd[%p]=\"", cmd);
	puts (cmd ? cmd : "NULL");	/* use puts - string may be loooong */
	puts ("\"\n");
#endif

	clear_ctrlc();		/* forget any previous Control C */

	if (!cmd || !*cmd) {
		return -1;	/* empty command */
	}

	if (strlen(cmd) >= CFG_CBSIZE) {
		puts ("## Command too long!\n");
		return -1;
	}

	strcpy (cmdbuf, cmd);

	/* Process separators and check for invalid
	 * repeatable commands
	 */

#ifdef DEBUG_PARSER
	printf ("[PROCESS_SEPARATORS] %s\n", cmd);
#endif
	while (*str) {

		/*
		 * Find separator, or string end
		 * Allow simple escape of ';' by writing "\;"
		 */
		for (inquotes = 0, sep = str; *sep; sep++) {
			if ((*sep=='\'') &&
			    (*(sep-1) != '\\'))
				inquotes=!inquotes;

			if (!inquotes &&
			    (*sep == ';') &&	/* separator		*/
			    ( sep != str) &&	/* past string start	*/
			    (*(sep-1) != '\\'))	/* and NOT escaped	*/
				break;
		}

		/*
		 * Limit the token to data between separators
		 */
		token = str;
		if (*sep) {
			str = sep + 1;	/* start of command for next pass */
			*sep = '\0';
		}
		else
			str = sep;	/* no more commands for next pass */
#ifdef DEBUG_PARSER
		printf ("token: \"%s\"\n", token);
#endif

		/* find macros in this token and replace them */
		process_macros (token, finaltoken);

		/* Extract arguments */
		if ((argc = parse_line (finaltoken, argv)) == 0) {
			rc = -1;	/* no command at all */
			continue;
		}

		/* Look up command in command table */
		if ((cmdtp = find_cmd(argv[0])) == NULL) {
			printf ("Unknown command '%s' - try 'help'\n", argv[0]);
			rc = -1;	/* give up after bad command */
			continue;
		}

		/* found - check max args */
		if (argc > cmdtp->maxargs) {
			printf ("Usage:\n%s\n", cmdtp->usage);
			rc = -1;
			continue;
		}

#if (CONFIG_COMMANDS & CFG_CMD_BOOTD)
		/* avoid "bootd" recursion */
		if (cmdtp->cmd == do_bootd) {
#ifdef DEBUG_PARSER
			printf ("[%s]\n", finaltoken);
#endif
			if (flag & CMD_FLAG_BOOTD) {
				puts ("'bootd' recursion detected\n");
				rc = -1;
				continue;
			} else {
				flag |= CMD_FLAG_BOOTD;
			}
		}
#endif	/* CFG_CMD_BOOTD */

		/* OK - call function to do the command */
		if ((cmdtp->cmd) (cmdtp, flag, argc, argv) != 0) {
			rc = -1;
		}

		repeatable &= cmdtp->repeatable;

		/* Did the user stop this? */
		if (had_ctrlc ())
			return 0;	/* if stopped then not repeatable */
	}

	return rc ? rc : repeatable;
}

/****************************************************************************/

#if (CONFIG_COMMANDS & CFG_CMD_RUN)
int do_run (cmd_tbl_t * cmdtp, int flag, int argc, char *argv[])
{
	int i;

	if (argc < 2) {
		printf ("Usage:\n%s\n", cmdtp->usage);
		return 1;
	}

	for (i=1; i<argc; ++i) {
		char *arg;

		if ((arg = getenv (argv[i])) == NULL) {
			printf ("## Error: \"%s\" not defined\n", argv[i]);
			return 1;
		}
#ifndef CFG_HUSH_PARSER
		if (run_command (arg, flag) == -1)
			return 1;
#else
		if (parse_string_outer(arg,
		    FLAG_PARSE_SEMICOLON | FLAG_EXIT_FROM_LOOP) != 0)
			return 1;
#endif
	}
	return 0;
}
#endif	/* CFG_CMD_RUN */
