// os_network.c
//

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>

#include "os_network.h"

#ifdef	_DEBUG
static void PrintIP(uint32_t ulIP)
{
	uint8_t	*puchIP = (uint8_t*)&ulIP;
	printf("%d.%d.%d.%d", puchIP[0], puchIP[1], puchIP[2], puchIP[3]);
}
#endif//_DEBUG
///////////////////////////////////////////////////////////////////////////////
// basic network routines
///////////////////////////////////////////////////////////////////////////////

#ifdef	WIN32
static void Win32SocketClean(void)
{
	WSACleanup();
}

static void Win32SocketStartup(void)
{
	static int	iIsSocketInited = 0;
	if (!iIsSocketInited)
	{
		WSADATA wsaData;
		if (0!=WSAStartup(MAKEWORD(2,1), &wsaData))
		{
			printf("not find a usable WinSock DLL!\n");
			exit(-1);
		}
		atexit(Win32SocketClean);
		iIsSocketInited = 1;
	}
}
#endif//WIN32

OsSocket_t	OsSocketOpen(int iAddrFamily, int iType, int iProtocol)
{
#ifdef	WIN32
	Win32SocketStartup();
#endif//WIN32
	return socket(iAddrFamily, iType, iProtocol);
}

bool_t	OsSocketClose(OsSocket_t Socket)
{
#ifdef	WIN32
	return 0==closesocket(Socket);
#else
	return 0==close(Socket);
#endif//!WIN32
}

bool_t OsConnect(OsSocket_t Socket, const struct sockaddr *pRemoteAddr, int	iAddrLen, int* iErrNo)
{
	struct timespec WaitTime =
	{
		.tv_sec	= 1,
		.tv_nsec= 0	//1ms
	};
	uint32_t	ulRetry;
	int			iErrCode = errno;

	for (ulRetry=0; ulRetry<OS_SOCKET_RETRY_TIMES; ulRetry++)
	{
		int	iReturn = connect(Socket, pRemoteAddr, iAddrLen);

		iErrCode = errno;
		if (0==iReturn)
		{
			return TRUE;
		}
		else if (EAGAIN==iErrCode)
		{
			nanosleep(&WaitTime, NULL);
			WaitTime.tv_sec = WaitTime.tv_sec<<1;
		}
		else
		{
			break;
		}
	}
	if (NULL!=iErrNo)
	{
		*iErrNo = iErrCode;
	}
	return FALSE;
}

bool_t OsConnectInternet(OsSocket_t Socket, uint32_t ulRemoteIP, uint16_t usPort)
{
	struct sockaddr_in	inRemoteAddr = {0};

	inRemoteAddr.sin_family = AF_INET;
	inRemoteAddr.sin_port = htons(usPort);
	inRemoteAddr.sin_addr.s_addr = ulRemoteIP;
	return connect(Socket, (struct sockaddr*)&inRemoteAddr, sizeof(inRemoteAddr))>=0;
}

bool_t OsBind(OsSocket_t Socket, const struct sockaddr *pAddr, int iAddrLen)
{
	return 0==bind(Socket, pAddr, iAddrLen);
}

bool_t	OsBindInternet(OsSocket_t Socket, uint32_t ulIP, uint16_t usPort)
{
	struct sockaddr_in	inAddr = {0};

	inAddr.sin_family = AF_INET;
	inAddr.sin_port = htons(usPort);
	inAddr.sin_addr.s_addr = ulIP;
	return OsBind(Socket, (struct sockaddr*)&inAddr, sizeof(struct sockaddr_in));
}

bool_t OsGetIfInfoByName(const char* cszIfName, int32_t* piIfIndex, MacAddr_t MacAddr)
{
	struct ifreq	ifr;
	int32_t			fd = socket(PF_INET, SOCK_STREAM, 0);

	if (0 > fd)
	{
		return FALSE;
	}
	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_name, cszIfName, sizeof(ifr.ifr_name));
	if (NULL!=piIfIndex)
	{
		if (0 > ioctl(fd, SIOCGIFINDEX, &ifr))
		{
			close(fd);
			return FALSE;
		}
		*piIfIndex = ifr.ifr_ifindex;
	}
	if (NULL!=MacAddr)
	{
		if (0 > ioctl(fd, SIOCGIFHWADDR, &ifr))
		{
			close(fd);
			return FALSE;
		}
		memcpy(MacAddr, ifr.ifr_addr.sa_data, sizeof(MacAddr_t));
	}
	return 0==close(fd);
}

bool_t	OsBindIfNameOnRAWPkt(OsSocket_t Socket, const char* cszIfName)
{
	struct sockaddr_ll from = {0};

	if (OsGetIfInfoByName(cszIfName, &from.sll_ifindex, NULL))
	{
		from.sll_family = AF_PACKET;
		return 0==bind(Socket, (struct sockaddr*)&from, sizeof(from));
	}
	return FALSE;
}

bool_t	OsListen(OsSocket_t Socket, int iBacklog)
{
	return 0==listen(Socket, iBacklog);
}

OsSocket_t	OsAccept(OsSocket_t Socket, struct sockaddr *pAddr, int *pAddrLen)
{
	return accept(Socket, pAddr, pAddrLen);
}

OsSocket_t	OsAcceptInternet(OsSocket_t Socket, uint32_t *pulIP, uint16_t *pusPort)
{
	struct sockaddr_in	inAddr		= {0};
	int					iAddrLen	= sizeof(struct sockaddr_in);
	OsSocket_t			SocketAccept= accept(Socket, (struct sockaddr*)&inAddr, &iAddrLen);

	if (NULL!=pulIP)
	{
		*pulIP = inAddr.sin_addr.s_addr;
	}
	if (NULL!=pusPort)
	{
		*pusPort = ntohs(inAddr.sin_port);
	}
#ifdef	_DEBUG
	printf("Received from ");
	PrintIP(inAddr.sin_addr.s_addr);
	printf(" %d\r\n", ntohs(inAddr.sin_port));
#endif//DEBUG
	return SocketAccept;
}

int32_t	OsRecv(OsSocket_t Socket, void *pBuf, int32_t iLen, int iFlags)
{
	return recv(Socket, pBuf, iLen, iFlags);
}

int32_t	OsRecvFrom(OsSocket_t Socket, void *pBuf, int32_t iLen, int iFlags, 
				   struct sockaddr *From, int32_t *iFromLen)
{
	struct sockaddr	sockFrom;
	int32_t			iLength;

	if (NULL==From || NULL==iFromLen)
	{
		return recvfrom(Socket, pBuf, iLen, iFlags, &sockFrom, &iLength);
	}
	else
	{
		return recvfrom(Socket, pBuf, iLen, iFlags, From, iFromLen);
	}
}

int32_t	OsSend(OsSocket_t Socket, const void *pBuf, int32_t iLen, int iFlags)
{
	return send(Socket, pBuf, iLen, iFlags);
}

int32_t OsSendTo(OsSocket_t Socket, const void *pBuf, int32_t iLen,
				 int iFlags, uint32_t ulRemoteIP, uint16_t usPort)
{
	struct sockaddr_in	inRemoteAddr = {0};

	inRemoteAddr.sin_family = AF_INET;
	inRemoteAddr.sin_port = htons(usPort);
	inRemoteAddr.sin_addr.s_addr = ulRemoteIP;
	return sendto(Socket, pBuf, iLen, iFlags, (struct sockaddr*)&inRemoteAddr, sizeof(inRemoteAddr));
}
///////////////////////////////////////////////////////////////////////////////
// socket test routines
///////////////////////////////////////////////////////////////////////////////
bool_t	OsIsPacketIncoming(OsSocket_t Socket, uint32_t ulTimeout)
{
	fd_set	FdSet;
	struct	timeval timeout = {0};

	timeout.tv_sec = ulTimeout/1000;			//seconds
	timeout.tv_usec = (ulTimeout%1000)*1000;	//microsecond
	if (0==ulTimeout)
	{
		timeout.tv_usec = 1;
	}

	FD_ZERO(&FdSet);
	FD_SET(Socket, &FdSet);
	select(Socket + 1, &FdSet, NULL, NULL, &timeout);

	return FD_ISSET(Socket, &FdSet);
}
