// os_file.c
//

#include <stdio.h>
#include <string.h>

#include "os_mem.h"
#include "os_file.h"

const char*	OsFileName(const char *cszFullFileName)
{
	const char	*cszFileName = strrchr(cszFullFileName, '\\');
	if (NULL==cszFileName)
	{
		cszFileName = strrchr(cszFullFileName, '/');
	}
	if (NULL==cszFileName)
	{
		cszFileName = strrchr(cszFullFileName, ':');
	}
	return (NULL==cszFileName)?cszFullFileName:cszFileName+1;
}

OsFile_t	OsFileOpen(const char *cszFileName, const char *cszOpenMode)
{
	return fopen(cszFileName, cszOpenMode);
}

void		OsFileClose(OsFile_t FileHandle)
{
	fclose(FileHandle);
}

uint32_t	OsGetFileSize(const char *cszFile, bool_t fIsText)
{
	FILE		*pFile = fopen(cszFile, fIsText?"rt":"rb");
	uint32_t	ulFileSize = 0;
	if (NULL!=pFile)
	{
		fseek(pFile, 0, SEEK_END);
		ulFileSize = ftell(pFile);
		fclose(pFile);
	}
	return ulFileSize;
}

uint8_t*	OsGetFileContent(const char *cszFile, bool_t fIsText)
{
	FILE	*pFile = fopen(cszFile, fIsText?"rt":"rb");
	uint8_t	*szContent = NULL;
	if (NULL!=pFile)
	{
		uint32_t	ulFileSize = 0;
		
		fseek(pFile, 0, SEEK_END);
		ulFileSize = ftell(pFile);
		szContent = OsMemNew(ulFileSize+1, FALSE);
		fseek(pFile, 0, SEEK_SET);
		fread(szContent, sizeof(uint8_t), ulFileSize, pFile);
		fclose(pFile);
		szContent[ulFileSize] = '\0';
	}
	return szContent;
}

#ifdef	WIN32
#include <direct.h>
#include <windows.h>

char*	OsGetCurrentDirectory(void)
{
	char		*szPath = NULL;
	uint32_t	ulSize = GetCurrentDirectory(0, szPath);
	if (ulSize!=0 && NULL!=(szPath=OsMemNew(ulSize+1, FALSE)))
	{
		GetCurrentDirectory(ulSize+1, szPath);
	}
	return szPath;
}

bool_t	OsSetCurrentDirectory(const char *cszPath)
{
	return SetCurrentDirectory(cszPath);
}

static OsFileEntry_t*	FileEntryNew(const char *cszFileName)
{
	uint32_t		ulSize = strlen(cszFileName)+1;
	OsFileEntry_t	*pEntry = NULL;
	
	if (ulSize>sizeof(pEntry->szFileName))
	{
		ulSize -= sizeof(pEntry->szFileName);
	}
	ulSize += sizeof(OsFileEntry_t);
	if (NULL!=(pEntry=(OsFileEntry_t*)OsMemNew(ulSize, FALSE)))
	{
		pEntry->pNext = NULL;
		pEntry->pChild = NULL;
		strcpy(pEntry->szFileName, cszFileName);
	}
	return pEntry;
}

static OsFileEntry_t*	GetFileList(
	const char	*cszPath,
	const char	*szFileName,
	int32_t		lDepth
	)
{
	OsFileEntry_t	*pEntry = NULL;
	OsFileEntry_t	**ppEntry = &pEntry;
	WIN32_FIND_DATA	FindFileData = {0};
	HANDLE			hFind = INVALID_HANDLE_VALUE;

	chdir(cszPath);
	if (INVALID_HANDLE_VALUE==(hFind=FindFirstFile(szFileName, &FindFileData)))
	{
		return NULL;
	}

	do
	{
		if (FILE_ATTRIBUTE_DIRECTORY!=FindFileData.dwFileAttributes)
		{
			*ppEntry = FileEntryNew(FindFileData.cFileName);
			ppEntry = &((*ppEntry)->pNext);
		}
		else if (0!=lDepth
			&& 0!=strcmp(".", FindFileData.cFileName)
			&& 0!=strcmp("..", FindFileData.cFileName)
			)
		{
			*ppEntry = FileEntryNew(FindFileData.cFileName);
			(*ppEntry)->pChild = GetFileList(FindFileData.cFileName, 
				szFileName, lDepth<0?lDepth:lDepth-1);
			ppEntry = &((*ppEntry)->pNext);
		}
	} while (FindNextFile(hFind, &FindFileData));
	FindClose(hFind);
	return pEntry;
}

OsFileEntry_t*	OsGetFileList(
	const char	*cszPath,
	const char	*szFileName,
	int32_t		lDepth
	)
{
	// keep original current directory, later recover it.
	char			*szCurrentDirectory = OsGetCurrentDirectory();
	OsFileEntry_t	*pEntry = NULL;

	if (NULL==cszPath || SetCurrentDirectory(cszPath))
	{
		pEntry = GetFileList(".", (NULL==szFileName)?"*":szFileName, lDepth);
	}

	OsSetCurrentDirectory(szCurrentDirectory);
	OsMemFree(szCurrentDirectory);
	return pEntry;
}
#else
//#error TODO: Forrest
#endif//!WIN32

void OsFileEntryFree(OsFileEntry_t *pEntry)
{
	while (NULL!=pEntry)
	{
		OsFileEntry_t	*pNext = pEntry->pNext;
		if (NULL!=pEntry->pChild)
		{
			OsFileEntryFree(pEntry->pChild);
		}
		OsMemFree(pEntry);
		pEntry = pNext;
	}
}
