//*****************************************************************************
//*
//*
//*     SystemLinux.cpp
//*
//*
//*****************************************************************************
//
//  This file is an implementation of system functions for LINUX
//
//  Copyright  2003    Anton Zechner
//
#include <time.h>
#include <fcntl.h>
#include <sched.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <memory.h>
#include <malloc.h>
#include <unistd.h>
#include <pthread.h>
#include <semaphore.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/statfs.h>
#include <sys/statvfs.h>
#include <sys/resource.h>

#include "System.h"


#define     HAS_PATH_CACHE          0               // cache converted paresed paths
#define     MAP_DOS_FILE_BITS       0               // use execute bits to simulate HIDDEN SYSTEM and ACHIVE flags
#define     MAX_EX_HANDELS          64              // the count of similar open handels
#define     MAX_TEMP_PATH           256             // is the maximal path size whith can be converted
#define     SYS_API                 extern "C"      // call convention of the functions

static      int                     iTimeInit=0;    // is the time zone initalized
static      int volatile            iThreadCount=0; // is the count of running threads
static      int                     iHandleInit=0;  // is the handle mutex initalized
static      pthread_mutex_t         hHandleMutex;   // is the handle mutex for all handles
#define     HANDLE_INIT()           if(!iHandleInit){pthread_mutex_init(&hHandleMutex,0);iHandleInit=1;}
#define     HANDLE_LOCK()           pthread_mutex_lock  (&hHandleMutex);
#define     HANDLE_UNLOCK()         pthread_mutex_unlock(&hHandleMutex);

static      unsigned                uJiffMul =0x100000*10;
static      unsigned                uJiffLast=0xFFFFFFFF;
static      unsigned                uJiffInit=0;
static      unsigned                uJiffAdd =0;

#if         SYS_HAS_NAME_INFO

    #define EXDENDED_HANDLES        1

#endif

#if         MAP_DOS_FILE_BITS

    #define EXDENDED_HANDLES        1

#endif

#ifdef EXDENDED_HANDLES

    typedef struct
        {
        char   *pName;
        int     iHandle;
        #if     MAP_DOS_FILE_BITS
        int     iArchBit;
        #endif
        }ExHandle;

    static ExHandle             aHandles[MAX_EX_HANDELS];
    static unsigned volatile    uNextPos=0;

    static void SysDelExtHandle(SysFile hHandle)
                {
                unsigned    uPos=(unsigned)(hHandle)-1;

                if(uPos<MAX_EX_HANDELS)
                if(aHandles[uPos].pName)
                    {
                    if(aHandles[uPos].iArchBit)
                        {
                        struct stat sStat;
                        int         iFile;
                        int         iOk;

                                aHandles[uPos].iArchBit=0;
                        iFile = aHandles[uPos].iHandle;

                           iOk = fstat(iFile,&sStat);
                        if(iOk>=0)
                            {
                            fchmod(iFile,sStat.st_mode|S_IXUSR);
                            }
                        }

                    #if SYS_HAS_NAME_INFO
                    free(aHandles[uPos].pName);
                    #endif
                    aHandles[uPos].iHandle = -1;
                    }
                }

    static SysFile SysNewExtHandle(int iHandle,const char *pName,int *pError)
                {
                char       *pText;
                unsigned    uCount;
                unsigned    uPos;
                int         iLen;
                int         iNum;

                HANDLE_INIT();

                #if SYS_HAS_NAME_INFO
                iLen = strlen(pName);

                for(iNum=iLen;iNum>0;iNum--)
                    {
                    if(pName[iNum]!='/')continue;
                    iNum++;
                    break;
                    }

                    iLen -= iNum-1;
                    pText = (char*)malloc(iLen);
                if(!pText)
                    {
                    if(pError)*pError=SYS_ERR_UNKNOWN;
                    close(iHandle);
                    return 0;
                    }

                memcpy(pText,pName+iNum,iLen);

                #else

                pText = "";

                #endif

                for(uCount=0;uCount<MAX_EX_HANDELS;uCount++)
                    {
                    uPos = uNextPos%MAX_EX_HANDELS;
                    uNextPos++;

                    HANDLE_LOCK();

                    if(aHandles[uPos].pName)
                        {
                        HANDLE_UNLOCK();
                        continue;
                        }

                    aHandles[uPos].pName    = pText;
                    aHandles[uPos].iHandle  = iHandle;
                    aHandles[uPos].iArchBit = 0;
                    HANDLE_UNLOCK();

                    return (SysFile)(uPos+1);
                    }

                if(pError)*pError=SYS_ERR_NO_HANDLES;
                close(iHandle);

                return 0;
                }

    #define SYS_GET_NAME(h)         (((unsigned)((unsigned)(h)-1)>=MAX_EX_HANDELS)? "":aHandles[(unsigned)(h)-1].pName)
    #define SYS_GET_HANDLE(h)       (((unsigned)((unsigned)(h)-1)>=MAX_EX_HANDELS)? -1:aHandles[(unsigned)(h)-1].iHandle)
    #define SYS_NEW_HANDLE(h,n,e)   SysNewExtHandle(h,n,e)
    #define SYS_DEL_HANDLE(h)       SysDelExtHandle(h);

    #if     MAP_DOS_FILE_BITS
    #define SYS_SET_ARCHBIT(h,s)    if(s>0 && ((unsigned)((unsigned)(h)-1)<MAX_EX_HANDELS))aHandles[(unsigned)(h)-1].iArchBit=1;
    #else
    #define SYS_SET_ARCHBIT(h,s)
    #endif

#else

    #define SYS_GET_NAME(h)         ""
    #define SYS_GET_HANDLE(h)       (((int)h)-1)
    #define SYS_NEW_HANDLE(h,n,e)   ((SysFile)(h+1))
    #define SYS_DEL_HANDLE(h)
    #define SYS_SET_ARCHBIT(h,s)

#endif

#if         HAS_PATH_CACHE

    #error  "isn't implemented"

#else

    #define CALC_CHKSUM(a,b)
    #define CACHE_ADD(a,b,c)

#endif





typedef struct
    {
    int       iPriority;                                // is the thread priority (not supported)
    int       iStartOk;                                 // will be set to 1 if thread was stared
    void    (*pStartProc)(void*);                       // is a pointer to the therad function
    void     *pStartArg;                                // is the arument pointer for the thread
    }ThreadData;

typedef struct
    {
    int                 iId;                            // is the id of the semaphore 0x1255AA78
    int                 iValue;                         // is the current lock count (>0 = unlocked)
    int                 iNumber;                        // is an access count
    pthread_mutex_t     hAccess;                        // is the access mutex of the semathor
    pthread_mutex_t     hDummy;                         // is an internal used mutex
    pthread_cond_t      hWait;                          // is the global wait semathor
    }SemData;


#define     FIND_MAX    16                              // count of maximal similar open find handles
#define     FIND_LEN    256                             // maximal size of a find path
#define     FIND_ONE    (FIND_MAX+12345)

typedef struct
   {
   int                  iFree;                          // is the handle in use
   int                  iLen;                           // is the length of the find path
   int                  iNum;                           // is the length of the seach name
   DIR                 *pDir;                           // is the pointer to the directory entry
   char                 cFind[FIND_LEN+4];              // is the seach name with wildchards
   char                 cName[FIND_LEN+4];              // is the path name of the search
   }FindMem;




static  FindMem         aFindMem[FIND_MAX];
static  int             iFindInit=0;
//extern	int							drop_cache_FD;

//*****************************************************************************
//*
//*     SysError
//*
//*****************************************************************************
//  Converts an Linunx ErrorCode to a System-Error-Code
static int SysError()
{


    switch(errno)
        {
    case EPERM          : return SYS_ERR_ACCESS_DENIED;
    case EBADF          : return SYS_ERR_ACCESS_DENIED;
    case ENOENT         : return SYS_ERR_FILE_NOT_FOUND;
    case ENXIO          : return SYS_ERR_FILE_NOT_FOUND;
    case EACCES         : return SYS_ERR_ACCESS_DENIED;
    case ENOTBLK        : return SYS_ERR_ACCESS_DENIED;
    case EEXIST         : return SYS_ERR_FILE_NOT_FOUND;
    case EXDEV          : return SYS_ERR_FILE_NOT_FOUND;
    case ENODEV         : return SYS_ERR_FILE_NOT_FOUND;
    case ENOTDIR        : return SYS_ERR_PATH_NOTFOUND;
    case EINVAL         : return SYS_ERR_PARAM;
    case ENFILE         : return SYS_ERR_NO_HANDLES;
    case EMFILE         : return SYS_ERR_NO_HANDLES;
    case EFBIG          : return SYS_ERR_DISKFULL;
    case ENOSPC         : return SYS_ERR_DISKFULL;
    case EROFS          : return SYS_ERR_ACCESS_DENIED;
    case ENAMETOOLONG   : return SYS_ERR_INVALID_FILENAME;
    case ENOSTR         : return SYS_ERR_ACCESS_DENIED;
        }

return SYS_ERR_UNKNOWN;
}

//*****************************************************************************
//*
//*      strfcmp / strnxcmp
//*
//*****************************************************************************
//  Compares to strings, the second string could contain the wildchar ?.
//  pText       : is the pointer to the first string.
//  pSource     : is the pointer to the second string.
//                a '?' represents an sigle char.
//  iLen        : is the length of the strings in byte
//  returns         0 if the strings are equal
//                  1 if the strings are not equal
static int strfcmp(const char *pText,const char *pSource,int iLen)
{
register int    i;
register char   a,b;


    for(i=0;i<iLen;i++)                         // Compare strings
        {
        b=pSource[i];
        if(b=='?')continue;
        a=pText[i];
        if(a>='a' && a<='z')a+='A'-'a';
        if(b>='a' && b<='z')b+='A'-'a';
        if(a!=b)return  1;
        }


return 0;
}

//*****************************************************************************
//*
//*      strnxcmp
//*
//*****************************************************************************
//  Compares to strings, the second string could contain the wildchars * and ?
//  pText       : is the pointer to the first string.
//  iTextLen    : is the length of the first strings in byte
//  pSource     : is the pointer to the second string.
//                a '?' represents an sigle char.
//                a '*' represents several chars.
//  iLen        : is the length of the second strings in byte
//  returns         0 if the strings are equal
//                  1 if the strings are not equal
static int strnxcmp(const char *pText,int iTextLen,const char *pSource,int iSourceLen)
{
int     iTextPos,iFindPos;
int     i,iLen,iStar;



    if(iSourceLen<=0)return 0;


    iFindPos=0;
    iTextPos=0;


    for(;;)                                     // Suchen Schleife
        {
        if(iSourceLen<=0)break;
        if(  iTextLen<=0)break;

        iLen=0;
        while(iLen<iSourceLen)                  // Wort Lnge holen
            {
            if(pSource[iLen]=='*')break;
            iLen++;
            }
        if(iTextLen<iLen)return 1;
        if(iLen)
            {
            if(strfcmp(pText,pSource,iLen))return 1;
            iSourceLen-=iLen;pSource+=iLen;
              iTextLen-=iLen;  pText+=iLen;
            continue;
            }

        iStar=0;

        while(iSourceLen>0)                     // Wildcard berspringen
            {
            if(*pSource=='?'){iStar++;pSource++;iSourceLen--;continue;}
            if(*pSource!='*')break;
                pSource++;iSourceLen--;
            }

        if(iTextLen<iStar)return 1;
        if(iSourceLen<=0 )return 0;

        iTextLen -= iStar;
        pText    += iStar;
        iLen      = 0;

        while(iLen<iSourceLen)                  // Wort Lnge holen
            {
            if(pSource[iLen]=='*')break;
            iLen++;
            }

        while(iTextLen>=iLen)                   // Source Wortteil
            {
            if(!strfcmp(pText,pSource,iLen))
                {
                if(iSourceLen==iLen)return iTextLen!=iLen;
                i=strnxcmp( pText  +iLen,  iTextLen -iLen,
                            pSource+iLen,iSourceLen -iLen);

                if(i==0)return i;
                }

            iTextLen--;
            pText++;
            }


        return 1;
        }

//********************** Suchen_len <= 0 **************************************

    if(iSourceLen>0)                                // Stern am Wortende
        {
        while(iSourceLen>0)
            {
            if(*pSource!='*')return 1;
            iSourceLen--;
            pSource--;
            }
        return 0;
        }

    if(iTextLen>0)return 1;

return 0;
}

//*****************************************************************************
//*
//*     SysFullPath
//*
//*****************************************************************************
//  converts a relative path to a full path
//  pPath       : is the relative path
//  pBuffer     : is the buffer for the full path
//  iMax        : is the size of the buffer
//  returns the lenght in bytes of the full path cName.
static int SysFullPath(const char *pPath,char *pBuffer,int iMax)
{
char    *pStr;
int      iLen,iSlash;
int      iPos=0,iAdd;



    if(iMax<=0)return 0;

    pBuffer[0]=0;
    iLen=strlen(pPath);
    if(iLen+1>=iMax)return 0;

    if(pPath[0]!='/' && pPath[0]!='\\')
        {
        pStr=getcwd(pBuffer,iMax);
        if(pStr==NULL)return 0;
        iPos=strlen(pBuffer);
        if(iPos+1+iLen>=iMax)return 0;
        pBuffer[iPos]='/';
        iPos++;
        memcpy(pBuffer+iPos,pPath,iLen+1);
        iLen+=iPos;
        }
    else{
        memcpy(pBuffer,pPath,iLen+1);
        pBuffer[0]='/';
        iPos=1;
        }


    while(iPos<iLen)
        {                                               // filter "//"
        if(iPos>0 && (pBuffer[iPos]=='/' || pBuffer[iPos]=='\\'))
            {
            memmove(pBuffer+iPos,pBuffer+iPos+1,iLen-iPos);
            iLen-=1;
            continue;
            }
                                                        // filter "./"
        if(iPos>0 && pBuffer[iPos]=='.' && (pBuffer[iPos+1]=='/' || pBuffer[iPos+1]=='\\'))
            {
            memmove(pBuffer+iPos,pBuffer+iPos+2,iLen-iPos-1);
            iLen-=2;
            continue;
            }
                                                        // filter "."
        if(iPos>1 && pBuffer[iPos]=='.' && pBuffer[iPos+1]==0)
            {
            iPos--;
            pBuffer[iPos]=0;
            iLen=iPos;
            break;
            }
                                                        // filter "../"
        if(iPos>0 && pBuffer[iPos]=='.' && pBuffer[iPos+1]=='.' && (pBuffer[iPos+2]=='/' || pBuffer[iPos+2]=='\\'))
            {
            iSlash=iPos;

            for(iPos--;iPos>0;iPos--)
                {
                if(pBuffer[iPos-1]=='/')break;
                }

            iAdd = iSlash-iPos+3;

            memmove(pBuffer+iPos,pBuffer+iSlash+3,iLen-iSlash-2);
            iLen-=iAdd;
            continue;
            }
                                                        // filter ".."
        if(iPos>1 && pBuffer[iPos]=='.' && pBuffer[iPos+1]=='.' && pBuffer[iPos+2]==0)
            {
            for(iPos-=2;iPos>0;iPos--)
                {
                if(pBuffer[iPos]=='/')break;
                }

            if(iPos<=0)
                {
                pBuffer[0]='/';
                iPos=1;
                }

            pBuffer[iPos]=0;
            iLen=iPos;
            break;
            }

        for(;pBuffer[iPos];iPos++)                      // to next slash
            {
            if(pBuffer[iPos]!='/')
                {
                if(pBuffer[iPos]!='\\')continue;
                pBuffer[iPos]='/';
                }

            iPos++;
            break;
            }
        }



return iLen;
}

//*****************************************************************************
//*
//*     SysCheckPath
//*
//*****************************************************************************
//  converts a windows path to an linux path
//  pPath   : is the pointer to the windows path
//  pBuffer : is a buffer for the new linux path
//  returns a pointer to the new path
static const char *SysCheckPath(const char *pPath,char *pBuffer)
{
#if HAS_PATH_CACHE
unsigned        uChkSum=0;
#endif
#if SYS_HAS_SHORT_NAME
int             iNum,iPoint,iFoundLen=0;
char            cFound[sizeof(struct dirent)];
unsigned        uChkShr;
unsigned        uChkNow;
int             iShort;
#endif
DIR            *pDir;
struct dirent  *pResult;
struct dirent   sEntry;
struct stat     sStat;
char            cLast;
char            cChar;
int             iSize;
int             iLen;
int             iOk;
int             i,j;



    iLen = SysFullPath(pPath,pBuffer,MAX_TEMP_PATH);


       iOk = stat(pBuffer,&sStat);
    if(iOk>=0)                                          // exists the path
        {
        return pBuffer;
        }


    for(i=0;i<iLen;i=iSize)
        {
        for(iSize=i+1;iSize<iLen;iSize++)
            {
               cChar = pBuffer[iSize];
            if(cChar=='/')break;

            CALC_CHKSUM(uChkSum,cChar);
            }


        cLast           = pBuffer[iSize];
        pBuffer[iSize]  = 0;


           iOk = stat(pBuffer,&sStat);
        if(iOk>=0)                                      // exists the path
            {
            CACHE_ADD  (uChkSum,pBuffer,iSize);
            CALC_CHKSUM(uChkSum,'/'+i);
            pBuffer[iSize] = cLast;
            continue;
            }

        #if HAS_PATH_CACHE
        if(CACHE_FIND(uChkSum,pBuffer,iSize))           // is path in cache
            {
            CALC_CHKSUM(uChkSum,'/'+i);
            pBuffer[iSize] = cLast;
            continue;
            }
        #endif


        j =(!i)? 1:i;
        cChar       = pBuffer[j];
        pBuffer[j]  = 0;
        pDir        = opendir(pBuffer);                 // open directory entries
        pBuffer[j]  = cChar;

        if(!pDir)
            {
            pBuffer[iSize] = cLast;
            return pBuffer;
            }

        i++;

        #if SYS_HAS_SHORT_NAME
           iNum = iSize-i;
        if(iNum<=12 && iNum>=8 && pBuffer[i+2]=='~' &&
           iNum!= 9 &&(iNum==8 || pBuffer[i+8]=='.'))
            {
            uChkShr = 0;

            for(j=3;j<8;j++)
                {
                cChar = pBuffer[i+j];

                if(cChar<'0')break;
                if(cChar>'9' && cChar<'A')break;
                if(cChar>'Z' && cChar<'a')break;
                if(cChar>='a')
                    {
                    if(cChar>'z')break;
                    cChar -= 'a'-'A';
                    }

                if(cChar>='A')
                        cChar -= 'A'-10;
                else    cChar -= '0';

                uChkShr *= 36;
                uChkShr += (unsigned char)cChar;
                }


            if(j!=8 || (iNum> 9 && pBuffer[i+ 9]==' ') ||
                       (iNum>10 && pBuffer[i+10]==' ') ||
                       (iNum>11 && pBuffer[i+11]==' '))
                    iShort = 0;
            else    iShort = 1;

            cFound[0] = 0;
            }
        else{
            iShort  = 0;
            uChkShr = 0;
            }
        #endif


        for(;;)
            {
            readdir_r(pDir,&sEntry,&pResult);           // get next directory entry
            if(!pResult)break;

            #if SYS_HAS_SHORT_NAME
            if(!strncasecmp(sEntry.d_name,pBuffer+i,iNum))
                {
                if(sEntry.d_name[iNum]==0)break;

                if(iNum<=9)                             // is a '.' at the end
                if(sEntry.d_name[iNum  ]=='.')
                if(sEntry.d_name[iNum+1]== 0 )
                    {
                    memmove(pBuffer+iSize+1,pBuffer+iSize,iLen-iSize+1);
                    iSize++;
                    iLen++;
                    break;
                    }
                }

            #else
            if(!strcasecmp(sEntry.d_name,pBuffer+i))break;
            #endif

            #if SYS_HAS_SHORT_NAME
            if(!iShort || cFound[0])continue;           // searchfor a short path


                cChar = sEntry.d_name[0];
            if((cChar>='a' && cChar<='z') ||
               (cChar>='A' && cChar<='Z'))
                {
                if((cChar^pBuffer[i])&~0x20)continue;
                }
            else{
                if(cChar==' ')cChar='_';
                if(cChar!=pBuffer[i])continue;
                }

                cChar = sEntry.d_name[1];
            if((cChar>='a' && cChar<='z') ||
               (cChar>='A' && cChar<='Z'))
                {
                if((cChar^pBuffer[i+1])&~0x20)continue;
                }
            else{
                if(cChar==' ')cChar='_';
                if(cChar!=pBuffer[i+1])continue;
                }

            uChkNow = 0;
            iPoint  = 0;

            for(j=0;j<sEntry.d_name[j];j++)             // calculate checksum
                {
                cChar = sEntry.d_name[j];

                if(cChar>='a' && cChar<='z')cChar-='a'-'A';
                else if(cChar=='.')iPoint=j;

                uChkNow  = (uChkNow<<5)|(uChkNow>>27);
                uChkNow += (unsigned char)cChar;
                }

            uChkNow &= 0x01FFFFFF;

            if(uChkNow!=uChkShr)continue;

            if(iPoint+1==j)                             // last chat is a '.'
                {
                if(iNum!=8)continue;
                memcpy(cFound,sEntry.d_name,j);
                iFoundLen = j;
                continue;
                }

            if(iPoint==0)                               // no suffix
                {
                if(iNum!=8)continue;
                memcpy(cFound,sEntry.d_name,j);
                iFoundLen = j;
                continue;
                }

                cChar = sEntry.d_name[iPoint+1];
            if((cChar>='a' && cChar<='z') ||
               (cChar>='A' && cChar<='Z'))
                {
                if((cChar^pBuffer[i+9])&~0x20)break;
                }
            else{
                if(cChar==' ')cChar='_';
                if(cChar!=pBuffer[i+9])break;
                }

            if(iPoint+2==j)                             // one byte suffix
                {
                memcpy(cFound,sEntry.d_name,j);
                iFoundLen = j;
                continue;
                }

                cChar = sEntry.d_name[iPoint+2];
            if((cChar>='a' && cChar<='z') ||
               (cChar>='A' && cChar<='Z'))
                {
                if((cChar^pBuffer[i+10])&~0x20)break;
                }
            else{
                if(cChar==' ')cChar='_';
                if(cChar!=pBuffer[i+10])break;
                }

            if(iPoint+3==j)                             // tow byte suffix
                {
                memcpy(cFound,sEntry.d_name,j);
                iFoundLen = j;
                continue;
                }

                cChar = sEntry.d_name[iPoint+2];
            if((cChar>='a' && cChar<='z') ||
               (cChar>='A' && cChar<='Z'))
                {
                if((cChar^pBuffer[i+10])&~0x20)break;
                }
            else if(iPoint+4==j)
                {
                if(cChar==' ')cChar='_';
                if(cChar!=pBuffer[i+11])break;
                }
            else if(pBuffer[i+11]!='~')
                {
                break;
                }

            memcpy(cFound,sEntry.d_name,j);
            iFoundLen = j;
            #endif
            }

        closedir(pDir);

        if(pResult)                                     // found a same uncase sensidife name
            {
            memcpy(pBuffer+i,sEntry.d_name,iSize-i);
            CALC_CHKSUM(uChkSum,'/'+i);
            pBuffer[iSize] = cLast;
            continue;
            }

        #if SYS_HAS_SHORT_NAME
        if(iShort && cFound[0])                         // was a short path found
            {
            pBuffer[iSize] = cLast;
            CALC_CHKSUM(uChkSum,'/'+i);

            j = iFoundLen-(iSize-i);

            if(iLen+j<MAX_TEMP_PATH)
                {
                memmove(pBuffer+iSize+j,pBuffer+iSize,iLen-iSize+1);
                memcpy (pBuffer+i,cFound,iFoundLen);
                iSize += j;
                iLen  += j;
                continue;
                }
            }
        #endif

        pBuffer[iSize] = cLast;
        break;
        }



return pBuffer;
}

//*****************************************************************************
//*
//*     SysClearFindStruct
//*
//*****************************************************************************
//  Lscht eine Find-Struktur
//  pFind   : Zeiger auf die Find-Struktur
inline void SysClearFindStruct(SysFindStruct *pFind)
{

    pFind->dwSize=0;
    pFind->cName[0]=0;
    #if SYS_HAS_SHORT_NAME
    pFind->cShortName[0]=0;
    #endif
}

//*****************************************************************************
//*
//*     SysCopyShortName
//*
//*****************************************************************************
//  Converts a filename in a MSDOS 8.3 format
#if SYS_HAS_SHORT_NAME
inline void SysCopyShortName(char *pName,const char *pFullName)
{
unsigned    uChkNow;
char        cChar;
int         iPoint;
int         iNum;
int         i;



    if(pFullName[0]== 0)                            // don't convert emty names
        {
        pName[0] = 0;
        return;
        }

    if(pFullName[0]=='.')                           // don't convert "." and ".."
        {
        if(pFullName[1]==0)
            {
            pName[0] = '.';
            pName[1] = 0;
            return;
            }

        if(pFullName[1]=='.' && pFullName[2]==0)
            {
            pName[0] = '.';
            pName[1] = '.';
            pName[2] = 0;
            return;
            }
        }


    uChkNow =  0;
    iPoint  = -1;
    iNum    =  0;

    for(i=0;pFullName[i];i++)                       // calculate checksum
        {
        cChar = pFullName[i];

             if(cChar==' '){iNum=2;}
        else if(cChar=='.'){iNum++;iPoint=i;}
        else if(cChar>='a' && cChar<='z')cChar-='a'-'A';

        if(i<12)pName[i]=cChar;

        uChkNow  = (uChkNow<<5)|(uChkNow>>27);
        uChkNow += (unsigned char)cChar;
        }


    if(iNum<2 && i<12)                              // no checksum needed
    if((iPoint<0 && i<=8) || i-iPoint<=4)
        {
        if(iPoint+1==i)i--;
        pName[i] = 0;
        return;
        }

    uChkNow &= 0x01FFFFFF;

    if(pName[0]==' ')pName[0]='_';
    if(pName[1]==' ')pName[1]='_';
                     pName[2]='~';

    for(i=7;i>=3;i--)                               // write checksum to name
        {
           cChar = (char)(uChkNow%36);
        if(cChar>=10)
                cChar+='A'-10;
        else    cChar+='0';

        pName[i]=cChar;
        uChkNow/=36;
        }

    if(iPoint<0)                                    // exists a suffix
        {
        pName[8]=0;
        return;
        }


    pName[8]='.';

    for(iPoint++,i=9;i<12;i++)                      // copy suffix
        {
                cChar=pFullName[iPoint];
             if(cChar==0)break;
        else if(cChar>='a' && cChar<='z')cChar-='a'-'A';
        else if(cChar==' ')cChar='_';

        pName[i]=cChar;
        iPoint++;
        }

    if(i==9)                                        // remove the last point
        {
        i=8;
        }
    else if(i==12 && pFullName[iPoint])             // suffix larger than 3 chars ?
        {
        pName[11]='~';
        }

    pName[i] = 0;

}
#endif

//*****************************************************************************
//*
//*     SysTimeConvert     DOSTIME --> SysTime
//*
//*****************************************************************************
//  Konvertiert eine DOS Zeit in die Systemzeit
//  uDosTime    : Ist die DOS Zeit
//  pSysTime    : Zeiger auf Puffer fr die Systemzeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertDS(SysDW uDosTime ,SysTime *pSysTime)
{
static const int    aTpPDays [] = {  0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366, 366, 366,   0};
static const int    aTpDays  [] = {  0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365, 365, 365,   0};
unsigned            uVal,uYear,uDays;
unsigned short      wTime,wDate;



    wTime=(unsigned short) uDosTime;
    wDate=(unsigned short)(uDosTime>>16);

       uYear = (wDate>>9)+80;
    if(uYear>2038)
        {
        *pSysTime=0;
        return 0;
        }

    if(uYear&3)uDays  = aTpDays  [((wDate>>5)-1)&15];
    else       uDays  = aTpPDays [((wDate>>5)-1)&15];
               uDays += ((wDate)&0x1F)-1;
               uDays += (uYear-70  )*365;
               uDays += (uYear-70+1)>>2;

               uVal   = uDays*24;
               uVal  += (wTime>>11);
               uVal  *= 60;
               uVal  += (wTime>> 5)&0x3F;
               uVal  *= 60;
               uVal  += (wTime<< 1)&0x3F;
    *pSysTime =uVal;


return 1;
}

//*****************************************************************************
//*
//*     SysTimeConvert     SysTime --> DOSTIME
//*
//*****************************************************************************
//  Konvertiert eine eine Systemzeit in eine DOS-Zeit
//  uSysTime    : Ist die Systemzeit
//  pDosTime    : Zeiger auf Puffer fr die DOS Zeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertSD(SysTime uSysTime,SysDW *pDosTime)
{
struct tm      *pGlobal;
struct tm       sGlobal;
time_t          iTime;
unsigned short  wTime,wDate;


    iTime = uSysTime;

       pGlobal=gmtime_r(&iTime,&sGlobal);
    if(pGlobal==NULL || pGlobal->tm_year<80 || pGlobal->tm_year>143)
        {
        *pDosTime = 1<<5;
        return 0;
        }

    wDate = (pGlobal->tm_mday  ) | ((pGlobal->tm_mon+1)<<5) | ((pGlobal->tm_year-80)<< 9);
    wTime = (pGlobal->tm_sec>>1) | ( pGlobal->tm_min   <<5) | ( pGlobal->tm_hour    <<11);

    *pDosTime = wTime|(wDate<<16);


return 1;
}

//*****************************************************************************
//*
//*     SysTimeConvertSF        SysTime --> LONGTIME
//*
//*****************************************************************************
//  Konvertiert eine Systemzeit in eine Windows Filetime (100ns)
//  uSysTime    : Ist die SystemZeit
//  pWftTime    : Zeiger auf Puffer fr die Windows Filetime
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertSF(SysTime uSysTime,void *pWftTime)
{
union   {
        long long   i64;
        unsigned    dw[2];
        }uVal;


    if(uSysTime==0xFFFFFFFF || uSysTime==0)
        {
        ((unsigned*)pWftTime)[1] = 0xFFFFFFFF;
        ((unsigned*)pWftTime)[0] = 0xFFFFFFFF;
        return 0;
        }


    uVal.i64  = uSysTime;
    uVal.i64 *= 10000000;                       // Sekunden zu 100ns umwandeln
    uVal.i64 += 0x019db1ded53e8000LL;

    ((SysDW*)pWftTime)[1] = uVal.dw[1];         // Zeit speichern
    ((SysDW*)pWftTime)[0] = uVal.dw[0];


return 1;
}


//*****************************************************************************
//*
//*     SysTimeConvertFS        LONGTIME --> SysTime
//*
//*****************************************************************************
//  Konvertiert eine Windows Filetime in eine Systemzeit
//  uDosTime    : Ist der Zeiger auf die Windows Filetime
//  pSysTime    : Zeiger auf Puffer fr die Systemzeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertFS(const void *pWftTime,SysDW *pSysTime)
{
union   {
        long long   i64;
        unsigned    dw[2];
        }uVal;


    uVal.i64  = *(long long*)pWftTime;
    uVal.i64 -= 0x019db1ded53e8000LL;

    if(uVal.dw[1]&0x80000000)                   // Zeit vor Mi 1.Jan.1970
        {
        *pSysTime=0;
        return 0;
        }

    uVal.i64 /= 10000000;                       // 100ns zu Sekunden umwandeln

    if(uVal.dw[1]&0x80000000)                   // ber Jahr 2106
        {
        *pSysTime=0xFFFFFFFF;
        return 0;
        }

    *pSysTime = uVal.dw[0];                     // Systemzeit speichern


return 1;
}

//******************************************************************************
//*
//*     SysTimeConvertSU        SysTime --> time_t
//*
//******************************************************************************
//  Konvertiert eine Systemzeit in eine UTC Zeit(time_t)
//  uSysTime    : Ist der Zeiger auf die Windows Filetime
//  pUtcTime    : Zeiger auf Puffer fr die UTC-Zeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertSU(SysTime uSysTime,void *pUtcTime)
{

    *(time_t*)pUtcTime = uSysTime;


return (uSysTime==0xFFFFFFFF)? 0:1;
}

//******************************************************************************
//*
//*     SysTimeConvertUS    time_t --> SysTime
//*
//******************************************************************************
//  Konvertiert eine UTC Zeit(time_t) in eine Systemzeit
//  pUtcTime    : Ist der Zeiger auf die UTZ-Zeit
//  pSysTime    : Zeiger auf Puffer fr die Systemzeit
//  Ergibt 1 wenn erfolgreich ansonsten 0
SYS_API int SysTimeConvertUS(const void *pUtcTime,SysTime *pSysTime)
{

    *(SysTime*)pSysTime = *(SysTime*)pUtcTime;


return (*pSysTime==0xFFFFFFFF || *pSysTime==0)? 0:1;
}



//******************************************************************************
//*
//*     SysTimeGet
//*
//******************************************************************************
//  Returns the current system time
SYS_API SysTime SysTimeGet()
{
return time(0);
}

//*****************************************************************************
//*
//*     SysTimeToStruct
//*
//*****************************************************************************
//  Converts a SysTime value in a time structure
//  uTime   : global time in seconds from 1.Jan.2000
//  pData   : pointer to the time structure for the global time
//  uTime   : global time value that will converted
//  uMode   : is the conversion mode
//              0 = global time
//              1 = local time
//              2 = local time wihout daylight saving
SYS_API void SysTimeToStruct(SysTimeStruct *pData,SysTime uTime,unsigned uMode)
{
time_t      iUtc;
struct tm  *pTimeTm;


    if(!iTimeInit)
        {
        iTimeInit=1;
        time(0);
        }

    iUtc = uTime;

    switch(uMode)
        {
    case 2:     iUtc    -=  timezone;
    case 0:     pTimeTm  =  gmtime   (&iUtc);   break;
    case 1:     pTimeTm  =  localtime(&iUtc);   break;
    default:    memset(pData,0,sizeof(SysTimeStruct));
                return;
        }


    pData->usYear   = (unsigned short) (pTimeTm->tm_year+1900);
    pData->ucMonth  = (unsigned char ) (pTimeTm->tm_mon+1);
    pData->ucDay    = (unsigned char )  pTimeTm->tm_mday;
    pData->ucHour   = (unsigned char )  pTimeTm->tm_hour;
    pData->ucMinute = (unsigned char )  pTimeTm->tm_min;
    pData->ucSecond = (unsigned char )  pTimeTm->tm_sec;
    pData->ucIsDst  = (unsigned char )((pTimeTm->tm_isdst)? 1:0);

}

//*****************************************************************************
//*
//*     SysTimeFromData
//*
//*****************************************************************************
//  Converts the local time from a time structure in a SysTime value
//  pData   : pointer to the time structure
//  uMode   : is the conversion mode
//              0 = pData is a global time
//              1 = pData is a local time ucIsDst is valid
//              2 = pData is a local time ucIsDst is in valid
//  Returns the converted time or 0xFFFFFFFF at an error
SYS_API SysTime SysTimeFromStruct(const SysTimeStruct *pData,unsigned uMode)
{
time_t          iUtc;
struct tm       sTimeTm;


    if(!iTimeInit)
        {
        iTimeInit=1;
        time(0);
        }


    sTimeTm.tm_wday  =  0;
    sTimeTm.tm_yday  =  0;
    sTimeTm.tm_year  =  pData->usYear-1900;
    sTimeTm.tm_mon   =  pData->ucMonth-1;
    sTimeTm.tm_mday  =  pData->ucDay;
    sTimeTm.tm_hour  =  pData->ucHour;
    sTimeTm.tm_min   =  pData->ucMinute;
    sTimeTm.tm_sec   =  pData->ucSecond;

    switch(uMode)
        {
    case 0:     sTimeTm.tm_isdst =  0;
                iUtc  = mktime(&sTimeTm);
                if(iUtc<=0)return 0xFFFFFFFF;
                iUtc += timezone;
                break;

    case 1:     sTimeTm.tm_isdst =  (pData->ucIsDst)? 1:0;
                iUtc = mktime(&sTimeTm);
                break;

    case 2:     sTimeTm.tm_isdst = -1;
                iUtc = mktime(&sTimeTm);
                break;

    default:    return 0xFFFFFFFF;
        }


    if(iUtc<=0)return 0xFFFFFFFF;


return iUtc;
}

//*****************************************************************************
//*
//*     SysTimeToWeekday
//*
//*****************************************************************************
//  Converts a systime to the weekday
//  uSysTime    : is the systime value
//  Returns the weekday of the systemtime (0..6, sunday=0)
SYS_API SysDW SysTimeToWeekday(SysTime uSysTime)
{
unsigned    uTime;



    if(!iTimeInit)
        {
        iTimeInit=1;
        time(0);
        }


    uTime = uSysTime-timezone;


return ((uTime/(3600*24)+4))%7;
}

//*****************************************************************************
//*
//*     SysGetFullPath
//*
//*****************************************************************************
//  converts a relative path to a full path
//  pPath       : is the relative path
//  pBuffer     : is the buffer for the full path
//  iMax        : is the size of the buffer
//  returns the lenght in bytes of the full path cName.
int SysGetFullPath(const char *pPath,char *pBuffer,int iMax)
{
char    *pStr;
int      iLen,iSlash;
int      iPos=0,iAdd;



    if(iMax<=0)return 0;

    pBuffer[0]=0;
    iLen=strlen(pPath);
    if(iLen+1>=iMax)return 0;

    if(pPath[0]!='/')
        {
        pStr=getcwd(pBuffer,iMax);
        if(pStr==NULL)return 0;
        iPos=strlen(pBuffer);
        if(iPos+1+iLen>=iMax)return 0;
        pBuffer[iPos]='/';
        iPos++;
        memcpy(pBuffer+iPos,pPath,iLen+1);
        iLen+=iPos;
        }
    else{
        memcpy(pBuffer,pPath,iLen+1);
        iPos=1;
        }


    while(iPos<iLen)
        {
        if(iPos>0 && pBuffer[iPos]=='/')                // filter "//"
            {
            memmove(pBuffer+iPos,pBuffer+iPos+1,iLen-iPos);
            iLen-=1;
            continue;
            }
                                                        // filter "./"
        if(iPos>0 && pBuffer[iPos]=='.' && pBuffer[iPos+1]=='/')
            {
            memmove(pBuffer+iPos,pBuffer+iPos+2,iLen-iPos-1);
            iLen-=2;
            continue;
            }
                                                        // filter "."
        if(iPos>1 && pBuffer[iPos]=='.' && pBuffer[iPos+1]==0)
            {
            iPos--;
            pBuffer[iPos]=0;
            iLen=iPos;
            break;
            }

                                                        // filter "../"
        if(iPos>0 && pBuffer[iPos]=='.' && pBuffer[iPos+1]=='.' && pBuffer[iPos+2]=='/')
            {
            iSlash=iPos;

            for(iPos--;iPos>0;iPos--)
                {
                if(pBuffer[iPos-1]=='/')break;
                }

            iAdd = iSlash-iPos+3;

            memmove(pBuffer+iPos,pBuffer+iSlash+3,iLen-iSlash-2);
            iLen-=iAdd;
            continue;
            }
                                                        // filter ".."
        if(iPos>1 && pBuffer[iPos]=='.' && pBuffer[iPos+1]=='.' && pBuffer[iPos+2]==0)
            {
            for(iPos-=2;iPos>0;iPos--)
                {
                if(pBuffer[iPos]=='/')break;
                }

            if(iPos<=0)
                {
                pBuffer[0]='/';
                iPos=1;
                }

            pBuffer[iPos]=0;
            iLen=iPos;
            break;
            }

        for(;pBuffer[iPos];iPos++)                      // to next slash
            {
            if(pBuffer[iPos]!='/')continue;
            iPos++;
            break;
            }
        }



return iLen;
}


//*****************************************************************************
//*
//*     SysOpenEx
//*
//*****************************************************************************
//  Opens a file
//  pFilename   : name of the file
//  uFlags      : acces flags
//                  SYS_READ
//                  SYS_WRITE
//                  SYS_CREATE
//                  SYS_TRUNC
//  pError      : here the error code (below zero or 0 if ok) will be stored.
//                if this pontier is zero no error code will be stored.
//  Return the file handle or 0 if an error occurs
SYS_API SysFile SysOpenEx(const char *pFilename,unsigned uFlags,int *pError)
{
struct flock    sLock;
struct stat     sStat;
char            cTemp[MAX_TEMP_PATH];
int             iHandle;
SysDW           dwAccess;
SysDW           dwMode;
int             iOk;


    pFilename = SysCheckPath(pFilename,cTemp);
    dwAccess  = (uFlags&SYS_WRITE)? O_RDWR:O_RDONLY;
    dwMode    = (uFlags&SYS_WRITE)? S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH:S_IRUSR|S_IRGRP|S_IROTH;

    if(uFlags&SYS_TRUNC  )dwAccess|=O_TRUNC;
    if(uFlags&SYS_CREATE )dwAccess|=O_CREAT;


    #if MAP_DOS_FILE_BITS
    if(uFlags&SYS_ARCHIVE)dwMode  |=S_IXUSR;
    if(uFlags&SYS_SYSTEM )dwMode  |=S_IXGRP;
    if(uFlags&SYS_HIDDEN )dwMode  |=S_IXOTH;
    #endif

       iHandle=open(pFilename,dwAccess,dwMode);
    if(iHandle<0)
        {
        if(pError)*pError=SysError();
        return 0;
        }
    
    if(fstat(iHandle,&sStat)!=-1)
        {
        if(sStat.st_mode&S_IFDIR)
            {
            if(pError)*pError=SYS_ERR_ACCESS_DENIED;
            close(iHandle);
            return 0;
            }
        }


    if(uFlags&SYS_WRITE)
        {
        sLock.l_type   = F_WRLCK;
        sLock.l_whence = SEEK_SET;
        sLock.l_start  = 0;
        sLock.l_len    = 1;

           iOk = fcntl(iHandle,F_SETLK,&sLock);
        if(iOk<0)
            {
            if(errno==EPERM || errno==EAGAIN)
                {
                if(pError)*pError=SYS_ERR_ACCESS_DENIED;
                close(iHandle);
                return 0;
                }
            }
        }
    else{
        sLock.l_type   = F_RDLCK;
        sLock.l_whence = SEEK_SET;
        sLock.l_start  = 0;
        sLock.l_len    = 0;

           iOk = fcntl(iHandle,F_SETLK,&sLock);
        if(iOk<0)
            {
            if(errno==EPERM || errno==EAGAIN)
                {
                if(pError)*pError=SYS_ERR_ACCESS_DENIED;
                close(iHandle);
                return 0;
                }
            }
        }

return SYS_NEW_HANDLE(iHandle,pFilename,pError);
}



//*****************************************************************************
//*
//*     SysClose
//*
//*****************************************************************************
//  close a file
//  return 0 if all is ok, or an error code below zero
SYS_API int SysClose(SysFile hHandle)
{
int     iOk;
int     iFile;


    iFile = SYS_GET_HANDLE(hHandle);
            SYS_DEL_HANDLE(hHandle);
		
       iOk=close(iFile);
    if(iOk<0)
        {
        return SysError();
        }
    
    //if(!write(drop_cache_FD, "1", 1)) printf("write to drop_cache failed!\n");

return 0;
}

//*****************************************************************************
//*
//*     SysFlush
//*
//*****************************************************************************
//  flush the file buffers
//  return 0 if all is ok, or error code below zero
SYS_API int SysFlush(SysFile hHandle)
{
int     iOk;
int     iFile;


    iFile = SYS_GET_HANDLE(hHandle);

       iOk= fsync(iFile);
    if(iOk<0)
        {
        return SysError();
        }

return 0;
}

//*****************************************************************************
//*
//*     SysSeek
//*
//*****************************************************************************
//  seeks the file pointer
//  return the new file position or an error code below zero
SYS_API int SysSeek(SysFile hHandle,SysDW dwOffset,int iType)
{
SysDW   dwPos;
int     iFile;


    iFile = SYS_GET_HANDLE(hHandle);

    dwPos=lseek(iFile,dwOffset,iType);

    if(dwPos==0xFFFFFFFF)
        {
        return SysError();
        }


return dwPos;
}

//*****************************************************************************
//*
//*     SysTell
//*
//*****************************************************************************
//  return the file position or an error code below zero
SYS_API int SysTell(SysFile hHandle)
{
SysDW   dwPos;
int     iFile;


    iFile = SYS_GET_HANDLE(hHandle);
    dwPos=lseek(iFile,0,SEEK_CUR);

    if(dwPos==0xFFFFFFFF)
        {
        return SysError();
        }


return dwPos;
}

//*****************************************************************************
//*
//*     SysRead
//*
//*****************************************************************************
//  read data from a file
//  hHandle : file handle
//  pBuffer : buffer for read
//  dwSize  : byte count
//  return the count of read bytes or an error code below zero
SYS_API int SysRead(SysFile hHandle,void *pBuffer,SysDW dwSize)
{
int     iCount;
int     iFile;


    iFile = SYS_GET_HANDLE(hHandle);
				
       iCount =read(iFile,pBuffer,dwSize);
    if(iCount==-1)
        {
        return SysError();
        }

return iCount;
}

//*****************************************************************************
//*
//*     SysWrite
//*
//*****************************************************************************
//  write data to a file
//  hHandle : file handle
//  pBuffer : buffer to write
//  dwSize  : byte count
//  return the count of read bytes or an error code below zero
SYS_API int SysWrite(SysFile hHandle,const void *pBuffer,SysDW dwSize)
{
int     iCount;
int     iFile;


    iFile = SYS_GET_HANDLE(hHandle);


       iCount =write(iFile,pBuffer,dwSize);
    if(iCount==-1)
        {
        return SysError();
        }


    SYS_SET_ARCHBIT(hHandle,dwSize);


return iCount;
}


//*****************************************************************************
//*
//*     SysGetFileSize
//*
//*****************************************************************************
//  returns the file size or an error code below zero
SYS_API int SysGetFileSize(SysFile hHandle)
{
struct stat sStat;
int     iFile;
int     iOk;


    iFile = SYS_GET_HANDLE(hHandle);

       iOk = fstat(iFile,&sStat);
    if(iOk<0)
        {
        return SysError();
        }


return sStat.st_size;
}

//*****************************************************************************
//*
//*     SysSemaphoreCreate
//*
//*****************************************************************************
//  creates a new semaphore, the has the value 1 after creation and isn't locked
//  returns a semaphore handle
SYS_API SysSemaphore SysSemaphoreCreate()
{
SemData *pSem;


    pSem=(SemData*)malloc(sizeof(SemData));

    pthread_mutex_init(&pSem->hAccess, 0);
    pthread_mutex_init(&pSem->hDummy,  0);
    pthread_cond_init (&pSem->hWait  , 0);


    pSem->iId     = 0x1255AA78;
    pSem->iValue  = 1;
    pSem->iNumber = 0;


return (SysSemaphore)pSem;
}

//*****************************************************************************
//*
//*     SysSemaphoreCreateEx
//*
//*****************************************************************************
//  creates a new semaphore with an init value
//  iNumber : is the init value (>0 isn't locked)
//  returns a semaphore handle
SYS_API SysSemaphore SysSemaphoreCreateEx(int iNumber)
{
SemData *pSem;


        pSem=(SemData*)malloc(sizeof(SemData));
    if(!pSem)return 0;

    pthread_mutex_init(&pSem->hAccess, 0);
    pthread_mutex_init(&pSem->hDummy,  0);
    pthread_cond_init (&pSem->hWait  , 0);


    pSem->iId     = 0x1255AA78;
    pSem->iValue  = iNumber;
    pSem->iNumber = 0;


return (SysSemaphore)pSem;
}

//*****************************************************************************
//*
//*     SysSemaphoreDelete
//*
//*****************************************************************************
//  deletes a semaphore
//  returns 1 if the semaphore was deleted or 0 if an error occurs
SYS_API int SysSemaphoreDelete(SysSemaphore hSemaphore)
{
SemData *pSem;


    if(!hSemaphore)return 0;

       pSem=(SemData*)hSemaphore;
    if(pSem->iId!=0x1255AA78)return 0;

    pSem->iId     = 0;
    pSem->iValue  = 0x40000000;
    pSem->iNumber = 0;

    pthread_cond_signal  (&pSem->hWait);
    pthread_mutex_unlock (&pSem->hAccess);
    pthread_mutex_unlock (&pSem->hDummy);
    pthread_cond_destroy (&pSem->hWait);
    pthread_mutex_destroy(&pSem->hAccess);
    pthread_mutex_destroy(&pSem->hDummy);

    free(pSem);


return 1;
}


//*****************************************************************************
//*
//*     SysSemaphoreUnlock
//*
//*****************************************************************************
//  unlocks a sempahore, or increases the semaphore value if it was unlocked
//  returns 1 or 0 if ann error occurs
SYS_API int SysSemaphoreUnlock(SysSemaphore hSemaphore)
{
SemData *pSem;


    if(!hSemaphore)return 0;

       pSem=(SemData*)hSemaphore;
    if(pSem->iId!=0x1255AA78)return 0;

    pthread_mutex_lock(&pSem->hAccess);

       pSem->iValue++;
    if(pSem->iNumber>0)
        {
        pthread_cond_signal (&pSem->hWait);
        pthread_mutex_unlock(&pSem->hDummy);
        }

    pthread_mutex_unlock(&pSem->hAccess);


return 1;
}


//*****************************************************************************
//*
//*     SysSemaphoreLock
//*
//*****************************************************************************
//  waits until a sempahore is unlocket, or decreases the semaphore value
//  if it was unlocked
//  returns 1 or 0 if ann error occurs
SYS_API int SysSemaphoreLock(SysSemaphore hSemaphore)
{
SemData        *pSem;



    if(!hSemaphore)return 0;

    pSem=(SemData*)hSemaphore;
    if(pSem->iId!=0x1255AA78)return 0;

    pthread_mutex_lock(&pSem->hAccess);

       pSem->iValue--;
    if(pSem->iValue<0)
        {
        pthread_mutex_unlock(&pSem->hAccess);

        pSem->iNumber++;
        pthread_cond_wait(&pSem->hWait,&pSem->hDummy);

        if(pSem->iId!=0x1255AA78)return 0;

        pthread_mutex_lock  (&pSem->hAccess);
        pthread_mutex_unlock(&pSem->hDummy);
        pSem->iNumber--;
        pthread_mutex_unlock(&pSem->hAccess);

        return 1;
        }

    pthread_mutex_unlock(&pSem->hAccess);



return 1;
}


//*****************************************************************************
//*
//*     SysSemaphoreLockWait
//*
//*****************************************************************************
//  waits some time until a sempahore is unlocket, or decreases the semaphore
//  value if it was unlocked.
//  dwMillisec  : is the maximum wait time in milliseconds
//  returns     0: Timeout
//              1: Unlock
SYS_API int SysSemaphoreLockWait(SysSemaphore hSemaphore,SysDW dwMillisec)
{
SemData            *pSem;
struct timeval      sTv;
struct timespec     sTi;
int                 iRet;



    if(!hSemaphore)return 1;

       pSem=(SemData*)hSemaphore;
    if(pSem->iId!=0x1255AA78)return 1;

    pthread_mutex_lock(&pSem->hAccess);

       pSem->iValue--;
    if(pSem->iValue<0)
        {
        pSem->iNumber++;
        pthread_mutex_unlock(&pSem->hAccess);

        gettimeofday(&sTv, 0);

        sTi.tv_nsec  =(sTv.tv_usec + (dwMillisec % 1000) * 1000) * 1000;
        sTi.tv_sec   = sTv.tv_sec  + (dwMillisec / 1000) + (sTi.tv_nsec / 1000000000);
        sTi.tv_nsec %= 1000000000;

        iRet = pthread_cond_timedwait(&pSem->hWait,&pSem->hDummy,&sTi);

        if(pSem->iId!=0x1255AA78)return 0;

        if(iRet==ETIMEDOUT)
            {
            pthread_mutex_lock  (&pSem->hAccess);
            pthread_mutex_unlock(&pSem->hDummy);
            pSem->iValue++;
            pSem->iNumber--;
            pthread_mutex_unlock(&pSem->hAccess);
            return 0;
            }

        pthread_mutex_lock  (&pSem->hAccess);
        pthread_mutex_unlock(&pSem->hDummy);
        pSem->iNumber--;
        pthread_mutex_unlock(&pSem->hAccess);

        return 1;
        }

    pthread_mutex_unlock(&pSem->hAccess);


return 1;
}

//*****************************************************************************
//*
//*     SysSleep
//*
//*****************************************************************************
//  suspents the current thread for some time
//  dwMillisec      : is the time in millisekonds
SYS_API void SysSleep(SysDW dwMillisec)
{
struct timeval      sTv;
struct timespec     sTi;
pthread_mutex_t     tMtx;
pthread_cond_t      tCnd;


    gettimeofday(&sTv, 0);


    sTi.tv_nsec  =(sTv.tv_usec + (dwMillisec % 1000) * 1000) * 1000;
    sTi.tv_sec   = sTv.tv_sec  + (dwMillisec / 1000) + (sTi.tv_nsec / 1000000000);
    sTi.tv_nsec %= 1000000000;



    pthread_mutex_init    (&tMtx,0);
    pthread_cond_init     (&tCnd,0);

    pthread_mutex_lock    (&tMtx);
    pthread_cond_timedwait(&tCnd,&tMtx,&sTi);
    pthread_mutex_unlock  (&tMtx);

    pthread_cond_destroy  (&tCnd);
    pthread_mutex_destroy (&tMtx);

}


//*****************************************************************************
//*
//*     SysThreadExitWait
//*
//*****************************************************************************
//  waits until all threads are are closed
static void SysThreadExitWait(void)
{
int     iCount;


    for(;;)
        {
        HANDLE_LOCK();
        iCount=iThreadCount;
        HANDLE_UNLOCK();
        if(iCount<=0)break;
        sleep(1);
        }

}

//*****************************************************************************
//*
//*     SysThreadBegin
//*
//*****************************************************************************
//  local dummy function for thread starting
static void *SysThreadBegin(void *pParam)
{
ThreadData  sThread;



    memcpy(&sThread,pParam,sizeof(sThread));
    free(pParam);


    sThread.pStartProc(sThread.pStartArg);


    HANDLE_LOCK();
    iThreadCount--;
    HANDLE_UNLOCK();



return NULL;
}

//*****************************************************************************
//*
//*     SysThreadStart
//*
//*****************************************************************************
//  start a new thread
//  pStartProc  : function fpr the thread
//  dwStackSize : count of bytes for stack
//  pParam      : Parameter for pStartProc
//  iPriority   : -256 to 256 relative to the current priority or
//                    SYS_GLOBAL|iPriority for a global priority
//  pName       : taskname
//  returns the thread handle or 0 if an error occurs
SYS_API SysThread SysThreadStart(void( _cdecl *pStartProc)(void*),SysDW dwStackSize,void *pParam,int iPriority,const char *pName)
{
ThreadData     *pThread;
pthread_t       iThreadId;
pthread_attr_t  iAttr;
static int      iIsInit=0;



    if(!dwStackSize)return 0;

        pThread = (ThreadData*)malloc(sizeof(ThreadData));
    if(!pThread)return 0;

    if(!iIsInit)
        {
        iIsInit=1;
        HANDLE_INIT();
        atexit(SysThreadExitWait);
        }


    HANDLE_LOCK();
    iThreadCount++;
    HANDLE_UNLOCK();



    pThread->iPriority  = iPriority;
    pThread->pStartProc = pStartProc;
    pThread->pStartArg  = pParam;



        pthread_attr_init(&iAttr);
        pthread_attr_setstacksize(&iAttr,dwStackSize);
    if( pthread_create(&iThreadId,&iAttr,SysThreadBegin,pThread))
        {
        free(pThread);
        HANDLE_LOCK();
        iThreadCount--;
        HANDLE_UNLOCK();
        return 0;
        }


    pthread_detach(iThreadId);



return (SysThread)(iThreadId+1);
}


//*****************************************************************************
//*
//*     SysGetAttributes
//*
//*****************************************************************************
//  get the the attribute from a file
//  pFilename   : pointer to file name
//  return ns the attriputes or an error number below zero
//              	SYS_READ
//              	SYS_WRITE
//              	SYS_DIR
//              	SYS_HIDDEN
//              	SYS_ARCHIVE
SYS_API int SysGetAttributes(const char *pFilename)
{
char        cTemp[MAX_TEMP_PATH];
struct stat sStat;
int         iRet;
int         iOk;



    pFilename = SysCheckPath(pFilename,cTemp);


       iOk = stat(pFilename,&sStat);
    if(iOk<0)
        {
        return SysError();
        }


    iRet=(sStat.st_mode&S_IWUSR)? SYS_READ|SYS_WRITE:SYS_READ;

    if(sStat.st_mode&S_IFDIR)iRet|=SYS_DIR;
	if(sStat.st_mode&S_IXUSR)iRet|=SYS_EXEC;

    #if MAP_DOS_FILE_BITS
    else{
        if(sStat.st_mode&S_IXOTH)iRet|=SYS_HIDDEN;
        if(sStat.st_mode&S_IXGRP)iRet|=SYS_SYSTEM;
        if(sStat.st_mode&S_IXUSR)iRet|=SYS_ARCHIVE;
        }
    #endif



return iRet;
}


//*****************************************************************************
//*
//*     SysSetAttributes
//*
//*****************************************************************************
//  set the the attribute for a file
//  pFilename   : pointer to file name
// 	dwAttr		: are the new attributes
//              	SYS_READ
//              	SYS_WRITE
//              	SYS_HIDDEN
//              	SYS_ARCHIVE
//					SYS_DIR
//  returns null if ok or an error number below zero
//					SYS_ERR_??? (a value <0)
SYS_API int SysSetAttributes(const char *pFilename,SysDW dwAttr)
{
char        cTemp[MAX_TEMP_PATH];
int         iMode;
int         iOk;


    pFilename = SysCheckPath(pFilename,cTemp);

    if(dwAttr&(SYS_VOLUME|SYS_DIR))
        {
        return SYS_ERR_PARAM;
        }


    iMode = (dwAttr&SYS_WRITE)? S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH:S_IRUSR|S_IRGRP|S_IROTH;

	if(dwAttr&SYS_EXEC   )iMode|=S_IXUSR;

    #if MAP_DOS_FILE_BITS
    if(dwAttr&SYS_ARCHIVE)iMode|=S_IXUSR;
    if(dwAttr&SYS_SYSTEM )iMode|=S_IXGRP;
    if(dwAttr&SYS_HIDDEN )iMode|=S_IXOTH;
    #endif


       iOk = chmod(pFilename,iMode);
    if(iOk<0)
        {
        return SysError();
        }



return 0;
}



//*****************************************************************************
//*
//*     SysFindFirstEx
//*
//*****************************************************************************
//  search for a file
//  pFilename   : find path (with '*' and '?' )
//  pFind       : pointer to file structer
//  pError      : here the error code (below zero or 0 if ok) will be stored.
//                if this pontier is zero no error code will be stored.
//  returns a find handle or 0 if an error occurs
SYS_API SysFind SysFindFirstEx(const char *pFilename,SysFindStruct *pFind,int *pError)
{
DIR        *pDir;
char        cTemp[MAX_TEMP_PATH];
char        cPath[FIND_LEN+4];
SysFind     hHandle;
struct stat sStat;
unsigned    uNr;
int         iFlag;
int         iAttr;
int         iLen;
int         i;




    if(!pFind)
        {
        if(pError)*pError=SYS_ERR_PARAM;
        return 0;
        }

    if(!iFindInit)                                  // init all
       {
       memset(aFindMem,0,sizeof(aFindMem));
       iFindInit=1;
       HANDLE_INIT();
       }

    HANDLE_LOCK();

    for(uNr=0;uNr<FIND_MAX;uNr++)                   // find empty handle
        {
        if(aFindMem[uNr].iFree==0)break;
        }

    if(uNr>=FIND_MAX)
        {
        HANDLE_UNLOCK();
        SysClearFindStruct(pFind);
        if(pError)*pError=SYS_ERR_NO_HANDLES;
        return 0;
        }

    aFindMem[uNr].iFree=1;

    HANDLE_UNLOCK();


//*****************************************************************************

    iLen=strlen(pFilename);

    for(i=iLen,iFlag=0;i>0;i--)
        {
        if(pFilename[i]=='/' )break;
        if(pFilename[i]=='\\')break;
        if(pFilename[i]!='*' )
        if(pFilename[i]!='?' )continue;
        iFlag=1;
        }


    if(!iFlag)                                      // no wildchars
        {
        pFilename = SysCheckPath(pFilename,cTemp);

        iLen=strlen(pFilename);

        for(i=iLen,iFlag=0;i>0;i--)
            {
            if(pFilename[i]=='/')break;
            if(pFilename[i]!='*')
            if(pFilename[i]!='?')continue;
            iFlag=1;
            }

        aFindMem[uNr].iFree=0;

        if(iLen-i>=(int)sizeof(pFind->cName))       // is the path size too long
            {
            SysClearFindStruct(pFind);
            if(pError)*pError=SYS_ERR_INVALID_FILENAME;
            return 0;
            }

        if(stat(pFilename,&sStat)==-1)              // exists the file
            {
            SysClearFindStruct(pFind);
            if(pError)*pError=SYS_ERR_FILE_NOT_FOUND;
            return 0;
            }

        if(pFilename[i]=='/')i++;
        strcpy(pFind->cName,pFilename+i);
        #if SYS_HAS_SHORT_NAME
        SysCopyShortName(pFind->cShortName,pFind->cName);
        #endif

        pFind->dwSize       = (SysDW)sStat.st_size;
        pFind->dwTimeWrite  = (SysDW)sStat.st_mtime;
        pFind->dwTimeCreate = (SysDW)sStat.st_ctime;
        pFind->dwTimeAccess = (SysDW)sStat.st_atime;

		iAttr = (sStat.st_size>>(32-SYS_HIGHSHIFT))&SYS_HIGHMASK;

        if(sStat.st_mode&S_IWUSR)iAttr|=SYS_WRITE;
        if(sStat.st_mode&S_IRUSR)iAttr|=SYS_READ;
        if(sStat.st_mode&S_IFDIR)iAttr|=SYS_DIR;

        #if MAP_DOS_FILE_BITS
        else{
        if(sStat.st_mode&S_IXUSR)iAttr|=SYS_ARCHIVE;
        if(sStat.st_mode&S_IXGRP)iAttr|=SYS_SYSTEM;
        if(sStat.st_mode&S_IXOTH)iAttr|=SYS_HIDDEN;}
        #endif

        pFind->dwAttr=iAttr;

        return (SysFind)FIND_ONE;
        }


    if(i>FIND_LEN)                                  // is the path too long ?
        {
        SysClearFindStruct(pFind);
        aFindMem[uNr].iFree=0;
        if(pError)*pError=SYS_ERR_INVALID_FILENAME;
        return 0;
        }


    memcpy(cPath,pFilename,i);
    cPath[i]   = 0;
    iLen      -= i+1;
    pFilename += i+1;

    SysCheckPath(cPath,cTemp);                      // new path is in cTemp
    i = strlen(cTemp);

    if(i+iLen+2>=(int)sizeof(cTemp))
        {
        SysClearFindStruct(pFind);
        aFindMem[uNr].iFree=0;
        if(pError)*pError=SYS_ERR_INVALID_FILENAME;
        return 0;
        }

    memcpy(cTemp+i+1,pFilename,iLen+1);
    pFilename = cTemp;
    cTemp[i]  = '/';
    iLen     += i+1;


    memcpy(aFindMem[uNr].cName,pFilename,i);
           aFindMem[uNr].cName[i]=0;
           aFindMem[uNr].iLen =i;


    if(iLen-i>FIND_LEN)                             // is the name too long ?
        {
        SysClearFindStruct(pFind);
        aFindMem[uNr].iFree=0;
        if(pError)*pError=SYS_ERR_INVALID_FILENAME;
        return 0;
        }


    if(pFilename[i]=='/')i++;

    memcpy(aFindMem[uNr].cFind,pFilename+i,iLen-i+1);
           aFindMem[uNr].iNum = iLen-i;


    if(!aFindMem[uNr].cName[0])                     // "" to "."
        {
        aFindMem[uNr].cName[0] = '.';
        aFindMem[uNr].cName[1] = 0;
        aFindMem[uNr].iLen     = 1;
        }

       pDir=opendir(aFindMem[uNr].cName);           // open directory entries
    if(pDir==0)
        {
        SysClearFindStruct(pFind);
        aFindMem[uNr].iFree=0;
        if(pError)*pError=SYS_ERR_PATH_NOTFOUND;
        return 0;
        }

  i=aFindMem[uNr].iLen;
    aFindMem[uNr].cName[i]='/';i++;
    aFindMem[uNr].cName[i]=0;
    aFindMem[uNr].iLen++;
    aFindMem[uNr].pDir=pDir;

    hHandle = (SysFind)(uNr+1);

    if(SysFindNext(hHandle,pFind))                  // seach to first filename
        {
        SysFindClose(hHandle);
        if(pError)*pError=SYS_ERR_FILE_NOT_FOUND;
        return 0;
        }


    if(pError)*pError=SYS_ERR_NONE;


return hHandle;
}


//*****************************************************************************
//*
//*     SysFindNext
//*
//*****************************************************************************
//  search for the next file
//  hHandle : find handle
//  pFind   : pointer to file structer
//  returns 0 if a file was found or an error code below zero
SYS_API int SysFindNext(SysFind hHandle,SysFindStruct *pFind)
{
unsigned        uNr;
int             iLen;
int             iNum;
int             iAttr;
char           *pStr;
DIR            *pDir;
struct dirent  *pResult;
struct dirent   sEntry;
struct stat     sStat;



	if(!pFind)
		{
		return SYS_ERR_PARAM;
		}

       uNr = (int)hHandle-1;
    if(uNr>=FIND_MAX || aFindMem[uNr].iFree==0)
        {
        SysClearFindStruct(pFind);
        return (uNr==FIND_ONE)? SYS_ERR_FILE_NOT_FOUND:SYS_ERR_PARAM;
        }

    pDir=aFindMem[uNr].pDir;
    iLen=aFindMem[uNr].iLen;
    pStr=aFindMem[uNr].cName;

    for(;;)
        {
        readdir_r(pDir,&sEntry,&pResult);           // get next directory entry

        if(!pResult)
            {
            SysClearFindStruct(pFind);
            return SYS_ERR_FILE_NOT_FOUND;
            }

        iNum=strlen(sEntry.d_name);
        if(iNum>=(int)sizeof(pFind->cName))continue;
        if(strnxcmp(sEntry.d_name,iNum,aFindMem[uNr].cFind,aFindMem[uNr].iNum))continue;
        if(iLen+iNum>FIND_LEN)continue;

        memcpy(pStr+iLen,sEntry.d_name,iNum+1);
        if(stat(pStr,&sStat)==-1)continue;

        strcpy(pFind->cName,sEntry.d_name);
        #if SYS_HAS_SHORT_NAME
        SysCopyShortName(pFind->cShortName,pFind->cName);
        #endif

        pFind->dwSize       = (SysDW)sStat.st_size;
        pFind->dwTimeWrite  = (SysDW)sStat.st_mtime;
        pFind->dwTimeCreate = (SysDW)sStat.st_ctime;
        pFind->dwTimeAccess = (SysDW)sStat.st_atime;

		iAttr = (sStat.st_size>>(32-SYS_HIGHSHIFT))&SYS_HIGHMASK;

        if(sStat.st_mode&S_IWUSR)iAttr|=SYS_WRITE;
        if(sStat.st_mode&S_IRUSR)iAttr|=SYS_READ;
        if(sStat.st_mode&S_IFDIR)iAttr|=SYS_DIR;

        #if MAP_DOS_FILE_BITS
        else{
        if(sStat.st_mode&S_IXUSR)iAttr|=SYS_ARCHIVE;
        if(sStat.st_mode&S_IXGRP)iAttr|=SYS_SYSTEM;
        if(sStat.st_mode&S_IXOTH)iAttr|=SYS_HIDDEN;}
        #endif

        pFind->dwAttr=iAttr;

        break;
        }


return 0;
}


//*****************************************************************************
//*
//*     SysFindClose
//*
//*****************************************************************************
SYS_API int SysFindClose(SysFind hHandle)
{
unsigned    uNr;


    uNr = (int)hHandle-1;

    if(uNr>=FIND_MAX)
        {
        return (uNr==FIND_ONE)? SYS_ERR_NONE:SYS_ERR_PARAM;
        }

    if(aFindMem[uNr].iFree==0)
        {
        return SYS_ERR_PARAM;
        }


    closedir(aFindMem[uNr].pDir);
    aFindMem[uNr].iFree=0;


return 0;
}




//*****************************************************************************
//*
//*     SysGetFileInfo
//*
//*****************************************************************************
//  search for the next file
//  hHandle : find handle
//  pFind   : pointer to file structer
//  returns 0 if a file was found or an error code below zero
SYS_API int SysGetFileInfo(SysFile hHandle,SysFindStruct *pFind)
{
int             iAttr;
int             iFile;
struct stat     sStat;


    iFile = SYS_GET_HANDLE(hHandle);

    if(fstat(iFile,&sStat)==-1)
        {
        SysClearFindStruct(pFind);
        return SysError();
        }


    pFind->dwSize       = (SysDW)sStat.st_size;
    pFind->dwTimeWrite  = (SysDW)sStat.st_mtime;
    pFind->dwTimeCreate = (SysDW)sStat.st_ctime;
    pFind->dwTimeAccess = (SysDW)sStat.st_atime;

    iAttr = (sStat.st_size>>(32-SYS_HIGHSHIFT))&SYS_HIGHMASK;

    if(sStat.st_mode&S_IWUSR)iAttr|=SYS_WRITE;
    if(sStat.st_mode&S_IRUSR)iAttr|=SYS_READ;
    if(sStat.st_mode&S_IFDIR)iAttr|=SYS_DIR;

    #if MAP_DOS_FILE_BITS
    if(sStat.st_mode&S_IXUSR)iAttr|=SYS_ARCHIVE;
    if(sStat.st_mode&S_IXGRP)iAttr|=SYS_SYSTEM;
    if(sStat.st_mode&S_IXOTH)iAttr|=SYS_HIDDEN;
    #endif

    pFind->dwAttr=iAttr;

    #if SYS_HAS_NAME_INFO
    strncpy(pFind->cName,SYS_GET_NAME(hHandle),sizeof(pFind->cName));
    #else
    pFind->cName[0] = 0;
    #endif

    #if SYS_HAS_SHORT_NAME
    SysCopyShortName(pFind->cShortName,pFind->cName);
    #endif



return 0;
}



//*****************************************************************************
//*
//*     SysMkdir
//*
//*****************************************************************************
//  create a new directory
//  return 0 if all ok
SYS_API int SysMkdir(const char *pFilename)
{
char        cTemp[MAX_TEMP_PATH];


    pFilename = SysCheckPath(pFilename,cTemp);

    if(mkdir(pFilename,0777)!=-1)return 0;


return SysError();
}


//*****************************************************************************
//*
//*     SysRmdir
//*
//*****************************************************************************
//  remove a new directory
//  return 0 if all ok
SYS_API int SysRmdir(const char *pFilename)
{
char        cTemp[MAX_TEMP_PATH];


    pFilename = SysCheckPath(pFilename,cTemp);

    if(rmdir(pFilename)!=-1)return 0;
    if(errno==ENOENT)return SYS_ERR_PATH_NOTFOUND;


return SysError();
}


//*****************************************************************************
//*
//*     SysDelete
//*
//*****************************************************************************
//  delets a file
//  return 0 if all ok
SYS_API int SysDelete(const char *pFilename)
{
char        cTemp[MAX_TEMP_PATH];


    pFilename = SysCheckPath(pFilename,cTemp);

    if(remove(pFilename)!=-1)return 0;


return SysError();
}


//*****************************************************************************
//*
//*     SysMove
//*
//*****************************************************************************
//  moves or renames a file
SYS_API int SysMove(const char *pFilename,const char *pNewFilename)
{
char        cTemp1[MAX_TEMP_PATH];
char        cTemp2[MAX_TEMP_PATH];


    pFilename     = SysCheckPath(pFilename   ,cTemp1);
    pNewFilename  = SysCheckPath(pNewFilename,cTemp2);

    if(rename(pFilename,pNewFilename)==0)return 0;


return SysError();
}


//*****************************************************************************
//*
//*     SysGetFileTime
//*
//*****************************************************************************
//  get the all times of a opened file
SYS_API int SysGetFileTime(SysFile hHandle,SysDW *pWrite,SysDW *pCreate,SysDW *pAccess)
{
struct stat sStat;
int     iFile;
int     iOk;


    iFile = SYS_GET_HANDLE(hHandle);

       iOk = fstat(iFile,&sStat);
    if(iOk<0)
        {
        return SysError();
        }


    *pWrite  = sStat.st_mtime;
    *pCreate = sStat.st_ctime;
    *pAccess = sStat.st_atime;


return 0;
}


//*****************************************************************************
//*
//*     SysSetFileTime
//*
//*****************************************************************************
//  get the all times of an opened file
SYS_API int SysSetFileTime(SysFile hHandle,SysDW dwWrite,SysDW dwCreate,SysDW dwAccess)
{
timeval         sTv[2];
struct stat     sStat;
int             iFile;
int             iOk;
struct timespec ssTv[2];

    iFile = SYS_GET_HANDLE(hHandle);


    sTv[0].tv_sec   = dwAccess;
    sTv[1].tv_sec   = dwWrite;

    if(!dwWrite || !dwAccess)                           // Alter werte Laden
        {
           iOk = fstat(iFile,&sStat);
        if(iOk<0)
            {
            return SysError();
            }

        if(!dwAccess)sTv[0].tv_sec  = sStat.st_atime;
        if(!dwWrite )sTv[1].tv_sec  = sStat.st_mtime;
        }


    sTv[0].tv_usec  = 0;
    sTv[1].tv_usec  = 0;
				
				TIMEVAL_TO_TIMESPEC(sTv, ssTv);
				iOk = futimens(iFile, ssTv);
       //iOk = futimes(iFile,sTv);
    if(iOk<0)
        {
        return SysError();
        }


return 0;
}



//*****************************************************************************
//*
//*     SysGetVolumeInfo
//*
//*****************************************************************************
//  get info from a disk
//  pFilename   : name of the disk eg. "C:\"
//  pInfo       : pointer to the info stucture
//  iCalcFree   : should the free space be calculated
//  return 0 or an error code below zero
SYS_API int SysGetVolumeInfo(const char *pFilename,SysVolumeInfo *pInfo,int iCalcFree)
{
char            cTemp[MAX_TEMP_PATH];
struct statfs   sStatFs;
struct statvfs  sStat;
const char     *pType;
int             iOk;



    if(!pInfo)
        {
        return SYS_ERR_PARAM;
        }


    pFilename = SysCheckPath(pFilename,cTemp);


       iOk = statvfs(pFilename,&sStat);
    if(iOk<0)
        {
        memset(pInfo,0,sizeof(pInfo->cName));
        return SysError();
        }

    strcpy(pInfo->cFsType,"???");
    strcpy(pInfo->cName  ,"");

    pInfo->dwBytesPerSector = (sStat.f_bsize>=512)? 512:sStat.f_bsize;
    pInfo->dwSectorsPerUnit = (sStat.f_bsize>=512)? sStat.f_bsize/512:1;
    pInfo->dwTotalUnits     =  sStat.f_blocks;
    pInfo->dwFreeUnits      =  sStat.f_bfree;
    pInfo->dwSerialNumber   =  sStat.f_fsid;

    if(!iCalcFree)pInfo->dwFreeUnits=0;


       iOk = statfs(pFilename,&sStatFs);
    if(iOk<0)
        {
        strcpy(pInfo->cFsType,"???");
        strcpy(pInfo->cName  ,"");
        }
    else{
        switch(sStatFs.f_type)
            {
        case 0x137D     : pType="EXT";  break;
        case 0xEF51     : pType="EXT2"; break;
        case 0xEF53     : pType="EXT3"; break;
        case 0x9660     : pType="ISOFS";break;
        case 0x72b6     : pType="JFFS2";break;
        case 0x3153464a : pType="JFS";  break;
        case 0xF995E849 : pType="HPFS"; break;
        case 0x7275     : pType="ROM";  break;
        case 0x6969     : pType="NFS";  break;
        case 0x517B     : pType="SMB";  break;
        case 0x5346544e : pType="NTFS"; break;
        case 0x4d44     : pType="MSDOS";break;
        default:          pType="???";  break;
            }

        strcpy(pInfo->cFsType,pType);
        strcpy(pInfo->cName  ,"");
        }


return 0;
}



//*****************************************************************************
//*
//*     SysSetFileSize
//*
//*****************************************************************************
SYS_API int SysSetFileSize(SysFile hHandle,SysDW dwSize)
{
int     iFile;
int     iOk;


    iFile = SYS_GET_HANDLE(hHandle);

       iOk=ftruncate(iFile,dwSize);
    if(iOk<0)return SysError();


return 0;
}


//*****************************************************************************
//*
//*     SysTickInit
//*
//*****************************************************************************
static inline void SysTickInit()
{

    uJiffMul  = (0x100000*1000)/sysconf(_SC_CLK_TCK);
    uJiffLast = (unsigned)times(0);
    uJiffInit = 1;

}

//*****************************************************************************
//*
//*     SysTickCount
//*
//*****************************************************************************
SYS_API SysDW SysTickCount()
{
unsigned long long  uVal64;                                 // Linux Code mit JIFFES-Counter
unsigned            uJiffes;


    uJiffes = (unsigned)times(0);

    if(uJiffes<uJiffLast)
        {
        if(!uJiffInit)SysTickInit();
        uJiffAdd += 0x100000000ull;
        }

    uJiffLast   = uJiffes;
    uVal64      = uJiffes;
    uVal64     += uJiffAdd;
    uVal64     *= uJiffMul;

return (unsigned)(uVal64>>20);
}

//******************************************************************************
//*
//*		HeapCreate
//*
//******************************************************************************
void *HeapCreate(unsigned dwOptions,unsigned dwInitialSize,unsigned dwMaximumSize)
{
return (void*)0x22557733;
}

//******************************************************************************
//*
//*		HeapDestroy
//*
//******************************************************************************
int HeapDestroy(void *hHeap)
{
return (((unsigned)hHeap)==0x22557733);
}

//*****************************************************************************
//*
//*      HeapAlloc
//*
//*****************************************************************************
void *HeapAlloc(void *hHeap,SysDW dwFlags,SysDW dwBytes)
{
return malloc(dwBytes);
}

//*****************************************************************************
//*
//*      HeapReAlloc
//*
//*****************************************************************************
void *HeapReAlloc(void *hHeap,SysDW dwFlags,void *pOldMem,SysDW dwBytes)
{
return realloc(pOldMem,dwBytes);
}

//*****************************************************************************
//*
//*     HeapFree
//*
//*****************************************************************************
int HeapFree(void *hHeap,SysDW dwFlags,void *pMem)
{
    free(pMem);

return 1;
}

//*****************************************************************************
//*
//*      HeapAllocD
//*
//*****************************************************************************
#ifdef  HEAP_DEBUG
#define      MAX_ALLOCS 0x1000
char         cHeapNames [MAX_ALLOCS][28];
void        *pHeapPoints[MAX_ALLOCS];
unsigned     uHeapSizes [MAX_ALLOCS];
SysSemaphore hHeapSemaphore=0;

int dwHeapCount=0;
int dwHeapSize=0;
int dwHeapMax=0;

void   *HeapAllocD(void *hHeap,SysDW dwFlags,SysDW dwBytes,int iLine,const char *pFile)
{
void    *pMem;
char    *pData;
int      i,iLen;


    if(!hHeapSemaphore)
        {
        hHeapSemaphore=RTKCreateSemaphore(ST_COUNTING,1,NULL);
        }

        pMem=malloc(dwBytes);
    if(!pMem)return 0;

    SysSemaphoreLock(hHeapSemaphore);
    dwHeapCount++;
    for(i=0;i<MAX_ALLOCS;i++)if(!pHeapPoints[i])break;
    if(i<MAX_ALLOCS){pHeapPoints[i]=pMem;}
    SysSemaphoreUnlock(hHeapSemaphore);

    if(i<MAX_ALLOCS)
        {
        uHeapSizes[i]=dwBytes;
        pData=cHeapNames[i];
        SysSemaphoreLock(hHeapSemaphore);
        dwHeapSize+=dwBytes;
        SysSemaphoreUnlock(hHeapSemaphore);

        i=strlen(pFile);
        for(;i>0;i--)
            {
            if(pFile[i]=='\\'){i++;break;}
            if(pFile[i]=='/' ){i++;break;}
            }

           iLen = strlen(pFile+i);
        if(iLen>20)iLen=20;

        memcpy (pData,pFile+i,iLen);
        sprintf(pData+iLen,"(%i)",iLine);
        pData[27]=0;
        }

    if(dwHeapSize>dwHeapMax)dwHeapMax=dwHeapSize;


return pMem;
}


//*****************************************************************************
//*
//*      HeapReAllocD
//*
//*****************************************************************************
void *HeapReAllocD(void *hHeap,SysDW dwFlags,void *pOldMem,SysDW dwBytes,int iLine,const char *pFile)
{
void    *pMem;
char    *pData;
int      i,iLen;


    if(!pOldMem)return 0;

    if(!hHeapSemaphore)
        {
        hHeapSemaphore=SysSemaphoreCreate();
        }

        pMem=realloc(pOldMem,dwBytes);
    if(!pMem)return 0;

    SysSemaphoreLock(hHeapSemaphore);
    for(i=0;i<MAX_ALLOCS;i++)if(pHeapPoints[i]==pOldMem)break;
    if(i<MAX_ALLOCS){pHeapPoints[i]=pMem;}
    SysSemaphoreUnlock(hHeapSemaphore);

    if(i<MAX_ALLOCS)
        {
        SysSemaphoreLock(hHeapSemaphore);
        dwHeapSize-=uHeapSizes[i];
        dwHeapSize+=dwBytes;
        SysSemaphoreUnlock(hHeapSemaphore);
        uHeapSizes[i]=dwBytes;
        pData=cHeapNames[i];

        i=strlen(pFile);
        for(;i>0;i--)
            {
            if(pFile[i]=='\\'){i++;break;}
            if(pFile[i]=='/' ){i++;break;}
            }


           iLen = strlen(pFile+i);
        if(iLen>20)iLen=20;

        memcpy (pData,pFile+i,iLen);
        sprintf(pData+iLen,"(%i)",iLine);
        pData[27]=0;
        }

    if(dwHeapSize>dwHeapMax)dwHeapMax=dwHeapSize;


return pMem;
}

//*****************************************************************************
//*
//*      HeapFreeD
//*
//*****************************************************************************
int HeapFreeD(void *hHeap,SysDW dwFlags,void *pMem,int iLine,const char *pFile)
{
char    *pData=(char*)pMem;
int      i;


    if(!pMem)
        return 0;

    SysSemaphoreLock(hHeapSemaphore);
    dwHeapCount--;
    SysSemaphoreLock(hHeapSemaphore);

    for(i=0;i<MAX_ALLOCS;i++)if(pHeapPoints[i]==pMem)break;
    if(i<MAX_ALLOCS)
        {
        SysSemaphoreLock(hHeapSemaphore);
        dwHeapSize-=uHeapSizes[i];
        SysSemaphoreUnlock(hHeapSemaphore);
        cHeapNames [i][0]=0;
        pHeapPoints[i]=0;
        uHeapSizes [i]=0;
        }


    free(pMem);


return 1;
}

//*****************************************************************************
//*
//*      HeapPrint
//*
//*****************************************************************************
int HeapPrint()
{
char    *pData;
int      i;


    SysSleep(500);

    #ifdef NO_PRINT
    printf("\nHEAP COUNT = %i   size=%i  max=%i",dwHeapCount,dwHeapSize,dwHeapMax);
    #else
    nprintf("\nHEAP COUNT = %i   size=%i  max=%i",dwHeapCount,dwHeapSize,dwHeapMax);
    #endif

    for(i=0;i<MAX_ALLOCS;i++)
        {
        if(!pHeapPoints[i])continue;
        pData=cHeapNames[i];
        #ifdef NO_PRINT
        printf("\n   %08X --> \"%s\" = %i",pHeapPoints[i],pData,uHeapSizes[i]);
        #else
        nprintf("\n   %08X --> \"%s\" = %i",pHeapPoints[i],pData,uHeapSizes[i]);
        #endif
        }

    SysSleep(300);


return dwHeapCount;
}
#endif




