//*****************************************************************************
//*
//*
//*		SocketUtility.cpp
//*
//*
//*****************************************************************************
//
//	This is a UDP/TCP library for
//
//		LINUX
//		WINDOWS
//		RT-TARGET
//
//
//	Copyright  2008	Anton Zechner
//

#include	<stdio.h>
#include	<memory.h>
#include	"SocketUtility.h"

#ifndef			NMB_LOG
#define			NMB_LOG							1
#define			NMB_LOGGING(MES)		time(&timep);fprintf(nmblog, MES, ctime(&timep)); fflush(nmblog)
#define			NMB_LOGARG(MES)			fprintf(nmblog, "(%d)", MES); fflush(nmblog)
						extern FILE		*nmblog;
						extern time_t timep;
#include    <time.h>						
#include 		<errno.h>
#endif

#ifndef		NO_DYNAMIC_MEM
#define 	NO_DYNAMIC_MEM		0
#endif

#ifndef		RECV_TIMEOUT
#define 	RECV_TIMEOUT		0
#endif

#ifndef		SEND_TIMEOUT
#define 	SEND_TIMEOUT		15
#endif

#ifdef		_MSC_VER
#pragma 	warning( disable :4035 )
#pragma 	warning( disable :4996 )
#endif

#define		SUAPI		extern "C"
#define 	LOOPBACK	(*(unsigned*)"\x7F\0\0\1")
#define 	IPV4TOV6	"\0\0\0\0\0\0\0\0\0\0\xFF\xFF\0\0\0\0"
static int	iIpV6Allowed  = SU_USE_IPV6;
static int	iDualStack    = SU_USE_IPV6;
static char cHostName[64] = "";

//*****************************************************************************
//*
//*		WINDOWS
//*
//*****************************************************************************
#if (defined(WIN32) || defined(WIN64)) && !defined(RTTARGET) && !defined(LWIP)

	#pragma		comment(lib,"wsock32.lib")
	#pragma		comment(lib,"ws2_32.lib")
	#include	<winsock2.h>
	#include	<windows.h>
	#include	<stdlib.h>

	#define 	SysSleep					Sleep
	#define 	SysTickCount				GetTickCount
	#define		LEN_TYPE					int*
	#define 	SysDW						unsigned
	#define		SEND_FLAGS					0
	#define		RECV_FLAGS					0

	#define 	Semaphore					HANDLE
	#define 	SemaphoreInit()				CreateSemaphore(0,1,0x70000000,0)
	#define 	SemaphoreExit(h)			CloseHandle(h)
	#define		SemaphoreLock(h)			WaitForSingleObject(h,0xFFFFFFFF)
	#define		SemaphoreUnlock(h)			ReleaseSemaphore(h,1,0)
	#define 	StartTimeout(pParam)		{HANDLE	hHandle;DWORD dwId;hHandle=CreateThread(0,1024,TimeoutProc,pParam,0,&dwId);if(hHandle)CloseHandle(hHandle);}

	static		Semaphore	hGlobalSem =0;

	static DWORD WINAPI TimeoutProc(LPVOID pParam)	// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =GetTickCount();
		unsigned	uDiff;

		pCancel->iRunning=1;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = GetTickCount()-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			Sleep(100);
			}

		pCancel->iRunning=0;

		return 0;
		}


//*****************************************************************************
//*
//*		RTTARGET
//*
//*****************************************************************************
#elif defined(RTTARGET)	&& !defined(LWIP)

	#if			SU_USE_IPV6
	#error		"RTTARGET don't supports IPv6"
	#endif

	#include	<rttarget.h>
	#include	<rtipapi.h>
	#include	<socket.h>
	#include	<clock.h>
	#define 	SysSleep(dwMillisec)		RTKDelay(CLKMilliSecsToTicks(dwMillisec))
	#define 	SysTickCount()				CLKTicksToMilliSecs(RTKGetTime())
	#define		LEN_TYPE					int*
	#define 	SysDW						unsigned
	#define		SEND_FLAGS					0
	#define		RECV_FLAGS					0

	#define 	Semaphore					RTKSemaphore
	#define 	SemaphoreInit()				RTKOpenSemaphore(ST_COUNTING,1,0,0)
	#define 	SemaphoreExit(h)			RTKDeleteSemaphore(&h)
	#define 	SemaphoreLock(h)			RTKWait(h)
	#define 	SemaphoreUnlock(h)			RTKSignal(h)

	#define 	StartTimeout(pParam)		RTKCreateThread(TimeoutProc,RTKGetTaskPrio(RTKCurrentTaskHandle()),1024,0,pParam,"TcpOpen Timeout");

	static		Semaphore	hGlobalSem =0;

	static void _cdecl TimeoutProc(void *pParam)	// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =CLKTicksToMilliSecs(RTKGetTime());
		unsigned	uDiff;

		pCancel->iRunning=1;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = CLKTicksToMilliSecs(RTKGetTime())-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			RTKDelay(CLKMilliSecsToTicks(100));
			}

		pCancel->iRunning=0;
		}


//*****************************************************************************
//*
//*		LWIP mit RTTARGET
//*
//*****************************************************************************
#elif defined(RTTARGET)	&& defined(LWIP)

	#if			SU_USE_IPV6
	#error		"LWIP don't supports IPv6"
	#endif

	#ifndef		INVALID_SOCKET
	#define		INVALID_SOCKET	(~0)
	#endif

	#ifndef		SOCKET_ERROR
	#define		SOCKET_ERROR	(-1)
	#endif

	#define 	LWIP_PROVIDE_NO_ERRNO

	
	#include	<rttarget.h>
	#include	<rtipapi.h>
	#include	<string.h>
	#include	<stdlib.h>
	#include	<clock.h>

	#include	"../src/include/lwip/sockets.h"
	#include	"../src/include/lwip/netif.h"
	#include	"../src/include/lwip/arch.h"
	#include	"../src/include/lwip/tcp.h"

	#define 	SysSleep(dwMillisec)		RTKDelay(CLKMilliSecsToTicks(dwMillisec))
	#define 	SysTickCount()				CLKTicksToMilliSecs(RTKGetTime())
	#define		LEN_TYPE					unsigned*
	#define 	SysDW						unsigned
	#define		SOCKET						int
	#define		SEND_FLAGS					0
	#define		RECV_FLAGS					0
	#define 	gethostbyname(a)			0

	#define 	Semaphore					RTKSemaphore
	#define 	SemaphoreInit()				RTKOpenSemaphore(ST_COUNTING,1,0,0)
	#define 	SemaphoreExit(h)			RTKDeleteSemaphore(&h)
	#define 	SemaphoreLock(h)			RTKWait(h)
	#define 	SemaphoreUnlock(h)			RTKSignal(h)

	#define 	StartTimeout(pParam)		RTKCreateThread(TimeoutProc,RTKGetTaskPrio(RTKCurrentTaskHandle()),1024,0,pParam,"TcpOpen Timeout");

	static		Semaphore					hGlobalSem =0;
	static		Semaphore					hInfoSem   =0;

	static void _cdecl TimeoutProc(void *pParam)	// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =CLKTicksToMilliSecs(RTKGetTime());
		unsigned	uDiff;

		pCancel->iRunning=1;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = CLKTicksToMilliSecs(RTKGetTime())-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			RTKDelay(CLKMilliSecsToTicks(100));
			}

		pCancel->iRunning=0;
		}

	struct hostent 
		{
        char   *h_name;						// official name of host 
        char  **h_aliases;					// alias list 
        short   h_addrtype;             	// host address type 
        short   h_length;               	// length of address 
        char  **h_addr_list;				// list of addresses 
		};

	typedef struct addrinfo_ex
		{
		int                 ai_flags;		// AI_PASSIVE, AI_CANONNAME, AI_NUMERICHOST
		int                 ai_family;      // PF_xxx
		int                 ai_socktype;    // SOCK_xxx
		int                 ai_protocol;    // 0 or IPPROTO_xxx for IPv4 and IPv6
		size_t              ai_addrlen;     // Length of ai_addr
		char			   *ai_canonname;   // Canonical name for nodename
		struct sockaddr    *ai_addr;        // Binary address
		struct addrinfo_ex *ai_next;        // Next structure in linked list
		struct sockaddr     ai_adr;         // Binary address
		}addrinfo_ex_lwip;

	static addrinfo_ex_lwip	aNetInfo[10];

//*****************************************************************************
//*
//*		LWIP mit WINDOWS
//*
//*****************************************************************************
#elif (defined(WIN32) || defined(WIN64)) && defined(LWIP)

	#if			SU_USE_IPV6
	#error		"LWIP don't supports IPv6"
	#endif

	#ifndef		INVALID_SOCKET
	#define		INVALID_SOCKET	(~0)
	#endif

	#ifndef		SOCKET_ERROR
	#define		SOCKET_ERROR	(-1)
	#endif

	#define 	LWIP_PROVIDE_NO_ERRNO

	#include	<string.h>
	#include	<stdlib.h>
	#include	"../src/include/lwip/sockets.h"
	#include	"../src/include/lwip/netif.h"
	#include	"../src/include/lwip/arch.h"
	#include	"../src/include/lwip/tcp.h"

	#define 	SysSleep					Sleep
	#define 	SysTickCount				GetTickCount
	#define		LEN_TYPE					unsigned*
	#define		SOCKET						int
	#define 	WINAPI						__stdcall
	#define 	SysDW						unsigned
	#define 	DWORD						unsigned 
	#define 	HANDLE						void* 
	#define 	LPVOID						void* 
	#define		SEND_FLAGS					0
	#define		RECV_FLAGS					0
	#define 	gethostbyname(a)			0

	#define 	Semaphore					HANDLE
	#define 	SemaphoreInit()				CreateSemaphore(0,1,0x70000000,0)
	#define 	SemaphoreExit(h)			CloseHandle(h)
	#define		SemaphoreLock(h)			WaitForSingleObject(h,0xFFFFFFFF)
	#define		SemaphoreUnlock(h)			ReleaseSemaphore(h,1,0)
	#define 	StartTimeout(pParam)		{HANDLE	hHandle;DWORD dwId;hHandle=CreateThread(0,1024,TimeoutProc,pParam,0,&dwId);if(hHandle)CloseHandle(hHandle);}

	typedef		DWORD								   (__stdcall  *PTHREAD_START_ROUTINE)(LPVOID);
	extern "C"	void            __declspec(dllexport) * __stdcall	CreateSemaphoreA(void*,long,long,const char*);
	extern "C"	unsigned long	__declspec(dllexport)   __stdcall	WaitForSingleObject(void *,unsigned long);
	extern "C"	unsigned long	__declspec(dllexport)   __stdcall	GetTickCount(void);
	extern "C"	int				__declspec(dllexport)   __stdcall	ReleaseSemaphore(void *,long,long*);
	extern "C"	int				__declspec(dllexport)   __stdcall	CloseHandle(void*);
	extern "C"	void            __declspec(dllexport)   __stdcall	Sleep(unsigned long);
	extern "C"	void			__declspec(dllexport) * __stdcall	CreateThread(void*,DWORD,PTHREAD_START_ROUTINE,LPVOID,DWORD,DWORD*);

	
	static		Semaphore					hGlobalSem =0;
	static		Semaphore					hInfoSem   =0;

	static DWORD WINAPI TimeoutProc(LPVOID pParam)	// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =GetTickCount();
		unsigned	uDiff;

		pCancel->iRunning=1;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = GetTickCount()-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			Sleep(100);
			}

		pCancel->iRunning=0;

		return 0;
		}

	struct hostent 
		{
        char   *h_name;						// official name of host 
        char  **h_aliases;					// alias list 
        short   h_addrtype;             	// host address type 
        short   h_length;               	// length of address 
        char  **h_addr_list;				// list of addresses 
		};

	typedef struct addrinfo_ex
		{
		int                 ai_flags;		// AI_PASSIVE, AI_CANONNAME, AI_NUMERICHOST
		int                 ai_family;      // PF_xxx
		int                 ai_socktype;    // SOCK_xxx
		int                 ai_protocol;    // 0 or IPPROTO_xxx for IPv4 and IPv6
		size_t              ai_addrlen;     // Length of ai_addr
		char			   *ai_canonname;   // Canonical name for nodename
		struct sockaddr    *ai_addr;        // Binary address
		struct addrinfo_ex *ai_next;        // Next structure in linked list
		struct sockaddr     ai_adr;         // Binary address
		}addrinfo_ex_lwip;

	static addrinfo_ex_lwip	aNetInfo[10];


//*****************************************************************************
//*
//*		LINUX
//*
//*****************************************************************************
#else

	#include	<unistd.h>
	#include	<sys/time.h>
	#include	<sys/times.h>
	#include	<sys/ioctl.h>
	#include	<sys/socket.h>
	#include	<netinet/tcp.h>
	#include	<arpa/inet.h>
	#include	<semaphore.h>
	#include	<pthread.h>
	#include	<signal.h>
	#include	<malloc.h>
	#include	<netdb.h>
	#include	<sched.h>
	#include	<time.h>

	#define		SOCKET	int

	#ifndef		SOCKET_ERROR
	#define		SOCKET_ERROR	(-1)
	#endif

	#ifndef		INVALID_SOCKET
	#define		INVALID_SOCKET	(~0)
	#endif

	#ifndef 	TCP_NODELAY
	#define		TCP_NODELAY		(1)
	#endif

	#ifndef 	TCP_QUICKACK
	#define		TCP_QUICKACK	(12)
	#endif

	#define 	ioctlsocket 				ioctl
	#define 	closesocket(Socket) 		close(Socket)
	#define		LEN_TYPE					socklen_t*
	#define 	SysDW						unsigned
	#define		SEND_FLAGS					MSG_NOSIGNAL
	#define		RECV_FLAGS					MSG_NOSIGNAL
	typedef		pthread_mutex_t			   *Semaphore;


	extern int	sem_timedwait(sem_t*,const struct timespec*);

	static pthread_mutex_t *SemaphoreInit()
		{
		pthread_mutex_t	*pSem;
		pSem = (pthread_mutex_t*)malloc(sizeof(pthread_mutex_t));
		pthread_mutex_init(pSem,0);
		return pSem;
		}

	static inline void SemaphoreExit(pthread_mutex_t *pSem)
		{
		if(!pSem)return;
		pthread_mutex_destroy(pSem);
		free(pSem);
		}

	static inline void SemaphoreLock(pthread_mutex_t *pSem)
		{
		pthread_mutex_lock(pSem);
		}

	static inline void SemaphoreUnlock(pthread_mutex_t *pSem)
		{
		pthread_mutex_unlock(pSem);
		}

	static void SysSleep(unsigned uMilliSec)
		{
		struct timeval		sTime;
		struct timespec 	sSpec;
		pthread_mutex_t 	sMutex;
		pthread_cond_t  	sCond;


		gettimeofday(&sTime,0);


		sSpec.tv_nsec  =(sTime.tv_usec + (uMilliSec % 1000) * 1000) * 1000;
		sSpec.tv_sec   = sTime.tv_sec  + (uMilliSec / 1000) + (sSpec.tv_nsec / 1000000000);
		sSpec.tv_nsec %= 1000000000;


		pthread_mutex_init    (&sMutex,0);
		pthread_cond_init     (&sCond ,0);

		pthread_mutex_lock    (&sMutex);
		pthread_cond_timedwait(&sCond, &sMutex, &sSpec);
		pthread_mutex_unlock  (&sMutex);

		pthread_cond_destroy  (&sCond );
		pthread_mutex_destroy (&sMutex);
		}

	static unsigned		 	uJiffMul  = 0x100000*10;			// Multiplikator fr Jiffes
	static unsigned 	 	uJiffLast = 0xFFFFFFFF;
	static unsigned 	 	uJiffInit = 0;
	static unsigned long 	uJiffAdd  = 0;
	static Semaphore		hGlobalSem= 0;


	static inline void SysTickInit()
		{
		uJiffMul  = (0x100000*1000)/sysconf(_SC_CLK_TCK);
		uJiffLast = (unsigned)times(0);
		uJiffInit = 1;
		}

	static inline unsigned SysTickCount()
		{
		unsigned long long	uVal64;								// Linux Code mit JIFFES-Counter
		unsigned			uJiffes;

		uJiffes = (unsigned)times(0);

		if(uJiffes<uJiffLast)
			{
			if(!uJiffInit)SysTickInit();
			uJiffAdd += 0x100000000ull;
			}

		uJiffLast	= uJiffes;
		uVal64  	= uJiffes;
		uVal64     += uJiffAdd;
		uVal64     *= uJiffMul;

		return (unsigned)(uVal64>>20);
		}

	static void *TimeoutProc(void *pParam)						// Timeout function for TCP connect
		{
		TcpCancel  *pCancel =(TcpCancel*)pParam;
		unsigned 	uTimeout=pCancel->iTimeout;
		unsigned	uStart  =SysTickCount();
		unsigned	uDiff;

		pCancel->iRunning=1;

		while(pCancel->iValue==0x5A71D34E)
			{
			uDiff = SysTickCount()-uStart;

			if(uDiff>=uTimeout)
				{
				TcpOpenCancel(pCancel);
				break;
				}

			SysSleep(100);
			}

		pCancel->iRunning=0;

		return 0;
		}

	static void StartTimeout(void *pParam)						// Start timeout thread
		{
		pthread_t       iThreadId;
		pthread_attr_t  iAttr;


			pthread_attr_init(&iAttr);
			pthread_attr_setstacksize(&iAttr,1024);
		if(!pthread_create(&iThreadId,&iAttr,TimeoutProc,pParam))
			{
			pthread_detach(iThreadId);
			}
		}


#endif


//*****************************************************************************
//*
//*		Handel definitions
//*
//*****************************************************************************
// REM AZ gegen header tauschen
#if linux

#define sockaddr_in6_		sockaddr_in6

#else

struct in_addr6_
	{
    unsigned char   		s6_addr[16];						// IPv6 address
    };

struct sockaddr_in6_
	{
    short           		sin6_family;						// AF_INET6
    unsigned short  		sin6_port;							// Transport level port number
    unsigned long   		sin6_flowinfo;    					// IPv6 flow information
    struct in_addr6_		sin6_addr;							// IPv6 address
	unsigned long			sin6_scope_id;						// Set of interfaces for a scope.
	};

#endif

typedef	struct
	{
	IpAddr					sIpAddr;
	SOCKET					iSocket;
	unsigned				uNoneBlocking;
	#if SU_USE_IPV6
	union
	{
	struct sockaddr_in		sAddr;
	struct sockaddr_in6_	sAddr6;
	};
	#else
	struct sockaddr_in		sAddr;
	#endif
	#ifdef RTTARGET
	RTKSemaphore			hSemaphor;
	#endif
	}UdpAddr;

typedef	struct
	{
	IpAddr					sIpAddr;
	SOCKET					iSocket;
	unsigned				uNoneBlocking;
	void				   *hServer;
	#if SU_USE_IPV6
	union
	{
	struct sockaddr_in		sAddr;
	struct sockaddr_in6_	sAddr6;
	};
	#else
	struct sockaddr_in		sAddr;
	#endif

	#if (defined(WIN32) || defined(WIN64)) && !defined(RTTARGET) && !defined(LWIP)
	WSAEVENT				hEvent;
	#endif
	}TcpAddr;


//*****************************************************************************
//*
//*		Several Macros
//*
//*****************************************************************************

#if SU_USE_IPV6
#define 	MULTICAST_V6			"\xFF\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01"
#define 	SET_V4_ADDR(p,a)		{ (p)->uAddr[0] =SU_IPV4_ID;  (p)->uAddr[1]=(a);}
#define 	GET_V4_ADDR(p)			(((p)->uAddr[0]==SU_IPV4_ID)? (p)->uAddr[1]:0)
#else
#define 	SET_V4_ADDR(p,a)		{*(p)=(a);}
#define 	GET_V4_ADDR(p)			(*(p))
#endif

#ifndef		IPPROTO_IPV6
#define		IPPROTO_IPV6			41
#endif

#ifndef		IPV6_V6ONLY
#define		IPV6_V6ONLY				27
#endif

#ifndef		AI_PASSIVE
#define		AI_PASSIVE				1
#endif



//*****************************************************************************
//*
//*		Use dynamic or static memory
//*
//*****************************************************************************

#if	NO_DYNAMIC_MEM

#ifndef		SOCKETS_MAX
#define 	SOCKETS_MAX		128
#endif
#define		HANDLE_SIZE		((sizeof(TcpAddr)>sizeof(UdpAddr))? sizeof(TcpAddr):sizeof(UdpAddr))

static	Semaphore		hHandleSemaphore=0;
static	unsigned		uReserveFlags[(SOCKETS_MAX+31)/32];
static	char 			cHandleMemory[ SOCKETS_MAX*HANDLE_SIZE];
extern	void			SocketAllocError(void);

inline void *MallocHandle(int)
{
unsigned 	uBits;
int			i,j;



	if(!hHandleSemaphore)hHandleSemaphore=SemaphoreInit();

	SemaphoreLock(hHandleSemaphore);

	for(i=0;i<(SOCKETS_MAX+31)/32;i++)						// find an empty slot
		{
		   uBits=uReserveFlags[i];
		if(uBits==0xFFFFFFFF)continue;

		for(j=0;j<32;j++,uBits>>=1)
			{
			if(!(uBits&1))break;
			}

		if(i*32+j>=SOCKETS_MAX)break;

		uReserveFlags[i]|=1<<j;

		SemaphoreUnlock(hHandleSemaphore);

		return cHandleMemory+(i*32+j)*HANDLE_SIZE;
		}


	SemaphoreUnlock(hHandleSemaphore);


	#ifndef		NO_ALLOC_ERROR
	SocketAllocError();
	#endif



return 0;
}

inline void FreeHandle(void *pMemory)
{
unsigned uPos;


	uPos  = (char*)pMemory-cHandleMemory;
	uPos /= HANDLE_SIZE;

	if(uPos>=SOCKETS_MAX)return;


	SemaphoreLock(hHandleSemaphore);
	uReserveFlags[uPos/32] &= ~(1<<(uPos&0x1F));
	SemaphoreUnlock(hHandleSemaphore);

}

#else

#include	<malloc.h>

	inline void *MallocHandle(int iSize)  {	return malloc(iSize);}
	inline void  FreeHandle(void *pMemory){	free(pMemory);}

#endif




//*****************************************************************************
//*
//*		InitSocket
//*
//*****************************************************************************
static	int				iSocketInit=0;
#define InitSocket()	if(!iSocketInit)InitSocketProc();
static	TcpHandle		TcpHandleAlloc(SOCKET iSocket,void *pAddr,int iServer,int iNoBlock);
static	UdpHandle		UdpHandleAlloc(SOCKET iSocket,void *pAddr,int iServer,int iNoBlock);


#ifdef LWIP

static void _cdecl ExitSocketProc()
	{
	}

static void InitSocketProc()
	{
	struct netif   *pInterface;
	unsigned		uPos;

	iSocketInit = 1;
	
	if(!hGlobalSem)hGlobalSem  = SemaphoreInit();
	if(!hInfoSem  )hInfoSem    = SemaphoreInit();

	#if LWIP_NETIF_HOSTNAME

	for(pInterface=netif_list;pInterface;pInterface=pInterface->next)
		{
		if(! pInterface->hostname)continue;
		if(!*pInterface->hostname)continue;
		
		IpSetHostname(pInterface->hostname);
		break;			
		}

	#endif 

	aNetInfo[0].ai_flags			= 0x04;
	aNetInfo[0].ai_family			= 2;
	aNetInfo[0].ai_socktype			= 0;
	aNetInfo[0].ai_protocol			= 0;
	aNetInfo[0].ai_addrlen			= 16;
	aNetInfo[0].ai_canonname		= 0;
	aNetInfo[0].ai_addr				= &aNetInfo[0].ai_adr;
	aNetInfo[0].ai_next				= 0;
	aNetInfo[0].ai_adr.sa_len		= 16;
	aNetInfo[0].ai_adr.sa_family	= 2;	
			 
	aNetInfo[1].ai_flags			= 0x04;
	aNetInfo[1].ai_family			= 2;
	aNetInfo[1].ai_socktype			= 0;
	aNetInfo[1].ai_protocol			= 0;
	aNetInfo[1].ai_addrlen			= 16;
	aNetInfo[1].ai_canonname		= 0;
	aNetInfo[1].ai_addr				= &aNetInfo[1].ai_adr;
	aNetInfo[1].ai_next				= 0;
	aNetInfo[1].ai_adr.sa_len		= 16;
	aNetInfo[1].ai_adr.sa_family	= 2;	
	aNetInfo[1].ai_adr.sa_data[2]	= 127;
	aNetInfo[1].ai_adr.sa_data[3]	= 0;
	aNetInfo[1].ai_adr.sa_data[4]	= 0;
	aNetInfo[1].ai_adr.sa_data[5]	= 1;

	uPos = 2;

	for(pInterface=netif_list;pInterface;pInterface=pInterface->next)
		{
		if(!pInterface->ip_addr.addr)continue;
	
		*(unsigned*)(aNetInfo[uPos].ai_adr.sa_data+2) = pInterface->ip_addr.addr;

		aNetInfo[uPos].ai_flags			= 0x04;
		aNetInfo[uPos].ai_family		= 2;
		aNetInfo[uPos].ai_socktype		= 0;
		aNetInfo[uPos].ai_protocol		= 0;
		aNetInfo[uPos].ai_addrlen		= 16;
		aNetInfo[uPos].ai_canonname		= 0;
		aNetInfo[uPos].ai_addr			= &aNetInfo[uPos].ai_adr;
		aNetInfo[uPos].ai_next			=  aNetInfo+uPos+1;
		aNetInfo[uPos].ai_adr.sa_len	= 16;
		aNetInfo[uPos].ai_adr.sa_family	= 2;	
						
		   uPos++;
		if(uPos>=8)break;
		}

	aNetInfo[uPos-1].ai_next = 0;
	}


#elif !defined(RTTARGET) && (defined(WIN32) || defined(WIN64))

static void _cdecl ExitSocketProc()
	{
	WSACleanup();
	}

static void InitSocketProc()
	{
	WORD    wVersion;
	WSADATA sWsaData;


	wVersion = 0x0202;
	if(WSAStartup(wVersion,&sWsaData))
		{
		wVersion = 0x0002;
		WSAStartup(wVersion,&sWsaData);
		}

	iSocketInit = 1;
	hGlobalSem  = SemaphoreInit();
	atexit(ExitSocketProc);

	#if SU_USE_IPV6

	int		iSocket;

	   iSocket = socket(AF_INET6,SOCK_DGRAM,IPPROTO_UDP);
	if(iSocket==INVALID_SOCKET)
		{
		iIpV6Allowed = 0;
		}
	else{
		closesocket(iSocket);
		}

	#endif
	}

#else

static void InitSocketProc()
	{
	iSocketInit = 1;
	hGlobalSem  = SemaphoreInit();


	#if SU_USE_IPV6

	int		iSocket;

	   iSocket = socket(AF_INET6,SOCK_DGRAM,IPPROTO_UDP);
	if(iSocket==INVALID_SOCKET)
		{
		iIpV6Allowed = 0;
		}
	else{
		closesocket(iSocket);
		}

	#endif
	}

#endif


//*****************************************************************************
//*
//*		getaddrinfo_ex
//*
//*****************************************************************************
//	Dummy fr Win200 weil hier getaddrinfo in der ws2_32.dll nicht existiert.

#ifdef LWIP 


	void freeaddrinfo_ex(struct addrinfo_ex *pAddrInfo)
	{
		if(!pAddrInfo)return;
		SemaphoreUnlock(hInfoSem);
	}

	static int getaddrinfo_ex(const char *pNodeName,const char *pServname,const struct addrinfo_ex *pHints,struct addrinfo_ex **pResult)
	{
	unsigned char  *pData;
	unsigned		uNum;
	unsigned		uVal;


	if(!pNodeName || !*pNodeName)
		{
		*pResult = 0;
		return -1;
		}

	SemaphoreLock(hInfoSem);

	if(*pNodeName<='9' && *pNodeName>='0')
		{
		pData = (unsigned char*)aNetInfo[0].ai_adr.sa_data+2;
		
		for(uNum=0;uNum<4;uNum++)						// decode ip address
			{
			   uVal  = atoi(pNodeName);
			if(uVal>255)break;

			pData[uNum] = uVal;
				  
			while(*pNodeName<='9' && *pNodeName>='0')
				{
				pNodeName++;
				}
			
			if(uNum==3)
				{
				if(*pNodeName!='\0')break;
				}
			else{
				if(*pNodeName!='.' )break;
				}

			pNodeName++;
			}

		if(uNum!=4)
			{
			SemaphoreUnlock(hInfoSem);
			*pResult = 0;
			return -1;
			}
		
		*pResult = aNetInfo+0;

		return 0;
		}

	if(*pNodeName=='l' && !strcmp(pNodeName,"localhost"))
		{
		*pResult = aNetInfo+1;

		return 0;
		}
		
	if(!strcmp(pNodeName,cHostName))
		{
		*pResult = aNetInfo+2;
		return 0;
		}

	SemaphoreUnlock(hInfoSem);
	*pResult = 0;
	return -1;
	}

#elif !defined(RTTARGET) && !defined(LWIP) && (defined(WIN32) || defined(WIN64))


	typedef int	(WSAAPI *getaddrinfo_type )(const char *,const char *,const struct addrinfo_ex *,struct addrinfo_ex **);
	typedef void(WSAAPI *freeaddrinfo_type)(struct addrinfo_ex *);

	typedef struct addrinfo_ex
		{
		int                 ai_flags;					// AI_PASSIVE, AI_CANONNAME, AI_NUMERICHOST
		int                 ai_family;      			// PF_xxx
		int                 ai_socktype;    			// SOCK_xxx
		int                 ai_protocol;    			// 0 or IPPROTO_xxx for IPv4 and IPv6
		size_t              ai_addrlen;     			// Length of ai_addr
		char			   *ai_canonname;   			// Canonical name for nodename
		struct sockaddr    *ai_addr;        			// Binary address
		struct addrinfo_ex *ai_next;        			// Next structure in linked list
		}addrinfo_ex_;



	static	HMODULE				hDll	= 0;
	static	int					iInit	= 0;
	static	getaddrinfo_type	pProc   = 0;
	static	freeaddrinfo_type	pFree	= 0;

	void freeaddrinfo_ex(struct addrinfo_ex *pAddrInfo)
	{
		if(!pFree)return;
		pFree(pAddrInfo);
	}

	static int getaddrinfo_ex(const char *pNodeName,const char *pServname,const struct addrinfo_ex *pHints,struct addrinfo_ex **pResult)
	{


		if(!iInit)
			{
			InitSocket();

			iInit = 1;

			if(!hDll)
				{
				#if WIN32
				hDll = LoadLibrary("ws2_32.dll");
				#else
				hDll = LoadLibrary("ws2_64.dll");
				#endif
				if(!hDll)return -1;
				}

			if(!pProc)pProc = (getaddrinfo_type )GetProcAddress(hDll,"getaddrinfo" );
			if(!pFree)pFree = (freeaddrinfo_type)GetProcAddress(hDll,"freeaddrinfo");
			}


		if(!pProc)return -1;


	return pProc(pNodeName,pServname,pHints,pResult);
	}


#elif defined(RTTARGET) && !defined(LWIP)


	typedef struct addrinfo_ex
		{
		int                 ai_flags;						// AI_PASSIVE, AI_CANONNAME, AI_NUMERICHOST
		int                 ai_family;      				// PF_xxx
		int                 ai_socktype;    				// SOCK_xxx
		int                 ai_protocol;    				// 0 or IPPROTO_xxx for IPv4 and IPv6
		size_t              ai_addrlen;     				// Length of ai_addr
		char			   *ai_canonname;   				// Canonical name for nodename
		struct sockaddr    *ai_addr;        				// Binary address
		struct addrinfo_ex *ai_next;        				// Next structure in linked list
		}addrinfo_ex_;

	#define		getaddrinfo_ex(a,b,c,d)	-1
	#define 	freeaddrinfo_ex(a)


#elif defined(LWIP)


	typedef struct addrinfo_ex
		{
		int                 ai_flags;						// AI_PASSIVE, AI_CANONNAME, AI_NUMERICHOST
		int                 ai_family;      				// PF_xxx
		int                 ai_socktype;    				// SOCK_xxx
		int                 ai_protocol;    				// 0 or IPPROTO_xxx for IPv4 and IPv6
		size_t              ai_addrlen;     				// Length of ai_addr
		char			   *ai_canonname;   				// Canonical name for nodename
		struct sockaddr    *ai_addr;        				// Binary address
		struct addrinfo_ex *ai_next;        				// Next structure in linked list
		}addrinfo_ex_;

	#define		getaddrinfo_ex(a,b,c,d)	-1
	#define 	freeaddrinfo_ex(a)


#else


	// LINUX
	#define		freeaddrinfo_ex	freeaddrinfo
	#define		getaddrinfo_ex	getaddrinfo
	#define		addrinfo_ex		addrinfo

#endif

//*****************************************************************************
//*
//*		CreateServerSocket
//*
//*****************************************************************************
//	Creates a server socked with automatic IPV4/IPV6 any address conversion
//	pAddrPtr	: pointer to the address pointer. It the address pointer
//				  is zero, a any address is used.
//	pTemp		: temorary address object.
//	iMode		: socket mode  (0=tcp 1=udp)
//	Returns the socket or INVALID_SOCKET, and fills the address if the
//	address pointer is zero.
static SOCKET CreateServerSocket(IpAddr **pAddrPtr,IpAddr *pTemp,int iMode)
{
IpAddr     *pAddr=*pAddrPtr;
SOCKET		iSock;



	#if SU_USE_IPV6

	int	iValue;

	if(!iIpV6Allowed)									// if no IPv6 possible use IPv4 any address
		{
		if(!pAddr)										// if zero then use 0.0.0.0
			{
			SET_V4_ADDR(pTemp,0);
			*pAddrPtr = pTemp;
			return INVALID_SOCKET;
			}

		if(pAddr->uAddr[0]==0 && pAddr->uAddr[1]==0 && pAddr->uAddr[2]==0)
			{
			if(pAddr->uAddr[3]==0)						// use "0.0.0.0" instead of "::"
				{
				SET_V4_ADDR(pTemp,0);
				*pAddrPtr = pTemp;
				}
			else if(pAddr->uAddr[3]==1)					// use "127.0.0.1" instead of "::1"
				{
				SET_V4_ADDR(pTemp,LOOPBACK);
				*pAddrPtr = pTemp;
				}
			}

		return INVALID_SOCKET;
		}
								
	if(iMode)											// is it udp						
		{
		if(!pAddr)										// if zero then use 0.0.0.0
			{
			SET_V4_ADDR(pTemp,0);
			*pAddrPtr = pTemp;
			return INVALID_SOCKET;
			}

		if(pAddr->uAddr[0]==SU_IPV4_ID)					// only IPV4 on udp
			{
			return INVALID_SOCKET;
			}
		}

	iSock  = socket(AF_INET6,(iMode)? SOCK_DGRAM:SOCK_STREAM,(iMode)? IPPROTO_UDP:IPPROTO_TCP);
	iValue = 0;											// enable dual-stack on the IPv6 socket

	if(setsockopt(iSock,IPPROTO_IPV6,IPV6_V6ONLY,(char*)&iValue,sizeof(iValue))==SOCKET_ERROR)
		{
		iDualStack = 0;

		if(!pAddr)
			{
			SET_V4_ADDR(pTemp,0);
			*pAddrPtr = pTemp;
			closesocket(iSock);
			return INVALID_SOCKET;
			}

		if(pAddr->uAddr[0]==SU_IPV4_ID)					// if IPv4 don't use the IPv6 socket
			{
			closesocket(iSock);
			return INVALID_SOCKET;
			}

		return iSock;
		}


	iDualStack = 1;

														// is it a any address
	if(!pAddr || (pAddr->uAddr[0]==SU_IPV4_ID && GET_V4_ADDR(pAddr)==0))
		{
		memset(pTemp,0,sizeof(IpAddr));					// use "::" instead of "0.0.0.0"
		*pAddrPtr = pTemp;
		}

	#else

	if(!pAddr)											// use any address "0.0.0.0"
		{
		SET_V4_ADDR(pTemp,0);
		*pAddrPtr = pTemp;
		}

	iSock = INVALID_SOCKET;

	#endif


return iSock;
}

//*****************************************************************************
//*
//*		CreateTcpMulti
//*
//*****************************************************************************
//	Adds a IPv4 any listen port to an IPv6 any port
//	hBase		: is the IPv6 base handle
//	uPort		: is the uesed port number
//	iFlags		: Several Flags for the connection:
//					TCP_NOBLOCKING = sets the socket in the no blocking mode
#if SU_USE_IPV6
static void CreateTcpMulti(TcpHandle hBase,unsigned short uPort,int iFlags)
{
unsigned long			lNoneBlocking=1;
struct sockaddr_in		sAddr;
TcpAddr			       *pData;
TcpHandle				hNext;
SOCKET					iSock;
int						iErr;



	   iSock=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
	if(iSock==INVALID_SOCKET)return;


	memset(&sAddr,0,sizeof(sAddr));

	sAddr.sin_family      =  AF_INET;
	sAddr.sin_port 		  =  htons(uPort);

		iErr = bind(iSock,(const struct sockaddr*)&sAddr,sizeof(sAddr));
	if(!iErr)iErr = listen(iSock,8);

	if(iErr)
		{
		closesocket(iSock);
		return;
		}

	if(iFlags&TCP_NOBLOCKING)							// set socket in none blocking mode
		{
		lNoneBlocking = 1;

		if(ioctlsocket(iSock,FIONBIO,&lNoneBlocking))
			{
			closesocket(iSock);
			return;
			}
		}



		hNext = TcpHandleAlloc(iSock,&sAddr,1,iFlags&TCP_NOBLOCKING);
	if(!hNext)return;

	pData = (TcpAddr*)hBase;
	pData->hServer  = hNext;


}
#endif

//*****************************************************************************
//*
//*		IpAddress
//*
//*****************************************************************************
//	Converts an ip-string in an ip-address
//	pAddr		: Poiter to the adress memory
//	pIp			: IP-String  i.e. "127.0.0.1"
//	pMask		: Subnetmask for broadcast addresses (NULL for normal adresses)
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int IpAddress(IpAddr *pAddr,const char *pIp,const char *pMask)
{
unsigned		uIpAddr,uIpMask;



    if(!pIp || !*pIp)
        {
		SET_V4_ADDR(pAddr,INADDR_NONE);
        return 0;
        }

	if(!IpHostname(pAddr,pIp))
		{
		return 0;
		}

	#if SU_USE_IPV6

	if(pMask && IsIpV6(pAddr))
		{
		memcpy(pAddr->uAddr,MULTICAST_V6,16);
		return 1;
		}

	#endif
   	if(pMask)
		{
		   uIpMask = inet_addr(pMask);
		if(uIpMask==INADDR_NONE)
			{
			SET_V4_ADDR(pAddr,INADDR_NONE);
			return 0;
			}

		uIpAddr = GET_V4_ADDR(pAddr)|~uIpMask;

		SET_V4_ADDR(pAddr,uIpAddr);
		}





return 1;
}

//*****************************************************************************
//*
//*		IpGetHostname
//*
//*****************************************************************************
//	Gets the hostname of this station.
//	Returns a pointer to the hostname
SUAPI const char *IpGetHostname()
{

	#if !defined(RTTARGET) && !defined(LWIP)

	if(!cHostName[0])
		{
		InitSocket();
		gethostname(cHostName,sizeof(cHostName));
		}

	#endif

return cHostName;
}

//*****************************************************************************
//*
//*		IpSetHostname
//*
//*****************************************************************************
//	Sets the hostname for this station
//	Returns the length of the hostname
SUAPI int IpSetHostname(const char *pName)
{
int		iPos;


	for(iPos=0;iPos<sizeof(cHostName)-1;iPos++)
		{
		if(!pName[iPos])break;
		cHostName[iPos] = pName[iPos];
		}

	cHostName[iPos] = 0;


return iPos;
}

//*****************************************************************************
//*
//*		IpIsV6Address
//*
//*****************************************************************************
//	Check a name if it is an IP V6 ip address, i.e. :
//				"fd9e:21a7:a92c:2323::1"
//				"fc00::/7"
//				"2001:db8::2"
//				"[2001:db8::2]"
//				"[::ffff:192.168.32.1]"
//				"::1"
//				"::"
//	pName	: is the pointer to the name
//	Returns >=1 if it an IP V6 ip address, or 0 in the other case
//					0 = is a hostname or an IPv4 address
//					1 = a valid IPv6 address  "1:2:3:4:5:6:7:8"
//					2 = any address "::"
//					3 = loopback address "::1"
//				  >=7 = offset from sup IP "::ffff:192.168.0.1"
SUAPI int IpIsV6Address(const char *pName)
{
char			cSign;
const char	   *pStart;
unsigned short	aWords[8+5];
unsigned		uValue;
int				iDouble;
int				iBraket;
int				iPoint;
int				iLast;
int				iSegs;
int				iSub;
int				iLen;
int				iRet;




	pStart  = pName;
	uValue  = 0;
	iBraket = 0;
	iDouble = 0;
	iPoint	= 0;
	iSegs	= 0;
	iLast	= 1;
	iLen	= 0;
	iSub	= 0;


	if(*pName=='[')
		{
		iBraket=1;
		pName++;
		}


	for(;;pName++)
		{
		   cSign = *pName;
		if(cSign==0)break;
		if(cSign=='.')										// special IPv4 "::FFFF:192.168.0.1"
			{
			if(uValue>0x255)return 0;

			if(iPoint==0)									// first IPv4 part
				{
				if(iSegs<1)return 0;
				if(iSegs>6)return 0;
				if(iSegs<6 && !iDouble)return 0;
				if(uValue%16>9)return 0;

				iSegs--;

				if(aWords[iSegs]!=0xFFFF)return 0;

				for(iSegs--;iSegs>=0;iSegs--)
					{
					if(aWords[iSegs])return 0;
					}

				iPoint = iLast;
				iSub   = 1;
				}
			else if(iSub>=3)								// only 4 sub parts
				{
				return 0;
				}
			else{
				iSub++;
				}


			uValue = 0;
			iLen   = 0;

			continue;
			}

		if(cSign==':')
			{
			aWords[iSegs] = (unsigned short)uValue;
			uValue		  = 0;

			   iSegs++;
			if(iSegs>=8 || iPoint)return 0;

			iLen = 0;

			if(pName[1]==':')								// found "::"
				{
				if(iDouble      )return 0;
				if(iSegs>=7     )return 0;
				if(pName[2]==':')return 0;
				if(pName[2]==']' && !iBraket)return 0;

				aWords[iSegs] = 0;
				pName   += 1;
				iSegs   += 1;
				iDouble  = iSegs;

				continue;
				}

			if(pName[1]== 0 )return 0;						// "1234:"  is not allowed
			if(pName[1]==']')return 0;						// "[234:]" is not allowed

			iLast = pName-pStart+1;

			continue;
			}


		if(cSign<='9' && cSign>='0')
			{
			uValue <<= 4;
			uValue  += cSign-'0';
			iLen    += 1;
			}
		else if(cSign==']')									// check end of text
			{
			if(iBraket)break;
			return 0;
			}
		else if(iPoint)
			{
			return 1;
			}
		else if(cSign>='a' && cSign<='f')
			{
			uValue <<= 4;
			uValue  += cSign-'a'+10;
			iLen    += 1;
			}
		else if(cSign>='A' && cSign<='F')
			{
			uValue <<= 4;
			uValue  += cSign-'A'+10;
			iLen    += 1;
			}
		else{
			return 0;
			}

		if(iLen>4)return 0;
		}



	if(iPoint)												// special IPv4 "::FFFF:192.168.0.1"
		{
		if(iSub==3 && uValue<=0x255)return iPoint;
		}
	else{
		if(iSegs==7 || iDouble)
			{
			if(uValue==0)									// is it any address "::"
				{
				iRet = 2;
				}
			else if(uValue==1)								// is it loopback "::1"
				{
				iRet = 3;
				}
			else{
				return 1;
				}

			for(iSegs--;iSegs>=0;iSegs--)					// check for zerros
				{
				if(aWords[iSegs])return 1;
				}

			return iRet;
			}
		}




return 0;
}

//*****************************************************************************
//*
//*		IpHostname
//*
//*****************************************************************************
//	Converts a text to an IP Address
//	pAddr		: Here is the IP address saved
//	pHostname	: Is the  i.e. "127.0.0.1" "www.sbox.tugraz.at" "ax-progr-22"
//			      NULL is the own host name
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int IpHostname(IpAddr *pAddr,const char *pHostname)
{
char				 cMyName[128];
char			   **pAddresses;
struct hostent	    *pHost;
struct addrinfo_ex  *pData;
struct addrinfo_ex	 sInfo;
struct addrinfo_ex  *pPtr;
struct sockaddr_in  *pIp;
int					 iRet;
int					 iLen;



	InitSocket();

	pData = 0;

	if(pHostname)
		{										// don't resolve IPV6 addresses if only IPV4 pressend
		if(iIpV6Allowed==0)
			{
			   iRet = IpIsV6Address(pHostname);
			if(iRet==0)							// it isn't an IPv6 addess
				{
				}
			else if(iRet==2)					// any address "::" used as "0.0.0.0"
				{
				SET_V4_ADDR(pAddr,0);
				return 1;
				}
			else if(iRet==3)					// loopback "::1" used as "127.0.0.1"
				{
				SET_V4_ADDR(pAddr,LOOPBACK);
				return 1;
				}
			else if(iRet<7)						// can't convert IPv6 to an IPv4 address
				{
				SET_V4_ADDR(pAddr,INADDR_NONE);
				return 0;
				}
			else if(*pHostname=='[')			// sub ip "[::FFFF:192.168.0.1]"
				{
				pHostname += iRet;

				for(iLen=0;;iLen++)
					{
					if(pHostname[iLen]==']')break;
					cMyName[iLen]=pHostname[iLen];
					}

				cMyName[iLen]=0;
				pHostname = cMyName;
				}
			else{								// sub ip "::FFFF:192.168.0.1"
				pHostname += iRet;
				}
			}


		memset(&sInfo,0,sizeof(sInfo));


		   iRet = getaddrinfo_ex(pHostname,0,&sInfo,&pData);
		if(iRet==0)
			{
			pPtr = pData;

			while(pPtr)
				{
				pIp = (struct sockaddr_in*)(pPtr->ai_addr);

				if(iRet==0 && pIp->sin_family==AF_INET)
					{
					SET_V4_ADDR(pAddr,*(unsigned*)&pIp->sin_addr);
					iRet=1;
					}

				#if SU_USE_IPV6
				else if(iRet<=1 && pIp->sin_family==AF_INET6 && iIpV6Allowed)
					{
					struct sockaddr_in6_ *pIp6=(struct sockaddr_in6_*)pIp;
					memcpy(pAddr->uAddr,pIp6->sin6_addr.s6_addr,16);
					iRet = 2;
					}
				#endif

				pPtr = pPtr->ai_next;
				}


			freeaddrinfo_ex(pData);

			if(iRet)return 1;
			}
		}


//****************** old version with gethostbyname ***************************


	if(!pHostname)								// use the own hostname
		{
		#if defined(RTTARGET) || defined(LWIP)
		
		if(cHostName[0]==0)
			{
			SET_V4_ADDR(pAddr,INADDR_NONE);
			return 0;
			}
		
		#else
		
		if(gethostname(cMyName,sizeof(cMyName))== SOCKET_ERROR)
			{
			SET_V4_ADDR(pAddr,INADDR_NONE);
			return 0;
			}
		
		#endif
		
		pHostname = cMyName;
		}

		pHost=gethostbyname((char*)pHostname);
	if(!pHost)
		{
		SET_V4_ADDR(pAddr,INADDR_NONE);
		return 0;
		}

	pAddresses=pHost->h_addr_list;

	if(!pAddresses[0])
		{
		SET_V4_ADDR(pAddr,INADDR_NONE);
		return 0;
		}


	   SET_V4_ADDR(pAddr,*(unsigned*)pAddresses[0]);
	if(GET_V4_ADDR(pAddr)==INADDR_NONE)return 0;



return 1;
}

//*****************************************************************************
//*
//*		UdpHandleAlloc
//*
//*****************************************************************************/
//	Creates a new UDP handel
//	Socket		: Socket number for the handle
//	pAddr		: pointer to the "sockaddr_in" data
//	iNoBlock	: 1 for a nonblocking socket, 0 for a blocking socket
//	Returns the pointer for the pData
static UdpHandle UdpHandleAlloc(SOCKET iSocket,void *pAddr,int iNoBlock)
{
UdpAddr	*pData;


		pData=(UdpAddr*)MallocHandle(sizeof(UdpAddr));
	if(!pData)
		{
		closesocket(iSocket);
		return 0;
		}


	pData->iSocket	     = iSocket;
	pData->uNoneBlocking = iNoBlock;

	memcpy(&pData->sAddr,pAddr,sizeof(pData->sAddr));

	#ifdef RTTARGET
	pData->hSemaphor=RTKCreateSemaphore(ST_COUNTING,1,NULL);
	#endif

	#if SU_USE_IPV6
	if(pData->sAddr.sin_family==AF_INET)
		{
		pData->sIpAddr.uAddr[0]=SU_IPV4_ID;
		pData->sIpAddr.uAddr[1]=pData->sAddr.sin_addr.s_addr;
		}
	else{
		memcpy(pData->sIpAddr.uAddr,pData->sAddr6.sin6_addr.s6_addr,16);
		}
	#else
	pData->sIpAddr=pData->sAddr.sin_addr.s_addr;
	#endif


return (UdpHandle)pData;
}


//*****************************************************************************
//*
//*		UdpHandleFree
//*
//*****************************************************************************/
//	Releases an UDP handle
static inline void UdpHandleFree(UdpHandle hHandle)
{

	if(!hHandle)return;
	#ifdef RTTARGET
	RTKDeleteSemaphore(&((UdpAddr*)hHandle)->hSemaphor);
	#endif
	FreeHandle(hHandle);
}


//*****************************************************************************
//*
//*		UdpOpen
//*
//*****************************************************************************/
//	Creates a UDP socket
//	pAddr		: Is the IP adress for the socked
//	uPort		: UDP port number (0=assign an unique port numper)
//	iBroadcast	: Broadcast socket, (when 1 a packet to "255.255.255.255"
//				  can send on all ethernet cards)
//	iNoBlock	: Blocking mode  (0=blocking 1=none-blocking)
//	Retrns 0 at an error, or a UDP handle
SUAPI UdpHandle UdpOpen(IpAddr *pAddr,unsigned short uPort,int iBroadcast,int iNoBlock)
{
void				   *pData;
struct sockaddr_in		sData;
#if SU_USE_IPV6
struct sockaddr_in6_	sData6;
#endif
SOCKET					iSock;
IpAddr					iTemp;
unsigned				uAddr;
int						iNameSize,iErr;
unsigned long			lNoneBlocking=1;



	InitSocket();

	iNameSize = sizeof(struct sockaddr_in);
	iSock     = CreateServerSocket(&pAddr,&iTemp,1);


	#if SU_USE_IPV6

	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{												// open socket
		if(iSock==INVALID_SOCKET)iSock=socket(AF_INET6,SOCK_DGRAM,IPPROTO_UDP);
		if(iSock==INVALID_SOCKET)
			{
			return 0;
			}

		memset(&sData6,0,sizeof(sData6));				//  build local address

		sData6.sin6_family      =  AF_INET6;
		sData6.sin6_port 		=  htons(uPort);

		if(iBroadcast)
				memcpy(sData6.sin6_addr.s6_addr,MULTICAST_V6,16);
		else	memcpy(sData6.sin6_addr.s6_addr,pAddr->uAddr,16);

		iErr  = bind(iSock,(const struct sockaddr*)&sData6,sizeof(sData6));
		pData = &sData6;
		}
	else{
	#endif 												// open socket
		if(iSock==INVALID_SOCKET)iSock=socket(AF_INET,SOCK_DGRAM,IPPROTO_UDP);
		if(iSock==INVALID_SOCKET)
			{
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpOpen: iSock==INVALID_SOCKET\n\n");
#endif				
			return 0;
			}

		memset(&sData,0,sizeof(sData));					//  build local address

		uAddr				  =  GET_V4_ADDR(pAddr);

		#if defined(RTTARGET) && RTT32_VER<523
		if(iBroadcast)uAddr	  =  INADDR_ANY;			// RTOS don't receive broadcasts on common ports
		#endif

		sData.sin_family      =  AF_INET;
		//sData.sin_addr.s_addr =  uAddr;
		sData.sin_addr.s_addr =  INADDR_ANY;
		sData.sin_port 		  =  htons(uPort);

		iErr  = bind(iSock,(const struct sockaddr*)&sData,sizeof(sData));
		pData = &sData;
	#if SU_USE_IPV6
		}
	#endif


	if(iErr)
		{
		closesocket(iSock);
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpOpen: bind error");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif
		return 0;
		}

	if(iNoBlock)
		{
		lNoneBlocking = 1;
		if(ioctlsocket(iSock,FIONBIO,&lNoneBlocking))
			{
			closesocket(iSock);
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpOpen: ioctlsocket error");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif
			return 0;
			}
		}

	#if !defined(RTTARGET) || RTT32_VER>=523

	if(iBroadcast)
		{
		int iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_BROADCAST,(char*)&iVal,sizeof(iVal)) == -1)
			{
			closesocket(iSock);
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpOpen: setsockopt error");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif
			return 0;
			}
		}

	#endif


return UdpHandleAlloc(iSock,pData,iNoBlock);
}

//*****************************************************************************
//*
//*		UdpGetPort
//*
//*****************************************************************************/
//	Gets the assigned Port which was was used at UdpOpen.
//	hHandle		: UDP handle of the socket
//	Returns 0 if an error occurs or the assigned port number
SUAPI int UdpGetPort(UdpHandle hHandle)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
struct sockaddr	sInfo;
int				iLen;
int				iErr;


	if(!pAddr)return 0;


	   iLen = sizeof(sInfo);
	   iErr = getsockname(pAddr->iSocket,(struct sockaddr*)&sInfo,(LEN_TYPE)&iLen);
	if(iErr)return 0;


	if(sInfo.sa_family==AF_INET)
		{
		struct sockaddr_in	*pInfo4;
		pInfo4 = (struct sockaddr_in*)&sInfo;
		return htons(pInfo4->sin_port);
		}

	#if SU_USE_IPV6
	if(sInfo.sa_family==AF_INET6)
		{
		struct sockaddr_in6_	*pInfo6;
		pInfo6= (struct sockaddr_in6_*)&sInfo;
		return htons(pInfo6->sin6_port);
		}
	#endif



return 0;
}

//*****************************************************************************
//*
//*		UdpBlock
//*
//*****************************************************************************/
//	Changes the blocking mode of a UDP socket
//	hHandle		: UDP handle of the socket
//	iOn			: 0=none-blocking  1=blocking
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int UdpBlock(UdpHandle hHandle,int iOn)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
unsigned long	lNoneBlocking;



	if(!pAddr)return 0;

	if(!iOn && !pAddr->uNoneBlocking)
		{
		lNoneBlocking = 1;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=1;
		}

	if(iOn &&  pAddr->uNoneBlocking)
		{
		lNoneBlocking = 0;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=0;
		}


return 1;
}

//*****************************************************************************
//*
//*		UdpSetBuffers
//*
//*****************************************************************************/
//	Changes the send and receive buffer sizes of a socket
//	hHandle		: TCP handle of the socket
//	iRecvSize	: Size for the receive buffer (0=none-change)
//	iSendSize	: Size for the send    buffer (0=none-change)
//	Returns		Bit0 :  0=receive buffer not change  1=changed
//				Bit1 :  0=send    buffer not change  1=changed
SUAPI int UdpSetBuffers(UdpHandle hHandle,int iRecvSize,int iSendSize)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
int				iRet=0;


	if(!pAddr)return 0;

	#ifndef RTTARGET

	if(iRecvSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_RCVBUF,(char*)&iRecvSize,sizeof(iRecvSize)) != -1)
		{
		iRet |=1;
		}

	if(iSendSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_SNDBUF,(char*)&iSendSize,sizeof(iSendSize)) != -1)
		{
		iRet |=2;
		}

	#endif


return iRet;
}


//*****************************************************************************
//*
//*		UdpShutdown
//*
//*****************************************************************************/
//	Disconnects a UDP connection. After this call no data will send or receive
//	on the socket
//	hHandle		: Handle of the UDP socket
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int UdpShutdown(UdpHandle hHandle)
{
UdpAddr	   *pAddr=(UdpAddr*)hHandle;


	if(!hHandle)return 0;
	if(shutdown(pAddr->iSocket,2))return 0;


return 1;
}

//*****************************************************************************
//*
//*		UdpClose
//*
//*****************************************************************************/
//	Closes a UDP socket
//	hHandle		: Handle of the UDP socket
//	Returns 0 if an error occurs or 1 if all ok
//  added by Leon: error return: -1(shutdown), -2(close), -3()
SUAPI int UdpClose(UdpHandle	hHandle)
{
UdpAddr   *pAddr=(UdpAddr*)hHandle;

	if(!hHandle){
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpClose: !hHandle\n\n");
#endif		
		return 0;
	}
		// 2=>SHUT_RDWR
	/*
	if(shutdown   (pAddr->iSocket,2) == -1){
		perror("shutdown");
		return -1;
	}
	*/
	if(closesocket(pAddr->iSocket  ) == -1){
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpClose: close failed");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif		
		return -2;
	}
	UdpHandleFree(hHandle);

return 1;
}

//*****************************************************************************
//*
//*		UdpPut
//*
//*****************************************************************************/
//	Sends a packet to a destination address, to the same port
//	pAddr : IP-Address of the destination
//	pMsg	 : Pointer to the packet
//	iLen	 : Size of the packet
//	Returns the count of the sended iBytes, or zero at an error
SUAPI int UdpPut(UdpHandle hHandle,IpAddr *pAddr,const void *pMsg,int iLen)
{
UdpAddr			       *pData=(UdpAddr*)hHandle;
struct sockaddr_in		sInput;
unsigned short			usPort;
int						iBytes;


	if(!hHandle){
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpPut: !hHandle\n\n");		
#endif		
		return 0;
	}

	usPort = pData->sAddr.sin_port;

	#if SU_USE_IPV6

	if(pData->sAddr.sin_family==AF_INET6)
		{
		usPort = pData->sAddr6.sin6_port;
		}

	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{
		struct sockaddr_in6_	sInput6;

		sInput6.sin6_family	=  AF_INET6;
		sInput6.sin6_port 	=  usPort;
		memcpy(&sInput6.sin6_addr,pAddr->uAddr,16);

		#ifdef RTTARGET
		RTKWait(pData->hSemaphor);
		#endif

		iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,SEND_FLAGS,(const struct sockaddr*)&sInput6,sizeof(sInput6));

		#ifdef RTTARGET
		RTKSignal(pData->hSemaphor);
		#endif

		if(iBytes<iLen){
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpPut: iBytes<iLen");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif				
			return 0;
		}

		return iBytes;
		}

	#endif

	sInput.sin_family		=  AF_INET;
	sInput.sin_addr.s_addr	=  GET_V4_ADDR(pAddr);
	sInput.sin_port 		=  usPort;

	#ifdef RTTARGET
	RTKWait(pData->hSemaphor);
	#endif

	iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,SEND_FLAGS,(const struct sockaddr*)&sInput,sizeof(sInput));

	#ifdef RTTARGET
	RTKSignal(pData->hSemaphor);
	#endif

	if(iBytes<iLen){
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpPut: iBytes<iLen");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif		
		return 0;
	}


return iBytes;
}

//*****************************************************************************
//*
//*		UdpPutTo
//*
//*****************************************************************************/
//	Sends a packet to a destination address to a other port
//	pAddr	: IP-Address of the destination
//	pMsg	: Pointer to the packet
//	iLen	: Size of the packet
//	uPort	: Port number of the destination
//	Returns the count of the sended bytes, or zero at an error
SUAPI int UdpPutTo(UdpHandle hHandle,IpAddr *pAddr,const void *pMsg,int iLen,unsigned short uPort)
{
UdpAddr			   *pData=(UdpAddr*)hHandle;
struct sockaddr_in	sInput;
int					iBytes;


	if(!hHandle){
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpPutTo: !hHandle\n\n");		
#endif				
		return 0;
	}

	#if SU_USE_IPV6

	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{
		struct sockaddr_in6_	sInput6;

		sInput6.sin6_family	=  AF_INET6;
		sInput6.sin6_port 	=  htons(uPort);
		memcpy(&sInput6.sin6_addr,pAddr->uAddr,16);

		#ifdef RTTARGET
		RTKWait(pData->hSemaphor);
		#endif

		iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,SEND_FLAGS,(const struct sockaddr*)&sInput6,sizeof(sInput6));

		#ifdef RTTARGET
		RTKSignal(pData->hSemaphor);
		#endif

		if(iBytes<iLen){
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpPutTo: iBytes<iLen");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif					
			return 0;
		}

		return iBytes;
		}

	#endif

	sInput.sin_family       =  AF_INET;
	sInput.sin_addr.s_addr  =  GET_V4_ADDR(pAddr);
	sInput.sin_port 		=  htons(uPort);

	#ifdef RTTARGET
	RTKWait(pData->hSemaphor);
	#endif

	iBytes = sendto(pData->iSocket,(char*)pMsg,iLen,SEND_FLAGS,(const struct sockaddr*)&sInput,sizeof(sInput));

	#ifdef RTTARGET
	RTKSignal(pData->hSemaphor);
	#endif

	if(iBytes<iLen)
		{
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpPutTo: iBytes<iLen");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif					
		return 0;
		}


return iBytes;
}

//*****************************************************************************
//*
//*		UdpGet
//*
//*****************************************************************************/
//	Reseives a packet
//	Form		: UDP handle from the socket
//	pRecv		: If this pointer is not zero,the ip address of the
//				  sender will stored there.
//	pMsg		: Pointer to the packet buffer
//	iLen		: Size of the packet buffer
//	Returns the count of the sended bytes, or zero at an error
SUAPI int UdpGet(UdpHandle hFrom,IpAddr *pRecv,void *pMsg,int iLen,unsigned short *pPort)
{
UdpAddr			       *pAddr=(UdpAddr*)hFrom;
#if SU_USE_IPV6
struct sockaddr_in6_	sInput;
#else
struct sockaddr_in		sInput;
#endif
int 					iSize;
int						iBytes;

	if(!hFrom)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		if(pPort)*pPort=0;
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpGet: !hFrom\n\n");			
#endif					
		return -1;
		}


	#if SU_USE_IPV6
	sInput.sin6_family = pAddr->sAddr.sin_family;
	#endif


	   iSize  = sizeof(sInput);
	   iBytes = recvfrom(pAddr->iSocket,(char*)pMsg,iLen,RECV_FLAGS,(struct sockaddr*)&sInput,(LEN_TYPE)&iSize);
	/*
	printf("\n===========================\n");
	//printf("UDP/ADDR: %s\n", inet_ntoa(sInput.sin_addr.s_addr));
	printf("UDP/PORT: %d\n", htons(sInput.sin_port));
	printf("UDP/BYTE: %d\n", iBytes);
	printf("\n===========================\n");
	*/
	if(iBytes<=0)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		if(pPort)*pPort=0;
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpGet: iBytes<=0");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif				
		return iBytes;
		}

	#if SU_USE_IPV6
	if(sInput.sin6_family==AF_INET)
		{
		if(pRecv)SET_V4_ADDR(pRecv,((struct sockaddr_in*)&sInput)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sInput)->sin_port);
		}
	else{
		if(pRecv)memcpy(pRecv->uAddr,sInput.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sInput.sin6_port);
		}
	#else
	if(pRecv)*pRecv=sInput.sin_addr.s_addr;
	if(pPort)*pPort=htons(sInput.sin_port);
	#endif

return iBytes;
}

//*****************************************************************************
//*
//*		UdpGetWait
//*
//*****************************************************************************/
//	Reseives a packet with a timeout. The function returns only
//  if a packet receives, or the timeout is expired.
//	hForm		: UDP handle from the socket
//	pRecv		: If this pointer is not zero,the ip address of the
//				  sender will stored there.
//	pMsg		: Pointer to the packet buffer
//	iLen		: Size of the packet buffer
//	iTimeout	: Time in millisecounds for the timeout
//	pPort		: If this pointer is not zero,the receifed port address
//	Returns the count of the sended iBytes, or zero at an error
SUAPI int UdpGetWait(UdpHandle hFrom,IpAddr *pRecv,void *pMsg,int iLen,int iTimeout,unsigned short *pPort)
{
UdpAddr				   *pAddr=(UdpAddr*)hFrom;
fd_set					sFdRead;
struct timeval			sTimeVal;
#if SU_USE_IPV6
struct sockaddr_in6_	sInput;
#else
struct sockaddr_in		sInput;
#endif
SysDW			    	uWait;
SysDW				    uStart;
int						iBytes;
int						iCount;
int						iSize;


	if(!hFrom)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		if(pPort)*pPort=0;
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpGetWait: !hFrom\n\n");			
#endif				
		return -1;
		}

	if(iTimeout<0)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		if(pPort)*pPort=0;
		return 0;
		}


	uWait=iTimeout;
	uStart=SysTickCount();

	sTimeVal.tv_sec   = uWait/1000;
	sTimeVal.tv_usec  =(uWait%1000)*1000;

	FD_ZERO(&sFdRead);
	FD_SET(pAddr->iSocket,&sFdRead);

	   iCount=select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal);
	if(iCount<=0)
		{
		if(pRecv)SET_V4_ADDR(pRecv,0);
		if(pPort)*pPort=0;
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpGetWait: iCount<=0");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");			
#endif		
		return iCount;
		}

	#if SU_USE_IPV6
	sInput.sin6_family = pAddr->sAddr.sin_family;
	#endif

	   iSize  = sizeof(sInput);
	   iBytes = recvfrom(pAddr->iSocket,(char*)pMsg,iLen,RECV_FLAGS,(struct sockaddr*)&sInput,(LEN_TYPE)&iSize);
	if(iBytes<=0)
		{
#if			NMB_LOG
			NMB_LOGGING("%s\tUdpGetWait: iBytes<=0");
			NMB_LOGARG(errno);
			NMB_LOGGING("\n\n");
#endif			
		if(pRecv)SET_V4_ADDR(pRecv,0);
		if(pPort)*pPort=0;
		return iBytes;
		}


	#if SU_USE_IPV6
	if(sInput.sin6_family==AF_INET)
		{
		if(pRecv)SET_V4_ADDR(pRecv,((struct sockaddr_in*)&sInput)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sInput)->sin_port);
		}
	else{
		if(pRecv)memcpy(pRecv->uAddr,sInput.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sInput.sin6_port);
		}
	#else
	if(pRecv)*pRecv=sInput.sin_addr.s_addr;
	if(pPort)*pPort=htons(sInput.sin_port);
	#endif



return iBytes;
}

//*****************************************************************************
//*
//*		UdpWait
//*
//*****************************************************************************/
//	Waits until the next data from a UDP socket was received.
//	hHandle		: Handle of the UDP socket
//	iTimeout	: Timeout in millisecounds
//	Returns -1 if an error occurs.
//  Returns  0 on timeouts.
//  Returns  the count of data in the the input buffer.
SUAPI int UdpWait(UdpHandle hHandle,int iTimeout)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
unsigned long	uReady;



	if(hHandle==0)return -1;
	if(iTimeout<0)
		{
		return 0;
		}

	sTimeVal.tv_sec  =  iTimeout/1000;
	sTimeVal.tv_usec = (iTimeout%1000)*1000;

	FD_ZERO(&sFdRead);
	FD_SET(pAddr->iSocket,&sFdRead);

	if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))// wait until data received
		{
		return 0;
		}

	if(ioctlsocket(pAddr->iSocket,FIONREAD,&uReady))	// get byte count in queue
		{
		return -1;
		}


return uReady;
}

//*****************************************************************************
//*
//*		UdpWaitMulti
//*
//*****************************************************************************/
//	Waits until the next data from a UDP socket array was received.
//	pHandle		: Pointer to the handle array of the UDP sockets
//	iCount		: Is the count of handles in the array (0..n)
//				  If this value is zero the function sleeps iTimeout ms.
//	iTimeout	: Timeout in millisecounds
//				: Is the offset for the first retuned handle in the array,
//				  if more than one is signaled.
//  Returns  -1 on timeout, or the number of the first handle with received data.
SUAPI int UdpWaitMulti(UdpHandle *pHandle,int iCount,int iTimeout,int iOffset)
{
UdpAddr		  **pAddr=(UdpAddr**)pHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
int				iPos;
int				iMax;



	if(pHandle==0)return -1;
	if(iTimeout<0 || iCount<0)
		{
		return -1;
		}

	if(iCount==0)
		{
		SysSleep(iTimeout);
		return -1;
		}

	sTimeVal.tv_sec  =  iTimeout/1000;
	sTimeVal.tv_usec = (iTimeout%1000)*1000;

	FD_ZERO(&sFdRead);

	iMax = 1;

	for(iPos=0;iPos<iCount;iPos++)						// set all sockets
		{
		if(iMax<=(int)pAddr[iPos]->iSocket)
			{
			iMax = pAddr[iPos]->iSocket+1;
			}

		FD_SET(pAddr[iPos]->iSocket,&sFdRead);
		}


	if(!select(iMax,&sFdRead,0,0,&sTimeVal))			// wait until data received
		{
		return -1;
		}


	if(iOffset<0)iOffset=0;


	for(iPos=0;iPos<iCount;iPos++,iOffset++)			// find signaled socket
		{
		if(iOffset>=iCount)iOffset=0;

		if(FD_ISSET(pAddr[iOffset]->iSocket,&sFdRead))
			{
			return iPos;
			}
		}



return -1;
}

//*****************************************************************************
//*
//*		UdpGetCount
//*
//*****************************************************************************/
//	Returns the size of the next packet in the receive buffer of the socket
//	hHandle	: UDP handle from the socket
//	Return the size or 0 if no packet is in the buffer, or -1 on an error
SUAPI int UdpGetCount(UdpHandle hHandle)
{
UdpAddr		   *pAddr=(UdpAddr*)hHandle;
unsigned long	uDataReady;


	if(!hHandle)return 0;
	if(ioctlsocket(pAddr->iSocket,FIONREAD,&uDataReady))return -1;


return uDataReady;
}


//*****************************************************************************
//*
//*		UdpGetLocal
//*
//*****************************************************************************
//	Gets the IP-Addres of the local side of the handle
//	hHandle		: UDP handle of the socket
//	pAddr		: Poiter to the adress memory (0 means not get)
//	pPort		: Pointer fpr the port number (0 means not get)
//  Returns 0 on errors or 1 if ok
SUAPI int UdpGetLocal(UdpHandle hHandle,IpAddr *pAddr,unsigned short *pPort)
{
UdpAddr			       *pData=(UdpAddr*)hHandle;
#if SU_USE_IPV6
struct sockaddr_in6_	sAddr;
#else
struct sockaddr_in		sAddr;
#endif
int						iLen;



	iLen = sizeof(sAddr);

	if(!hHandle || getsockname(pData->iSocket,(struct sockaddr*)&sAddr,(LEN_TYPE)&iLen))
		{
		#if SU_USE_IPV6

		if(pAddr)
			{
			pAddr->uAddr[0]=SU_IPV4_ID;
			pAddr->uAddr[1]=0;
			}

		if(pPort)*pPort=0;

		#else

		if(pAddr)*pAddr=0;
		if(pPort)*pPort=0;

		#endif

		return 0;
		}



	#if SU_USE_IPV6

	if(sAddr.sin6_family==AF_INET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,((struct sockaddr_in*)&sAddr)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sAddr)->sin_port);
		}
	else{
		if(pAddr)memcpy(pAddr->uAddr,sAddr.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sAddr.sin6_port);
		}

	#else

	if(pAddr)*pAddr=sAddr.sin_addr.s_addr;
	if(pPort)*pPort=htons(sAddr.sin_port);

	#endif



return 1;
}

//*****************************************************************************
//*
//*		TcpHandleAlloc
//*
//*****************************************************************************/
//	Creates a new TCP handel
//	iSocket		: iSocket number for the pData
//	pAddr		: pointer to the "sockaddr_in" data
//	iNoBlock	: 1 for a nonblocking socket, 0 for a blocking socket
//	Returns the pointer for the pData
static TcpHandle TcpHandleAlloc(SOCKET iSocket,void *pAddr,int iServer,int iNoBlock)
{
TcpAddr	*pData;


		pData=(TcpAddr*)MallocHandle(sizeof(TcpAddr));
	if(!pData)
		{
		closesocket(iSocket);
		return 0;
		}

	pData->iSocket        =  iSocket;
	pData->uNoneBlocking  =  iNoBlock;
	pData->hServer        = (iServer)? (void*)1:0;

	memcpy(&pData->sAddr,pAddr,sizeof(pData->sAddr));


	#if SU_USE_IPV6

	if(pData->sAddr.sin_family==AF_INET)
		{
		pData->sIpAddr.uAddr[0]=SU_IPV4_ID;
		pData->sIpAddr.uAddr[1]=pData->sAddr.sin_addr.s_addr;
		}
	else{
		memcpy(pData->sIpAddr.uAddr,pData->sAddr6.sin6_addr.s6_addr,16);
		}

	#else

	pData->sIpAddr=pData->sAddr.sin_addr.s_addr;

	#endif

	#if (defined(WIN32) || defined(WIN64)) && !defined(RTTARGET) && !defined(LWIP)

	pData->hEvent = 0;

	#endif

	if(!iServer)
		{
		#if SEND_TIMEOUT>0
		TcpSendTimeout((TcpHandle)pData,SEND_TIMEOUT*1000);
		#endif 	

		#if RECV_TIMEOUT>0
		TcpRecvTimeout((TcpHandle)pData,RECV_TIMEOUT*1000);
		#endif 	
		}


return (TcpHandle)pData;
}


//*****************************************************************************
//*
//*		TcpHandleFree
//*
//*****************************************************************************/
//	Releases a TCP handle
static inline void TcpHandleFree(TcpHandle hHandle)
{

	if(!hHandle)return;

	#if (defined(WIN32) || defined(WIN64)) && !defined(RTTARGET) && !defined(LWIP)

	TcpAddr	*pData=(TcpAddr*)hHandle;

	if(pData->hEvent)
		{
		WSACloseEvent(pData->hEvent);
		pData->hEvent = 0;
		}

	#endif

	FreeHandle(hHandle);
}


//*****************************************************************************
//*
//*		TcpOpenCancel
//*
//*****************************************************************************
//	Cancel the open process of a tcp socket
//	pCancel		: handle to the open process
//	Retuns 1 if cancel was done or 0 if not.
SUAPI int TcpOpenCancel(TcpCancel *pCancel)
{
int		iRet=0;



	SemaphoreLock(hGlobalSem);

	if(pCancel)
	if(pCancel->iValue==0x5A71D34E)
	if(pCancel->iSocket!=INVALID_SOCKET)
		{
		shutdown   (pCancel->iSocket,2);
		closesocket(pCancel->iSocket  );
		pCancel->iSocket = INVALID_SOCKET;
		iRet=1;
		}

	SemaphoreUnlock(hGlobalSem);


return iRet;
}

//*****************************************************************************
//*
//*		TcpOpenWait
//*
//*****************************************************************************/
//	Creates a TCP socket
//	pAddr		: Is the IP adress.  i.e. IpAddress(&iAddr,"127.0.0.1",0);
//				  If this pointer is zero an it is a sever socket the any
//				  address will be used.
//	uPort		: TCP port number
//	iTimeout	: Is the timeout for the connect procces in ms. (0=no timeout)
//	pCancel		: Handle for the cancel operation (0=no cancel operation)
//	iServer		: Is the connection a sever (1) or a client (0)
//	iFlags		: Several Flags for the connection:
//					TCP_NOBLOCKING = sets the socket in the no blocking mode
//					TCP_KEEP_ALIVE = enable sending of keepalive packets
//					TCP_NODELAYACK = disable nagle algorithm
//					Bit 16..31     = is the local port (0=auto, use TCP_USEPORT(a))
//	Retrns 0 at an error, or a TCP handle
SUAPI TcpHandle TcpOpenWait(IpAddr *pAddr,unsigned short uPort,int iTimeout,TcpCancel *pCancel,int iServer,int iFlags)
{
void				   *pData;
void				   *pTemp;
struct sockaddr_in		sAddr;
struct sockaddr_in		sTemp;
#if SU_USE_IPV6
struct sockaddr_in6_	sAddr6;
struct sockaddr_in6_	sTemp6;
int						iMulti=0;
#endif
TcpHandle				hHandle;
TcpCancel				sData;
SOCKET					iSock;
IpAddr					iTemp;
int						iSize;
int						iNameSize,iErr;
unsigned long			lNoneBlocking = 1;



	InitSocket();

	iNameSize = sizeof(struct sockaddr_in);
	iSock	  = INVALID_SOCKET;


	if(iServer)											// is it IPV4 any-sever-socket
		{
		iSock = CreateServerSocket(&pAddr,&iTemp,0);
		}
	else if(!pAddr)										// any address on clients not allowed
		{
		return 0;
		}


	#if SU_USE_IPV6
	if(pAddr->uAddr[0]!=SU_IPV4_ID)
		{												// open socket
		if(iSock==INVALID_SOCKET)iSock=socket(AF_INET6,SOCK_STREAM,IPPROTO_TCP);
		if(iSock==INVALID_SOCKET)return 0;

		memset(&sAddr6,0,sizeof(sAddr6));				// build local address
		memset(&sTemp6,0,sizeof(sTemp6));

		sAddr6.sin6_family      =  AF_INET6;
		sAddr6.sin6_port 		=  htons(uPort);

		memcpy(sAddr6.sin6_addr.s6_addr,pAddr->uAddr,16);

		sTemp6.sin6_family      =  AF_INET6;
		sTemp6.sin6_port 	    =  (unsigned short)htons(iFlags>>16);

		pTemp = &sTemp6;
		pData = &sAddr6;
		iSize = sizeof(sAddr6);
		}
	else if(iSock!=INVALID_SOCKET)						// IPV4 on dual socket
		{											
		memset(&sAddr6,0,sizeof(sAddr6));				// build local address
		memset(&sTemp6,0,sizeof(sTemp6));

		sAddr6.sin6_family      =  AF_INET6;
		sAddr6.sin6_port 		=  htons(uPort);
		
		memcpy(sAddr6.sin6_addr.s6_addr,IPV4TOV6,12);
		memcpy(sAddr6.sin6_addr.s6_addr+12,pAddr->uAddr+1,4);

		sTemp6.sin6_family      =  AF_INET6;
		sTemp6.sin6_port 	    =  (unsigned short)htons(iFlags>>16);

		pTemp = &sTemp6;
		pData = &sAddr6;
		iSize = sizeof(sAddr6);
		}
	else{
	#endif 												// open socket
		if(iSock==INVALID_SOCKET)iSock=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
		if(iSock==INVALID_SOCKET)return 0;

		memset(&sAddr,0,sizeof(sAddr));					// build local address
		memset(&sTemp,0,sizeof(sTemp));

		sAddr.sin_family      =  AF_INET;
		sAddr.sin_addr.s_addr =  GET_V4_ADDR(pAddr);
		sAddr.sin_port 		  =  htons(uPort);

		sTemp.sin_family      =  AF_INET;
		sTemp.sin_port 		  =  (unsigned short)htons(iFlags>>16);

		pTemp = &sTemp;
		pData = &sAddr;
		iSize = sizeof(sAddr);
	#if SU_USE_IPV6
		}
	#endif


	if(iTimeout>0)										// Timeout via Thread
		{
		if(!pCancel)pCancel	=&sData;
		SemaphoreLock(hGlobalSem);
		pCancel->iTimeout	= iTimeout;
		pCancel->iSocket	= iSock;
		pCancel->iValue		= 0x5A71D34E;
		pCancel->iRunning	= 0;
		SemaphoreUnlock(hGlobalSem);
		StartTimeout(pCancel);

		while(!pCancel->iRunning)
			{
			SysSleep(20);
			}
		}
	else if(pCancel)									// set cancel handle
		{
		SemaphoreLock(hGlobalSem);
		pCancel->iTimeout	= 0;
		pCancel->iSocket	= iSock;
		pCancel->iValue		= 0x5A71D34E;
		pCancel->iRunning	= 0;
		SemaphoreUnlock(hGlobalSem);
		}


	if(iFlags&TCP_REUSEADDR)							// enable reuse of the address
		{
		int iVal;

		#if defined(RTTARGET) && !defined(LWIP)
		
		iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_REUSESOCK,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}

		#endif 	

		iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_REUSEADDR,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}
		}


	if(iServer)											// install a server socket
		{
		    iErr = bind(iSock,(const struct sockaddr*)pData,iSize);
		if(!iErr)iErr = listen(iSock,8);
		#if SU_USE_IPV6
		if(!iErr && !iDualStack && iIpV6Allowed && IsIpAny(pAddr) && IsIpV6(pAddr))
			{
			iMulti = 1;
			}
		#endif
		}
	else{
		if(iFlags&0xFFFF0000)							// use special local port
			{
			bind(iSock,(const struct sockaddr*)pTemp,iSize);
			}

		iErr = connect(iSock,(const struct sockaddr*)pData,iSize);

		#if linux

		if(iErr)SysSleep(50);							// sometimes on linux connect(...) returns immiditly

		#endif
		}


	if(pCancel)											// reset cancel handle
		{
		SemaphoreLock(hGlobalSem);
		pCancel->iValue  = 0;
		SemaphoreUnlock(hGlobalSem);

		while(pCancel->iRunning)						// wait until thread is stopped
			{
			SysSleep(50);
			}

		if(pCancel->iSocket==INVALID_SOCKET)			// socket cloesd by cancel function
			{
			return 0;
			}
		}


	if(iErr)											// can't open the socket
		{
		closesocket(iSock);
		return 0;
		}

	if(iFlags&TCP_NOBLOCKING)							// set socket in none blocking mode
		{
		lNoneBlocking = 1;

		if(ioctlsocket(iSock,FIONBIO,&lNoneBlocking))
			{
			closesocket(iSock);
			return 0;
			}
		}

	if(iFlags&TCP_KEEP_ALIVE)							// enable keepalive option
		{
		int iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,SO_KEEPALIVE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}
		}

	if(iFlags&TCP_NODELAYACK)							// disable nagle algorithm
		{
		#ifndef RTTARGET

		int iVal=1;

		if(setsockopt(iSock,SOL_SOCKET,TCP_NODELAY,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}


		#elif !defined(LWIP)

		int iVal=0;

		if(setsockopt(iSock,SOL_SOCKET,SO_NAGLE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			closesocket(iSock);
			return 0;
			}

		iVal = 0;

		if(setsockopt(iSock,SOL_SOCKET,SO_DELAYED_ACK,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}

		#endif


		#if linux

		iVal = 1;

		if(setsockopt(iSock,IPPROTO_TCP,TCP_QUICKACK,(const char*)&iVal,sizeof(iVal)==SOCKET_ERROR))
			{
			closesocket(iSock);
			return 0;
			}

		#endif
		}

	hHandle = TcpHandleAlloc(iSock,pData,iServer,iFlags&TCP_NOBLOCKING);

	#if SU_USE_IPV6
	if(iMulti && hHandle)CreateTcpMulti(hHandle,uPort,iFlags);
	#endif


return hHandle;
}

//*****************************************************************************
//*
//*		TcpAccept
//*
//*****************************************************************************/
//	Accepts a connection on a server soket
//	hHandle	: Handle of the sever socket
//	pAddr	: If this pointer is not zero, the ip address of the client will
//			  stored there.
//	pPort	: If this pointer is not zero, the port number of the client will
//			  stored there.
//	Returns a handle for the TCP connetion or 0 if no connetion is opened
SUAPI TcpHandle TcpAccept(TcpHandle hHandle,IpAddr *pAddr,unsigned short *pPort)
{
TcpAddr	       *pData=(TcpAddr*)hHandle;
TcpHandle		hSock;
unsigned		uStart;
unsigned		uDiff;




	if(!hHandle)
		{
		return 0;
		}

	if(pData->uNoneBlocking)
		{
		return TcpAcceptWait(hHandle,pAddr,pPort,0);
		}

	for(;;)
		{
		uStart   = SysTickCount();

		   hSock = TcpAcceptWait(hHandle,pAddr,pPort,12345678);
		if(hSock)break;

		   uDiff = SysTickCount()-uStart;
		if(uDiff<500)SysSleep(200);
		}


	if(hSock==(TcpHandle)1)return NULL;



return hSock;
}

//*****************************************************************************
//*
//*		TcpAcceptWait
//*
//*****************************************************************************/
//	Accepts a connection on a server soket with a timeout, the function
//	returns only if a connection is accepted, or the timeout is expired.
//	hHandle		: Handle of the sever socket
//	pAddr	: If this pointer is not zero, the ip address of the client will
//			  stored there.
//	pPort	: If this pointer is not zero, the port number of the client will
//			  stored there.
//	iTimeout	: Timeout in ms
//	Returns a handle for the TCP connetion or 0 if no connetion is opened
SUAPI TcpHandle TcpAcceptWait(TcpHandle hHandle,IpAddr *pAddr,unsigned short *pPort,int iTimeout)
{
TcpAddr			       *pData=(TcpAddr*)hHandle;
TcpHandle				hError;
struct timeval			sTimeVal;
fd_set					sFdRead;
unsigned	 			uStart;
#if SU_USE_IPV6
TcpAddr			       *pNext;
struct sockaddr_in6_	sAddr;
#else
struct sockaddr_in		sAddr;
#endif
int 					iWait;
int						iSize;
int						iSock;
int						iMax;



	if(!hHandle)
		{
		SysSleep(iTimeout);
		return 0;
		}

	if(iTimeout==12345678)								// hack for TcpAccept
		{
		iWait		 = 5000;
		hError		 = (TcpHandle)1;
		}
	else{
		iWait		 = iTimeout;
		hError		 = 0;
		}


	sTimeVal.tv_sec  = iWait/1000;
	sTimeVal.tv_usec =(iWait%1000)*1000;
	uStart			 = SysTickCount();
	iSize			 = sizeof(sAddr);
	iMax			 = pData->iSocket+1;


	FD_ZERO(&sFdRead);
	FD_SET(pData->iSocket,&sFdRead);


	#if SU_USE_IPV6

	if(pData->hServer && pData->hServer!=(TcpHandle)1)	// is it a multiple port
		{
		pNext = (TcpAddr*)pData->hServer;
		FD_SET(pNext->iSocket,&sFdRead);
		if(iMax<=(int)pNext->iSocket)iMax=pNext->iSocket+1;
		}
	else{
		pNext = 0;
		}

	#endif


	if(select(iMax,&sFdRead,0,0,&sTimeVal)<=0)
		{
		if(pAddr)SET_V4_ADDR(pAddr,0);
		return 0;
		}


	#if SU_USE_IPV6
	if(pNext && FD_ISSET(pNext->iSocket,&sFdRead))		// is the multiple port signaled
		{
		pData = pNext;
		}
	#endif



	#if SU_USE_IPV6
	sAddr.sin6_family = IsIpV6(&pData->sIpAddr)? AF_INET6:AF_INET;
	#endif


	   iSock = accept(pData->iSocket,(struct sockaddr*)&sAddr,(LEN_TYPE)&iSize);
	if(iSock==INVALID_SOCKET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,0);
		return hError;
		}



	#if SU_USE_IPV6
	if(sAddr.sin6_family==AF_INET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,((struct sockaddr_in*)&sAddr)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sAddr)->sin_port);
		}
	else{
		if(pAddr)memcpy(pAddr->uAddr,sAddr.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sAddr.sin6_port);
		}
	#else
	if(pAddr)*pAddr=sAddr.sin_addr.s_addr;
	if(pPort)*pPort=htons(sAddr.sin_port);
	#endif



return TcpHandleAlloc(iSock,&sAddr,0,0);
}

//*****************************************************************************
//*
//*		TcpSetFlags
//*
//*****************************************************************************
//	Changes the flags of a TCP socket
//	hHandle		: TCP handle of the socket
//	iFlags		: Several Flags for the connection:
//					TCP_NOBLOCKING = sets the socket in the no blocking mode
//					TCP_KEEP_ALIVE = enable sending of keepalive packets
//					TCP_NODELAYACK = disable nagle algorithm
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpSetFlags(TcpHandle hHandle,int iFlags)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;


	if(!pAddr)return 0;


	if(iFlags&TCP_NOBLOCKING)							// set socket in none blocking mode
		{
		unsigned long	lNoneBlocking = 1;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))
			{
			return 0;
			}
		}

	if(iFlags&TCP_KEEP_ALIVE)							// enable keepalive option
		{
		int iVal=1;

		if(setsockopt(pAddr->iSocket,SOL_SOCKET,SO_KEEPALIVE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}
		}

	if(iFlags&TCP_NODELAYACK)							// disable nagle algorithm
		{
		#ifndef RTTARGET

		int iVal=1;

		if(setsockopt(pAddr->iSocket,SOL_SOCKET,TCP_NODELAY,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}


		#elif !defined(LWIP)

		int iVal=0;


		if(setsockopt(pAddr->iSocket,SOL_SOCKET,SO_NAGLE,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}

		iVal = 0;

		if(setsockopt(pAddr->iSocket,SOL_SOCKET,SO_DELAYED_ACK,(const char*)&iVal,sizeof(iVal))==SOCKET_ERROR)
			{
			return 0;
			}

		#endif

		#if linux

		iVal = 1;

		if(setsockopt(pAddr->iSocket,IPPROTO_TCP,TCP_QUICKACK,(const char *)&iVal,sizeof(iVal)==SOCKET_ERROR))
			{
			return 0;
			}

		#endif
		}



return 1;
}

//*****************************************************************************
//*
//*		TcpBlock
//*
//*****************************************************************************/
//	Changes the blocking mode of a TCP socket
//	hHandle		: TCP handle of the socket
//	iOn			: 0=none-blocking  1=blocking
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpBlock(TcpHandle hHandle,int iOn)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
unsigned long	lNoneBlocking;


	if(!pAddr)return 0;

	if(!iOn && !pAddr->uNoneBlocking)
		{
		lNoneBlocking = 1;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=1;
		}

	if(iOn &&  pAddr->uNoneBlocking)
		{
		lNoneBlocking = 0;

		if(ioctlsocket(pAddr->iSocket,FIONBIO,&lNoneBlocking))return 0;

		pAddr->uNoneBlocking=0;
		}


return 1;
}

//*****************************************************************************
//*
//*		TcpSetBuffers
//*
//*****************************************************************************/
//	Changes the send and receive buffer sizes of a socket
//	hHandle		: TCP handle of the socket
//	iRecvSize	: Size for the receive buffer (0=none-change)
//	iSendSize	: Size for the send    buffer (0=none-change)
//	Returns		Bit0 :  0=receive buffer not change  1=changed
//				Bit1 :  0=send    buffer not change  1=changed
SUAPI int TcpSetBuffers(TcpHandle hHandle,int iRecvSize,int iSendSize)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
int				iRet=0;


	if(!pAddr)return 0;


	#ifndef RTTARGET

	if(iRecvSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_RCVBUF,(char*)&iRecvSize,sizeof(iRecvSize)) != -1)
		{
		iRet |=1;
		}

	if(iSendSize && setsockopt(pAddr->iSocket,SOL_SOCKET,SO_SNDBUF,(char*)&iSendSize,sizeof(iSendSize)) != -1)
		{
		iRet |=2;
		}

	#endif



return iRet;
}

//*****************************************************************************
//*
//*		TcpSendTimeout
//*
//*****************************************************************************
//	Sets the send timeout of a socket.
//	hHandle		: TCP handle from the socket
//	iTimeout	: Is the timeout in milliseconds
//	Return the 0 if ok, or -1 if an error occurs
int TcpSendTimeout(TcpHandle hHandle, int iTimeout)
{
TcpAddr    *pAddr  = (TcpAddr*)hHandle;
int			iError;


	if(!pAddr)return -1;


	if(iTimeout<500)
		{
		iTimeout = 500;
		}


	#if defined(RTTARGET) && !defined(LWIP)
		{
		struct timeval	sTime;

		sTime.tv_sec	=  iTimeout/1000;
		sTime.tv_usec	= (iTimeout%1000)*1000;

		iError = setsockopt(pAddr->iSocket,SOL_SOCKET,SO_SEND_TIMEO,(const char*)&sTime,sizeof(sTime));
		}
	#endif

	#if (defined(WIN32) || defined(WIN64)) && (!defined(RTTARGET) || defined(LWIP))
		{
		int		iVal;

		iVal = iTimeout;

		iError = setsockopt(pAddr->iSocket,SOL_SOCKET,SO_SNDTIMEO,(const char*)&iVal,sizeof(iVal));
		}
	#endif

	#if linux
		{
		struct timeval	sTime;

		sTime.tv_sec	=  iTimeout/1000;
		sTime.tv_usec	= (iTimeout%1000)*1000;

		iError = setsockopt(pAddr->iSocket,SOL_SOCKET,SO_SNDTIMEO,(const char*)&sTime,sizeof(sTime));
		}
	#endif


return iError;
}


//*****************************************************************************
//*
//*		TcpRecvTimeout
//*
//*****************************************************************************
//	Sets the receive timeout of a socket.
//	hHandle		: TCP handle from the socket
//	iTimeout	: Is the timeout in milliseconds
//	Return the 0 if ok, or -1 if an error occurs
int TcpRecvTimeout(TcpHandle hHandle, int iTimeout)
{
TcpAddr	   *pAddr  = (TcpAddr*)hHandle;
int			iError;


	if(!pAddr)return -1;

	if(iTimeout<500)
		{
		iTimeout = 500;
		}



	#if defined(RTTARGET) && !defined(LWIP)
		{
		struct timeval	sTime;

		sTime.tv_sec	=  iTimeout/1000;
		sTime.tv_usec	= (iTimeout%1000)*1000;

		iError = setsockopt(pAddr->iSocket,SOL_SOCKET,SO_RCV_TIMEO,(const char*)&sTime,sizeof(sTime));
		}
	#elif defined(WIN32) || defined(WIN64) || defined(LWIP)
		{
		int		iVal;

		iVal = iTimeout;

		iError = setsockopt(pAddr->iSocket,SOL_SOCKET,SO_RCVTIMEO,(const char*)&iVal,sizeof(iVal));
		}
	#elif linux
		{
		struct timeval	sTime;

		sTime.tv_sec	=  iTimeout/1000;
		sTime.tv_usec	= (iTimeout%1000)*1000;

		iError = setsockopt(pAddr->iSocket,SOL_SOCKET,SO_RCVTIMEO,(const char*)&sTime,sizeof(sTime));
		}
	#endif


return iError;
}


//*****************************************************************************
//*
//*		TcpShutdown
//*
//*****************************************************************************/
//	Disconnects a TCP socket. After this call no data will send or receive
//	on the socket
//	hHandle		: Handle of the TCP socket
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpShutdown(TcpHandle	hHandle)
{
TcpAddr	   *pAddr=(TcpAddr*)hHandle;


	if(!hHandle)return 0;
	if(shutdown(pAddr->iSocket,2))return 0;


return 1;
}

//*****************************************************************************
//*
//*		TcpClose
//*
//*****************************************************************************/
//	Closes a TCP socket TCP socket
//	hHandle		: Handle of the TCP socket
//	Returns 0 if an error occurs or 1 if all ok
SUAPI int TcpClose(TcpHandle hHandle)
{
TcpAddr	 *pAddr=(TcpAddr*)hHandle;


	if(!hHandle)return 0;

	#if SU_USE_IPV6
	if(pAddr->hServer && pAddr->hServer!=(TcpHandle)1)
		{
		TcpClose((TcpHandle)pAddr->hServer);
		}
	#endif

	#ifndef LWIP
	shutdown(pAddr->iSocket,2);
	#endif 	

	closesocket(pAddr->iSocket);
	TcpHandleFree(hHandle);


return 1;
}


//*****************************************************************************
//*
//*		TcpPut
//*
//*****************************************************************************/
//	Sends data on a TCP socket
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Size of the data buffer
//	Returns the count of the sended bytes, or zero if an error occurs.
SUAPI int TcpPut(TcpHandle hHandle,const void *pMsg,int iLen)
{
TcpAddr	   *pAddr=(TcpAddr*)hHandle;
int			iBytes=0,iPos,iErr;


	if(!hHandle)return 0;

	for(iPos=0,iErr=0;iPos<iLen;iPos+=iBytes)
		{
		   iBytes = send(pAddr->iSocket,(char*)pMsg+iPos,iLen-iPos,SEND_FLAGS);
		if(iBytes<0)
			{
			if(TcpIsBroken(hHandle))return 0;

			iErr++;
			if(iErr>4)return 0;
			iBytes=0;
			SysSleep(10);
			continue;
			}

		iErr=0;
		}


return iBytes;
}

//*****************************************************************************
//*
//*		TcpGet
//*
//*****************************************************************************/
//	Reveives data from a TCP socket. If the socket is none blocking the
// 	function returns immediately. On blocking sockets, the function waits
//	if all bytes received.
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Size of the data buffer
//	Returns -1 if the connection is brocken or no byte is received on blocking sokets
//  Returns  0 on NONE_BLOCKING sockets if no byte is received.
//	Returns the count of the received bytes
SUAPI int TcpGet(TcpHandle hHandle,void *pMsg,int iLen)
{
TcpAddr	   *pAddr=(TcpAddr*)hHandle;
int			iBytes;


	if(!hHandle)return -1;

	if(pAddr->uNoneBlocking)
		{
		struct timeval	sTimeVal;
		fd_set			sFdRead;

		sTimeVal.tv_sec  =  0;
		sTimeVal.tv_usec	=  0;

		FD_ZERO(&sFdRead);
		FD_SET(pAddr->iSocket,&sFdRead);

		if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))return 0;

		if(iLen==0)										// empty buffer
			{
			if(TcpIsBroken(hHandle))return -1;
			return 0;
			}

		iBytes = recv(pAddr->iSocket,(char*)pMsg,iLen,RECV_FLAGS);

		if(iBytes<0)return 0;
		}
	else{
		if(iLen==0)										// empty buffer
			{
			if(TcpIsBroken(hHandle))return -1;
			return 0;
			}

		iBytes = recv(pAddr->iSocket,(char*)pMsg,iLen,RECV_FLAGS);

		if(iBytes<0)return -1;
		}


return iBytes;
}

//*****************************************************************************
//*
//*		TcpGetWait
//*
//*****************************************************************************/
//	Receives data from a TCP socket with a timeout, the function returns only
//  if all data bytes are receives, or the timeout is expired.
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Count of the bytes you will receive
//	iTimeout	: iTimeout in millisecounds
//	Returns -1 if the connection is brocken or no byte is received on blocking sokets
//  Returns  0 on NONE_BLOCKING sockets if no byte is received.
//	Returns the count of the received bytes
SUAPI int TcpGetWait(TcpHandle hHandle,void *pMsg,int iLen,int iTimeout)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
SysDW			uStart;
int 			iWait;
int				iBytes;
int				iNum;


	if(hHandle==0)return -1;
	if(iTimeout<0)
		{
		if(TcpIsBroken(hHandle))return -1;
		return 0;
		}

	iWait  = iTimeout;
	uStart = SysTickCount();

	for(iBytes=0;iBytes<iLen;)
		{
		sTimeVal.tv_sec  =  iWait/1000;
		sTimeVal.tv_usec = (iWait%1000)*1000;

		FD_ZERO(&sFdRead);
		FD_SET(pAddr->iSocket,&sFdRead);
														// wait until data received
		if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))
			{
			   iWait=iTimeout-(SysTickCount()-uStart);
			if(iWait<=0)return iBytes;

			if(TcpIsBroken(hHandle))					// check for broken connection
				{
				return (!iBytes)? -1:iBytes;
				}

			SysSleep((iWait>300)? 30:10);
			continue;
			}

		if(iLen==iBytes)								// empty buffer
			{
			iNum = 0;
			}
		else{
			   iNum = recv(pAddr->iSocket,(char*)pMsg+iBytes,iLen-iBytes,RECV_FLAGS);
			if(iNum<0)return -1;
			}

		if(iNum==0)										// check for broken connection
			{
			if(pAddr->uNoneBlocking==0 || TcpIsBroken(hHandle))
				{
				return (!iBytes)? -1:iBytes;
				}
			}


		iBytes += iNum;

		   iWait =iTimeout-(SysTickCount()-uStart);
		if(iWait<=0)return iBytes;
		if(iNum ==0)SysSleep((iWait>300)? 30:10);		// BUG in Windows ip stack !!!
		}


return iBytes;
}

//*****************************************************************************
//*
//*		TcpGetNext
//*
//*****************************************************************************/
//	Receives the next data from a TCP socket with a timeout, the function
//  returns if one or more data bytes are receives, or the timeout time is expired.
//	hHandle		: Handle of the TCP socket
//	pMsg		: Pointer to the data buffer
//	iLen		: Count of the bytes you will maximal receive
//	iTimeout	: Timeout in millisecounds
//	Returns -1 if the connection is brocken or no byte is received on blocking sokets
//  Returns  0 on timeouts.
//	Returns the count of the received bytes
SUAPI int TcpGetNext(TcpHandle hHandle,void *pMsg,int iLen,int iTimeout)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
unsigned long	uReady;
SysDW			uStart;
int 			iWait;
int				iNum;


	if(hHandle==0)return -1;
	if(iTimeout<0)
		{
		if(TcpIsBroken(hHandle))return -1;
		return 0;
		}

	iWait  = iTimeout;
	uStart = SysTickCount();

	for(;;)
		{
		sTimeVal.tv_sec  =  iWait/1000;
		sTimeVal.tv_usec = (iWait%1000)*1000;

		FD_ZERO(&sFdRead);
		FD_SET(pAddr->iSocket,&sFdRead);
														// wait until data received
		if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))
			{
			   iWait=iTimeout-(SysTickCount()-uStart);
			if(iWait<=0)return 0;

			if(TcpIsBroken(hHandle))					// check for broken connection
				{
				return -1;
				}

			SysSleep((iWait>300)? 30:10);

			continue;
			}

		if(ioctlsocket(pAddr->iSocket,FIONREAD,&uReady))// get byte count in queue
			{
			return -1;
			}

		if((int)uReady>iLen)uReady=iLen;

		if(uReady==0)									// empty buffer
			{
			iNum = 0;
			}
		else{
			iNum = recv(pAddr->iSocket,(char*)pMsg,uReady,RECV_FLAGS);
			}

		if(iNum>0)break;
		if(iNum<0)
			{
			return -1;
			}

		if(pAddr->uNoneBlocking==0)
			{
			return -1;
			}


		if(TcpIsBroken(hHandle))						// check for broken connection
			{
			return -1;
			}


		   iWait =iTimeout-(SysTickCount()-uStart);
		if(iWait<=0)return 0;

		SysSleep((iWait>300)? 30:10);					// BUG in Windows ip stack !!!
		}


return iNum;
}

//*****************************************************************************
//*
//*		TcpWait
//*
//*****************************************************************************/
//	Waits until the next data from a TCP socket was received.
//	hHandle		: Handle of the TCP socket
//	iTimeout	: Timeout in millisecounds
//	Returns -1 if the connection is brocken
//  Returns  0 on timeouts.
//  Returns  the count of data in the the input buffer
SUAPI int TcpWait(TcpHandle hHandle,int iTimeout)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
unsigned long	uReady;



	if(hHandle==0)return -1;
	if(iTimeout<0)
		{
		if(TcpIsBroken(hHandle))return -1;
		return 0;
		}

	sTimeVal.tv_sec  =  iTimeout/1000;
	sTimeVal.tv_usec = (iTimeout%1000)*1000;

	FD_ZERO(&sFdRead);
	FD_SET(pAddr->iSocket,&sFdRead);

	if(!select(pAddr->iSocket+1,&sFdRead,0,0,&sTimeVal))// wait until data received
		{
		if(TcpIsBroken(hHandle))return -1;				// check for broken connection
		return 0;
		}

	if(ioctlsocket(pAddr->iSocket,FIONREAD,&uReady))	// get byte count in queue
		{
		return -1;
		}


return uReady;
}

//*****************************************************************************
//*
//*		TcpWaitMulti
//*
//*****************************************************************************/
//	Waits until the next data from a TCP socket array was received.
//	pHandle		: Pointer to the handle array of the TCP sockets
//	iCount		: Is the count of handles in the array (0..n)
//				  If this value is zero the function sleeps iTimeout ms.
//	iTimeout	: Timeout in millisecounds
//				: Is the offset for the first retuned handle in the array,
//				  if more than one is signaled.
//  Returns  -1 on timeout, or the number of the first handle with received data.
SUAPI int TcpWaitMulti(TcpHandle *pHandle,int iCount,int iTimeout,int iOffset)
{
TcpAddr		  **pAddr=(TcpAddr**)pHandle;
struct timeval	sTimeVal;
fd_set			sFdRead;
int				iPos;
int				iMax;



	if(pHandle==0)return -1;
	if(iTimeout<0 || iCount<0)
		{
		return -1;
		}

	if(iCount==0)
		{
		SysSleep(iTimeout);
		return -1;
		}

	sTimeVal.tv_sec  =  iTimeout/1000;
	sTimeVal.tv_usec = (iTimeout%1000)*1000;

	FD_ZERO(&sFdRead);

	iMax = 1;

	for(iPos=0;iPos<iCount;iPos++)						// set all sockets
		{
		if(iMax<=(int)pAddr[iPos]->iSocket)
			{
			iMax = pAddr[iPos]->iSocket+1;
			}

		FD_SET(pAddr[iPos]->iSocket,&sFdRead);
		}


	if(!select(iMax,&sFdRead,0,0,&sTimeVal))			// wait until data received
		{
		return -1;
		}


	if(iOffset<0)iOffset=0;


	for(iPos=0;iPos<iCount;iPos++,iOffset++)			// find signaled socket
		{
		if(iOffset>=iCount)iOffset=0;

		if(FD_ISSET(pAddr[iOffset]->iSocket,&sFdRead))
			{
			return iPos;
			}
		}



return -1;
}

//*****************************************************************************
//*
//*		TcpGetCount
//*
//*****************************************************************************/
//	Gets the size of stored data in the receive buffer of the socket.
//	hHandle	: TCP handle from the socket
//	Return the size or 0 if no data is in the buffer, or -1 on an error occurs.
SUAPI int TcpGetCount(TcpHandle hHandle)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;
unsigned long	uDataReady;


	if(!hHandle)return 0;
	if(ioctlsocket(pAddr->iSocket,FIONREAD,&uDataReady))return -1;


return uDataReady;
}

//*****************************************************************************
//*
//*		TcpIsBroken
//*
//*****************************************************************************/
//	Checks if a tcp connection is brocken or closed.
//	Attention, a closed socket can have data in his queue.
//	hHandle	: TCP handle from the socket
//	Returns 1 if closed or broken, and 0 if the connection is online.
SUAPI int TcpIsBroken(TcpHandle hHandle)
{
TcpAddr		   *pAddr=(TcpAddr*)hHandle;



	if(hHandle==0)return 1;
	if(pAddr->iSocket==INVALID_SOCKET)return 1;


	#if defined(LWIP)
	{
	int		iState;
	
	iState = lwip_state(pAddr->iSocket);
													// check the socket state
	if(iState==CLOSED || iState>=FIN_WAIT_1 || iState==SOCKET_ERROR)
		{
		return 1;
		}
	}
	#elif defined(RTTARGET)

	if(!xn_tcp_is_connect(pAddr->iSocket))			// check the socket state
		{
		return 1;
		}

	#elif (defined(WIN32) || defined(WIN64))

	if(!pAddr->hEvent)								// exists the close event
		{
		pAddr->hEvent = WSACreateEvent();
		WSAEventSelect(pAddr->iSocket,pAddr->hEvent,FD_CLOSE);
		}
													// is the close envet signaled
	if(WSAWaitForMultipleEvents(1,&pAddr->hEvent,TRUE,0,TRUE)==WSA_WAIT_EVENT_0)
		{
		return 1;
		}

	#elif linux
	{
	struct tcp_info		sInfo;
	socklen_t			iLen;

	iLen = sizeof(sInfo);

	if(getsockopt(pAddr->iSocket,SOL_TCP,TCP_INFO,(char*)&sInfo,&iLen)==SOCKET_ERROR)
		{
		return 1;
		}

	if(sInfo.tcpi_state==TCP_CLOSE || sInfo.tcpi_state>=TCP_FIN_WAIT1)
		{
		return 1;
		}
	}
	#endif



return 0;
}

//*****************************************************************************
//*
//*		TcpGetLocal
//*
//*****************************************************************************
//	Gets the IP address from the local side of the handle
//	hHandle		: TCP handle of the socket
//	pAddr		: Poiter to the adress memory (0 means not get)
//	pPort		: Pointer fpr the port number (0 means not get)
//  Returns 0 on errors, or 1 if ok
SUAPI int TcpGetLocal(TcpHandle hHandle,IpAddr *pAddr,unsigned short *pPort)
{
TcpAddr			       *pData=(TcpAddr*)hHandle;
#if SU_USE_IPV6
struct sockaddr_in6_	sAddr;
#else
struct sockaddr_in		sAddr;
#endif
int						iLen;



	iLen = sizeof(sAddr);

	if(!hHandle || getsockname(pData->iSocket,(struct sockaddr*)&sAddr,(LEN_TYPE)&iLen))
		{
		#if SU_USE_IPV6

		if(pAddr)
			{
			pAddr->uAddr[0]=SU_IPV4_ID;
			pAddr->uAddr[1]=0;
			}

		if(pPort)*pPort=0;

		#else

		if(pAddr)*pAddr=0;
		if(pPort)*pPort=0;

		#endif

		return 0;
		}



	#if SU_USE_IPV6

	if(sAddr.sin6_family==AF_INET)
		{
		if(pAddr)SET_V4_ADDR(pAddr,((struct sockaddr_in*)&sAddr)->sin_addr.s_addr);
		if(pPort)*pPort = htons   (((struct sockaddr_in*)&sAddr)->sin_port);
		}
	else{
		if(pAddr)memcpy(pAddr->uAddr,sAddr.sin6_addr.s6_addr,16);
		if(pPort)*pPort=htons(sAddr.sin6_port);
		}

	#else

	if(pAddr)*pAddr=sAddr.sin_addr.s_addr;
	if(pPort)*pPort=htons(sAddr.sin_port);

	#endif



return 1;
}

//*****************************************************************************
//*
//*		IpToString
//*
//*****************************************************************************
//	Converts an ip address to a string
//	pAddr	: ip address (should be nor smaler than 42 bytes)
//	pIp		: pointer to string memory
//	Returns the size of the string.
SUAPI int	IpToString(IpAddr *pAddr,char *pIp)
{
int		iLen;
union	{
		unsigned long ulValue;
		unsigned char ucData[4];
		}u;


	#if SU_USE_IPV6
	if(pAddr->uAddr[0]==SU_IPV4_ID)
		{
		u.ulValue=pAddr->uAddr[1];
		iLen=sprintf(pIp,"%i.%i.%i.%i",u.ucData[0],u.ucData[1],u.ucData[2],u.ucData[3]);
		}
	else if(pAddr->uAddr[2]==SU_IPV4_TAG && !pAddr->uAddr[1] && !pAddr->uAddr[0])
		{
		u.ulValue=pAddr->uAddr[3];
		iLen=sprintf(pIp,"%i.%i.%i.%i",u.ucData[0],u.ucData[1],u.ucData[2],u.ucData[3]);
		}
	else{
		unsigned char  *pData;
		int				iPos;
		int				iMax;
		int				iOff;

		iMax =  0;
		iLen =  0;
		iOff = -1;

		pData = (unsigned char*)(pAddr->uAddr);

		for(iPos=0;iPos<8;iPos++)					// search for the largest zero area
			{
			if(!pData[iPos*2] && !pData[iPos*2+1])
				{
				iLen++;
				continue;
				}

			if(iLen>iMax)
				{
				iOff=iPos-iLen;
				iMax=iLen;
				}

			iLen=0;
			}

		if(iLen>iMax)
			{
			iOff=iPos-iLen;
			iMax=iLen;
			iLen=0;
			};

		for(iPos=0,iLen=0;iPos<8;iPos++)			// print the values
			{
			if(iPos==iOff)
				{
				pIp[iLen]=':';iLen++;
				pIp[iLen]=':';iLen++;
				iPos += iMax-1;
				continue;
				}

			if(pData[iPos*2])
					iLen+=sprintf(pIp+iLen,"%X%02X",pData[iPos*2  ],pData[iPos*2+1]);
			else	iLen+=sprintf(pIp+iLen,"%X"    ,pData[iPos*2+1]);

			if(iPos<7)
				{
				pIp[iLen]=':';
				iLen++;
				}
			}

		pIp[iLen]=0;
		}

	#else
	u.ulValue=*pAddr;
	iLen=sprintf(pIp,"%i.%i.%i.%i",u.ucData[0],u.ucData[1],u.ucData[2],u.ucData[3]);
	#endif


return iLen;
}



