/*
 *  Boa, an http server, realms.c implements realm password policies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <grp.h>
#include <sys/types.h>
#include "basic_auth.h"
#include "users.h"
#include "realms.h"
#include "boa.h"

/* Error messages. */
static char out_of_memory[] = "Out of memory.";
static char no_realms_list[] = "Could not complete realms list";

/* String in configuration file and numeric category association. */
struct protection {
     char name[32];
     enum prot_category category;
};

/* Supported categories: anonymous, password. */
static struct protection prot_table[last_prot_category] = {
     {"anonymous", anonymous_prot_category},
     {"password", password_prot_category}
};

/*
 * List contains all defined realms from the boa configuration file.
 */
struct realm *realms = NULL;

void init_realm(void);
#ifndef DISABLE_DEBUG
void dump_realms(void);
#endif
void add_realm_group(char *realm_name, char *group_name);
void add_prot_space(char *realm_name, char *space);
void add_prot_realm(char *realm_name, char *category_name);
struct realm *find_realm(const char *realm_name);
void set_user_realm_list(struct user *user_item);
static struct realm *new_realm(const char *realm_name);
static struct realm *insert_realm(struct realm *realm);
static struct realm_group *new_group(const char *group_name);
static struct realm_group *insert_group(struct realm *realm,
					struct realm_group *group);
static enum prot_category valid_category(const char *category_name);


/* Free all allocated memory in the lists realms. */
void
init_realm(void)
{
     struct realm *realm;
     struct realm *next_realm;
     struct realm_group *group;
     struct realm_group *next_group;

     /* Free memory in the list realms. */
     realm = realms;
     while (realm != NULL) {
	  group = realm->groups;
	  while (group != NULL) {
	       free(group->name);
	       next_group = group->next;
	       free(group);
	       group = next_group;
	  }
	  free(realm->name);
	  free(realm->space);
	  next_realm = realm->next;
	  free(realm);
	  realm = next_realm;
     }
     realms = NULL;
}

#ifndef DISABLE_DEBUG
/* Debug, dump all realm items. */
void
dump_realms(void)
{
     struct realm *realm_item;
     struct realm_group *group_item;

     realm_item = realms;
     while (realm_item != NULL) {
	  fprintf(stderr, "category [%d] ", realm_item->category);
	  fprintf(stderr, "[%s] ", realm_item->name);
	  fprintf(stderr, "[%s] ", realm_item->space);
	  group_item = realm_item->groups;
	  while (group_item != NULL) {
	       fprintf(stderr, "( [%s] ", group_item->name);
	       fprintf(stderr, "[%d] )", group_item->id);
	       group_item = group_item->next;
	  }
	  fprintf(stderr, "\n");
	  realm_item = realm_item->next;
     }
}
#endif

/*
 * Configuration file: Realm <realm_name> <group>
 * Called each time the configuration file parser finds the tag.
 */
void
add_realm_group(char *realm_name, char *group_name)
{
     struct realm *realm_item;
     struct realm_group *group_item;

     /* Find an existing realm or create a new one. */
     realm_item = find_realm(realm_name);
     if (realm_item == NULL) {
	  realm_item = new_realm(realm_name);
	  if (realm_item == NULL) {
	       perror(out_of_memory);
	       WARN(out_of_memory);
	       _exit(EXIT_FAILURE);
	  }
	  realm_item = insert_realm(realm_item);
     }

     /* Insert the group in the realm. The group name can be non-existent. */
     group_item = new_group(group_name);
     if (group_item == NULL) {
	  perror(out_of_memory);
	  WARN(out_of_memory);
	  _exit(EXIT_FAILURE);
     }
     group_item = insert_group(realm_item, group_item);
}

/*
 * Configuration file: ProtSpace <realm_name> <protection space>
 * Called each time the configuration file parser finds the tag.
 */
void
add_prot_space(char *realm_name, char *space)
{
     struct realm *realm_item;

     /* Find existing realm or fail. */
     realm_item = find_realm(realm_name);
     if (realm_item == NULL) {
	  WARN(realm_name);
	  _exit(EXIT_FAILURE);
     }
     realm_item->space = strdup(space);
}

/*
 * Configuration file: ProtRealm <realm_name> [password|anonymous]
 * Called each time the configuration file parser finds the tag.
 */
void
add_prot_realm(char *realm_name, char *category_name)
{
     enum prot_category i;
     struct realm *realm_item;

     /* Find existing realm or fail. */
     realm_item = find_realm(realm_name);
     if (realm_item == NULL) {
	  WARN(realm_name);
	  _exit(EXIT_FAILURE);
     }

     /* Verify the category. */
     i = valid_category(category_name);
     if (i < 0) {
	  WARN(category_name);
	  _exit(EXIT_FAILURE);
     }
     realm_item->category = prot_table[i].category;
}

/* Return a struct realm* to a defined realm_name or NULL. */
struct realm*
find_realm(const char *realm_name)
{
     struct realm *realm_item;

     realm_item = realms;
     while (realm_item != NULL) {
	  if (STR_EQ(realm_item->name, realm_name)) {
	       break;
	  }
	  realm_item = realm_item->next;
     }
     return realm_item;
}

/* Setup the list of all accesible realms */
void 
set_user_realm_list(struct user *user_item)
{
     char buf[LEN_REALM_LIST];
     struct realm *ritem;
     int pos = 0;
  
     /* for each realm */
     for (ritem = realms; ritem != NULL; ritem = ritem->next) {
	  struct realm_group *rgroup;
	  char added = 0;
    
	  /* for each group giving access to the realm */
	  rgroup = ritem->groups;
	  while (!added && rgroup != NULL) {
	       int i = 0;
      
	       /* for each group ID */
	       while (!added && i < user_item->no_groups) {
		    gid_t gid = user_item->groups[i];
                 
		    if (gid == rgroup->id) {
			 int size = sizeof(buf) - 1 - pos;
			 int n;
			 n = snprintf(buf + pos, 
				      size, 
				      "%c%s",
				      REALM_LIST_SEPARATOR,
				      ritem->name);
			 /* handle tedious C99 standard as well */
			 if (n < 0 || n >= size) {
			      goto error;
			 }
			 pos += n;
			 /* avoid duplicate appearence */
			 added = 1;
		    } 
		    ++i;
	       }
	       rgroup = rgroup->next;
	  }
     }
  
     if (pos + 2 > LEN_REALM_LIST) {
	  goto error;
     }
     buf[pos] = REALM_LIST_SEPARATOR;
     buf [pos + 1] = '\0';
     user_item->realm_list = strdup(buf);
     return;

error:
     WARN(no_realms_list);
     _exit(EXIT_FAILURE);
}

/* Return a struct realm* to an intialised realm_name or NULL. */
static struct realm*
new_realm(const char *realm_name)
{
     struct realm *realm;

     realm = (struct realm *)malloc(sizeof(struct realm));
     if (realm == NULL) {
	  return NULL;
     }
     realm->name = strdup(realm_name);
     if (realm->name == NULL) {
	  free(realm);
	  return NULL;
     }
     realm->category = password_prot_category;
     realm->space = NULL;
     realm->groups = NULL;
     realm->next = NULL;

     return realm;
}

/* Return a struct realm* realm inserted in realms.*/
static struct realm*
insert_realm(struct realm *realm)
{
     struct realm *realm_item;

     if (realms == NULL) {
	  /* Empty list, insert first. */
	  realms = realm;
	  realm_item = realms;
     } else {
	  /* Insert last. TODO sorted list scales better. */
	  realm_item = realms;
	  while(realm_item->next != NULL) {
	       realm_item = realm_item->next;
	  }
	  realm_item->next = realm;
	  realm_item = realm_item->next;
     }

     return realm_item;
}

/* Return a struct group* to an intialised group_name or NULL. */
static struct realm_group*
new_group(const char *group_name)
{
     struct realm_group *group_item;

     /* Create a new group element and initialize it. */
     group_item = (struct realm_group *)malloc(sizeof(struct realm_group));
     if (group_item == NULL) {
	  return NULL;
     }
     /* A non-existing group name does not get a group id. */
     group_item->name = strdup(group_name);
     if (group_item->name == NULL) {
	  free(group_item);
	  return NULL;
     }
     /* Initialised from UNIX groups. */
     group_item->id = USHRT_MAX;
     group_item->next = NULL;

     return group_item;
}

/* Return a struct realm_group* group inserted in struct realm*.*/
static struct realm_group*
insert_group(struct realm *realm, struct realm_group *group)
{
     struct realm_group *group_item;

     if (realm->groups == NULL) {
	  /* Empty list, insert first. */
	  realm->groups = group;
	  group_item = realm->groups;
     } else {
	  /* Insert last. TODO sorted list scales better. */
	  group_item = realm->groups;
	  while(group_item->next != NULL) {
	       group_item = group_item->next;
	  }
	  group_item->next = group;
	  group_item = group_item->next;
     }

     return group_item;
}

/* Return an index to a valid category associated with category_name or -1. */
static enum prot_category
valid_category(const char *category_name)
{
     enum prot_category i;

     for (i = anonymous_prot_category; i < last_prot_category; i++) {
	  if (STR_EQ(prot_table[i].name, category_name)) {
	       break;
	  }
     }
     if (i == last_prot_category) {
	  return -1;
     }
     return i;
}
