
/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
#include <common.h>
#include <malloc.h>
#include <net.h>
#include "ftgmac100.h"


#ifdef CONFIG_FTGMAC100

static const char version[] =
	"Faraday FTGMAC100 Driver\n";


#define inl(addr) 			(*((volatile u32 *)(addr)))
#define inw(addr)			(*((volatile u16 *)(addr)))
#define outl(value, addr)  	(*((volatile u32 *)(addr)) = value)
#define outb(value, addr)	(*((volatile u8 *)(addr)) = value)


/*------------------------------------------------------------------------
 .
 . Configuration options, for the experienced user to change.
 .
 -------------------------------------------------------------------------*/



/*
 . DEBUGGING LEVELS
 .
 . 0 for normal operation
 . 1 for slightly more details
 . >2 for various levels of increasingly useless information
 .    2 for interrupt tracking, status flags
 .    3 for packet info
 .    4 for complete packet dumps
*/

#define DO_PRINT(args...) printk(args)

//#define FTGMAC100_DEBUG 5 // Must be defined in makefile

#if (FTGMAC100_DEBUG > 2 )
#define PRINTK3(args...) DO_PRINT(args)
#else
#define PRINTK3(args...)
#endif

#if FTGMAC100_DEBUG > 1
#define PRINTK2(args...) DO_PRINT(args)
#else
#define PRINTK2(args...)
#endif

#ifdef FTGMAC100_DEBUG
#define PRINTK(args...) DO_PRINT(args)
#else
#define PRINTK(args...)
#endif


///#define FTGMAC100_TIMER
#define PHY_MARVELL			0 //for MARVELL 88E111
#define PHY_ICPLUS			1 //for ICPLUS IP1001
#define PHY_ICPLUS_100	2 //for ICPLUS IP101A
#define PHY_AR					3 

static int PHY_MODE = PHY_AR;

/*------------------------------------------------------------------------
 .
 . The internal workings of the driver.  If you are changing anything
 . here with the SMC stuff, you should have the datasheet and know
 . what you are doing.
 .
 -------------------------------------------------------------------------*/
#define CARDNAME "FTGMAC100"




#ifdef FTGMAC100_TIMER
	static struct timer_list ftgmac100_timer;
#endif




#define ETH_ZLEN 60

#ifdef  CONFIG_SMC_USE_32_BIT
#define USE_32_BIT
#else
#undef USE_32_BIT
#endif
/*-----------------------------------------------------------------
 .
 .  The driver can be entered at any of the following entry points.
 .
 .------------------------------------------------------------------  */

extern int eth_init(bd_t *bd);
extern void eth_halt(void);
extern int eth_rx(void);
extern int eth_send(volatile void *packet, int length);


int initialized = 0;



/*
 . This is called by  register_netdev().  It is responsible for
 . checking the portlist for the FTMAC100 series chipset.  If it finds
 . one, then it will initialize the device, find the hardware information,
 . and sets up the appropriate device parameters.
 . NOTE: Interrupts are *OFF* when this procedure is called.
 .
 . NB:This shouldn't be static since it is referred to externally.
*/
int ftgmac100_init(struct eth_device *dev,bd_t *bd);

/*
 . This is called by  unregister_netdev().  It is responsible for
 . cleaning up before the driver is finally unregistered and discarded.
*/
void ftgmac100_destructor(struct eth_device *dev);

/*
 . The kernel calls this function when someone wants to use the eth_device,
 . typically 'ifconfig ethX up'.
*/
static int ftgmac100_open(struct eth_device *dev);


/*
 . This is called by the kernel in response to 'ifconfig ethX down'.  It
 . is responsible for cleaning up everything that the open routine
 . does, and maybe putting the card into a powerdown state.
*/
static int ftgmac100_close(struct eth_device *dev);


/*
 . This is a separate procedure to handle the receipt of a packet, to
 . leave the interrupt code looking slightly cleaner
*/
inline static int ftgmac100_rcv( struct eth_device *dev );



/*
 ------------------------------------------------------------
 .
 . Internal routines
 .
 ------------------------------------------------------------
*/



/*
 . A rather simple routine to print out a packet for debugging purposes.
*/
#if FTGMAC100_DEBUG > 2
static void print_packet( byte *, int );
#endif




/* this does a soft reset on the device */
static void ftgmac100_reset( struct eth_device *dev );

/* Enable Interrupts, Receive, and Transmit */
static void ftgmac100_enable( struct eth_device *dev );

/* this puts the device in an inactive state */
static void ftgmac100_shutdown( unsigned int ioaddr );

/* Routines to Read and Write the PHY Registers across the
   MII Management Interface
*/


void put_mac(int base, char *mac_addr)
{
	int val;

	val = ((u32)mac_addr[0])<<8 | (u32)mac_addr[1];
	outl(val, base);
	val = ((((u32)mac_addr[2])<<24)&0xff000000) |
		  ((((u32)mac_addr[3])<<16)&0xff0000) |
		  ((((u32)mac_addr[4])<<8)&0xff00)  |
		  ((((u32)mac_addr[5])<<0)&0xff);
	outl(val, base+4);

}

void get_mac(int base, char *mac_addr)
{
	int val;
	val = inl(base);
	mac_addr[0] = (val>>8)&0xff;
	mac_addr[1] = val&0xff;
	val = inl(base);
	mac_addr[2] = (val>>24)&0xff;
	mac_addr[3] = (val>>16)&0xff;
	mac_addr[4] = (val>>8)&0xff;
	mac_addr[5] = val&0xff;
}

// --------------------------------------------------------------------
// 	Print the Ethernet address
// --------------------------------------------------------------------
void print_mac(char *mac_addr)
{
	int i;

	DO_PRINT("ADDR: ");
	for (i = 0; i < 5; i++)
	{
		DO_PRINT("%2.2x:", mac_addr[i] );
	}
	DO_PRINT("%2.2x \n", mac_addr[5] );
}




/*
 . Function: ftgmac100_reset( struct eth_device* dev )
 . Purpose:
 .  	This sets the SMC91111 chip to its normal state, hopefully from whatever
 . 	mess that any other DOS driver has put it in.
 .
 . Maybe I should reset more registers to defaults in here?  SOFTRST  should
 . do that for me.
 .
 . Method:
 .	1.  send a SOFT RESET
 .	2.  wait for it to finish
 .	3.  enable autorelease mode
 .	4.  reset the memory management unit
 .	5.  clear all interrupts
 .
*/
static void ftgmac100_reset( struct eth_device* dev )
{
	//struct ftgmac100_local *lp 	= (struct ftgmac100_local *)dev->priv;
	unsigned int tmp;
	unsigned int	ioaddr = dev->iobase;

	PRINTK2("%s:ftgmac100_reset\n", dev->name);

	outl( SW_RST_bit, ioaddr + MACCR_REG );

#ifdef not_complete_yet
	/* Setup for fast accesses if requested */
	/* If the card/system can't handle it then there will */
	/* be no recovery except for a hard reset or power cycle */
	if (dev->dma)
	{
		outw( inw( ioaddr + CONFIG_REG ) | CONFIG_NO_WAIT,	ioaddr + CONFIG_REG );
	}
#endif /* end_of_not */

	/* this should pause enough for the chip to be happy */
	for (; (inl( ioaddr + MACCR_REG ) & SW_RST_bit) != 0; )
	{
		mdelay(10);
		PRINTK3("RESET: reset not complete yet\n" );
	}
		tmp = inl( ioaddr + MACCR_REG );
		tmp |= 0x80000;
		outl(tmp, ioaddr + MACCR_REG );
	outl( 0, ioaddr + IER_REG );			/* Disable all interrupts */
}


/*
 . Function: ftgmac100_enable
 . Purpose: let the chip talk to the outside work
 . Method:
 .	1.  Enable the transmitter
 .	2.  Enable the receiver
 .	3.  Enable interrupts
*/
static void ftgmac100_enable( struct eth_device *dev )
{
	unsigned int ioaddr 	= dev->iobase;
	int i;
	struct ftgmac100_local *lp 	= (struct ftgmac100_local *)dev->priv;
	unsigned int tmp_rsize;		//Richard
	unsigned int rfifo_rsize;	//Richard
	unsigned int tfifo_rsize;	//Richard
	unsigned int rxbuf_size;

	PRINTK2("%s:ftgmac100_enable\n", dev->name);

#ifdef NC_Body
	rxbuf_size = RX_BUF_SIZE & 0x3fff;
	outl( rxbuf_size , ioaddr + RBSR_REG); //for NC Body
#endif

	for (i=0; i<RXDES_NUM; ++i)
	{
		lp->rx_descs[i].RXPKT_RDY = RX_OWNBY_FTGMAC100;				// owned by FTMAC100
	}
	lp->rx_idx = 0;

	for (i=0; i<TXDES_NUM; ++i)
	{
		lp->tx_descs[i].TXDMA_OWN = TX_OWNBY_SOFTWARE;			// owned by software
	}
	lp->tx_idx = 0;


	/* set the MAC address */
	put_mac(ioaddr + MAC_MADR_REG, dev->enetaddr);

	outl( lp->rx_descs_dma, ioaddr + RXR_BADR_REG);
	outl( lp->tx_descs_dma, ioaddr + TXR_BADR_REG);
	outl( 0x00001010, ioaddr + ITC_REG);					// Ȭ document ҫĳ
	///outl( 0x0, ioaddr + ITC_REG);
	///outl( (1UL<<TXPOLL_CNT)|(1UL<<RXPOLL_CNT), ioaddr + APTC_REG);
	outl( (0UL<<TXPOLL_CNT)|(0x1<<RXPOLL_CNT), ioaddr + APTC_REG);
	//outl( 0x1df, ioaddr + DBLAC_REG );						// Ȭ document ҫĳ
	outl( 0x22f97, ioaddr + DBLAC_REG ); //Richard burst

	outl( inl(FCR_REG)|0x1, ioaddr + FCR_REG );				// enable flow control
	outl( inl(BPR_REG)|0x1, ioaddr + BPR_REG );				// enable back pressure register

	// +++++ Richard +++++ //
	tmp_rsize = inl( ioaddr + FEAR_REG );
	rfifo_rsize = tmp_rsize & 0x00000007;
	tfifo_rsize = (tmp_rsize >> 3)& 0x00000007;

	tmp_rsize = inl( ioaddr + TPAFCR_REG );
	tmp_rsize &= 0x3f000000;
	tmp_rsize |= (tfifo_rsize << 27);
	tmp_rsize |= (rfifo_rsize << 24);

	outl(tmp_rsize, ioaddr + TPAFCR_REG);
	// ----- Richard ----- //

	/* now, enable interrupts */
	outl(
			PHYSTS_CHG_bit	|
			AHB_ERR_bit		|
///			RPKT_LOST_bit	|
///			RPKT2F_bit	|
///			TPKT_LOST_bit	|
///			TPKT2E_bit		|
///			NPTXBUF_UNAVA_bit		|
///			TPKT2F_bit	|
///			RXBUF_UNAVA_bit		|
			RPKT2B_bit
        	,ioaddr + IER_REG
        );

	/// enable trans/recv,...
	outl(lp->maccr_val, ioaddr + MACCR_REG );

#ifdef FTGMAC100_TIMER
	/// waiting to do: ӥHWd
	init_timer(&ftgmac100_timer);
	ftgmac100_timer.function = ftgmac100_timer_func;
	ftgmac100_timer.data = (unsigned long)dev;
	mod_timer(&ftgmac100_timer, jiffies + FTGMAC100_STROBE_TIME);
#endif
}

/*
 . Function: ftmac100_shutdown
 . Purpose:  closes down the chip.
 . Method:
 .	1. zero the interrupt mask
 .	2. clear the enable receive flag
 .	3. clear the enable xmit flags
 .
 . TODO:
 .   (1) maybe utilize power down mode.
 .	Why not yet?  Because while the chip will go into power down mode,
 .	the manual says that it will wake up in response to any I/O requests
 .	in the register space.   Empirical results do not show this working.
*/
static void ftgmac100_shutdown( unsigned int ioaddr )
{
	/// unmask interrupt mask register
	outl( 0, ioaddr + IER_REG );

	/// disable trans/recv,...
	outl( 0, ioaddr + MACCR_REG );
}


//static int time_old,time_new;
static int ftgmac100_send_packet(  struct eth_device *dev,void *packet, int length )
{
	volatile struct ftgmac100_local *lp 	= (struct ftgmac100_local *)dev->priv;
	volatile unsigned int ioaddr 	= dev->iobase,i=0;
	volatile TX_DESC *cur_desc;

  //printf("<d%x>",get_timer(0));
	PRINTK3("%s:ftgmac100_wait_to_send_packet\n", dev->name);
	cur_desc = &lp->tx_descs[lp->tx_idx];
	for (; cur_desc->TXDMA_OWN != TX_OWNBY_SOFTWARE; )		/// SŪ transmit descriptor iHϥ
	{
		DO_PRINT("Transmitting busy\n");
		udelay(10);
   	}
	length = ETH_ZLEN < length ? length : ETH_ZLEN;
	length = length > TX_BUF_SIZE ? TX_BUF_SIZE : length;

#if FTGMAC100_DEBUG > 2

	//DO_PRINT("Transmitting Packet\n");
	print_packet( packet, length ); //Justin
#endif
	//memcpy((char *)cur_desc->VIR_TXBUF_BADR, packet, length);		/// waiting to do: N data \h segment
	memcpy((char *)virt_to_phys( cur_desc->TXBUF_BADR ), packet, length); //Richard
//time_new=get_timer(0);
//if((time_new-time_old)<5000)
//printf("<%x>",inl(ioaddr + 0xa0));
{
	cur_desc->TXBUF_Size = length;
	cur_desc->LTS = 1;
	cur_desc->FTS = 1;
	cur_desc->TX2FIC = 0;
	cur_desc->TXIC = 0;
	cur_desc->TXDMA_OWN = TX_OWNBY_FTGMAC100;
	outl( 0xffffffff, ioaddr + TXPD_REG);
//printf("=>");
	lp->tx_idx = (lp->tx_idx + 1) % TXDES_NUM;
}
//else
//	printf("too late\n");
//time_old=get_timer(0);
//printf("<%x>",lp->tx_idx);
while (cur_desc->TXDMA_OWN == 1)
{
	udelay(10);
	i++;
	if(i>100)
	{
		printf("out\n");
		break;
	}
	if(cur_desc->TXDMA_OWN == 0)
		break;
//printf("<%x>",get_timer(0));
}
//printf("<%x>",inl(ioaddr + 0xa0));
	return length;
}

/*-------------------------------------------------------------------------
 |
 | smc_destructor( struct eth_device * dev )
 |   Input parameters:
 |	dev, pointer to the device structure
 |
 |   Output:
 |	None.
 |
 ---------------------------------------------------------------------------
*/
void ftgmac100_destructor(struct eth_device *dev)
{
	PRINTK3("%s:ftgmac100_destructor\n", dev->name);
}


/*
 * Open and Initialize the board
 *
 * Set up everything, reset the card, etc ..
 *
 */
static int ftgmac100_open(struct eth_device *dev)
{
	unsigned int	ioaddr = dev->iobase;

	PRINTK2("%s:ftgmac100_open\n", dev->name);

#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif

	/* reset the hardware */
	ftgmac100_reset( dev );
	ftgmac100_enable( dev );

	/* set the MAC address */
	put_mac(ioaddr + MAC_MADR_REG, dev->enetaddr);

	return 0;
}

#ifdef USE_32_BIT
void
insl32(r,b,l)
{
   int __i ;
   dword *__b2;

	__b2 = (dword *) b;
	for (__i = 0; __i < l; __i++)
   {
		  *(__b2 + __i) = *(dword *)(r+0x10000300);
	}
}
#endif

/*-------------------------------------------------------------
 .
 . ftmac100_rcv -  receive a packet from the card
 .
 . There is ( at least ) a packet waiting to be read from
 . chip-memory.
 .
 . o Read the status
 . o If an error, record it
 . o otherwise, read in the packet
 --------------------------------------------------------------
*/
static int ftgmac100_rcv(struct eth_device *dev)
{
	struct ftgmac100_local *lp = (struct ftgmac100_local *)dev->priv;
	int 	packet_length;
	volatile RX_DESC *cur_desc;
	int 	cpy_length;
	int		start_idx;
	int		seg_length;
	int 	rcv_cnt;
	//printf("<R>");
	//PRINTK3("%s:ftgmac100_rcv\n", dev->name);
	for (rcv_cnt=0; rcv_cnt<1; ++rcv_cnt)
	{
		packet_length = 0;
		start_idx = lp->rx_idx;

		for (; (cur_desc = &lp->rx_descs[lp->rx_idx])->RXPKT_RDY==RX_OWNBY_SOFTWARE; )
		{
			lp->rx_idx = (lp->rx_idx+1)%RXDES_NUM;
			if (cur_desc->FRS)
			{
				if (cur_desc->RX_ERR || cur_desc->CRC_ERR || cur_desc->FTL || cur_desc->RUNT || cur_desc->RX_ODD_NB)
				{
					cur_desc->RXPKT_RDY = RX_OWNBY_FTGMAC100;		///  frame wBz, ٵ hardware
					return 0;
				}

				//packet_length = cur_desc->ReceiveFrameLength;	//Richard
			}

			packet_length += cur_desc->VDBC; //Richard

			if ( cur_desc->LRS )		// packet's last frame
			{
				break;
			}
		}
		if (packet_length>0)			// @ packet
		{
			byte		* data;
//printf("R%x,%x>",get_timer(0),packet_length);
			data = NetRxPackets[0];
			cpy_length = 0;
			for (; start_idx!=lp->rx_idx; start_idx=(start_idx+1)%RXDES_NUM)
			{
				seg_length = min(packet_length - cpy_length, RX_BUF_SIZE);

				//memcpy(data+cpy_length, (char *)lp->rx_descs[start_idx].VIR_RXBUF_BADR, seg_length);
				memcpy(data+cpy_length, (char *)virt_to_phys( lp->rx_descs[start_idx].RXBUF_BADR ), seg_length); //Richard
//justin
//print_packet(data+cpy_length, seg_length );
				cpy_length += seg_length;
				lp->rx_descs[start_idx].RXPKT_RDY = RX_OWNBY_FTGMAC100;			///  frame wBz, ٵ hardware
			}
			NetReceive(NetRxPackets[0], packet_length);
#if	FTGMAC100_DEBUG > 4
			DO_PRINT("Receiving Packet\n");
			print_packet( data, packet_length );
#endif
			return packet_length;
		}
	}
	return 0;
}



/*----------------------------------------------------
 . ftmac100_close
 .
 . this makes the board clean up everything that it can
 . and not talk to the outside world.   Caused by
 . an 'ifconfig ethX down'
 .
 -----------------------------------------------------*/
static int ftgmac100_close(struct eth_device *dev)
{
	//netif_stop_queue(dev);
	//dev->start = 0;

	PRINTK2("%s:ftgmac100_close\n", dev->name);


	/* clear everything */
	ftgmac100_shutdown( dev->iobase );

	/* Update the statistics here. */
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif

	return 0;
}


//---PHY CONTROL AND CONFIGURATION-----------------------------------------



#if FTGMAC100_DEBUG > 2
static void print_packet( byte * buf, int length )
{
#if 1
#if FTGMAC100_DEBUG > 3
	int i;
	int remainder;
	int lines;
#endif

	DO_PRINT("Packet of length %d \n", length );

#if FTGMAC100_DEBUG > 3
	lines = length / 16;
	remainder = length % 16;

	for ( i = 0; i < lines ; i ++ ) {
		int cur;

		for ( cur = 0; cur < 8; cur ++ ) {
			byte a, b;

			a = *(buf ++ );
			b = *(buf ++ );
			DO_PRINT("%02x%02x ", a, b );
		}
		DO_PRINT("\n");
	}
	for ( i = 0; i < remainder/2 ; i++ ) {
		byte a, b;

		a = *(buf ++ );
		b = *(buf ++ );
		DO_PRINT("%02x%02x ", a, b );
	}
	DO_PRINT("\n");
#endif
#endif
}
#endif


void ftgmac100_ringbuf_alloc(struct ftgmac100_local *lp)
{
	int i;

	lp->rx_descs = kmalloc( sizeof(RX_DESC)*(RXDES_NUM+1), GFP_DMA|GFP_KERNEL );
	if (lp->rx_descs == NULL )
	{
		DO_PRINT("Receive Ring Buffer allocation error\n");
		BUG();
	}
	lp->rx_descs =  (RX_DESC *)((int)(((char *)lp->rx_descs)+sizeof(RX_DESC)-1)&0xfffffff0);
	lp->rx_descs_dma = virt_to_phys(lp->rx_descs);
	memset(lp->rx_descs, 0, sizeof(RX_DESC)*RXDES_NUM);


	lp->rx_buf = kmalloc( RX_BUF_SIZE*RXDES_NUM, GFP_DMA|GFP_KERNEL );
	lp->rx_buf_dma = virt_to_phys(lp->rx_buf);

	if (lp->rx_buf == NULL || (( (u32)lp->rx_buf & 0x7)!=0) || (((u32)lp->rx_buf_dma & 0x7)!=0)) //Richard
	{
		DO_PRINT("Receive Ring Buffer allocation error, lp->rx_buf = %x\n", lp->rx_buf);
		BUG();
	}

	for (i=0; i<RXDES_NUM; ++i)
	{
#ifndef NC_Body
		lp->rx_descs[i].RXBUF_Size = RX_BUF_SIZE;
#endif
		lp->rx_descs[i].EDORR = 0;							// not last descriptor
		lp->rx_descs[i].RXBUF_BADR = lp->rx_buf_dma+RX_BUF_SIZE*i;
		//lp->rx_descs[i].VIR_RXBUF_BADR = virt_to_phys( lp->rx_descs[i].RXBUF_BADR );//Richard
	}
	lp->rx_descs[RXDES_NUM-1].EDORR = 1;					// is last descriptor


	lp->tx_descs = kmalloc( sizeof(TX_DESC)*(TXDES_NUM+1), GFP_DMA|GFP_KERNEL );
	if (lp->tx_descs == NULL)
	{
		DO_PRINT("Transmit Ring Buffer allocation error\n");
		BUG();
	}
	lp->tx_descs =  (TX_DESC *)((int)(((char *)lp->tx_descs)+sizeof(TX_DESC)-1)&0xfffffff0);
	lp->tx_descs_dma = virt_to_phys(lp->tx_descs);
	memset(lp->tx_descs, 0, sizeof(TX_DESC)*TXDES_NUM);

	lp->tx_buf = kmalloc( TX_BUF_SIZE*TXDES_NUM, GFP_DMA|GFP_KERNEL );
	if (lp->tx_buf == NULL || (( (u32)lp->tx_buf % 4)!=0))
	{
		DO_PRINT("Transmit Ring Buffer allocation error\n");
		BUG();
	}
	lp->tx_buf_dma = virt_to_phys(lp->tx_buf);

	for (i=0; i<TXDES_NUM; ++i)
	{
		lp->tx_descs[i].EDOTR = 0;							// not last descriptor
		lp->tx_descs[i].TXBUF_BADR = lp->tx_buf_dma+TX_BUF_SIZE*i;
		//lp->tx_descs[i].VIR_TXBUF_BADR = virt_to_phys( lp->tx_descs[i].TXBUF_BADR );//Richard
	}
	lp->tx_descs[TXDES_NUM-1].EDOTR = 1;					// is last descriptor
	PRINTK("lp->rx_descs = %x, lp->rx_rx_descs_dma = %x\n", lp->rx_descs, lp->rx_descs_dma);
	PRINTK("lp->rx_buf = %x, lp->rx_buf_dma = %x\n", lp->rx_buf, lp->rx_buf_dma);
	PRINTK("lp->tx_descs = %x, lp->tx_rx_descs_dma = %x\n", lp->tx_descs, lp->tx_descs_dma);
	PRINTK("lp->tx_buf = %x, lp->tx_buf_dma = %x\n", lp->tx_buf, lp->tx_buf_dma);
}


int ftgmac100_initialize(bd_t *bis)
{

	int card_number = 0;
	struct eth_device *dev;





		dev = (struct eth_device *)malloc(sizeof *dev);

		sprintf (dev->name, "FTGMAC#%d", card_number);

		dev->iobase = CONFIG_FTGMAC100_BASE;
		dev->init = ftgmac100_init;
		dev->halt = ftgmac100_close;
		dev->send = ftgmac100_send_packet;
		dev->recv = ftgmac100_rcv;

		eth_register (dev);
		card_number++;



	return card_number;
}



static char ftgmac100_mac_addr[] = {0x00, 0x42, 0x70, 0x00, 0x30, 0x52}; //GM8185.h CONFIG_ETHADDR define

void ftgmac100_phy_rw_waiting(unsigned int ioaddr)
{
	unsigned int tmp;

	do
	{
		mdelay(10);
		tmp =inl(ioaddr + PHYCR_REG);
	}while((tmp&(PHY_READ_bit|PHY_WRITE_bit))>0);
}

static word ftgmac100_read_phy_register(unsigned int ioaddr, byte phyaddr, byte phyreg)
{
	unsigned int tmp;

	tmp =inl(ioaddr + PHYCR_REG);

	tmp &= 0x3000003F;
	tmp |=(phyaddr<<16);
	tmp |=(phyreg<<(16+5));
	tmp |=PHY_READ_bit;

	outl( tmp, ioaddr + PHYCR_REG );

	ftgmac100_phy_rw_waiting(ioaddr);

	return((inl(ioaddr + PHYDATA_REG)>>16));
}

static void ftgmac100_write_phy_register(unsigned int ioaddr,
	byte phyaddr, byte phyreg, word phydata)
{
	unsigned int tmp;

	tmp =inl(ioaddr + PHYCR_REG);

	tmp &= 0x3000003F;
	tmp |=(phyaddr<<16);
	tmp |=(phyreg<<(16+5));
	tmp |=PHY_WRITE_bit;

	outl( phydata, ioaddr + PHYDATA_REG );

	outl( tmp, ioaddr + PHYCR_REG );

	ftgmac100_phy_rw_waiting(ioaddr);

}

int ftgmac100_init(struct eth_device *dev, bd_t *bis)
{
	struct ftgmac100_local *lp;
	int i;
	unsigned int tmp,tmp_1;
	char *s, *e;

	if(PHY_MODE == PHY_MARVELL)
  	printf("FTMAC with MARVELL PHY support\n");
	else if(PHY_MODE == PHY_ICPLUS)
  	printf("FTMAC with ICPLUS PHY IP1001 support\n");
	else if(PHY_MODE == PHY_ICPLUS_100)
  	printf("FTMAC with ICPLUS PHY IP101A support\n");
	else if(PHY_MODE == PHY_AR)
  	printf("FTMAC with AR PHY support\n");
  	    
	if (initialized == 0)
	{
		initialized = 1;

		dev->iobase = CONFIG_FTGMAC100_BASE;
		/* Initialize the private structure. */
		dev->priv = (void *)malloc(sizeof(struct ftgmac100_local));
		if (dev->priv == NULL)
		{
			DO_PRINT("out of memory\n");
			return 0;
		}
#ifdef CONFIG_PLATFORM_GM8181
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x6C) |= (1<<16);//output mode enable
		*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x7C) &= ~(1<<20);
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x38) = (*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x38)) & ~(1<<8);// turn on GMAC clock
#ifdef UART4_CLOCK
		//enable clock
		*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x3c) &= ~(1<<20);
		*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x3c) &= ~(1<<4);
#ifndef has_osc
		//GPIO2 reset PHY
		*(volatile unsigned int *)(CONFIG_GPIO_BASE + 0x08) |= (1<<0);
		*(volatile unsigned int *)(CONFIG_GPIO_BASE + 0x00) &= ~(1<<0);
#endif
		//set UART4
		*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x44) |= 0x2;
		tmp = *(volatile unsigned int *)(CONFIG_PMU_BASE + 0x50) & 0xFFF3FFFF;
		*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x50) = tmp | (0x3 << 18);
		//set PLL2 clock
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) &= 0xFFFF;
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) |= (0x2195 << 16);
		//set div value
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70) = (*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70)) & (~0x3E000000);// set clock
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70) = (*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70)) | (0x5 << 25);
		//tmp = *(volatile unsigned int *)(CONFIG_PMU_BASE + 0x78) & 0xFFE07FFF;
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x78) = tmp | (4 << 15);
#ifndef has_osc
		for(i=0;i<10;i++)
			udelay(10000);

		*(volatile unsigned int *)(CONFIG_GPIO_BASE + 0x00) |= (1<<0);

		for(i=0;i<300;i++)
			udelay(10000);
#endif
#endif
#endif

#if 0
		tmp = ftgmac100_read_phy_register(dev->iobase, PHY_ADDR, 0x1);
		printf("read_phy reg 0x1=0x%x\n",tmp);
#endif

		/// --------------------------------------------------------------------
		///		ftmac100_local
		/// --------------------------------------------------------------------
		memset(dev->priv, 0, sizeof(struct ftgmac100_local));
		strcpy(dev->name, "eth0");
		//copy the mac address
		memcpy(dev->enetaddr,ftgmac100_mac_addr,6);
		lp = (struct ftgmac100_local *)dev->priv;
		lp->maccr_val = FULLDUP_bit | CRC_APD_bit | RXMAC_EN_bit | TXMAC_EN_bit  | RXDMA_EN_bit	| TXDMA_EN_bit|RX_BROADPKT_bit;

#ifndef RMII_MODE
#if 0		//print PHY status
		for(i = 0; i <= 31; i++)
		{
				tmp = ftgmac100_read_phy_register(dev->iobase, PHY_ADDR, i);
				printf("read_phy reg %d=0x%x\n",i, tmp);
		}
#endif
#if 0//internal loopback test
		tmp = ftgmac100_read_phy_register(dev->iobase, PHY_ADDR, 0x0);
		tmp &= ~(1<<6);
		tmp &= ~(1<<12);
		tmp &= ~(1<<13);
		//tmp |= (1<<14);
		ftgmac100_write_phy_register(dev->iobase, PHY_ADDR, 0x0, tmp);
		printf("set reg 0=%x\n",tmp);
		tmp = ftgmac100_read_phy_register(dev->iobase, PHY_ADDR, 0x0);
		printf("reg reg 0=%x\n",tmp);
#endif
		tmp = ftgmac100_read_phy_register(dev->iobase, PHY_ADDR, 0x11);
		//printk("read_phy=%x\n",tmp);
		if(PHY_MODE == PHY_MARVELL)	
			tmp = (tmp & PHY_SPEED_mask)>>14;

		if(PHY_MODE == PHY_ICPLUS)
		{		
			if((tmp & 0x1000) == 0)
			{
					tmp_1 = inl( dev->iobase + MACCR_REG );
					tmp_1 &= ~FULLDUP_bit;
					lp->maccr_val &= ~FULLDUP_bit;
					outl(tmp_1, dev->iobase + MACCR_REG );
					printf("HALF\n");	
			}
			else
			{
					tmp_1 = inl( dev->iobase + MACCR_REG );
					tmp_1 |= FULLDUP_bit;
					lp->maccr_val |= FULLDUP_bit;
					outl(tmp_1, dev->iobase + MACCR_REG );
					printf("FULL\n");	
			}		
			tmp = (tmp & 0x6000)>>13;					
		}	
		else if(PHY_MODE == PHY_AR)
		{		
			if((tmp & 0x2000) == 0)
			{
					tmp_1 = inl( dev->iobase + MACCR_REG );
					tmp_1 &= ~FULLDUP_bit;
					lp->maccr_val &= ~FULLDUP_bit;
					outl(tmp_1, dev->iobase + MACCR_REG );
					printf("HALF\n");	
			}
			else
			{
					tmp_1 = inl( dev->iobase + MACCR_REG );
					tmp_1 |= FULLDUP_bit;
					lp->maccr_val |= FULLDUP_bit;
					outl(tmp_1, dev->iobase + MACCR_REG );
					printf("FULL\n");	
			}		
			tmp = (tmp & 0x6000)>>14;					
		}		
		else if(PHY_MODE == PHY_ICPLUS_100)
		{
			tmp_1 = ftgmac100_read_phy_register(dev->iobase, PHY_ADDR, 0x5 );
			tmp = (tmp_1 & 0x80)>>7;
			
			if((tmp_1 & 0x140) == 0)
			{
					tmp_1 = inl( dev->iobase + MACCR_REG );
					tmp_1 &= ~FULLDUP_bit;
					lp->maccr_val &= ~FULLDUP_bit;
					outl(tmp_1, dev->iobase + MACCR_REG );
					printf("HALF\n");	
			}
			else
			{
					tmp_1 = inl( dev->iobase + MACCR_REG );
					tmp_1 |= FULLDUP_bit;
					lp->maccr_val |= FULLDUP_bit;
					outl(tmp_1, dev->iobase + MACCR_REG );
					printf("FULL\n");	
			}		
		}

		if(tmp == PHY_SPEED_1G)
		{
#ifdef CONFIG_PLATFORM_GM8181
			//set div value
			//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70) = (*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70)) & (~0x3E000000);// set clock
			//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70) = (*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x70)) | (0x5 << 25);
			//set GMII clock
			*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x6C) |= (1<<17);
			//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) &= 0xFFFF;
			//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) |= (0x27DD << 16);

//#ifdef UART4_CLOCK
			//GPIO2 reset PHY
			//*(volatile unsigned int *)(CONFIG_GPIO_BASE + 0x00) &= ~(1<<0);
			//set div value
			//tmp = *(volatile unsigned int *)(CONFIG_PMU_BASE + 0x78) & 0xFFE07FFF;
			//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x78) = tmp | (4 << 15);
			//printf("reset clock again\n");
			//*(volatile unsigned int *)(CONFIG_GPIO_BASE + 0x00) |= (1<<0);
//#endif
#endif
#ifdef CONFIG_PLATFORM_GM8185_v2
			tmp = *(volatile unsigned int *)(CONFIG_PMU_BASE + 0x118) & 0x1FF;
			*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x118) = tmp | (0x3E8F07 << 8);
#endif
			printk("PHY_SPEED_1G\n");
			lp->maccr_val |= (GMAC_MODE_bit | SPEED_100);

			//change RX clock delay value
			tmp = ftgmac100_read_phy_register(dev->iobase, PHY_ADDR, 0x10);
			ftgmac100_write_phy_register(dev->iobase, PHY_ADDR, 0x10, tmp & 0xFFFFFFFE);
			//printf("read reg 16=%x\n",tmp);
		}
		else
		{
#ifdef CONFIG_PLATFORM_GM8181
//#ifndef UART4_CLOCK
			//set MII clock
			//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) &= 0xFFFF;
			//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) |= (0x2195 << 16);
//#endif
//			*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x6C) |= (1<<17);
#endif
#ifdef CONFIG_PLATFORM_GM8185_v2
	tmp = *(volatile unsigned int *)(CONFIG_PMU_BASE + 0x118) & 0x1FF;
	*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x118) = tmp | (0x0C890B << 8);
#endif
			printk("PHY_SPEED_100M\n");
			lp->maccr_val &= ~GMAC_MODE_bit;
			lp->maccr_val |= SPEED_100;
		}
#else
#ifdef CONFIG_PLATFORM_GM8181
		//set RMII clock
		*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x6C) |= (1<<17);
		*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x6C) |= (1<<12);
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) &= 0xFFFF;
		//*(volatile unsigned int *)(CONFIG_PMU_BASE + 0x34) |= (0x2197 << 16);
#endif
		printk("RMII mode\n");
		lp->maccr_val &= ~GMAC_MODE_bit;
		lp->maccr_val |= SPEED_100;
#endif
		//printk("MACCR=%x\n",lp->maccr_val);

		ftgmac100_ringbuf_alloc(lp);

		s = getenv ("ethaddr");
	if (s == NULL){
		//copy the default mac address
		memcpy(dev->enetaddr,ftgmac100_mac_addr,6);
	}
	else{
		for(i = 0; i < 6; i++) {
			dev->enetaddr[i] = s ? simple_strtoul (s, &e, 16) : 0;
			if (s){
				s = (*e) ? e + 1 : e;
			}
		}
	}
	}
	ftgmac100_open(dev);

	return 0;
}


#endif
