/*
 * A common driver to support Winbond SPI FLash, the current support list
 * should refer the array, winbond_spi_flash_table[], below
 * $Author: wdshih $
 * $Log: winbond.c,v $
 * Revision 1.11  2011/07/15 07:52:15  wdshih
 * fix multi-probe abort issue
 *
 * Revision 1.10  2011/07/13 07:52:56  wdshih
 * fix command buf_addr not initial and data abort issue
 *
 * Revision 1.9  2011/07/07 03:36:23  wdshih
 * add MX25L256 spi flash
 *
 * Revision 1.8  2011/06/27 01:11:13  wdshih
 * fix SPI flash write busy issue
 *
 * Revision 1.7  2011/03/25 05:52:11  wdshih
 * add spi flash
 *
 * Revision 1.6  2011/01/19 11:52:59  mars_ch
 * *: fix probe search bug
 *
 * Revision 1.5  2011/01/18 09:34:47  mars_ch
 * *: add MXIC SPI flash and speed up SPI flash booting
 *
 * Revision 1.4  2010/10/22 03:10:38  mars_ch
 * *: temp hack to make spi flash erase work
 *
 * Revision 1.3  2010/07/22 07:49:27  mars_ch
 * *: add winbond compatibale spi flash support
 *
 * Revision 1.2  2010/07/07 05:39:43  mars_ch
 * *: add spi flash W25Q128BV support
 *
*/
#include <common.h>
#include <malloc.h>
#include <spi_flash.h>

#include <environment.h>//Gary
#include "spi_flash_internal.h"

#define WINBOND_SPI_FLASH_JEDEC_M_ID    0xEF
#define MXIC_SPI_FLASH_JEDEC_M_ID       0xC2
#define NUMONYX_SPI_FLASH_JEDEC_M_ID    0x20
#define EON_SPI_FLASH_JEDEC_M_ID    		0x1C
#define SELF_SPI_FLASH_JEDEC_M_ID       0x68

#define CMD_W25_WREN		0x06	/* Write Enable */
#define CMD_W25_WRDI		0x04	/* Write Disable */
#define CMD_W25_RDSR		0x05	/* Read Status Register */
#define CMD_W25_WRSR		0x01	/* Write Status Register */
#define CMD_W25_READ		0x03	/* Read Data Bytes */
#define CMD_W25_FAST_READ	0x0b	/* Read Data Bytes at Higher Speed */
#define CMD_W25_PP		0x02	/* Page Program */
#define CMD_W25_SE		0x20	/* Sector (4K) Erase */
#define CMD_W25_BE		0xd8	/* Block (64K) Erase */
#define CMD_W25_CE		0xc7	/* Chip Erase */
#define CMD_W25_DP		0xb9	/* Deep Power-down */
#define CMD_W25_RES		0xab	/* Release from DP, and Read Signature */

#define WINBOND_ID_NOCHECK		0x7878
#define WINBOND_ID_W25P16		0x2015
#define WINBOND_ID_W25X16		0x3015
#define WINBOND_ID_W25X32		0x3016
#define WINBOND_ID_W25X64		0x3017
#define WINBOND_ID_W25Q64CV		0x4017
#define WINBOND_ID_W25Q128BV	0x4018

#define MXIC_ID_MX25L12845E     0x2018
#define MXIC_ID_MX25L25635E     0x2019
#define NUMONYX_ID_N25Q128      0xBA18
#define EON_ID_EN25Q128     		0x3018
#define SELF_ID_MX25L12845E     0x1234

#define WINBOND_SR_WIP		(1 << 0)	/* Write-in-Progress */
#define WINBOND_SR_WEL		(1 << 1)	/* Write-enable latency */

struct winbond_spi_flash_params {
	uint16_t	id;
	/* Log2 of page size in power-of-two mode */
	uint8_t		byte_mode;	/* 3: 3 byte mode, 4:4 byte mode(over 128Mb flash) */
	uint8_t		l2_page_size;
	uint16_t	pages_per_sector;
	uint16_t	sectors_per_block;
	uint16_t	nr_blocks;
	const char	*name;
};

struct winbond_spi_flash {
	struct spi_flash flash;	
	const struct winbond_spi_flash_params *params;
};

inline struct winbond_spi_flash *
to_winbond_spi_flash(struct spi_flash *flash)
{
	return container_of(flash, struct winbond_spi_flash, flash);
}

static const struct winbond_spi_flash_params winbond_spi_flash_table[] = {
	{
		.id			= WINBOND_ID_NOCHECK,
		.byte_mode			= 3,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 256,
		.name			= "WB_NOCHECK",
	},
	{
		.id			= WINBOND_ID_W25X16,
		.byte_mode			= 3,
		.l2_page_size		= 8,
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 32,
		.name			= "W25X16",
	},
	{
		.id			= WINBOND_ID_W25X32,
		.byte_mode			= 3,
		.l2_page_size		= 8,
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 64,
		.name			= "W25X32",
	},
	{
		.id			= WINBOND_ID_W25X64,
		.byte_mode			= 3,
		.l2_page_size		= 8,
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 128,
		.name			= "W25X64",
	},
	{ //ycmo091110
		.id			= WINBOND_ID_W25P16,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 256,
		.sectors_per_block	= 32,
		.nr_blocks		= 1,
		.name			= "W25X64",
	},
	{ //Mars Cheng 20101020
		.id			= WINBOND_ID_W25Q64CV,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 128,
		.name			= "W25Q64CV",
	},
	{ //Mars Cheng 20100707
		.id			= WINBOND_ID_W25Q128BV,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 256,
		.name			= "W25Q128BV",
	}
};

static const struct winbond_spi_flash_params mxic_spi_flash_table[] = {
	{ //Mars Cheng 20101220
		.id			= MXIC_ID_MX25L12845E,
		.byte_mode			= 3,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 256,
		.name			= "MX25L12845E",
	},
	{ 
		.id			= MXIC_ID_MX25L25635E,
		.byte_mode			= 4,
		.l2_page_size		= 9, //512bytes
		.pages_per_sector	= 8,
		.sectors_per_block	= 16,
		.nr_blocks		= 512,
		.name			= "MX25L25635E",
	},	
};

static const struct winbond_spi_flash_params eon_spi_flash_table[] = {
	{ 
		.id			= EON_ID_EN25Q128,
		.byte_mode			= 3,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 256,
		.name			= "EN25Q128",
	},
};

static const struct winbond_spi_flash_params numonyx_spi_flash_table[] = {
	{ //Mars Cheng 20101220
		.id			= NUMONYX_ID_N25Q128,
		.byte_mode			= 3,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 256,
		.name			= "N25Q128",
	},
};

static const struct winbond_spi_flash_params SELF_spi_flash_table[] = {
	{ 
		.id			= SELF_ID_MX25L12845E,
		.byte_mode			= 3,
		.l2_page_size		= 8, //256bytes
		.pages_per_sector	= 16,
		.sectors_per_block	= 16,
		.nr_blocks		= 256,
		.name			= "SELF",
	},
};

static int winbond_wait_ready(struct spi_flash *flash, unsigned long check_bit, unsigned long timeout)
{
	struct spi_slave *spi = flash->spi;
	unsigned long timebase;
	int ret;
	u8 status;
	u8 cmd[5] = { CMD_W25_RDSR, 0xff, 0xff, 0xff, 0xff};

    if ((check_bit != WINBOND_SR_WIP) && (check_bit != WINBOND_SR_WEL)) {
        debug("SF: Fatal error: check_bit is not WINBOND_SR_WIP or WINBOND_SR_WEL!!\n");
        while(1);
    }

	set_wsize(8);
	ret = spi_xfer(spi, 32, &cmd[0], NULL, SPI_XFER_BEGIN);
	if (ret) {
		debug("SF: Failed to send command %02x: %d\n", cmd, ret);
		return ret;
	}

	timebase = get_timer(0);
	do {
		ret = spi_xfer(spi, 8, NULL, &status, 0);
		if (ret) {
			debug("SF: Failed to get status for cmd %02x: %d\n", cmd, ret);
			return -1;
		}

		if ((status & check_bit) == 0)
			break;

	} while (get_timer(timebase) < timeout);

	spi_xfer(spi, 0, NULL, NULL, SPI_XFER_END);

	if ((status & check_bit) == 0)
		return 0;

	debug("SF: Timed out on command %02x: %d\n", cmd, ret);
	/* Timed out */
	return -1;
}

static int spi_4byte_mode(struct spi_flash *flash, int enable) 
{ 
	struct winbond_spi_flash *stm = to_winbond_spi_flash(flash);
	int ret = -1;
  u8 code; 

	if(stm->params->byte_mode == 4){
	  if (enable) 
	  		code = 0xB7; /* EN4B, enter 4-byte mode */ 
	  else 
	 		code = 0xE9; /* EX4B, exit 4-byte mode */ 
	
		ret = spi_flash_read_common(flash, &code, 1, 0, 0);
		if (ret < 0) {
			debug("SF: Set 4 byte mode failed\n");
			goto exit;
		}  
	}
	ret = 0;
exit:	
  return ret; 
} 

/*
 * Assemble the address part of a command for Winbond devices in
 * non-power-of-two page size mode.
 */
static void winbond_build_address(struct winbond_spi_flash *stm, u8 *cmd, u32 offset)
{
	if(stm->params->byte_mode == 4){
		cmd[0] = (offset >> 24) & 0xFF;
		cmd[1] = (offset >> 16) & 0xFF;
		cmd[2] = (offset >> 8) & 0xFF;
		cmd[3] = offset & 0xFF;		
	}else {
		cmd[0] = (offset >> 16) & 0xFF;
		cmd[1] = (offset >> 8) & 0xFF;
		cmd[2] = offset & 0xFF;
	}
}

static int winbond_read_fast(struct spi_flash *flash,
		u32 offset, size_t len, void *buf)
{
	struct winbond_spi_flash *stm = to_winbond_spi_flash(flash);
	u8 cmd[5];

	cmd[0] = CMD_READ_ARRAY_FAST;
	winbond_build_address(stm, cmd + 1, offset);

	return spi_flash_read_common(flash, cmd, stm->params->byte_mode + 1, buf, len);
}

static int winbond_read(struct spi_flash *flash,
		u32 offset, size_t len, void *buf)
{
    int ret = -1;
	struct winbond_spi_flash *stm = to_winbond_spi_flash(flash);
	u8 cmd[5];

	spi_4byte_mode(flash, 1);

	cmd[0] = CMD_READ_ARRAY_SLOW;
	winbond_build_address(stm, cmd + 1, offset);

    ret = winbond_wait_ready(flash, WINBOND_SR_WIP, SPI_FLASH_PROG_TIMEOUT);
    if (ret < 0) {
        debug("SF: Winbond timed out\n");
    }
  ret = spi_flash_read_common(flash, cmd, stm->params->byte_mode + 1, buf, len);
  spi_4byte_mode(flash, 0);
  
	return ret;
}

static int winbond_write(struct spi_flash *flash,
		u32 offset, size_t len, const void *buf)
{
	struct winbond_spi_flash *stm = to_winbond_spi_flash(flash);
	unsigned long byte_addr;
	unsigned long write_size;
	size_t chunk_len;
	size_t actual;
	int ret;
	u32 offset_addr;
	u8 cmd[5];

	write_size = 256; /* page program one time 256 data bytes */
	byte_addr = offset % write_size;
	offset_addr = offset;

	ret = spi_4byte_mode(flash, 1);
	if (ret)
		goto out;
		
	ret = spi_claim_bus(flash->spi);
	if (ret) {
		debug("SF: Unable to claim SPI bus\n");
		goto out;
	}

	for (actual = 0; actual < len; actual += chunk_len) {
		
		chunk_len = min(len - actual, write_size - byte_addr);


		cmd[0] = CMD_W25_PP;
		
  	if(stm->params->byte_mode == 4){
  		cmd[1] = (offset_addr >> 24) & 0xFF;
  		cmd[2] = (offset_addr >> 16) & 0xFF;
  		cmd[3] = (offset_addr >> 8) & 0xFF;
  		cmd[4] = offset_addr & 0xFF;		
  		
  		debug("PP: 0x%p => cmd = { 0x%02x 0x%02x%02x%02x%02x } chunk_len = %d\n",
  			buf + actual,	cmd[0], cmd[1], cmd[2], cmd[3], cmd[4], chunk_len);		
  	}else {
  		cmd[1] = (offset_addr >> 16) & 0xFF;
  		cmd[2] = (offset_addr >> 8) & 0xFF;
  		cmd[3] = offset_addr & 0xFF;
  		
  		debug("PP: 0x%p => cmd = { 0x%02x 0x%02x%02x%02x } chunk_len = %d\n",
  			buf + actual,	cmd[0], cmd[1], cmd[2], cmd[3], chunk_len);		
  	}
					
		ret = spi_flash_cmd(flash->spi, CMD_W25_WREN, NULL, 0);
		if (ret < 0) {
			debug("SF: Enabling Write failed\n");
			goto out;
		}

		ret = spi_flash_cmd_write(flash->spi, cmd, stm->params->byte_mode + 1,
				buf + actual, chunk_len);
		if (ret < 0) {
			debug("SF: Winbond Page Program failed\n");
			goto out;
		}

		ret = winbond_wait_ready(flash, WINBOND_SR_WEL, SPI_FLASH_PROG_TIMEOUT);
		if (ret < 0) {
			debug("SF: Winbond page programming timed out\n");
			goto out;
		}

		offset_addr += chunk_len;
		byte_addr = 0;
	}
		
	debug("SF: Winbond: Successfully programmed %u bytes @ 0x%x\n",
			len, offset);
	ret = 0;

out:
	spi_release_bus(flash->spi);
	spi_4byte_mode(flash, 0);
	
	return ret;
}

extern void spi_swap_disable(void);
extern void spi_swap_enable(void);

int winbond_erase(struct spi_flash *flash, u32 offset, size_t len)
{
	struct winbond_spi_flash *stm = to_winbond_spi_flash(flash);
	unsigned long sector_size;
	unsigned int page_shift;
	size_t actual;
	int ret;
	u8 cmd[5];

	/*
	 * This function currently uses sector erase only.
	 * probably speed things up by using bulk erase
	 * when possible.
	 */
	
//	printf("%s : offset: %p  len:%p \n",__func__,offset,len);
	page_shift = stm->params->l2_page_size;
	sector_size = (1 << page_shift) * stm->params->pages_per_sector;

//	printf("%s : sector_size: %p  \n",__func__,sector_size);
	if (offset % sector_size || len % sector_size) {
		debug("SF: Erase offset/length not multiple of sector size\n");
		return -1;
	}

	len /= sector_size;
	cmd[0] = (sector_size==0x10000)? CMD_W25_BE: CMD_W25_SE; //ycmo0911: W25P16 sector_size = 64KB

	ret = spi_4byte_mode(flash, 1);
	if (ret)
		goto out_erase;
		
	ret = spi_claim_bus(flash->spi);
	if (ret) {
		debug("SF: Unable to claim SPI bus\n");
		goto out_erase;
	}

	for (actual = 0; actual < len; actual++) {
		winbond_build_address(stm, &cmd[1], offset + actual * sector_size);
		if(stm->params->byte_mode == 4)
			printf("Erase: %02x %02x %02x %02x %02x\n", cmd[0], cmd[1], cmd[2], cmd[3], cmd[4]);			
		else
			printf("Erase: %02x %02x %02x %02x\n", cmd[0], cmd[1], cmd[2], cmd[3]);
		
		spi_swap_disable();
		ret = spi_flash_cmd(flash->spi, CMD_W25_WREN, NULL, 0);
		if (ret < 0) {
			debug("SF: Enabling Write failed\n");
			goto out_erase;
		}

		ret = spi_flash_cmd_write(flash->spi, cmd, stm->params->byte_mode + 1, NULL, 0);
		if (ret < 0) {
			debug("SF: Winbond sector erase failed\n");
			goto out_erase;
		}

		ret = winbond_wait_ready(flash, WINBOND_SR_WEL, (2*CFG_HZ_CLOCK));	/* Up to 2 seconds */
		if (ret < 0) {
			debug("SF: Winbond sector erase timed out\n");
			goto out_erase;
		}
    spi_swap_enable();		
	}


	debug("SF: Winbond: Successfully erased %u bytes @ 0x%x\n",
			len * sector_size, offset);
	ret = 0;

out_erase:
	spi_release_bus(flash->spi);
	spi_4byte_mode(flash, 0);
	return ret;
}

struct spi_flash* spi_flash_probe_nocheck(struct spi_slave *spi)
{
	
	const struct winbond_spi_flash_params *params = NULL;
	unsigned long page_size = 0;
	struct winbond_spi_flash *stm = NULL;

	params = &winbond_spi_flash_table[0];

	stm = malloc(sizeof(struct winbond_spi_flash));
	if (!stm) {
		debug("SF: Failed to allocate memory\n");
		return NULL;
	}

	stm->params = params;
	stm->flash.spi = spi;
	stm->flash.name = params->name;

	/* Assuming power-of-two page size initially. */
	page_size = 1 << params->l2_page_size;

	stm->flash.write = winbond_write;
	stm->flash.erase = winbond_erase;
//	stm->flash.read = winbond_read_fast;
	stm->flash.read = winbond_read; //ycmo091110
	stm->flash.size = page_size * params->pages_per_sector
				* params->sectors_per_block
				* params->nr_blocks;

	return &stm->flash;
}

struct spi_flash* spi_flash_probe_winbond(struct spi_slave *spi, u8 *idcode)
{
	const struct winbond_spi_flash_params *params;
    int table_size = 0;
	unsigned long page_size;
	struct winbond_spi_flash *stm;
	unsigned int i;

    if (idcode[0] == WINBOND_SPI_FLASH_JEDEC_M_ID) {
        params = winbond_spi_flash_table;
        table_size = ARRAY_SIZE(winbond_spi_flash_table);
    } else if (idcode[0] == MXIC_SPI_FLASH_JEDEC_M_ID) {
        params = mxic_spi_flash_table;
        table_size = ARRAY_SIZE(mxic_spi_flash_table);
    } else if (idcode[0] == EON_SPI_FLASH_JEDEC_M_ID) {
        params = eon_spi_flash_table;
        table_size = ARRAY_SIZE(eon_spi_flash_table);        
    } else if (idcode[0] == NUMONYX_SPI_FLASH_JEDEC_M_ID) {
        params = numonyx_spi_flash_table;
        table_size = ARRAY_SIZE(numonyx_spi_flash_table);
    } else if (idcode[0] == SELF_SPI_FLASH_JEDEC_M_ID) {
        params = SELF_spi_flash_table;
        table_size = ARRAY_SIZE(SELF_spi_flash_table);          
    } else {
        printf("SF: Unsupported ID 0x%x02x%02x%02x\n", idcode[0], idcode[1], idcode[2]);
        return NULL;
    }

    for (i = 0; i < table_size; i++) {
        if (params->id == ((idcode[1] << 8) | idcode[2])) {
            break;
        }
        params++;
    }

    if (i == table_size) {
		debug("SF: Unsupported ID 0x%x02x%02x%02x\n", idcode[0], idcode[1], idcode[2]);
		return NULL;
    }
    
	stm = malloc(sizeof(struct winbond_spi_flash));
	if (!stm) {
		debug("SF: Failed to allocate memory\n");
		return NULL;
	}

	stm->params = params;
	stm->flash.spi = spi;
	stm->flash.name = params->name;

	/* Assuming power-of-two page size initially. */
	page_size = 1 << params->l2_page_size;

	stm->flash.write = winbond_write;
	stm->flash.erase = winbond_erase;
//	stm->flash.read = winbond_read_fast;
	stm->flash.read = winbond_read; //ycmo091110
	stm->flash.size = page_size * params->pages_per_sector
				* params->sectors_per_block
				* params->nr_blocks;

	debug("SF: Detected %s with page size %u, total %u bytes\n",
			params->name, page_size, stm->flash.size);

	return &stm->flash;
}
