#!/bin/sh

# WRAPPER SCRIPT FOR TZO COMMAND LINE CLIENT
#
# Copyright 2007-8 Tzolkin Corporation.
# Licensed under terms of GNU GPL version 2
# Alternate licensing is available if requested.
#
# Version: Same as tzoupdate --help
#
# Author: Scott Prive
#
# This Wrapper is an integral part of the TZO Update for UNIX/Linux client.
# Please do not bypass this wrapper or invoke tzoupdate.c directly.

# If you had to do something special to install this on an untested OS, or
# unsupported toolchain (such as Sun or Microsoft C compliler) please let us know.
# devsupport@tzo.com
#
# /tmp/tzo-killfile
# This file helps SHUT DOWN the TZO client in event of a major problem.
# Please DO NOT delete this file without understanding what the problem was (usually it is bad authentication)


# BEGIN FUNCTIONS -- SCROLL DOWN FOR THE MAIN CODE

# detect_os - dynamically set OSSTRING to value of: OS, provider, distribution name, kernel-version, CPUtype.
# For inclusion in HTTP USER-Agent variable.
# Return format:  [ OS <distribution release> (kernel-version CPU-arch) ]
detect_os()
{

    if [ -f ${LSBRELEASE} ]; then
        # Info not currently provided by lsb_release
        OS=`uname -s` 				# OS name: "Linux" 
        DIST=`lsb_release -i|cut -d: -f2`	# distributor: "CentOS"
        REV=`lsb_release -r|cut -d: -f2`	# release: "4.4"
        PSEUDONAME=`lsb_release -d|cut -d: -f2`
        KERNEL=`uname -r`
        MACH=`uname -m`
    
        # Final string
        OSSTRING="${OS}<${DIST}${REV}>(${PSUEDONAME} ${KERNEL} ${MACH})"
    elif [ -x ${TZODETECT} ]; then
        # Older or non-LSB systems such as Ubuntu 7.04 lack the lsb_release utility.
        # tzodetect.sh will recover the same information.
        OSSTRING=`${TZODETECT}`
    else
        # this shouldn't happen, but does if installed by hand, incompletely.
        OSSTRING='BAD_INSTALL: tzoupdate v1.12, no tzodetect.sh found, no lsb_release'
        #exit 127
    fi
}

# get the location of IPFILE (IP address cookie)
detect_log_filename()
{
    # if -f supplied,  grep that file for the log name.
    if [ -f "${TZOCONFIG}" ]; then
        # This may not be portable. I did not put any effort into
        # stripping end of line for Windows or MacOS
        IPTMP=`grep IPFILE ${TZOCONFIG}`
        IPFILE=`echo $IPTMP | tr -d 'IPFILE='|tr -d '\015'` #also eliminate \r
    else
        echo "ERROR: Could not open TZO config file (${TZOCONFIG}). Aborting.."
    fi
}

# detect age of IP cookie file.
# This helps throttle updates if the user restarts the client. This helps avoid the
# rare (but confusing) 'IP blocked for 60 seconds' Response if 2 updates arrive close together.
detect_ipfile_age()
{
    if [ -f ${IPFILE} ]; then
        TIME_FILE=`stat --format='%Y' $IPFILE`
	#workaround if OS has no 'stat' command (Solaris, Scratchbox, OS X)
	if [ "x${TIME_FILE}" == "x" ]; then
		TIME_FILE = 0
	fi
        TIME_CLOCK=`date +%s`
        IPFILE_AGE=`expr ${TIME_CLOCK} "-" ${TIME_FILE}`
    else 
        # file does not exist, so make assumption this is a new install (no wait)
        IPFILE_AGE=60
    fi
    # Is IPFILE age < 60 seconds? Force 60 second wait to be safe..
    if [ $IPFILE_AGE -lt 60 ]; then
        #echo "Hmm. You just tried updating the server 60 seconds ago."
        #echo "To prevent server abuse, script will pause for 60 seconds."
        sleep 60
    fi
}

# detect_killfile - checks for shutdown file, created on fatal configuration or account errors.
# Owner of installation must fix the problem AND remove the killfile
detect_killfile()
{
	if [ -f ${KILLFILE} ]; then
		# We don't suppress this error even if QUIETMODE
	    MESSAGE="ERROR: Killfile found (${KILLFILE}). Please FIX your TZO config/account (then remove killfile)."
		echo "${MESSAGE}" > ${TZOLOG}
	    exit 3
	fi
}

# create_killfile - In event of problem requiring user input, this halts TZO client and prevents
# any restart. Example is if the TZO account is EXPIRED, this file keeps the client from running.
# manually remove the killfile once the original problem is solved.
create_killfile()
{
	if [ $RETURNCODE -eq 2 ]; then
		MSG="TZO Account shutdown. Reason: tzoupdate exited with code 2. Expired TZO??"
	elif [ $RETURNCODE -eq 3 ]; then
		MSG="TZO Account shutdown. Reason: tzoupdate exited with code 3. Bad host/key/email email in tzoconfig??"
	fi
	echo "${MSG}" > ${KILLFILE}
	echo "${OUTPUT}" >> ${KILLFILE}
}

# detect_args - This function is depricated, vestige function from the first tzoupdate version.
# Function attempted to guess location of tzoupdate.conf.
# This newest version of tzoupdate uses /etc/, and rpm/deb packages help install .conf there.
detect_args()
{

	if [ $# -eq 0 ]; then
	    # no arguments? OK asume TZOCONFIG is in same dir
	    TZOARGS="-f ${TZOCONFIG}"
	else
	    TZOARGS="$@"
	fi
}

# invoke_tzoupdate - calls basic TZO update library, captures text output, and examines Exit code.
invoke_tzoupdate()
{
    if [ ! -x "${TZOUPDATE}" ]; then
        echo "ERROR: The TZO file (${TZOUPDATE}) is NOT executable. Exiting.."
        exit 127
    fi
	OUTPUT=`${TZOUPDATE} ${TZOARGS} -u "${OSSTRING}" -v`   #verbose output of tzoupdate
	RETURNCODE=$?           # exit code
}



###########################################################
# START WRAPPER SCRIPT - MAIN
###########################################################

# NOTES:
# 1) Installation is usually via RPM or DEB file, solving most install issues.
#    If you install manually, please review INSTALL file.
# 2) Paths must be 100% fully-qualified.

# DAEMON mode:
# possible values: 1 or 0
# Should always be 1 because this client is intended to run in the background continually.
#
DAEMON="1"

# LOCATIONS
KILLFILE="/tmp/tzo-killfile"    # cookie/killfile if major/fatal update error
TZOUPDATE="/usr/local/bin/tzoupdate"         # path to TZO Update
TZOCONFIG="/etc/tzoupdate.conf"  	# Place for all TZO settings
IPFILE=""                       # This location gets set later (parsed from TZOCONFIG).
TZOLOG="/tmp/tzolog.txt"        # Save the output of this client here
# OS info
LSBRELEASE="/usr/bin/lsb_release"  	    # Detecting OS on newer (LSB-compliant) Linux
TZODETECT="/usr/local/bin/tzodetect.sh" # Detecting OS, if non-LSB system

# Get TZO domainname, sanity-check the value
DOMAIN=`grep DOMAIN $TZOCONFIG|cut -f2 -d=`
if [ $DOMAIN = "example.tzo.com" ]; then
	# Error - $TZOCONFIG is is still configured with example.tzo.com. User must fix config
	echo "ERROR: Config file $TZOCONFIG is invalid (contains domain $DOMAIN)."
	echo "Please enter your TZO domain into $TZOCONFIG. Now exiting.."
	exit 127
fi


# DAEMON CHECKING
# If KILLFILE is found, UN-SET DAEMON variable so this client can shut down 
if [ $DAEMON -eq 1 ] && [ ! -f "$KILLFILE" ]; then
	KEEPRUNNING=1
else
	KEEPRUNNING=0
fi

while [ $KEEPRUNNING -eq 1 ] && [ ! -f "$KILLFILE" ]
do
    # the 'do' continues to end of script...
    
    # IF $returncode exists, this is the second (or greater) pass through the loop... meaning we are in daemon
    # mode and at least 1 pass. Therefore this is a good place to put the mandatory 10 minute waite between
    # each 'run' of tzoupdate. The mandatory 10 minute wait prevents runaway execution or client abuse.
    if [ "xr$RETURNCODE" != "xr" ]; then
        # If return code exists, must not be our first run. Sleep now 600 seconds.
        sleep 600
    else
        # No return code, must be our first time, continue on
        sleep 1
    fi
    
    # QUIETMODE - Debug variable for running inside a test shell. This manages output levels.
    # Use 1 to suppress all STDOUT output except critical errors; 0 for more  extra debug information.
    # QUIETMODE has NO impact on disk logging to $TZOLOG.
    
    QUIETMODE=1;
    RETURNCODE=""                   # Initialize.. (Exit code $? of tzoupdate)
    OUTPUT=""                       # Initialize.. (Verbose output of tzoupdate)
    TZOARGS=""                      # Initialize.. (we set this here in loop, so we can detect modified tzoupdate.conf)
    
    #set -x		# debug
    detect_args			# sanity configuration, error checking
    detect_killfile		# avoids abuse updates  (terminates in event of unfixed fatal error)
    detect_log_filename # set $IPFILE
    #detect_ipfile_age   # set $IPFILE_AGE. Detects if we've run >1x within 60 secs.
    detect_os           # set $OSSTRING
    invoke_tzoupdate	# set $OUTPUT, $RETURNCODE. 
    
    # post-processing - RETURNCODE indicates state of the update
    # create and set $MESSAGE
    #
    if [ "r$RETURNCODE" = "r0" ]; then
        # GOOD
        MESSAGE="Success: tzoupdate reports no errors. You may view $TZOLOG for details."
        if [ $QUIETMODE -ne 1 ]; then
            echo "${MESSAGE}"
        fi
    elif [ "r$RETURNCODE" = "r1" ]; then
        # Happens if either a) the network is down or b) you tried to update
        # more than one time within the last 60 seconds.
        MESSAGE="Error: Temporary network issue, retrying in 60 seconds." 
        if [ $QUIETMODE -ne 1 ]; then
            echo "${MESSAGE}"
        fi
        sleep 60;
    elif [ $RETURNCODE -eq 2 ]; then
        # FATAL ERROR (EXPIRED TZO!)
        # We don't suppress this error even if QUIETMODE
        MESSAGE="Error $RETURNCODE: TZO indicates Account EXPIRED. Please see $TZOLOG for details."
        echo "${MESSAGE}"
        create_killfile
    elif [ $RETURNCODE -eq 3 ]; then
        # FATAL ERROR (CHECK ACCOUNT CONFIG)
        # We don't suppress this error even if QUIETMODE
        MESSAGE="Error $RETURNCODE: Fatal error with your TZO Account. Please see $TZOLOG for details."
        # NOTE: remove the killfile AFTER you have resolved the account
        echo "${MESSAGE}"
        create_killfile
    elif [ $RETURNCODE -eq 4 ]; then
        # WARNING (typo in script or installation, usage problem?)
        # We don't suppress this error even if QUIETMODE
        MESSAGE="Error $RETURNCODE: tzoupdate Usage Error. Please see --help option"
        echo "${MESSAGE}"
    fi
    
    # LOG OUR RESULTS, overwritten after each loop (10 minutes) 
    DATE=`date`
    HOST=`hostname`
    echo "* Results of your most recent tzoupdate. Please review this log for your update status." > ${TZOLOG}
    echo "* Local system's time is: ${DATE}, and local hostname is: ${HOST}." >> ${TZOLOG}
    echo "* 'tzoupdate' return code: ${RETURNCODE}. tzoupdate output text: (see next line)" >> ${TZOLOG}
    echo "${OUTPUT}">> ${TZOLOG}
        
    # done from while loop, daemon mode
done


