/*******************************************************************************
#                                                                              #
#      MJPG-streamer allows to stream JPG frames from an input-plugin          #
#      to several output plugins                                               #
#                                                                              #
#      Copyright (C) 2007 Tom Stöveken                                         #
#                                                                              #
# This program is free software; you can redistribute it and/or modify         #
# it under the terms of the GNU General Public License as published by         #
# the Free Software Foundation; version 2 of the License.                      #
#                                                                              #
# This program is distributed in the hope that it will be useful,              #
# but WITHOUT ANY WARRANTY; without even the implied warranty of               #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                #
# GNU General Public License for more details.                                 #
#                                                                              #
# You should have received a copy of the GNU General Public License            #
# along with this program; if not, write to the Free Software                  #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    #
#                                                                              #
*******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <linux/videodev.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <getopt.h>
#include <pthread.h>
#include <sys/ipc.h>
#include <sys/shm.h>

typedef struct _JPEGFrame{
    int             on_demand;
    int             buf_ready;
    unsigned int		datalen;
    unsigned char 		buf[0];	
} SJPEGFrame;

#define JPEG_SHM_KEY 7000
#define VID_OVER_HTTP			10
#define SIGVID          40

#include "../../mjpg_streamer.h"
#include "../../utils.h"

#define MJPG_OVER_HTTP_FILE  "/var/run/mjpg_over_http"
#define INPUT_PLUGIN_NAME "SHM input plugin"
#define MAX_ARGUMENTS 32

/* private functions and variables to this plugin */
static pthread_t   shm;
static globals     *pglobal;

SJPEGFrame *Jpegframe;
int pid = 0, shm_key = 0, shm_id;

int read_pid_from_file(char* pid_file)
{
	FILE *fp;
	int ret = 0;
	char buf[16]="";
	if ( pid_file != NULL ) {
		fp = fopen(pid_file, "r");
		if ( fp != NULL ) {
			fgets(buf,sizeof(buf),fp);
			ret = atoi(buf);
			fclose(fp);
		}
	}
	return ret;
}

void *generic_shm_read(unsigned int key, unsigned int size, int *shm_id)
{
	void *shared_memory=(void *) 0;
	*shm_id = shmget((key_t)key, (size_t)size, 0666|IPC_CREAT);
	if (*shm_id != -1)
	{
	       shared_memory = shmat(*shm_id,(void *) 0,0);
		if(shared_memory != -1)
			return shared_memory;
	}
	return 0;
}

static void create_pid_file(char *path)
{
	FILE *fp;
	fp =  fopen(path, "w") ;
	if(fp == NULL) {
		printf("Failed to create PID file\n");
		return;
	}
	fprintf(fp, "%d", getpid());
	fclose(fp);	
}

void *shm_thread( void *);
void shm_cleanup(void *);
void help(void);

static int delay = 1000;

/*** plugin interface functions ***/
int input_init(input_parameter *param) {
  char *argv[MAX_ARGUMENTS]={NULL};
  int argc=1, i, width=0, height=0;
  char *s;
  char tmp[64];

  /* convert the single parameter-string to an array of strings */
  argv[0] = INPUT_PLUGIN_NAME;
  if ( param->parameter_string != NULL && strlen(param->parameter_string) != 0 ) {
    char *arg=NULL, *saveptr=NULL, *token=NULL;

    arg=(char *)strdup(param->parameter_string);

    if ( strchr(arg, ' ') != NULL ) {
      token=strtok_r(arg, " ", &saveptr);
      if ( token != NULL ) {
        argv[argc] = strdup(token);
        argc++;
        while ( (token=strtok_r(NULL, " ", &saveptr)) != NULL ) {
          argv[argc] = strdup(token);
          argc++;
          if (argc >= MAX_ARGUMENTS) {
            IPRINT("ERROR: too many arguments to input plugin\n");
            return 1;
          }
        }
      }
    }
  }

  /* show all parameters for DBG purposes */
  for (i=0; i<argc; i++) {
    DBG("argv[%d]=%s\n", i, argv[i]);
  }

  reset_getopt();
  while(1) {
    int option_index = 0, c=0;
    static struct option long_options[] = \
    {
      {"h", no_argument, 0, 0},
      {"help", no_argument, 0, 0},
      {"d", required_argument, 0, 0},
      {"delay", required_argument, 0, 0},
      {"r", required_argument, 0, 0},
      {"resolution", required_argument, 0, 0},
      {"k", required_argument, 0, 0},
      {"key", required_argument, 0, 0},
      {0, 0, 0, 0}
    };

    c = getopt_long_only(argc, argv, "", long_options, &option_index);

    /* no more options to parse */
    if (c == -1) break;

    /* unrecognized option */
    if (c == '?'){
      help();
      return 1;
    }

    switch (option_index) {
      /* h, help */
      case 0:
      case 1:
        DBG("case 0,1\n");
        help();
        return 1;
        break;

      /* d, delay */
      case 2:
      case 3:
        DBG("case 2,3\n");
        delay = atoi(optarg);
        break;

      /* r, resolution */
      case 4:
      case 5:
        DBG("case 4,5\n");
        width = -1;
        height = -1;

        /* parse value as decimal value */
        width  = strtol(optarg, &s, 10);
        height = strtol(s+1, NULL, 10);
        break;

      /* r, resolution */
      case 6:
      case 7:
        DBG("case 6,7\n");
        shm_key = atoi(optarg);
        break;

      default:
        DBG("default case\n");
        help();
        return 1;
    }
  }

  pglobal = param->global;
  pglobal->channelID=shm_key-JPEG_SHM_KEY;	 // Get channel ID back from shm_key - JPEG_SHM_KEY
  sprintf(tmp, "%s%d.pid", MJPG_OVER_HTTP_FILE, pglobal->channelID);
  create_pid_file(tmp);	
  IPRINT("delay.............: %i\n", delay);

  Jpegframe = (SJPEGFrame *) generic_shm_read(shm_key,
                sizeof(SJPEGFrame) + ((width * height) * 0.85), &shm_id);

  if (Jpegframe == NULL) {
    fprintf(stderr, "share memory initial error\n");
    exit(1);
  }

  pglobal->buf = malloc((width * height) * 0.85);
  if (pglobal->buf == NULL) {
    fprintf(stderr, "could not allocate memory\n");
    exit(1);
  }

  return 0;
}

int input_stop(void) {
  DBG("will cancel input thread\n");
  pthread_cancel(shm);

  return 0;
}

int input_run(void) {
  if( pthread_create(&shm, 0, shm_thread, NULL) != 0) {
    fprintf(stderr, "could not start shm thread\n");
    exit(EXIT_FAILURE);
  }
  pthread_detach(shm);

  return 0;
}

/*** private functions for this plugin below ***/
void help(void) {
    fprintf(stderr, " ---------------------------------------------------------------\n" \
                    " Help for input plugin..: "INPUT_PLUGIN_NAME"\n" \
                    " ---------------------------------------------------------------\n" \
                    " The following parameters can be passed to this plugin:\n\n" \
                    " [-d | --delay ]........: delay to pause between frames\n" \
                    " ---------------------------------------------------------------\n");
}

/* the single writer thread */
void *shm_thread( void *arg ) {
  int connect=0, sent=0;
  union sigval rt_sigval;
  char pid_path[32];
  /* set cleanup handler to cleanup allocated ressources */
  pthread_cleanup_push(shm_cleanup, NULL);

  while( !pglobal->stop ) {
    pthread_mutex_lock( &pglobal->db );
    connect=pglobal->connectNumber;
    pthread_mutex_unlock( &pglobal->db );
  
    if(connect==0) {
      usleep(500000);
      Jpegframe->on_demand = 0;
      sent = 0;
    } else {
      if (sent == 0) {
        Jpegframe->on_demand = 1;
  
        while (pid == 0) {
          sprintf(pid_path, "/var/run/mdengine_ch%d.pid", shm_key - JPEG_SHM_KEY);
          pid = read_pid_from_file(pid_path);
          sleep(1);
        }
        rt_sigval.sival_int = VID_OVER_HTTP;
        if(sigqueue(pid, SIGVID, rt_sigval) == -1)
                printf("send error\n");
        sent = 1;
      }

      pthread_mutex_lock( &pglobal->db );

      /* grab a frame */
      while(1) {
        if (Jpegframe->buf_ready == 1) {
          /* copy JPG picture to global buffer */
          pglobal->size = Jpegframe->datalen;
          memcpy(pglobal->buf, Jpegframe->buf, Jpegframe->datalen);
          Jpegframe->buf_ready = 0;
          break;
        }
        usleep(delay*1000);
      }

      /* signal fresh_frame */
      pthread_cond_broadcast(&pglobal->db_update);
      pthread_mutex_unlock( &pglobal->db );

      usleep(delay*1000);
    }
  }

  DBG("leaving input thread, calling cleanup function now\n");
  pthread_cleanup_pop(1);

  pglobal->input_stoped = 1;
  return NULL;
}

void shm_cleanup(void *arg) {
  if (pglobal->buf != NULL) free(pglobal->buf);

  if (shmdt(Jpegframe) == -1)
    perror("shmdt");

//  if (shmctl(shm_id, IPC_RMID, 0) == -1)
//    perror("shmctl");
}

