/*******************************************************************************
# GSPCAV1 streaming input-plugin for MJPG-streamer                             #
#                                                                              #
# This plugin is intended to work with gspcav1 compatible devices              #
# Intention is to use webcams that support JPG encoding by the webcam itself   #
#                                                                              #
# Copyright (C) 2007 Tom Stöveken                                              #
#                                                                              #
# This program is free software; you can redistribute it and/or modify         #
# it under the terms of the GNU General Public License as published by         #
# the Free Software Foundation; either version 2 of the License, or            #
# (at your option) any later version.                                          #
#                                                                              #
# This program is distributed in the hope that it will be useful,              #
# but WITHOUT ANY WARRANTY; without even the implied warranty of               #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                #
# GNU General Public License for more details.                                 #
#                                                                              #
# You should have received a copy of the GNU General Public License            #
# along with this program; if not, write to the Free Software                  #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    #
#                                                                              #
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <getopt.h>
#include <pthread.h>

#include "spcaframe.h"
#include "spcav4l.h"
//#include "utils.h"

#include "../../utils.h"
#include "../../mjpg_streamer.h"
#include "ioctl_je.h"
#include "FrameRateControl.h"


#define INPUT_PLUGIN_NAME "GSPCAV1 webcam grabber"
#define FMJPEG_ENCODER_DEV  "/dev/mjenc" 	//major:10 minior:61
#define MAX(x, y) ((x) > (y) ? (x) : (y))
#define MIN(x, y) ((x) < (y) ? (x) : (y))
//#define IMAGE_COMP		3
#define IMAGE_QUALITY	60
#define IMAGE_RST		0


#define SET_COMP(p, index,hsamp,vsamp)  \
  (p->rgComponentInfo[index].m_u8HSamplingFrequency = (hsamp), \
   p->rgComponentInfo[index].m_u8VSamplingFrequency = (vsamp))



#define MAX_ARGUMENTS 32
#define GEMTEK_NOTES 0
enum YUVPictureType {
	YUV_PIC_TYPE_420=0,
	YUV_PIC_TYPE_422=1,
	YUV_PIC_TYPE_211=2,
	YUV_PIC_TYPE_333=3,
	YUV_PIC_TYPE_222=4,
	YUV_PIC_TYPE_111=5,
};

enum JPEGPictureType {
	JPEG_PIC_WITHOUT_HT_QT=0,	/* Picture does not include Huffmann and Quant tables */
	JPEG_PIC_WITH_HT_QT=1		/* Picture include Huffmann and Quant tables */
};

typedef struct _JPEGEncDev {
	int fd;
	unsigned int width;
	unsigned int height;
	int quality;

#ifdef SHOW_PERFORMANCE_STATS
	struct timeval start_time;			/* start time since opened */
	unsigned int pic_count;			/* total encoded picture count. */
	unsigned int bytes_count;			/* total encoded bytes count. */
	unsigned int bytes_truncated;		/* total bytes being truncated */
	unsigned int fail_encode_count;	/* total failed attempt to encode frame */
#endif

	FJPEG_ENC_PARAM 	enc_param;
	FJPEG_ENC_QTBLS 	qtbls;
} SJPEGEncDev;

static const struct {
  const char *string;
  const int width, height;
} resolutions[] = {
  { "QSIF", 160,  120  },
  { "QCIF", 176,  144  },
  { "CGA",  320,  200  },
  { "QVGA", 320,  240  },
  { "CIF",  352,  288  },
  { "VGA",  640,  480  },
  { "SVGA", 800,  600  },
  { "XGA",  1024, 768  },
  { "SXGA", 1280, 1024 }
};
#if GEMTEK_NOTES

WXGA 	1280*800 
VGA		640*480 
QVGA	320*192 

#endif

static const struct {
  const char *string;
  const int format;
} formats[] = {
  { "r16", VIDEO_PALETTE_RGB565  },
  { "r24", VIDEO_PALETTE_RGB24   },
  { "r32", VIDEO_PALETTE_RGB32   },
  { "yuv", VIDEO_PALETTE_YUV420P },
  { "jpg", VIDEO_PALETTE_JPEG    }
};

#if GEMTEK_NOTES
v4l1Setting.palette = VIDEO_PALETTE_YUV420P;
#endif

/* private functions and variables to this plugin */
pthread_t cam;
struct vdIn *videoIn;
static globals *pglobal;

void *cam_thread( void *);
void cam_cleanup(void *);
void help(void);

static HANDLE_JPEG_ENC jpegenc_alloc_dev()
{
	SJPEGEncDev *pJPEGEncDev=NULL;
	unsigned char *buf=NULL;
	
	// Allocate device
	if ( (pJPEGEncDev=malloc(sizeof(SJPEGEncDev))) == NULL) {
		perror("malloc");
		goto ERROR;
	}
	memset(pJPEGEncDev, 0x0, sizeof(SJPEGEncDev));
	pJPEGEncDev->fd = -1;
	
	return (HANDLE_JPEG_ENC) pJPEGEncDev;

ERROR:
	if (pJPEGEncDev)
		free(pJPEGEncDev);

	return (HANDLE_JPEG_ENC) NULL;
	
}

static void  jpegenc_reset_dev(HANDLE_JPEG_ENC handle) 
{
	SJPEGEncDev *pJPEGEncDev=(SJPEGEncDev *) handle;

	if (pJPEGEncDev==NULL)
		return;
	
	if (pJPEGEncDev->fd >= 0) {
		close(pJPEGEncDev->fd);
		pJPEGEncDev->fd = -1;
	}
	
}

// TODO: passing pointer may not work... since the handle may have been pass by value..
static void  jpegenc_dealloc_dev(HANDLE_JPEG_ENC *pHandle)
{
	if (*pHandle == NULL)
		return;

	free(*pHandle);
	*pHandle=NULL;
}
/////////////////////////////
// JPEG Encode Interface APIs  //
////////////////////////////


HANDLE_JPEG_ENC  jpegenc_open(const SJPEGEncSetting *pSetting)
{
	HANDLE_JPEG_ENC handle=NULL;
	SJPEGEncDev *context=NULL;
	int fmjpeg_enc_fd=-1;
	unsigned int max_h_samp, max_v_samp;
	int i;
	unsigned int image_size[3];
	unsigned int yuv_size=0;
	FJPEG_ENC_PARAM *enc_param;

	//printf("Open JPEG encoder device\n");


	if ((handle=jpegenc_alloc_dev())==NULL)
		goto ERROR;
	context = (SJPEGEncDev*) handle;
	

	////// Open the encoder device 
	if ( (fmjpeg_enc_fd=open(FMJPEG_ENCODER_DEV,O_RDWR)) < 0) {
	  	perror(FMJPEG_ENCODER_DEV);
		goto ERROR;
      	}
	context->fd = fmjpeg_enc_fd;
	

	////// FJPEG_ENC_PARAM:  parameter for jpeg encoder initialization
	enc_param=&context->enc_param;
	memset(enc_param, 0, sizeof(FJPEG_ENC_PARAM));
	enc_param->u32API_version=MJPEGE_VER;
//    	enc_param->u32ImageQuality = pSetting->quality;  
		enc_param->sample = 0x00;  // JCS_420
    	enc_param->u32RestartInterval = 0;
    	enc_param->u32ImageWidth=pSetting->width; 
    	enc_param->u32ImageHeight=pSetting->height;
    	enc_param->u82D = pSetting->grabtype;
		enc_param->roi_x = 0;
		enc_param->roi_y = 0;
		enc_param->roi_w = pSetting->width;
		enc_param->roi_h = pSetting->height;
#if 0	
    	enc_param->u32API_version=MJPEGVER;
    	enc_param->u32ImageQuality = pSetting->quality;  
    	enc_param->u32RestartInterval = 0;
    	enc_param->u32ImageWidth=pSetting->width; 
    	enc_param->u32ImageHeight=pSetting->height;

	// To describe the YUV format through the following encoding parameters  
    	enc_param->u8NumComponents = IMAGE_COMP; 	/* the input image has 3 components 'YUV' */
    	enc_param->u8JPGPIC = JPEG_PIC_WITH_HT_QT;	// JPEG_PIC_WITHOUT_HT_QT;	
	if ( pSetting->grabtype  != 0xFFFFFFFF)
	    enc_param->u82D = pSetting->grabtype;
    	else 
	    enc_param->u82D = 1; 

	// ROI information
	enc_param->roi_enable = 0;


	/* Motion Detection */
	enc_param->u32ImageMotionDetection = pSetting->md_enabled;	
  	int YUVsampling = YUV_PIC_TYPE_420;	 	// TODO: FIXME... avctx->fmjpeg_yuv_format;

	if(enc_param->u8NumComponents==1)
		YUVsampling = 5; // if there is only one component, it is gray, so we force it YUV111

	switch (YUVsampling) {
	case YUV_PIC_TYPE_420:	  
		SET_COMP(enc_param, 0, 2, 2);	
		SET_COMP(enc_param, 1, 1,1);   
		SET_COMP(enc_param, 2, 1,1);	
		break;
       case YUV_PIC_TYPE_422:	  
	   	SET_COMP(enc_param, 0, 4,1);	
		SET_COMP(enc_param, 1, 2,1);   
		SET_COMP(enc_param, 2, 2,1);	
		break;
       case YUV_PIC_TYPE_211:	  
	   	SET_COMP(enc_param, 0, 2,1);	
		SET_COMP(enc_param, 1, 1,1);   
		SET_COMP(enc_param, 2, 1,1);	
		break;
       case YUV_PIC_TYPE_333:	  
	   	SET_COMP(enc_param, 0, 3,1);	
		SET_COMP(enc_param, 1, 3,1);   
		SET_COMP(enc_param, 2, 3,1);	
		break;
       case YUV_PIC_TYPE_222:	  
	   	SET_COMP(enc_param, 0, 2,1);	
		SET_COMP(enc_param, 1, 2,1);   
		SET_COMP(enc_param, 2, 2,1);	
		break;
       case YUV_PIC_TYPE_111:	  
	   	SET_COMP(enc_param, 0, 1,1);	
		SET_COMP(enc_param, 1, 1,1);   
		SET_COMP(enc_param, 2, 1,1);	
		break;
       default:   
	   	break;
	}  
	// to set each component's sampling factor (horizontally and vertically)
  	   

	// get the maximum horizontal sampling factor
	max_h_samp=MAX(enc_param->rgComponentInfo[0].m_u8HSamplingFrequency,
            				MAX(enc_param->rgComponentInfo[1].m_u8HSamplingFrequency,
            						enc_param->rgComponentInfo[2].m_u8HSamplingFrequency));

	// get the maximum horizontal sampling factor
	max_v_samp= MAX(enc_param->rgComponentInfo[0].m_u8VSamplingFrequency,
            				MAX(enc_param->rgComponentInfo[1].m_u8VSamplingFrequency,
            					enc_param->rgComponentInfo[2].m_u8VSamplingFrequency)); 
                 
	// calculate each component size according to its maximum sampling factor
  	// and individual sampling factor
	for(i=0;i<enc_param->u8NumComponents;i++)  {
        	image_size[i]=
			(((enc_param->rgComponentInfo[i].m_u8HSamplingFrequency*enc_param->u32ImageWidth) /max_h_samp) *
               	((enc_param->rgComponentInfo[i].m_u8VSamplingFrequency*enc_param->u32ImageHeight) /max_v_samp));
		yuv_size += image_size[i];
	}
	//printf("yuv_size: %d\n", yuv_size);

#endif


	if ( ioctl(fmjpeg_enc_fd, FMJPEG_IOCTL_ENCODE_CREATE, enc_param) < 0 )  {
		perror("ioctl - FMJPEG_IOCTL_ENCODE_CREATE");
		goto ERROR;
	}


	//printf("JPEG encoder created [%p]\n", handle);
	return handle;

ERROR:

	jpegenc_reset_dev(handle);
	jpegenc_dealloc_dev(&handle);
	return (HANDLE_JPEG_ENC) NULL;

}


/*** plugin interface functions ***/
/******************************************************************************
Description.: This function initializes the plugin. It parses the commandline-
              parameter and stores the default and parsed values in the
              appropriate variables.
Input Value.: param contains among others the command-line string
Return Value: 0 if everything is fine
              1 if "--help" was triggered, in this case the calling programm
              should stop running and leave.
******************************************************************************/
int input_init(input_parameter *param) {
  char *argv[MAX_ARGUMENTS]={NULL}, *dev = "/dev/video0", *s;
  int i,argc=1, width=640, height=480,fps=6,quality=80,format=VIDEO_PALETTE_YUV420P;//format=VIDEO_PALETTE_JPEG
 int ingrab=32;
  /* convert the single parameter-string to an array of strings */
  argv[0] = INPUT_PLUGIN_NAME;
  if ( param->parameter_string != NULL && strlen(param->parameter_string) != 0 ) {
    char *arg=NULL, *saveptr=NULL, *token=NULL;

    arg=(char *)strdup(param->parameter_string);

    if ( strchr(arg, ' ') != NULL ) {
      token=strtok_r(arg, " ", &saveptr);
      if ( token != NULL ) {
        argv[argc] = strdup(token);
        argc++;
        while ( (token=strtok_r(NULL, " ", &saveptr)) != NULL ) {
          argv[argc] = strdup(token);
          argc++;
          if (argc >= MAX_ARGUMENTS) {
            IPRINT("ERROR: too many arguments to input plugin\n");
            return 1;
          }
        }
      }
    }
  }

  /* show all parameters for DBG purposes */
  for (i=0; i<argc; i++) {
    DBG("argv[%d]=%s\n", i, argv[i]);
  }

  reset_getopt();
  while(1) {
    int option_index = 0, c=0;
    static struct option long_options[] = \
    {
      {"h", no_argument, 0, 0},
      {"help", no_argument, 0, 0},
      {"d", required_argument, 0, 0},
      {"device", required_argument, 0, 0},
      {"r", required_argument, 0, 0},
      {"resolution", required_argument, 0, 0},
       {"f", required_argument, 0, 0},
      {"fps", required_argument, 0, 0},
       {"y", no_argument, 0, 0},
      {"yuv", no_argument, 0, 0},
       {"q", required_argument, 0, 0},
      {"quality", required_argument, 0, 0},
	{"i", required_argument, 0, 0},
      {"ingrab", required_argument, 0, 0},
      {0, 0, 0, 0}
    };

    c = getopt_long_only(argc, argv, "", long_options, &option_index);

    /* no more options to parse */
    if (c == -1) break;

    /* unrecognized option */
    if (c == '?'){
      help();
      return 1;
    }

    switch (option_index) {
      /* h, help */
      case 0:
      case 1:
        DBG("case 0,1\n");
        help();
        return 1;
        break;

      /* d, device */
      case 2:
      case 3:
        DBG("case 2,3\n");
        dev = strdup(optarg);
        break;

      /* r, resolution */
      case 4:
      case 5:
        DBG("case 4,5\n");
        width = -1;
        height = -1;

        /* try to find the resolution in lookup table "resolutions" */
        for ( i=0; i < LENGTH_OF(resolutions); i++ ) {
          if ( strcmp(resolutions[i].string, optarg) == 0 ) {
            width  = resolutions[i].width;
            height = resolutions[i].height;
          }
        }
        /* done if width and height were set */
        if(width != -1 && height != -1)
          break;
        /* parse value as decimal value */
        width  = strtol(optarg, &s, 10);
        height = strtol(s+1, NULL, 10);
        break;

	 /* f, fps */
	 //TODO:
      case 6:
      case 7:
        DBG("case 6,7\n");
        fps=atoi(optarg);
        break;
	  /* y, yuv */
      case 8:
      case 9:
        DBG("case 8,9\n");
        format = VIDEO_PALETTE_YUV420P;
        break;
	/*q,quality*/	
      case 10:
      case 11:
        DBG("case 10,11\n");
        quality=atoi(optarg);
        break;
	/*i,ingrab*/	
      case 12:
      case 13:
        DBG("case 12,13\n");
        ingrab=atoi(optarg);
        break;
		
      default:
        DBG("default case\n");
        help();
        return 1;
    }
  }

  /* keep a pointer to the global variables */
  pglobal = param->global;

  /* allocate webcam datastructure */
  videoIn = malloc(sizeof(struct vdIn));
  if ( videoIn == NULL ) {
    IPRINT("not enough memory for videoIn\n");
    exit(EXIT_FAILURE);
  }
  memset(videoIn, 0, sizeof(struct vdIn));

  /* display the parsed values */
 #if 0 
  IPRINT("Using V4L1 device.: %s\n", dev);
  IPRINT("Desired Resolution: %i x %i\n", width, height);
  IPRINT("Frames Per Second.: %i\n", fps);
  IPRINT("Format............: %d\n", format);
#endif

///* ingrab type value */
videoIn->ingrab=ingrab;//FGRABTYPE_SENSOR_OV9710

  /* open video device and prepare data structure */
  if (init_videoIn(videoIn, dev, width, height, format, 1) != 0) {
    IPRINT("init_VideoIn failed\n");
    closelog();
    exit(EXIT_FAILURE);
  }
	////// Open JPEG encoder device.

	videoIn->jpegEncSetting.width = width;
	videoIn->jpegEncSetting.height = height;
	videoIn->jpegEncSetting.fps = fps; 
	videoIn->jpegEncSetting.quality = quality;

	if ( (videoIn->hJPEGEnc=jpegenc_open(&videoIn->jpegEncSetting)) == NULL){
		closelog();
    		exit(EXIT_FAILURE);
	}
		
		
  
//printf("Success\n");
  return 0;
	
  
}


int jpegenc_encode_frame(HANDLE_JPEG_ENC handle,
				SYUVFrame *pYUVFrame,unsigned char *buf)
{
	SJPEGEncDev *context = (HANDLE_JPEG_ENC) handle;

	FJPEG_ENC_FRAME	enc_frame;
#if 0
	FJPEG_ENC_PARAM *enc_param = &context->enc_param;

	if (enc_param->u8NumComponents != IMAGE_COMP) {
		fprintf(stderr, "# of components %d not equal to 3\n", enc_param->u8NumComponents);
		return -1;
	}
#endif	

		enc_frame.roi_x = -1;
		enc_frame.roi_y = -1;
		enc_frame.roi_w = -1;
		enc_frame.roi_h = -1;
		enc_frame.u8JPGPIC = 1;//context->snapshot
		enc_frame.pu8YUVAddr[0]=pYUVFrame->yAddr;  // capture YUV virtual addres
		enc_frame.pu8YUVAddr[1]=pYUVFrame->uAddr;  // capture YUV virtual addres
		enc_frame.pu8YUVAddr[2]=pYUVFrame->vAddr;  // capture YUV virtual addres
		enc_frame.u32ImageQuality = videoIn->jpegEncSetting.quality;//80;
		enc_frame.pu8BitstreamAddr = buf;
		
		if (ioctl(context->fd, FMJPEG_IOCTL_ENCODE_ONE, &enc_frame) < 0 )	{
			perror("ioctl - FMJPEG_IOCTL_ENCODE_ONE");
			goto ERROR;
		}

	return enc_frame.bitstream_size;

#if 0
	enc_param->pu8YUVAddr[0]=pYUVFrame->yAddr;  // capture YUV virtual addres
       enc_param->pu8YUVAddr[1]=pYUVFrame->uAddr;  // capture YUV virtual addres
       enc_param->pu8YUVAddr[2]=pYUVFrame->vAddr;  // capture YUV virtual addres


	enc_param->pu8BitstreamAddr =buf;
	

	if (ioctl(context->fd, FMJPEG_IOCTL_ENCODE_ONE, enc_param) < 0 )   {
		perror("ioctl - FMJPEG_IOCTL_ENCODE_ONE");
		goto ERROR;
	}


		
	return enc_param->bitstream_size;
#endif
ERROR:
	return -1;
}


/******************************************************************************
Description.: Stops the execution of worker thread
Input Value.: -
Return Value: always 0
******************************************************************************/
int input_stop(void) {
  DBG("will cancel input thread\n");
  pthread_cancel(cam);

  return 0;
}

/******************************************************************************
Description.: spins of a worker thread
Input Value.: -
Return Value: always 0
******************************************************************************/
int input_run(void) {
  pglobal->buf = malloc(videoIn->framesizeIn);
  if (pglobal->buf == NULL) {
    fprintf(stderr, "could not allocate memory\n");
    exit(EXIT_FAILURE);
  }

  pthread_create(&cam, 0, cam_thread, NULL);
  pthread_detach(cam);

  return 0;
}

/******************************************************************************
Description.: process commands, allows to set certain runtime configurations
              and settings like pan/tilt, colors, saturation etc.
Input Value.: * cmd specifies the command, a complete list is maintained in
                the file "input.h"
              * value is used for commands that make use of a parameter.
Return Value: depends in the command, for most cases 0 means no errors and
              -1 signals an error. This is just rule of thumb, not more!
******************************************************************************/
int input_cmd(in_cmd_type cmd, int value) {
  int res=0;

  return res;
}

/*** private functions for this plugin below ***/
/******************************************************************************
Description.: print a help message to stderr
Input Value.: -
Return Value: -
******************************************************************************/
void help(void) {
  int i;

  fprintf(stderr, " ---------------------------------------------------------------\n" \
                  " Help for input plugin..: "INPUT_PLUGIN_NAME"\n" \
                  " ---------------------------------------------------------------\n" \
                  " The following parameters can be passed to this plugin:\n\n" \
                  " [-d | --device ].......: video device to open (your camera)\n" \
                  " [-r | --resolution ]...: the resolution of the video device,\n" \
                  "                          can be one of the following strings:\n" \
                  "                          ");

  for ( i=0; i < LENGTH_OF(resolutions); i++ ) {
    fprintf(stderr, "%s ", resolutions[i].string);
    if ( (i+1)%6 == 0)
      fprintf(stderr, "\n                          ");
  }
  fprintf(stderr, "\n                          or a custom value like the following" \
                  "\n                          example: 640x480\n" \
                  " [ -f | --format ]......: grabbing format, should be set to 'jpg'\n" \
                  "                          can be: ");
  for ( i=0; i < LENGTH_OF(formats); i++ ) {
    fprintf(stderr, "%s ", formats[i].string);
    if ( (i+1)%6 == 0)
      fprintf(stderr, "\n                          ");
  }
  fprintf(stderr, "\n");
  fprintf(stderr, " ---------------------------------------------------------------\n\n");
}

/******************************************************************************
Description.: this thread worker grabs a frame and copies it to the global buffer
Input Value.: unused
Return Value: unused, always NULL
******************************************************************************/
void *cam_thread( void *arg ) {
  int iframe = 0;
  unsigned char *pictureData = NULL;
  struct frame_t *headerframe;
  SFrameRateControl frameRateControl;
int connect=0;
int i=0;
  /* set cleanup handler to cleanup allocated ressources */
  pthread_cleanup_push(cam_cleanup, NULL);
  
  // Set frame rate
	  FrameRateControlInit(&frameRateControl, videoIn->jpegEncSetting.fps);

  while( !pglobal->stop ) {
  pthread_mutex_lock( &pglobal->db );
	connect=pglobal->connectNumber;
   pthread_mutex_unlock( &pglobal->db );

	if(connect==0)
	{
		usleep(500000);
	}
	else
		{
		    /* grab a frame */
			for(;i<3;i++)
			{
			    if(v4lGrab(videoIn) < 0 ) {
			      IPRINT("Error grabbing frames\n");
			      exit(EXIT_FAILURE);
			    }
			}
			FrameRateControl_start(&frameRateControl);
		    if( v4lGrab(videoIn) < 0 ) {
		      IPRINT("Error grabbing frames\n");
		      exit(EXIT_FAILURE);
		    }

		    iframe=(videoIn->frame_cour +(OUTFRMNUMB-1))% OUTFRMNUMB;
		    //videoIn->framelock[iframe]++;
		    headerframe=(struct frame_t*)videoIn->ptframe[iframe];
		    pictureData = videoIn->ptframe[iframe]+sizeof(struct frame_t);
		  //  videoIn->framelock[iframe]--;

		    /* copy JPG picture to global buffer */
		    pthread_mutex_lock( &pglobal->db );

		//    pglobal->size = get_jpegsize(pictureData, headerframe->size);

		     pglobal->size = videoIn->jpegsize;
		    memcpy(pglobal->buf, pictureData, pglobal->size);

		//printf("pglobal->size=[%d]  videoIn->framesizeIn=[%d]\n",pglobal->size,videoIn->framesizeIn);
		    /* signal fresh_frame */
		    pthread_cond_broadcast(&pglobal->db_update);
		    pthread_mutex_unlock( &pglobal->db );

		    DBG("waiting for next frame\n");
			FrameRateControl_wait_before_next_frame(&frameRateControl);


		}

	

  }

  DBG("leaving input thread, calling cleanup function now\n");
  pthread_cleanup_pop(1);

  return NULL;
}

/******************************************************************************
Description.: 
Input Value.: 
Return Value: 
******************************************************************************/
void cam_cleanup(void *arg) {
  static unsigned char first_run=1;

  if ( !first_run ) {
    DBG("already cleaned up ressources\n");
    return;
  }

  first_run = 0;
  IPRINT("cleaning up ressources allocated by input thread\n");
  close_v4l(videoIn);
  if(videoIn->hJPEGEnc){
  	jpegenc_close(videoIn->hJPEGEnc);
	videoIn->hJPEGEnc=NULL;
  	}
  	
  //if (videoIn->tmpbuffer != NULL) free(videoIn->tmpbuffer);
  if (videoIn != NULL) free(videoIn);
  if (pglobal->buf != NULL) free(pglobal->buf);
}




