/*******************************************************************************
#                                                                              #
#      MJPG-streamer allows to stream JPG frames from an input-plugin          #
#      to several output plugins                                               #
#                                                                              #
#      Copyright (C) 2007 Tom Stöveken                                         #
#                                                                              #
# This program is free software; you can redistribute it and/or modify         #
# it under the terms of the GNU General Public License as published by         #
# the Free Software Foundation; version 2 of the License.                      #
#                                                                              #
# This program is distributed in the hope that it will be useful,              #
# but WITHOUT ANY WARRANTY; without even the implied warranty of               #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                #
# GNU General Public License for more details.                                 #
#                                                                              #
# You should have received a copy of the GNU General Public License            #
# along with this program; if not, write to the Free Software                  #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA    #
#                                                                              #
*******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <linux/videodev.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <getopt.h>
#include <pthread.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include "frames_server_api.h"

#include "../../mjpg_streamer.h"
#include "../../utils.h"

#define INPUT_PLUGIN_NAME "FC input plugin"
#define MAX_ARGUMENTS 32

/* private functions and variables to this plugin */
static pthread_t   fc;
static globals     *pglobal;

void *fc_thread( void *);
void fc_cleanup(void *);
void help(void);

static int delay = 1000;
size_t buf_size;
HANDLE *handle;
int channel;

/*** plugin interface functions ***/
int input_init(input_parameter *param) {
  char *argv[MAX_ARGUMENTS]={NULL};
  int argc=1, i, width=0, height=0;
  char *s;

  /* convert the single parameter-string to an array of strings */
  argv[0] = INPUT_PLUGIN_NAME;
  if ( param->parameter_string != NULL && strlen(param->parameter_string) != 0 ) {
    char *arg=NULL, *saveptr=NULL, *token=NULL;

    arg=(char *)strdup(param->parameter_string);

    if ( strchr(arg, ' ') != NULL ) {
      token=strtok_r(arg, " ", &saveptr);
      if ( token != NULL ) {
        argv[argc] = strdup(token);
        argc++;
        while ( (token=strtok_r(NULL, " ", &saveptr)) != NULL ) {
          argv[argc] = strdup(token);
          argc++;
          if (argc >= MAX_ARGUMENTS) {
            IPRINT("ERROR: too many arguments to input plugin\n");
            return 1;
          }
        }
      }
    }
  }

  /* show all parameters for DBG purposes */
  for (i=0; i<argc; i++) {
    DBG("argv[%d]=%s\n", i, argv[i]);
  }

  reset_getopt();
  while(1) {
    int option_index = 0, c=0;
    static struct option long_options[] = \
    {
      {"h", no_argument, 0, 0},
      {"help", no_argument, 0, 0},
      {"d", required_argument, 0, 0},
      {"delay", required_argument, 0, 0},
      {"r", required_argument, 0, 0},
      {"resolution", required_argument, 0, 0},
      {"c", required_argument, 0, 0},
      {"channel", required_argument, 0, 0},
      {0, 0, 0, 0}
    };

    c = getopt_long_only(argc, argv, "", long_options, &option_index);

    /* no more options to parse */
    if (c == -1) break;

    /* unrecognized option */
    if (c == '?'){
      help();
      return 1;
    }

    switch (option_index) {
      /* h, help */
      case 0:
      case 1:
        DBG("case 0,1\n");
        help();
        return 1;
        break;

      /* d, delay */
      case 2:
      case 3:
        DBG("case 2,3\n");
        delay = atoi(optarg);
        break;

      /* r, resolution */
      case 4:
      case 5:
        DBG("case 4,5\n");
        width = -1;
        height = -1;

        /* parse value as decimal value */
        width  = strtol(optarg, &s, 10);
        height = strtol(s+1, NULL, 10);
        break;
        /* r, resolution */
      case 6:
      case 7:
        DBG("case 6,7\n");
        channel = atoi(optarg);
        break;

      default:
        DBG("default case\n");
        help();
        return 1;
    }
  }

  pglobal = param->global;

  IPRINT("delay.............: %i\n", delay);

  buf_size = (width * height) >> 3;
  pglobal->buf = malloc(buf_size);
  if (pglobal->buf == NULL) {
    fprintf(stderr, "could not allocate memory\n");
    exit(1);
  }

  if ((handle = frames_client_init()) == NULL)
    exit(1);

  return 0;
}

int input_stop(void) {
  DBG("will cancel input thread\n");
  pthread_cancel(fc);

  return 0;
}

int input_run(void) {
  if( pthread_create(&fc, 0, fc_thread, NULL) != 0) {
    fprintf(stderr, "could not start shm thread\n");
    exit(EXIT_FAILURE);
  }
  pthread_detach(fc);

  return 0;
}

/*** private functions for this plugin below ***/
void help(void) {
    fprintf(stderr, " ---------------------------------------------------------------\n" \
                    " Help for input plugin..: "INPUT_PLUGIN_NAME"\n" \
                    " ---------------------------------------------------------------\n" \
                    " The following parameters can be passed to this plugin:\n\n" \
                    " [-d | --delay ]........: delay to pause between frames\n" \
                    " ---------------------------------------------------------------\n");
}

/* the single writer thread */
void *fc_thread( void *arg ) {
  int connect=0;
  size_t datalen;
  unsigned int seqNum = 0, refer_seqNum = 0;
  struct timeval sts;
  int key_frame;
  /* set cleanup handler to cleanup allocated ressources */
  pthread_cleanup_push(fc_cleanup, NULL);

  while( !pglobal->stop ) {
    pthread_mutex_lock( &pglobal->db );
    connect=pglobal->connectNumber;
    pthread_mutex_unlock( &pglobal->db );
  
    if(connect==0) {
      usleep(500000);
      seqNum = 0;
      frames_client_disconnect(handle, channel, MSGQ_R_MJPG_STREAMER);
    } else {
      frames_client_connect(handle, channel, MSGQ_R_MJPG_STREAMER);
      pthread_mutex_lock( &pglobal->db );

      /* grab a frame */
      while(1) {
        if (frames_client_read_frame(handle, channel, MEDIA_CAT_VIDEO_INDEX, MSGQ_R_MJPG_STREAMER, MSG_CMD_GET_FRAME_BY_SEQNUM_FORWARD, pglobal->buf, buf_size, &datalen, &seqNum, &refer_seqNum, &sts, &key_frame) >= 0) {
          DBG("seqNum = %d\n", seqNum);            
          pglobal->size = datalen;
          seqNum++;
          break;
        }
        usleep(delay*1000);
      }

      /* signal fresh_frame */
      pthread_cond_broadcast(&pglobal->db_update);
      pthread_mutex_unlock( &pglobal->db );

      usleep(delay*1000);
    }
  }

  frames_client_disconnect(handle, channel, MSGQ_R_MJPG_STREAMER);

  DBG("leaving input thread, calling cleanup function now\n");
  pthread_cleanup_pop(1);

  pglobal->input_stoped = 1;
  return NULL;
}

void fc_cleanup(void *arg) {
  if (pglobal->buf != NULL) free(pglobal->buf);
  frames_client_deinit(handle);
}

