#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "debug.h"
#include "rbuf.h"


/// Internal functions.
inline int rbuf_room(const rbuf_t *rb)
{
	return (rb->size - rb->len);
}


/// API functions
int rbuf_init(rbuf_t *rb, int size)
{
	DBG_PRINTF(DBG_INFO, "rbuf_init.  size: %d\n", size);

	if ( (rb->buf=malloc(size)) == NULL) {
		DBG_PRINTF(DBG_ERROR, "Error: malloc\n");
		return RBUF_ERR_MALLOC;
	}
	rb->size = size;
	rb->len = 0;
	rb->ridx = rb->widx = 0;
	return RBUF_ERR_OK;
}


int rbuf_advance_read_pos(rbuf_t *rb, int nlen)
{
	DBG_PRINTF(DBG_INFO, "rbuf_advance_read_pos  %d bytes\n", nlen);

	if (rb->len < nlen) {
		DBG_PRINTF(DBG_WARNING,  "Warning! Advance %d bytes while only %d bytes in buffer\n",
									nlen, rb->len);	
		return RBUF_ERR_READ_INSUFFICIENT_DATA;
	}
	rb->len -= nlen;
	rb->ridx = (rb->ridx+nlen) % rb->size;
	DBG_PRINTF(DBG_INFO, "rb: len %d, ridx %d, widx %d\n", rb->len, rb->ridx, rb->widx);
	return RBUF_ERR_OK;
}


int rbuf_read(rbuf_t *rb, void *buf, int nlen)
{
	DBG_PRINTF(DBG_INFO, "rbuf_read %d bytes\n", nlen);

	if (rb->len < nlen) {
		DBG_PRINTF(DBG_WARNING, "Warning! Advance %d bytes while only %d bytes in buffer\n",
									nlen, rb->len);	
		return RBUF_ERR_READ_INSUFFICIENT_DATA;
	}

	// Data could overwrap, so we may have to copy 1) tail data, then followed by an optional 2) head data.
	int taillen=rb->size-rb->ridx;

	if (taillen < nlen) {
		// overwrap reading
		memcpy((unsigned char*)buf, &rb->buf[rb->ridx], taillen);	// Tail data
		memcpy((unsigned char*)buf+taillen, rb->buf, nlen-taillen);	// head data
	} else {
		// normal reading
		memcpy((unsigned char*)buf, &rb->buf[rb->ridx], nlen);
	}

	rb->len -= nlen;
	rb->ridx = (rb->ridx+nlen) % rb->size;
	DBG_PRINTF(DBG_INFO, "rb: len %d, ridx %d, widx %d\n", rb->len, rb->ridx, rb->widx);
	return RBUF_ERR_OK;
}


int rbuf_write(rbuf_t *rb, const void *data, int nlen)
{
	DBG_PRINTF(DBG_INFO, "rbuf_write %d bytes\n", nlen);

	if (nlen > rb->size) {
		DBG_PRINTF(DBG_ERROR, "Error: data of %d bytes to be written is too great than rign buffer size %d\n", 
												nlen, rb->size);
		return RBUF_ERR_WRITE_DATA_TOO_LARGE;
	}
	if (nlen > rbuf_room(rb)) {
		DBG_PRINTF(DBG_ERROR, "Error: data of %d bytes to be written is too great than avaiable rign buffer space %d\n", 
							nlen, rbuf_room(rb));
		return RBUF_ERR_WRITE_INSUFFICIENT_SPACE;
	}
		
	// Again, it's likely we need to store the data into two portions, 1) tail room, and optionally 2) head room.
	int tailroom=rb->size-rb->widx;

	if (tailroom < nlen) {
		// overwrap writing.
		memcpy(&rb->buf[rb->widx], (unsigned char*)data, tailroom);		// tail
		memcpy(rb->buf, (unsigned char*)data+tailroom, nlen-tailroom);	// head room
	} else {
		// normal.
		memcpy(&rb->buf[rb->widx], (unsigned char*)data, nlen);
	}

	rb->len += nlen;
	rb->widx = (rb->widx+nlen) % rb->size;
	DBG_PRINTF(DBG_INFO, "rb: len %d, ridx %d, widx %d\n", rb->len, rb->ridx, rb->widx);
	return RBUF_ERR_OK;
}



void rbuf_release(rbuf_t *rb)
{
	DBG_PRINTF(DBG_INFO, "rbuf_release\n");

	if (rb) {
		if (rb->buf) 	free (rb->buf);
	}
}

void rbuf_dump(const rbuf_t *rb)
{
	DBG_PRINTF(DBG_INFO, "rbuf_dump\n");

	DBG_PRINTF(DBG_INFO, "rbuf::  size %d, len %d, buf: %p, ridx %d, widx: %d\n",
				rb->size, rb->len, rb->buf, rb->ridx, rb->widx);
}
