#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "debug.h"
#include "prealarm.h"


/// frame queue 
static void fqueue_init(frame_queue_t *q)	{ STAILQ_INIT(q);	}

static frame_t *fqueue_detach_head(frame_queue_t *q)
{
	frame_t *frame=STAILQ_FIRST(q);
	
	if (frame) STAILQ_REMOVE_HEAD(q, entry);
	return frame;
}

static void fqueue_insert_tail(frame_queue_t *q, frame_t *frame) {  STAILQ_INSERT_TAIL(q, frame, entry);	}

static void fqueue_dump(frame_queue_t *q) 
{
	frame_t *frame;

	STAILQ_FOREACH(frame, q, entry) {
		DBG_PRINTF(DBG_INFO, "frame: len: %d, pts (%d,%d)\n", 
						frame->len, frame->pts.tv_sec, frame->pts.tv_usec);
	}
}


/// prealarm API
int prealarm_init(prealarm_buf_t *preAlarm, int bufsize, int fcount)
{	
	DBG_PRINTF(DBG_INFO, "prealarm_init.  buffer size: %d,  frame count: %d\n", bufsize, fcount);

	if (preAlarm) {
		int i;
		frame_t *frame;
		
		memset(preAlarm, 0x0, sizeof(prealarm_buf_t));

		if (bufsize <= 0) {
			preAlarm->disabled=1;
			return PREALARM_ERR_OK;
		}

		// init ring buffer
		if (rbuf_init(&preAlarm->rbuf, bufsize) < 0) {
			goto ERROR;
		}
		preAlarm->pRbuf=&preAlarm->rbuf;

		// init frame queue..
		fqueue_init(&preAlarm->activeQ);
		preAlarm->pActiveQ = &preAlarm->activeQ;
		fqueue_init(&preAlarm->freeQ);
		preAlarm->pFreeQ = &preAlarm->freeQ;

		// Add frames to the free queue.
		frame=preAlarm->mem=malloc(fcount*sizeof(frame_t));
		if (!frame) {
			DBG_PRINTF(DBG_ERROR, "Error: malloc\n");
			goto ERROR;
		}
		for (i=0; i < fcount; i++, frame++) {
			fqueue_insert_tail(preAlarm->pFreeQ, frame);
		}
		return PREALARM_ERR_OK;

ERROR:
		prealarm_release(preAlarm);	
	}
	return PREALARM_ERR_GENERIC;	

}

int prealarm_init1(prealarm_buf_t *preAlarm, int bitrateKbps, int duration)
{
	int bytes = (bitrateKbps * duration * 1000) / 8 * 3/2;
	int fcount = duration * 100; 	// Assume max 100 fps.
	return prealarm_init(preAlarm, bytes, fcount);
}


int prealarm_store(prealarm_buf_t *preAlarm, frame_t *frame)
{
	int rc;
	frame_t *new_frame, *oldest_frame;

	DBG_PRINTF(DBG_INFO, "prealarm_store  frame %d bytes, pts (%d,%d) \n", 
					frame->len, frame->pts.tv_sec, frame->pts.tv_usec);

	if (preAlarm->disabled) {
		return PREALARM_ERR_GENERIC;
	}

	// Fetch a frame from free queue.
	if ( (new_frame=fqueue_detach_head(preAlarm->pFreeQ)) == NULL) {

		// No more free frame_t, kick the oldest out of the active queue..
		oldest_frame=fqueue_detach_head(preAlarm->pActiveQ);
		
		if (!oldest_frame) {
			DBG_PRINTF(DBG_ERROR, "Error: no more free frame\n");
			return PREALARM_ERR_STORE_NO_FREE_FRAME;
		}

		// Advance the read position with this old frame datalen.
		rbuf_advance_read_pos(preAlarm->pRbuf, oldest_frame->len);

		// reuse the frame.
		new_frame=oldest_frame;
	}

	// shallow copy the data..
	memcpy(new_frame, frame, sizeof(frame_t));
			
WRITE_AGAIN:
	// Store the new frame data into the ring buffer.
	rc=rbuf_write(preAlarm->pRbuf, new_frame->buf, new_frame->len);

	if (rc == RBUF_ERR_WRITE_INSUFFICIENT_SPACE) {
				
		// No space in rbuffer, so discard the oldest frame in the active Q.
		if ( (oldest_frame=fqueue_detach_head(preAlarm->pActiveQ)) == NULL) {
			// We have dumped everything out from ring buffer but still don't have room to take new frame....
			rc = RBUF_ERR_WRITE_DATA_TOO_LARGE;
			goto WRITE_DONE;
		}

		DBG_PRINTF(DBG_INFO, "Discard old frame len %d, pts (%d, %d)\n", 
					oldest_frame->len, oldest_frame->pts.tv_sec, oldest_frame->pts.tv_usec);
		
		// Advance the read position with this old frame datalen.
		rbuf_advance_read_pos(preAlarm->pRbuf, oldest_frame->len);

		// Recycle the frame.
		fqueue_insert_tail(preAlarm->pFreeQ, oldest_frame);
			
		// Try write again.
		goto WRITE_AGAIN;
	}
	
WRITE_DONE:

	if (rc >= 0) {
		fqueue_insert_tail(preAlarm->pActiveQ, new_frame);
		return PREALARM_ERR_OK;
	} else {
		// Encounter one of these errors:
		//	RBUF_ERR_WRITE_DATA_TOO_LARGE
		//	...

		// Discard and recyle new frame..
		fqueue_insert_tail(preAlarm->pFreeQ, new_frame);
		return PREALARM_ERR_GENERIC;
	}
	
}

int prealarm_get(prealarm_buf_t *preAlarm, frame_t *frame)
{
	frame_t * stored_frame;
	
	DBG_PRINTF(DBG_INFO, "prealarm_get  frame: buffer size %d\n", frame->size);

	if (preAlarm->disabled)
		return PREALARM_ERR_GET_NO_STORED_FRAME;

	if ( (stored_frame=fqueue_detach_head(preAlarm->pActiveQ)) == NULL) {
		return PREALARM_ERR_GET_NO_STORED_FRAME;
	}

	if (stored_frame->prepare_get_frame_cb) {
		stored_frame->prepare_get_frame_cb(stored_frame->context, frame);
	}
	
	if (frame->size < stored_frame->len) {
		DBG_PRINTF(DBG_WARNING, "Warning: buffer too small.  Truncation occurs.\n"
						"Attempt to read %d bytes data into %d bytes buffer !\n",
						stored_frame->len, frame->size);
		
		rbuf_read(preAlarm->pRbuf, frame->buf, frame->size);
		rbuf_advance_read_pos(preAlarm->pRbuf, stored_frame->len - frame->size);
		frame->len = frame->size;
	}  else {
		rbuf_read(preAlarm->pRbuf, frame->buf, stored_frame->len);	
		frame->len = stored_frame->len;
	}
	frame->pts = stored_frame->pts;
			
#ifdef MULTI_SLICE
    // restore the slice count and the array of the slice size
    int i;
    frame->slice_count = stored_frame->slice_count;
    for (i = 0 ; i < frame->slice_count ; i++)
        frame->slice_size[i] = stored_frame->slice_size[i];
#endif
	// recycle the frame.
	fqueue_insert_tail(preAlarm->pFreeQ, stored_frame);
	return PREALARM_ERR_OK;
	
}

void prealarm_dump(prealarm_buf_t *preAlarm)
{
	DBG_PRINTF(DBG_INFO, "prealarm_dump\n");

	DBG_PRINTF(DBG_INFO, "\n\n--- Active Queue ---\n");
	fqueue_dump(preAlarm->pActiveQ);
	
	DBG_PRINTF(DBG_INFO, "\n\n--- Free Queue ---\n");
	fqueue_dump(preAlarm->pFreeQ);

	DBG_PRINTF(DBG_INFO, "\n\n--- Ring buffer ---\n");
	rbuf_dump(preAlarm->pRbuf);
}

void prealarm_release(prealarm_buf_t *preAlarm)
{
	DBG_PRINTF(DBG_INFO, "prealarm_release\n");

	if (preAlarm) {
		if (preAlarm->pRbuf)	rbuf_release(preAlarm->pRbuf);
		if (preAlarm->mem)	free(preAlarm->mem);
	}
}
