/***************************************************************
  * rbuf.h
  *	A ring buffer implementation which can be used as a FIFO.
  *
  *	However, no locking mechaism is used for serialize concurrent access 
  *   among multiple readers.  
  *
  *	It can only support single reader, single writer.
  *	This means it can support 
  *		1 thread: as both reader and writer
  *		2 threads: 1 reader, 1 writer.
  ***************************************************************/
#ifndef _RBUF_H
#define _RBUF_H


#ifdef __cplusplus
extern "C" {
#endif


typedef struct _rbuf {
	unsigned char *buf;		/* the buffer of ring buffer */
	int size;					/* size of memory pointed by buf */
	int len;					/* length of data in buf */
	int ridx;					/* read index */
	int widx;					/* write index */
} rbuf_t;


/// Error
#define RBUF_ERR_OK									0
#define RBUF_ERR_MALLOC							-1
#define RBUF_ERR_READ_INSUFFICIENT_DATA			-2
#define RBUF_ERR_WRITE_DATA_TOO_LARGE			-3
#define RBUF_ERR_WRITE_INSUFFICIENT_SPACE		-4



/*******************************************************
  * rbuf_init
  *	initialize a ring buffer
  *
  * @ Parameters
  *	[IN] rb		ring buffer to be init.
  *	[IN] size		Bounded size of the ring buffer.
  *
  * @Return value
  *   >=0, success
  *		RBUF_ERR_OK
  *  < 0, failure
  *		RBUF_ERR_MALLOC
  *******************************************************/
extern int rbuf_init(rbuf_t *rb, int size);



/**********************************************************************
  * rbuf_advance_read_pos
  *	Advance read position of a ring buffer.
  *
  * @ Parameters
  *	[IN] rb		the ring buffer.
  *	[IN] nlen		number of bytes to be advanced (not really consumed).
  *
  * @Return value
  *	If successful, value >=0 is returned and read position is adavnced by 'nlen' bytes.
  *		  RBUF_ERR_OK
  *   If failed, value < 0 is returned and read position remained as before.
  *		RBUF_ERR_READ_INSUFFICIENT_DATA
  **********************************************************************/
extern int rbuf_advance_read_pos(rbuf_t *rb, int nlen);


/**************************************************************
  * rbuf_read
  *	Consume data from a ring buffer.
  *
  * @ Parameters
  *	[IN] rb		the ring buffer.
  *	[OUT] buf	the buffer to store the data. It must has size greater 
  *				or equal to 'nlen' bytes.
  *	[IN] nlen		number of bytes of data to be read (consumed).
  *
  * @Return value
  *	if successfulm, >=0 is returned, and 'nlen' bytes of data is copied into buf, 
  *	the read position is also advanced by 'nlen' bytes.
  *		RBUF_ERR_OK
  *
  *	If failed, < 0 is returned and read position remains as before.
  *		RBUF_ERR_READ_INSUFFICIENT_DATA
  ***************************************************************/
extern int rbuf_read(rbuf_t *rb, void *buf, int nlen);


/**************************************************************
  * rbuf_write
  *	Write data into to a ring buffer 
  *
  * @ Parameters
  *	[IN] rb		the ring buffer.
  *	[IN] data		the data to be stored into the ring buffer
  *	[IN] nlen 		number of bytes of data to be written (produced).
  *
  * @Return value
  *	>=0 if sucessful
  *		RBUF_ERR_OK
  *
  *   < 0 failed.
  * 		RBUF_ERR_WRITE_DATA_TOO_LARGE
  *		RBUF_ERR_WRITE_INSUFFICIENT_SPACE
  ***************************************************************/
extern int rbuf_write(rbuf_t *rb, const void *data, int nlen);


/*********************************************
  * rbuf_release
  *	Function to clean up the internals of a ring buffer
  *
  * @ Parameters
  *	[IN] rb	the ring buffer.
  *
  * @Return value
  *	none
  ***********************************************/
extern void rbuf_release(rbuf_t *rb);


/*************************************
 * rbuf_dump
 *	Dump out the ring buffer information.
 *
 * @Parameters
 *	[IN] rb 	the ring buffer.
 *
 * @REturn value
 *	none
 **************************************/
extern void rbuf_dump(const rbuf_t *rb);


#ifdef __cplusplus
}
#endif

#endif /* _RBUF_H */
