/**********
This library is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the
Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version. (See <http://www.gnu.org/copyleft/lesser.html>.)

This library is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
more details.

You should have received a copy of the GNU Lesser General Public License
along with this library; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
**********/
// "liveMedia"
// Copyright (c) 1996-2008 Live Networks, Inc.  All rights reserved.
// A RTSP server
// Implementation

#include "RTSPServer.hh"
#include "RTSPCommon.hh"
#include "Base64.hh"
#include <GroupsockHelper.hh>
#include "debug.h"



#if defined(__WIN32__) || defined(_WIN32) || defined(_QNX4)
#else
#include <signal.h>
#define USE_SIGNALS 1
#endif
#include <time.h> // for "strftime()" and "gmtime()"


////////// RTSPServer //////////

// TODO: later we may have more than 1 server..
static RTSPServer* theOnlyRTSPServer=NULL;

extern int g_rtcpSREnabled;


RTSPServer*
RTSPServer::createNew(UsageEnvironment& env, Port ourPort,
		      UserAuthenticationDatabase* authDatabase,
		      unsigned reclamationTestSeconds, int rtcpSREnabled) {
  int ourSocket = -1;

  do {
    ourSocket = setUpOurSocket(env, ourPort);
    if (ourSocket == -1) break;

    return new RTSPServer(env, ourSocket, ourPort, authDatabase,
			  reclamationTestSeconds, rtcpSREnabled);
  } while (0);

  if (ourSocket != -1) ::closeSocket(ourSocket);
  return NULL;
}

Boolean RTSPServer::lookupByName(UsageEnvironment& env,
				 char const* name,
				 RTSPServer*& resultServer) {
  resultServer = NULL; // unless we succeed

  Medium* medium;
  if (!Medium::lookupByName(env, name, medium)) return False;

  if (!medium->isRTSPServer()) {
    env.setResultMsg(name, " is not a RTSP server");
    return False;
  }

  resultServer = (RTSPServer*)medium;
  return True;
}

void RTSPServer::addServerMediaSession(ServerMediaSession* serverMediaSession) {
  if (serverMediaSession == NULL) return;

  char const* sessionName = serverMediaSession->streamName();
  DBG_PRINTF(DBG_INFO,"RTSPServer add SMS %p %s\n", serverMediaSession, sessionName);
  if (sessionName == NULL) sessionName = "";
  ServerMediaSession* existingSession
    = (ServerMediaSession*)
    (fServerMediaSessions->Add(sessionName,
			       (void*)serverMediaSession));
  removeServerMediaSession(existingSession); // if any
}

ServerMediaSession* RTSPServer::lookupServerMediaSession(char const* streamName) {
  return (ServerMediaSession*)(fServerMediaSessions->Lookup(streamName));
}

void RTSPServer::removeServerMediaSession(ServerMediaSession* serverMediaSession) {
  if (serverMediaSession == NULL) return;
	
	DBG_PRINTF(DBG_INFO,"RTSPServer remove SMS %p %s, referenceCount = %d\n", 
		serverMediaSession, serverMediaSession->streamName(), serverMediaSession->referenceCount());
		
  fServerMediaSessions->Remove(serverMediaSession->streamName());
  if (serverMediaSession->referenceCount() == 0) {
  	DBG_PRINTF(DBG_INFO,"Medium::close() %p\n", serverMediaSession); 
    Medium::close(serverMediaSession);
  } else {
    serverMediaSession->deleteWhenUnreferenced() = True;
  }
}

void RTSPServer::removeServerMediaSession(char const* streamName) {
  removeServerMediaSession(lookupServerMediaSession(streamName));
}

void RTSPServer::addUser(char const *name, char const *password) {
	if (fAuthDB) {
  		fAuthDB->addUserRecord(name, password);
	}
}

void RTSPServer::removeUser(char const *name) {
	if (fAuthDB) {
  		fAuthDB->removeUserRecord(name);
	}
}

void RTSPServer::updateUser(char const *name, char const *password) {
	if (fAuthDB) {
  		fAuthDB->removeUserRecord(name);
  		fAuthDB->addUserRecord(name, password);
	}
}


void RTSPServer::setDSCP(int videoDSCP, int audioDSCP)
{
	RTSPClientSession *session;

	fVideoDSCP = videoDSCP;
	fAudioDSCP = audioDSCP;
	
	for (session=fClientSessionHead; session; session=session->fNext) {
		session->setDSCP(videoDSCP, audioDSCP);
	}		
}



RTSPServer::RTSPClientSession *RTSPServer::findRTSPClientSession(int sockfd) {
	RTSPClientSession *session;

	for (session=fClientSessionHead; session; session=session->fNext) {

		if (sockfd==session->getSocketDescriptor()) {
			return session;
		}
	}
	return NULL;
}


void RTSPServer::deleteRTSPClientSession(int sockfd) 
{
	RTSPClientSession *clientSession;

	clientSession=findRTSPClientSession(sockfd);
	if (!clientSession) {
		fprintf(stderr, "Warning: can't find RTSPClientSession with sockfd: %d\n", sockfd);
		return;
	}
	deleteRTSPClientSession(clientSession);
}

void RTSPServer::addRTSPClientSession(RTSPClientSession *clientSession)
{
	clientSession->fNext = fClientSessionHead;
	fClientSessionHead = clientSession;
}

void RTSPServer::deleteRTSPClientSession(RTSPClientSession *clientSession) {
	RTSPClientSession *prev, *ptr;

	// printf("Deleting RTSPClientSession: %p\n", clientSession);
		
	prev=NULL;
	for (ptr=fClientSessionHead; ptr; ptr=ptr->fNext) {

		if (ptr==clientSession) {

			if (prev) 
				prev->fNext = ptr->fNext;
			else
				fClientSessionHead = ptr->fNext;
			
			delete ptr;		
			// printf("RTSPClientSession deleted\n");
			break;
		}
		prev=ptr;
	}
}

void RTSPServer::closeRTSPClientSessions(ServerMediaSession* serverMediaSession) {
	RTSPClientSession *prev, *ptr, *next;

	prev=NULL;
	for (ptr=fClientSessionHead; ptr; ptr=next) {
		next = ptr->fNext;
		if(ptr->getServerMediaSession() == serverMediaSession) {
			if (prev)
				prev->fNext =next;				
			else
				fClientSessionHead=next;	// At the head.
			DBG_PRINTF(DBG_INFO,"delete RTSPClientSession: %p\n", ptr);
			delete ptr;
		} else {
			prev = ptr;
		}
	}
}

char* RTSPServer::rtspURLPrefix(int clientSocket) const {
  struct sockaddr_in ourAddress;
  if (clientSocket < 0) {
    // Use our default IP address in the URL:
    ourAddress.sin_addr.s_addr = ReceivingInterfaceAddr != 0
      ? ReceivingInterfaceAddr
      : ourIPAddress(envir()); // hack
  } else {
    SOCKLEN_T namelen = sizeof ourAddress;
    getsockname(clientSocket, (struct sockaddr*)&ourAddress, &namelen);
  }

  char urlBuffer[100]; // more than big enough for "rtsp://<ip-address>:<port>/"

  portNumBits portNumHostOrder = ntohs(fServerPort.num());
  if (portNumHostOrder == 554 /* the default port number */) {
    sprintf(urlBuffer, "rtsp://%s/", our_inet_ntoa(ourAddress.sin_addr));
  } else {
    sprintf(urlBuffer, "rtsp://%s:%hu/",
	    our_inet_ntoa(ourAddress.sin_addr), portNumHostOrder);
  }

  return strDup(urlBuffer);
}

char* RTSPServer
::rtspURL(ServerMediaSession const* serverMediaSession, int clientSocket) const {
  char* urlPrefix = rtspURLPrefix(clientSocket);
  char const* sessionName = serverMediaSession->streamName();

  char* resultURL = new char[strlen(urlPrefix) + strlen(sessionName) + 1];
  sprintf(resultURL, "%s%s", urlPrefix, sessionName);

  delete[] urlPrefix;
  return resultURL;
}

#define LISTEN_BACKLOG_SIZE 20

int RTSPServer::setUpOurSocket(UsageEnvironment& env, Port& ourPort) {
  int ourSocket = -1;

  do {
    ourSocket = setupStreamSocket(env, ourPort);
    if (ourSocket < 0) break;

    // Make sure we have a big send buffer:
    if (!increaseSendBufferTo(env, ourSocket, 50*1024)) break;

    // Allow multiple simultaneous connections:
    if (listen(ourSocket, LISTEN_BACKLOG_SIZE) < 0) {
      env.setResultErrMsg("listen() failed: ");
      break;
    }

    if (ourPort.num() == 0) {
      // bind() will have chosen a port for us; return it also:
      if (!getSourcePort(env, ourSocket, ourPort)) break;
    }

    return ourSocket;
  } while (0);

  if (ourSocket != -1) ::closeSocket(ourSocket);
  return -1;
}

Boolean RTSPServer
::specialClientAccessCheck(int /*clientSocket*/, struct sockaddr_in& /*clientAddr*/, char const* /*urlSuffix*/) {
  // default implementation
  return True;
}

Boolean RTSPServer
::customAuthenticationOK(int /*clientSocket*/, struct sockaddr_in& /*clientAddr*/, char const* /*urlSuffix*/, 
		char const* /*fullRequestStr*/) {
	// default implementation
	return False;
}

void RTSPServer::logFailedAuthentication(const char *username, const char *clientIPAddress)
{
	/* default implementation */
}

void RTSPServer::logSessionStart(unsigned sessionId, const char *channelName, const char *username, const char* clientIPAddress)
{
	/* default implementation */
}

void RTSPServer::logSessionStop(unsigned sessionId, const char *channelName, const char *username, const char* clientIPAddress)
{
	/* default implementation */
}


RTSPServer::RTSPServer(UsageEnvironment& env,
		       int ourSocket, Port ourPort,
		       UserAuthenticationDatabase* authDatabase,
		       unsigned reclamationTestSeconds,
		       int rtcpSREnabled)
  : Medium(env),
    fServerSocket(ourSocket), fServerPort(ourPort),
    fAuthDB(authDatabase), fReclamationTestSeconds(reclamationTestSeconds),
    fServerMediaSessions(HashTable::create(STRING_HASH_KEYS)),
    fSessionIdCounter(0), fClientSessionHead(NULL),
    fVideoDSCP(0), fAudioDSCP(0) {
#ifdef USE_SIGNALS
  // Ignore the SIGPIPE signal, so that clients on the same host that are killed
  // don't also kill us:
  signal(SIGPIPE, SIG_IGN);
#endif

  setCloseOnExec(fServerSocket, True);

  // Arrange to handle connections from others:
  env.taskScheduler().turnOnBackgroundReadHandling(fServerSocket,
        (TaskScheduler::BackgroundHandlerProc*)&incomingConnectionHandler,
						   this);

   g_rtcpSREnabled=rtcpSREnabled;
		
  // Finally assign theOnlyRTSPServer to this..
  theOnlyRTSPServer = this;
}

RTSPServer::~RTSPServer() {
  // Turn off background read handling:
  envir().taskScheduler().turnOffBackgroundReadHandling(fServerSocket);

  ::closeSocket(fServerSocket);

  // Remove all server media sessions (they'll get deleted when they're finished):
  while (1) {
    ServerMediaSession* serverMediaSession
      = (ServerMediaSession*)fServerMediaSessions->RemoveNext();
    if (serverMediaSession == NULL) break;
    removeServerMediaSession(serverMediaSession);
  }

  // Finally, delete the session table itself:
  delete fServerMediaSessions;

#if 1	// Hack.  Steve, 2009/07/07
	theOnlyRTSPServer=NULL;
#endif
}

Boolean RTSPServer::isRTSPServer() const {
  return True;
}

void RTSPServer::incomingConnectionHandler(void* instance, int /*mask*/) {
  RTSPServer* server = (RTSPServer*)instance;
  server->incomingConnectionHandler1();
}

// TODO: Hack to avoid excessive heap fragmentation.
#include <malloc.h>
#include <signal.h>

static void attempt_reset_stream_server()
{
	struct mallinfo info;
	
	info = mallinfo();
	if (info.arena > 500 * 1024) {
		DBG_PRINTF(DBG_INFO,"Heap/total [%d,%d] grows too large. Restart server!\n", info.arena, info.usmblks);
		kill(getpid(), SIGTERM);	
	}
}

void RTSPServer::prepareConnection(int clientSocket)
{
	//attempt_reset_stream_server();

	makeSocketNonBlocking(clientSocket);

	// Steve, 2009/07/09.
	// Note: To avoid RTPoverTCP client not able to handle burst condition, 
	//		 increase from 50K to 150K for send buffer size.  This should not place burden
	//		 on the normal UDP usage as we don't have too much data to send.
	increaseSendBufferTo(envir(), clientSocket, 50 *1024);

	if (!fReclamationTestSeconds) {
		// No RTCP keepalive, fallback to TCP keepalive. 1 (on), 3 (times), 30 (idle), 30 (intvl) == 2 minutes
		setTCPSocketKeepalive(clientSocket, 1, 3, 30, 30);
	}
}

void RTSPServer::incomingConnectionHandler1() {
  struct sockaddr_in clientAddr;
  SOCKLEN_T clientAddrLen = sizeof clientAddr;
  RTSPClientSession *tmp;
  int clientSocket = accept(fServerSocket, (struct sockaddr*)&clientAddr,
			    &clientAddrLen);
  if (clientSocket < 0) {
    int err = envir().getErrno();
    if (err != EWOULDBLOCK) {
        envir().setResultErrMsg("accept() failed: ");
    }
    return;
  }

  prepareConnection(clientSocket);
  

#if defined(DEBUG) || defined(DEBUG_CONNECTIONS)
  fprintf(stderr, "accept()ed connection from %s\n", our_inet_ntoa(clientAddr.sin_addr));
#endif

	fprintf(stderr, "accept()ed connection from %s\n", our_inet_ntoa(clientAddr.sin_addr));

	// Create a new object for this RTSP session:
	tmp = new RTSPClientSession(*this, ++fSessionIdCounter, clientSocket, clientAddr);
	if (tmp) {
		addRTSPClientSession(tmp);		
	}
}


////////// RTSPServer::RTSPClientSession implementation //////////

RTSPServer::RTSPClientSession
::RTSPClientSession(RTSPServer& ourServer, unsigned sessionId,
	      int clientSocket, struct sockaddr_in clientAddr)
  : fOurServer(ourServer), fOurSessionId(sessionId),
    fOurServerMediaSession(NULL),
    fClientSocket(clientSocket), fClientAddr(clientAddr),
    fLivenessCheckTask(NULL),
    fIsMulticast(False), fSessionIsActive(True), fStreamAfterSETUP(False),
    fTCPStreamIdCount(0), fNumStreamStates(0), fStreamStates(NULL), fNext(NULL) {

	// Arrange to handle incoming requests:
	strcpy(fUsername, "anonymous");
	fAuthenOK=False;
	fFailedAuthenCount=0;
	fUserAgentID=USER_AGENT_UNKNOWN;
	fStreamingMode = RTP_UDP;	// default..


	fUseExternalAddress = False;
	fExtHostname[0] = '\0';
	fExtPort=0;
	
	
  	resetRequestBuffer();
  	envir().taskScheduler().turnOnBackgroundReadHandling(fClientSocket,
     		(TaskScheduler::BackgroundHandlerProc*)&incomingRequestHandler, this);
  	noteLiveness();
}

RTSPServer::RTSPClientSession::~RTSPClientSession() {

	if (!fAuthenOK && fFailedAuthenCount > 0) {
		fOurServer.logFailedAuthentication(fUsername, inet_ntoa(fClientAddr.sin_addr));
	}

	// Turn off any liveness checking:
	envir().taskScheduler().unscheduleDelayedTask(fLivenessCheckTask);

	// Turn off background read handling:
	envir().taskScheduler().turnOffBackgroundReadHandling(fClientSocket);

	::closeSocket(fClientSocket);

	reclaimStreamStates();

	if (fOurServerMediaSession != NULL) {
		
		fOurServer.logSessionStop(fOurSessionId, fOurServerMediaSession->channelName(), 
									fUsername,  inet_ntoa(fClientAddr.sin_addr));
		
		fOurServerMediaSession->decrementReferenceCount();

		// TODO: remove...
		// printf("~RTSPClientSession(): reference count = %d\n", fOurServerMediaSession->referenceCount());
		if (fOurServerMediaSession->referenceCount() == 0
			&& fOurServerMediaSession->deleteWhenUnreferenced()) {
      			fOurServer.removeServerMediaSession(fOurServerMediaSession);
    		}
  	}
}

Boolean RTSPServer::RTSPClientSession::needAuthentication()
{	
	return True;
}

// TODO: move this to other place?
void 
RTSPServer::RTSPClientSession::determineUserAgent(const char *urlPreSuffix, const char *fullRequestStr)
{
	// printf("urlPreSuffix: %s\n", urlPreSuffix);

	do {
		// RULE 1. check against stream URL prefix..
		if (!strcasecmp(urlPreSuffix, "onvif")) {
			fUserAgentID = USER_AGENT_ONVIF;
			break;	// found.
		}
	
		// RULE 2. check against user agent
		char userAgent[RTSP_PARAM_STRING_MAX];
		if ( parseUserAgentHeader(fullRequestStr, userAgent, sizeof(userAgent)) ) {
			if (strcasestr(userAgent, "VLC media player") != NULL) {
				fUserAgentID = USER_AGENT_VLC;
				break;
			} else if(strcasestr(userAgent, "QuickTime") != NULL) {
				fUserAgentID = USER_AGENT_QUICKTIME;
				break;
			} else if(strcasestr(userAgent, "BRICKCOM_MEDIA_PLAYER") != NULL) {
				fUserAgentID = USER_AGENT_BRICKCOM;
				break;
			}
			else if (!strncasecmp(userAgent, "JMF RTSP Player", 15)) {
				fUserAgentID = USER_AGENT_JMF;
				break;
			}
		}
		// RULEs...
	} while (0);


	// printf("media player type: %d\n", fUserAgentID);	
}



void RTSPServer::RTSPClientSession::parseExternalAddress(char *hostPortStr)
{
	// TODO: IPv6 ???
	int len = strlen(hostPortStr);
	char *start=hostPortStr;
	char *p=hostPortStr+len-1;

	if (len > 0) {
		// Search from the end...	
		while (p >= start) {

			if (*p == ':') {
				// port found!
				*p = '\0';
				fExtPort = atoi(p+1);
				break;
			}
			p--;
		}
		strcpy(fExtHostname, start);
		if (strcmp(fExtHostname, ""))
			fUseExternalAddress = True;
	}

}

char* RTSPServer::RTSPClientSession::
	getRTSPURL(ServerMediaSession const* serverMediaSession)
{
	if (fUseExternalAddress) {
		char urlPrefix[350];
		char const* sessionName = serverMediaSession->streamName();
		char* resultURL;

		if(fExtPort == 554 || fExtPort == 0)
			sprintf(urlPrefix, "rtsp://%s/", fExtHostname);
		else
			sprintf(urlPrefix, "rtsp://%s:%hu/", fExtHostname, fExtPort);
			
		resultURL = new char[strlen(urlPrefix) + strlen(sessionName) + 1];
		sprintf(resultURL, "%s%s", urlPrefix, sessionName);
		return resultURL;
		

	} else {
		char* urlPrefix = fOurServer.rtspURLPrefix(fClientSocket);		// TODO: ???
		char const* sessionName = serverMediaSession->streamName();

		char* resultURL = new char[strlen(urlPrefix) + strlen(sessionName) + 1];
		sprintf(resultURL, "%s%s", urlPrefix, sessionName);

		delete[] urlPrefix;
		return resultURL;
	}
}

int RTSPServer::RTSPClientSession::readRequest(unsigned char *ptr, unsigned size)
{
	struct sockaddr_in dummy; // 'from' address, meaningless in this case	
	return readSocket(envir(), fClientSocket, ptr, size, dummy);
}

int RTSPServer::RTSPClientSession::sendResponse()
{
	return send(fClientSocket, (char const*)fResponseBuffer, strlen((char*)fResponseBuffer), 0);	
}

void RTSPServer::RTSPClientSession::reclaimStreamStates() {
  for (unsigned i = 0; i < fNumStreamStates; ++i) {
    if (fStreamStates[i].subsession != NULL) {
      fStreamStates[i].subsession->deleteStream(fOurSessionId,
						fStreamStates[i].streamToken);
    }
  }
  delete[] fStreamStates; fStreamStates = NULL;
  fNumStreamStates = 0;
}

void RTSPServer::RTSPClientSession::resetRequestBuffer() {
  fRequestBytesAlreadySeen = 0;
  fRequestBufferBytesLeft = sizeof fRequestBuffer;
  fLastCRLF = &fRequestBuffer[-3]; // hack
}

void RTSPServer::RTSPClientSession::incomingRequestHandler(void* instance, int /*mask*/) {
  ((RTSPClientSession*)instance)->incomingRequestHandler1();
}

void RTSPServer::RTSPClientSession::incomingRequestHandler1() {
  int bytesRead = readRequest(&fRequestBuffer[fRequestBytesAlreadySeen], fRequestBufferBytesLeft);
  handleRequestBytes(bytesRead);
}

void RTSPServer::RTSPClientSession::handleAlternativeRequestByte(void* instance, u_int8_t requestByte) {
  ((RTSPClientSession*)instance)->handleAlternativeRequestByte1(requestByte);
}

void RTSPServer::RTSPClientSession::handleAlternativeRequestByte1(u_int8_t requestByte) {
  // Add this character to our buffer; then try to handle the data that we have buffered so far:
  if (fRequestBufferBytesLeft == 0|| fRequestBytesAlreadySeen >= RTSP_BUFFER_SIZE) return;
  fRequestBuffer[fRequestBytesAlreadySeen] = requestByte;
  handleRequestBytes(1);
}

void RTSPServer::RTSPClientSession::handleRequestBytes(int newBytesRead) {
  noteLiveness();

  if (newBytesRead <= 0 || (unsigned)newBytesRead >= fRequestBufferBytesLeft) {
    // Either the client socket has died, or the request was too big for us.
    // Terminate this connection:
#ifdef DEBUG
    fprintf(stderr, "RTSPClientSession[%p]::handleRequestBytes() read %d new bytes (of %d); terminating connection!\n", this, newBytesRead, fRequestBufferBytesLeft);
#endif
		// Remove RTSPClientSession from a linked list maintained server.
		this->fOurServer.deleteRTSPClientSession(this);	// delete this;
		return;
	}

  Boolean endOfMsg = False;
  unsigned char* ptr = &fRequestBuffer[fRequestBytesAlreadySeen];
#ifdef DEBUG
  ptr[newBytesRead] = '\0';
  fprintf(stderr, "RTSPClientSession[%p]::handleRequestBytes() read %d new bytes:%s\n", this, newBytesRead, ptr);
#endif

  // Look for the end of the message: <CR><LF><CR><LF>
  unsigned char *tmpPtr = ptr;
  if (fRequestBytesAlreadySeen > 0) --tmpPtr;
      // in case the last read ended with a <CR>
  while (tmpPtr < &ptr[newBytesRead-1]) {
    if (*tmpPtr == '\r' && *(tmpPtr+1) == '\n') {
      if (tmpPtr - fLastCRLF == 2) { // This is it:
	endOfMsg = True;
	break;
      }
      fLastCRLF = tmpPtr;
    }
    ++tmpPtr;
  }

  fRequestBufferBytesLeft -= newBytesRead;
  fRequestBytesAlreadySeen += newBytesRead;

  if (!endOfMsg) return; // subsequent reads will be needed to complete the request

  // Parse the request string into command name and 'CSeq',
  // then handle the command:
  fRequestBuffer[fRequestBytesAlreadySeen] = '\0';
  char cmdName[RTSP_PARAM_STRING_MAX];
  char hostPortStr[300];		// MAX FQDN 255 + ';' + xxxxx port
  char urlPreSuffix[RTSP_PARAM_STRING_MAX];
  char urlSuffix[RTSP_PARAM_STRING_MAX];
  char cseq[RTSP_PARAM_STRING_MAX];
  if (!parseRTSPRequestString((char*)fRequestBuffer, fRequestBytesAlreadySeen,
			      cmdName, sizeof cmdName,
			      hostPortStr, sizeof hostPortStr,
			      urlPreSuffix, sizeof urlPreSuffix,
			      urlSuffix, sizeof urlSuffix,
			      cseq, sizeof cseq)) {
#ifdef DEBUG
    fprintf(stderr, "parseRTSPRequestString() failed!\n");
#endif
    handleCmd_bad(cseq);
  } else {
#ifdef DEBUG
    fprintf(stderr, "parseRTSPRequestString() returned \n"
    					"\tcmdName \"%s\"\n"
    					"\thost:port \"%s\"\n"
    					"\turlPreSuffix \"%s\"\n"
    					"\turlSuffix \"%s\"\n", 
    					cmdName, hostPortStr, urlPreSuffix, urlSuffix);
#endif
    if (strcmp(cmdName, "OPTIONS") == 0) {
      handleCmd_OPTIONS(cseq);
    } else if (strcmp(cmdName, "DESCRIBE") == 0) {
      determineUserAgent(urlPreSuffix, (char const*)fRequestBuffer);  	// Steve, 2010/12/03.
      parseExternalAddress(hostPortStr);
	  
      handleCmd_DESCRIBE(cseq, urlSuffix, (char const*)fRequestBuffer);
    } else if (strcmp(cmdName, "SETUP") == 0) {
      handleCmd_SETUP(cseq, urlPreSuffix, urlSuffix, (char const*)fRequestBuffer);
    } else if (strcmp(cmdName, "TEARDOWN") == 0
	       || strcmp(cmdName, "PLAY") == 0
	       || strcmp(cmdName, "PAUSE") == 0
	       || strcmp(cmdName, "GET_PARAMETER") == 0
	       || strcmp(cmdName, "SET_PARAMETER") == 0) {
      handleCmd_withinSession(cmdName, urlPreSuffix, urlSuffix, cseq,
			      (char const*)fRequestBuffer);
    } else {
      handleCmd_notSupported(cseq);
    }
  }

#ifdef DEBUG
  fprintf(stderr, "sending response: %s", fResponseBuffer);
#endif

  // TODO: ??? What if things have been buffered...
  sendResponse();

  if (strcmp(cmdName, "SETUP") == 0 && fStreamAfterSETUP) {
    // The client has asked for streaming to commence now, rather than after a
    // subsequent "PLAY" command.  So, simulate the effect of a "PLAY" command:
    handleCmd_withinSession("PLAY", urlPreSuffix, urlSuffix, cseq,
			    (char const*)fRequestBuffer);
  }

  resetRequestBuffer(); // to prepare for any subsequent request
  if (!fSessionIsActive) {
	fprintf(stderr, "RTSPClientSession[%p] is not long active; terminating connection!\n", this);
  	this->fOurServer.deleteRTSPClientSession(this);	// delete this;
  }
}

// Handler routines for specific RTSP commands:

// Generate a "Date:" header for use in a RTSP response:
static char const* dateHeader() {
  static char buf[200];
#if !defined(_WIN32_WCE)
  time_t tt = time(NULL);
  strftime(buf, sizeof buf, "Date: %a, %b %d %Y %H:%M:%S GMT\r\n", gmtime(&tt));
#else
  // WinCE apparently doesn't have "time()", "strftime()", or "gmtime()",
  // so generate the "Date:" header a different, WinCE-specific way.
  // (Thanks to Pierre l'Hussiez for this code)
  SYSTEMTIME SystemTime;
  GetSystemTime(&SystemTime);
  WCHAR dateFormat[] = L"ddd, MMM dd yyyy";
  WCHAR timeFormat[] = L"HH:mm:ss GMT\r\n";
  WCHAR inBuf[200];
  DWORD locale = LOCALE_NEUTRAL;

  int ret = GetDateFormat(locale, 0, &SystemTime,
			  (LPTSTR)dateFormat, (LPTSTR)inBuf, sizeof inBuf);
  inBuf[ret - 1] = ' ';
  ret = GetTimeFormat(locale, 0, &SystemTime,
		      (LPTSTR)timeFormat,
		      (LPTSTR)inBuf + ret, (sizeof inBuf) - ret);
  wcstombs(buf, inBuf, wcslen(inBuf));
#endif
  return buf;
}

static char const* allowedCommandNames
//  = "OPTIONS, DESCRIBE, SETUP, TEARDOWN, PLAY, PAUSE, GET_PARAMETER, SET_PARAMETER";
= "OPTIONS, DESCRIBE, SETUP, TEARDOWN, PLAY, PAUSE, SET_PARAMETER"; // TEMP HACK to stop VLC from using "GET_PARAMETER" as a client 'keep-alive' indicator; we don't need this, and it currently causes problems for RTP-over-TCP streams.

void RTSPServer::RTSPClientSession::handleCmd_bad(char const* /*cseq*/) {
  // Don't do anything with "cseq", because it might be nonsense
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 400 Bad Request\r\n%sAllow: %s\r\n\r\n",
	   dateHeader(), allowedCommandNames);
}

void RTSPServer::RTSPClientSession::handleCmd_notSupported(char const* cseq) {
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 405 Method Not Allowed\r\nCSeq: %s\r\n%sAllow: %s\r\n\r\n",
	   cseq, dateHeader(), allowedCommandNames);
}

void RTSPServer::RTSPClientSession::handleCmd_notFound(char const* cseq) {
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 404 Stream Not Found\r\nCSeq: %s\r\n%s\r\n",
	   cseq, dateHeader());
  fSessionIsActive = False; // triggers deletion of ourself after responding
}

void RTSPServer::RTSPClientSession::handleCmd_unsupportedTransport(char const* cseq) {
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 461 Unsupported Transport\r\nCSeq: %s\r\n%s\r\n",
	   cseq, dateHeader());
  fSessionIsActive = False; // triggers deletion of ourself after responding
}

void RTSPServer::RTSPClientSession::handleCmd_OPTIONS(char const* cseq) {
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 200 OK\r\nCSeq: %s\r\n%sPublic: %s\r\n\r\n",
	   cseq, dateHeader(), allowedCommandNames);
}


void RTSPServer::RTSPClientSession
::handleCmd_DESCRIBE(char const* cseq, char const* urlSuffix,
		     char const* fullRequestStr) {
  char* sdpDescription = NULL;
  char* rtspURL = NULL;
  do {
      if (!authenticationOK("DESCRIBE", cseq, urlSuffix, fullRequestStr)) {
	  	if (++fFailedAuthenCount > 5)	// Disconnect after 6 failed attempt..
			fSessionIsActive = False;
		break;
	}
	fAuthenOK= True;
	
    // We should really check that the request contains an "Accept:" #####
    // for "application/sdp", because that's what we're sending back #####

    // Begin by looking up the "ServerMediaSession" object for the
    // specified "urlSuffix":
    ServerMediaSession* session = fOurServer.lookupServerMediaSession(urlSuffix);
    if (session == NULL  ||  !session->isEnabled()) {
      handleCmd_notFound(cseq);
      break;
    }

    sdpDescription = session->generateSDPDescription(fUserAgentID);

    // Then, assemble a SDP description for this session:
    if (sdpDescription == NULL) {
      // This usually means that a file name that was specified for a
      // "ServerMediaSubsession" does not exist.
      snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	       "RTSP/1.0 404 File Not Found, Or In Incorrect Format\r\n"
	       "CSeq: %s\r\n"
	       "%s\r\n",
	       cseq,
	       dateHeader());
     break;
    }
    unsigned sdpDescriptionSize = strlen(sdpDescription);
	
    // Also, generate our RTSP URL, for the "Content-Base:" header
    // (which is necessary to ensure that the correct URL gets used in
    // subsequent "SETUP" requests).
    rtspURL = getRTSPURL(session);
		
    if(fUserAgentID == USER_AGENT_QUICKTIME) {
    	snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	     "RTSP/1.0 200 OK\r\nCSeq: %s\r\n"
	     "%s"
	     "Content-Base: %s/\r\n"
	     "x-Accept-Dynamic-Rate: 0\r\n"
	     "Content-Type: application/sdp\r\n"
	     "Content-Length: %d\r\n\r\n"
	     "%s",
	     cseq,
	     dateHeader(),
	     rtspURL,
	     sdpDescriptionSize,
	     sdpDescription);		
    }
    else {
    	snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	     "RTSP/1.0 200 OK\r\nCSeq: %s\r\n"
	     "%s"
	     "Content-Base: %s/\r\n"
	     "Content-Type: application/sdp\r\n"
	     "Content-Length: %d\r\n\r\n"
	     "%s",
	     cseq,
	     dateHeader(),
	     rtspURL,
	     sdpDescriptionSize,
	     sdpDescription);		
    }
	

  } while (0);

  delete[] sdpDescription;
  delete[] rtspURL;
}


static void parseTransportHeader(char const* buf,
				 StreamingMode& streamingMode,
				 char*& streamingModeString,
				 char*& destinationAddressStr,
				 u_int8_t& destinationTTL,
				 portNumBits& clientRTPPortNum, // if UDP
				 portNumBits& clientRTCPPortNum, // if UDP
				 unsigned char& rtpChannelId, // if TCP
				 unsigned char& rtcpChannelId // if TCP
				 ) {
  // Initialize the result parameters to default values:
  streamingMode = RTP_UDP;
  streamingModeString = NULL;
  destinationAddressStr = NULL;
  destinationTTL = 255;
  clientRTPPortNum = 0;
  clientRTCPPortNum = 1;
  rtpChannelId = rtcpChannelId = 0xFF;

  portNumBits p1, p2;
  unsigned ttl, rtpCid, rtcpCid;

  // First, find "Transport:"
  while (1) {
    if (*buf == '\0') return; // not found
    if (_strncasecmp(buf, "Transport: ", 11) == 0) break;
    ++buf;
  }

  // Then, run through each of the fields, looking for ones we handle:
  char const* fields = buf + 11;
  char* field = strDupSize(fields);
  while (sscanf(fields, "%[^;]", field) == 1) {
    if (strcmp(field, "RTP/AVP/TCP") == 0) {
      streamingMode = RTP_TCP;
    } else if (strcmp(field, "RAW/RAW/UDP") == 0 ||
	       strcmp(field, "MP2T/H2221/UDP") == 0) {
      streamingMode = RAW_UDP;
      streamingModeString = strDup(field);
    } else if (_strncasecmp(field, "destination=", 12) == 0) {
      delete[] destinationAddressStr;
      destinationAddressStr = strDup(field+12);
    } else if (sscanf(field, "ttl%u", &ttl) == 1) {
      destinationTTL = (u_int8_t)ttl;
    } else if (sscanf(field, "client_port=%hu-%hu", &p1, &p2) == 2) {
	clientRTPPortNum = p1;
	clientRTCPPortNum = p2;
    } else if (sscanf(field, "client_port=%hu", &p1) == 1) {
	clientRTPPortNum = p1;
	clientRTCPPortNum = streamingMode == RAW_UDP ? 0 : p1 + 1;
    } else if (sscanf(field, "interleaved=%u-%u", &rtpCid, &rtcpCid) == 2) {
      rtpChannelId = (unsigned char)rtpCid;
      rtcpChannelId = (unsigned char)rtcpCid;
    }

    fields += strlen(field);
    while (*fields == ';') ++fields; // skip over separating ';' chars
    if (*fields == '\0' || *fields == '\r' || *fields == '\n') break;
  }
  delete[] field;
}

static Boolean parsePlayNowHeader(char const* buf) {
  // Find "x-playNow:" header, if present
  while (1) {
    if (*buf == '\0') return False; // not found
    if (_strncasecmp(buf, "x-playNow:", 10) == 0) break;
    ++buf;
  }

  return True;
}

void RTSPServer::RTSPClientSession
::handleCmd_SETUP(char const* cseq,
		  char const* urlPreSuffix, char const* urlSuffix,
		  char const* fullRequestStr) {
  // "urlPreSuffix" should be the session (stream) name, and
  // "urlSuffix" should be the subsession (track) name.
  char const* streamName = urlPreSuffix;
  char const* trackId = urlSuffix;

  // Check whether we have existing session state, and, if so, whether it's
  // for the session that's named in "streamName".  (Note that we don't
  // support more than one concurrent session on the same client connection.) #####
  if (fOurServerMediaSession != NULL
      && strcmp(streamName, fOurServerMediaSession->streamName()) != 0) {
    fOurServerMediaSession = NULL;
  }
  if (fOurServerMediaSession == NULL) {
    // Set up this session's state.

    // Look up the "ServerMediaSession" object for the specified stream:
    if (streamName[0] != '\0' ||
	fOurServer.lookupServerMediaSession("") != NULL) { // normal case
    } else { // weird case: there was no track id in the URL
      streamName = urlSuffix;
      trackId = NULL;
    }
    fOurServerMediaSession = fOurServer.lookupServerMediaSession(streamName);
    if (fOurServerMediaSession == NULL  || !fOurServerMediaSession->isEnabled()) {
      handleCmd_notFound(cseq);
      return;
    }

    fOurServerMediaSession->incrementReferenceCount();

    // Set up our array of states for this session's subsessions (tracks):
    reclaimStreamStates();
    ServerMediaSubsessionIterator iter(*fOurServerMediaSession);
    for (fNumStreamStates = 0; iter.next() != NULL; ++fNumStreamStates) {}
    fStreamStates = new struct streamState[fNumStreamStates];
    iter.reset();
    ServerMediaSubsession* subsession;
    for (unsigned i = 0; i < fNumStreamStates; ++i) {
      subsession = iter.next();
      fStreamStates[i].subsession = subsession;
      fStreamStates[i].streamToken = NULL; // for now; reset by SETUP later
	  fStreamStates[i].sendSocket=-1;  // TCP, UDP, RTP
      fStreamStates[i].rtcpSocket=-1;	// RTCP
    }
  }

  // Look up information for the specified subsession (track):
  ServerMediaSubsession* subsession = NULL;
  unsigned streamNum;
  if (trackId != NULL && trackId[0] != '\0') { // normal case
    for (streamNum = 0; streamNum < fNumStreamStates; ++streamNum) {
      subsession = fStreamStates[streamNum].subsession;
      if (subsession != NULL && strcmp(trackId, subsession->trackId()) == 0) break;
    }
    if (streamNum >= fNumStreamStates) {
      // The specified track id doesn't exist, so this request fails:
      handleCmd_notFound(cseq);
      return;
    }
  } else {
    // Weird case: there was no track id in the URL.
    // This works only if we have only one subsession:
    if (fNumStreamStates != 1) {
      handleCmd_bad(cseq);
      return;
    }
    streamNum = 0;
    subsession = fStreamStates[streamNum].subsession;
  }
  // ASSERT: subsession != NULL

  // Look for a "Transport:" header in the request string,
  // to extract client parameters:
  StreamingMode streamingMode;
  char* streamingModeString = NULL; // set when RAW_UDP streaming is specified
  char* clientsDestinationAddressStr;
  u_int8_t clientsDestinationTTL;
  portNumBits clientRTPPortNum, clientRTCPPortNum;
  unsigned char rtpChannelId, rtcpChannelId;
  parseTransportHeader(fullRequestStr, streamingMode, streamingModeString,
		       clientsDestinationAddressStr, clientsDestinationTTL,
		       clientRTPPortNum, clientRTCPPortNum,
		       rtpChannelId, rtcpChannelId);
   if (fStreamingMode == RTP_TCP_HTTP) {
	// force to RTP_TCP regardless of parsing.
	streamingMode = RTP_TCP;
   }

  if (streamingMode == RTP_TCP && rtpChannelId == 0xFF) {
    // TCP streaming was requested, but with no "interleaving=" fields.
    // (QuickTime Player sometimes does this.)  Set the RTP and RTCP channel ids to
    // proper values:
    rtpChannelId = fTCPStreamIdCount; rtcpChannelId = fTCPStreamIdCount+1;
  }
  fTCPStreamIdCount += 2;

  Port clientRTPPort(clientRTPPortNum);
  Port clientRTCPPort(clientRTCPPortNum);

  // Next, check whether a "Range:" header is present in the request.
  // This isn't legal, but some clients do this to combine "SETUP" and "PLAY":
  double rangeStart = 0.0, rangeEnd = 0.0;
  fStreamAfterSETUP = parseRangeHeader(fullRequestStr, rangeStart, rangeEnd) ||
                      parsePlayNowHeader(fullRequestStr);

  // Then, get server parameters from the 'subsession':
  int tcpSocketNum = streamingMode == RTP_TCP ? fClientSocket : -1;
  netAddressBits destinationAddress = 0;
  u_int8_t destinationTTL = 255;
#ifdef RTSP_ALLOW_CLIENT_DESTINATION_SETTING
  if (clientsDestinationAddressStr != NULL) {
    // Use the client-provided "destination" address.
    // Note: This potentially allows the server to be used in denial-of-service
    // attacks, so don't enable this code unless you're sure that clients are
    // trusted.
    destinationAddress = our_inet_addr(clientsDestinationAddressStr);
  }
  // Also use the client-provided TTL.
  destinationTTL = clientsDestinationTTL;
#endif
  delete[] clientsDestinationAddressStr;
  Port serverRTPPort(0);
  Port serverRTCPPort(0);
  subsession->getStreamParameters(fOurSessionId, fClientAddr.sin_addr.s_addr,
				  clientRTPPort, clientRTCPPort,
				  tcpSocketNum, rtpChannelId, rtcpChannelId,
				  destinationAddress, destinationTTL, fIsMulticast,
				  serverRTPPort, serverRTCPPort,
				  fStreamStates[streamNum].streamToken,
				  fStreamStates[streamNum].sendSocket,
				  fStreamStates[streamNum].rtcpSocket);

  subsession->setSocketMode(tcpSocketNum == -1 ? UDP : TCP);

  struct in_addr destinationAddr; destinationAddr.s_addr = destinationAddress;
  char* destAddrStr = strDup(our_inet_ntoa(destinationAddr));
  struct sockaddr_in sourceAddr; SOCKLEN_T namelen = sizeof sourceAddr;
  getsockname(fClientSocket, (struct sockaddr*)&sourceAddr, &namelen);
  char* sourceAddrStr = strDup(our_inet_ntoa(sourceAddr.sin_addr));
  if (fIsMulticast) {
    switch (streamingMode) {
    case RTP_UDP:
        if(fUserAgentID == USER_AGENT_QUICKTIME) {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
                 "RTSP/1.0 200 OK\r\n"
                 "CSeq: %s\r\n"
                 "%s"
                 "Transport: RTP/AVP;multicast;destination=%s;source=%s;port=%d-%d;ttl=%d\r\n"
                 "x-dynamic-rate: 0\r\n"
                 "Session: %d\r\n\r\n",
                 cseq,
                 dateHeader(),
                 destAddrStr, sourceAddrStr, ntohs(serverRTPPort.num()), ntohs(serverRTCPPort.num()), destinationTTL,
                 fOurSessionId);
        }
        else {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
                 "RTSP/1.0 200 OK\r\n"
                 "CSeq: %s\r\n"
                 "%s"
                 "Transport: RTP/AVP;multicast;destination=%s;source=%s;port=%d-%d;ttl=%d\r\n"
                 "Session: %d\r\n\r\n",
                 cseq,
                 dateHeader(),
                 destAddrStr, sourceAddrStr, ntohs(serverRTPPort.num()), ntohs(serverRTCPPort.num()), destinationTTL,
                 fOurSessionId);
        }
        break;
    case RTP_TCP:
        // multicast streams can't be sent via TCP
        handleCmd_unsupportedTransport(cseq);
        break;
    case RAW_UDP:
        if(fUserAgentID == USER_AGENT_QUICKTIME) {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
                 "RTSP/1.0 200 OK\r\n"
                 "CSeq: %s\r\n"
                 "%s"
                 "Transport: %s;multicast;destination=%s;source=%s;port=%d;ttl=%d\r\n"
                 "x-dynamic-rate: 0\r\n"
                 "Session: %d\r\n\r\n",
                 cseq,
                 dateHeader(),
                 streamingModeString, destAddrStr, sourceAddrStr, ntohs(serverRTPPort.num()), destinationTTL,
                 fOurSessionId);
        }
        else {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
                 "RTSP/1.0 200 OK\r\n"
                 "CSeq: %s\r\n"
                 "%s"
                 "Transport: %s;multicast;destination=%s;source=%s;port=%d;ttl=%d\r\n"
                 "Session: %d\r\n\r\n",
                 cseq,
                 dateHeader(),
                 streamingModeString, destAddrStr, sourceAddrStr, ntohs(serverRTPPort.num()), destinationTTL,
                 fOurSessionId);			
        }
        break;
    }
  } else {
    switch (streamingMode) {
    case RTP_UDP: {
        if(fUserAgentID == USER_AGENT_QUICKTIME) {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	       "RTSP/1.0 200 OK\r\n"
	       "CSeq: %s\r\n"
	       "%s"
	       "Transport: RTP/AVP;unicast;destination=%s;source=%s;client_port=%d-%d;server_port=%d-%d\r\n"
		"x-dynamic-rate: 0\r\n"
	       "Session: %d\r\n\r\n",
	       cseq,
	       dateHeader(),
	       destAddrStr, sourceAddrStr, ntohs(clientRTPPort.num()), ntohs(clientRTCPPort.num()), ntohs(serverRTPPort.num()), ntohs(serverRTCPPort.num()),
	       fOurSessionId);
        }
	else {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	       "RTSP/1.0 200 OK\r\n"
	       "CSeq: %s\r\n"
	       "%s"
	       "Transport: RTP/AVP;unicast;destination=%s;source=%s;client_port=%d-%d;server_port=%d-%d\r\n"
		"Session: %d\r\n\r\n",
	       cseq,
	       dateHeader(),
	       destAddrStr, sourceAddrStr, ntohs(clientRTPPort.num()), ntohs(clientRTCPPort.num()), ntohs(serverRTPPort.num()), ntohs(serverRTCPPort.num()),
	       fOurSessionId);
	}
    break;
    }
    case RTP_TCP: {
        if(fUserAgentID == USER_AGENT_QUICKTIME) {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	       "RTSP/1.0 200 OK\r\n"
	       "CSeq: %s\r\n"
	       "%s"
	       "Transport: RTP/AVP/TCP;unicast;destination=%s;source=%s;interleaved=%d-%d\r\n"
	       "x-dynamic-rate: 0\r\n"
	       "Session: %d\r\n\r\n",
	       cseq,
	       dateHeader(),
	       destAddrStr, sourceAddrStr, rtpChannelId, rtcpChannelId,
	       fOurSessionId);
	}
	else {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	       "RTSP/1.0 200 OK\r\n"
	       "CSeq: %s\r\n"
	       "%s"
	       "Transport: RTP/AVP/TCP;unicast;destination=%s;source=%s;interleaved=%d-%d\r\n"
	       "Session: %d\r\n\r\n",
	       cseq,
	       dateHeader(),
	       destAddrStr, sourceAddrStr, rtpChannelId, rtcpChannelId,
	       fOurSessionId);		
	}
    break;
    }
    case RAW_UDP: {
        if(fUserAgentID == USER_AGENT_QUICKTIME) {		
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	       "RTSP/1.0 200 OK\r\n"
	       "CSeq: %s\r\n"
	       "%s"
	       "Transport: %s;unicast;destination=%s;source=%s;client_port=%d;server_port=%d\r\n"
	       "x-dynamic-rate: 0\r\n"
	       "Session: %d\r\n\r\n",
	       cseq,
	       dateHeader(),
	       streamingModeString, destAddrStr, sourceAddrStr, ntohs(clientRTPPort.num()), ntohs(serverRTPPort.num()),
	       fOurSessionId);
        }
        else {
            snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	       "RTSP/1.0 200 OK\r\n"
	       "CSeq: %s\r\n"
	       "%s"
	       "Transport: %s;unicast;destination=%s;source=%s;client_port=%d;server_port=%d\r\n"
	       "Session: %d\r\n\r\n",
	       cseq,
	       dateHeader(),
	       streamingModeString, destAddrStr, sourceAddrStr, ntohs(clientRTPPort.num()), ntohs(serverRTPPort.num()),
	       fOurSessionId);	
        }
    break;
    }
    }
  }
  delete[] destAddrStr; delete[] sourceAddrStr; delete[] streamingModeString;
}

void RTSPServer::RTSPClientSession
::handleCmd_withinSession(char const* cmdName,
			  char const* urlPreSuffix, char const* urlSuffix,
			  char const* cseq, char const* fullRequestStr) {
  // This will either be:
  // - a non-aggregated operation, if "urlPreSuffix" is the session (stream)
  //   name and "urlSuffix" is the subsession (track) name, or
  // - a aggregated operation, if "urlSuffix" is the session (stream) name,
  //   or "urlPreSuffix" is the session (stream) name, and "urlSuffix"
  //   is empty.
  // First, figure out which of these it is:
  if (fOurServerMediaSession == NULL) { // There wasn't a previous SETUP!
    handleCmd_notSupported(cseq);
    return;
  }
  ServerMediaSubsession* subsession;
  if (urlSuffix[0] != '\0' &&
      strcmp(fOurServerMediaSession->streamName(), urlPreSuffix) == 0) {
    // Non-aggregated operation.
    // Look up the media subsession whose track id is "urlSuffix":
    ServerMediaSubsessionIterator iter(*fOurServerMediaSession);
    while ((subsession = iter.next()) != NULL) {
      if (strcmp(subsession->trackId(), urlSuffix) == 0) break; // success
    }
    if (subsession == NULL) { // no such track!
      handleCmd_notFound(cseq);
      return;
    }
  } else if (strcmp(fOurServerMediaSession->streamName(), urlSuffix) == 0 ||
	     strcmp(fOurServerMediaSession->streamName(), urlPreSuffix) == 0) {
    // Aggregated operation
    subsession = NULL;
  } else { // the request doesn't match a known stream and/or track at all!
    handleCmd_notFound(cseq);
    return;
  }

  if (strcmp(cmdName, "TEARDOWN") == 0) {
    handleCmd_TEARDOWN(subsession, cseq);
  } else if (strcmp(cmdName, "PLAY") == 0) {
    handleCmd_PLAY(subsession, cseq, fullRequestStr);
  } else if (strcmp(cmdName, "PAUSE") == 0) {
    handleCmd_PAUSE(subsession, cseq);
  } else if (strcmp(cmdName, "GET_PARAMETER") == 0) {
    handleCmd_GET_PARAMETER(subsession, cseq, fullRequestStr);
  } else if (strcmp(cmdName, "SET_PARAMETER") == 0) {
    handleCmd_SET_PARAMETER(subsession, cseq, fullRequestStr);
  }
}

void RTSPServer::RTSPClientSession
::handleCmd_TEARDOWN(ServerMediaSubsession* /*subsession*/, char const* cseq) {
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 200 OK\r\nCSeq: %s\r\n%s\r\n",
	   cseq, dateHeader());
  fSessionIsActive = False; // triggers deletion of ourself after responding
}

static Boolean parseScaleHeader(char const* buf, float& scale) {
  // Initialize the result parameter to a default value:
  scale = 1.0;

  // First, find "Scale:"
  while (1) {
    if (*buf == '\0') return False; // not found
    if (_strncasecmp(buf, "Scale: ", 7) == 0) break;
    ++buf;
  }

  // Then, run through each of the fields, looking for ones we handle:
  char const* fields = buf + 7;
  while (*fields == ' ') ++fields;
  float sc;
  if (sscanf(fields, "%f", &sc) == 1) {
    scale = sc;
  } else {
    return False; // The header is malformed
  }

  return True;
}

void RTSPServer::RTSPClientSession
  ::handleCmd_PLAY(ServerMediaSubsession* subsession, char const* cseq,
		   char const* fullRequestStr) {
  char* rtspURL = getRTSPURL(fOurServerMediaSession);
  unsigned rtspURLSize = strlen(rtspURL);

  //// Parse the client's "Scale:" header, if any:
  float scale;
  Boolean sawScaleHeader = parseScaleHeader(fullRequestStr, scale);

  // Try to set the stream's scale factor to this value:
  if (subsession == NULL /*aggregate op*/) {
    fOurServerMediaSession->testScaleFactor(scale);
  } else {
    subsession->testScaleFactor(scale);
  }

  char buf[100];
  char* scaleHeader;
  if (!sawScaleHeader) {
    buf[0] = '\0'; // Because we didn't see a Scale: header, don't send one back
  } else {
    sprintf(buf, "Scale: %f\r\n", scale);
  }
  scaleHeader = strDup(buf);

  //// Parse the client's "Range:" header, if any:
  double rangeStart = 0.0, rangeEnd = 0.0;
  Boolean sawRangeHeader = parseRangeHeader(fullRequestStr, rangeStart, rangeEnd);

  // Use this information, plus the stream's duration (if known), to create
  // our own "Range:" header, for the response:
  float duration = subsession == NULL /*aggregate op*/
    ? fOurServerMediaSession->duration() : subsession->duration();
  if (duration < 0.0) {
    // We're an aggregate PLAY, but the subsessions have different durations.
    // Use the largest of these durations in our header
    duration = -duration;
  }

  if (rangeEnd < 0.0 || rangeEnd > duration) rangeEnd = duration;
  if (rangeStart < 0.0) {
    rangeStart = 0.0;
  } else if (rangeEnd > 0.0 && scale > 0.0 && rangeStart > rangeEnd) {
    rangeStart = rangeEnd;
  }

  char* rangeHeader;
  if (!sawRangeHeader) {
    buf[0] = '\0'; // Because we didn't see a Range: header, don't send one back
  } else if (rangeEnd == 0.0 && scale >= 0.0) {
    sprintf(buf, "Range: npt=%.3f-\r\n", rangeStart);
  } else {
    sprintf(buf, "Range: npt=%.3f-%.3f\r\n", rangeStart, rangeEnd);
  }
  rangeHeader = strDup(buf);

  // Create a "RTP-Info:" line.  It will get filled in from each subsession's state:
  char const* rtpInfoFmt =
    "%s" // "RTP-Info:", plus any preceding rtpInfo items
    "%s" // comma separator, if needed
    "url=%s/%s"
    ";seq=%d"
    ";rtptime=%u"
    ;
  unsigned rtpInfoFmtSize = strlen(rtpInfoFmt);
  char* rtpInfo = strDup("RTP-Info: ");
  unsigned i, numRTPInfoItems = 0;

  // Do any required seeking/scaling on each subsession, before starting streaming:
  for (i = 0; i < fNumStreamStates; ++i) {
    if (subsession == NULL /* means: aggregated operation */
	|| subsession == fStreamStates[i].subsession) {
      if (sawScaleHeader) {
	fStreamStates[i].subsession->setStreamScale(fOurSessionId,
						    fStreamStates[i].streamToken,
						    scale);
      }
      if (sawRangeHeader) {
	fStreamStates[i].subsession->seekStream(fOurSessionId,
						fStreamStates[i].streamToken,
						rangeStart);
      }
    }
  }

   setDSCP(fOurServer.fVideoDSCP, fOurServer.fAudioDSCP);

  // Now, start streaming:
  for (i = 0; i < fNumStreamStates; ++i) {
    if (subsession == NULL /* means: aggregated operation */
	|| subsession == fStreamStates[i].subsession) {
      unsigned short rtpSeqNum = 0;
      unsigned rtpTimestamp = 0;
      fStreamStates[i].subsession->startStream(fOurSessionId,
					       fStreamStates[i].streamToken,
					       (TaskFunc*)noteClientLiveness, this,
					       rtpSeqNum, rtpTimestamp,
					       handleAlternativeRequestByte, this);
	// needed for filtering on multicast meta session
	// A big hack now.. 	Skip the metadata subsession printout..
	char const* sdpLines = fStreamStates[i].subsession->sdpLines();
	if (sdpLines && 
		(   ( strstr(sdpLines, "gt.metadata") != NULL && fUserAgentID != USER_AGENT_BRICKCOM ) ||
		    (strstr(sdpLines, "vnd.onvif.metadata") != NULL && fUserAgentID != USER_AGENT_ONVIF) 
		) 
	   )  {	continue;  }

      const char *urlSuffix = fStreamStates[i].subsession->trackId();
      char* prevRTPInfo = rtpInfo;
      unsigned rtpInfoSize = rtpInfoFmtSize
	+ strlen(prevRTPInfo)
	+ 1
	+ rtspURLSize + strlen(urlSuffix)
	+ 5 /*max unsigned short len*/
	+ 10 /*max unsigned (32-bit) len*/
	+ 2 /*allows for trailing \r\n at final end of string*/;
      rtpInfo = new char[rtpInfoSize];
      sprintf(rtpInfo, rtpInfoFmt,
	      prevRTPInfo,
	      numRTPInfoItems++ == 0 ? "" : ",",
	      rtspURL, urlSuffix,
	      rtpSeqNum,
	      rtpTimestamp
	      );
      delete[] prevRTPInfo;
    }
  }

   fOurServer.logSessionStart(fOurSessionId, fOurServerMediaSession->channelName(), 
   									fUsername, inet_ntoa(fClientAddr.sin_addr));

  if (numRTPInfoItems == 0) {
    rtpInfo[0] = '\0';
  } else {
    unsigned rtpInfoLen = strlen(rtpInfo);
    rtpInfo[rtpInfoLen] = '\r';
    rtpInfo[rtpInfoLen+1] = '\n';
    rtpInfo[rtpInfoLen+2] = '\0';
  }
  
  // Fill in the response:
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 200 OK\r\n"
	   "CSeq: %s\r\n"
	   "%s"
	   "%s"
	   "%s"
	   "Session: %d\r\n"
	   "%s\r\n",
	   cseq,
	   dateHeader(),
	   scaleHeader,
	   rangeHeader,
	   fOurSessionId,
	   rtpInfo);
  delete[] rtpInfo; delete[] rangeHeader;
  delete[] scaleHeader; delete[] rtspURL;
}

void RTSPServer::RTSPClientSession
  ::handleCmd_PAUSE(ServerMediaSubsession* subsession, char const* cseq) {
  for (unsigned i = 0; i < fNumStreamStates; ++i) {
    if (subsession == NULL /* means: aggregated operation */
	|| subsession == fStreamStates[i].subsession) {
      fStreamStates[i].subsession->pauseStream(fOurSessionId,
					       fStreamStates[i].streamToken);
    }
  }
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 200 OK\r\nCSeq: %s\r\n%sSession: %d\r\n\r\n",
	   cseq, dateHeader(), fOurSessionId);
}

void RTSPServer::RTSPClientSession
::handleCmd_GET_PARAMETER(ServerMediaSubsession* subsession, char const* cseq,
			  char const* /*fullRequestStr*/) {
  // We implement "GET_PARAMETER" just as a 'keep alive',
  // and send back an empty response:
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 200 OK\r\nCSeq: %s\r\n%sSession: %d\r\n\r\n",
	   cseq, dateHeader(), fOurSessionId);
}

void RTSPServer::RTSPClientSession
::handleCmd_SET_PARAMETER(ServerMediaSubsession* /*subsession*/, char const* cseq,
			  char const* /*fullRequestStr*/) {
  // By default, we don't implement "SET_PARAMETER":

  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 200 OK\r\nCSeq: %s\r\n%sSession: %d\r\n\r\n",
	   cseq, dateHeader(), fOurSessionId);
  // handleCmd_notSupported(cseq);
}

static Boolean parseAuthorizationHeader(char const* buf,
					char const*& username,
					char const*& realm,
					char const*& nonce, char const*& uri,
					char const*& response) {
  // Initialize the result parameters to default values:
  username = realm = nonce = uri = response = NULL;

  // First, find "Authorization:"
  while (1) {
    if (*buf == '\0') return False; // not found
    if (_strncasecmp(buf, "Authorization: Digest ", 22) == 0) break;
    ++buf;
  }

  // Then, run through each of the fields, looking for ones we handle:
  char const* fields = buf + 22;
  while (*fields == ' ') ++fields;
  char* parameter = strDupSize(fields);
  char* value = strDupSize(fields);
  while (1) {
    value[0] = '\0';
    if (sscanf(fields, "%[^=]=\"%[^\"]\"", parameter, value) != 2 &&
	sscanf(fields, "%[^=]=\"\"", parameter) != 1) {
      break;
    }
    if (strcmp(parameter, "username") == 0) {
      username = strDup(value);
    } else if (strcmp(parameter, "realm") == 0) {
      realm = strDup(value);
    } else if (strcmp(parameter, "nonce") == 0) {
      nonce = strDup(value);
    } else if (strcmp(parameter, "uri") == 0) {
      uri = strDup(value);
    } else if (strcmp(parameter, "response") == 0) {
      response = strDup(value);
    }

    fields += strlen(parameter) + 2 /*="*/ + strlen(value) + 1 /*"*/;
    while (*fields == ',' || *fields == ' ') ++fields;
        // skip over any separating ',' and ' ' chars
    if (*fields == '\0' || *fields == '\r' || *fields == '\n') break;
  }
  delete[] parameter; delete[] value;
  return True;
}

static Boolean parseBasicAuthorizationHeader(char const* buf,
					char const*& username,
					char const*& password) {
  // Initialize the result parameters to default values:
  username = password = NULL;

  // First, find "Authorization:"
  while (1) {
    if (*buf == '\0') return False; // not found
    if (_strncasecmp(buf, "Authorization: Basic ", 21) == 0) break;
    ++buf;
  }

  // Then, run through each of the fields, looking for ones we handle:
  char const* fields = buf + 21;
  while (*fields == ' ') ++fields;
  char* parameter = strDupSize(fields);

  if (sscanf(fields, "%s\r\n", parameter) != 1) {
    delete[] parameter;
    return False;
  }

  unsigned size;
  unsigned char* username_password = base64Decode(parameter, size);
  username_password[size] = '\0';

  if (size == 1) {
    delete[] parameter,username_password;
    return False;
  }

  char* tmp = strchr((char*)username_password, 0x3a);
  *tmp = '\0';
  tmp++;

  if (strlen((const char *)username_password)) username = strDup((const char*) username_password);
  if (strlen((const char *)tmp)) password = strDup((const char*) tmp);

  delete[] parameter,username_password;
  return True;
}

Boolean RTSPServer::RTSPClientSession
::authenticationOK(char const* cmdName, char const* cseq,
		   char const* urlSuffix, char const* fullRequestStr) {

  if (!fOurServer.specialClientAccessCheck(fClientSocket, fClientAddr, urlSuffix)) {
    snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
             "RTSP/1.0 401 Unauthorized\r\n"
             "CSeq: %s\r\n"
             "%s"
             "\r\n",
             cseq, dateHeader());
    return False;
  }
  
   // User-provided custom authentication scheme. If failed, fall back to old mechanism.
  if ( !needAuthentication() || 
  	fOurServer.customAuthenticationOK(fClientSocket, fClientAddr, urlSuffix, fullRequestStr)) {
	return True;
  }

  // If we weren't set up with an authentication database, we're OK:
  if (fOurServer.fAuthDB == NULL) {
  	return True;
  }

  if (fOurServer.fAuthDB->getAuthType() == RTSP_AUTH_DIGEST) {
	
  char const* username = NULL; char const* realm = NULL; char const* nonce = NULL;
  char const* uri = NULL; char const* response = NULL;
  Boolean success = False;

  do {
    // To authenticate, we first need to have a nonce set up
    // from a previous attempt:
    if (fCurrentAuthenticator.nonce() == NULL) break;

    // Next, the request needs to contain an "Authorization:" header,
    // containing a username, (our) realm, (our) nonce, uri,
    // and response string:
    if (!parseAuthorizationHeader(fullRequestStr,
				  username, realm, nonce, uri, response)
	|| username == NULL
	|| realm == NULL || strcmp(realm, fCurrentAuthenticator.realm()) != 0
	|| nonce == NULL || strcmp(nonce, fCurrentAuthenticator.nonce()) != 0
	|| uri == NULL || response == NULL) {
      break;
    }

	if (!strcmp(username, ""))
    		strcpy(fUsername, "anonymous");
	else
    		strcpy(fUsername, username);	// TODO: watch for overrun..
		
    // Next, the username has to be known to us:
    char const* password = fOurServer.fAuthDB->lookupPassword(username);
#ifdef DEBUG
    fprintf(stderr, "lookupPassword(%s) returned password %s\n", username, password);
#endif
    if (password == NULL) break;
    fCurrentAuthenticator.
      setUsernameAndPassword(username, password,
			     fOurServer.fAuthDB->passwordsAreMD5());

    // Finally, compute a digest response from the information that we have,
    // and compare it to the one that we were given:
    char const* ourResponse
      = fCurrentAuthenticator.computeDigestResponse(cmdName, uri);
    success = (strcmp(ourResponse, response) == 0);
    fCurrentAuthenticator.reclaimDigestResponse(ourResponse);
  } while (0);

  delete[] (char*)username; delete[] (char*)realm; delete[] (char*)nonce;
  delete[] (char*)uri; delete[] (char*)response;
  if (success) return True;

  // If we get here, there was some kind of authentication failure.
  // Send back a "401 Unauthorized" response, with a new random nonce:
  fCurrentAuthenticator.setRealmAndRandomNonce(fOurServer.fAuthDB->realm());
  snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
	   "RTSP/1.0 401 Unauthorized\r\n"
	   "CSeq: %s\r\n"
	   "%s"
	   "WWW-Authenticate: Digest realm=\"%s\", nonce=\"%s\"\r\n\r\n",
	   cseq,
	   dateHeader(),
	   fCurrentAuthenticator.realm(), fCurrentAuthenticator.nonce());

	   
  } else if (fOurServer.fAuthDB->getAuthType() == RTSP_AUTH_BASIC) {
    Boolean success = False;
    char const* username = NULL;char const* password = NULL;

    do {
      if (!parseBasicAuthorizationHeader(fullRequestStr, username, password)) {
            break;
          }

          if (username == NULL || password == NULL) break;

          if (!strcmp(username, ""))
                  strcpy(fUsername, "anonymous");
          else
                  strcpy(fUsername, username);    // TODO: watch for overrun..
              
          // Next, the username has to be known to us:
          char const* our_password = fOurServer.fAuthDB->lookupPassword(username);
          if (our_password == NULL) break;
          fCurrentAuthenticator.
            setUsernameAndPassword(username, password,
                       fOurServer.fAuthDB->passwordsAreMD5());
      
          success = (strcmp(our_password, password) == 0);
        } while (0);

        if (username != NULL) delete[] (char*)username;
        if (password != NULL) delete[] (char*)password;
        if (success) return True;

      fCurrentAuthenticator.setRealmAndNonce(fOurServer.fAuthDB->realm(), NULL);
      snprintf((char*)fResponseBuffer, sizeof fResponseBuffer,
           "RTSP/1.0 401 Unauthorized\r\n"
           "CSeq: %s\r\n"
           "%s"
           "WWW-Authenticate: Basic realm=\"%s\"\r\n\r\n",
           cseq,
           dateHeader(),
           fCurrentAuthenticator.realm());
  }
  return False;
}

void RTSPServer::RTSPClientSession::noteLiveness() {
  if (fOurServer.fReclamationTestSeconds > 0) {
    envir().taskScheduler()
      .rescheduleDelayedTask(fLivenessCheckTask,
			     fOurServer.fReclamationTestSeconds*1000000,
			     (TaskFunc*)livenessTimeoutTask, this);
  }
}

void RTSPServer::RTSPClientSession
::noteClientLiveness(RTSPClientSession* clientSession) {
  clientSession->noteLiveness();
}

void RTSPServer::RTSPClientSession::setDSCP(int videoDSCP, int audioDSCP)
{
	int i;
	ServerMediaSubsession *subsession;

	for (i = 0; i < fNumStreamStates; i++) {
		subsession=fStreamStates[i].subsession;

		if (subsession) {
			switch(subsession->getMediaType()) {
			case MediaTypeVideo:
				setSocketDSCP(fStreamStates[i].sendSocket, PF_INET, videoDSCP);
				setSocketDSCP(fStreamStates[i].rtcpSocket, PF_INET, videoDSCP);
				break;
			case MediaTypeAudio:
				if (subsession->getSocketMode() == UDP) {
					setSocketDSCP(fStreamStates[i].sendSocket, PF_INET, audioDSCP);
					setSocketDSCP(fStreamStates[i].rtcpSocket, PF_INET, audioDSCP);
				}
				break;
			}
		}
	}
}


void RTSPServer::RTSPClientSession
::livenessTimeoutTask(RTSPClientSession* clientSession) {
  // If this gets called, the client session is assumed to have timed out,
  // so delete it:

  // However, we don't timeout multicast sessions, because to do so would require
  // closing all client sessions that have requested the stream - not just this one.
  // Also, the multicast stream itself would usually not be halted, in any case.
  if (clientSession->isMulticast()) return;

#ifdef DEBUG
  fprintf(stderr, "RTSP client session from %s has timed out (due to inactivity)\n", our_inet_ntoa(clientSession->fClientAddr.sin_addr));
#endif

      fprintf(stderr, "RTSP client session from %s has timed out (due to inactivity)\n", our_inet_ntoa(clientSession->fClientAddr.sin_addr));
	clientSession->fOurServer.deleteRTSPClientSession(clientSession);	// delete clientSession;
}


////////// UserAuthenticationDatabase implementation //////////

UserAuthenticationDatabase::UserAuthenticationDatabase(char const* realm,
						       Boolean passwordsAreMD5)
  : fTable(HashTable::create(STRING_HASH_KEYS)),
    fRealm(strDup(realm == NULL ? "LIVE555 Streaming Media" : realm)),
    fPasswordsAreMD5(passwordsAreMD5) {
}

UserAuthenticationDatabase::~UserAuthenticationDatabase() {
  delete[] fRealm;
  delete fTable;
}

void UserAuthenticationDatabase::addUserRecord(char const* username,
					       char const* password) {
  fTable->Add(username, (void*)(strDup(password)));
}

void UserAuthenticationDatabase::removeUserRecord(char const* username) {
  char* password = (char*)(fTable->Lookup(username));
  fTable->Remove(username);
  delete[] password;
}

char const* UserAuthenticationDatabase::lookupPassword(char const* username) {
  return (char const*)(fTable->Lookup(username));
}


void deleteRTSPClientConnectionFromDB(int sockfd)
{
	if (sockfd < 0)
		return;

	// printf("deleteRTSPClientConnectionFromDB(): %d\n", sockfd);
	
	if (theOnlyRTSPServer) {
		theOnlyRTSPServer->deleteRTSPClientSession(sockfd);
	}
}

