/**********
This library is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the
Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version. (See <http://www.gnu.org/copyleft/lesser.html>.)

This library is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
more details.

You should have received a copy of the GNU Lesser General Public License
along with this library; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
**********/
// "liveMedia"
// Copyright (c) 1996-2008 Live Networks, Inc.  All rights reserved.
// An abstraction of a network interface used for RTP (or RTCP).
// (This allows the RTP-over-TCP hack (RFC 2326, section 10.12) to
// be implemented transparently.)
// Implementation

#include "RTPInterface.hh"
#include <GroupsockHelper.hh>
#include <stdio.h>



// TODO: static initialization seems OK.. but is alive as process is alive...
static HashTable* writeSockTable = HashTable::create(ONE_WORD_HASH_KEYS);


////////// Helper Functions - Definition //////////

// Helper routines and data structures, used to implement

// Reading RTP-over-TCP is implemented using two levels of hash tables.
// The top-level hash table maps TCP socket numbers to a
// "SocketDescriptor" that contains a hash table for each of the
// sub-channels that are reading from this socket.

static HashTable* socketHashTable(UsageEnvironment& env) {
  _Tables* ourTables = _Tables::getOurTables(env);
  if (ourTables->socketTable == NULL) {
    // Create a new socket number -> SocketDescriptor mapping table:
    ourTables->socketTable = HashTable::create(ONE_WORD_HASH_KEYS);
  }
  return (HashTable*)(ourTables->socketTable);
}

class SocketDescriptor {
public:
  SocketDescriptor(UsageEnvironment& env, int socketNum);
  virtual ~SocketDescriptor();

  void registerRTPInterface(unsigned char streamChannelId,
			    RTPInterface* rtpInterface);
  RTPInterface* lookupRTPInterface(unsigned char streamChannelId);
  void deregisterRTPInterface(unsigned char streamChannelId);

  void setServerRequestAlternativeByteHandler(ServerRequestAlternativeByteHandler* handler, void* clientData) {
    fServerRequestAlternativeByteHandler = handler;
    fServerRequestAlternativeByteHandlerClientData = clientData;
  }

private:
  static void tcpReadHandler(SocketDescriptor*, int mask);
  void tcpReadHandler1(int mask, UsageEnvironment& env, int socketNum);

private:
  UsageEnvironment& fEnv;
  int fOurSocketNum;
  HashTable* fSubChannelHashTable;
  ServerRequestAlternativeByteHandler* fServerRequestAlternativeByteHandler;
  void* fServerRequestAlternativeByteHandlerClientData;
  u_int8_t fStreamChannelId, fSizeByte1;
  enum { AWAITING_DOLLAR, AWAITING_STREAM_CHANNEL_ID, AWAITING_SIZE1, AWAITING_SIZE2 } fTCPReadingState;
};

static SocketDescriptor* lookupSocketDescriptor(UsageEnvironment& env, int sockNum, Boolean createIfNotFound = True) {
  HashTable* table = socketHashTable(env);	// TODO: createIfNotFound);
  if (table == NULL) return NULL;

  char const* key = (char const*)(long)sockNum;
  SocketDescriptor* socketDescriptor = (SocketDescriptor*)(table->Lookup(key));
  if (socketDescriptor == NULL && createIfNotFound) {
    socketDescriptor = new SocketDescriptor(env, sockNum);
    table->Add((char const*)(long)(sockNum), socketDescriptor);
  }

  return socketDescriptor;
}

static void removeSocketDescription(UsageEnvironment& env, int sockNum) {
  char const* key = (char const*)(long)sockNum;
  HashTable* table = socketHashTable(env);
  table->Remove(key);

  if (table->IsEmpty()) {
    // We can also delete the table (to reclaim space):
    _Tables* ourTables = _Tables::getOurTables(env);
    delete table;
    ourTables->socketTable = NULL;
    ourTables->reclaimIfPossible();
  }
}

////////// RTPInterface - Implementation //////////

RTPInterface::RTPInterface(Medium* owner, Groupsock* gs)
  : fOwner(owner), fGS(gs),
    fTCPStreams(NULL),
    fNextTCPReadSize(0), fNextTCPReadStreamSocketNum(-1),
    fNextTCPReadStreamChannelId(0xFF), fReadHandlerProc(NULL),
    fAuxReadHandlerFunc(NULL), fAuxReadHandlerClientData(NULL) {
  // Make the socket non-blocking, even though it will be read from only asynchronously, when packets arrive.
  // The reason for this is that, in some OSs, reads on a blocking socket can (allegedly) sometimes block,
  // even if the socket was previously reported (e.g., by "select()") as having data available.
  // (This can supposedly happen if the UDP checksum fails, for example.)
  makeSocketNonBlocking(fGS->socketNum());
  increaseSendBufferTo(envir(), fGS->socketNum(), 50*1024);
}

RTPInterface::~RTPInterface() {
  delete fTCPStreams;
}

void RTPInterface::setStreamSocket(int sockNum,
				   unsigned char streamChannelId) {
  fGS->removeAllDestinations();
  addStreamSocket(sockNum, streamChannelId);
}

void RTPInterface::addStreamSocket(int sockNum,
				   unsigned char streamChannelId) {
  if (sockNum < 0) return;

  for (tcpStreamRecord* streams = fTCPStreams; streams != NULL;
       streams = streams->fNext) {
    if (streams->fStreamSocketNum == sockNum
	&& streams->fStreamChannelId == streamChannelId) {
      return; // we already have it
    }
  }
  fTCPStreams = new tcpStreamRecord(envir(), sockNum, streamChannelId, fTCPStreams);
}

static void deregisterSocket(UsageEnvironment& env, int sockNum, unsigned char streamChannelId) {
  SocketDescriptor* socketDescriptor = lookupSocketDescriptor(env, sockNum, False);
  if (socketDescriptor != NULL) {
    socketDescriptor->deregisterRTPInterface(streamChannelId);
        // Note: This may delete "socketDescriptor",
        // if no more interfaces are using this socket
  }
}

void RTPInterface::removeStreamSocket(int sockNum,
				      unsigned char streamChannelId) {
  for (tcpStreamRecord** streamsPtr = &fTCPStreams; *streamsPtr != NULL;
       streamsPtr = &((*streamsPtr)->fNext)) {
    if ((*streamsPtr)->fStreamSocketNum == sockNum
	&& (*streamsPtr)->fStreamChannelId == streamChannelId) {
      deregisterSocket(envir(), sockNum, streamChannelId);

      // Then remove the record pointed to by *streamsPtr :
      tcpStreamRecord* next = (*streamsPtr)->fNext;
      (*streamsPtr)->fNext = NULL;
      delete (*streamsPtr);
      *streamsPtr = next;
      return;
    }
  }
}

void RTPInterface
::setServerRequestAlternativeByteHandler(int socketNum, ServerRequestAlternativeByteHandler* handler, void* clientData) {

  SocketDescriptor* socketDescriptor = lookupSocketDescriptor(envir(), socketNum);

  if (socketDescriptor != NULL) socketDescriptor->setServerRequestAlternativeByteHandler(handler, clientData);
}

void RTPInterface::sendPacket(unsigned char* packet, unsigned packetSize, 
		Boolean lastFragmentEndFrame, Boolean firstFragmentInKeyFrame) {
	// Normal case: Send as a UDP packet:
	fGS->output(envir(), fGS->ttl(), packet, packetSize);

	// Also, send over each of our TCP sockets:	
	for (tcpStreamRecord* streams = fTCPStreams; streams != NULL; streams = streams->fNext) {
		streams->fOutSocketDesc->sendPacket(streams,
				packet, packetSize, lastFragmentEndFrame,  firstFragmentInKeyFrame);
	}
}


void RTPInterface
::startNetworkReading(TaskScheduler::BackgroundHandlerProc* handlerProc) {
  // Normal case: Arrange to read UDP packets:
  envir().taskScheduler().
    turnOnBackgroundReadHandling(fGS->socketNum(), handlerProc, fOwner);

  // Also, receive RTP over TCP, on each of our TCP connections:
  fReadHandlerProc = handlerProc;
  for (tcpStreamRecord* streams = fTCPStreams; streams != NULL;
       streams = streams->fNext) {
    // Get a socket descriptor for "streams->fStreamSocketNum":
    SocketDescriptor* socketDescriptor
      = lookupSocketDescriptor(envir(), streams->fStreamSocketNum);
    if (socketDescriptor == NULL) {
      socketDescriptor
	= new SocketDescriptor(envir(), streams->fStreamSocketNum);
      socketHashTable(envir())->Add((char const*)(long)(streams->fStreamSocketNum),
				    socketDescriptor);
    }

    // Tell it about our subChannel:
    socketDescriptor->registerRTPInterface(streams->fStreamChannelId, this);
  }
}

Boolean RTPInterface::handleRead(unsigned char* buffer,
				 unsigned bufferMaxSize,
				 unsigned& bytesRead,
				 struct sockaddr_in& fromAddress) {
  Boolean readSuccess;
  if (fNextTCPReadStreamSocketNum < 0) {
    // Normal case: read from the (datagram) 'groupsock':
    readSuccess = fGS->handleRead(buffer, bufferMaxSize, bytesRead, fromAddress);
  } else {
    // Read from the TCP connection:
    bytesRead = 0;
    unsigned totBytesToRead = fNextTCPReadSize;
    if (totBytesToRead > bufferMaxSize) totBytesToRead = bufferMaxSize;
    unsigned curBytesToRead = totBytesToRead;
    int curBytesRead;
    while ((curBytesRead = readSocket(envir(), fNextTCPReadStreamSocketNum,
				      &buffer[bytesRead], curBytesToRead,
				      fromAddress)) > 0) {
      bytesRead += curBytesRead;
      if (bytesRead >= totBytesToRead) break;
      curBytesToRead -= curBytesRead;
    }
    if (curBytesRead <= 0) {
      bytesRead = 0;
      readSuccess = False;
    } else {
      readSuccess = True;
    }
    fNextTCPReadStreamSocketNum = -1; // default, for next time
  }

  if (readSuccess && fAuxReadHandlerFunc != NULL) {
    // Also pass the newly-read packet data to our auxilliary handler:
    (*fAuxReadHandlerFunc)(fAuxReadHandlerClientData, buffer, bytesRead);
  }
  return readSuccess;
}

void RTPInterface::stopNetworkReading() {
  // Normal case
  envir().taskScheduler().turnOffBackgroundReadHandling(fGS->socketNum());

  // Also turn off read handling on each of our TCP connections:
  for (tcpStreamRecord* streams = fTCPStreams; streams != NULL;
       streams = streams->fNext) {
    deregisterSocket(envir(), streams->fStreamSocketNum, streams->fStreamChannelId);
  }
}


////////// Helper Functions - Implementation /////////


SocketDescriptor::SocketDescriptor(UsageEnvironment& env, int socketNum)
  :fEnv(env), fOurSocketNum(socketNum),
    fSubChannelHashTable(HashTable::create(ONE_WORD_HASH_KEYS)),
   fServerRequestAlternativeByteHandler(NULL), fServerRequestAlternativeByteHandlerClientData(NULL),
   fTCPReadingState(AWAITING_DOLLAR) {
}

SocketDescriptor::~SocketDescriptor() {
  delete fSubChannelHashTable;
}

void SocketDescriptor::registerRTPInterface(unsigned char streamChannelId,
					    RTPInterface* rtpInterface) {
  Boolean isFirstRegistration = fSubChannelHashTable->IsEmpty();
  fSubChannelHashTable->Add((char const*)(long)streamChannelId,
			    rtpInterface);

  if (isFirstRegistration) {
    // Arrange to handle reads on this TCP socket:
    TaskScheduler::BackgroundHandlerProc* handler
      = (TaskScheduler::BackgroundHandlerProc*)&tcpReadHandler;
    fEnv.taskScheduler().
      turnOnBackgroundReadHandling(fOurSocketNum, handler, this);
  }
}

RTPInterface* SocketDescriptor
::lookupRTPInterface(unsigned char streamChannelId) {
  char const* lookupArg = (char const*)(long)streamChannelId;
  return (RTPInterface*)(fSubChannelHashTable->Lookup(lookupArg));
}

void SocketDescriptor
::deregisterRTPInterface(unsigned char streamChannelId) {
  fSubChannelHashTable->Remove((char const*)(long)streamChannelId);

  if (fSubChannelHashTable->IsEmpty()) {
    // No more interfaces are using us, so it's curtains for us now
    fEnv.taskScheduler().turnOffBackgroundReadHandling(fOurSocketNum);
    removeSocketDescription(fEnv, fOurSocketNum);
    delete this;
  }
}

extern void deleteRTSPClientConnectionFromDB(int sockfd);


void SocketDescriptor::tcpReadHandler(SocketDescriptor* socketDescriptor, int mask) {
  socketDescriptor->tcpReadHandler1(mask, socketDescriptor->fEnv, socketDescriptor->fOurSocketNum);
}

void SocketDescriptor::tcpReadHandler1(int mask, UsageEnvironment& env, int socketNum) {
	// We expect the following data over the TCP channel:
	//   optional RTSP command or response bytes (before the first '$' character)
	//   a '$' character
	//   a 1-byte channel id
	//   a 2-byte packet size (in network byte order)
	//   the packet data.
	// However, because the socket is being read asynchronously, this data might arrive in pieces.
  
	u_int8_t c;
	struct sockaddr_in fromAddress;
	struct timeval timeout; timeout.tv_sec = 0; timeout.tv_usec = 0;
    	Boolean socketClosed=False;
	while (1) {
		// int result = readSocket(env, socketNum, &c, 1, fromAddress);
		int result = readSocket(env, socketNum, &c, 1, fromAddress, &timeout, &socketClosed);

		if (result != 1) { // error reading TCP socket, or no more data available
#ifdef DEBUG
			fprintf(stderr, "Error: readSocket() from RTP/RTSP connection. socket: %d, result: %d, closed: %d\n", 
												socketNum, result, socketClosed);
#endif		
			if (result < 0  || socketClosed == True) {
	  			env.taskScheduler().turnOffBackgroundReadHandling(socketNum); // stops further calls to us
				deleteRTSPClientConnectionFromDB(socketNum);
			}
			return;
    		}
    
		switch (fTCPReadingState) {
		case AWAITING_DOLLAR:
			if (c == '$') {
				fTCPReadingState = AWAITING_STREAM_CHANNEL_ID;
			} else {
				// This character is part of a RTSP request or command, which is handled separately:
				if (fServerRequestAlternativeByteHandler != NULL) {
	    				(*fServerRequestAlternativeByteHandler)(fServerRequestAlternativeByteHandlerClientData, c);
				}
			}
			break;
		
		case AWAITING_STREAM_CHANNEL_ID: 
			// The byte that we read is the stream channel id.
			fStreamChannelId = c;
			fTCPReadingState = AWAITING_SIZE1;
			break;
		
		case AWAITING_SIZE1:
			// The byte that we read is the first (high) byte of the 16-bit RTP or RTCP packet 'size'.
			fSizeByte1 = c;
			fTCPReadingState = AWAITING_SIZE2;
			break;
    
		case AWAITING_SIZE2:
			// The byte that we read is the second (low) byte of the 16-bit RTP or RTCP packet 'size'.
			unsigned short size = (fSizeByte1<<8)|c;
			fTCPReadingState = AWAITING_DOLLAR;
	
			// Now that we know how much packet data to read, call this subchannel's read handler:
			RTPInterface* rtpInterface = lookupRTPInterface(fStreamChannelId);
			if (rtpInterface != NULL) {
	  			rtpInterface->fNextTCPReadSize = size;
	  			rtpInterface->fNextTCPReadStreamSocketNum = socketNum;
	  			rtpInterface->fNextTCPReadStreamChannelId = fStreamChannelId;
	  			if (rtpInterface->fReadHandlerProc != NULL) {
#ifdef DEBUG
	    				fprintf(stderr, "SocketDescriptor::tcpReadHandler() reading %d bytes on channel %d\n", 
	    						rtpInterface->fNextTCPReadSize, rtpInterface->fNextTCPReadStreamChannelId);
#endif
	    				rtpInterface->fReadHandlerProc(rtpInterface->fOwner, mask);
	  			}
			}
			return;
    		}
   	}
}


// TODO: ....
#if 0
void SocketDescriptor::tcpReadHandler(SocketDescriptor* socketDescriptor,
				      int mask) {
  do {
    UsageEnvironment& env = socketDescriptor->fEnv; // abbrev
    int socketNum = socketDescriptor->fOurSocketNum;

    // Begin by reading and discarding any characters that aren't '$'.
    // Any such characters are probably regular RTSP responses or
    // commands from the server.  At present, we can't do anything with
    // these, because we have taken complete control of reading this socket.
    // (Later, fix) #####
    unsigned char c;
    struct sockaddr_in fromAddress;
    struct timeval timeout; timeout.tv_sec = 0; timeout.tv_usec = 0;
    Boolean socketClosed=False;
    do {
	int result = readSocket(env, socketNum, &c, 1, fromAddress, &timeout, &socketClosed);
	if (result != 1) { // error reading TCP socket
#ifdef DEBUG
		fprintf(stderr, "Error: readSocket() from RTP/RTSP connection. socket: %d, result: %d, closed: %d\n", 
							socketNum, result, socketClosed);
#endif
		if (result < 0  || socketClosed == True) {
	  		env.taskScheduler().turnOffBackgroundReadHandling(socketNum); // stops further calls to us
			deleteRTSPClientConnectionFromDB(socketNum);
		}
		return;
	}
    } while (c != '$');

    // The next byte is the stream channel id:
    unsigned char streamChannelId;
    if (readSocket(env, socketNum, &streamChannelId, 1, fromAddress)
	!= 1) break;
    RTPInterface* rtpInterface
      = socketDescriptor->lookupRTPInterface(streamChannelId);
    if (rtpInterface == NULL) break; // we're not interested in this channel

    // The next two bytes are the RTP or RTCP packet size (in network order)
    unsigned short size;
    if (readSocketExact(env, socketNum, (unsigned char*)&size, 2,
			fromAddress) != 2) break;
    rtpInterface->fNextTCPReadSize = ntohs(size);
    rtpInterface->fNextTCPReadStreamSocketNum = socketNum;
    rtpInterface->fNextTCPReadStreamChannelId = streamChannelId;
#ifdef DEBUG
    printf("SocketDescriptor::tcpReadHandler() reading %d bytes on channel %d\n", 
    								rtpInterface->fNextTCPReadSize, streamChannelId);
#endif

    // Now that we have the data set up, call this subchannel's
    // read handler:
    if (rtpInterface->fReadHandlerProc != NULL) {
      rtpInterface->fReadHandlerProc(rtpInterface->fOwner, mask);
    }

  } while (0);
}
#endif



////////// tcpStreamRecord implementation //////////

tcpStreamRecord
::tcpStreamRecord(UsageEnvironment& env, 
		int streamSocketNum, unsigned char streamChannelId, tcpStreamRecord* next)
	:fNext(next), fStreamSocketNum(streamSocketNum), fStreamChannelId(streamChannelId) {

#ifdef DEBUG
	printf("[%p] tcpStreamRecord::tcpStreamRecord(sock %d, channel id %d)\n", 
									this, streamSocketNum, streamChannelId);
#endif

	fOutSocketDesc = OutputSocketDescriptor::getInstance(env, this);
	fHasStartQueuingInSlowMode=fFirstKeyFrameFragmentSeen = False;	// start with fast mode...
	fLastFrameFragmentSeen =True;
	fOverflowCurOffset = fOverflowDataLen = fOverflowBufferSize = 0;	
    	fOverflowBuffer=NULL;
}

tcpStreamRecord::~tcpStreamRecord() {

	if (fOverflowBuffer)
		free(fOverflowBuffer);

	if (fOutSocketDesc) {
		fOutSocketDesc->release(this);
		if (!fOutSocketDesc->hasSubChannels()) {
#ifdef DEBUG
			printf("[%p] socket %d is empty now !!!!! Remove it from table for deletion.\n", 
					fOutSocketDesc, fOutSocketDesc->fOurSocketNum);
#endif
			writeSockTable ->Remove((char const*)(long)(fOutSocketDesc->fOurSocketNum));
			delete fOutSocketDesc;
		}
	}

	delete fNext;
}


/* enqueueOverflowDataVector
  *	Data vector contains 2 parts
  *	   1) header  (	'$', channel Id, RTP packet len) - The full 4 bytes may have been sent partially
  *	   2) RTP packet
  */
int tcpStreamRecord::enqueueOverflowDataVector(
						unsigned char *header, unsigned headerSize,
						unsigned char *packet, unsigned packetSize)
{
	unsigned dataSize = headerSize + packetSize;
	
#ifdef DEBUG
	printf("Channel[%d:%d] - enqueue %d+%d bytes\n", fStreamSocketNum, fStreamChannelId, headerSize, packetSize);
#endif

	if (fOverflowDataLen + dataSize > fOverflowBufferSize) {

		// Reallocate buffer...
		// TODO: Some day review this carefully.  
		unsigned newBufferSize = 
				( (fOverflowDataLen + dataSize + DEFAULT_BUFFER_INCREMENT_SIZE -1) /
						DEFAULT_BUFFER_INCREMENT_SIZE) * DEFAULT_BUFFER_INCREMENT_SIZE;
		
#ifdef DEBUG
		printf("[%p] Request %d bytes: reallocate buffer size from %d to %d bytes\n", 
						this, fOverflowDataLen + dataSize, fOverflowBufferSize, newBufferSize);
#endif

		if (newBufferSize > MAXIMUM_BUFFER_SIZE) {

#if 1
			fprintf(stderr, "Warning: overflow data %d bytes exceeds max %d bytes.\n",
											newBufferSize, MAXIMUM_BUFFER_SIZE);
#endif

			if (headerSize == 4) {
				return -1;	// Discard a complete header + packet
			}		
			// A partial header-packet is sent.. make sure this packet is stored in its completion.
		}
		
		unsigned char *newBuffer=(unsigned char *)realloc(fOverflowBuffer, newBufferSize);
		if (!newBuffer) {
#ifdef DEBUG
			fprintf(stderr, "Error: can't realloc() \n");
#endif
			return -1;
		}
		fOverflowBuffer = newBuffer;
		fOverflowBufferSize = newBufferSize;
	}

	// Copy the header + packet
	if (headerSize > 0) {
		memcpy(&fOverflowBuffer[fOverflowDataLen], header, headerSize);
		fOverflowDataLen += headerSize;
	}
	memcpy(&fOverflowBuffer[fOverflowDataLen], packet, packetSize);
	fOverflowDataLen += packetSize;
	
	return 0;
}


Boolean tcpStreamRecord::sendOverflowDataToCompletion()
{
#ifdef DEBUG
	printf("Channel[%d:%d]->sendOverflowDataToCompletion()\n", fStreamSocketNum, fStreamChannelId);
#endif
	if (fOverflowCurOffset < fOverflowDataLen) {		
		// Has overflow data	

		// TODO: it seems send() will wanna send the data completely... please optimize..
		int bytesSent=send(fStreamSocketNum, &fOverflowBuffer[fOverflowCurOffset],  
									fOverflowDataLen -fOverflowCurOffset, 0);
		if (bytesSent > 0) {
#ifdef DEBUG
			printf("bytesSent: %d\n", bytesSent);
#endif
			fOverflowCurOffset+=bytesSent;
		}
		
		if (fOverflowCurOffset <fOverflowDataLen) {
#ifdef DEBUG
			printf("Still has data: %d/%d\n", fOverflowCurOffset, fOverflowDataLen);
#endif
			return False;	// still has more data..
		}
		fOverflowCurOffset = fOverflowDataLen=0;
	}

	// TODO: Shall we release memory when done???

#ifdef DEBUG
	if (fLastFrameFragmentSeen)
		printf("No more data\n");	
	else {
		printf("Still has data\n");
		dumpInfo();
	}
#endif
	return (fLastFrameFragmentSeen) ? True: False;
}

void tcpStreamRecord::dumpInfo() 
{
#ifdef DEBUG
	printf("Channel[%d:%d] - sending %d,  offset/end/size %lu, %lu, %lu\n",
			fStreamSocketNum, fStreamChannelId, fHasStartQueuingInSlowMode,
				fOverflowCurOffset, fOverflowDataLen, fOverflowBufferSize);
#endif
}

OutputSocketDescriptor *OutputSocketDescriptor::getInstance(
				UsageEnvironment& env, tcpStreamRecord* subChannel)
{
	int socketNum = subChannel->fStreamSocketNum;
	if (socketNum < 0)
		return NULL;

	// Search existing socket descriptor with the matching socket number..
	OutputSocketDescriptor *socketDesc=(OutputSocketDescriptor *) 
				writeSockTable ->Lookup((char const*)(long)socketNum);
	
	if (!socketDesc) {

		// Not found.. create a new instance.
		socketDesc = new OutputSocketDescriptor(env, socketNum);
		if (!socketDesc) {
#ifdef DEBUG
			fprintf(stderr, "Error: new OutputSocketDescriptor()\n");
#endif
			return NULL;
		}
		writeSockTable ->Add((char const*)(long)(socketNum), socketDesc);
	}

	//Register our channel with this socket descriptor..
	socketDesc->registerSubchannel(subChannel);
	return socketDesc;
}


void OutputSocketDescriptor::release(tcpStreamRecord* subChannel)
{
	unregisterSubchannel(subChannel);
}

Boolean OutputSocketDescriptor::hasSubChannels()
{
	return (!fSubChannelHashTable->IsEmpty());
}

OutputSocketDescriptor::OutputSocketDescriptor(UsageEnvironment& env, int socketNum):
	fEnv(env),
	fSubChannelHashTable(HashTable::create(ONE_WORD_HASH_KEYS)),
	fOurSocketNum(socketNum), fUsingSlowMode(False),
	fSlowSendTask(NULL)
{
#ifdef DEBUG
	printf("[%p] OutputSocketDescriptor::OutputSocketDescriptor(%d)\n", this, socketNum); 
#endif
}


OutputSocketDescriptor::~OutputSocketDescriptor()
{
#ifdef DEBUG
	printf("[%p] OutputSocketDescriptor::~OutputSocketDescriptor(%d)\n", this, fOurSocketNum); 
#endif

	delete fSubChannelHashTable;

	if (fSlowSendTask) {
		fEnv.taskScheduler().unscheduleDelayedTask(fSlowSendTask);
	}
}

int OutputSocketDescriptor::registerSubchannel(tcpStreamRecord *subChannel)
{
#ifdef DEBUG
	printf("Registering channel[%d] to socket [%d]\n", subChannel->fStreamChannelId, fOurSocketNum);
#endif

	if (fSubChannelHashTable->Lookup((char const*)(long)subChannel->fStreamChannelId)) {
#ifdef DEBUG
		printf("Already registered: %d\n", subChannel->fStreamChannelId);
#endif
		return 0;
	}

	// TODO: We shall check the state of the socket before adding..
	// Add to the table.
      	fSubChannelHashTable->Add((char const*)(long)(subChannel->fStreamChannelId), subChannel);
	return 0;
}

int OutputSocketDescriptor::unregisterSubchannel(tcpStreamRecord *subChannel)
{
#ifdef DEBUG
	printf("Unregistering channel[%d] from socket[%d]\n", subChannel->fStreamChannelId, fOurSocketNum);
#endif

 	fSubChannelHashTable->Remove((char const*)(long)subChannel->fStreamChannelId);
	return 0;
}

void OutputSocketDescriptor::enterSlowMode()
{
	tcpStreamRecord* subChannel;

#ifdef DEBUG
	printf("Socket[%d]->enterSlowMode()\n", fOurSocketNum);
#endif

	fUsingSlowMode=True;

	// TODO: This is rather waste of resource...
	HashTable::Iterator* iter = HashTable::Iterator::create(*fSubChannelHashTable);
	char const* key;
	while ((subChannel= (tcpStreamRecord*)(iter->next(key))) != 0) {
		subChannel->fHasStartQueuingInSlowMode = False;
	}	
	delete iter;	 

#ifdef DEBUG
	if (fSlowSendTask) {
		printf(" !!!! Warning: reschedule active send task.. should not happen..\n");
      }
#endif
	// Pre-schedule slow sending task...
	fSlowSendTask=fEnv.taskScheduler().scheduleDelayedTask(100 * 1000, sendQueuedTCPData, this);

#ifdef DEBUG
	if (!fSlowSendTask)
		printf("Error: Canot schedule send task\n");
#endif

}

void OutputSocketDescriptor::enterFastMode()
{
	tcpStreamRecord* subChannel;

#ifdef DEBUG
	printf("Socket[%d]->enterFastMode()\n", fOurSocketNum);
#endif


	fSlowSendTask=NULL;
	fUsingSlowMode=False;

	HashTable::Iterator* iter = HashTable::Iterator::create(*fSubChannelHashTable);
	char const* key;
	while ((subChannel= (tcpStreamRecord*)(iter->next(key))) != 0) {
		subChannel->fFirstKeyFrameFragmentSeen= False;
	}	
	delete iter;	 

}

int OutputSocketDescriptor::sendPacket(tcpStreamRecord* subChannel,
			unsigned char* packet, unsigned packetSize, 
			Boolean lastFragmentEndFrame, Boolean firstFragmentInKeyFrame)
{
	int bytesSent, bytesToSend;
	struct iovec vec[2];
	unsigned char header[4];

#ifdef DEBUG 
	printf("chan[%d:%d]::sendPacket() - %d bytes, last fragment %d, first key fragment %d\n",
					fOurSocketNum, subChannel->fStreamChannelId,
					packetSize, lastFragmentEndFrame, firstFragmentInKeyFrame);
#endif

	if (!fUsingSlowMode) {

		/* FAST mode: data is sent as fast as possible for this socket descriptor.  */
RETRY:
		if (subChannel->fFirstKeyFrameFragmentSeen) {
			
			// No overflow data, attempt to send new packet.
			header[0] = '$';
			header[1] = subChannel->fStreamChannelId;
			header[2] = (char) ((packetSize&0xFF00)>>8);
			header[3] = (char) (packetSize&0xFF);

			vec[0].iov_base = header;
			vec[0].iov_len = sizeof(header);
			vec[1].iov_base = packet;
			vec[1].iov_len = packetSize;
			bytesToSend = packetSize + 4;

			// Use writev to save multiple user-kernel context switch.
			bytesSent = writev(fOurSocketNum, vec, 2);			

			if (bytesSent < bytesToSend) {
				if (bytesSent == -1) {
					if (errno != EAGAIN && errno != EINTR) {
#ifdef DEBUG
						fprintf(stderr, "Warning: writev() fatal error. Skip packet\n");
#endif
						return -1;
					}
					bytesSent = 0;
				}

				// Can't send at full speed. Assume TCP connection is slow, switch to SLOW mode. 				
#ifdef DEBUG
				printf("writev(): %d/%d bytes => SLOW mode\n", bytesSent, bytesToSend);
#endif
				enterSlowMode();
				
				// Enqueue data to this subchannel and mark it the unfinished channel...	
				if (bytesSent < 4) {
					subChannel->enqueueOverflowDataVector(&header[bytesSent], 4-bytesSent,
												packet, packetSize);
				} else {
					subChannel->enqueueOverflowDataVector(NULL, 0,
												&packet[bytesSent-4], bytesToSend - bytesSent);
				}
				subChannel->fLastFrameFragmentSeen = lastFragmentEndFrame;

				fLastUnfinishChannel = subChannel->fStreamChannelId;
				subChannel->fHasStartQueuingInSlowMode = True;				
			}
			
		}else {

			// have not seen the first fragment...
			if (firstFragmentInKeyFrame) {
#ifdef DEBUG				
				printf("First key frame fragment\n");
#endif
				subChannel->fFirstKeyFrameFragmentSeen = True;
				goto RETRY;
			}
#ifdef DEBUG
			printf("Drop data\n");
#endif
		}
	}else {

		/* SLOW mode */
		
		/* This check allows at most one frame for to be queued and sent in slow mode. 
		  * What it does is to reset the fLastFrameFragmentSeen to 'False' 
		  *  if it just finishes sending  a complete frame ..
		  */
		if (!subChannel->fHasStartQueuingInSlowMode) {

			if (subChannel->fLastFrameFragmentSeen)
				subChannel->fLastFrameFragmentSeen = False;
			
			subChannel->fHasStartQueuingInSlowMode = True;
		}
		
		if (!subChannel->fLastFrameFragmentSeen) {

			// Enqueue more data...
			header[0] = '$';
			header[1] = subChannel->fStreamChannelId;
			header[2] = (char) ((packetSize&0xFF00)>>8);
			header[3] = (char) (packetSize&0xFF);
			
			subChannel->enqueueOverflowDataVector(header, sizeof(header), packet, packetSize);
			if (lastFragmentEndFrame) {
				subChannel->fLastFrameFragmentSeen=True;
#if DEBUG
				printf("Stop queuing\n");
				subChannel->dumpInfo();
#endif
			}
			return 0;
		} else {
#ifdef DEBUG
			// drop..
			printf("Drop data\n");
#endif
			return -1;
		}
	}
	return 0;
}


void OutputSocketDescriptor::sendQueuedTCPData(void* clientData)
{
#ifdef DEBUG
	printf("[%p] OutputSocketDescriptor::sendQueuedTCPData()\n", clientData);
#endif
	((OutputSocketDescriptor *)clientData)->sendQueuedTCPData1();
}
 
void OutputSocketDescriptor ::sendQueuedTCPData1()
{
	tcpStreamRecord* subChannel;

#ifdef DEBUG
	printf("[Socket %d]->sendQueuedTCPData1()\n", fOurSocketNum);
#endif

	// Start from the last unfinished channel.	
  	subChannel= (tcpStreamRecord*) fSubChannelHashTable->Lookup((char const*)(long)fLastUnfinishChannel);
#ifdef DEBUG
	if (!subChannel) {
		printf("Warning: cannot find channel object with ID: %d\n", fLastUnfinishChannel);
	}
#endif
	HashTable::Iterator* iter = HashTable::Iterator::create(*fSubChannelHashTable);
	char const* key;

	if (subChannel) {
		if (!subChannel->sendOverflowDataToCompletion())
			goto SCHEDULE_NEXT_SEND;
	}

	// Loop through each channel to see if anything left unsent...

	// TODO: Shall we make it member?
//	HashTable::Iterator* iter = HashTable::Iterator::create(*fSubChannelHashTable);
//	char const* key;
	while ((subChannel= (tcpStreamRecord*)(iter->next(key))) != 0) {
		if (!subChannel->sendOverflowDataToCompletion()) {
			fLastUnfinishChannel = subChannel->fStreamChannelId;
			delete iter;
			goto SCHEDULE_NEXT_SEND;
		}
	}

	delete iter;	 

	enterFastMode();	
	return;

	
SCHEDULE_NEXT_SEND:
	
	// Re-schedule in next 100 ms
	fSlowSendTask=fEnv.taskScheduler().scheduleDelayedTask(100 * 1000, sendQueuedTCPData, this);	
}
