/******************************************************************************

  Copyright (C), 2014-2024, Hisilicon Tech. Co., Ltd.

 ******************************************************************************
  File Name     : drv_hdmi_edid.c
  Version       : Initial Draft
  Author        : Hisilicon multimedia software group
  Created       : 2014/12/10
  Description   :
  History       :
  Date          : 2014/12/10
  Author        : t00273561
  Modification  :
*******************************************************************************/
#include "drv_hdmi_platform.h"
#include "drv_hdmi_intf.h"
#include "drv_hdmi_edid.h"
#include "drv_hdmi_edid_test.h"

/********************** private define ********************************************/
#define  BIT0_MASK                				0x01
#define  BIT1_MASK                				0x02
#define  BIT2_MASK                				0x04
#define  BIT3_MASK                				0x08
#define  BIT4_MASK                				0x10
#define  BIT5_MASK                				0x20
#define  BIT6_MASK                				0x40
#define  BIT7_MASK                				0x80

#define  EDID_EXTENSION_BLK_ADDR 				0x7e

/* VENDOR  INFO */
#define  EDID_VEND_NAME_CHAR_MASK				0x1F
#define  EDID_VEND_CHAR_LOW_INVALID			0
#define  EDID_VEND_CHAR_HIGH_INVALID			27
#define  EDID_VEND_YEAR_BASE 					1990

/* STD TIMING */
#define  EDID_STDTMING_UNUSED_FLAG				0x01
#define  EDID_STDTMING_RATE_BASE				60
#define  EDID_HORACTIVE_FACTOR					8
#define  EDID_HORACTIVE_BASE					31
#define  EDID_REFRESH_RATE_MASK 				0x3F
#define  EDID_ASPECT_RATIO_MASK				0xC0
#define  EDID_ASPECT_RATIO_16_10   			0  
#define  EDID_ASPECT_RATIO_4_3     			1  
#define  EDID_ASPECT_RATIO_5_4   				2  
#define  EDID_ASPECT_RATIO_16_9    			3  

/* Detailed Timing Descriptor */
#define  EDID_FST_BLK_DTD_OFFSET				0x36
#define  EDID_DTD_SIZE							18
#define  EDID_PIXCLK_KHZ_FACTOR 				10

#define  EDID_UPPER_NIBBLE_MASK				0xF0
#define  EDID_LOWER_NIBBLE_MASK				0x0F

#define  EDID_HSO_MASK 						0xC0
#define  EDID_HSPW_MASK 						0x30
#define  EDID_VSO_MASK 						0x0C 
#define  EDID_VSPW_MASK 						0x03

#define  EDID_STEREO_MASK 						0x60
#define  EDID_STEREO_SEQUENTIAL_R 				0x02
#define  EDID_STEREO_SEQUENTIAL_L 				0x04
#define  EDID_STEREO_INTERLEAVED_2R 			0x03
#define  EDID_STEREO_INTERLEAVED_2L 			0x05
#define  EDID_STEREO_INTERLEAVED_4 			0x06
#define  EDID_STEREO_INTERLEAVED_SBS			0x07 

#define  EDID_SYNC_SIGNAL_TYPE_MASK 			0x0E
#define  EDID_SYNC_DCS_WS_0 					0x00
#define  EDID_SYNC_DCS_WS_1 					0x01
#define  EDID_SYNC_DCS_DS_2 					0x02
#define  EDID_SYNC_DCS_DS_3 					0x03
#define  EDID_SYNC_DSS_VN_HN_4 				0x04
#define  EDID_SYNC_DSS_VN_HP_5 				0x05
#define  EDID_SYNC_DSS_VP_HN_6 				0x06
#define  EDID_SYNC_DSS_VP_HP_7 				0x07

/* EXTENSION BLOCK */
#define  EDID_CEA_EXTVERSION3_TAG         		0x02
#define  EDID_CEA_EXTVERSION3_REVISION        0x03
#define  EDID_DB_LEN_MASK  					0x1F
#define  EDID_DB_TAG_CODE_MASK 				0xE0

/* Tag Code */
#define  EDID_REVERSED_DATA_BLOCK    			0x00
#define  EDID_AUDIO_DATA_BLOCK    				0x01
#define  EDID_VIDEO_DATA_BLOCK    				0x02
#define  EDID_VENDOR_DATA_BLOCK   				0x03
#define  EDID_SPEAKER_DATA_BLOCK  				0x04
#define  EDID_VESA_DTC_DATA_BLOCK 				0x05
#define  EDID_USE_EXT_DATA_BLOCK  				0x07

/* Ext Tag Code */
#define  EDID_VIDEO_CAPABILITY_DB      		0x00
//#define   EDID_VENDOR_SPECIFIC_VIDEO_DB 		0x01
//#define   EDID_VESA_DISPLAY_DEVICE_DB     		0x02
//#define   EDID_RESERVED_VESA_VIDEO_DB   		0x03
//#define   EDID_VESA_VIDEO_TIMING_DB   		0x04
#define  EDID_COLORIMETRY_DB           		0x05
#define  EDID_VIDEO_FMT_PREFER_DB				0x0d
#define  EDID_Y420_VIDEO_DB           			0x0e
#define  EDID_Y420_CAPABILITY_MAP_DB  			0x0f
//#define   EDID_MISCELLANENOUS_AUDIO_FIELDS  0x10
//#define   EDID_VENDOR_SPECIFIC_AUDIO_DB 		0x11
#define  EDID_RESERVED_HDMI_AUDIO_DB  			0x12
#define  EDID_INFOFRAME_DB  					0x20

/* AUDIO DATA BLOCK */
#define  EDID_AUDIO_FORMAT_MASK   				0x78
#define  EDID_AUDIO_CHANNEL_MASK   			0x07
#define  EDID_AUDIO_EXT_TYPE_CODE 				0xf8
#define  EDID_AUDIO_BITRATE_FACTOR				8

/* VIDEO DATA BLOCK */
#define  EDID_VIC_NATIVE_MAX            		64
#define  EDID_VIC_LOWER7_MASK            		0x7F
#define  EDID_VIC_INVALID_LOW            		128
#define  EDID_VIC_INVALID_HIGH					192
#define  EDID_VIC_INVALID_ZERO					0

/* HFVSDB */
#define  EDID_HFVSDB_VERSION					1
#define  EDID_MAX_HDMI14_TMDS_RATE				340
#define  EDID_TMDS_FACTOR 						5
#define  EDID_IEEE_VSDB_1ST 					0x03
#define  EDID_IEEE_VSDB_2ND 					0x0c
#define  EDID_IEEE_VSDB_3RD 					0x00

#define  EDID_IEEE_HFVSDB_1ST 					0xd8
#define  EDID_IEEE_HFVSDB_2ND 					0x5d
#define  EDID_IEEE_HFVSDB_3RD 					0xc4


/* VSDB */
#define  EDID_CEC_INVALID_ADDR 				0xF
#define  EDID_IMG_SIZE_MASK 					0x18
#define  EDID_HDMI_3D_LEN_MASK 				0x1F
#define  EDID_HDMI_VIC_LEN_MASK 				0xE0
#define  EDID_3DMULTI_PRESENT_LOWER8			0x01
#define  EDID_3DMULTI_PRESENT_UPPER8			0x02

#define EDID_NULL_CHK(p)	do{\
	if(HI_NULL==p)\
	{EDID_ERR("NULL pionter!\n");return HI_FAILURE;}\
}while(0);

#define EDID_SUCCESS_CHK(res)	do{\
	if(HI_FAILURE==res)\
	{EDID_ERR("return failure!\n");return HI_FAILURE;}\
}while(0);

typedef struct
{
    HI_U8 au8PixelClk[2];
    HI_U8 u8H_Active;
    HI_U8 u8H_Blank;
    HI_U8 u8H_ActiveBlank;
    HI_U8 u8V_Active;
    HI_U8 u8V_Blank;
    HI_U8 u8V_ActiveBlank;
    HI_U8 u8H_SyncOffset;
    HI_U8 u8H_SyncPulseWidth;
    HI_U8 u8VS_OffsetPulseWidth;
    HI_U8 u8HS_Offset_VS_Offset;
    HI_U8 u8H_ImageSize;
    HI_U8 u8V_ImageSize;
    HI_U8 u8H_V_ImageSize;
    HI_U8 u8H_Border;
    HI_U8 u8V_Border;
    HI_U8 u8Flags;
} __attribute__((packed))EDID_DTD_TIMING_BLOCK;


typedef struct 
{
	HI_U8 au8Header[8];
	/* Vendor & product info */
	HI_U8 au8MfgId[2];
	HI_U8 au8ProdCode[2];
	/* FIXME: byte order */
	HI_U8 au8Serial[4]; 
	HI_U8 u8MfgWeek;
	HI_U8 u8MfgYear;
	/* EDID version */
	HI_U8 u8Version;
	HI_U8 u8Revision;
	/* Display info: */
	HI_U8 u8Input;
	HI_U8 u8WidthCm;
	HI_U8 u8HeightCm;
	HI_U8 u8Gamma;
	HI_U8 u8Features;
	/* Color characteristics */
	HI_U8 u8RedGreenLow;
	HI_U8 u8BlackWhiteLow;
	HI_U8 u8RedX;
	HI_U8 u8RedY;
	HI_U8 u8GreenX;
	HI_U8 u8GreenY;
	HI_U8 u8BlueX;
	HI_U8 u8BlueY;
	HI_U8 u8WhiteX;
	HI_U8 u8WhiteY;
    HI_U8 au8EstTiming[3];
    HI_U8 au8StdTiming[16];
    HI_U8 au8DetailedTiming[72];
	/* Number of 128 byte ext. blocks */
	HI_U8 u8ExtBlocks;
	/* Checksum */
	HI_U8 u8ChkSum;
} __attribute__((packed))EDID_FIRST_BLOCK_INFO;



/********************** private interface ********************************************/

static HI_S32 EdidBlockChkSum(HI_U8 *pau8Data,
							HDMI_EDID_STATUS_S *pstStatus)
{
    HI_U8 	i = 0;
    HI_U32 	u32CheckSum = 0;

    EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

    for(i = 0; i < HDMI_EDID_BLOCK_SIZE; i++)
    {
        u32CheckSum += pau8Data[i];
    }

    if((u32CheckSum & 0xff) != 0x00)
    {
    	pstStatus->enParseErr = EDID_PARSE_ERR_CHECKSUM;
		EDID_ERR("u32CheckSum:0x%02x\n",u32CheckSum&0xff);
		return HI_FAILURE;
    }
	
    return HI_SUCCESS;
}

static HI_S32 EdidHeaderChk(HI_U8 *pau8Data,
								HDMI_EDID_STATUS_S *pstStatus)
{
    HI_U32 i = 0;
    const HI_U8 au8Blkheader[] ={0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00};

	EDID_NULL_CHK(pstStatus);

    for (i = 0; i < 8; i++)
    {  
        if(pau8Data[i] != au8Blkheader[i])
        {
        	pstStatus->enParseErr = EDID_PARSE_ERR_HEADER;
			EDID_ERR("Header Chk: DATA[%d]=0x%x\n",i,pau8Data[i]);
			return HI_FAILURE;
        }
    }
	
    return HI_SUCCESS;
}

static HI_S32 EdidVendorInfo(HDMI_EDID_INFO_S *pstEdidInfo)
{
    HI_U16 							i = 0;
	HI_U16 							u16Data = 0;
    HDMI_SINK_CAPABILITY_S 			*pstCap = HI_NULL;
    HDMI_EDID_MANUFACTURE_INFO_S 	*pstVendor = HI_NULL;
	EDID_FIRST_BLOCK_INFO			*pstFstBlk = HI_NULL;
	HDMI_EDID_STATUS_S				*pstStatus = HI_NULL;
	
	EDID_NULL_CHK(pstEdidInfo);
	pstFstBlk	= (EDID_FIRST_BLOCK_INFO *)pstEdidInfo->au8EdidRaw;
	EDID_NULL_CHK(pstFstBlk);
	pstCap 		= &pstEdidInfo->stCapability;
	pstStatus		= &pstEdidInfo->stStatus;
	pstVendor 	= &pstCap->stMfrsInfo;

	u16Data = (pstFstBlk->au8MfgId[0]<<8) | (pstFstBlk->au8MfgId[1]);

	/* 2Bytes(16bits) width,5-bits a character. */
	for(i = 0; i < 3; i++)
	{
	    pstVendor->u8MfrsName[2 - i]= ((u16Data & (EDID_VEND_NAME_CHAR_MASK << (5*i))) >> (5*i));
		
		/* 'A'~'Z' is refered to 1~26 */
		if((EDID_VEND_CHAR_LOW_INVALID < pstVendor->u8MfrsName[2 - i])
			&& (EDID_VEND_CHAR_HIGH_INVALID > pstVendor->u8MfrsName[2 - i]))
		{		
			pstVendor->u8MfrsName[2 - i] += 'A' - 1;
		}
		else
		{
			pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_VENDOR_INVALID;
		    EDID_WARN("Vend INFO MFG NAME!\n");			
		}
	}
	
	pstVendor->u32ProductCode       = (pstFstBlk->au8ProdCode[1] << 8) | pstFstBlk->au8ProdCode[0];
	pstVendor->u32SerialNumber      = (pstFstBlk->au8Serial[3] << 24) | (pstFstBlk->au8Serial[2] << 16) \
									  | (pstFstBlk->au8Serial[1] << 8) | (pstFstBlk->au8Serial[0]);
	pstVendor->u32Week 				= pstFstBlk->u8MfgWeek ;
	pstVendor->u32Year 				= pstFstBlk->u8MfgYear + EDID_VEND_YEAR_BASE ;

	HDMI_DBG("Vend INFO u8MfrsName		:%s\n",pstVendor->u8MfrsName);
	HDMI_DBG("Vend INFO u32ProductCode	:%d\n",pstVendor->u32ProductCode);
	HDMI_DBG("Vend INFO u32SerialNumber	:%d\n",pstVendor->u32SerialNumber);
	HDMI_DBG("Vend INFO u32Year			:%d\n",pstVendor->u32Year);
	HDMI_DBG("Vend INFO u32Week			:%d\n",pstVendor->u32Week);

    return HI_SUCCESS;
}

static HI_S32 EdidVersion(HDMI_EDID_INFO_S *pstEdidInfo)
{
    HDMI_SINK_CAPABILITY_S 	*pstCap = HI_NULL;
	EDID_FIRST_BLOCK_INFO 	*pEdidFstInfo = HI_NULL;
	HDMI_EDID_STATUS_S		*pstStatus = HI_NULL;

	EDID_NULL_CHK(pstEdidInfo);
	pstCap = &pstEdidInfo->stCapability;
	EDID_NULL_CHK(pstCap);
    pEdidFstInfo = (EDID_FIRST_BLOCK_INFO*)pstEdidInfo->au8EdidRaw;
	EDID_NULL_CHK(pEdidFstInfo);
	pstStatus	= &pstEdidInfo->stStatus;

    pstCap->u8Version  = pEdidFstInfo->u8Version;
    pstCap->u8Revision = pEdidFstInfo->u8Revision;
	
	if ( (pstCap->u8Version != 1) || (pstCap->u8Revision != 3 ) )
	{
		EDID_ERR("EDID first blk is not version1.3:%d.%d\n!",
					pstCap->u8Version,pstCap->u8Revision);
		pstStatus->enParseErr = EDID_PARSE_ERR_FST_BLK_VER;
		return HI_FAILURE;
		
	}

    HDMI_DBG("VER u8Version:\t%02d\n",pstCap->u8Version);
	HDMI_DBG("VER u8Revision:\t%02d\n",pstCap->u8Revision);

    return HI_SUCCESS;
}


static HI_S32 EdidStdTiming(HDMI_EDID_INFO_S *pstEdidInfo)
{
	HI_U8					i = 0;
	HI_U8 					u8AspRatio = 0;
    HI_U8  					*pau8Data = HI_NULL;
    HDMI_SINK_CAPABILITY_S 	*pstCap = HI_NULL;
	EDID_FIRST_BLOCK_INFO 	*pstFstBlk = HI_NULL;

	EDID_NULL_CHK(pstEdidInfo);
	pstCap = &pstEdidInfo->stCapability;
    pstFstBlk = (EDID_FIRST_BLOCK_INFO*)pstEdidInfo->au8EdidRaw;
	EDID_NULL_CHK(pstFstBlk);
	pau8Data = pstFstBlk->au8StdTiming;
	EDID_NULL_CHK(pau8Data);

	HDMI_MEMSET(pstCap->stStdTiming,0,sizeof(pstCap->stStdTiming));

    for(i = 0; i < HDMI_EDID_MAX_STDTIMNG_COUNT ; i++,pau8Data+=2)
    {
        if((pau8Data[0] == EDID_STDTMING_UNUSED_FLAG)
			&&(pau8Data[1]==EDID_STDTMING_UNUSED_FLAG))
        {
            HDMI_DBG("STDTIMNG field[%d] un-used!\n",i);
        }
        else
        {
            pstCap->stStdTiming[i].u32HorActive = (pau8Data[0] + EDID_HORACTIVE_BASE) * EDID_HORACTIVE_FACTOR;
			pstCap->stStdTiming[i].u32RefreshRate	= (pau8Data[1] & EDID_REFRESH_RATE_MASK) + EDID_STDTMING_RATE_BASE;
            u8AspRatio = (pau8Data[1] & EDID_ASPECT_RATIO_MASK) >> 6;
			
			switch(u8AspRatio)
			{
				case EDID_ASPECT_RATIO_16_10:
					pstCap->stStdTiming[i].u32VerActive = pstCap->stStdTiming[i].u32HorActive*10/16 ;
					break;
				case EDID_ASPECT_RATIO_5_4:
					pstCap->stStdTiming[i].u32VerActive = pstCap->stStdTiming[i].u32HorActive*4/5 ;
					break;
				case EDID_ASPECT_RATIO_4_3:
					pstCap->stStdTiming[i].u32VerActive = pstCap->stStdTiming[i].u32HorActive*3/4 ;
					break;
				case EDID_ASPECT_RATIO_16_9:
					pstCap->stStdTiming[i].u32VerActive = pstCap->stStdTiming[i].u32HorActive*9/16 ;
					break;

				default:
					break;
			}
			HDMI_DBG("STDTIMNG field[%d] u32HorActive:%u\n",i,pstCap->stStdTiming[i].u32HorActive);
			HDMI_DBG("STDTIMNG field[%d] u32VerActive:%u\n",i,pstCap->stStdTiming[i].u32VerActive);
			HDMI_DBG("STDTIMNG field[%d] u32RefreshRate:%u\n",i,pstCap->stStdTiming[i].u32RefreshRate);
            
        }
		
    }
	
    return HI_SUCCESS;
}


static HI_S32 EdidEstablishTiming(HDMI_EDID_INFO_S *pstEdidInfo)
{
	HI_U32	i = 0;
    HDMI_SINK_CAPABILITY_S 	*pstCap = HI_NULL;
	EDID_FIRST_BLOCK_INFO	*pstFstBlk = HI_NULL;

	EDID_NULL_CHK(pstEdidInfo);

	pstCap = &pstEdidInfo->stCapability;
	pstFstBlk = (EDID_FIRST_BLOCK_INFO	*)pstEdidInfo->au8EdidRaw;
	EDID_NULL_CHK(pstFstBlk);


	for (i=0; i <= HDMI_EDID_ESTABTIMG_BUTT; i++)
	{
		if(pstCap->u32EstabNum >= HDMI_EDID_ESTABTIMG_BUTT)
		{
			EDID_WARN("EstablishTiming u32EstabNum over:%d!\n",pstCap->u32EstabNum);
			break;
		}
		/*if BYTE(i/8),BIT[i%8] is 1*/
		if ( pstFstBlk->au8EstTiming[i/8] & (0x01<<(i%8)) )
		{
			pstCap->au32EstabTiming[pstCap->u32EstabNum++] = i;
			HDMI_DBG("EstablishTiming u32EstabTiming[%d]:%d\n",pstCap->u32EstabNum,i);
		}
	}

	return HI_SUCCESS;
}


static HI_S32 EdidDetailTiming(HDMI_SINK_CAPABILITY_S *pstCap,
										HDMI_EDID_STATUS_S *pstStatus,
										HI_U8 *pau8Data,
										HI_U8 u8Len)
{
    HI_U32	 				u32Temp = 0;
	HDMI_EDID_PRE_TIMING_S	*pstCapPreTiming = HI_NULL; 
	EDID_DTD_TIMING_BLOCK 	*pstDTD = HI_NULL;

	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

	if (pstCap->u32PerferTimingNum >= HDMI_EDID_MAX_DETAIL_TIMING_COUNT
		|| u8Len < EDID_DTD_SIZE)
	{
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_DTD_OVER;
		EDID_WARN("DTDs over max DTDs cnt or less u8Len:%d!\n",u8Len);
		return HI_SUCCESS;
	}

	pstDTD = (EDID_DTD_TIMING_BLOCK*)pau8Data;
	pstCapPreTiming = &pstCap->stPerferTiming[pstCap->u32PerferTimingNum];
	
    /* pixel clock ,KHz */
    u32Temp = pstDTD->au8PixelClk[1];
	u32Temp <<= 8;
	u32Temp |= pstDTD->au8PixelClk[0];
    u32Temp *= EDID_PIXCLK_KHZ_FACTOR;
	pstCapPreTiming->u32PixelClk = u32Temp;
	HDMI_DBG("DTDs pixel CLK(KHz):%d\n",u32Temp);
	
	if (u32Temp > 0)
	{
		HDMI_DBG("DTDs u32PerferTimingNum:%d\n",pstCap->u32PerferTimingNum);
		pstCap->u32PerferTimingNum++;
	}
	else
	{
		HDMI_DBG("DTDs empty DTDs,CurNum:%d\n",pstCap->u32PerferTimingNum);
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_DTD_INVALID;
		return HI_SUCCESS;
	}
	

     /* VFB ,2+4 bits */ 
    u32Temp = pstDTD->u8HS_Offset_VS_Offset & EDID_VSO_MASK  ;
	u32Temp <<= 2;
    u32Temp |= (pstDTD->u8VS_OffsetPulseWidth & EDID_UPPER_NIBBLE_MASK) >> 4 ;
	//u32Temp += pstDTD->u8V_Border ;
	pstCapPreTiming->u32VFB = u32Temp;
	HDMI_DBG("DTDs VFB :%d\n",u32Temp);

	/*v_active_blank ( vblack) = vfront + vback + vsync = VFB(vfront+ vsync) + VBB(vback )*/

    /* VBB ,4+8 bits*/ 
    u32Temp = pstDTD->u8V_ActiveBlank & EDID_LOWER_NIBBLE_MASK;
    u32Temp <<= 8;
	u32Temp |= pstDTD->u8V_Blank ;
    //pstCapPreTiming->u32VBB = u32Temp - pstCapPreTiming->u32VFB;
	if (u32Temp >= pstCapPreTiming->u32VFB)
	{
		pstCapPreTiming->u32VBB = u32Temp - pstCapPreTiming->u32VFB;
		
	}
	else
	{
		HDMI_WARN("DTDs invalid V_Blank < u32VFB (%d<%d)\n",u32Temp,pstCapPreTiming->u32VFB);
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_DTD_INVALID;
		pstCapPreTiming->u32VBB = 0;
	}
    HDMI_DBG("DTDs VBB :%d\n",pstCapPreTiming->u32VBB);

    /* VACT ,4+8 bits */
    u32Temp = pstDTD->u8V_ActiveBlank & EDID_UPPER_NIBBLE_MASK;
	u32Temp <<= 4;
	u32Temp|= pstDTD->u8V_Active ;
    pstCapPreTiming->u32VACT = u32Temp;
    HDMI_DBG("DTDs VACT :%d\n",u32Temp);

    /* HFB ,2+8 bits */
    u32Temp = pstDTD->u8HS_Offset_VS_Offset &  EDID_HSO_MASK;
	u32Temp <<= 2;
    u32Temp |= pstDTD->u8H_SyncOffset ;
	//u32Temp += pstDTD->u8H_Border ;
	pstCapPreTiming->u32HFB = u32Temp;
    HDMI_DBG("DTDs HFB :%d\n",u32Temp);

	/*h_active_blank ( hblack ) == hfront + hback + hsync = HFB(hfront + hsync) + HBB(hback)*/
    /* HBB , 4+8 bits */ 
    u32Temp = (pstDTD->u8H_ActiveBlank & EDID_LOWER_NIBBLE_MASK);
	u32Temp <<= 8;
	u32Temp |= pstDTD->u8H_Blank ;//hblank
	if (u32Temp >= pstCapPreTiming->u32HFB)
	{
		pstCapPreTiming->u32HBB = u32Temp - pstCapPreTiming->u32HFB;
		
	}
	else
	{
		HDMI_WARN("DTDs invalid H_Blank < HFB (%d<%d)\n",u32Temp,pstCapPreTiming->u32HFB);
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_DTD_INVALID;
		pstCapPreTiming->u32HBB = 0;
	}
	HDMI_DBG("DTDs HBB :%d\n",pstCapPreTiming->u32HBB);
    


    /* HACT,4+8 bits */
    u32Temp = pstDTD->u8H_ActiveBlank & EDID_UPPER_NIBBLE_MASK;
    u32Temp <<= 4;
	u32Temp	|= pstDTD->u8H_Active ;
    pstCapPreTiming->u32HACT = u32Temp;
    HDMI_DBG("DTDs HACT :%d\n",u32Temp);

	/* VPW,2+4 bits */
    u32Temp = pstDTD->u8HS_Offset_VS_Offset & EDID_VSPW_MASK ;
	u32Temp <<= 4;
    u32Temp |= (pstDTD->u8VS_OffsetPulseWidth & EDID_LOWER_NIBBLE_MASK);
    pstCapPreTiming->u32VPW = u32Temp;
	HDMI_DBG("DTDs VPW :%d\n",u32Temp);

    /* HPW,2+8 bits */
    u32Temp = pstDTD->u8HS_Offset_VS_Offset & EDID_HSPW_MASK ;
	u32Temp <<= 4;							
    u32Temp |= pstDTD->u8H_SyncPulseWidth ;	
    pstCapPreTiming->u32HPW = u32Temp;
	HDMI_DBG("DTDs HPW :%d\n",u32Temp);

    /* H image size(mm),4+8 bits */
    u32Temp = pstDTD->u8H_V_ImageSize & EDID_UPPER_NIBBLE_MASK ;
	u32Temp <<= 4;
	u32Temp |= pstDTD->u8H_ImageSize ;
    pstCapPreTiming->u32ImageWidth = u32Temp;
    HDMI_DBG("DTDs H image size(mm) :%d\n",u32Temp);


    /* V image size(mm),4+8 bits */
    u32Temp = pstDTD->u8H_V_ImageSize & EDID_LOWER_NIBBLE_MASK;
	u32Temp <<= 8;
    u32Temp |= pstDTD->u8V_ImageSize ;
    pstCapPreTiming->u32ImageHeight = u32Temp;
	HDMI_DBG("DTDs V image size(mm) :%d\n",u32Temp);

	/*Interlaced flag*/
    if(pstDTD->u8Flags & BIT7_MASK)         
	{      
        pstCapPreTiming->bInterlace = HI_TRUE;
		HDMI_DBG("DTDs Output mode: interlaced\n");
	}
	else
	{
        pstCapPreTiming->bInterlace = HI_FALSE;
		HDMI_DBG("DTDs Output mode: progressive\n");
	}

	/*Stereo Viewing Support*/
	switch(((pstDTD->u8Flags & EDID_STEREO_MASK ) >> 4)|(pstDTD->u8Flags & BIT0_MASK))
	{
		case EDID_STEREO_SEQUENTIAL_L :
			HDMI_DBG("DTDs stereo sequential L\n");
			break;

		case EDID_STEREO_INTERLEAVED_2R :
			HDMI_DBG("DTDs stereo interleaved 2R\n");
			break;

		case EDID_STEREO_INTERLEAVED_2L :
			HDMI_DBG("DTDs stereo interleaved 2L\n");
			break;

		case EDID_STEREO_INTERLEAVED_4 :
			 HDMI_DBG("DTDs stereo interleaved 4\n");
			break;

		case EDID_STEREO_INTERLEAVED_SBS :
			HDMI_DBG("DTDs stereo interleaved SBS\n");
			break;

		default:
			HDMI_DBG("DTDs no stereo \n");
			break;

	}

	/*Analog Sync Signal Definitions*/
    if(0 == (pstDTD->u8Flags & BIT4_MASK))   
	{
		switch((pstDTD->u8Flags & EDID_SYNC_SIGNAL_TYPE_MASK ) >> 1)
		{
			/*Analog Composite Sync - Without Serrations - Sync On Green Signal only*/
			case 0x00:          		
                HDMI_DBG("DTDs sync acs ws green\n");
				break;
				
			/*Analog Composite Sync - Without Serrations - Sync On all three (RGB) video signals*/
			case 0x01:                 
                HDMI_DBG("DTDs sync acs ws all\n");
				break;

			/*Analog Composite Sync - With Serrations (H-sync during V-sync); - Sync On Green Signal only*/
			case 0x02:                 
                HDMI_DBG("DTDs sync acs ds green\n");
				break;

			/*Analog Composite Sync - With Serrations (H-sync during V-sync); - Sync On all three (RGB) video signals*/
			case 0x03:                  
                HDMI_DBG("DTDs sync acs ds all\n");
                break;

			/*Bipolar Analog Composite Sync - Without Serrations; - Sync On Green Signal only*/
			case 0x04:                 
                HDMI_DBG("DTDs sync bacs ws green\n");
                break;

			/*Bipolar Analog Composite Sync - Without Serrations; - Sync On all three (RGB) video signals*/
			case 0x05:                 
                HDMI_DBG("DTDs sync bacs ws all\n");
                break;
				
			/*Bipolar Analog Composite Sync - With Serrations (H-sync during V-sync); - Sync On Green Signal only*/
			case 0x06:                  
                HDMI_DBG("DTDs sync bacs ds green\n");
                break;

			/*Bipolar Analog Composite Sync - With Serrations (H-sync during V-sync); - Sync On all three (RGB) video signals*/
			case 0x07:                  
                HDMI_DBG("DTDs sync bacs ds all\n");
                break;
				
			default:
				break;

          }
    }
	/* Digital Sync Signal Definitions */
    else                
    {
 		switch((pstDTD->u8Flags & EDID_SYNC_SIGNAL_TYPE_MASK) >> 1)
		{
			
			/* DCS:Digital Composite Sync */
			case EDID_SYNC_DCS_WS_0:
            case EDID_SYNC_DCS_WS_1:                  
                pstCapPreTiming->bIHS = HI_FALSE;
                pstCapPreTiming->bIVS = HI_FALSE;
                break;

			case EDID_SYNC_DCS_DS_2:                   
			case EDID_SYNC_DCS_DS_3:
                break;
				
			/* DSS:Digital Separate Sync */
			case EDID_SYNC_DSS_VN_HN_4:                  
                pstCapPreTiming->bIHS = HI_FALSE;
                pstCapPreTiming->bIVS = HI_FALSE;
                break;

			case EDID_SYNC_DSS_VN_HP_5:                  
                pstCapPreTiming->bIHS = HI_TRUE;
                pstCapPreTiming->bIVS = HI_FALSE;
                break;

			case EDID_SYNC_DSS_VP_HN_6:                  
                pstCapPreTiming->bIHS = HI_FALSE;
                pstCapPreTiming->bIVS = HI_TRUE;
                break;
				
			case EDID_SYNC_DSS_VP_HP_7:                  
                pstCapPreTiming->bIHS = HI_TRUE;
                pstCapPreTiming->bIVS = HI_TRUE;
                break;

			default:
				break;

		}
    }
    pstCapPreTiming->bIDV = HI_FALSE;
    return HI_SUCCESS;
}

static HI_S32 EdidExtNum(HDMI_EDID_INFO_S *pstEdidInfo)
{
	HDMI_EDID_STATUS_S		*pstStatus = HI_NULL;
    HI_U8  					*pau8Data = HI_NULL;
    HDMI_SINK_CAPABILITY_S 	*pstCap = HI_NULL;

	EDID_NULL_CHK(pstEdidInfo);
	
	pstStatus 	= &pstEdidInfo->stStatus;
	pau8Data 	= pstEdidInfo->au8EdidRaw;
	pstCap 		= &pstEdidInfo->stCapability;

	pstCap->u8ExtBlockNum = (HI_U8)pau8Data[EDID_EXTENSION_BLK_ADDR];
	pstCap->bSupportHdmi = HI_FALSE;

	HDMI_DBG("Ext-BLK num:%d\n",pstCap->u8ExtBlockNum);
	
	if(pstCap->u8ExtBlockNum == 0)
	{
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_EXT_BLK_ZERO;
		EDID_WARN("Ext-BLK zero: 0x%02x \n",pstCap->u8ExtBlockNum);
	}
	else if(pstCap->u8ExtBlockNum > (HDMI_EDID_MAX_BLOCK_NUM - 1))
	{
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_EXT_BLK_OVER;
		EDID_WARN("Ext-BLK cnt over: 0x%02x \n",pstCap->u8ExtBlockNum);

	}

	return HI_SUCCESS;
}


static HI_S32 EdidFirstBlkParse(HDMI_EDID_INFO_S *pstEdidInfo)
{
	HI_S32					s32Ret = HI_SUCCESS;
	HDMI_EDID_STATUS_S		*pstStatus = HI_NULL;
    HI_U8  					*pau8Data = HI_NULL;
	HDMI_SINK_CAPABILITY_S 	*pstCap = HI_NULL;
	EDID_FIRST_BLOCK_INFO	*pFstInfo = HI_NULL;


	EDID_NULL_CHK(pstEdidInfo);
	pstStatus = &pstEdidInfo->stStatus;
	pau8Data = pstEdidInfo->au8EdidRaw;
	pstCap = &pstEdidInfo->stCapability;
	pFstInfo = (EDID_FIRST_BLOCK_INFO *)pau8Data;
	EDID_NULL_CHK(pau8Data);
	

	s32Ret |= EdidBlockChkSum(pau8Data,pstStatus);
	EDID_SUCCESS_CHK(s32Ret);

    s32Ret |= EdidHeaderChk(pau8Data,pstStatus);
	EDID_SUCCESS_CHK(s32Ret);

    s32Ret |= EdidVendorInfo(pstEdidInfo);
	EDID_SUCCESS_CHK(s32Ret);
	
    s32Ret |= EdidVersion(pstEdidInfo);
	EDID_SUCCESS_CHK(s32Ret);
	
    s32Ret |= EdidEstablishTiming(pstEdidInfo);
	EDID_SUCCESS_CHK(s32Ret);
	
    s32Ret |= EdidStdTiming(pstEdidInfo);
	EDID_SUCCESS_CHK(s32Ret);
	
    s32Ret |= EdidDetailTiming(pstCap,pstStatus,pFstInfo->au8DetailedTiming,EDID_DTD_SIZE);
	EDID_SUCCESS_CHK(s32Ret);
	
	s32Ret |= EdidExtNum(pstEdidInfo);
	EDID_SUCCESS_CHK(s32Ret);

	EDID_INFO("EdidFirstBlkParse return 0x%x!\n",s32Ret);
    return s32Ret;
}


static HI_S32 EdidAudioDB(HDMI_SINK_CAPABILITY_S *pstCap,
									HDMI_EDID_STATUS_S *pstStatus,
									HI_U8 *pau8Data, 
									HI_U8 u8Len)
{
	HI_U8 						i = 0;
	HI_U8 						u8FrmCode = 0; 
	HI_U8 						u8Byte = 0;
	HI_U8 						u8Count = 0;
	HI_U8 						u8CurNum = 0;
    HDMI_EDID_AUDIO_INFO_S	 	*pstAudioInfo = HI_NULL;

	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);
	pstAudioInfo = pstCap->stAudioInfo;
	
    /*Each Short Audio Descriptor is 3-bytes long*/
    for(i = 0; i < (u8Len / 3); i++)
    {
    	u8Count = 0;
        u8CurNum = pstCap->u32AudioInfoNum;
        

        if(u8CurNum >= HDMI_EDID_MAX_AUDIO_CAP_COUNT)
        {
        	pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_AUDIO_CNT_OVER;
            EDID_WARN("ADB Audio Capability Count over max-num:%d \n",HDMI_EDID_MAX_AUDIO_CAP_COUNT);
            break;
        }
		/* Byte-0 */
		u8Byte = pau8Data[i*3];
		HDMI_DBG("ADB [u8Byte0]:%02x\n",u8Byte);
		
        u8FrmCode = (u8Byte & EDID_AUDIO_FORMAT_MASK) >> 3;
        pstAudioInfo[u8CurNum].u8AudChannel = (u8Byte & EDID_AUDIO_CHANNEL_MASK) + 1;
		pstAudioInfo[u8CurNum].enAudFmtCode = (HDMI_AUDIO_FORMAT_CODE_E)u8FrmCode;

		HDMI_DBG("ADB ====u8CurNum====:%d\n",u8CurNum);
		HDMI_DBG("ADB u8FrmCode:%d\n",u8FrmCode);
		HDMI_DBG("ADB u8AudChannel:%d\n",pstAudioInfo[u8CurNum].u8AudChannel);
		HDMI_DBG("ADB enAudFmtCode:%d\n",pstAudioInfo[u8CurNum].enAudFmtCode);

		
        //if(u8FrmCode > HDMI_AUDIO_FORMAT_CODE_RESERVED 
		//&& u8FrmCode < HDMI_AUDIO_FORMAT_CODE_BUTT)
		if(u8FrmCode > HDMI_AUDIO_CODING_TYPE_STREAM 
		&& u8FrmCode < HDMI_AUDIO_CODING_TYPE_BUTT)
		{
            pstCap->u32AudioInfoNum++;
        }
		else
		{
			pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_AUDIO_FMT_INVALID;
			EDID_WARN("ADB CEA-861-F not support Audio Frm:%d \n",u8FrmCode);
			break;
		}

		/* Byte-1 */
        u8Byte = pau8Data[i*3 + 1];
		HDMI_DBG("ADB [u8Byte1]:%02x\n",u8Byte);
        if( u8Byte & BIT0_MASK )
        {
            pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count] = HDMI_SAMPLE_RATE_32K;
			HDMI_DBG("ADB num[%d].enSupportSampleRate:%d\n",u8Count,
				pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count]);
            u8Count++;
        }
        if( u8Byte & BIT1_MASK )
        {
            pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count] = HDMI_SAMPLE_RATE_44K;
			HDMI_DBG("ADB num[%d].enSupportSampleRate:%d\n",u8Count,
				pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count]);
            u8Count++;
        }
        if( u8Byte & BIT2_MASK )
        {
            pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count] = HDMI_SAMPLE_RATE_48K;
			HDMI_DBG("ADB num[%d].enSupportSampleRate:%d\n",u8Count,
				pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count]);
            u8Count++;
        }
        if( u8Byte & BIT3_MASK )
        {
            pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count] = HDMI_SAMPLE_RATE_88K;
			HDMI_DBG("ADB num[%d].enSupportSampleRate:%d\n",u8Count,
				pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count]);
            u8Count++;
        }
        if( u8Byte & BIT4_MASK )
        {
            pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count] = HDMI_SAMPLE_RATE_96K;
			HDMI_DBG("ADB num[%d].enSupportSampleRate:%d\n",u8Count,
				pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count]);
            u8Count++;
        }
		/*  CEA-861-F add for HDMI2.0 */
		if (u8FrmCode >= 1 && u8FrmCode <=14) 
		{
	        if( u8Byte & BIT5_MASK )
	        {
	            pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count] = HDMI_SAMPLE_RATE_176K;
				HDMI_DBG("ADB num[%d].enSupportSampleRate:%d\n",u8Count,
				pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count]);
	            u8Count++;
	        }
	        if( u8Byte & BIT6_MASK )
	        {
	            pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count] = HDMI_SAMPLE_RATE_192K;
				HDMI_DBG("ADB num[%d].enSupportSampleRate:%d\n",u8Count,
				pstAudioInfo[u8CurNum].enSupportSampleRate[u8Count]);
	            u8Count++;
	        }
		}
		
        pstAudioInfo[u8CurNum].u32SupportSampleRateNum = u8Count;

		/* Byte-2 */
        if(1 == u8FrmCode)
        {
            u8Count = 0;
            u8Byte = pau8Data[i*3 + 2];
			HDMI_DBG("ADB [u8Byte2]:%02x\n",u8Byte);


            if(u8Byte & BIT0_MASK)
            {
                pstAudioInfo[u8CurNum].enSupportBitDepth[u8Count] = HDMI_AUDIO_BIT_DEPTH_16;
				HDMI_DBG("ADB Num[%d].enSupportBitDepth:%d\n",u8Count,
					pstAudioInfo[u8CurNum].enSupportBitDepth[u8Count]);
                u8Count++;
            }
            if(u8Byte & BIT1_MASK)
            {
                pstAudioInfo[u8CurNum].enSupportBitDepth[u8Count] = HDMI_AUDIO_BIT_DEPTH_20;
				HDMI_DBG("ADB Num[%d].enSupportBitDepth:%d\n",u8Count,
					pstAudioInfo[u8CurNum].enSupportBitDepth[u8Count]);
                u8Count++;
				
            }
            if(u8Byte & BIT2_MASK)
            {
                pstAudioInfo[u8CurNum].enSupportBitDepth[u8Count] = HDMI_AUDIO_BIT_DEPTH_24;
				HDMI_DBG("ADB Num[%d].enSupportBitDepth:%d\n",u8Count,
					pstAudioInfo[u8CurNum].enSupportBitDepth[u8Count]);
                u8Count++;
            }

            pstAudioInfo[u8CurNum].u32SupportBitDepthNum = u8Count;
        }
        else if ((u8FrmCode > 1) && (u8FrmCode < 9))
        {
            pstAudioInfo[u8CurNum].u32MaxBitRate = pau8Data[i*3 + 2] / EDID_AUDIO_BITRATE_FACTOR;
        }
		/* the following part is CEA-861-F for HDMI2.0,we don't use. */
        else if((u8FrmCode >= 9) && (u8FrmCode < 15))
        {
			HDMI_DBG("ADB Profile:0x%x\n",pau8Data[i*3 + 2]);
        }
		else if (u8FrmCode == 15)
		{
			u8Byte	= pau8Data[i*3 + 2];
			HDMI_DBG("ADB ExtTypeCode:0x%x\n",(u8Byte & EDID_AUDIO_EXT_TYPE_CODE) >> 3);
			HDMI_DBG("ADB MPEG Surround (MPS)(BIT[0]):%d\n",u8Byte & BIT0_MASK);
			HDMI_DBG("ADB 960_TL(BIT[1]):0x%x\n",(u8Byte & BIT1_MASK)>>1);
			HDMI_DBG("ADB 1024_TL(BIT[2]):0x%x\n",(u8Byte & BIT2_MASK)>>2);
		}
		else
		{
			/* reserved */
		}
    }

    return HI_SUCCESS;
}

static HI_S32 EdidVideoDB(HDMI_SINK_CAPABILITY_S *pstCap,
								HDMI_EDID_STATUS_S *pstStatus,
								HI_U8 *pau8Data, 
								HI_U8 u8Len)
{
    HI_U8 	i = 0;
	HI_U32 	u32InplicitNative = EDID_VIC_INVALID_ZERO;
	HI_U32	u32VicAll = EDID_VIC_INVALID_ZERO;
    HI_U32 	u32VicLower = EDID_VIC_INVALID_ZERO;

    EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

    for(i = 0; i < u8Len; i++)
    {	
    	u32VicAll = pau8Data[i];
        u32VicLower = pau8Data[i] & EDID_VIC_LOWER7_MASK;

		/*avoid full over*/
		if ( pstCap->u32SupportVICNum >= HDMI_EDID_MAX_VIC_COUNT )
		{
			pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_VIC_CNT_OVER;
			EDID_WARN("VDB vic count over:%d !\n",HDMI_EDID_MAX_VIC_COUNT);
			break;
		}

		/*avoid invalid vic*/
		if ( u32VicAll == EDID_VIC_INVALID_ZERO ) 
		{
			pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_VIC_INVALID;
			EDID_WARN("VDB the %d u32Vic: %d ! \n",i,u32VicAll);
			continue;
		}

		/*explicit native*/
		if ( (EDID_VIC_INVALID_ZERO == pstCap->u32NativeFormat) && 
			 (u32VicAll & BIT7_MASK) && (u32VicLower <= EDID_VIC_NATIVE_MAX) )
		{
			pstCap->u32NativeFormat = u32VicLower;
	    }

		/* set the first valid VIC as implicit native  */
		if ( EDID_VIC_INVALID_ZERO == u32InplicitNative)
		{
			u32InplicitNative = u32VicAll;
		}
		
		if ((u32VicAll & BIT7_MASK) && (u32VicLower <= EDID_VIC_NATIVE_MAX))
		{
			pstCap->au32SupportFormat[pstCap->u32SupportVICNum] = u32VicLower ; 
		}
		else
		{
			pstCap->au32SupportFormat[pstCap->u32SupportVICNum] = u32VicAll;
		}
		HDMI_DBG("VDB Num[%d] u32VicAll :%d(0x%02x)\n",i,u32VicAll,u32VicAll);
		HDMI_DBG("VDB Num[%d] u32RealVic:%d\n",i,pstCap->au32SupportFormat[pstCap->u32SupportVICNum]);
		
		pstCap->u32SupportVICNum++;

	}

	if (EDID_VIC_INVALID_ZERO == pstCap->u32NativeFormat)
	{
		pstCap->u32NativeFormat = u32InplicitNative;
	}
	
	HDMI_DBG("VDB Native VIC:%d\n",pstCap->u32NativeFormat);

	EDID_INFO("HI_SUCCESS\n");
    return HI_SUCCESS;
}

static HI_S32 EdidHFVSDB(HDMI_SINK_CAPABILITY_S *pstCap,
									HDMI_EDID_STATUS_S *pstStatus,
									HI_U8 *pau8Data, 
									HI_U8 u8Len )
{
	
	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

	if (u8Len < 7 || u8Len > 31)
	{
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_BLOCKLEN_INVALID;
		EDID_WARN("HFVSDB u8Len:%d\n",u8Len);
		return HI_SUCCESS;
	}

	/* Byte-3 ,Version*/
	if (pau8Data[3] != EDID_HFVSDB_VERSION)
	{
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_HFVSDB_INVALID;
		EDID_WARN("HFVSDB Version:%d\n",pau8Data[3]);
	}
	HDMI_DBG("HFVSDB [byte-3]:0x%02x\n",pau8Data[3]);
	
	/* Byte-4 */
	pstCap->u32MaxTMDSClock = pau8Data[4];
	HDMI_DBG("VSDB [byte-4]:0x%02x\n",pstCap->u32MaxTMDSClock);
	pstCap->u32MaxTMDSClock *= EDID_TMDS_FACTOR;
	pstCap->bSupportHdmi_2_0 = ( pstCap->u32MaxTMDSClock > EDID_MAX_HDMI14_TMDS_RATE ) ? HI_TRUE : HI_FALSE ;
	HDMI_DBG("HFVSDB Max TMDS Colock:%d\n", pstCap->u32MaxTMDSClock);

	/* Byte-5 */
	HDMI_DBG("HFVSDB [byte-5]:0x%02x\n",pau8Data[5]);
	pstCap->bSupport3dOsdDisparity 			= (pau8Data[5] & BIT0_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->bSupport3dDualView	 			= (pau8Data[5] & BIT1_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->bSupport3dIndependentView 		= (pau8Data[5] & BIT2_MASK) ? HI_TRUE : HI_FALSE ;
#if 1
	pstCap->bSupportLte340McscScrameble 	= (pau8Data[5] & BIT3_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->bSupportRRCapable 				= (pau8Data[5] & BIT6_MASK) ? HI_TRUE : HI_FALSE ;
#endif
	pstCap->bSupportSCDC					= (pau8Data[5] & BIT7_MASK) ? HI_TRUE : HI_FALSE ;

	/* Byte-6 */
	HDMI_DBG("HFVSDB [byte-6]:0x%02x\n",pau8Data[6]);
	pstCap->stDeepColorY420.bDeepColor30Bit = (pau8Data[6] & BIT0_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stDeepColorY420.bDeepColor36Bit = (pau8Data[6] & BIT1_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stDeepColorY420.bDeepColor48Bit = (pau8Data[6] & BIT2_MASK) ? HI_TRUE : HI_FALSE ;

	return HI_SUCCESS;
}


static HI_S32 EdidVSDB(HDMI_SINK_CAPABILITY_S *pstCap,
								HDMI_EDID_STATUS_S *pstStatus,
								HI_U8 *pau8Data, 
								HI_U8 u8Len )
{
    HI_U8 i = 0;
	HI_U8 u8Byte = 0;
    HI_U8 u8Offset = 0;
	HI_U8 u8HdmiVicLen = 0;
	HI_U8 u8Hdmi3DLen = 0;
	HI_BOOL b3DPresent = 0;
	HI_U8 u83DMultiPresent = 0;
	HI_U8 u83DStructure = 0;

	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

	/*Byte-0~2, IEEE have been parsed outside the func */

	/*Byte-3~4 ,CEC Addr */
    if( u8Len < 5 )
    {
        EDID_WARN("VSDB len:%d\n",u8Len);
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_BLOCKLEN_INVALID;
        return HI_SUCCESS;
    }

    pstCap->stCECAddr.u8PhyAddrA = (pau8Data[3] & EDID_UPPER_NIBBLE_MASK) >> 4 ;
    pstCap->stCECAddr.u8PhyAddrB = (pau8Data[3] & EDID_LOWER_NIBBLE_MASK) ;
    pstCap->stCECAddr.u8PhyAddrC = (pau8Data[4] & EDID_UPPER_NIBBLE_MASK) >> 4 ;
    pstCap->stCECAddr.u8PhyAddrD = (pau8Data[4] & EDID_LOWER_NIBBLE_MASK) ;

    if (   (pstCap->stCECAddr.u8PhyAddrA != EDID_CEC_INVALID_ADDR )
		&& (pstCap->stCECAddr.u8PhyAddrB != EDID_CEC_INVALID_ADDR )
		&& (pstCap->stCECAddr.u8PhyAddrC != EDID_CEC_INVALID_ADDR )
		&& (pstCap->stCECAddr.u8PhyAddrD != EDID_CEC_INVALID_ADDR )   )
    {
        pstCap->stCECAddr.bPhyAddrValid = HI_TRUE ;
    }
    else
    {
        pstCap->stCECAddr.bPhyAddrValid = HI_FALSE ;
    }

	/* Byte-5 */
    if(u8Len < 6)
    {
    	EDID_INFO("VSDB len:%d\n",u8Len);
        return HI_SUCCESS;
    }
	u8Byte = pau8Data[5];
	HDMI_DBG("VSDB [byte-5]:0x%02x\n",u8Byte);
    pstCap->bSupportDVIDual    			= (u8Byte & BIT0_MASK) ? HI_TRUE : HI_FALSE ;
    pstCap->stDeepColor.bDeepColorY444  = (u8Byte & BIT3_MASK) ? HI_TRUE : HI_FALSE ;
    pstCap->stDeepColor.bDeepColor30Bit = (u8Byte & BIT4_MASK) ? HI_TRUE : HI_FALSE ;
    pstCap->stDeepColor.bDeepColor36Bit = (u8Byte & BIT5_MASK) ? HI_TRUE : HI_FALSE ;
    pstCap->stDeepColor.bDeepColor48Bit = (u8Byte & BIT6_MASK) ? HI_TRUE : HI_FALSE ;
    pstCap->bSupportsAI 				= (u8Byte & BIT7_MASK) ? HI_TRUE : HI_FALSE ;

	/* Byte-6 */
    if(u8Len < 7)
    {
    	EDID_INFO("VSDB len:%d\n",u8Len);
        return HI_SUCCESS;
    }

	pstCap->u32MaxTMDSClock = pau8Data[6];
	HDMI_DBG("VSDB [byte-6]:0x%02x\n",pstCap->u32MaxTMDSClock);
	pstCap->u32MaxTMDSClock *= EDID_TMDS_FACTOR;
	pstCap->bSupportHdmi_2_0 = ( pstCap->u32MaxTMDSClock >= EDID_MAX_HDMI14_TMDS_RATE ) ? HI_TRUE : HI_FALSE ;
	HDMI_DBG("VSDB Max TMDS Colock:%d\n", pstCap->u32MaxTMDSClock);

	/* Byte-7 */
    if(u8Len < 8)
    {
    	EDID_INFO("VSDB len:%d\n",u8Len);
        return HI_SUCCESS;
    }
#if 0
    /* NOTE:HDMI_Video_present is used as 3d support tag now. 
    		we don't use 3D_present to judge 3dsupport */
    pstCap->st3DInfo.bSupport3D = (pau8Data[7] & BIT5_MASK) ? HI_TRUE : HI_FALSE ;
#endif
	u8Byte = pau8Data[7];
	HDMI_DBG("VSDB [byte-7]:0x%02x\n",u8Byte);

	pstCap->bHDMIVideoPresent 		= (u8Byte & BIT5_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->bILatencyFieldsPresent	= (u8Byte & BIT6_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->bLatencyFieldsPresent	= (u8Byte & BIT7_MASK) ? HI_TRUE : HI_FALSE ;
	HDMI_DBG("VSDB HDMI_Video_present:%d\n",pstCap->bHDMIVideoPresent);
	HDMI_DBG("VSDB bI_Latency_Fields_Present:%d\n",pstCap->bILatencyFieldsPresent);
	HDMI_DBG("VSDB bLatency_Fields_Present:%d\n",pstCap->bLatencyFieldsPresent);
    HDMI_DBG("VSDB CNC:0x%02x\n",(u8Byte & EDID_LOWER_NIBBLE_MASK));

	/* Byte8~11,A&V Latecy */
	
	if(u8Len < 12)
	{
		EDID_INFO("VSDB len:%d\n",u8Len);
		return HI_SUCCESS;
	}

	pstCap->u8VideoLatency 			 = pau8Data[8];
	pstCap->u8AudioLatency 			 = pau8Data[9];
	pstCap->u8InterlacedVideoLatency = pau8Data[10];
	pstCap->u8InterlacedAudioLatency = pau8Data[11];	
    HDMI_DBG("VSDB u8Video_Latency[u8Byte-8]:%d(0x%02x)\n",pstCap->u8VideoLatency,pstCap->u8VideoLatency);
    HDMI_DBG("VSDB u8Audio_Latency[u8Byte-9]:%d(0x%02x)\n",pstCap->u8AudioLatency,pstCap->u8AudioLatency);
    HDMI_DBG("VSDB Interlaced_Video_Latency[u8Byte-10]:%d(0x%02x)\n",pstCap->u8InterlacedVideoLatency,pstCap->u8InterlacedVideoLatency);
	HDMI_DBG("VSDB Interlaced_Audio_Latency[u8Byte-11]:%d(0x%02x)\n",pstCap->u8InterlacedAudioLatency,pstCap->u8InterlacedAudioLatency);

	/* Byte-12  */
	if(u8Len < 13)
    {
    	EDID_INFO("VSDB len:%d\n",u8Len);
        return HI_SUCCESS;
    }
	
	u8Byte = pau8Data[12];
	HDMI_DBG("VSDB [u8Byte-12]:0x%02x\n",u8Byte);
	if (pstCap->bHDMIVideoPresent)
	{
	    HDMI_DBG("VSDB u8ImagSize:%d \n",((u8Byte & (BIT4_MASK | BIT3_MASK) ) >> 3));
	    u83DMultiPresent = (u8Byte & (BIT6_MASK | BIT5_MASK)) >> 5;
		b3DPresent = (u8Byte & BIT7_MASK) ? HI_TRUE : HI_FALSE ;
		pstCap->st3DInfo.bSupport3D = (u8Byte & BIT7_MASK) ? HI_TRUE : HI_FALSE ;
	    HDMI_DBG("VSDB 3D present:%d \n",b3DPresent);
		HDMI_DBG("VSDB u83DMultiPresent:%d \n",u83DMultiPresent);
	}
	
	if(u8Len < 14)
    {
    	EDID_INFO("VSDB len:%d\n",u8Len);
        return HI_SUCCESS;
    }

 	/* Byte-13 */
	u8Byte = pau8Data[13];
	HDMI_DBG("VSDB [u8Byte-13]:0x%02x\n",u8Byte);
    u8Hdmi3DLen = (u8Byte & EDID_HDMI_3D_LEN_MASK );
    u8HdmiVicLen = (u8Byte & EDID_HDMI_VIC_LEN_MASK ) >> 5;
	HDMI_DBG("VSDB u8Hdmi3DLen:%d \n",u8Hdmi3DLen);
	HDMI_DBG("VSDB u8HdmiVicLen:%d \n",u8HdmiVicLen);
		
	/* Byte-14 & following,VIC*/
	u8Offset = 14;
    if ((u8HdmiVicLen > 0) && (u8Len >= (u8HdmiVicLen + u8Offset)))
    {
        for(i = 0; i < u8HdmiVicLen; i++)
        {
            u8Byte = pau8Data[u8Offset++];
			HDMI_DBG("VSDB [u8Byte-%d]:0x%02x\n",u8Offset-1,u8Byte);

			if (pstCap->u32SupportVICNum < HDMI_EDID_MAX_VIC_COUNT)
			{
				pstCap->au32SupportFormat[pstCap->u32SupportVICNum++] = HDMI_EDID_MAX_REAL_VIC + u8Byte;
			}
			else
			{	
				pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_VIC_CNT_OVER;
				EDID_WARN("VSDB VICNum over :%d\n",HDMI_EDID_MAX_VIC_COUNT);
				break;
			}
        }
    }

	/* Byte-following, 3D */
    if((u8Hdmi3DLen > 0) && (u8Len >= (u8Hdmi3DLen + u8Offset)) && b3DPresent) 
    {
        if ((u83DMultiPresent == EDID_3DMULTI_PRESENT_LOWER8) 
			|| (u83DMultiPresent == EDID_3DMULTI_PRESENT_UPPER8))
        {
	        /* 3d structure_All_15...8 reseved */
	        u8Byte = pau8Data[u8Offset++] & 0xFF;
			HDMI_DBG("VSDB 3D_Structure_ALL_158[u8Byte-%d]:0x%02x\n",u8Offset-1,u8Byte);
	        
	        pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_SIDE_BY_SIDE_HALF] = (u8Byte & BIT0_MASK) ? HI_TRUE : HI_FALSE;           

	        u8Byte = pau8Data[u8Offset++] & 0xFF;
			HDMI_DBG("VSDB 3D_Structure_ALL_70[u8Byte-%d]:0x%02x\n",u8Offset-1,u8Byte);

			pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_FRAME_PACKETING]	= (u8Byte & BIT0_MASK) ? HI_TRUE : HI_FALSE;           
			pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_FIELD_ALTERNATIVE]	= (u8Byte & BIT1_MASK) ? HI_TRUE : HI_FALSE;         
			pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_LINE_ALTERNATIVE] 	= (u8Byte & BIT2_MASK) ? HI_TRUE : HI_FALSE;                  
			pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_SIDE_BY_SIDE_FULL]	= (u8Byte & BIT3_MASK) ? HI_TRUE : HI_FALSE;          
			pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_L_DEPTH] 			= (u8Byte & BIT4_MASK) ? HI_TRUE : HI_FALSE;
			pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_L_DEPTH_GRAPHICS_GRAPHICS_DEPTH] \
																			= (u8Byte & BIT5_MASK) ? HI_TRUE : HI_FALSE;
			pstCap->st3DInfo.bSupport3DType[HDMI_EDID_3D_TOP_AND_BOTTOM] 	= (u8Byte & BIT6_MASK) ? HI_TRUE : HI_FALSE;          
        }

		/* we don't use */
        if(u83DMultiPresent == EDID_3DMULTI_PRESENT_UPPER8)
        {
            /* 3D_MASK_0...15 reseved */
            u8Byte = pau8Data[u8Offset++] & 0xFF;
            HDMI_DBG("VSDB 3D_MASK_158 (u8Byte-%d):0x%x\n",u8Offset-1,u8Byte);
            u8Byte = pau8Data[u8Offset++] & 0xFF;
            HDMI_DBG("VSDB 3D_MASK_70 (u8Byte-%d):0x%x\n",u8Offset-1,u8Byte);
        }

    }

	/* we don't use */
    for(i = u8Offset ; i < u8Len; i++)
    {
    	u8Byte = pau8Data[u8Offset];
		HDMI_DBG("VSDB [u8Byte-%d]:0x%02x\n",u8Offset,u8Byte);
        u83DStructure = (u8Byte & EDID_UPPER_NIBBLE_MASK)>>4;
        HDMI_DBG("VSDB 2D_VIC_order:0x%02x\n", u83DStructure);
        u83DStructure = u8Byte & EDID_LOWER_NIBBLE_MASK;
		HDMI_DBG("VSDB 3D_Structure:0x%02x\n", u83DStructure);
        if(u83DStructure >= 0x08)
        {
            HDMI_DBG("VSDB 3D_Detailed > 0x08 :0x%02x\n", u83DStructure);
        }

        u8Offset++;
        HDMI_DBG("VSDB 3D_Detailed :0x%02x\n", u83DStructure);
    }
    
    return HI_SUCCESS;
}

static HI_S32 EdidSpeakerDB(HDMI_SINK_CAPABILITY_S *pstCap,
									HDMI_EDID_STATUS_S *pstStatus,
									HI_U8 *pau8Data, 
									HI_U8 u8Len)
{
	HI_U8 i = 0;
    HI_U8 u8Byte = 0;

	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

	if (u8Len < 2)
	{
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_BLOCKLEN_INVALID;
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_SPKDB_INVALID;
		EDID_WARN("SPKDB u8Len:%d\n",u8Len);
		return HI_SUCCESS;
	}

	/* Byte-0 */
	u8Byte = pau8Data[0];
    HDMI_DBG("SPKDB Byte-0:0x%02x\n",u8Byte);

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_FL_FR]   = (u8Byte & BIT0_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_LFE] 	  = (u8Byte & BIT1_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_FC] 	  = (u8Byte & BIT2_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_RL_RR]   = (u8Byte & BIT3_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_RC] 	  = (u8Byte & BIT4_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_FLC_FRC] = (u8Byte & BIT5_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_RLC_RRC] = (u8Byte & BIT6_MASK) ? HI_TRUE : HI_FALSE ;
	
	/* NOTE: the following is CEA-861-F add for HDMI2.0 */
	pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_FLW_FRW] = (u8Byte & BIT7_MASK) ? HI_TRUE : HI_FALSE ;

	
	/* Byte-1 */
    u8Byte = pau8Data[1];
    HDMI_DBG("SPKDB Byte-1:0x%02x\n",u8Byte);

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_FLH_FRH] = (u8Byte & BIT0_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_TC] 	  = (u8Byte & BIT1_MASK) ? HI_TRUE : HI_FALSE ;

    pstCap->bSupportAudioSpeaker[HDMI_EDID_AUDIO_SPEAKER_FCH] 	  = (u8Byte & BIT2_MASK) ? HI_TRUE : HI_FALSE ;

	for(i = 0; i <= HDMI_EDID_AUDIO_SPEAKER_FCH; i++)
	{
		if(pstCap->bSupportAudioSpeaker[i])
		{
			HDMI_DBG("SPKDB bSupportAudioSpeaker[%d]:HI_TRUE\n",i);
		}
	}
	
	/* Byte-2 , reversed */
	HDMI_DBG("SPKDB Byte-2:0x%02x\n",pau8Data[2]);

    return HI_SUCCESS;
}

static HI_S32 EdidColormetryDB(HDMI_SINK_CAPABILITY_S *pstCap,
										HDMI_EDID_STATUS_S *pstStatus,
										HI_U8 *pau8Data, 
										HI_U8 u8Len)
{
	HI_U8 i = 0;
	HI_U8 u8Byte = 0;
	HI_BOOL *pBool = HI_NULL;
	
	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

	/*1 ext tag code + 2 bytes payload*/
	if (u8Len < 3)
	{
		pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_BLOCKLEN_INVALID;
		EDID_ERR("ColormetryDB Len!\n");
		return HI_SUCCESS;
	}

	/* Byte-0 */
	u8Byte = pau8Data[0];
	HDMI_DBG("ColormetryDB u8Byte0:0x%02x\n",u8Byte);
	pstCap->stColorMetry.bxvYCC601 		= (u8Byte & BIT0_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stColorMetry.bxvYCC709 		= (u8Byte & BIT1_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stColorMetry.bsYCC601 		= (u8Byte & BIT2_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stColorMetry.bAdobleYCC601 	= (u8Byte & BIT3_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stColorMetry.bAdobleRGB 	= (u8Byte & BIT4_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stColorMetry.bBT2020cYCC 	= (u8Byte & BIT5_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stColorMetry.bBT2020YCC 	= (u8Byte & BIT6_MASK) ? HI_TRUE : HI_FALSE ;
	pstCap->stColorMetry.bBT2020RGB 	= (u8Byte & BIT7_MASK) ? HI_TRUE : HI_FALSE ;

	for(i = 0,pBool = (HI_BOOL *)&pstCap->stColorMetry;
		i<8 && pBool;
		i++,pBool++)
	{
		HDMI_DBG("ColormetryDB stColorMetry[%d]:0x%02x\n",i,*pBool);
	}
	
	/* Byte-1 */
	pstCap->u8MDBit = pau8Data[1];
	HDMI_DBG("ColormetryDB u8Byte1:0x%02x\n",pstCap->u8MDBit);
	HDMI_DBG("ColormetryDB gamut-related metadata[BIT0~3]:0x%x!\n",(pstCap->u8MDBit & EDID_LOWER_NIBBLE_MASK));

	return HI_SUCCESS;
	
}

static HI_S32 EdidY420VideoDB(HDMI_SINK_CAPABILITY_S *pstCap,
										HDMI_EDID_STATUS_S *pstStatus, 
										HI_U8 *pau8Data, 
										HI_U8 u8Len)
{
	HI_U8			i = 0;
	HI_U32			u32Vic = 0;

	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);

	for(i = 0; i < u8Len-1; i++)
    {
        u32Vic = pau8Data[i] ;

		if (pstCap->u32OnlySupportY420VICNum > HDMI_EDID_MAX_VIC_COUNT)
		{
			pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_Y420VDB_VIC_OVER;
			EDID_WARN("Y420VideoDB vic count over:%d !\n",HDMI_EDID_MAX_VIC_COUNT);
			break;
		}
		
		if (   ( u32Vic == EDID_VIC_INVALID_ZERO) 
			|| ( (u32Vic >= EDID_VIC_INVALID_LOW ) && (u32Vic <= EDID_VIC_INVALID_HIGH) )   )
		{
			pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_VIC_INVALID;
			EDID_WARN("Y420VideoDB the %d is invalid Vic: %d ! \n",i,u32Vic);
			continue;
		}
	
		pstCap->au32OnlySupportY420Format[pstCap->u32OnlySupportY420VICNum++] = u32Vic ; 
		HDMI_DBG("Y420VideoDB u32Vic:%u(0x%02x)\n",u32Vic,u32Vic);
	}

	return HI_SUCCESS;
}

static HI_S32 EdidY420CapMapDB(HDMI_SINK_CAPABILITY_S *pstCap,
											HI_U8 *pau8Data, 
											HI_U8 u8Len)
{
	HI_U32			i = 0;
	HI_U32			u32Loop = 0;

	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	
	/*pstCap->au32SupportY420Format  is  SVDs parsed from Video Data Block(s) in order.
		When the Length field is  1, indicates that all the SVDs  support YUV420 .  */
	if (1 == u8Len)
	{
		for (i = 0 ; i < pstCap->u32SupportVICNum ; i++)
		{
			if (pstCap->au32SupportFormat[i] <= HDMI_EDID_MAX_REAL_VIC)
			{
				pstCap->au32SupportY420Format[pstCap->u32SupportY420VICNum++] \
											= pstCap->au32SupportFormat[i];
			}
		}

	}
	/* Or,for SVDs which support YUV420, the corresponding bit  shall be set to 1 here. */
	else  
	{
		u32Loop = u8Len*8;
		for(i = 0; (i < u32Loop) && (i < pstCap->u32SupportVICNum); i++)
	    {
	    	/* BYTE n(i/8),BIT x(i%8) */
	    	if (pau8Data[i/8] & (0x01<<(i%8)) )
	    	{
				pstCap->au32SupportY420Format[pstCap->u32SupportY420VICNum++] \
												= pstCap->au32SupportFormat[i] ;
	    	}
		}
	}
	
	return HI_SUCCESS;
}


static HI_S32 EdidExtDataBlockParse(HDMI_SINK_CAPABILITY_S *pstCap,
														HDMI_EDID_STATUS_S *pstStatus,
														HI_U8 *pau8Data, 
														HI_U8 u8Len)
{
	HI_S32	u32Ret = HI_SUCCESS;
	HI_U8	u8ExtTagCode = 0;

	EDID_NULL_CHK(pstCap);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstStatus);
	
	u8ExtTagCode = pau8Data[0];
	pau8Data++;

	HDMI_DBG("EXT-BLK u8ExtTagCode=%d(0x%02x)\n",u8ExtTagCode,u8ExtTagCode);
	
    switch(u8ExtTagCode)
    {			
		case 	EDID_COLORIMETRY_DB:    		
				u32Ret = EdidColormetryDB(pstCap,pstStatus,pau8Data,u8Len);
				EDID_SUCCESS_CHK(u32Ret);
        		break;
				
		case	EDID_Y420_VIDEO_DB :
				u32Ret = EdidY420VideoDB(pstCap,pstStatus,pau8Data,u8Len);
				EDID_SUCCESS_CHK(u32Ret);
				break;
				
		case	EDID_Y420_CAPABILITY_MAP_DB :       
				u32Ret = EdidY420CapMapDB(pstCap,pau8Data,u8Len);
				EDID_SUCCESS_CHK(u32Ret);
				break;
 
		
		case	EDID_VIDEO_CAPABILITY_DB :   
    	case	EDID_VIDEO_FMT_PREFER_DB:  	    
		case	EDID_RESERVED_HDMI_AUDIO_DB:        	 
		case 	EDID_INFOFRAME_DB:
     	default:
				EDID_WARN("EXT-BLK un-parse Ext data block!Ext-Tag Code:%d!\n",u8ExtTagCode);
        		break;


    }
    return HI_SUCCESS;
}

static HI_S32 EdidExtentionBlockParse(HDMI_EDID_INFO_S *pstEdidInfo,
												HI_U8 u8BlkNum)
{
    HI_U8 						u8BlkOffset = 0;
	HI_U8 						u8DTDOffset = 0;
	HI_U8						u8BlkLen = 0;
	HI_U8						u8DBTagCode = 0;
    HI_S32 						s32Ret = HI_SUCCESS;
	HDMI_EDID_STATUS_S			*pstStatus = HI_NULL;
    HI_U8  						*pau8Data = HI_NULL;
    HDMI_SINK_CAPABILITY_S 		*pstCap = HI_NULL;
	HDMI_EDID_COLOR_SPACE_S 	*pstClrSpace= HI_NULL;

	EDID_NULL_CHK(pstEdidInfo);
	pstStatus 	= &pstEdidInfo->stStatus;
	pau8Data 	= pstEdidInfo->au8EdidRaw + (u8BlkNum*HDMI_EDID_BLOCK_SIZE);
	pstCap 		= &pstEdidInfo->stCapability;
	EDID_NULL_CHK(pau8Data);

    s32Ret = EdidBlockChkSum(pau8Data,pstStatus);
	EDID_SUCCESS_CHK(s32Ret);

	/* Byte-0 :TAG */
    if(pau8Data[0] != EDID_CEA_EXTVERSION3_TAG)
    {
    	pstStatus->enParseErr = EDID_PARSE_ERR_TAG_UNKNOWN;
        EDID_ERR("EXT_BLK CEA TAG:0x%02x\n",pau8Data[0]);
        return HI_FAILURE;
    }

	/* Byte-1 :REVISION */
    if (pau8Data[1] != EDID_CEA_EXTVERSION3_REVISION)
    {
    	pstStatus->enParseErr = EDID_PARSE_ERR_CEA_REVERION;
        EDID_ERR("EXT_BLK CEA861 REVISION: 0x%02x\n", pau8Data[1]);
        return HI_FAILURE;
    }

	/* Byte-2 :DTDs Offset */
    u8DTDOffset = pau8Data[2];
    HDMI_DBG("EXT_BLK u8DTDOffset:%d\n",u8DTDOffset);
    if(u8DTDOffset < 4)
    {
        EDID_INFO("EXT_BLK NO DTDs & reserved Data Blks :d=%d\n",u8DTDOffset);
        return HI_SUCCESS;
    }

	/* Byte-3 :Basic Audio,Color Space,DTDs Cnt */
	pstClrSpace = &pstCap->stColorSpace;
    pstClrSpace->bRGB444   = HI_TRUE;
    pstClrSpace->bYCbCr422 = (pau8Data[3] & BIT4_MASK ? HI_TRUE : HI_FALSE) ;
    pstClrSpace->bYCbCr444 = (pau8Data[3] & BIT5_MASK ? HI_TRUE : HI_FALSE) ;
	
    HDMI_DBG("EXT_BLK IT Underscan:0x%02x\n", ((pau8Data[3] & BIT7_MASK)>>7));
	HDMI_DBG("EXT_BLK basic audio:0x%02x\n",  (pau8Data[3] & BIT6_MASK)>>6);
	HDMI_DBG("EXT_BLK bYCbCr444:%d\n", pstClrSpace->bYCbCr444);
	HDMI_DBG("EXT_BLK bYCbCr422:%d\n", pstClrSpace->bYCbCr422);
	HDMI_DBG("EXT_BLK native DTDS:0x%02x\n",	pau8Data[3] & EDID_LOWER_NIBBLE_MASK);
	
	/* sevaral Data Block */
    for (u8BlkOffset = 4 ,pau8Data += u8BlkOffset,u8BlkLen=0 ; 
	     (u8BlkOffset < u8DTDOffset) && (pau8Data != HI_NULL); 
		 u8BlkOffset += u8BlkLen + 1)
    {
    	pau8Data += u8BlkLen;
        u8BlkLen = (*pau8Data) & EDID_DB_LEN_MASK;
		u8DBTagCode = ( (*pau8Data) & EDID_DB_TAG_CODE_MASK ) >> 5;
		HDMI_DBG("EXT_BLK u8BlkOffset:%d\n",u8BlkOffset);
		HDMI_DBG("EXT_BLK byte_contex:0x%02x\n",*pau8Data);
        HDMI_DBG("EXT_BLK u8DBTagCode:%d\n",u8DBTagCode);
		HDMI_DBG("EXT_BLK u8BlkLen:%d\n",u8BlkLen);
		
		pau8Data++;
		/* Tag Code */
        switch(u8DBTagCode)
        {
            case	EDID_AUDIO_DATA_BLOCK:
                	s32Ret = EdidAudioDB(pstCap,pstStatus,pau8Data,u8BlkLen);
					EDID_SUCCESS_CHK(s32Ret);
                	break;
				
            case 	EDID_VIDEO_DATA_BLOCK:
                	s32Ret = EdidVideoDB(pstCap,pstStatus,pau8Data,u8BlkLen);
					EDID_SUCCESS_CHK(s32Ret);
                	break;
				
            case 	EDID_VENDOR_DATA_BLOCK:				
					if (   u8BlkLen >= 3
						&& (pau8Data[0] == EDID_IEEE_VSDB_1ST) 
						&& (pau8Data[1] == EDID_IEEE_VSDB_2ND)
						&& (pau8Data[2] == EDID_IEEE_VSDB_3RD)   )
					{
						/* IEEE  code indicate whether it is HDMI1.4b or HDMI2.0*/
						pstCap->bSupportHdmi = HI_TRUE;
	                	s32Ret = EdidVSDB(pstCap,pstStatus,pau8Data,u8BlkLen);
						EDID_SUCCESS_CHK(s32Ret);
					}
					else if (  u8BlkLen >= 3 
							&& (pau8Data[0] == EDID_IEEE_HFVSDB_1ST) 
							&& (pau8Data[1] == EDID_IEEE_HFVSDB_2ND) 
							&& (pau8Data[2] == EDID_IEEE_HFVSDB_3RD)  )
					{
						s32Ret = EdidHFVSDB(pstCap,pstStatus,pau8Data,u8BlkLen);
						EDID_SUCCESS_CHK(s32Ret);
					}
					else
					{
						if(u8BlkLen >= 3 )
						{
							pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_VSDB_INVALID;
						 	EDID_WARN("(HF)VSDB IEEE:0x%02x, 0x%02x, 0x%02x\n", 
							pau8Data[0], pau8Data[1], pau8Data[2]);
						}
						else
						{
							pstStatus->u32ParseWarn |= 1<<EDID_PARSE_WARN_BLOCKLEN_INVALID;
							EDID_WARN("(HF)VSDB  u8BlkLen:%d\n",u8BlkLen);
						}
					}
					break;
				
	        case 	EDID_SPEAKER_DATA_BLOCK:
	                s32Ret = EdidSpeakerDB(pstCap,pstStatus,pau8Data,u8BlkLen);
					EDID_SUCCESS_CHK(s32Ret);
	                break;
            

            case 	EDID_USE_EXT_DATA_BLOCK:
					/* Ext Tag Code */
					s32Ret = EdidExtDataBlockParse(pstCap,pstStatus,pau8Data,u8BlkLen);
					EDID_SUCCESS_CHK(s32Ret);
	                break;

			case 	EDID_VESA_DTC_DATA_BLOCK:
            default:
	                EDID_WARN("EXT_BLK resvered/un-parse block tag code!\n");
	                break;
        }

    }

	/* DTDs */
	while( ((HDMI_EDID_BLOCK_SIZE-1) - u8BlkOffset) > EDID_DTD_SIZE)
	{
		
		s32Ret = EdidDetailTiming(pstCap,pstStatus,pau8Data,EDID_DTD_SIZE);
		EDID_SUCCESS_CHK(s32Ret);
		u8BlkOffset += EDID_DTD_SIZE;
		pau8Data += EDID_DTD_SIZE;
	}
	
	EDID_INFO("HI_SUCCESS\n");
    return HI_SUCCESS;

}

static HI_S32 EdidRawParse(HDMI_EDID_INFO_S *pstEdidInfo)
{
	HI_U8					u8BlkNum = 0;
    HI_S32 					s32Ret = HI_SUCCESS;
	HDMI_EDID_STATUS_S		*pstStatus = HI_NULL;
    HI_U8  					*pau8Data = HI_NULL;
    HDMI_SINK_CAPABILITY_S 	*pstCap = HI_NULL;

	EDID_NULL_CHK(pstEdidInfo);
	pstStatus 	= &pstEdidInfo->stStatus;
	pau8Data 	= pstEdidInfo->au8EdidRaw;
	pstCap 		= &pstEdidInfo->stCapability;
	EDID_NULL_CHK(pstStatus);
	EDID_NULL_CHK(pau8Data);
	EDID_NULL_CHK(pstCap);
	
    s32Ret = EdidFirstBlkParse(pstEdidInfo);
	EDID_SUCCESS_CHK(s32Ret);

    for (u8BlkNum = 1; u8BlkNum <= pstCap->u8ExtBlockNum; u8BlkNum++)
    {
        s32Ret = EdidExtentionBlockParse(pstEdidInfo,u8BlkNum);
        EDID_SUCCESS_CHK(s32Ret);
    }

    return HI_SUCCESS;
}

/********************** public interface ********************************************/


HI_S32 DRV_HDMI_EdidUpdate(HDMI_EDID_INFO_S *pstEdidInfo,
									HDMI_EDID_UPDATE_MODE_E enMode)
{

	HI_S32 					s32Ret = HI_SUCCESS;
	HDMI_EDID_STATUS_S		*pstStatus = HI_NULL;
    HI_U8  					*pau8Data = HI_NULL;
    HDMI_SINK_CAPABILITY_S 	*pstCap = HI_NULL;
	HDMI_DEVICE_S			*pstHdmiDev = HI_NULL;
    HI_U8                   u8EdidRawData[HDMI_EDID_TOTAL_SIZE];
    
	EDID_NULL_CHK(pstEdidInfo);
	pstStatus 	= &pstEdidInfo->stStatus;
	pstCap 		= &pstEdidInfo->stCapability;
	pau8Data 	= pstEdidInfo->au8EdidRaw;
	EDID_NULL_CHK(pau8Data);

	HDMI_MEMSET(pstCap,0,sizeof(HDMI_SINK_CAPABILITY_S));
	HDMI_MEMSET(pau8Data,0,sizeof(pstEdidInfo->au8EdidRaw));

	pstStatus->bCapSink = HI_FALSE;
	pstStatus->bCapValid = HI_FALSE;
	pstStatus->bRawValid = HI_FALSE;
	pstStatus->u32RawLen = 0;
	pstStatus->enParseErr = EDID_PARSE_ERR_NONE;
	pstStatus->u32ParseWarn = EDID_PARSE_WARN_NONE;
	
	if ( HDMI_EDID_UPDATE_SINK == enMode )
	{
		pstHdmiDev = container_of(pstEdidInfo, typeof(*pstHdmiDev), stEdidInfo); 
		EDID_NULL_CHK(pstHdmiDev);
		EDID_NULL_CHK(pstHdmiDev->pstHdmiHal);
		EDID_NULL_CHK(pstHdmiDev->pstHdmiHal->HAL_HDMI_EdidRawDataRead);
		s32Ret = pstHdmiDev->pstHdmiHal->HAL_HDMI_EdidRawDataRead(pstHdmiDev->pstHdmiHal, HDMI_EDID_TOTAL_SIZE, u8EdidRawData);	
        pstStatus->bCapSink = HI_TRUE;
	}
#if HDMI_EDID_RAW_TEST
	else
	{
		s32Ret = DRV_HDMI_EdidTestRawRead(enMode,pau8Data,HDMI_EDID_TOTAL_SIZE);
		pstStatus->bCapSink = HI_FALSE;
	}
#endif

    if ( (0==s32Ret) || (s32Ret % HDMI_EDID_BLOCK_SIZE) )
    {
     	pstStatus->u32RawUpdateErrCnt++;
        EDID_ERR("\nRead EDID raw data fail(0x%x)!\n",s32Ret);
        return HI_FAILURE;
    }
	pstStatus->u32RawLen = (HI_U32)s32Ret;
	pstStatus->bRawValid = HI_TRUE;
#ifdef HDMI_HDCP_SUPPORT    
    pstHdmiDev->pstHdmiHal->HAL_HDMI_SinkHdcp22CapabilityGet(pstHdmiDev->pstHdmiHal, \
                            &pstEdidInfo->stCapability.bSupportHDCP_2_2);
#endif

    if (HDMI_EDID_UPDATE_SINK == enMode)
    {
        /* EDID is not change,so do not paser again*/
        if (!HDMI_MEMCMP(pau8Data, u8EdidRawData, HDMI_EDID_TOTAL_SIZE))
        {
            EDID_INFO("EDID is not change, so do not paser again\n");
            return HI_SUCCESS;
        }
        else
        {
            HDMI_MEMCPY(pau8Data, u8EdidRawData, HDMI_EDID_TOTAL_SIZE);
        }
    }

	s32Ret = EdidRawParse(pstEdidInfo);

    EDID_SUCCESS_CHK(s32Ret);
	pstStatus->bCapValid = HI_TRUE;
	return HI_SUCCESS;
}


HDMI_EDID_DATA_E DRV_HDMI_EdidCapabilityGet(HDMI_EDID_INFO_S *pstEdidInfo,
									 					const HDMI_SINK_CAPABILITY_S **pstCapDest)
{
	HDMI_EDID_STATUS_S		*pstStatus = HI_NULL;
	//HDMI_SINK_CAPABILITY_S	*pstCapSrc  = HI_NULL;

	EDID_NULL_CHK(pstCapDest);
	EDID_NULL_CHK(pstEdidInfo);

	pstStatus = &pstEdidInfo->stStatus;
	*pstCapDest = &pstEdidInfo->stCapability;
    
	if (pstStatus->bCapValid)
	{
		//HDMI_MEMCPY(pstCapDest,pstCapSrc,sizeof(HDMI_SINK_CAPABILITY_S));
		return (pstStatus->bCapValid) ? HDMI_EDID_DATA_VALIDSINK : HDMI_EDID_DATA_VALIDTEST;
	}
	else
	{
		pstStatus->u32CapGetErrCnt++;
		//HDMI_MEMCPY(pstCapDest,pstCapSrc,sizeof(HDMI_SINK_CAPABILITY_S));
		return HDMI_EDID_DATA_INVALID;
	}
    
}


HI_S32 DRV_HDMI_EdidRawGet(HDMI_EDID_INFO_S *pstEdidInfo,
											HI_U8 *pu8RawDataDest, 
											HI_U32 u32Len)
{
	HDMI_EDID_STATUS_S		*pstStatus = HI_NULL;
    HI_U8  					*pau8DataSrc = HI_NULL;
	HI_U32					u32TmpLen = 0;

	EDID_NULL_CHK(pu8RawDataDest);
	EDID_NULL_CHK(pstEdidInfo);

	pstStatus = &pstEdidInfo->stStatus;
	pau8DataSrc = pstEdidInfo->au8EdidRaw;
	EDID_NULL_CHK(pau8DataSrc);

	if ( (u32Len > HDMI_EDID_TOTAL_SIZE) || (HI_FALSE == pstStatus->bRawValid) )
	{
		pstStatus->u32RawGetErrCnt++;
		HDMI_ERR("u32Len:%d;bRawValid:%d\n",u32Len,pstStatus->bRawValid);
		return HI_FAILURE;
	}

	u32TmpLen = (u32Len <= pstStatus->u32RawLen) ? u32Len : pstStatus->u32RawLen;
	HDMI_MEMCPY(pu8RawDataDest,pau8DataSrc,u32TmpLen);

	return (HI_S32)u32TmpLen;
}


HI_S32 DRV_HDMI_EdidRawForceGet(HDMI_EDID_INFO_S *pstEdidInfo,
											HI_U8 *pu8RawData,
											HI_U32 u32Len)
{
	HI_S32 s32Ret = 0;

	s32Ret = DRV_HDMI_EdidUpdate(pstEdidInfo,HDMI_EDID_UPDATE_SINK)	;
	EDID_SUCCESS_CHK(s32Ret);
	
	s32Ret = DRV_HDMI_EdidRawGet(pstEdidInfo,pu8RawData ,u32Len) ;
	return s32Ret;
}


HI_S32 DRV_HDMI_EdidStatusGet(HDMI_EDID_INFO_S *pstEdidInfo,
											HDMI_EDID_STATUS_S *pstStatus)
{
	
	EDID_NULL_CHK(pstEdidInfo);
	EDID_NULL_CHK(pstStatus);
	
	HDMI_MEMCPY(pstStatus,&pstEdidInfo->stStatus,sizeof(HDMI_EDID_STATUS_S));

	return HI_SUCCESS;	
}

