/**
 * ddr_training_ctl.c
 *
 * Copyright (c) 2009-2014, HiSilicon Technologies Co., Ltd.
 * All rights reserved.
 *
 * DDR training control.
 */
#include <ddr_interface.h>
#include "ddr_training_impl.h"

#ifdef DDR_SW_TRAINING_FUNC_PUBLIC
/**
 * ddr_sw_training_func
 * @ddrtr_result
 *
 *
 */
int ddr_sw_training_func(void *ddrtr_result)
{
	int result = 0;

	/* clear stat register */
	REG_WRITE(0x0, DDR_REG_BASE_SYSCTRL + SYSCTRL_DDR_TRAINING_STAT);

	result  = ddr_wl_if();
	result += ddr_dataeye_training_if(ddrtr_result);

#ifdef DDR_HW_TRAINING_CONFIG
	if (result && !ddr_training_check_bypass(DDR_BYPASS_HW_MASK)) {
		result  = ddr_hw_training_if();
		result += ddr_dataeye_training_if(ddrtr_result);
	}
#endif

#ifdef DDR_MPR_TRAINING_CONFIG
	if (result && !ddr_training_check_bypass(DDR_BYPASS_MPR_MASK)) {
		result  = ddr_mpr_training_if();
		result += ddr_dataeye_training_if(ddrtr_result);
	}
#endif

	result += ddr_gating_if();
	result += ddr_vref_training_if(ddrtr_result);

	if (!result)
		ddr_training_suc();
	return result;
}
#endif /* DDR_SW_TRAINING_FUNC_PUBLIC */

#if defined(DDR_HW_TRAINING_CONFIG) && defined(DDR_HW_TRAINING_FUNC_PUBLIC)
/**
 * ddr_hw_training_func
 * @void
 *
 *
 */
int ddr_hw_training_func(void)
{
	unsigned int base_dmc, base_phy;
	struct tr_relate_reg relate_reg;
	int result = 0;
	int i;

	/* hardware training disable */
	if (ddr_training_check_bypass(DDR_BYPASS_HW_MASK))
		return 0;

	ddr_training_save_reg(&relate_reg, DDR_BYPASS_HW_MASK);
	for (i = 0; i < DDR_PHY_NUM; i++) {
		if (ddr_training_phy_disable(i))
			continue;

		ddr_training_get_base(i, &base_dmc, &base_phy);
		result += ddr_hw_training(base_dmc, base_phy);
	}
	ddr_training_restore_reg(&relate_reg);

	return result;
}
#else
int ddr_hw_training_func(void)
{
	DDR_WARNING("Not support DDR HW training.");
	return 0;
}
#endif /* DDR_HW_TRAINING_CONFIG */

#if defined(DDR_MPR_TRAINING_CONFIG) && defined(DDR_MPR_TRAINING_FUNC_PUBLIC)
/**
 * ddr_mpr_training_func
 * @void
 *
 *
 */
int ddr_mpr_training_func(void)
{
	unsigned int base_dmc, base_phy;
	struct tr_relate_reg relate_reg;
	int result = 0;
	int i;

	/* MPR training disable */
	if (ddr_training_check_bypass(DDR_BYPASS_MPR_MASK))
		return 0;

	ddr_training_save_reg(&relate_reg, DDR_BYPASS_MPR_MASK);
	for (i = 0; i < DDR_PHY_NUM; i++) {
		if (ddr_training_phy_disable(i))
			continue;

		ddr_training_get_base(i, &base_dmc, &base_phy);
		result += ddr_mpr_training(base_dmc, base_phy);
	}
	ddr_training_restore_reg(&relate_reg);

	return result;
}
#else
int ddr_mpr_training_func(void)
{
	DDR_WARNING("Not support DDR MPR training.");
	return 0;
}
#endif /* DDR_MPR_TRAINING_CONFIG */

#if defined(DDR_WL_TRAINING_CONFIG) && defined(DDR_WL_FUNC_PUBLIC)
/**
 * ddr_wl_func
 * @void
 *
 *
 */
int ddr_wl_func(void)
{
	unsigned int base_dmc, base_phy;
	struct tr_relate_reg relate_reg;
	int result = 0;
	int i;

	/* write leveling disable */
	if (ddr_training_check_bypass(DDR_BYPASS_WL_MASK))
		return 0;

	ddr_training_save_reg(&relate_reg, DDR_BYPASS_WL_MASK);
	for (i = 0; i < DDR_PHY_NUM; i++) {
		if (ddr_training_phy_disable(i))
			continue;

		ddr_training_get_base(i, &base_dmc, &base_phy);
		result += ddr_write_leveling(base_dmc, base_phy);
	}
	ddr_training_restore_reg(&relate_reg);

	return result;
}
#else
int ddr_wl_func()
{
	DDR_WARNING("Not support DDR WL training.");
	return 0;
}
#endif /* DDR_WL_TRAINING_CONFIG */

#if defined(DDR_GATE_TRAINING_CONFIG) && defined(DDR_GATING_FUNC_PUBLIC)
/**
 * ddr_gating_func
 * @void
 *
 *
 */
int ddr_gating_func(void)
{
	unsigned int base_dmc, base_phy;
	struct tr_relate_reg relate_reg;
	int result = 0;
	int i;

	/* gate training disable */
	if (ddr_training_check_bypass(DDR_BYPASS_GATE_MASK)) {
		for (i = 0; i < DDR_PHY_NUM; i++) {
			if (ddr_training_phy_disable(i))
				continue;

			ddr_training_get_base(i, &base_dmc, &base_phy);
			/* check hardware gating */
			if (REG_READ(base_phy + DDR_PHY_PHYINITSTATUS)
				& PHY_INITSTATUS_GT_MASK) {
				DDR_FATAL("PHY[%x] hw gating fail.", base_phy);
				ddr_training_stat(DDR_ERR_HW_GATING,
					base_phy, -1, -1);
				return -1;
			}
		}
		return 0;
	}

	ddr_training_save_reg(&relate_reg, DDR_BYPASS_GATE_MASK);
	for (i = 0; i < DDR_PHY_NUM; i++) {
		if (ddr_training_phy_disable(i))
			continue;

		ddr_training_get_base(i, &base_dmc, &base_phy);
		ddr_ddrt_init(base_dmc, DDR_DDRT_MODE_GATE);
		ddr_training_switch_axi(i, &relate_reg);
		result += ddr_gate_training(base_dmc, base_phy);
	}
	ddr_training_restore_reg(&relate_reg);

	return result;
}
#else
int ddr_gating_func(void)
{
	DDR_WARNING("Not support DDR gate training.");
	return 0;
}
#endif /* DDR_GATE_TRAINING_CONFIG */

#ifdef DDR_DATAEYE_TRAINING_FUNC_PUBLIC
/**
 * ddr_dataeye_training_func
 * @ddrtr_result
 *
 *
 */
int ddr_dataeye_training_func(void *ddrtr_result)
{
	unsigned int base_dmc, base_phy;
	struct tr_relate_reg relate_reg;
	int result = 0;
	int i;

	/* dataeye training disable */
	if (ddr_training_check_bypass(DDR_BYPASS_DATAEYE_MASK))
		return 0;

	ddr_training_save_reg(&relate_reg, DDR_BYPASS_DATAEYE_MASK);
	for (i = 0; i < DDR_PHY_NUM; i++) {
		if (ddr_training_phy_disable(i))
			continue;

		ddr_training_get_base(i, &base_dmc, &base_phy);
		ddr_ddrt_init(base_dmc, DDR_DDRT_MODE_GATE);
		ddr_training_switch_axi(i, &relate_reg);
		result += ddr_dataeye_training(base_dmc, base_phy,
			ddrtr_result);
	}
	ddr_training_restore_reg(&relate_reg);

	return result;
}
#endif /* DDR_DATAEYE_TRAINING_FUNC_PUBLIC */

#if defined(DDR_VREF_TRAINING_CONFIG) && defined(DDR_VREF_TRAINING_FUNC_PUBLIC)
/**
 * ddr_vref_training_func
 * @void
 *
 *
 */
int ddr_vref_training_func(void *ddrtr_result)
{
	unsigned int base_dmc, base_phy;
	struct tr_relate_reg relate_reg;
	int result = 0;
	int i;

	/* vref training disable */
	if (ddr_training_check_bypass(DDR_BYPASS_VREF_MASK))
		return 0;

	ddr_training_save_reg(&relate_reg, DDR_BYPASS_VREF_MASK);
	for (i = 0; i < DDR_PHY_NUM; i++) {
		if (ddr_training_phy_disable(i))
			continue;

		ddr_training_get_base(i, &base_dmc, &base_phy);
		ddr_ddrt_init(base_dmc, DDR_DDRT_MODE_GATE);
		ddr_training_switch_axi(i, &relate_reg);
		result += ddr_vref_training(base_dmc, base_phy,
			ddrtr_result);
	}
	ddr_training_restore_reg(&relate_reg);

	return result;
}
#else
int ddr_vref_training_func(void *ddrtr_result)
{
	DDR_WARNING("Not support DDR vref training.");
	return 0;
}
#endif /* DDR_VREF_TRAINING_CONFIG */

#if defined(DDR_AC_TRAINING_CONFIG) && defined(DDR_AC_TRAINING_FUNC_PUBLIC)
/**
 * ddr_ac_training_func
 * @void
 *
 *
 */
int ddr_ac_training_func(void)
{
	int result = 0;
	int i;
	unsigned int base_dmc, base_phy;
	struct tr_relate_reg relate_reg;

	/* AC training disable */
	if (ddr_training_check_bypass(DDR_BYPASS_AC_MASK))
		return 0;

	ddr_training_save_reg(&relate_reg, DDR_BYPASS_AC_MASK);
	for (i = 0; i < DDR_PHY_NUM; i++) {
		if (ddr_training_phy_disable(i))
			continue;

		ddr_training_get_base(i, &base_dmc, &base_phy);
		result += ddr_ac_training(base_dmc, base_phy);
	}
	ddr_training_restore_reg(&relate_reg);

	return result;
}
#else
int ddr_ac_training_func(void)
{
	DDR_WARNING("Not support DDR AC training.");
	return 0;
}
#endif /* DDR_MPR_TRAINING_CONFIG */

/**
 * ddr_sw_training_if
 * @ddrtr_result
 *
 *
 */
int ddr_sw_training_if(void *ddrtr_result)
{
	return DDR_SW_TRAINING_FUNC(ddrtr_result);
}

/**
 * ddr_hw_training_if
 * @void
 *
 *
 */
int ddr_hw_training_if(void)
{
	return DDR_HW_TRAINING_FUNC();
}

/**
 * ddr_mpr_training_if
 * @void
 *
 *
 */
int ddr_mpr_training_if(void)
{
	return DDR_MPR_TRAINING_FUNC();
}

/**
 * ddr_wl_if
 * @void
 *
 *
 */
int ddr_wl_if(void)
{
	return DDR_WL_FUNC();
}

/**
 * ddr_gating_if
 * @void
 *
 *
 */
int ddr_gating_if(void)
{
	return DDR_GATING_FUNC();
}

/**
 * ddr_dataeye_training_if
 * @ddrtr_result
 *
 *
 */
int ddr_dataeye_training_if(void *ddrtr_result)
{
	return DDR_DATAEYE_TRAINING_FUNC(ddrtr_result);
}

/**
 * ddr_vref_training_if
 * @void
 *
 *
 */
int ddr_vref_training_if(void *ddrtr_result)
{
	return DDR_VREF_TRAINING_FUNC(ddrtr_result);
}

/**
 * ddr_ac_training_if
 * @void
 *
 *
 */
int ddr_ac_training_if(void)
{
	return DDR_AC_TRAINING_FUNC();
}
