#include <linux/init.h>
#include <linux/module.h>

#include <linux/poll.h>
#include <linux/kdev_t.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <asm/uaccess.h>
#include <linux/slab.h>
#include <linux/device.h>
#include "gpp/mvGpp.h"


#define DEV_NAME "svr500_gpio"

MODULE_AUTHOR("Michael Hsu <michael.hsu@seenergy.com.tw>");
MODULE_DESCRIPTION("SVR500 GPIO Driver");
MODULE_LICENSE("GPL");

static int major;
module_param(major, int, 0);
MODULE_PARM_DESC(major, "Major device number");

#define SVR500_GPIO_DI_PIN_COUNT 8
static u8 svr500_gpio_di_pin[SVR500_GPIO_DI_PIN_COUNT] = { 20, 21, 22, 23, 24, 25, 26, 27 };

#define SVR500_GPIO_DO_PIN_COUNT 4
static u8 svr500_gpio_do_pin[SVR500_GPIO_DO_PIN_COUNT] = { 30, 31, 32, 48 };

#define SVR500_GPIO_DO_DATA_BYTE 0x00
#define SVR500_GPIO_DO_DATA_STATUS_BYTE 0x01
#define SVR500_GPIO_DI_DATA_BYTE 0x02
#define SVR500_GPIO_MAX_DATA_BYTE 0x03


static inline void __set_direction(unsigned pin, int input)
{
    u32 grp = pin >> 5;
    u32 mask = (1 << (pin & 0x1F));

    if (input)
        mvGppTypeSet(grp, mask, MV_GPP_IN & mask);
    else
        mvGppTypeSet(grp, mask, MV_GPP_OUT & mask);
}

static void __set_level(unsigned pin, int high)
{
    u32 grp = pin >> 5;
    u32 mask = (1 << (pin & 0x1F));

    if (high)
        mvGppValueSet (grp, mask, mask);
    else
        mvGppValueSet (grp, mask, 0);
}

static int __get_value(unsigned pin)
{
    u32 val;
    u32 grp = pin >> 5;
    u32 mask = (1 << (pin & 0x1F));

    val = mvGppValueGet(grp, mask) ^ mvGppPolarityGet(grp, mask);
    return (val >> (pin & 31)) & 1;
}

static inline void __set_init(void)
{
    u32 idx = 0;

    for (idx = 0; idx < SVR500_GPIO_DI_PIN_COUNT; idx++)
        __set_direction(svr500_gpio_di_pin[idx], 1);

    for (idx = 0; idx < SVR500_GPIO_DO_PIN_COUNT; idx++)
        __set_direction(svr500_gpio_do_pin[idx], 0);

    for (idx = 0; idx < SVR500_GPIO_DO_PIN_COUNT; idx++)
        __set_level(svr500_gpio_do_pin[idx], 0);
}

static void __set_all_do_level(unsigned char data_byte)
{
    u32 idx = 0;
    data_byte &= 0x0F;

    //printk("[SET] DO_DATA_BYTE: %x\n", data_byte);
    for (idx = 0; idx < SVR500_GPIO_DO_PIN_COUNT; idx++)
    {
        __set_level(svr500_gpio_do_pin[idx], data_byte & 0x01);
        data_byte = data_byte >> 1;
    }
}

static unsigned char __get_all_do_value(void)
{
    u32 idx = 0;
    u32 val = 0;
    u8 data_byte = 0;

    for (idx = 0; idx < SVR500_GPIO_DO_PIN_COUNT; idx++)
    {
        val = __get_value(svr500_gpio_do_pin[idx]);
        if (val)
            data_byte |= (0x01 << idx);
    }
    //printk("[GET] DO_DATA_BYTE: %x\n", data_byte);
    return data_byte;
}

static unsigned char __get_all_di_value(void)
{
    u32 idx = 0;
    u32 val = 0;
    u8 data_byte = 0;

    for (idx = 0; idx < SVR500_GPIO_DI_PIN_COUNT; idx++)
    {
        val = __get_value(svr500_gpio_di_pin[idx]);
        if (val)
            data_byte |= (0x01 << idx);
    }
    //printk("[SET] DI_DATA_BYTE: %x\n", data_byte);
    return data_byte;
}


struct svr500_gpio_dev {
    struct cdev cdev;
} *svr500_gpio_devp;

struct class *svr500_gpio_class;


int svr500_gpio_open(struct inode *inode, struct file *filp)
{
    return 0;
}

int svr500_gpio_release(struct inode *inode, struct file *filp)
{
    return 0;
}

ssize_t svr500_gpio_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
    unsigned char data[SVR500_GPIO_MAX_DATA_BYTE];

    count = (count > SVR500_GPIO_MAX_DATA_BYTE)?SVR500_GPIO_MAX_DATA_BYTE:count;

    data[SVR500_GPIO_DO_DATA_BYTE] = 0;
    data[SVR500_GPIO_DO_DATA_STATUS_BYTE] = __get_all_do_value();
    data[SVR500_GPIO_DI_DATA_BYTE] = __get_all_di_value();
    if (copy_to_user(buf, data, count))
        return -EFAULT;
    return count;
}

ssize_t svr500_gpio_write(struct file *filp, const char __user *buf, size_t count, loff_t *f_pos)
{
    unsigned char data[SVR500_GPIO_MAX_DATA_BYTE];

    count = (count > SVR500_GPIO_MAX_DATA_BYTE)?SVR500_GPIO_MAX_DATA_BYTE:count;

    if (copy_from_user(data, buf, count))
        return -EFAULT;

    __set_all_do_level(data[SVR500_GPIO_DO_DATA_BYTE]);
    return count;
}

unsigned int svr500_gpio_poll(struct file *filp, poll_table *wait)
{
    return POLLIN | POLLRDNORM | POLLOUT | POLLWRNORM;
}


static struct file_operations svr500_gpio_fops = {
    .owner = THIS_MODULE,
    .read = svr500_gpio_read,
    .write = svr500_gpio_write,
    .poll = svr500_gpio_poll,
    .open = svr500_gpio_open,
    .release = svr500_gpio_release,
};


static int __init svr500_gpio_init(void)
{
    dev_t dev_id;
    int retval;

    if (major) {
        dev_id = MKDEV(major, 0);
        retval = register_chrdev_region(dev_id, 1, DEV_NAME);
    } else {
        retval = alloc_chrdev_region(&dev_id, 0, 1, DEV_NAME);
        major = MAJOR(dev_id);
    }
    printk(KERN_ALERT "Register SVR500 GPIO Driver (%d, %d)\n", MAJOR(dev_id), MINOR(dev_id));

    svr500_gpio_class = class_create(THIS_MODULE, DEV_NAME);

    svr500_gpio_devp = kmalloc(sizeof(struct svr500_gpio_dev), GFP_KERNEL); 
    if (NULL == svr500_gpio_devp)
        printk(KERN_ALERT "kmalloc failed\n");

    cdev_init (&(svr500_gpio_devp->cdev), &svr500_gpio_fops);
    svr500_gpio_devp->cdev.owner = THIS_MODULE;
    if (cdev_add(&(svr500_gpio_devp->cdev), dev_id, 1) < 0)
        printk(KERN_NOTICE "cdev_add failed\n");

    device_create(svr500_gpio_class, NULL, dev_id, NULL, "svr500_gpio%d", 0);
    __set_init ();
    printk(KERN_ALERT "SVR500 GPIO Driver Initialized.\n");
    return 0;
}

static void __exit svr500_gpio_exit(void)
{
    dev_t dev_id = MKDEV(major, 0);

    if (svr500_gpio_devp)
    {
        cdev_del(&(svr500_gpio_devp->cdev));
    }

    unregister_chrdev_region(dev_id, 1);
    class_destroy (svr500_gpio_class);
    printk(KERN_ALERT "Goodbye, cruel world (%d)\n", major);
}


module_init(svr500_gpio_init);
module_exit(svr500_gpio_exit);
