/*
 * Copyright (c) 2000-2002 by Dima Epshtein
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/platform_device.h>

extern const struct hc_driver ehci_driver;
/*-------------------------------------------------------------------------*/

static const struct hc_driver ehci_feroceon_hc_driver = {
	.description = hcd_name,
	.product_desc = "Feroceon EHCI",
	.hcd_priv_size = sizeof(struct ehci_hcd),

	/*
	 * generic hardware linkage
	 */
	.irq = ehci_irq,
	.flags = HCD_MEMORY | HCD_USB2,

	/*
	 * basic lifecycle operations
	 */
	.reset = ehci_init,
	.start = ehci_run,
	.stop = ehci_stop,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue = ehci_urb_enqueue,
	.urb_dequeue = ehci_urb_dequeue,
	.endpoint_disable = ehci_endpoint_disable,

	/*
	 * scheduling support
	 */
	.get_frame_number = ehci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data = ehci_hub_status_data,
	.hub_control = ehci_hub_control,
#ifdef	CONFIG_PM
	.hub_suspend = ehci_hub_suspend,
	.hub_resume = ehci_hub_resume,
#endif
};

static int ehci_hcd_feroceon_drv_probe(struct platform_device *dev) 
{ 
	const struct hc_driver *driver = &ehci_feroceon_hc_driver; 
	int retval;
	struct usb_hcd *hcd = NULL;
	struct resource *res;
	struct ehci_hcd *ehci;

	hcd = usb_create_hcd (driver, &dev->dev, dev->dev.bus_id);
	if (hcd == NULL) {
		printk("%s: hcd_alloc failed\n", __FUNCTION__);
		return -ENOMEM;
	}

	res = platform_get_resource(dev, IORESOURCE_IRQ, 0);
	if (!res) {
		printk("Found HC with no IRQ!\n");
		return -ENODEV;
	}
	hcd->irq = res->start;

	res = platform_get_resource(dev, IORESOURCE_MEM, 0);
	if (!res) {
		printk("Found HC with no REGS!\n");
		return -ENODEV;
	}
	hcd->rsrc_start = res->start;
	hcd->rsrc_len = res->end - res->start + 1;
	if (!request_mem_region(hcd->rsrc_start, hcd->rsrc_len,
			       driver->description)) {
		printk("Controller is allocated ready");
		return -EBUSY;
	}

	hcd->regs = ioremap(hcd->rsrc_start, hcd->rsrc_len) + 0x100;
	if (hcd->regs == NULL) {
		printk("Error memory mapping\n");
		retval = -EFAULT;
		goto free_mem;
	}
	

	ehci = hcd_to_ehci(hcd);
	ehci->caps = hcd->regs;
	ehci->regs = hcd->regs + HC_LENGTH(readl(&ehci->caps->hc_capbase));
	/* cache this readonly data; minimize chip reads */
	ehci->hcs_params = readl(&ehci->caps->hcs_params);
	ehci->is_tdi_rh_tt = 1;		/* It's TDI/ARC USB IP */

	/* ehci_hcd_init(hcd_to_ehci(hcd)); */
	retval = usb_add_hcd (hcd, hcd->irq, IRQF_DISABLED);
	if (retval != 0)
	{
		printk("%s: usb_add_hcd failed, retval=0x%x\n",
			       __FUNCTION__, retval); 
		retval = -ENOMEM; 
	}

	goto good;

	iounmap(hcd->regs);
free_mem:
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
good:
	return retval; 
} 
 
static int ehci_hcd_feroceon_drv_remove(struct platform_device *dev) 
{ 
	struct usb_hcd *hcd = platform_get_drvdata(dev);

	printk("USB: ehci_platform_remove\n");

	usb_remove_hcd(hcd);
	iounmap(hcd->regs);
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);

	return 0;
} 
 

MODULE_ALIAS("feroceon usb");
static struct platform_driver ehci_hcd_feroceon_driver = {
	.probe = ehci_hcd_feroceon_drv_probe,
	.remove = ehci_hcd_feroceon_drv_remove,
	.driver = {
		.name = "Feroceon ehci",
		.bus = &platform_bus_type
	}
};
