/*
 * Board setup routines for the Freescale Sandpoint Test Platform.
 *
 * Author: Mark A. Greer <mgreer@mvista.com>
 *
 * 2000-2003 (c) MontaVista Software, Inc.  This file is licensed under
 * the terms of the GNU General Public License version 2.  This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

/*
 * This file adds support for the Freescale Sandpoint Test Platform.
 * These boards have a PPMC slot for the processor so any combination
 * of cpu and host bridge can be attached.  This port is for an 8240 PPMC
 * module from Freescale and other closely related cpu/host bridge
 * combinations (e.g., 750/755/7400 with MPC107 host bridge).
 * The sandpoint itself has a Windbond 83c553 (PCI-ISA bridge, 2 DMA ctlrs, 2
 * cascaded 8259 interrupt ctlrs, 8254 Timer/Counter, and an IDE ctlr), a
 * National 87308 (RTC, 2 UARTs, Keyboard & mouse ctlrs, and a floppy ctlr),
 * and 4 PCI slots (only 2 of which are usable; the other 2 are keyed for 3.3V
 * but are really 5V).
 *
 * The firmware on the sandpoint is called DINK (not my acronym :).  This port
 * depends on DINK to do some basic initialization (e.g., initialize the memory
 * ctlr) and to ensure that the processor is using MAP B (CHRP map).
 *
 * Since Freescale listened to our suggestions for improvement, we now have
 * the Sandpoint X3 board.  All of the PCI slots are available, it uses
 * the serial interrupt interface (just a hardware thing we need to
 * configure properly).
 *
 * Use the default X3 switch settings.  The interrupts are then:
 *		EPIC	Source
 *		  0	SIOINT 		(8259, active low)
 *		  1	PCI #1
 *		  2	PCI #2
 *		  3	PCI #3
 *		  4	PCI #4
 *		  7	Winbond INTC	(IDE interrupt)
 *		  8	Winbond INTD	(IDE interrupt)
 *
 *
 * Freescale has finally released a version of DINK32 that correctly
 * (seemingly) initalizes the memory controller correctly, regardless
 * of the amount of memory in the system.  Once a method of determining
 * what version of DINK initializes the system for us, if applicable, is
 * found, we can hopefully stop hardcoding 32MB of RAM.
 */

#include <linux/initrd.h>
#include <linux/delay.h>
#include <linux/ide.h>
#include <linux/root_dev.h>

#include <asm/time.h>
#include <asm/i8259.h>
#include <asm/udbg.h>
#include <asm/todc.h>
#include <asm/mpic.h>
#include <asm/todc.h>

/*
 * Interrupt setup and service.  Interrrupts on the Sandpoint come
 * from the four PCI slots plus the 8259 in the Winbond Super I/O (SIO).
 * The 8259 is cascaded from EPIC IRQ0, IRQ1-4 map to PCI slots 1-4,
 * IDE is on EPIC 7 and 8.
 */
static void sandpoint_8259_cascade(unsigned int irq, struct irq_desc *desc,
		struct pt_regs *regs)
{
	unsigned int cascade_irq = i8259_irq(regs);

	if (cascade_irq != NO_IRQ)
		generic_handle_irq(cascade_irq, regs);
	desc->chip->eoi(irq);
}

static void __init
sandpoint_init_irq(void)
{
	struct mpic *mpic;
	struct device_node *dnp, *cascade_node = NULL;
	void *prop;
	int size;
	unsigned int cascade_irq;
	phys_addr_t paddr;

	dnp = of_find_node_by_type(NULL, "open-pic");
	if (dnp == NULL)
		return;

	prop = (struct device_node *)get_property(dnp, "reg", &size);
	paddr = (phys_addr_t)of_translate_address(dnp, prop);

	mpic = mpic_alloc(dnp, paddr, MPIC_PRIMARY, 0, 0, " EPIC     ");

	prop = (struct device_node *)get_property(dnp, "clock-ratio", &size);
	if (prop != NULL)
		mpic_set_clk_ratio(mpic, *(u32 *)prop);

	prop = (struct device_node *)get_property(dnp, "serial-mode", &size);
	if (prop != NULL)
		mpic_set_serial_int(mpic, *(u32 *)prop);

	mpic_assign_isu(mpic, 0, paddr + 0x10200);
	mpic_init(mpic);

	for_each_node_by_type(dnp, "interrupt-controller")
		if (device_is_compatible(dnp, "chrp,iic")) {
			cascade_node = dnp;
			break;
		}

	if (cascade_node == NULL)
		return;

	cascade_irq = irq_of_parse_and_map(cascade_node, 0);

	if (cascade_irq == NO_IRQ)
		return;

	i8259_init(cascade_node, 0);
	set_irq_chained_handler(cascade_irq, sandpoint_8259_cascade);
}

static int __init
add_bridge(struct device_node *dev)
{
	int len;
	struct pci_controller *hose;
	int *bus_range;

	printk("Adding PCI host bridge %s\n", dev->full_name);

	bus_range = (int *) get_property(dev, "bus-range", &len);
	if (bus_range == NULL || len < 2 * sizeof(int))
		printk(KERN_WARNING "Can't get bus-range for %s, assume"
				" bus 0\n", dev->full_name);

	hose = pcibios_alloc_controller();
	if (hose == NULL)
		return -ENOMEM;
	hose->first_busno = bus_range ? bus_range[0] : 0;
	hose->last_busno = bus_range ? bus_range[1] : 0xff;
	setup_indirect_pci(hose, 0xfec00000, 0xfee00000);

	/* Interpret the "ranges" property */
	/* This also maps the I/O region and sets isa_io/mem_base */
	pci_process_bridge_OF_ranges(hose, dev, 1);
	return 0;
}

/*
 * Freescale Sandpoint interrupt routing.
 */
static inline int
x3_map_irq(struct pci_dev *dev, unsigned char idsel, unsigned char pin)
{
	static char pci_irq_table[][4] =
	/*
	 *	PCI IDSEL/INTPIN->INTLINE
	 * 	   A   B   C   D
	 */
	{
		{ 0, 0, 0, 0 },	/* IDSEL 11 - i8259 on Winbond */
		{ 0, 0, 0, 0 },	/* IDSEL 12 - unused */
		{ 2, 5, 4, 3 },	/* IDSEL 13 - PCI slot 1 */
		{ 3, 2, 5, 4 },	/* IDSEL 14 - PCI slot 2 */
		{ 4, 3, 2, 5 },	/* IDSEL 15 - PCI slot 3 */
		{ 5, 4, 3, 2 },	/* IDSEL 16 - PCI slot 4 */
	};

	const long min_idsel = 11, max_idsel = 16, irqs_per_slot = 4;
	return PCI_IRQ_TABLE_LOOKUP;
}

#ifdef CONFIG_PPC_TODC
TODC_ALLOC()
#endif

static void __init
sandpoint_setup_arch(void)
{
	struct device_node *dnp;

#ifdef CONFIG_BLK_DEV_INITRD
	if (initrd_start)
		ROOT_DEV = Root_RAM0;
	else
#endif
#ifdef	CONFIG_ROOT_NFS
		ROOT_DEV = Root_NFS;
#else
		ROOT_DEV = Root_HDA1;
#endif

	for (dnp = NULL; (dnp=of_find_node_by_type(dnp,"pci")) != NULL;)
		add_bridge(dnp);
	ppc_md.pci_swizzle = common_swizzle;
	ppc_md.pci_map_irq = x3_map_irq;

	printk(KERN_INFO "Port by MontaVista Software, Inc. <source@mvista.com>\n");

#ifdef CONFIG_PPC_TODC
	TODC_INIT(TODC_TYPE_PC97307, 0x70, 0x00, 0x71, 8);

	ppc_md.nvram_read_val = todc_mc146818_read_val;
	ppc_md.nvram_write_val = todc_mc146818_write_val;

	ppc_md.time_init = todc_time_init;
	ppc_md.set_rtc_time = todc_set_rtc_time;
	ppc_md.get_rtc_time = todc_get_rtc_time;
#endif

	/* DINK32 12.3 and below do not correctly enable any caches.
	 * We will do this now with good known values.  Future versions
	 * of DINK32 are supposed to get this correct.
	 */
	if (cpu_has_feature(CPU_FTR_SPEC7450))
		/* 745x is different.  We only want to pass along enable. */
		_set_L2CR(L2CR_L2E);
	else if (cpu_has_feature(CPU_FTR_L2CR))
		/* All modules have 1MB of L2.  We also assume that an
		 * L2 divisor of 3 will work.
		 */
		_set_L2CR(L2CR_L2E | L2CR_L2SIZ_1MB | L2CR_L2CLK_DIV3
				| L2CR_L2RAM_PIPE | L2CR_L2OH_1_0 | L2CR_L2DF);
#if 0
	/* Untested right now. */
	if (cpu_has_feature(CPU_FTR_L3CR)) {
		/* Magic value. */
		_set_L3CR(0x8f032000);
	}
#endif
}

static void
sandpoint_restart(char *cmd)
{
	void _nmask_and_or_msr(unsigned long nmask, unsigned long or_val);

	local_irq_disable();
	_nmask_and_or_msr(0, MSR_IP); /* Set exception prefix high - firmware */

	/* Reset system via Port 92 */
	outb(0x00, 0x92);
	outb(0x01, 0x92);

	for(;;);	/* Spin until reset happens */
}

static void
sandpoint_power_off(void)
{
	local_irq_disable();
	for(;;);	/* No way to shut power off with software */
	/* NOTREACHED */
}

static void
sandpoint_halt(void)
{
	sandpoint_power_off();
	/* NOTREACHED */
}

#if defined(CONFIG_BLK_DEV_IDE) || defined(CONFIG_BLK_DEV_IDE_MODULE)
/*
 * IDE support.
 */
static int		sandpoint_ide_ports_known = 0;
static unsigned long	sandpoint_ide_regbase[MAX_HWIFS];
static unsigned long	sandpoint_ide_ctl_regbase[MAX_HWIFS];
static unsigned long	sandpoint_idedma_regbase;

static void
sandpoint_ide_probe(void)
{
	struct pci_dev *pdev = pci_get_device(PCI_VENDOR_ID_WINBOND,
			PCI_DEVICE_ID_WINBOND_82C105, NULL);

	if (pdev) {
		sandpoint_ide_regbase[0]=pdev->resource[0].start;
		sandpoint_ide_regbase[1]=pdev->resource[2].start;
		sandpoint_ide_ctl_regbase[0]=pdev->resource[1].start;
		sandpoint_ide_ctl_regbase[1]=pdev->resource[3].start;
		sandpoint_idedma_regbase=pdev->resource[4].start;
		pci_dev_put(pdev);
	}

	sandpoint_ide_ports_known = 1;
}

static unsigned long
sandpoint_ide_default_io_base(int index)
{
	if (sandpoint_ide_ports_known == 0)
		sandpoint_ide_probe();

	return sandpoint_ide_regbase[index];
}

static void __init
sandpoint_ide_init_hwif_ports(hw_regs_t *hw, unsigned long data_port,
		unsigned long ctrl_port, int *irq)
{
	unsigned long reg = data_port;
	uint	alt_status_base;
	int	i;

	for (i = IDE_DATA_OFFSET; i <= IDE_STATUS_OFFSET; i++) {
		hw->io_ports[i] = reg++;
	}

	if (data_port == sandpoint_ide_regbase[0]) {
		alt_status_base = sandpoint_ide_ctl_regbase[0] + 2;
		hw->irq = 14;
	}
	else if (data_port == sandpoint_ide_regbase[1]) {
		alt_status_base = sandpoint_ide_ctl_regbase[1] + 2;
		hw->irq = 15;
	}
	else {
		alt_status_base = 0;
		hw->irq = 0;
	}

	if (ctrl_port) {
		hw->io_ports[IDE_CONTROL_OFFSET] = ctrl_port;
	} else {
		hw->io_ports[IDE_CONTROL_OFFSET] = alt_status_base;
	}

	if (irq != NULL) {
		*irq = hw->irq;
	}
}

static int __init
sandpoint_ide_init(void)
{
	/* doesn't find hda without this */
	ppc_ide_md.default_io_base = sandpoint_ide_default_io_base;
	/* get error msgs without this */
	ppc_ide_md.ide_init_hwif = sandpoint_ide_init_hwif_ports;
	return 0;
}
arch_initcall(sandpoint_ide_init);
#endif

void __init
sandpoint_pcibios_fixup(void)
{
	struct pci_dev *dev = NULL;

	for_each_pci_dev(dev)
		pci_read_irq_line(dev);
}

#define	PLATFORM_NAME	"Freescale Sandpoint"

static int __init
sandpoint_probe(void)
{
	unsigned long root = of_get_flat_dt_root();

	if (of_flat_dt_is_compatible(root, "Sandpoint"))
		return 1;
	return 0;
}

define_machine(sandpoint) {
	.name		= PLATFORM_NAME,
	.probe		= sandpoint_probe,
	.setup_arch	= sandpoint_setup_arch,
	.init_IRQ	= sandpoint_init_irq,
	.pcibios_fixup	= sandpoint_pcibios_fixup,
	.get_irq	= mpic_get_irq,
	.restart	= sandpoint_restart,
	.power_off	= sandpoint_power_off,
	.halt		= sandpoint_halt,
	.calibrate_decr	= generic_calibrate_decr,
	.progress	= udbg_progress,
};
