/*
 * Freescale mpc10[67] & mpc824[015] specific code.
 *
 * Author: Mark A. Greer <mgreer@mvista.com>
 *
 * 2001 (c) MontaVista, Software, Inc.  This file is licensed under
 * the terms of the GNU General Public License version 2.  This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <stdarg.h>
#include <stddef.h>
#include "types.h"
#include "string.h"
#include "stdio.h"
#include "io.h"

/* Map B (CHRP Map) Defines */
#define	MPC10X_MAPB_CNFG_ADDR		0xfec00000
#define	MPC10X_MAPB_CNFG_DATA		0xfee00000

/* Define offsets for the memory controller registers in the config space */
#define MPC10X_MCTLR_MEM_START_1	0x80	/* Banks 0-3 */
#define MPC10X_MCTLR_MEM_START_2	0x84	/* Banks 4-7 */
#define MPC10X_MCTLR_EXT_MEM_START_1	0x88	/* Banks 0-3 */
#define MPC10X_MCTLR_EXT_MEM_START_2	0x8c	/* Banks 4-7 */

#define MPC10X_MCTLR_MEM_END_1		0x90	/* Banks 0-3 */
#define MPC10X_MCTLR_MEM_END_2		0x94	/* Banks 4-7 */
#define MPC10X_MCTLR_EXT_MEM_END_1	0x98	/* Banks 0-3 */
#define MPC10X_MCTLR_EXT_MEM_END_2	0x9c	/* Banks 4-7 */

#define MPC10X_MCTLR_MEM_BANK_ENABLES	0xa0

#define	PCI_DEVFN(slot,func)	((((slot) & 0x1f) << 3) | ((func) & 0x07))

/* Indirect PCI config space access routines */
static inline void
pci_indirect_read_config_byte(u32 *cfg_addr, u32 *cfg_data, int devfn,
		int offset, u8 *val)
{
	out_be32(cfg_addr,
		((offset & 0xfc) << 24) | (devfn << 16) | (0 << 8) | 0x80);
	*val = in_8((u8 *)(cfg_data + (offset & 3)));
	return;
}

static inline void
pci_indirect_read_config_dword(u32 *cfg_addr, u32 *cfg_data, int devfn,
		int offset, u32 *val)
{
	out_be32(cfg_addr,
		((offset & 0xfc) << 24) | (devfn << 16) | (0 << 8) | 0x80);
	*val = in_le32(cfg_data + (offset & 3));
	return;
}

/*
 * Read the memory controller registers to determine the amount of memory in
 * the system.  This assumes that the firmware has correctly set up the memory
 * controller registers.
 * Assume memory map B (CHRP).
 */
u32
mpc10x_get_mem_size(void)
{
	u32 *config_addr, *config_data, val;
	u32 start, end, total, offset, i;
	u8 bank_enables;

	config_addr = (u32 *)MPC10X_MAPB_CNFG_ADDR;
	config_data = (u32 *)MPC10X_MAPB_CNFG_DATA;

	pci_indirect_read_config_byte(config_addr, config_data, PCI_DEVFN(0,0),
			MPC10X_MCTLR_MEM_BANK_ENABLES, &bank_enables);

	total = 0;

	for (i=0; i<8; i++) {
		if (bank_enables & (1 << i)) {
			offset = MPC10X_MCTLR_MEM_START_1 + ((i > 3) ? 4 : 0);
			pci_indirect_read_config_dword(config_addr, config_data,
					PCI_DEVFN(0,0), offset, &val);
			start = (val >> ((i & 3) << 3)) & 0xff;

			offset = MPC10X_MCTLR_EXT_MEM_START_1 + ((i>3) ? 4 : 0);
			pci_indirect_read_config_dword(config_addr, config_data,
					PCI_DEVFN(0,0), offset, &val);
			val = (val >> ((i & 3) << 3)) & 0x03;
			start = (val << 28) | (start << 20);

			offset = MPC10X_MCTLR_MEM_END_1 + ((i > 3) ? 4 : 0);
			pci_indirect_read_config_dword(config_addr, config_data,
					PCI_DEVFN(0,0), offset, &val);
			end = (val >> ((i & 3) << 3)) & 0xff;

			offset = MPC10X_MCTLR_EXT_MEM_END_1 + ((i > 3) ? 4 : 0);
			pci_indirect_read_config_dword(config_addr, config_data,
					PCI_DEVFN(0,0), offset, &val);
			val = (val >> ((i & 3) << 3)) & 0x03;
			end = (val << 28) | (end << 20) | 0xfffff;

			total += (end - start + 1);
		}
	}

	return total;
}
