#include <linux/bcd.h>

#include "mvS35390A.h"
#include "mvS35390AReg.h"

/*#define MV_DEBUG*/
#ifdef MV_DEBUG
#define DB(x) x
#else
#define DB(x)
#endif

static MV_STATUS mvRtcRegWrite(MV_U32 addr, MV_U8 *data, MV_U32 len);
static MV_STATUS mvRtcRegRead(MV_U32 addr, MV_U8 *data, MV_U32 len);

MV_32  twentyfourhour; /* XXX: FIXME */

static MV_U8 s35390aHr2Reg(MV_32 hour)
{
	if (twentyfourhour)
		return BIN2BCD(hour);

	if (hour < 12)
		return BIN2BCD(hour);

	return 0x40 | BIN2BCD(hour - 12);
}

static MV_U32 s35390aReg2Hr(MV_U8 reg)
{
	MV_U32 hour;

	if (twentyfourhour)
		return BCD2BIN(reg & 0x3f);

	hour = BCD2BIN(reg & 0x3f);
	if (reg & 0x40)
		hour += 12;

	return hour;
}
static int s35390aReset(MV_VOID)
{
	MV_U8 tempValue[1];

	if (mvRtcRegRead(S35390A_CMD_STATUS1, tempValue, sizeof(tempValue)) < 0)
		return -EIO;


	if (!(tempValue[0] & (S35390A_FLAG_POC | S35390A_FLAG_BLD)))
		return 0;

	tempValue[0] |= (S35390A_FLAG_RESET | S35390A_FLAG_24H);
	tempValue[0] &= 0xf0;
	return mvRtcRegWrite(S35390A_CMD_STATUS1, tempValue, sizeof(tempValue));
}

static int s35390aDisableTestMode(MV_VOID)
{
	char tempValue[1];

	if (mvRtcRegRead(S35390A_CMD_STATUS2, tempValue, sizeof(tempValue)) < 0)
		return -EIO;

	if (!(tempValue[0] & S35390A_FLAG_TEST))
		return 0;

	tempValue[0] &= ~S35390A_FLAG_TEST;
	return mvRtcRegWrite(S35390A_CMD_STATUS2, tempValue, sizeof(tempValue));
}


/*******************************************************************************
* mvRtcS35390ATimeSet - Set the Alarm of the Real time clock
*
* DESCRIPTION:
*
* INPUT:
*       time - A pointer to a structure RTC_TIME (defined in mvS35390A.h).
* OUTPUT:
*       None.
*
* RETURN:
*       None.
*
*******************************************************************************/
MV_VOID mvRtcS35390AAlarmSet(MV_RTC_TIME* time)
{
	return;
}
/*******************************************************************************
* mvRtcS35390ATimeSet - Update the Real Time Clock.
*
* DESCRIPTION:
*       This function sets a new time and date to the RTC from the given 
*       structure RTC_TIME . All fields within the structure must be assigned 
*		with a value prior to the use of this function.
*
* INPUT:
*       time - A pointer to a structure RTC_TIME (defined in mvS35390A.h).
* OUTPUT:
*       None.
*
* RETURN:
*       None.
*
*******************************************************************************/
MV_VOID mvRtcS35390ATimeSet(MV_RTC_TIME* time)
{
	MV_U8 tempValue[7];
	MV_32 i, err;

	DB(mvOsPrintf("%s: tm is secs=%d, mins=%d, hours=%d mday=%d, "
		"mon=%d, year=%d, wday=%d\n", __func__, time->seconds,
		time->minutes, time->hours, time->date, time->month, time->year,
		time->day));

	tempValue[S35390A_BYTE_YEAR] = BIN2BCD(time->year);
	tempValue[S35390A_BYTE_MONTH] = BIN2BCD(time->month);
	tempValue[S35390A_BYTE_DAY] = BIN2BCD(time->date);
	tempValue[S35390A_BYTE_WDAY] = BIN2BCD(time->day);
	tempValue[S35390A_BYTE_HOURS] = s35390aHr2Reg(time->hours);
	tempValue[S35390A_BYTE_MINS] = BIN2BCD(time->minutes);
	tempValue[S35390A_BYTE_SECS] = BIN2BCD(time->seconds);

	/* This chip expects the bits of each byte to be in reverse order */
	for (i = 0; i < 7; ++i)
		tempValue[i] = bitrev8(tempValue[i]);

	err = mvRtcRegWrite(S35390A_CMD_TIME1, tempValue, sizeof(tempValue));
	if (err < 0)
        mvOsPrintf("Error during mvRtcRegWrite\n");

	return;
}

/*******************************************************************************
* mvRtcS35390ATimeGet - Read the time from the RTC.
*
* DESCRIPTION:
*       This function reads the current time and date from the RTC into the 
*       structure RTC_TIME (defined in mvS35390A.h).
*
* INPUT:
*       time - A pointer to a structure TIME (defined in mvS35390A.h).
*
* OUTPUT:
*       The structure RTC_TIME is updated with the time read from the RTC.
*
* RETURN:
*       None.
*
*******************************************************************************/
MV_VOID mvRtcS35390ATimeGet(MV_RTC_TIME* time)
{
	MV_U8   tempValue[7];
	MV_32   i, err;

	err = mvRtcRegRead(S35390A_CMD_TIME1, tempValue, sizeof(tempValue));
	if (err < 0)
        mvOsPrintf("%s: Error during mvRtcRegRead\n", __func__);

	/* This chip returns the bits of each byte in reverse order */
	for (i = 0; i < 7; ++i) 
		tempValue[i] = bitrev8(tempValue[i]);

	time->seconds = BCD2BIN(tempValue[S35390A_BYTE_SECS]);
	time->minutes = BCD2BIN(tempValue[S35390A_BYTE_MINS]);
	time->hours = s35390aReg2Hr(tempValue[S35390A_BYTE_HOURS]);
	time->day = BCD2BIN(tempValue[S35390A_BYTE_WDAY]);
	time->date = BCD2BIN(tempValue[S35390A_BYTE_DAY]);
	time->month = BCD2BIN(tempValue[S35390A_BYTE_MONTH]);
	time->year = BCD2BIN(tempValue[S35390A_BYTE_YEAR]);

	DB(mvOsPrintf("%s: tm is secs=%d, mins=%d, hours=%d, mday=%d, "
		"mon=%d, year=%d, wday=%d\n", __func__, time->seconds,
		time->minutes, time->hours, time->date, time->month, time->year,
		time->day));

	return;	
}

/*******************************************************************************
* mvRtcS35390AInit - Initialize the clock.
*
* DESCRIPTION:
*       This function initialize the clock registers and read\write functions
*
* INPUT:
*       None.
*
* OUTPUT:
*       None.
*
* RETURN:
*       None.
*
*******************************************************************************/
MV_VOID mvRtcS35390AInit(MV_VOID)
{
    MV_32   err;
	MV_U8   tempValue[1];

	err = s35390aReset();
	if (err < 0) {
		mvOsPrintf("error resetting chip\n");
	}

	err = s35390aDisableTestMode();
	if (err < 0) {
		mvOsPrintf("error disabling test mode\n");
	}

	err = mvRtcRegRead(S35390A_CMD_STATUS1, tempValue, sizeof(tempValue));
	if (err < 0) {
		mvOsPrintf("error checking 12/24 hour mode\n");
	}
	if (tempValue[0] & S35390A_FLAG_24H)
		twentyfourhour = 1;
	else
		twentyfourhour = 0;

	return;
}


/* assumption twsi is initialized !!! */
/*******************************************************************************
* mvRtcRegRead - Read a RTC Register
*
* DESCRIPTION:
*       This function reads a register from the RTC offset.
*
* INPUT:
*       addr - slave address. Seiko RTC has multiple slave address.
*       len - length of the register. Seiko RTC registers are multi-byte
*
* OUTPUT:
*       data - reg read from address addr.
*
* RETURN:
*       None.
*
*******************************************************************************/
static MV_STATUS mvRtcRegRead(MV_U32 addr, MV_U8 *data, MV_U32 len)
{
	MV_TWSI_SLAVE   twsiSlave;
    MV_32           ret;
	
	twsiSlave.slaveAddr.type = mvBoardRtcTwsiAddrTypeGet();
	twsiSlave.slaveAddr.address = S35390A_BASE_ADDR | addr;
	twsiSlave.validOffset = MV_FALSE;
	twsiSlave.offset = 0;
	twsiSlave.moreThen256 = MV_FALSE;
	ret = mvTwsiRead (&twsiSlave, data, len);
    if (ret != MV_OK)
        return -1;

	return 0;
}

/*******************************************************************************
* mvRtcRegWrite - Write a RTC Register.
*
* DESCRIPTION:
*       This function writes to a register at given address.
*
* INPUT:
*       addr - slave address. Seiko RTC has multiple slave address.
*       len - length of the register. Seiko RTC registers are multi-byte
*
* OUTPUT:
*       data - reg write to addr address.
*
* RETURN:
*       None.
*
*******************************************************************************/
static MV_STATUS mvRtcRegWrite(MV_U32 addr, MV_U8 *data, MV_U32 len)
{
	MV_TWSI_SLAVE   twsiSlave;
    MV_32           ret;
	
	twsiSlave.slaveAddr.type = mvBoardRtcTwsiAddrTypeGet();
	twsiSlave.slaveAddr.address = S35390A_BASE_ADDR | addr;
	twsiSlave.validOffset = MV_FALSE;
	twsiSlave.offset = 0;
	twsiSlave.moreThen256 = MV_FALSE;
	ret = mvTwsiWrite (&twsiSlave, data, len);
    if (ret != MV_OK)
        return -1;

	return 0;
}
