#include <time.h>
#include "attributes.h"
#include "ftpparse.h"
#include "buffer.h"
#include "stralloc.h"
#include "scan.h"
#include "ip4.h"
#include "alloc.h"
#include "byte.h"
#include "getln.h"
#include "ftplib.h"
#include "strhash.h"
#include "fmt.h"
#include "uogetopt.h"
#include "uotime.h"
#include "uo_wildmat.h"
#include "urlparse.h"
#include "timeoutio.h"
#include "ftpcopy.h"
#include "bailout.h"
#include "str.h"
#include "error.h"
#include "open.h"
#include "readwrite.h"
#include "close.h"
#include "nowarn.h"
#include "get_cwd.h"
#include "api_futimes.h"
#include "api_utimes.h"
#include "case.h"
#include "mdtm.h"
#include "mlsx.h"
#include "api_dir.h"
#include "mysleep.h"
#include <sys/stat.h>


buffer io_i;
buffer io_o;
static buffer io_d;
static stralloc io_d_mem;
int data_sock=-1;
static strhash hash_ids;
static strhash hash_fns;
static const char *o_user;
static const char *o_pass;
static const char *o_acct;
unsigned long o_loglevel=1;
unsigned long o_login_sleep=5;
static int may_mlsx=1;
static int o_do_delete=1;
static int o_dry_run=0;
static int o_symlink_hack=0;
static int o_directories_only=0;
static int o_tolower=0;
static int o_mdtm=0;
static const char *o_list_options=0;
static stralloc o_exclude=STRALLOC_INIT;
static unsigned long o_max_days=0;
static unsigned long o_max_deletes=0;
static unsigned long count_deletes=0;
static unsigned long o_tries=1;
static unsigned long o_rate_limit=0;
static int o_interactive;
static int o_ignore_time;
static int o_ignore_size;
unsigned long o_timeout=30;
stralloc remoteip;
int o_force_select;
static int o_bps;

static int loop(stralloc *r_dir, stralloc *l_dir);
static void usage(void) attribute_noreturn;

MKTIMEOUTREAD(o_timeout)

/* static void logmem(char *s, size_t len) { LOGMEM(s,len); LOGFLUSH(); } */

static int do_symlink(const char *f,const char *t)
{
	static stralloc s=STRALLOC_INIT;
	unsigned int l1=0;
	unsigned int slash=0;
	const char *p;
	if (o_dry_run)
		return 0;
	while (f[l1] && t[l1]==f[l1]) {
		if (t[l1]=='/') slash=l1;
		l1++;
	}
	if (!f[l1] && t[l1]=='/')
		slash=l1;

	if (!slash) {
		if (!stralloc_cats(&s,f) || !stralloc_0(&s)) oom();
		return symlink(s.s,t);
	}
	if (!f[l1]) {
		static stralloc x;
		if (!stralloc_copys(&x,f)) oom();
		if (!stralloc_append(&x,"/")) oom();
		if (!stralloc_0(&x)) oom();
		f=x.s;
	}
	slash++;
	p=t+slash+1;
	s.len=0;
	while (*p) {
		if (*p=='/') 
			if (!stralloc_cats(&s,"../")) oom();
		p++;
	}
	if (!stralloc_cats(&s,f+slash)) oom();
	if (s.len==0) if (!stralloc_cats(&s,".")) oom();
	if (!stralloc_0(&s)) oom();
	return symlink(s.s,t);
}

static stralloc *
canon(stralloc *l_dir) /* feed me with 0 terminated strings only */
{
	static stralloc c=STRALLOC_INIT;
	char *p,*end,*w;
	if (!stralloc_copy(&c,l_dir)) oom();
	if (!stralloc_readyplus(&c,l_dir->len+1)) oom();
	p=l_dir->s;
	end=l_dir->s+l_dir->len;
	w=c.s;

	while (*p) {	
		if (p[0]=='/' && p[1]=='.' && p[2]=='.' && (!p[3] || p[3]=='/')) {
			while (w!=c.s && w[-1]!='/') w--;
			*w='/';
			p+=4;
		} else if (p[0]=='/' && p[1]=='/')
			p++;
		else
			*w++=*p++;
	}
	c.len=w-c.s;
	if (!stralloc_0(&c)) oom();
	return &c;
}

static void
hash_it(struct ftpparse *x, stralloc *l_dir, int id_also)
{
	stralloc *t;
	t=canon(l_dir);

	if (1!=strhash_enter(&hash_fns,1,t->s,t->len,0,0,0))
		oom();

	if (!x->idlen || !id_also) {
		if (1!=strhash_enter(&hash_ids,
				1,t->s,t->len,
				1,t->s,t->len)) 
			oom();
	} else {
		if (1!=strhash_enter(&hash_ids,
				1,x->id,x->idlen,
				1,t->s,t->len)) 
			oom();
	}
}

static void
handle_rate_limit(struct taia *start, unsigned long bytes)
{
	struct taia now;
	struct taia diff;
	double sec;
	double bps;
	double slp;

	/* no use */
	if (bytes < 65536) return;
	if (bytes < o_rate_limit) return;

	taia_now(&now);
	taia_sub(&diff,&now,start);
	sec=taia_approx(&diff);
	if (sec<=0.001)
		sec=0.01;

	bps=bytes/sec;
	if (bps <= o_rate_limit) return;

	slp=sec * (bps/(double)o_rate_limit)-sec;
	if (slp <= 0) return;
	if (slp > 1)
		/* try to not cause timeouts: this should get the TCP recv windows
		 * empty enough after a few seconds.
		 */
		slp=1;

	mysleep(slp);
}

static void
remove_dir (stralloc *s)
{
	stralloc sa=STRALLOC_INIT;
	const char *e;
	int flag;
	if (o_dry_run)
		return;
	if (0==unlink(s->s) || errno==error_noent)
		return;

	if (-1==api_dir_read(&sa, s->s))
		xbailout(100,errno,"failed to open/read ",s->s,0,0);

	s->len--;
	if (!stralloc_append(s,"/")) oom();

	for (e=api_dir_walkstart(&sa,&flag);
		e; e=api_dir_walknext(&sa,&flag)) {
		int pos;
		if (e[0]=='.') {
			if (e[1]=='.' && e[2]=='\0') continue;
			if (e[1]=='\0') continue;
		}
		pos=s->len;
		if (!stralloc_cats(s,e)) oom();
		if (!stralloc_0(s)) oom();
		if (-1==unlink(s->s))
			remove_dir(s);
		s->len=pos;
	}
	api_dir_free(&sa);
	s->len--; /* remove trailing / */
	s->s[s->len++]=0;
	if (-1==rmdir(s->s))
		xbailout(100,errno,"failed to rmdir ",s->s,0,0);
}

static int
download(struct ftpparse *x, stralloc *r_dir, stralloc *l_dir)
{
	char *p;
	int fd;
	const char *e;
	const char *slash;
	const char *cc;
	static stralloc tmpfn=STRALLOC_INIT;
	buffer save;
	static stralloc savemem=STRALLOC_INIT;
	struct stat st;
	char *fptr;
	unsigned int flen;
	int found;
	struct taia start;
	unsigned long bytes=0;
	time_t mtime;
	int pasv_retry=0;

	if (o_directories_only) {
		hash_it(x,l_dir,1);
		return 0;
	}
	if (o_bps || o_rate_limit)
		taia_now(&start);

	e=l_dir->s+l_dir->len;
	for (cc=slash=l_dir->s;cc!=e;cc++)
		if (*cc=='/')
			slash=cc;
	if (*slash=='/') slash++;
	if (!stralloc_copys(&tmpfn,".tmp.")
		|| !stralloc_catb(&tmpfn,slash,e-slash)
		|| !stralloc_0(&tmpfn)) oom();
	unlink(tmpfn.s);

	mtime=x->mtime;
	if (o_mdtm) {
		time_t t;
		switch (modtime_request(r_dir->s, &t))
		{ 
		case -1: o_mdtm=0; break;
		case 0:  break;
		case 1:  mtime=t; break;
		}
	}

	found=strhash_lookup(&hash_ids,x->id,x->idlen,&fptr,&flen);

	if (0==stat(l_dir->s,&st)) {

		int identical=1;
		if (!o_ignore_size && st.st_size != x->size)   identical=0;
		if (!o_ignore_time && st.st_mtime != x->mtime) identical=0;

		if (identical) {
			if (found) {
				if (0==link(fptr,tmpfn.s)
					|| 0==do_symlink(fptr,tmpfn.s))  {
					/* delete this in case both files are links to the
					 * same inode */
					unlink(l_dir->s);
					if (-1==rename(tmpfn.s,l_dir->s)) {
						int er=errno;
						unlink(tmpfn.s);
						xbailout(100,er,"failed to rename ", tmpfn.s,
								" to ",l_dir->s);
					}
					hash_it(x,l_dir,0);
					if (o_loglevel > 1)	
						log2(l_dir->s, ": is a link\n");
					return 1;
				}
				xbailout(100,errno,"failed to create link from ",
						fptr," to ", l_dir->s);
			}
			hash_it(x,l_dir,!found);
			if (o_loglevel > 1)	
				log2(l_dir->s, ": identical file found\n");
			return 1;
		}
		if (o_loglevel > 2)	{
			char nb[FMT_ULONG];
			log2(l_dir->s, ": lfacts: ");
			nb[fmt_ulong(nb,st.st_mtime)]=0;
			log2(nb," ");
			nb[fmt_ulong(nb,st.st_size)]=0;
			log2(nb,"\n");
		}
	}
	
	if (found) {
		if (o_dry_run)	 {
			if (o_loglevel > 1)
				log4(l_dir->s, "would be (sym)linked to ",fptr,"\n");
			return 1;
		}
		if (0==link(fptr,tmpfn.s)
			|| 0==do_symlink(fptr,tmpfn.s))  {
			if (-1==rename(tmpfn.s,l_dir->s)) {
				int er=errno;
				unlink(tmpfn.s);
				xbailout(100,er,"failed to rename ", tmpfn.s," to ",l_dir->s);
			}
			if (o_loglevel > 1)
				log4(l_dir->s, " (sym)linked to ",fptr,"\n");
			hash_it(x,l_dir,1);
			return 1;
		}
	}
	if (o_max_days && mtime+o_max_days*86400<uo_now()) {
		hash_it(x,l_dir,1);
		if (o_loglevel > 1)	
			log2(l_dir->s, ": too old\n");
		return 1;
	}
	if (o_dry_run) {
		if (o_loglevel)
			log2(l_dir->s, ": dry-run non-download successful\n");
		hash_it(x,l_dir,1);
		return 1;
	}


/* modify by Kent	*/
/* chmod mode to 664	*/
//	fd=open_trunc_mode(tmpfn.s,0644);
	fd=open_trunc_mode(tmpfn.s,0664);
/* end	*/
	if (fd==-1) xbailout(100,errno,"failed to open_trunc ",tmpfn.s,0,0);
	if (!stralloc_ready(&savemem,BUFFER_OUTSIZE)) oom();
	buffer_init(&save,(buffer_op_t)write,fd,savemem.s,BUFFER_OUTSIZE);

  retry_pasv:
	if (data_sock==-1)
		data_sock=do_pasv();
	cmdwrite2("RETR ",r_dir->s);
	p=ccread();
	if (!p) xbailout(111,errno,"failed to read RETR answer",0,0,0);
	if (*p!='1') {
		if (!pasv_retry++ && str_start(p,"425")) {
			/* at least one ftp server seems to close the data connection
			 * in case a RETR is done on a directory. So work around ...
			 */
			close(data_sock);
			data_sock=-1;
			goto retry_pasv;
		}
		warning(0,"got unwanted answer to `RETR ",r_dir->s,"': ",p);
		unlink(tmpfn.s);
		close(fd);
		return 0;
	}
	buffer_init(&io_d,(buffer_op_t)TIMEOUTREADFN(o_timeout),data_sock,
		io_d_mem.s,BUFFER_INSIZE);
	while (1) {
		int l;
		char *q;
		l=buffer_feed(&io_d);
		if (l==-1) xbailout(111,errno,"failed to read from remote",0,0,0);
		if (l==0) break;
		bytes+=l;
		q=buffer_peek(&io_d);
		if (-1==buffer_put(&save,q,l)) 
			xbailout(111,errno,"failed to write to ",tmpfn.s,0,0);
		buffer_seek(&io_d,l);
		if (o_rate_limit) 
			handle_rate_limit(&start,bytes);
	}
	close(data_sock);
	x2("RETR finish");
	if (buffer_flush(&save)) 
		xbailout(111,errno,"failed to write to ",tmpfn.s,0,0);
	data_sock=-1;
	if (-1==fsync(fd)) 
		xbailout(111,errno,"failed to fsync ",tmpfn.s,0,0);

	if (-1==api_futimes_1(fd,mtime,0,mtime,0))
		warning(errno,"failed to futimes ",tmpfn.s,0,0);
	if (-1==close(fd)) 
		xbailout(111,errno,"failed to close ",tmpfn.s,0,0);

	/* some version of reiserfs on linux didn't honor utimes() if it was 
	 * called before the close(). */
	if (-1==api_futimes_2(tmpfn.s,mtime,0,mtime,0))
		warning(errno,"failed to utimes ",tmpfn.s,0,0);

	if (-1==rename(tmpfn.s,l_dir->s)) {
		remove_dir(l_dir);
		if (-1==rename(tmpfn.s,l_dir->s)) {
			int er=errno;
			unlink(tmpfn.s);
			xbailout(100,er,"failed to rename ", tmpfn.s," to ",l_dir->s);
		}
	}
	if (x->size!=(long) bytes) {
		char nb[FMT_ULONG];
		/* This _most_ often means that the connection was closed 
		 * suddenly.
		 * Note that it also can mean that the file has been changed 
		 * between the getting of the listing and the download, and that
		 * everything is fine now. No, that's not unlikely. Try mirroring
		 * a big directory over a slow connection.
		 * It's hard to do the "right" thing here.
		 * XXX get a listing of that file and check the size? 
		 */
		nb[fmt_ulong(nb,x->size)]=0;
		log2(l_dir->s,": warning: expected ");
		log2(nb," bytes, but got ");
		nb[fmt_ulong(nb,bytes)]=0;
		log2(nb,"\n");
	}
	hash_it(x,l_dir,1);
	if (o_loglevel)
		log2(l_dir->s, ": download successful");
	if (o_bps) {
		struct taia stop;
		struct taia diff;
		unsigned long sec;
		char nb[FMT_ULONG];
		unsigned long bps;
		const char *what;
		/* could use double arithmetics, but what for? */
		taia_now(&stop);
		taia_sub(&diff,&stop,&start);
		sec=taia_approx(&diff);
		if (sec==0)
			sec=1;
		log1(", ");
		bps=bytes/sec;
		if (bps<10000) { nb[fmt_ulong(nb,bps)]=0; what=" B/s"; }
		else if (bps/1024<10000) { nb[fmt_ulong(nb,bps/1024)]=0;what=" KB/s";}
		else { nb[fmt_ulong(nb,bps/(1024*1024))]=0; what=" MB/s"; }
		log2(nb,what);
	}
	if (o_loglevel)
		log1("\n");
	return 1;
}

static int
handle_exclude (stralloc *ca)
{
	char *end=o_exclude.s+o_exclude.len;
	char *ptr=o_exclude.s;
	const char *info=0; /* keep gcc quiet */
	int exclude=0;
	while (ptr!=end) {
		int flag=(*ptr=='-');
		ptr++;
		if (uo_wildmat(ptr,ca->s,ca->len-1)) {
			if (exclude != flag) {
				exclude=flag;
				info=ptr;
				if (o_loglevel > 3) {
					write(1,ca->s,ca->len-1); 
					log4(": matched `",info,"', -> ",
						exclude ? "exclude" : "include");
					write(1,"\n",1);
				}
			}
		}
		ptr+=str_len(ptr)+1;
	}
	if (!info)
		info="end-of-list (default)";
	if (exclude) {
		if (o_loglevel > 1) {
			write(1,ca->s,ca->len-1); 
			log3(": excluded, matched `",info,"'\n");
		}
		return 1;
	}
	if (o_loglevel > 3) {
		write(1,ca->s,ca->len-1); 
		log3(": included, matched `",info,"'\n");
	}
	return 0;
}

static int
handle_hackish_symlink(struct ftpparse *x, stralloc *l_dir, 
	char *p, unsigned int l)
{
	char *q;
	char *end;
	q=p;
	end=q+l-4;
	while (q!=end) {
		if (*q==' ' && q[1]=='-'
			&& q[2]=='>' && q[3]==' ')
			{ q+=4 ; break; }
		q++;	
	}
	if (q!=end) {
		unsigned int pos;
		stralloc t=STRALLOC_INIT;
		stralloc idstr=STRALLOC_INIT;
		if (!stralloc_copy(&t,l_dir)) oom();
		t.s[t.len-1]='/';
		pos=t.len;
		if (!stralloc_catb(&t,q,str_len(q)) || !stralloc_0(&t)) 
			oom();
		if (o_tolower)
			case_lowers(t.s+pos);
		if (o_loglevel > 1) {
			log1("symlink '"); 
			write(1,t.s,t.len-1); 
			log1("'\n");
		}
		if (!stralloc_copy(&idstr,canon(&t))) oom();
		if (!stralloc_copy(&t,l_dir)) oom();
		t.s[t.len-1]='/';
		if (!stralloc_catb(&t,x->name,x->namelen)) oom();
		if (!stralloc_0(&t)) oom();
		unlink(t.s);
		rmdir(t.s);
		if (0!=do_symlink(idstr.s,t.s))  {
			if (errno==error_isdir) 
				remove_dir(&t);
			if (0!=do_symlink(idstr.s,t.s))
				xbailout(100,errno,"failed to create symlink "
					"from ", idstr.s, " to ",t.s);
		}
		hash_it(x,&t,1);
		stralloc_free(&t);
		stralloc_free(&idstr);
		return 1;
	}
	return 0;
}

static int
handle_directory(struct ftpparse *x, stralloc *r_dir, stralloc *l_dir)
{
	int found;
	char *fptr;
	unsigned int flen;
	int done=0;
	unsigned int lpos=l_dir->len;

	l_dir->s[l_dir->len-1]='/';
	if (!stralloc_catb(l_dir,x->name,x->namelen)
		|| !stralloc_0(l_dir)) oom();
	if (o_tolower)
		case_lowers(l_dir->s+lpos);
	found=strhash_lookup(&hash_ids,x->id,x->idlen,&fptr,&flen);
	if (found) {
		if (o_loglevel > 1)
			log4(l_dir->s,": (sym)linking, ID identical to `",
				fptr,"'\n");
		remove_dir(l_dir);
		if (0!=do_symlink(fptr,l_dir->s)) 
			xbailout(111,errno,"failed to create symlink from ",
				fptr, " to ",l_dir->s);
		hash_it(x,l_dir,1);
		done=1;
	} else {
		/* remote is a new directory. */
		/* local may be a symlink to a directory we already saw */
		struct stat st;
		if (0==lstat(l_dir->s,&st))
			if (!S_ISDIR(st.st_mode))
				unlink(l_dir->s);

		/* hash it now: "dir" -> "." symlinks */
		hash_it(x,l_dir,1);

		if (1==loop(r_dir, l_dir)) {
			if (!o_dry_run) 
				if (-1==api_utimes(l_dir->s,
					x->mtime,0, x->mtime, 0))
					warning(errno,"failed to call utimes on ",l_dir->s,0,0);
			done=1;
		}
	}
	l_dir->len=lpos;
	l_dir->s[l_dir->len-1]='\0';
	return done;
}

static int 
loop(stralloc *r_dir, stralloc *l_dir)
{
	char *p,*e;
	stralloc dirdata=STRALLOC_INIT;
	int olddirfd;
	int listno;
	int pasv_retries=0;

	cmdwrite2("CWD ",r_dir->s);

	p=ccread();
	if (!p) xbailout(111,errno,"failed to read CWD answer",0,0,0);
	if (*p!='2') 
		return 0;
	if (may_mlsx)
		listno=0;
	else
		listno=1;
  retry_listing_pasv:
	if (data_sock==-1)
		data_sock=do_pasv();
  retry_listing:
	if (listno==0)
		cmdwrite1("MLSD");
	else if (o_list_options)
		cmdwrite2("LIST ", o_list_options);
	else
		cmdwrite1("LIST");
	p=ccread();
	if (!p) xbailout(111,errno,"failed to read LIST answer",0,0,0);
	if (!pasv_retries++ && str_start(p,"425")) {
		/* in case ftp server lost track of the open PASV connection */
		close(data_sock);
		data_sock=-1;
		goto retry_listing_pasv;
	}
	if (listno==0 && *p=='5') {
		listno=1;
		if (!str_start(p,"501")) /* 501: MSLD on file */
			may_mlsx=0;
		goto retry_listing;
	}
	if (*p!='1')
		return 0;
	olddirfd=open_read(".");

	if (!o_dry_run)
		if (-1==chdir(l_dir->s)) {
			if (errno!=error_exist)
				remove_dir(l_dir);
/* modify by Kent	*/
//			if (-1==mkdir(l_dir->s,0755)) 
			if (-1==mkdir(l_dir->s,0777))
/* end	*/
				xbailout(100,errno,"failed to mkdir ",l_dir->s,0,0);
			if (-1==chdir(l_dir->s)) 
				xbailout(100,errno,"failed to chdir ",l_dir->s,0,0);
		}

    if (-1==ftp_read_list(data_sock,&dirdata))
		xbailout(111,errno,"failed to read remote directory",0,0,0);

	close(data_sock);
	data_sock=-1;
	x2("LIST");

	p=dirdata.s;
	e=dirdata.s+dirdata.len;
	while (p!=e) {
		struct ftpparse x;
		size_t l=str_len(p);
		int ok;
		if (may_mlsx)
			ok=mlsx_parse(&x,p,l);
		else
			ok=ftpparse(&x,p,l);
		if (!ok)
			log3("cannot parse LIST line: ",p,"\r\n");
		else if (x.name[0]=='.'
			&& (x.namelen==1 || (x.namelen==2 && x.name[1]=='.'))) {
			if (o_loglevel > 1) {
				log2(l_dir->s,"/");
				if (o_tolower)
					case_lowerb(x.name,x.namelen);
				logmem(x.name,x.namelen);
				log1(": ignored\n");
			}
		} else if (x.name[byte_chr(x.name,x.namelen,'/')] == '/') {
			/* file name with a slash in it: no good.
			 * Can't happen and has security implications. */
			log2(l_dir->s,"/");
			if (o_tolower)
				case_lowerb(x.name,x.namelen);
			logmem(x.name,x.namelen);
			log1(": ignored\n");
		} else {
			size_t rpos=r_dir->len;
			size_t lpos=l_dir->len;
			int done=0;
			stralloc *ca;
			if (o_loglevel > 2) {
				char nb1[FMT_ULONG];
				char nb2[FMT_ULONG];
				static stralloc lo;
				nb1[fmt_ulong(nb1,x.mtime)]=0;
				nb2[fmt_ulong(nb2,x.size)]=0;
				if (!stralloc_copyb(&lo,x.name,x.namelen)) oom();
				if (o_tolower)
					case_lowerb(lo.s,lo.len);
				log2(l_dir->s,"/");
				logmem(lo.s,lo.len);
				log4(": facts: ",nb1," ",nb2); log1("\n");
			}
			
			r_dir->s[r_dir->len-1]='/';
			if (!stralloc_catb(r_dir,x.name,x.namelen)
				|| !stralloc_0(r_dir)) oom();
			ca=canon(r_dir);
			if (o_exclude.len)
				done=handle_exclude(ca);
			if (!done && l>4 && *p=='l' && o_symlink_hack)
				done=handle_hackish_symlink(&x,l_dir,p,l);
			if (x.flagtrycwd && !done)
				done=handle_directory(&x,r_dir,l_dir);
			if (x.flagtryretr && !done) {
				l_dir->s[l_dir->len-1]='/';
				if (!stralloc_catb(l_dir,x.name,x.namelen)
					|| !stralloc_0(l_dir)) oom();
				if (o_tolower)
					case_lowers(l_dir->s+lpos);
				if (1==download(&x,r_dir, l_dir))
					done=1;
				l_dir->len=lpos;
				l_dir->s[l_dir->len-1]='\0';
			}
			r_dir->len=rpos;
			r_dir->s[r_dir->len-1]='\0';
		}
		p+=l+1;
	}
	stralloc_free(&dirdata);
	if (-1==fchdir(olddirfd)) /* ECANTHAPPEN, but we'd be in trouble */
		xbailout(111,errno,"failed to fchdir to upper level directory",0,0,0);
	close(olddirfd); /* ECANTHAPPEN */
	return 1; /* done */
}

static void
cwd_slash(void)
{
	char *p;
	cmdwrite1("CWD /");
	p=ccread();
	if (!p) xbailout(111,errno,"failed to read 'CWD /' answer",0,0,0);
	if (*p!='2')
		xbailout(100,0,"failed to 'CWD /': ",p,0,0);
}

/* try to find out information about the remote start directory.
 * Due to LIST ambiguities this will only work reliably with 
 * publicfile. Be paranoid.
 */
static void
initialdirectory(stralloc *dirdata, struct ftpparse *fp, stralloc *r_dir)
{
	char *p;
	int count;
	if (data_sock==-1)
		data_sock=do_pasv();

	cmdwrite2("LIST ",r_dir->s);
	p=ccread();
	if (!p) xbailout(111,errno,"failed to read LIST answer",0,0,0);
	if (*p!='1')
		xbailout(111,errno,"failed to read initial directory LIST answer",
			0,0,0);
	count=ftp_read_list(data_sock,dirdata);
	close(data_sock);
	data_sock=-1;
	x2("LIST");
	if (1!=count) 
		/* XXX could try LIST .. and parse the answer */
		return; /* doesn't help us. */
	if (1!=ftpparse(fp,dirdata->s,str_len(dirdata->s)))
		return;
	if (fp->namelen!=(int) r_dir->len-1
		|| byte_diff(fp->name,fp->namelen,r_dir->s))
		fp->idlen=0; /* used in onedirpair() */
}
/* try to find out whether the remote target is a file or directory.
 * we can't reliably use LIST: 
 * "LIST xxx" will return a list with one element "xxx" if xxx is a file
 * or "xxx" is a directory with exactly one file, which is called "xxx".
 *
 * We _could_ use LIST with an EPLF supporting FTP server, but we cannot
 * reliable detect that.
 */
static int 
initialentity(struct ftpparse *fp, stralloc *r_dir)
{
	int count=0;
	char *p;
	static stralloc dirdata=STRALLOC_INIT; /* we return a pointer into that */

	cwd_slash();
	cmdwrite2("MLST ",r_dir->s);
	p=ccread_oneline();
	if (!p) xbailout(111,errno,"failed to read MLST answer",0,0,0);
	if (*p=='2') {
		while (1) {
			p=ccread_oneline();
			if (str_start(p,"250 "))
				break;
			if (str_start(p,"250-"))
				continue;
			if (!stralloc_copys(&dirdata,p+1)) oom();
		}
		mlsx_parse(fp,dirdata.s,dirdata.len);
		if (fp->flagtrycwd && !fp->flagtryretr)
			return 1;
		return 0;
	}
	if (str_len(p)<4 && p[4]=='-') {
		/* we were in single-line mode */
		p=ccread();
		if (!p) xbailout(111,errno,"failed to read MLST answer",0,0,0);
	}


	cmdwrite2("CWD ",r_dir->s);
	p=ccread();
	if (!p) xbailout(111,errno,"failed to read CWD answer",0,0,0);
	if (*p=='2') {
		/* may be a directory.
		 * Note: publicfile allows "CWD" to a file, but LIST fails.
		 */
		if (data_sock==-1)
			data_sock=do_pasv();

		cmdwrite1("LIST");
		p=ccread();
		if (!p) xbailout(111,errno,"failed to read LIST answer",0,0,0);
		if (*p=='1') {
			count=ftp_read_list(data_sock,&dirdata);
			close(data_sock);
			data_sock=-1;
			x2("LIST");
			cwd_slash();
			if (-1!=count) {
				/* it's a directory */
				initialdirectory(&dirdata,fp,r_dir);
				return 1;
			}
		}
		close(data_sock);
		data_sock=-1;
		cwd_slash();
	}

	if (data_sock==-1)
		data_sock=do_pasv();

	cmdwrite2("LIST ",r_dir->s);
	p=ccread();
	if (!p) xbailout(111,errno,"failed to read LIST answer",0,0,0);
	if (*p!='1') xbailout(100,0,"failed to LIST the remote directory: ",p,0,0);

    count=ftp_read_list(data_sock,&dirdata);
	if (-1==count)
		xbailout(111,errno,"cannot read remote directory",0,0,0);
	close(data_sock);
	data_sock=-1;
	x2("LIST");
	if (count==0) /* typically happens if first entity is nonexistant */
		xbailout(111,0,"remote file/directory doesn't exist",0,0,0);
	if (count!=1)
		xbailout(111,0,"too many entries return by 'LIST ",r_dir->s,"'",0);
	/* might be a directory with one file, or a file */
	/* try to get the date. We need it later _if_ it is a file */
	ftpparse(fp,dirdata.s,dirdata.len);
	fp->flagtrycwd=0;
	fp->flagtryretr=1;
	return 0;
}



static void
delete(stralloc *dn)
{
	stralloc sa=STRALLOC_INIT;
	const char *e;
	int flag;
	size_t pos;
	if (-1==api_dir_read(&sa,dn->s))
		xbailout(111,errno,"failed to open/read ",dn->s,0,0);
	dn->len--;
	pos=dn->len;
	for (e=api_dir_walkstart(&sa,&flag);
		e; e=api_dir_walknext(&sa,&flag)) {
		struct stat st;
		int found;

		stralloc *t;
		dn->len=pos;
		if (e[0]=='.'
			&& (!e[1]
				|| (e[1]=='.' && e[2]==0)))
			continue;
		if (!stralloc_append(dn,"/")) oom();
		if (!stralloc_cats(dn,e)) oom();
		if (!stralloc_0(dn)) oom();
		if (-1==lstat(dn->s,&st))
			continue;
		if (S_ISDIR(st.st_mode)) {
			delete(dn);
			t=canon(dn);
			found=strhash_lookup(&hash_fns,t->s,t->len,0,0);
			if (!found) {
				if (-1==rmdir(dn->s))
					xbailout(100,errno,"failed to rmdir ",dn->s,0,0);
				if (o_loglevel)
					log2(dn->s,": not found on remote, deleted\n");
			} else {
				if (o_loglevel>3)
					log2(dn->s,": found on remote\n");
			}
			continue;
		}
		t=canon(dn);
		found=strhash_lookup(&hash_fns,t->s,t->len,0,0);
		if (!found) {
			if (o_max_deletes)
				if (count_deletes++ >= o_max_deletes) {
					if (o_loglevel)
						log2(dn->s,": not deleted due to --max-deletes\n");
					continue;
				}
			if (-1==unlink(dn->s)) {
				unsigned int er=errno;
				if (-1==rmdir(dn->s))
					xbailout(111,er,"failed to unlink ",dn->s,0,0);
			}
			if (o_loglevel)
				log2(dn->s,": not found on remote, deleted\n");
		} else {
			if (o_loglevel>3)
				log2(dn->s,": found on remote\n");
		}
	}
	dn->len=pos;
	if (!stralloc_0(dn)) oom();
	api_dir_free(&sa);
}

static void 
usage(void) 
{
	xbailout(2,0,
	  "usage: ftpcopy [options] host[:port] remotedir [localdir]\n"
	  "   or: ftpcopy [options] ftp://host[:port]/remotedir [localdir]\n"
	  "  use the --help option to get a description of the options."
	  ,0,0,0);
}

static void
callback_exclude(uogetopt_t *g, const char *s)
{
	(void) g;
	if (!stralloc_catb(&o_exclude,"-",1)
		|| !stralloc_cats(&o_exclude,s)
		|| !stralloc_0(&o_exclude)) oom();
}
static void
callback_include(uogetopt_t *g, const char *s)
{
	(void) g;
	if (!stralloc_catb(&o_exclude,"+",1)
		|| !stralloc_cats(&o_exclude,s)
		|| !stralloc_0(&o_exclude)) oom();
}

static uogetopt_t myopts[] =
{
{ 0, "", UOGO_TEXT, 0,0, "Login / username / password options:",0},
{'u', "user", UOGO_STRING, &o_user,0,
"Use NAME to login on the ftp server.\n"
"The default is `anonymous'. Use an empty name\n"
"to avoid a login.","NAME"},
{'p', "pass", UOGO_STRING, &o_pass,0,
"Use PASS as password to login on the ftp server.\n"
"The default is `anonymous@invalid.example'.","PASSWORD"},
{  0, "account", UOGO_STRING, &o_acct,0,
"Send ACCOUNT as account name during login phase.\n"
"Note: this is _not_ the user name, but the name\n"
"of what could be called a subaccount implemented\n"
"by a few servers. Use the -u (--user) option to\n"
"set the login name.","ACCOUNT"},
{'\0', "tries", UOGO_ULONG, &o_tries,0,
"Number of tries to connect and log in.\n"
"The default is 1.",0},
{'\0', "login-sleep", UOGO_ULONG, &o_login_sleep,0,
"Seconds to sleep after a failed login.\n"
/*2345678901234567890123456789012345678901234567890 */
"More precisely: ftpcopy will fall to sleep for\n"
"this many seconds after a try to connect or login\n"
"has failed. The default is 5.",0},

{ 0, "", UOGO_TEXT, 0,0, "Verbosity options:",0},
{'l', "loglevel", UOGO_ULONG, &o_loglevel,0,
"Controls the amount of logging done.\n"
"0: nothing except warnings and error messages.\n"
"1: downloads and deletes.\n"
"2: links/symlinks created, files we already got.\n"
"3: useless stuff.",0},
{0,"bps", UOGO_FLAG, &o_bps,1, 
"Log transfer rates.\n"
"This option causes ftpcopy to log byte / kilobyte /\n"
"megabyte per second information after successful\n"
"transfers.",0},

{ 0, "", UOGO_TEXT, 0,0, "File selection options:",0},
{'m', "max-days", UOGO_ULONG, &o_max_days,0,
"Restrict on modification time.\n"
"Download only files modified in the last\n"
"MAX days.","MAX"},
{'x', "exclude", UOGO_CALLBACK, callback_exclude,0,
"Exclude paths matching WILDCARD.\n"
"If WILDCARD matches the full path of the remote\n"
"file then the file will not be downloaded.\n"
/*2345678901234567890123456789012345678901234567890 */
"Use this option more than once to exclude multiple\n"
"patterns.","WILDCARD"},
{'i', "include", UOGO_CALLBACK, callback_include,0,
"Include paths matching WILDCARD.\n"
"If WILDCARD matches the full path of the remote\n"
"file then the file will be downloaded.\n"
"Use this option more than once to exclude multiple\n"
"patterns.\n"
"If both includes and excludes are used then the\n"
"last matching one will be honored. The list starts\n"
"with an implicit '--include *'","WILDCARD"},
{0, "ignore-time", UOGO_FLAG, &o_ignore_time,1,
"Ignore modification times.\n"
"... when comparing local and remote files.\n"
"This option may be used to ensure not to download\n"
"files if their modification time has changed.",0},
{0, "ignore-size", UOGO_FLAG, &o_ignore_size,1,
"Ignore file size.\n"
"... when comparing local and remote files.\n"
"This option may be used to not download files if\n"
"their size changed.",0},

{ 0, "", UOGO_TEXT, 0,0, "Deletion options:",0},
{'n', "no-delete", UOGO_FLAG, &o_do_delete,0,
"Do not delete files.\n"
"This influences the cleanup step when getting rid\n"
"of things the server doesn't have anymore, it does\n"
"not stop ftpcopy from deleting files in it's way\n"
"during downloads.",0},
{ 'M', "max-deletes", UOGO_ULONG, &o_max_deletes,0,
"Do not delete more then COUNT files.\n"
"This option may be useful to limit the impact\n"
"of a tempoary loss of files on the server.\n"
"The default is 0, meaning unlimited.\n"
"This, too, only influences the cleanup step.\n","COUNT"},

{ 0, "", UOGO_TEXT, 0,0, "Operational options:",0},
{'d', "directories-only", UOGO_FLAG, &o_directories_only,1,
"Only create the directory hierarchie.\n"
"Do not download files.\n"
"Any file in the tree will be deleted unless\n"
"the -n option is also given.",0},
{0, "dry-run", UOGO_FLAG, &o_dry_run,1,
"Don't do anything.\n"
"ftpcopy will only show what would be done.",0},
{'T', "timeout", UOGO_ULONG, &o_timeout,0,
"Timeout to use for read/write (sec.).\n"
"The default is 30 seconds.",0},
{0, "rate-limit", UOGO_ULONG, &o_rate_limit,0,
"Limit file download speed (bytes/sec).\n"
"This implements a very simple mechanism to limit\n"
"the transfer rate. The default is unlimited.",0},
{0, "interactive", UOGO_FLAG, &o_interactive,1,
"Read directories from stdin.\n"
"Two lines are read from the standard input for each\n"
"operation. The first line contains the remote\n"
"directory and the second one the local directory.\n"
"ftpcopy is ready to do another copy as soon as it\n"
"prints an END-OF-COPY line.",""},

{ 0, "", UOGO_TEXT, 0,0, "Workaround options:",0},
{'L', "list-options", UOGO_STRING, &o_list_options,1,
"Add OPTS to LIST command.\n"
"This allows to pass arbitrary options to the\n"
"FTP servers LIST command. Note that ftpcopy does\n"
"not cope well with recursive directory listings.","OPTS"},
{'s', "symlink-hack", UOGO_FLAG, &o_symlink_hack,1,
"Deal with symbolic links.\n"
"Only useful to mirror sites which create listings\n"
"through /bin/ls. Will fail if a file name in a\n"
"link contains a ` -> ' sequence.",0},
{0, "force-select", UOGO_FLAG, &iopause_force_select,1,
"Use select, not poll.\n"
"This works around a limitation of the socks5\n"
"reference implementation. Use this only if you\n"
"are using SOCKS. Note: Direct access to a DNS\n"
"resolver is needed if you use host names.",0},
{0, "mdtm", UOGO_FLAG, &o_mdtm,1,
"Use the MDTM command to get the remote time.\n"
"The default is to take the files timestamp from the\n"
"directory listing. Many servers don't provide time\n"
"zone information in them. While ftpcopy by default\n"
"assumes that the server runs in UTC this sometimes\n"
"doesn't work (it works most of the time, though).\n"
"The MDTM command offered by a growing number of\n"
"servers provides a workaround, but costs some\n"
"performance.",0},
{0, "tolower", UOGO_FLAG, &o_tolower,1,
"Change local names to lowercase.\n"
"Note: this option is slightly dangerous - misuse\n"
"can lead to incremented bandwidth usage.",0},

{0, 0, 0, 0, 0, 0, 0}           /* --help and --version */
};

static void
chdir_mkdir(const char *s)
{
	if (-1==chdir(s)) {
/* modify by Kent	*/
//		if (-1==mkdir(s,0755)) {
		if (-1==mkdir(s,0777)) {
/* end */
			if (errno!=error_exist) 
				xbailout(111,errno,"failed to mkdir ",s,0,0);
		}
		if (-1==chdir(s))
			xbailout(111,errno,"failed to chdir to ",s,0,0);
	}
}

static int
onedirpair(stralloc *remote, stralloc *local)
{
	struct stat st;
	struct ftpparse fp;
	strhash_destroy(&hash_ids);
	strhash_destroy(&hash_fns);
	if (-1==strhash_create(&hash_ids,16,32,strhash_hash)) oom();
	if (-1==strhash_create(&hash_fns,16,32,strhash_hash)) oom();
	byte_zero((char *)&fp,sizeof(fp));
	if (remote->s[0]!='/') {
		/* i like absolute paths. */
		stralloc t=STRALLOC_INIT;
		if (!stralloc_copy(&t,remote)) oom();
		if (!stralloc_copyb(remote,"/",1)) oom();
		if (!stralloc_cat(remote,&t)) oom();
		stralloc_free(&t);
	}
	if (!initialentity(&fp,remote)) {
		/* 'remote' is a file */
		unsigned int slash;
		/* 'local' now is a file, too */
		slash=str_rchr(local->s,'/');
		local->s[slash]=0;
		chdir_mkdir(local->s);
		local->s[slash]='/';
		if (0==stat(local->s,&st) && S_ISDIR(st.st_mode)) {
			slash=str_rchr(remote->s,'/');
			local->len--;
			if (local->s[local->len-1]=='/')
				slash++; /* don't need a seconds / */
			if (!stralloc_cats(local,remote->s+slash) 
			 || !stralloc_0(local)) oom();
		}

		if (download(&fp,remote,local))
			return 0;
	}
	/* 'remote' is a directory */
	chdir_mkdir(local->s);
	if (fp.idlen)
		/* for the weird symlinks to the servers root */
		hash_it(&fp,local,1);
	if (loop(remote,local)) {
		if (o_do_delete && !o_dry_run)
			delete(local);
		return 0;
	}
	return 1;
}

int 
main(int argc, char **argv)
{
	char *local_start_dir=0;
	const char *host,*remotedir,*localdir=0;
	stralloc d1=STRALLOC_INIT;
	stralloc d2=STRALLOC_INIT;
	stralloc proto=STRALLOC_INIT;
	stralloc user=STRALLOC_INIT;
	stralloc pass=STRALLOC_INIT;
	stralloc hostport=STRALLOC_INIT;
	stralloc rest=STRALLOC_INIT;
	int ldirfd;

	bailout_progname(argv[0]);
	flag_bailout_fatal_begin=3;

	if (!stralloc_ready(&io_d_mem,BUFFER_INSIZE)) oom();

	uogetopt (flag_bailout_log_name, PACKAGE, VERSION, 
		&argc, argv, uogetopt_out, 
	  "usage: ftpcopy [options] host[:port] remotedir [localdir]\n"
	  "   or: ftpcopy [options] ftp://host[:port]/remotedir [localdir]\n"
	  " localdir defaults to '.'. If it's not given then the -n option "
	  "must be used.\n",
	  myopts,
	  "\nReport bugs to <ftpcopy@bulkmail.ohse.de>");
	if (argc <2 || argc >4)
		usage();
	if (urlparse(argv[1],&proto,&user,&pass,&hostport,&rest)) {
		if (!stralloc_0(&proto)) oom();
		if (!str_equal(proto.s,"ftp")) 
			xbailout(100,0,"URL type `",proto.s,"' is not supported",0);
		if (!hostport.len) xbailout(100,0,"empty host in URL",0,0,0);
		if (!stralloc_0(&hostport)) oom();
		host=hostport.s;
		if (!rest.len) if (!stralloc_append(&rest,"/")) oom();
		if (!stralloc_0(&rest)) oom();
		remotedir=rest.s;
		if (!o_user && user.len) {
			if (!stralloc_0(&user)) oom();
			o_user=user.s;
		}
		if (!o_pass && pass.len) {
			if (!stralloc_0(&pass)) oom();
			o_pass=pass.s;
		}
		if (argv[2]) {
			localdir=argv[2];
			if (argv[3]) usage();
		}
	} else {
		host=argv[1];
		if (!argv[2])
			if (!o_interactive)
				usage();
		remotedir=argv[2];
		if (argv[3])
			localdir=argv[3];
	}
	if (!localdir && !o_interactive) {
		char dotdir[]=".";
		if (o_do_delete)
			xbailout(100,0,
				"default `localdir' (.) not allowed without the -n option",
				0,0,0);
		localdir=dotdir;
	}
	if (!o_user) o_user="anonymous";
	if (!o_pass) o_pass="anonymous@example.invalid";

	if (remotedir) { /* might be empty in interactive mode */
		if (!stralloc_copys(&d1,remotedir)) oom();
		if (!stralloc_0(&d1)) oom();
	}

	local_start_dir=get_cwd();
	if (!local_start_dir)
		xbailout(111,errno,"failed to get current directory",0,0,0);

	ldirfd=open_read(".");
	if (ldirfd==-1)
		xbailout(111,errno,"failed to open . for reading",0,0,0);
	
	connect_auth(host,o_user,o_pass,o_acct,o_tries);

	sx2("TYPE I");
	if (o_interactive) {
		int retcode=0;
		buffer io_stdin;
		char spc[BUFFER_INSIZE];
		buffer_init(&io_stdin,(buffer_op_t)read,0,spc,sizeof(spc));
		while (1) {
			int gotlf;
			char *p;
			if (-1==getln(&io_stdin,&d1,&gotlf,'\n'))
				xbailout(111,errno,"failed to read from stdin",0,0,0);
			if (d1.len==0)
				break;
			if (!gotlf)
				xbailout(111,errno,"unterminated line read from stdin",0,0,0);
			d1.len--;
			if (!stralloc_0(&d1)) oom();

			if (-1==getln(&io_stdin,&d2,&gotlf,'\n'))
				xbailout(111,errno,"failed to read from stdin",0,0,0);
			if (d2.len==0)
				break;
			if (!gotlf)
				xbailout(111,errno,"unterminated line read from stdin",0,0,0);
			d2.len--;
			if (!stralloc_0(&d2)) oom();
			if (d2.s[0]!='/') {
				stralloc x=STRALLOC_INIT;
				if (!stralloc_copys(&x,local_start_dir)) oom();
				if (!stralloc_append(&x,"/")) oom();
				if (!stralloc_cat(&x,&d2)) oom();
				if (!stralloc_copy(&d2,&x)) oom();
				stralloc_free(&x);
			}

			/* back to local start dir. make non-absolute directories work */
			if (-1==fchdir(ldirfd))
				xbailout(111,errno,"failed to fchdir to starting directory",
					0,0,0);
			/* back to remote root */
			cmdwrite1("CWD /");
			p=ccread();
			if (!p) xbailout(111,errno,"failed to read CWD answer",0,0,0);
			if (*p!='2') {
				log1(p);
				log1("\n");
				retcode=1;
			} else
				retcode=onedirpair(&d1,&d2);
			log1("END-OF-COPY\n");
		}
		return retcode;
	}
	if (*localdir=='/') {
		if (!stralloc_copys(&d2,localdir)) oom();
		if (!stralloc_0(&d2)) oom();
	} else {
		if (!stralloc_copys(&d2,local_start_dir)) oom();
		if (!stralloc_append(&d2,"/")) oom();
		if (!stralloc_cats(&d2,localdir)) oom();
		if (!stralloc_0(&d2)) oom();
	}
	return onedirpair(&d1,&d2);
}
