//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		volume.c
//
//	Abstract: 
//		provide volume operation fuction for cgi
//
//	HISTORY:
//		2002/09/26 Catherine Shen -- modify "Quick_Config_All_Disks" to
//			release list before restore the storage config file
//		2002/07/01 Catherine Shen -- modify "Quick_Config_All_Disks" to
//			keep the status of all volumes in every step it goes
//			through
//		2002/06/27 Catherine Shen -- add a new API, Quick_Config_All_Disks(),
//			to remove all volumes and make them into one or more new config
//		2002/06/21 Catherine Shen -- add a new API, Is_HD_Init, to 
//			detect if all HDs (volumes) are uninitialized
//		2002/06/11 Catherine Shen -- modify "Get_One_Volume_Info",
//			"Get_All_Volumes_Info", "Get_All_Volumes_Info_Ex",
//			"Get_All_Volumes_Info_Ex_2", "Get_One_Volume_Info_Ex"
//			so that when a RAID is resyncing, the status is still "S_READY" and
//			the "resync_per" field keeps the resync percentage
//		2002/05/20 Catherine Shen -- modify 
//			(1) "Get_All_Volumes_Info_Ex_2" & "Get_One_Volume_Info_Ex" :
//				to show only the drives present	in the RAID status file
//			(2) "Delete_Raid_Volume" & "Delete_Raid_Volume_Ex" :
//				only initialize the active drives
//		2002/04/22 Catherine Shen -- modify some code to solve the problems
//			caused by 2 processes racing modifying the same section in "storage.conf"
//			and unitentionally do something one the same disk
//		2002/04/09 Catherine Shen -- 
//			(1) check if success after "realloc"
//			(2) replace the code to directy assign a data structure
//				by "memcpy"
//		2002/03/29 Catherine Shen -- modified
//				"Create_Raid_Volume", "Create_Raid_Volume_Ex" to
//				solve the bug that when 2 or more volumes are
//				creating/removing at the same time, the volume number
//				is always changing, and the status may be set to another volume
//		2002/03/28 Catherine Shen -- modified
//			(1) "Add_Volume_Record", "Add_Volume_Record_Ex", "Remove_Volume_Record":
//				write directly w/o calling Util lib functions
//			(2) "Init_Single_Disk_Volume", "Create_Raid_Volume", "Create_Raid_Volume_Ex":
//				modify the volume status before returning the error code
//		2002/03/26 Catherine Shen -- modified 
//			(1) "Set_Volume_Status": to write until SUCCESS
//			(2) "Get_All_Volumes_Conf" series: not to Set_Volume_Status
//			(3) "Get_Raid_Disk_Volumes_Info_Ex_2": corret the volume numbers
//		2002/03/25 Catherine Shen -- modified
//			(1) "Remove_Volume_Record()":
//				when no records left to sort, return SUCCESS;
//			(2) "Get_Volume_Status":
//				Conf_Get_Field returns 0 means success
//		2002/03/24 Catherine Shen -- modify "Create_Raid_Volume_Ex()"
//			to update the "Drive X Exist" field to 1 for
//			each disk in the RAID array.
//		2002/03/21 Catherine Shen -- modify "Restore_Storage_Conf()"
//		2002/03/08 Catherine Shen -- modify the order to reinitialize
//			the spare disks in "Delete_Raid_Volume_Ex()"
//			to solve the bug that user can see the last single volume
//			only after all other volumes are ready.
//		2002/02/27 Catherine Shen -- add 'Restore_Storage_Conf()' to restore the storage.conf
//			when something is wrong w/ the file but the system was initialized.
//		2002/02/26 Catherine Shen -- make sure setting volume status to READY is successful
//		2002/02/25 Catherine Shen -- run quota functions after creating
//			or formatting volumes, modified
//			'Create_Raid_Volume(),
//			'Init_Single_Disk_Volume()',
//			'Create_Raid_Volume_Ex()'
//		2002/02/20 Catherine Shen -- support hot-spare
//			add 'Create_Raid_Volume_Ex()',
//				'Delete_Raid_Volume_Ex()',
//				'Add_Volume_Record_Ex()',
//				'Get_All_Volumes_Info_Ex_2()',
//				'Get_All_Configured_Vol_For_Share_Ex_2()',
//				'Get_All_Volumes_Conf_Ex_2()',
//				'Get_One_Volume_Conf_Ex()',
//				'Get_One_Volume_Info_Ex()',
//				'Get_Raid_Disk_Volumes_Info_Ex_2()'
//			modified 'Create_Storage_Conf()',
//				'Get_Volume_No_For_Drive()',
//				'Get_Volume_Path_For_Drive()',
//				and all functions that calls 'Create_Storage_Conf()'
//		2002/02/05 Catherine Shen -- create "Get_Volume_Path_For_Drive()"
//		2002/01/23 Catherine Shen -- modified "Create_Storage_Conf"
//		2001/12/28 Catherine Shen -- 
//			add 'Get_All_Configured_Vol_For_Share_Ex()',
//				'Get_All_Volumes_Info_Ex()',
//				'Get_All_Volumes_Conf_Ex()',
//				'Get_Uninit_Disk_Volumes_Conf_Ex()',
//				'Get_Single_Disk_Volumes_Conf_Ex()',
//				'Get_Raid_Disk_Volumes_Info_Ex()'
//			modify 'Get_Configured_Vol_No_For_Share()',
//				'Add_Volume_Record()',
//				'Remove_Volume_Record()',
//				'Get_Volume_No_For_Drive()'
//		2001/6/14	Louis Tsai created
//
//**************************************************************************
#include "storage.h"
#include "storage_err.h"
#include "NAS.h"
#include "nas_quota.h"	// Catherine 2002/02/25
#include "shm_conf.h"	// Catherine 2002/03/24
#include "eventlog.h"
#include "naslvm.h"	// LVM support
#include "msg.h"
#include "sysmgrd.h"
#ifdef	STORAGE_FILE_LOCK
#include "file_lock.h"
#endif
#include <time.h>
#include <stdlib.h>
#include <sys/file.h>

int m_cmp(const void *m1,const void *m2) //Catherine 2002/04/22
{

	VOLUME_CONF *a,*b;
	
	a = (VOLUME_CONF*) m1;
	b = (VOLUME_CONF*) m2;

	if (a->vol_no>b->vol_no) return 1;
	if (a->vol_no<b->vol_no) return -1;
	
	return 0;

}
void sort_volume_data(VOLUME_CONF *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(VOLUME_CONF),m_cmp);	
}

int m_cmp_for_show(const void *m1,const void *m2) //Catherine 2002/04/22
{

	VOLUME_CONF *a,*b;
	
	a = (VOLUME_CONF*) m1;
	b = (VOLUME_CONF*) m2;

	if(a->raid_level > b->raid_level ) return 1;
	if(a->raid_level < b->raid_level ) return -1;
	if(a->drive_no_list[0] > b->drive_no_list[0] ) return 1;
	if(a->drive_no_list[0] < b->drive_no_list[0] ) return -1;
	if(a->drive_no_list[0] == b->drive_no_list[0] ) return 0;

	return 0;

}
void sort_volume_data_for_show(VOLUME_CONF *vol_data_list,int list_cnt) //Catherine 2002/04/22
{
	qsort((void*)vol_data_list,list_cnt,sizeof(VOLUME_CONF),m_cmp_for_show);	
}

int m_cmp_for_show_ex(const void *m1,const void *m2) //Catherine 2002/04/22
{

	VOLUME_CONF_EX *a,*b;
	
	a = (VOLUME_CONF_EX*) m1;
	b = (VOLUME_CONF_EX*) m2;

	if(a->raid_level > b->raid_level ) return 1;
	if(a->raid_level < b->raid_level ) return -1;
	if(a->drive_no_list[0] > b->drive_no_list[0] ) return 1;
	if(a->drive_no_list[0] < b->drive_no_list[0] ) return -1;
	if(a->drive_no_list[0] == b->drive_no_list[0] ) return 0;

	return 0;

}

void sort_volume_data_for_show_ex(VOLUME_CONF_EX *vol_data_list,int list_cnt) //Catherine 2002/04/22
{
	qsort((void*)vol_data_list,list_cnt,sizeof(VOLUME_CONF_EX),m_cmp_for_show_ex);	
}

int m_cmp_acc(const void *int1, const void *int2) // Catherine 2002/05/20
{
	int *a, *b;
	a = (int*)int1;
	b = (int*)int2;
	if (*a > *b) return 1;
	if (*a < *b) return -1;
	return 0;
}

void sort_drive_no_for_show(int *drive_list, int list_cnt) // Catherine 2002/05/20
{
	qsort((void*)drive_list, list_cnt, sizeof(int), m_cmp_acc);
}

void backup_storage_conf()
{
	char cmd_line[80];
	sprintf(cmd_line,"cp %s %s > /dev/null",STORAGE_CONF,STORAGE_CONF_BAK);
	system(cmd_line);
}

void restore_storage_conf()
{
	char cmd_line[80];
	sprintf(cmd_line,"cp %s %s > /dev/null",STORAGE_CONF_BAK,STORAGE_CONF);
	system(cmd_line);
}

// If the model = NAS-4000 &&
// the device's "dev_NOEXIST" field of the "Disk" section is "0"
BOOL is_NAS4000_single_volume(int drive_no)
{
	char buf[BUF_SIZE], value[10];

	if (Get_Model_Name(buf, BUF_SIZE)<0)
		return FALSE;
	if (strcasecmp(buf, "NAS-4000")==0) { // is NAS-4000
		switch (drive_no) {
			case 1:
				strcpy(buf, "HDA_NOEXIST");
				break;
			case 2:
				strcpy(buf, "HDB_NOEXIST");
				break;
			case 3:
				strcpy(buf, "HDC_NOEXIST");
				break;
			case 4:
				strcpy(buf, "HDD_NOEXIST");
				break;
			default:
				return FALSE;
		}
		Get_Profile_String("Disk", buf, "", value, 10);

		if (strcasecmp(value, "0")==0)
			return TRUE;
	}

	return FALSE;
}


// Catherine 2002/02/25 to start volume quota functions
void start_vol_quota(int vol, int check)
{
	unsigned long quota_size;
	int ret;

	if (check) {
		ret = Check_User_Quota(vol, 1);
		if(ret != 0) {
		        int fd;
       			char *ptr[10];

       			fd = Open_Event_Log(1);
       			if(fd > 0) {
       				sprintf(ptr[0], "Failed to perform quotacheck on disk volume %d", vol); 
		        	Report_Event(fd, EVENTLOG_ERROR_TYPE, 1, ptr);
		        	Close_Event_Log(fd);
		        }
		}
	}

	if (Is_User_Quota_Enabled(&quota_size)) {
     		Enable_User_Quota_Disk(vol);
		Start_User_Quota(vol);
	}
}

// Catherine 2002/03/24 for potential hotswap problem
// ex. HW not stable & cause hotswap update "Drive x Exist" to 0,
// if user now remove the RAID, and re-create one,
// hotswap finds the disk and a "hot-plug" event is detected.
// Description : to set the "Drive x Exist" field of "Disk" section in "uLinux.conf"
int set_exist_flag(int drive_no, char *value)
{
	int ret;
	char hd_sec_name[HD_DEVICE_NAME_LENGTH];

	sprintf(hd_sec_name, "Drive %d Exist", drive_no);
	Conf_Set_Field(DEFAULT_CFG_FILE, "Disk", hd_sec_name, value);

	return ret;
}

// cram the fields of VOLUME_CONF_EX to VOLUME_CONF for clear_volume_related_messages only
void convert_to_volume_conf(VOLUME_CONF_EX* org_volconf, VOLUME_CONF *newvolconf)
{
	int i, j;

	if (org_volconf==NULL || newvolconf==NULL) return;

	newvolconf->vol_no = org_volconf->vol_no;
	strcpy(newvolconf->device_name, org_volconf->device_name);
	newvolconf->raid_level = org_volconf->raid_level;
	for (i=0; i<org_volconf->list_cnt; i++)
		newvolconf->drive_no_list[i] = org_volconf->drive_no_list[i];

	for (j=0; j<org_volconf->spare_list_cnt; j++, i++)
		newvolconf->drive_no_list[i] = org_volconf->spare_drive_list[j];

	newvolconf->list_cnt = org_volconf->list_cnt;
	newvolconf->status = org_volconf->status;
}

// clear the old messages/statuses for the old volume
void clear_volume_related_messages(VOLUME_CONF *vol_conf)
{
	char value[20], buf[BUF_SIZE];
	char *DS_CONF_FILE = "/etc/config/uLinux.conf";
	char *DISK_SESSION_NAME = "DISK";
	char *p = strrchr(vol_conf->device_name, '/');

	if (p==NULL)
		p= vol_conf->device_name;
	else
		p++;

	sprintf(value, "%s_OVER_98", p);
	String_To_Upper_Case(value);

	// clear the disk full message
	if ( GetProfileString(DISK_SESSION_NAME, value, "no", buf, sizeof(buf))>= 0) {
		if ( strncasecmp(buf, "yes", 3)==0) {// its original value over 98
			SendMessage("sysmond", MSG_SYSMGR, SYSMGR_DISK_CLEAR, 0);
			Conf_Remove_Field(DS_CONF_FILE, DISK_SESSION_NAME, value);
			//Update_Flash_Data(DS_CONF_FILE);
		}
	}
}

// clear the old messages/statuses for the old volume
void clear_volume_related_messages_ex(VOLUME_CONF_EX *vol_conf)
{
	VOLUME_CONF tmpvol;

	convert_to_volume_conf(vol_conf, &tmpvol);
	clear_volume_related_messages(&tmpvol);
}

#ifdef	STORAGE_FILE_LOCK
int repair_storage_conf_if_necessary()
{
	FILE *fp = NULL;
	int ret = SUCCESS;

#ifdef	DEBUG
printf("repair_storage_conf_if_necessary starts!\n");
#endif
	if (!NAS_File_Lock(STORAGE_CONF, 'r')) {
#ifdef	DEBUG
		printf("\tNAS_File_Lock(%s) fails!\n", STORAGE_CONF);
#endif
		return ERROR_LOCK_FILE;
	}

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) {
#ifdef	DEBUG
		printf("\t%s not exists!\n", STORAGE_CONF);
#endif
		NAS_File_Unlock(STORAGE_CONF);
		ret = Create_Storage_Conf();
	}
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		NAS_File_Unlock(STORAGE_CONF);
		if(strlen(tmp_line)== 0) {
#ifdef	DEBUG
			printf("\t%s is imcomplete!\n", STORAGE_CONF);
#endif
			ret = Restore_Storage_Conf();
		}
	}
#ifdef	DEBUG
printf("repair_storage_conf_if_necessary finishes!\n");
#endif
	return ret;
}
#endif

//=================================== Export function to CGI =================================

/*------------------------------------------------------------------------
According to HD configuration in uLinux.conf, this function configures each 
HD device to uninitialized single disk volume in storage.conf! Note that,
this function only be called when user setup the NAS system.
--------------------------------------------------------------------------*/
int Create_Storage_Conf()
{
	int i = 1,j,ret = 0,raid_cnt=0;
	RAID_DEVICE rd_list[MAX_DRIVE_NO];
	char device_name[HD_DEVICE_NAME_LENGTH];
	VOLUME_CONF_EX vol_data;
	FILE *fp;
	
	int drive_no = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);
	
#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(STORAGE_CONF, 'w') == 0)
		return ERROR_LOCK_FILE;
#endif
	//remove original storage.conf & create new	
	unlink(STORAGE_CONF);
	if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
		fclose(fp);
	else {
#ifdef	STORAGE_FILE_LOCK
		NAS_File_Unlock(STORAGE_CONF);
#endif
		return ERROR_FAIL;
	}
		
	while((ret = Get_Partition_Name(i,DATA_PART,device_name))>=0) {
		if(i>drive_no) break;
		if(Is_Raid_HD(i,NULL,0)) {
			i++;
			continue;
		}
		strcpy(vol_data.device_name,device_name);
		vol_data.list_cnt = 1;
		vol_data.drive_no_list[0] = i;
		vol_data.spare_list_cnt = 0; // Catherine 2002/02/22
		vol_data.raid_level = -2 ;
		Get_HD_DevName(i,device_name);

		if(!Is_HD_Exist(device_name)) 
			vol_data.status = S_NOT_EXIST;
		else if (is_NAS4000_single_volume(i)) {
			// Catherine 2002/01/23 
			vol_data.status = S_READY;
		}
		else
			vol_data.status = S_UNINITIALIZE;
		i++;
#ifdef	STORAGE_FILE_LOCK
		ret = Add_Volume_Record_Ex_2(&vol_data, FALSE);
#else
		ret = Add_Volume_Record_Ex(&vol_data);
#endif

		if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
			NAS_File_Unlock(STORAGE_CONF);
#endif
			return ret;
		}
	}
	raid_cnt = Get_All_RaidDev_Conf(rd_list,MAX_DRIVE_NO);
	if(raid_cnt>0) {
		for(i=0;i<raid_cnt;i++) {
			strcpy(vol_data.device_name,rd_list[i].name);

			vol_data.list_cnt = rd_list[i].data_drive_cnt;
			vol_data.spare_list_cnt = rd_list[i].spare_drive_cnt;
			if(strcmp(rd_list[i].raid_level,"linear")==0)
				vol_data.raid_level = -1;
			else
				vol_data.raid_level = atoi(rd_list[i].raid_level);
				
			for(j=0;j<rd_list[i].data_drive_cnt;j++) {
				vol_data.drive_no_list[j] = rd_list[i].data_drive_no[j];
			}

			for(j=0;j<rd_list[i].spare_drive_cnt;j++) {
				vol_data.spare_drive_list[j] = rd_list[i].spare_drive_no[j];
			}

			vol_data.status = S_READY;
#ifdef	STORAGE_FILE_LOCK
			ret = Add_Volume_Record_Ex_2(&vol_data, FALSE);
#else
			ret = Add_Volume_Record_Ex(&vol_data);
#endif

			if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
				NAS_File_Unlock(STORAGE_CONF);
#endif
				return ret;
			}
		}
	}
#ifdef	STORAGE_FILE_LOCK
	NAS_File_Unlock(STORAGE_CONF);
#endif
	return SUCCESS;
}

/*------------------------------------------------------------------------
According to HD configuration in uLinux.conf, this function configures each 
HD device not configured in "/etc/raidtab" to "ready" single disk volume
in storage.conf!
This is called when the storage.conf has incorrect information but raidtab
is valid.
Catherine created 2002/02/27
--------------------------------------------------------------------------*/
int Restore_Storage_Conf()
{
	int i = 1,j,ret = 0,raid_cnt=0;
	RAID_DEVICE rd_list[MAX_DRIVE_NO];
	char device_name[HD_DEVICE_NAME_LENGTH];
	VOLUME_CONF_EX vol_data;
	FILE *fp;
	
	int drive_no = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);
	
#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(STORAGE_CONF, 'w')==0)
		return ERROR_LOCK_FILE;
#endif
	//remove original storage.conf & create new	
	unlink(STORAGE_CONF);
	if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
		fclose(fp);
	else {
#ifdef	STORAGE_FILE_LOCK
		NAS_File_Unlock(STORAGE_CONF);
#endif
		return ERROR_FAIL;
	}
		
	while((ret = Get_Partition_Name(i,DATA_PART,device_name))>=0) {
		if(i>drive_no) break;
		if(Is_Raid_HD(i,NULL,0)) {
			i++;
			continue;
		}
		strcpy(vol_data.device_name,device_name);
		vol_data.list_cnt = 1;
		vol_data.drive_no_list[0] = i;
		vol_data.spare_list_cnt = 0; // Catherine 2002/02/22
		vol_data.raid_level = -2 ;
		Get_HD_DevName(i,device_name);

		/* modify by Kent 2002/03/04			*/
		/* to prevent the state of HD is  un-initialize	*/
		if (!Is_HD_Exist(device_name))
			vol_data.status=S_NOT_EXIST;
		else
		{
			if (!Is_HD_Valid(i))
				vol_data.status = S_UNINITIALIZE;
			else
				vol_data.status = S_READY;
		}

		i++;

#ifdef	STORAGE_FILE_LOCK
		ret = Add_Volume_Record_Ex_2(&vol_data, FALSE);
#else
		ret = Add_Volume_Record_Ex(&vol_data);
#endif

		if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
			NAS_File_Unlock(STORAGE_CONF);
#endif
			return ret;
		}
	}

	raid_cnt = Get_All_RaidDev_Conf(rd_list,MAX_DRIVE_NO);
	if(raid_cnt>0) {
		for(i=0;i<raid_cnt;i++) {
			strcpy(vol_data.device_name,rd_list[i].name);
			vol_data.list_cnt = rd_list[i].data_drive_cnt;
			vol_data.spare_list_cnt = rd_list[i].spare_drive_cnt;
			if(strcmp(rd_list[i].raid_level,"linear")==0)
				vol_data.raid_level = -1;
			else
				vol_data.raid_level = atoi(rd_list[i].raid_level);
				
			for(j=0;j<rd_list[i].data_drive_cnt;j++) {
				vol_data.drive_no_list[j] = rd_list[i].data_drive_no[j];
			}
			for(j=0;j<rd_list[i].spare_drive_cnt;j++) {
				vol_data.spare_drive_list[j] = rd_list[i].spare_drive_no[j];
			}
			vol_data.status = S_READY;
#ifdef	STORAGE_FILE_LOCK
			ret = Add_Volume_Record_Ex_2(&vol_data, FALSE);
#else
			ret = Add_Volume_Record_Ex(&vol_data);
#endif

			if(ret<0) {
#ifdef	STORAGE_FILE_LOCK
				NAS_File_Unlock(STORAGE_CONF);
#endif
				return ret;
			}
		}
	}
#ifdef	STORAGE_FILE_LOCK
	NAS_File_Unlock(STORAGE_CONF);
#endif
	return SUCCESS;
}

// ------- Initialize single disk volume function ------------//
int Init_Single_Disk_Volume(int vol_no)
{
	VOLUME_CONF vol_data;
	int ret; //, i;

	ret = Get_One_Volume_Conf(vol_no,&vol_data);
	if(ret<0) return ret;
	
	if(vol_data.raid_level != -2) return ERROR_FAIL;

	Set_Volume_Status(vol_no,S_UNINITIALIZE);

	Stop_User_Quota(vol_no); // Catherine 2002/02/25
	
	Set_Volume_Status(vol_no,S_INITIALIZING);

	if((ret=Init_NAS_Disk(vol_data.drive_no_list[0]))<0){
		Set_Volume_Status(vol_no,S_UNINITIALIZE); // Catherine 2002/03/28
		return ret;
	}
	
	Set_Volume_Status(vol_no,S_FORMATTING);
	if((ret=Format_Volume(vol_no))<0){
		Set_Volume_Status(vol_no,S_NOT_MOUNTED); // Catherine 2002/03/28
		return ret;
	}
	
	if((ret=Mount_Volume(vol_no))<0){
		Set_Volume_Status(vol_no,S_NOT_MOUNTED); // Catherine 2002/03/28
		return ret;
	}
	start_vol_quota(vol_no, 1); // Catherine 2002/02/25
	
	Set_Volume_Status(vol_no,S_READY);
	return SUCCESS;
}

// ------- Create raid volume function ---------------------//


//???????  how to process fail situation
int Create_Raid_Volume(int *vol_no_list,int list_cnt,int raid_level)
{
	int i=0,ret,vol_no;
	char rd_name[HD_DEVICE_NAME_LENGTH];
	VOLUME_CONF vol_data;
	
	//prepare raid volume data
	vol_data.list_cnt = list_cnt;
	vol_data.raid_level = raid_level;
	for(i=0;i<list_cnt;i++){
		VOLUME_CONF tmp_vol_data;
		ret = Get_One_Volume_Conf(vol_no_list[i],&tmp_vol_data);
		if(ret<0) return ret; // get volume data fail
		
		if(tmp_vol_data.raid_level != -2 ) return ERROR_FUNCTION_PARAMETER; //not signal volume
		if(tmp_vol_data.status != S_UNINITIALIZE &&
			tmp_vol_data.status != S_INVALID &&
			tmp_vol_data.status != S_READY  &&
			tmp_vol_data.status != S_NOT_MOUNTED)
			return ERROR_FAIL;
		vol_data.drive_no_list[i] = tmp_vol_data.drive_no_list[0];
	}
	
	
	if(vol_data.list_cnt > 1 && vol_data.raid_level >-2) {//for raid volume
		switch(vol_data.raid_level) {
			case -1://JBOD
				backup_storage_conf();	
				//update volume data
				for(i=0;i<vol_data.list_cnt;i++) {
					VOLUME_CONF tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[i]);
					Stop_User_Quota(vol_no);

					Get_One_Volume_Conf(vol_no, &tmpvol);
					clear_volume_related_messages(&tmpvol);

					Remove_Volume_Record(vol_no);
				}

				// create raid table
				Get_First_Available_RaidName(rd_name,HD_DEVICE_NAME_LENGTH);
				ret = Create_NAS_RaidDev(rd_name,vol_data.drive_no_list,vol_data.list_cnt,-1);
				if(ret < 0) return ret;
				
				strcpy(vol_data.device_name,rd_name);
				vol_data.status = S_INITIALIZING;
				ret = Add_Volume_Record(&vol_data);

				if (ret<0) return ret; // Catherine 2002/03/28

				//Initialize single disk
				for(i=0;i<vol_data.list_cnt;i++) {
					if((ret=Init_NAS_Disk(vol_data.drive_no_list[i]))<0){
						Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[i]),
							S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
				}
	
				//get availabe raid device name
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_CREATING_RAID);
				
				// do make raid
				Make_Raid(rd_name);
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_FORMATTING);
				Format_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]));
				if((ret=Mount_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0])))< 0 ) {
					Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),
						S_NOT_MOUNTED); // Catherine 2002/03/28
					return ret;
				}
				start_vol_quota(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),1);
				ret = Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_READY);

				break;
			case 0: //RAID 0
				backup_storage_conf();	
				//update volume data
				for(i=0;i<vol_data.list_cnt;i++) { // Catherine 2002/02/25 modified
					VOLUME_CONF tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[i]);
					Stop_User_Quota(vol_no);

					Get_One_Volume_Conf(vol_no, &tmpvol);
					clear_volume_related_messages(&tmpvol);

					Remove_Volume_Record(vol_no);
				}
		
				// create raid table
				Get_First_Available_RaidName(rd_name,HD_DEVICE_NAME_LENGTH);
				ret = Create_NAS_RaidDev(rd_name,vol_data.drive_no_list,vol_data.list_cnt,0);
				if(ret < 0) return ret;
				
				strcpy(vol_data.device_name,rd_name);
				vol_data.status = S_INITIALIZING;
				ret = Add_Volume_Record(&vol_data);

				if (ret<0) return ret; // Catherine 2002/03/28

				//Initialize single disk
				for(i=0;i<vol_data.list_cnt;i++) {
					if((ret=Init_NAS_Disk(vol_data.drive_no_list[i]))<0){
						Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[i]),
							S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
				}
	
				//get availabe raid device name
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_CREATING_RAID);
				
				// do make raid
				Make_Raid(rd_name);
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_FORMATTING);
				Format_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]));
				if((ret=Mount_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0])))< 0 ) {
					Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),
						S_NOT_MOUNTED); // Catherine 2002/03/28
					return ret;
				}
				start_vol_quota(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),1);
				ret = Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_READY);

				break;
			case 1: //RAID 1
				backup_storage_conf();	
				//update volume data
				//for(i=0;i<vol_data.list_cnt;i++) 
				for(i=0;i<vol_data.list_cnt;i++) { // Catherine 2002/02/25 modified
					VOLUME_CONF tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[i]);
					Stop_User_Quota(vol_no);

					Get_One_Volume_Conf(vol_no, &tmpvol);
					clear_volume_related_messages(&tmpvol);

					Remove_Volume_Record(vol_no);
				}
					
				// create raid table
				Get_First_Available_RaidName(rd_name,HD_DEVICE_NAME_LENGTH);
				ret = Create_NAS_RaidDev(rd_name,vol_data.drive_no_list,vol_data.list_cnt,1);
				if(ret < 0) return ret;
				
				strcpy(vol_data.device_name,rd_name);
				vol_data.status = S_INITIALIZING;
				
				ret = Add_Volume_Record(&vol_data);

				if (ret<0) return ret; // Catherine 2002/03/28

				//Initialize single disk
				for(i=0;i<vol_data.list_cnt;i++) {
					if((ret=Init_NAS_Disk(vol_data.drive_no_list[i]))<0){
						Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[i]),
							S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
				}
	
				//get availabe raid device name
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_CREATING_RAID);
				
				// do make raid
				Make_Raid(rd_name);
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_FORMATTING);
				Format_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]));
				if((ret=Mount_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0])))< 0 ) {
					Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),
						S_NOT_MOUNTED); // Catherine 2002/03/28
					return ret;
				}
				start_vol_quota(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),1);
				ret = Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_READY);

				break;
			case 4: //RAID 0+1
				backup_storage_conf();	
				
				//update volume data
				for(i=0;i<vol_data.list_cnt;i++) { // Catherine 2002/02/25 modified
					VOLUME_CONF tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[i]);
					Stop_User_Quota(vol_no);

					Get_One_Volume_Conf(vol_no, &tmpvol);
					clear_volume_related_messages(&tmpvol);

					Remove_Volume_Record(vol_no);
				}				
				
				// create raid table
				strcpy(rd_name,"/dev/md2");
				ret = Create_NAS_RaidDev(rd_name,vol_data.drive_no_list,vol_data.list_cnt,4);
				if(ret < 0) return ret;
				
				
				strcpy(vol_data.device_name,rd_name);
				vol_data.status = S_INITIALIZING;
				ret = Add_Volume_Record(&vol_data);

				if (ret<0) return ret; // Catherine 2002/03/28

				vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[0]); // Catherine 2002/03/28

				//Initialize single disk
				for(i=0;i<vol_data.list_cnt;i++) {
					if((ret=Init_NAS_Disk(vol_data.drive_no_list[i]))<0){
						Set_Volume_Status(vol_no, S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
				}
	
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_CREATING_RAID);

				//Umount single volume partition to avoid os bug
				for(i=0;i<vol_data.list_cnt;i++) {
					ret = Unmount_NAS_Partition(vol_data.drive_no_list[i],DATA_PART);
				}
				
				Unmount_Partition(rd_name);
				
				// do make raid
				if((ret = Make_Raid("/dev/md0"))<0) {
					Set_Volume_Status(vol_no, S_RAID_DIED); // Catherine 2002/03/28
					return ret;
				}
				if((ret = Make_Raid("/dev/md1"))<0) {
					Set_Volume_Status(vol_no, S_RAID_DIED); // Catherine 2002/03/28
					return ret;
				}
				if((ret = Make_Raid(rd_name))<0) {
					Set_Volume_Status(vol_no, S_RAID_DIED); // Catherine 2002/03/28
					return ret;
				}

				Set_Volume_Status(vol_no,S_FORMATTING);
				Format_Volume(vol_no);
				if((ret=Mount_Volume(vol_no))< 0 ) {
					Set_Volume_Status(vol_no, S_NOT_MOUNTED); // Catherine 2002/03/28
					return ret;
				}
				start_vol_quota(vol_no,1);
				ret = Set_Volume_Status(vol_no,S_READY);

				break;
			case 5: //RAID 5
				backup_storage_conf();	
				//update volume data
				for(i=0;i<vol_data.list_cnt;i++) { // Catherine 2002/02/25 modified
					VOLUME_CONF tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[i]);
					Stop_User_Quota(vol_no);

					Get_One_Volume_Conf(vol_no, &tmpvol);
					clear_volume_related_messages(&tmpvol);

					Remove_Volume_Record(vol_no);
				}
		
				// create raid table
				Get_First_Available_RaidName(rd_name,HD_DEVICE_NAME_LENGTH);
				ret = Create_NAS_RaidDev(rd_name,vol_data.drive_no_list,vol_data.list_cnt,5);
				if(ret < 0) return ret;

				strcpy(vol_data.device_name,rd_name);
				vol_data.status = S_INITIALIZING;
				
				ret = Add_Volume_Record(&vol_data);

				if (ret<0) return ret; // Catherine 2002/03/28

				//Initialize single disk
				for(i=0;i<vol_data.list_cnt;i++) {
					if((ret=Init_NAS_Disk(vol_data.drive_no_list[i]))<0){
						Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),
							S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
				}
	
				//get availabe raid device name
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_CREATING_RAID);
				
				// do make raid
				Make_Raid(rd_name);
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_FORMATTING);
				Format_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]));
				if((ret=Mount_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0])))< 0 ) {
					Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),
						S_NOT_MOUNTED); // Catherine 2002/03/28
					return ret;
				}
				start_vol_quota(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),1);
				ret = Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_READY);

				break;
			default:
				return ERROR_FUNCTION_PARAMETER;
				break;
		}
	}
	else 
		return ERROR_FUNCTION_PARAMETER;
	return SUCCESS;
} 

int Delete_Raid_Volume(int vol_no)
{
	VOLUME_CONF raid_vol_data,single_vol_data;
	int ret,i, tmp_vol_no; 
	ret = Get_One_Volume_Conf(vol_no,&raid_vol_data);
	if(ret<0) return ret;
	
	if(raid_vol_data.raid_level < -1) return ERROR_FAIL;
	
	
	//removing raid
	Set_Volume_Status(vol_no,S_REMOVING_RAID);

	Stop_User_Quota(vol_no); // Catherine 2002/02/25
	
	if(raid_vol_data.status != S_NOT_ACTIVE) {
		
		Umount_Volume(vol_no);
		
		if((ret=Stop_Raid(raid_vol_data.device_name))<0){
			Mount_Volume(vol_no);
			start_vol_quota(vol_no, 0);

			Set_Volume_Status(vol_no,S_READY);

			return ret;
		}
	}
	
	if(raid_vol_data.raid_level == 4) {  // raid0+1
		if((ret=Stop_Raid("/dev/md0"))<0) return ret;
		if((ret=Stop_Raid("/dev/md1"))<0) return ret;
		if((ret=Delete_RaidDev("/dev/md0"))<0) return ret;
		if((ret=Delete_RaidDev("/dev/md1"))<0) return ret;
	}
	
	if((ret=Delete_RaidDev(raid_vol_data.device_name))<0) {
		return ret;
	}

	// clear old messages
	clear_volume_related_messages(&raid_vol_data);

	//remove volume record from storage.conf
	Remove_Volume_Record(vol_no);
	
	//reinitialize single disk volume belong to this volume
	for(i=0;i<raid_vol_data.list_cnt;i++){
		Get_Partition_Name(raid_vol_data.drive_no_list[i],DATA_PART,single_vol_data.device_name);
		single_vol_data.list_cnt = 1;
		single_vol_data.drive_no_list[0] = raid_vol_data.drive_no_list[i];
		single_vol_data.raid_level = -2;
		single_vol_data.status = S_INITIALIZING;
		ret = Add_Volume_Record(&single_vol_data);
		if(ret<0) {
			return ret;
		}
	}
	for(i=0;i<raid_vol_data.list_cnt;i++){
		tmp_vol_no = Get_Volume_No_For_Drive(raid_vol_data.drive_no_list[i]);
		if(tmp_vol_no>0) {
			char device_name[HD_DEVICE_NAME_LENGTH];

			Get_HD_DevName(raid_vol_data.drive_no_list[i],device_name);
			if (Is_HD_Exist(device_name))
				ret = Init_Single_Disk_Volume(tmp_vol_no);
			else
				Set_Volume_Status(tmp_vol_no, S_NOT_EXIST);
		}
		else
			return ERROR_FAIL;
		sleep(1);
	}
	return SUCCESS;
}

//---------------------------------------------------------------------------------------------//
int Format_Volume(int vol_no)
{
	VOLUME_CONF vol_data;
	int fs_no,ret;
	
	
	
	if((ret = Get_One_Volume_Conf(vol_no,&vol_data))== SUCCESS) {
		
		fs_no = Get_FileSystem_Type();
		
		//chek raid5 ?
		if(vol_data.raid_level == 5)
			ret = Format_Partition(vol_data.device_name,TRUE,fs_no);
		else
			ret = Format_Partition(vol_data.device_name,FALSE,fs_no);
		
	}
	else
		ret = ERROR_FORMAT;
		
	return ret;
}

int Scan_Volume(int vol_no,int action)
{
	
	VOLUME_CONF vol_data;
	int fs_no,ret;

	if((ret = Get_One_Volume_Conf(vol_no,&vol_data))== SUCCESS) {
		fs_no = Get_Volume_FileSystem_Type(vol_no); // ReiserFS support
		ret = Scan_Partition(vol_data.device_name,action,fs_no);
	}
	else
		ret = ERROR_SCAN;
		
	return ret;
}

int Mount_Volume(int vol_no)
{
	VOLUME_CONF vol_data;
	int fs_no,ret;
	char mp_str[BUF_SIZE];
	
	if((ret = Get_One_Volume_Conf(vol_no,&vol_data))== SUCCESS) {
		Get_MP_String(vol_data.drive_no_list[0],DATA_PART,mp_str);
		fs_no = Get_Volume_FileSystem_Type(vol_no); // ReiserFS Support
		ret = Mount_Partition(vol_data.device_name,mp_str,fs_no);		
	}
	else
		ret = ERROR_MOUNT_FAIL;
	return ret;
}

int Umount_Volume(int vol_no)
{	
	VOLUME_CONF vol_data;
	int ret;
//	char tmp_str[256];
	
	if((ret = Get_One_Volume_Conf(vol_no,&vol_data))== SUCCESS) {
		ret = Unmount_Partition(vol_data.device_name);
	}
	else
		ret = ERROR_UMOUNT_FAIL;
	return ret;
}

//------------------------------------------------------------------------------------//


int Add_Volume_Record(VOLUME_CONF *vol_data)
{
	int vol_cnt=0,i=0,j=0;
	VOLUME_CONF *vol_data_list=NULL;
	FILE *fp;
	time_t the_time;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE], tmp_str[10];
	void *ptr=NULL;

	vol_cnt = Get_All_Volumes_Conf_Ex(&vol_data_list);
	
	if(vol_cnt >=MAX_VOLUME_NO) {
		free(vol_data_list);
		return ERROR_OUT_OF_RANGE;
	}
	if(vol_cnt >=0) {
		ptr = realloc((void*)vol_data_list,
			(vol_cnt+1)*sizeof(VOLUME_CONF));

		if (ptr==NULL) {
			if (vol_data_list) free(vol_data_list);
			return ERROR_OUT_OF_MEMORY;
		}
		vol_data_list = (VOLUME_CONF *)ptr;

		// Catherine 2002/04/22 ==>
		memcpy((void*)&vol_data_list[vol_cnt],
			(void*)vol_data,
			sizeof(VOLUME_CONF));

		{
			int i,j,vol_no,flag;

			for (i=1; i<MAX_VOLUME_NO; i++) {
				flag=0;
				for (j=0; j<vol_cnt; j++) {
					if (i==vol_data_list[j].vol_no) {
						flag=1;
						break;
					}
				}
				if (flag==0) {
					vol_no=i;
					break;
				}
			}
			printf("New volume number is %d\n", vol_no);
			vol_data_list[vol_cnt].vol_no=vol_no;
			vol_data->vol_no=vol_no;
		}
		// <== Catherine 2002/04/22

		vol_cnt++;
		sort_volume_data(vol_data_list,vol_cnt);	

#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(STORAGE_CONF, 'w')==0) {
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}

		// open /etc/config/storage.conf for write
		if ((fp = fopen(STORAGE_CONF, "w+")) == NULL) {
			free(vol_data_list);
			NAS_File_Unlock(STORAGE_CONF);
			return ERROR_OPEN_FILE;
		}
#else
		if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
			fclose(fp);
		else {
			free(vol_data_list);
			return ERROR_FAIL;
		}
// Catherine 2002/03/28 ==>
		// open /etc/config/storage.conf for write
		if ((fd=open(STORAGE_CONF, O_CREAT | O_RDWR))<0) {
			free(vol_data_list);
			return ERROR_OPEN_FILE;
		}
		// exclusively lock file
		if (flock(fd, LOCK_EX)<0) {
			close(fd);
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}
#endif
		// write the new content into the config file
		for(i=0;i<vol_cnt;i++){
			// volume header
			sprintf(tmpbuf, "[VOLUME %d]\n", vol_data_list[i].vol_no); // Catherine 2002/04/22
			strcpy(buf, tmpbuf);
			// device name
			sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].device_name);
			strcat(buf, tmpbuf);
			// raid level
			sprintf(tmpbuf, "raid level = %d\n", vol_data_list[i].raid_level);
			strcat(buf, tmpbuf);
			// drive no
			strcpy(tmpbuf,"drive no = ");
			for(j=0;j<vol_data_list[i].list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].drive_no_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");

			strcat(buf, tmpbuf);
			// status
			sprintf(tmpbuf, "status = %d\n", vol_data_list[i].status);
			strcat(buf, tmpbuf);
			// record time
			(void) time(&the_time);
			sprintf(tmpbuf, "record_time = %s\n", ctime(&the_time));
			strcat(buf, tmpbuf);
#ifdef	STORAGE_FILE_LOCK
			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(vol_data_list);
				NAS_File_Unlock(STORAGE_CONF);
				return ERROR_WRITE_FILE;
			}
#else
			if (write(fd, buf, strlen(buf))<0) {
				flock(fd, LOCK_UN);
				close(fd);
				free(vol_data_list);
				return ERROR_WRITE_FILE;
			}
#endif
		}
		// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
		fclose(fp);
		NAS_File_Unlock(STORAGE_CONF);
#else
		flock(fd, LOCK_UN);
		close(fd);
#endif
// <== Catherine 2002/03/28
		free(vol_data_list);
	}
	return vol_cnt;
	
}

int Remove_Volume_Record(int vol_no)
{
	int ret=0,vol_cnt=0,i=0,j=0;
	char vol_section[20],tmp_str[10];	
	VOLUME_CONF *vol_data_list;
	FILE *fp;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	time_t the_time;
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE];
	
	//remove volume record
	sprintf(vol_section,"VOLUME %d",vol_no);
	ret = Conf_Remove_Section(STORAGE_CONF,vol_section);
	if(ret<0) return ret;
	
	//sort volume record
	vol_cnt = Get_All_Volumes_Conf_Ex(&vol_data_list);
	if (vol_cnt<0)
		return ERROR_NOT_FOUND;
	if (vol_cnt==0) // Catherine 2002/03/25
		return SUCCESS;

	sort_volume_data(vol_data_list,vol_cnt);

#ifdef	STORAGE_FILE_LOCK
	if (NAS_File_Lock(STORAGE_CONF, 'w') == 0) {
		free(vol_data_list);
		return ERROR_LOCK_FILE;
	}
#endif
	if((fp = fopen(STORAGE_CONF,"w+")) != NULL) {
#ifndef	STORAGE_FILE_LOCK
		fclose(fp);
#endif
	}
	else {
		free(vol_data_list);
#ifdef	STORAGE_FILE_LOCK
		NAS_File_Unlock(STORAGE_CONF);
#endif
		return ERROR_FAIL;
	}
#ifndef	STORAGE_FILE_LOCK
	// open /etc/config/storage.conf for write
	if ((fd=open(STORAGE_CONF, O_CREAT | O_RDWR))<0) {
		free(vol_data_list);
		return ERROR_OPEN_FILE;
	}
	// exclusively lock file
	if (flock(fd, LOCK_EX)<0) {
		close(fd);
		free(vol_data_list);
		return ERROR_LOCK_FILE;
	}
#endif
	// write the new content into the config file
	for(i=0;i<vol_cnt;i++){
		// volume header
		sprintf(tmpbuf, "[VOLUME %d]\n", vol_data_list[i].vol_no); // Catherine 2002/04/22
		strcpy(buf, tmpbuf);
		// device name
		sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].device_name);
		strcat(buf, tmpbuf);
		// raid level
		sprintf(tmpbuf, "raid level = %d\n", vol_data_list[i].raid_level);
		strcat(buf, tmpbuf);
		// drive no
		strcpy(tmpbuf,"drive no = ");
		for(j=0;j<vol_data_list[i].list_cnt;j++) {
			sprintf(tmp_str,"%d,",vol_data_list[i].drive_no_list[j]);
			strcat(tmpbuf,tmp_str);
		}
		tmpbuf[strlen(tmpbuf)-1] = '\0';
		strcat(tmpbuf, "\n");

		strcat(buf, tmpbuf);
		// status
		sprintf(tmpbuf, "status = %d\n", vol_data_list[i].status);
		strcat(buf, tmpbuf);
		// record time
		(void) time(&the_time);
		sprintf(tmpbuf, "record_time = %s\n", ctime(&the_time));
		strcat(buf, tmpbuf);
#ifndef	STORAGE_FILE_LOCK
		if (write(fd, buf, strlen(buf))<0) {
			flock(fd, LOCK_UN);
			close(fd);
			free(vol_data_list);
			return ERROR_WRITE_FILE;
		}
#else
		if (fputs(buf, fp)<0) {
			free(vol_data_list);
			fclose(fp);
			NAS_File_Unlock(STORAGE_CONF);
			return ERROR_WRITE_FILE;
		}
#endif
	}
#ifndef	STORAGE_FILE_LOCK	
	// unlock the file & close it
	flock(fd, LOCK_UN);
	close(fd);
#else
	fclose(fp);
	NAS_File_Unlock(STORAGE_CONF);
#endif
	free(vol_data_list);
	return SUCCESS;
}

int Get_Volume_No_For_Drive(int drive_no)
{
	int vol_cnt=0,i=0,j=0;
	VOLUME_CONF_EX *vol_data_list=NULL;

	//sort volume record
	vol_cnt = Get_All_Volumes_Conf_Ex_2(&vol_data_list);

	if (vol_cnt<=0) return ERROR_NOT_FOUND;

	for(i=0;i<vol_cnt;i++) {
		for(j=0;j<vol_data_list[i].list_cnt;j++) {
			if(vol_data_list[i].drive_no_list[j] == drive_no) {
				i=vol_data_list[i].vol_no; // Catherine 2002/04/22
				free(vol_data_list);
				return i; // Catherine 2002/04/22
			}
		}
		for(j=0;j<vol_data_list[i].spare_list_cnt;j++) {
			if(vol_data_list[i].spare_drive_list[j] == drive_no) {
				i=vol_data_list[i].vol_no; // Catherine 2002/04/22
				free(vol_data_list);
				return i; // Catherine 2002/04/22
			}
		}
	}
	free(vol_data_list);
	return ERROR_NOT_FOUND;
}

char *Get_Volume_Message(VOLUME_CONF raid_vol_data){
	return NULL;
}
/*----------------- Get Volume Info function -----------------*/
int Get_One_Volume_Info(int vol_no,VOLUME_INFO *vol_info)
{
	VOLUME_CONF vol_data;
	PARTITION_INFO part_info;
	int ret;
#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;
	
	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	ret = Get_One_Volume_Conf(vol_no,&vol_data);
	
	if(ret<0) return ret;
	
	vol_info->vol_data = vol_data;
		
	if(vol_info->vol_data.status > S_READY && vol_info->vol_data.status < S_REBUILDING_RAID) {
		vol_info->total_size = -1;
		vol_info->free_size  = -1;
		vol_info->resync_per = -1;
		return SUCCESS;	
	}
		
	ret=Get_Partition_Info(vol_data.device_name,&part_info);
	if(ret != SUCCESS) {
		switch (ret) {
			case PARTITION_NOT_EXIST : 
				if(vol_info->vol_data.raid_level != -2)
					vol_info->vol_data.status = S_NOT_ACTIVE;
				else
					vol_info->vol_data.status = S_UNINITIALIZE;
				break;
			case ERROR_NOT_MOUNTED :
				if (Verify_Hidden_Conf_File(vol_info->vol_data.drive_no_list[0])<0 )
					vol_info->vol_data.status = S_INVALID;
				else
				if(vol_info->vol_data.status != S_UNINITIALIZE)
					vol_info->vol_data.status = S_NOT_MOUNTED;
				break;
			default:
				vol_info->vol_data.status = S_UNKNOW_ERROR;
		}
		vol_info->total_size = -1;
		vol_info->free_size  = -1;
		vol_info->resync_per = -1;
	}
	else if(vol_info->vol_data.status <= S_READY || vol_info->vol_data.status == S_REBUILDING_RAID) 
	{
		vol_info->total_size = part_info.total_size;
		vol_info->free_size  = part_info.free_size;
		vol_info->resync_per = -1;

	
		if(vol_info->vol_data.raid_level == -2) {// Its single volume
			if(!Is_Volume_HD_Valid(vol_no)){
				if(Is_Volume_HD_Exist(vol_no))
					vol_info->vol_data.status = S_INVALID;
				else
					vol_info->vol_data.status = S_NOT_EXIST;	
			}
			else
				vol_info->vol_data.status = S_READY;
				
		}
		else {// Its raid volume
			RAID_STATUS sts_rd;
			ret = Get_One_RaidDev_Status(vol_data.device_name,&sts_rd);
			if(ret == SUCCESS) {
				// Catherine 2002/06/11 for resync only ==>
				if(sts_rd.resync_per >= RAID_RESYNC_PER_BASE) {
					vol_info->resync_per = sts_rd.resync_per-RAID_RESYNC_PER_BASE;
					vol_info->vol_data.status = S_READY;
				} else // <== Catherine 2002/06/11 for resync only
				if(sts_rd.resync_per > -1){
					vol_info->resync_per = sts_rd.resync_per;
					vol_info->vol_data.status = S_REBUILDING_RAID;
				}
				else if(sts_rd.active_drive_cnt==vol_info->vol_data.list_cnt-1){
					if(vol_info->vol_data.raid_level == MIRROR || vol_info->vol_data.raid_level == RAID5)
						vol_info->vol_data.status = S_DEGRADED;
					else if(vol_info->vol_data.raid_level == STRIPING)

						vol_info->vol_data.status = S_NOT_ACTIVE;
				}
				else if(sts_rd.active_drive_cnt<vol_info->vol_data.list_cnt-1)
					vol_info->vol_data.status=S_NOT_ACTIVE;
				else
					vol_info->vol_data.status = S_READY;
			}
			else 
			if(ret == ERROR_NOT_FOUND)
				vol_info->vol_data.status = S_NOT_ACTIVE;
			else
				vol_info->vol_data.status = S_UNKNOW_ERROR;
		}	
	}
	
	Set_Volume_Status(vol_no,vol_info->vol_data.status);
	return SUCCESS;
	
}
int Get_All_Volumes_Info(VOLUME_INFO *vol_info_list,int list_cnt)
{
	VOLUME_CONF vol_data_list[MAX_VOLUME_NO];
	PARTITION_INFO part_info;
	int vol_cnt,i,ret;
#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;
	
	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	ret = Get_All_Volumes_Conf(vol_data_list,MAX_VOLUME_NO);

	if(ret<=0) return ret;
	
	vol_cnt = ret;

	sort_volume_data_for_show(vol_data_list, vol_cnt); // Catherine for test

	if(list_cnt == 0) return vol_cnt;

	for(i=0;i<vol_cnt;i++) {
//		vol_info_list[i].vol_data = vol_data_list[i];
		/* modify by Kent 2001/11/30 */
		memcpy(&vol_info_list[i].vol_data, &vol_data_list[i], sizeof(VOLUME_CONF));
		
		if(vol_info_list[i].vol_data.status > S_READY && vol_info_list[i].vol_data.status < S_REBUILDING_RAID) {
			vol_info_list[i].total_size = -1;
			vol_info_list[i].free_size  = -1;
			vol_info_list[i].resync_per = -1;
			continue;
		}
	
		ret=Get_Partition_Info(vol_data_list[i].device_name,&part_info);
		if(ret != SUCCESS) {

			switch (ret) {
				case PARTITION_NOT_EXIST : 
					if(vol_info_list[i].vol_data.raid_level != -2)
						vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					else if(Is_Volume_HD_Exist(vol_info_list[i].vol_data.vol_no)) 
						vol_info_list[i].vol_data.status = S_UNINITIALIZE;
					else
						vol_info_list[i].vol_data.status = S_NOT_EXIST;
					break;
				case ERROR_NOT_MOUNTED :
						if(Verify_Hidden_Conf_File(vol_info_list[i].vol_data.drive_no_list[0])<0 )
							vol_info_list[i].vol_data.status = S_INVALID;
						else
						if(vol_info_list[i].vol_data.status != S_UNINITIALIZE)
							vol_info_list[i].vol_data.status = S_NOT_MOUNTED;
						break;
				default:
						vol_info_list[i].vol_data.status = S_UNKNOW_ERROR;
			}
			vol_info_list[i].total_size = -1;
			vol_info_list[i].free_size  = -1;
			vol_info_list[i].resync_per = -1;
		}
		else if(vol_info_list[i].vol_data.status <= S_READY || vol_info_list[i].vol_data.status == S_REBUILDING_RAID) 
		{
			vol_info_list[i].total_size = part_info.total_size;
			vol_info_list[i].free_size  = part_info.free_size;
			vol_info_list[i].resync_per = -1;
	
			if(vol_info_list[i].vol_data.raid_level == -2) { // Its single volume
				if(!Is_Volume_HD_Valid(vol_info_list[i].vol_data.vol_no)){
					if(Is_Volume_HD_Exist(vol_info_list[i].vol_data.vol_no)) {
						if(Verify_Hidden_Conf_File(vol_info_list[i].vol_data.drive_no_list[0])  <= ERROR_HD_DEVICE_NAME )
							vol_info_list[i].vol_data.status = S_INVALID;
						else
							vol_info_list[i].vol_data.status = S_UNINITIALIZE;
						
					}
					else
						vol_info_list[i].vol_data.status = S_NOT_EXIST;
				}
				else
					vol_info_list[i].vol_data.status = S_READY;
			}
			else {// Its raid volume
				RAID_STATUS sts_rd;
				ret = Get_One_RaidDev_Status(vol_data_list[i].device_name,&sts_rd);
				if(ret == SUCCESS) {
					// Catherine 2002/06/11 for resync only ==>
					if(sts_rd.resync_per >= RAID_RESYNC_PER_BASE) {
						vol_info_list[i].resync_per = sts_rd.resync_per-RAID_RESYNC_PER_BASE;
						vol_info_list[i].vol_data.status = S_READY;
					} else // <== Catherine 2002/06/11 for resync only
					if(sts_rd.resync_per > -1){
						vol_info_list[i].resync_per = sts_rd.resync_per;
						vol_info_list[i].vol_data.status = S_REBUILDING_RAID;
					}
					else if(sts_rd.active_drive_cnt == vol_info_list[i].vol_data.list_cnt-1){
						if(vol_info_list[i].vol_data.raid_level == MIRROR || vol_info_list[i].vol_data.raid_level == RAID5)
							vol_info_list[i].vol_data.status = S_DEGRADED;
						else if(vol_info_list[i].vol_data.raid_level == STRIPING)
							vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					}
					else if(sts_rd.active_drive_cnt < vol_info_list[i].vol_data.list_cnt-1)
						vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					else
						vol_info_list[i].vol_data.status = S_READY;
				}
				else 
				if(ret == ERROR_NOT_FOUND)
					vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
				else
					vol_info_list[i].vol_data.status = S_UNKNOW_ERROR;
			}	
		}
		
		//recheck hd while status is ready
		Set_Volume_Status(vol_info_list[i].vol_data.vol_no,vol_info_list[i].vol_data.status); // Catherine 2002/04/22
	}
	
	return vol_cnt;
}

int Get_All_Volumes_Conf(VOLUME_CONF *vol_data_list,int list_cnt)
{
	int i,cnt=0,ret;
	VOLUME_CONF vol_data;
	
	for(i=1;i<MAX_VOLUME_NO+1;i++) {
		ret = Get_One_Volume_Conf(i,&vol_data);
		if(ret < 0)
			continue;
		else {
			if(vol_data_list != NULL && list_cnt > 0) {
				// Catherine 2002/04/22
				memcpy((void*)&vol_data_list[cnt],
					(void*) &vol_data,
					sizeof(VOLUME_CONF));
			}
			
			cnt++;
		}
	}
	return cnt;
}


int Get_One_Volume_Conf(int vol_no,VOLUME_CONF *vol_data)
{
	char vol_section[20];
	char tmp_buf[BUF_SIZE],drv_no_str[4];
	int drive_no_cnt=1,ret;
	
	sprintf(vol_section,"VOLUME %d",vol_no);


	//set vol no
	vol_data->vol_no = vol_no;
	// get device name
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"device name",vol_data->device_name,HD_DEVICE_NAME_LENGTH);
	if(ret<0) return ret;
		
	// get raid level
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"raid level",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	vol_data->raid_level = atoi(tmp_buf);		
	//get drive no list & list count
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"drive no",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	while(Get_String_Field(tmp_buf,drive_no_cnt,',',drv_no_str,4)>=0) {	
		vol_data->drive_no_list[drive_no_cnt-1] = atoi(drv_no_str);
		drive_no_cnt++;
	}
	vol_data->list_cnt = drive_no_cnt-1;
	//get volume status
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"status",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	vol_data->status = atoi(tmp_buf);
	
	return SUCCESS;
}

int Get_Uninit_Disk_Volumes_Conf(VOLUME_CONF *vol_data_list,int list_cnt)
{
	int i,cnt=0,vol_cnt=0;
	VOLUME_INFO vol_info[20];
#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;
	
	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else{
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	//update current info;
	vol_cnt = Get_All_Volumes_Info(vol_info,20);
	
	for(i=0;i<vol_cnt;i++) {
		//just get single disk volume
		if(vol_info[i].vol_data.raid_level == -2 &&
			( vol_info[i].vol_data.status == S_UNINITIALIZE || vol_info[i].vol_data.status == S_INVALID)) {
				if(!Is_Volume_HD_Exist(vol_info[i].vol_data.vol_no)) continue;

				// Catherine 2002/04/22
				memcpy((void*)&vol_data_list[cnt],
					(void*)&(vol_info[i].vol_data),
					sizeof(VOLUME_CONF));

				cnt++;
				if (cnt>=list_cnt)
					return cnt;
		}
	}
	return cnt;
}

int Get_Single_Disk_Volumes_Conf(VOLUME_CONF *vol_data_list,int list_cnt)
{	
	int i,cnt=0;
	VOLUME_INFO vol_info[20];
	int vol_cnt=0;

#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	//update current info;
	vol_cnt = Get_All_Volumes_Info(vol_info,20);
	
	for(i=0;i<vol_cnt;i++) {
		//just get single disk volume
		if(vol_info[i].vol_data.raid_level == -2) {
			// Catherine 2002/04/22
			memcpy((void*)&vol_data_list[cnt],
				(void*)&(vol_info[i].vol_data),
				sizeof(VOLUME_CONF));

			/* Catherine 2002/04/23 remove dummy code	
			if(vol_data_list[cnt].status == S_READY) {
				if(!Is_Volume_HD_Valid(vol_data_list[cnt].vol_no)){
					if(Is_Volume_HD_Exist(vol_data_list[cnt].vol_no))
						vol_data_list[cnt].status = S_INVALID;
					else 
						vol_data_list[cnt].status = S_NOT_EXIST;
					
					Set_Volume_Status(vol_data_list[cnt].vol_no,vol_data_list[cnt].status);
				}
			}
			*/
			if(vol_data_list[cnt].status == S_UNINITIALIZE) {
				if(!Is_Volume_HD_Exist(vol_data_list[cnt].vol_no)) continue;
			}

			cnt++;
		}
	}
	sort_volume_data_for_show(vol_data_list, cnt); // Catherine 2002/04/22

	return cnt;
}

int Get_Raid_Disk_Volumes_Info(VOLUME_INFO *vol_info_list,int list_cnt)
{
	
	int i,cnt=0,ret;
	VOLUME_INFO m_info;

#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else{
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	for(i=1;i<MAX_VOLUME_NO+1;i++) {
		ret = Get_One_Volume_Info(i,&m_info);
		if(ret < 0)
			continue;
		else {
			if(m_info.vol_data.raid_level > -2) {
				memcpy((void*)&(vol_info_list[cnt]),
					(void*)&m_info,
					sizeof(VOLUME_INFO));

				/* Catherine 2002/04/23 remove dummy cod
				if(m_info.vol_data.status == S_READY) {
					if(!Is_Volume_HD_Valid(i)){
						if(Is_Volume_HD_Exist(i))
							m_info.vol_data.status = S_INVALID;
						else
							m_info.vol_data.status = S_NOT_EXIST;
						
						Set_Volume_Status(i,m_info.vol_data.status);
					}
				}
				*/
				cnt++;
				if (cnt>=list_cnt) {
					return cnt;
				}

			}
		}
	}
	return cnt;
}

int Get_All_Configured_Vol_For_Share(VOLUME_INFO *mount_vol,int list_cnt)
{
	VOLUME_INFO vol_info[MAX_VOLUME_NO];
	int vol_cnt,i,ret_cnt=0;
	
	vol_cnt = Get_All_Volumes_Info(vol_info,MAX_VOLUME_NO);
	for(i=0;i<vol_cnt;i++) {
		if(vol_info[i].vol_data.status == S_READY ||
			vol_info[i].vol_data.status == S_DEGRADED ||
			vol_info[i].vol_data.status == S_REBUILDING_RAID) {
			//mount_vol[ret_cnt++] = vol_info[i];
			memcpy((void*)&(mount_vol[ret_cnt++]),
				(void*) &(vol_info[i]),
				sizeof(VOLUME_INFO));
		}
	}
		
	return ret_cnt;
}


int Get_Configured_Vol_No_For_Share(char *share_name)
{
	NAS_SHARE_INFO	share_info;
	VOLUME_INFO	*vol_info;
	int vol_cnt,i;
	char mp_str[128];
		
	strcpy(share_info.share_name,share_name);
	share_info.read_list_cnt = 0;       
	share_info.read_list_ptr = NULL;       
	share_info.write_list_cnt = 0;     
	share_info.read_list_ptr = NULL;
	Get_NAS_Share_Info_Ex(&share_info);
	
	vol_cnt = Get_All_Configured_Vol_For_Share_Ex(&vol_info);

	if (vol_cnt<=0)
		return -1;

	for(i=0;i<vol_cnt;i++) {
		Get_MP_String(vol_info[i].vol_data.drive_no_list[0],DATA_PART,mp_str);
		if( strstr(share_info.path,mp_str) != NULL ) {
			if (share_info.read_list_cnt>0)
				free(share_info.read_list_ptr);
			if (share_info.write_list_cnt>0)
				free(share_info.write_list_ptr);
			free(vol_info);
			return i+1;
		}
	}	
	if (share_info.read_list_cnt>0)
		free(share_info.read_list_ptr);
	if (share_info.write_list_cnt>0)
		free(share_info.write_list_ptr);
	free(vol_info);
	return -1;

}


/*------------------- Volume status function -----------------------------*/

//--------------------------------------------------//
// this function will check each HD belong to this  //
// volume. The check item list as belows,	    //
// 1. HD exist					    //
// 2. check .conf is correct              	    //		   		
//--------------------------------------------------//
BOOL Is_Volume_HD_Exist(int vol_no)
{
	VOLUME_CONF vol_data;
	int ret,i;
	char device_name[HD_DEVICE_NAME_LENGTH];

	if((ret = Get_One_Volume_Conf(vol_no,&vol_data))<0) return FALSE;

	for(i=0;i<vol_data.list_cnt;i++){
		Get_HD_DevName(vol_data.drive_no_list[i],device_name);
		if(!Is_HD_Exist(device_name)) return FALSE;
	}
	
	return TRUE;
}

BOOL Is_Volume_HD_Valid(int vol_no)
{
	VOLUME_CONF vol_data;
	int ret,i;
	
	if((ret = Get_One_Volume_Conf(vol_no,&vol_data))<0) return FALSE;
	for(i=0;i<vol_data.list_cnt;i++){
		if(! Is_HD_Valid(vol_data.drive_no_list[i])) 
			return FALSE;
	}
		
	return TRUE;
}

int Set_Volume_Status(int vol_no,int status)
{
	char vol_section[20],tmp_str[10];
	int ret, retstatus, i=0;

	sprintf(vol_section,"VOLUME %d",vol_no);
	sprintf(tmp_str,"%d",status);
// Catherine 2002/03/26	==>
//	return (Conf_Set_Field(STORAGE_CONF,vol_section,"status",tmp_str));

	ret = Conf_Set_Field(STORAGE_CONF,vol_section,"status",tmp_str);
	retstatus = Get_Volume_Status(vol_no);
	while (retstatus!=status && i++<100) {
		sleep(1);
		ret = Conf_Set_Field(STORAGE_CONF,vol_section,"status",tmp_str);
		retstatus = Get_Volume_Status(vol_no);
	}
	return ret;
// <== Catherine 2002/03/26
}

int Get_Volume_Status(int vol_no)
{
	char vol_section[20],tmp_str[10],ret;
	sprintf(vol_section,"VOLUME %d",vol_no);
	
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"status",tmp_str,10);
	//if(ret <= 0) return ret;
//	if(ret < 0) return ret; // Catherine 2002/03/25
	if(ret!=0)
                return ret; // Catherine 2002/03/25
        else
                return (atoi(tmp_str));
}


//-------------------- for printer spool -----------------------//
BOOL Is_Spool_Partition_Ready()
{
	char device_name[HD_DEVICE_NAME_LENGTH];
	
	Get_HD_DevName(2,device_name);
	if(!Is_HD_Exist(device_name))	return FALSE;
	if(!Is_HD_Valid(2))   		return FALSE;
	
	return TRUE;	
}

//-------------------- for dynamic memory allocation -----------------------//

int Get_All_Configured_Vol_For_Share_Ex(VOLUME_INFO **mount_vol_ptr)
{
	VOLUME_INFO *vol_info=NULL,*mount_vol=NULL;
	int vol_cnt,i,ret_cnt=0;
	void *ptr=NULL;

	vol_cnt = Get_All_Volumes_Info_Ex(&vol_info);

	if (vol_cnt<=0)
		return 0;

	for(i=0;i<vol_cnt;i++) {
		if(vol_info[i].vol_data.status == S_READY ||
			vol_info[i].vol_data.status == S_DEGRADED ||
			vol_info[i].vol_data.status == S_REBUILDING_RAID) {
			ret_cnt++;

			// Catherine 2002/04/09 ==>
			//mount_vol = (VOLUME_INFO *) realloc(mount_vol,
			//		ret_cnt * sizeof(VOLUME_INFO));
			ptr = realloc((void*)mount_vol,
				ret_cnt * sizeof(VOLUME_INFO));

			if (ptr==NULL) {
#ifdef DEBUG
				printf("Fail to re-allocate buffer!\n");
#endif
				if (mount_vol) free(mount_vol);
				free(vol_info);
				return ERROR_OUT_OF_MEMORY;
			}
			mount_vol = (VOLUME_INFO *)ptr;
			// <== Catherine 2002/04/09

			memcpy(&(mount_vol[ret_cnt-1]), &(vol_info[i]), sizeof(VOLUME_INFO));
		}
	}

	free(vol_info);
	*mount_vol_ptr = mount_vol;
	return ret_cnt;
}

int Get_All_Volumes_Info_Ex(VOLUME_INFO **vol_info_list_ptr)
{
	VOLUME_CONF *vol_data_list=NULL;
	PARTITION_INFO part_info;
	VOLUME_INFO *vol_info_list=NULL;
	int vol_cnt,i,ret;

#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	ret = Get_All_Volumes_Conf_Ex(&vol_data_list);

	if(ret<=0) return ret;
	
	vol_cnt = ret;

	sort_volume_data_for_show(vol_data_list, vol_cnt); // Catherine 2002/04/22

	vol_info_list = (VOLUME_INFO*)calloc(vol_cnt, sizeof(VOLUME_INFO));

	if (vol_info_list==NULL) { // Catherine 2002/04/09
		free(vol_data_list);
		return ERROR_OUT_OF_MEMORY;
	}

	for(i=0;i<vol_cnt;i++) {
		memcpy(&vol_info_list[i].vol_data, &vol_data_list[i], sizeof(VOLUME_CONF));

		if(vol_info_list[i].vol_data.status > S_READY &&
			vol_info_list[i].vol_data.status < S_REBUILDING_RAID) {
			vol_info_list[i].total_size = -1;
			vol_info_list[i].free_size  = -1;
			vol_info_list[i].resync_per = -1;
			continue;
		}

		ret=Get_Partition_Info(vol_data_list[i].device_name,&part_info);
		if(ret != SUCCESS) {
			switch (ret) {
				case PARTITION_NOT_EXIST : 
					if(vol_info_list[i].vol_data.raid_level != -2)
						vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					else if(Is_Volume_HD_Exist(vol_info_list[i].vol_data.vol_no)) 
						vol_info_list[i].vol_data.status = S_UNINITIALIZE;
					else
						vol_info_list[i].vol_data.status = S_NOT_EXIST;
					break;
				case ERROR_NOT_MOUNTED :
					if(Verify_Hidden_Conf_File(vol_info_list[i].vol_data.drive_no_list[0])<0 )
						vol_info_list[i].vol_data.status = S_INVALID;
					else
					if(vol_info_list[i].vol_data.status != S_UNINITIALIZE)
						vol_info_list[i].vol_data.status = S_NOT_MOUNTED;
					break;
				default:
					vol_info_list[i].vol_data.status = S_UNKNOW_ERROR;
			}
			vol_info_list[i].total_size = -1;
			vol_info_list[i].free_size  = -1;
			vol_info_list[i].resync_per = -1;
		}
		else if(vol_info_list[i].vol_data.status <= S_READY || vol_info_list[i].vol_data.status == S_REBUILDING_RAID) 
		{
			vol_info_list[i].total_size = part_info.total_size;
			vol_info_list[i].free_size  = part_info.free_size;
			vol_info_list[i].resync_per = -1;

			if(vol_info_list[i].vol_data.raid_level == -2) { // Its single volume
				if(!Is_Volume_HD_Valid(vol_info_list[i].vol_data.vol_no)){
					if(Is_Volume_HD_Exist(vol_info_list[i].vol_data.vol_no)) {
						if(Verify_Hidden_Conf_File(vol_info_list[i].vol_data.drive_no_list[0])  <= ERROR_HD_DEVICE_NAME )
							vol_info_list[i].vol_data.status = S_INVALID;
						else
							vol_info_list[i].vol_data.status = S_UNINITIALIZE;
					}
					else
						vol_info_list[i].vol_data.status = S_NOT_EXIST;
				}
				else
					vol_info_list[i].vol_data.status = S_READY;
			}
			else {// Its raid volume
				RAID_STATUS sts_rd;
				ret = Get_One_RaidDev_Status(vol_data_list[i].device_name,&sts_rd);
				if(ret == SUCCESS) {
					// Catherine 2002/06/11 ==>
					if (sts_rd.resync_per>=RAID_RESYNC_PER_BASE) {
						vol_info_list[i].resync_per = sts_rd.resync_per-RAID_RESYNC_PER_BASE;
						vol_info_list[i].vol_data.status = S_READY;
					} else // <== Catherine 2002/06/11 
					if(sts_rd.resync_per > -1){
						vol_info_list[i].resync_per = sts_rd.resync_per;
						vol_info_list[i].vol_data.status = S_REBUILDING_RAID;
					}
					else if(sts_rd.active_drive_cnt == vol_info_list[i].vol_data.list_cnt-1){
						if(vol_info_list[i].vol_data.raid_level == MIRROR || vol_info_list[i].vol_data.raid_level ==RAID5)
							vol_info_list[i].vol_data.status = S_DEGRADED;
						else if(vol_info_list[i].vol_data.raid_level == STRIPING)
							vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
												
					}

					else if(sts_rd.active_drive_cnt < vol_info_list[i].vol_data.list_cnt-1)
						vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					else
						vol_info_list[i].vol_data.status = S_READY;
				}
				else if (ret == ERROR_NOT_FOUND)
					vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
				else
					vol_info_list[i].vol_data.status = S_UNKNOW_ERROR;
			}	
		}

		//recheck hd while status is ready
		Set_Volume_Status(vol_info_list[i].vol_data.vol_no,vol_info_list[i].vol_data.status);	// Catherine 2002/04/22
	}
	free(vol_data_list);
	*vol_info_list_ptr = vol_info_list;
	return vol_cnt;
}

int Get_All_Volumes_Conf_Ex(VOLUME_CONF **vol_data_list_ptr)
{
	int i,cnt=0,ret;
	VOLUME_CONF *vol_data_list=NULL;
	VOLUME_CONF vol_data;
	void *ptr=NULL;

	for(i=1;i<MAX_VOLUME_NO+1;i++) {
		ret = Get_One_Volume_Conf(i,&vol_data);
		if (ret < 0)
			continue;
		else {
			ptr = realloc((void*)vol_data_list,
				(cnt+1)*sizeof(VOLUME_CONF));

			if (ptr==NULL) {
				if (vol_data_list) free(vol_data_list);
				return ERROR_OUT_OF_MEMORY;
			}
			vol_data_list = (VOLUME_CONF *)ptr;

			// Catherine 2002/04/22
			memcpy((void*)&vol_data_list[cnt],
				(void*) &vol_data,
				sizeof(VOLUME_CONF));
			cnt++;
		}
	}
	*vol_data_list_ptr = vol_data_list;
	return cnt;
}

int Get_Uninit_Disk_Volumes_Conf_Ex(VOLUME_CONF **vol_data_list_ptr)
{
	VOLUME_CONF *vol_data_list=NULL;
	int i,cnt=0,vol_cnt=0;
	VOLUME_INFO *vol_info=NULL;
	void *ptr=NULL;
	
#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else{
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	//update current info;
	vol_cnt=Get_All_Volumes_Info_Ex(&vol_info);
	if (vol_cnt<=0) return 0;

	for(i=0;i<vol_cnt;i++) {
		//just get single disk volume
		if(vol_info[i].vol_data.raid_level == -2 &&
			( vol_info[i].vol_data.status == S_UNINITIALIZE || vol_info[i].vol_data.status == S_INVALID)) {
			if(!Is_Volume_HD_Exist(vol_info[i].vol_data.vol_no)) continue;

			// Catherine 2002/04/09 ==>
			ptr = realloc((void*)vol_data_list,
				(cnt+1)*sizeof(VOLUME_CONF));

			if (ptr==NULL) {
				if (vol_data_list) free(vol_data_list);
				free(vol_info);
				return ERROR_OUT_OF_MEMORY;
			}
			vol_data_list = (VOLUME_CONF *)ptr;
			// <== Catherine 2002/04/09

			memcpy((void*)vol_data_list,
				(void*)&(vol_info[i].vol_data),
				sizeof(VOLUME_CONF));
			cnt++;
		}
	}
	free(vol_info);
	*vol_data_list_ptr = vol_data_list;
	return cnt;
}

int Get_Single_Disk_Volumes_Conf_Ex(VOLUME_CONF **vol_data_list_ptr)
{	
	int i,cnt=0,vol_cnt=0;
	VOLUME_INFO *vol_info=NULL;
	VOLUME_CONF *vol_data_list=NULL;
	void *ptr=NULL;
	
#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	//update current info;
	vol_cnt=Get_All_Volumes_Info_Ex(&vol_info);
	if (vol_cnt<=0) return 0;
	
	for(i=0;i<vol_cnt;i++) {
		//just get single disk volume
		if(vol_info[i].vol_data.raid_level == -2) {
			ptr = realloc((void*)vol_data_list,
				(cnt+1)*sizeof(VOLUME_CONF));
			if (ptr==NULL) {
				if (vol_data_list) free(vol_data_list);
				free(vol_info);
				return ERROR_OUT_OF_MEMORY;
			}
			vol_data_list = (VOLUME_CONF *)ptr;

			// Catherine 2002/04/22
			memcpy((void*)&vol_data_list[cnt],
				(void*)&(vol_info[i].vol_data),
				sizeof(VOLUME_CONF));

			/* Catherine 2002/04/23 remove dummy code
			if(vol_data_list[cnt].status == S_READY) {
				if(!Is_Volume_HD_Valid(vol_data_list[cnt].vol_no)){
					if(Is_Volume_HD_Exist(vol_data_list[cnt].vol_no))
						vol_data_list[cnt].status = S_INVALID;
					else 
						vol_data_list[cnt].status = S_NOT_EXIST;
						
					Set_Volume_Status(vol_data_list[cnt].vol_no,vol_data_list[cnt].status);
				}
			}
			*/
			if(vol_data_list[cnt].status == S_UNINITIALIZE) {
				if(!Is_Volume_HD_Exist(vol_data_list[cnt].vol_no)) continue;
			}

			cnt++;
		}
	}
	free(vol_info);
	sort_volume_data_for_show(vol_data_list, cnt); // Catherine 2002/04/22
	*vol_data_list_ptr=vol_data_list;
	return cnt;
}

int Get_Raid_Disk_Volumes_Info_Ex(VOLUME_INFO **vol_info_list_ptr)
{
	int i,cnt=0,ret;
	VOLUME_INFO m_info;
	VOLUME_INFO *vol_info_list=NULL;
	void *ptr=NULL;

#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else{
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	for(i=1; i<MAX_VOLUME_NO+1; i++) {
		ret = Get_One_Volume_Info(i,&m_info);
		if(ret < 0)
			continue;
		else {
			if(m_info.vol_data.raid_level > -2) {
				// Catherine 2002/04/09 ==>
				ptr = realloc((void*)vol_info_list,
					(cnt+1)*sizeof(VOLUME_INFO));
				if (ptr==NULL) {
					if (vol_info_list) free(vol_info_list);
					return ERROR_OUT_OF_MEMORY;
				}
				vol_info_list = (VOLUME_INFO *)ptr;

				//vol_info_list[cnt] = m_info;
				memcpy((void*)&(vol_info_list[cnt]),
					(void*)&m_info,
					sizeof(VOLUME_INFO));
				// <== Catherine 2002/04/09

				/* Catherine 2002/04/23 remove dummy code	
				if(m_info.vol_data.status == S_READY) {
					if(!Is_Volume_HD_Valid(i)){
						if(Is_Volume_HD_Exist(i))
							m_info.vol_data.status = S_INVALID;
						else
							m_info.vol_data.status = S_NOT_EXIST;
						
						Set_Volume_Status(m_info.vol_data.vol_no,m_info.vol_data.status); // Catherine 2002/04/22
					}
				}
				*/
				cnt++;
			}
		}
	}
	*vol_info_list_ptr = vol_info_list;
	return cnt;
}

/*********************************************************
* Get_Volume_Path_For_Drive
* Description : getting the volume path for the drive_no
* Output : SUCCESS -- vol_path contains a valid volume path
*	ERROR_NOT_FOUND -- the drive is not found in any volume
*			of the storage.conf
*********************************************************/
int Get_Volume_Path_For_Drive(int drive_no, char* vol_path)
{
	int vol_cnt=0,i=0,j=0;
	VOLUME_CONF_EX *vol_data_list=NULL;
	
	//sort volume record
	vol_cnt = Get_All_Volumes_Conf_Ex_2(&vol_data_list);
	if (vol_cnt<=0) return ERROR_NOT_FOUND;

	for(i=0;i<vol_cnt;i++) {
		for(j=0;j<vol_data_list[i].list_cnt;j++) {
			if(vol_data_list[i].drive_no_list[j] == drive_no) {
				strcpy(vol_path, vol_data_list[i].device_name);
				free(vol_data_list);
				return SUCCESS;
			}
		}
		for(j=0;j<vol_data_list[i].spare_list_cnt;j++) {
			if(vol_data_list[i].spare_drive_list[j] == drive_no) {
				strcpy(vol_path, vol_data_list[i].device_name);
				free(vol_data_list);
				return SUCCESS;
			}
		}
	}
	free(vol_data_list);
	return ERROR_NOT_FOUND;

}

int m_cmp_ex(const void *m1,const void *m2) // Catherine 2002/04/22
{

	VOLUME_CONF_EX *a,*b;
	
	a = (VOLUME_CONF_EX*) m1;
	b = (VOLUME_CONF_EX*) m2;

	if (a->vol_no>b->vol_no) return 1;
	if (a->vol_no<b->vol_no) return -1;

	return 0;

}
void sort_volume_data_ex(VOLUME_CONF_EX *vol_data_list,int list_cnt)
{
	qsort((void*)vol_data_list,list_cnt,sizeof(VOLUME_CONF_EX),m_cmp_ex);	 // Catherine 2002/04/22
}

/**********************************************************************
** Create_Raid_Volume_Ex
** Description:	Create a RAID volume w/ spare disk support if level=1 or 5
** Output:	SUCCESS -- create successfully
**		ERROR_FAIL
**		ERROR_FUNCTION_PARAMETER
***********************************************************************/
int Create_Raid_Volume_Ex(int *vol_no_list,int list_cnt,
	int *spare_vol_no_list,int spare_list_cnt,int raid_level)
{
	int i=0,ret,vol_no;
	char rd_name[HD_DEVICE_NAME_LENGTH];
	VOLUME_CONF_EX vol_data;
	VOLUME_CONF_EX tmp_vol_data;

	//prepare raid volume data
	vol_data.list_cnt = list_cnt;
	vol_data.raid_level = raid_level;
	for(i=0;i<list_cnt;i++){

		ret = Get_One_Volume_Conf_Ex(vol_no_list[i],&tmp_vol_data);
		if(ret<0) return ret; // get volume data fail

		if(tmp_vol_data.raid_level != -2 )
			return ERROR_FUNCTION_PARAMETER; //not signal volume
		if(tmp_vol_data.status != S_UNINITIALIZE &&
			tmp_vol_data.status != S_INVALID &&
			tmp_vol_data.status != S_READY &&
			tmp_vol_data.status != S_NOT_MOUNTED)
			return ERROR_FAIL;
		vol_data.drive_no_list[i] = tmp_vol_data.drive_no_list[0];
	}

	if ((raid_level==1) || (raid_level==5)) {
		vol_data.spare_list_cnt = spare_list_cnt;
		for(i=0;i<spare_list_cnt;i++){

			ret = Get_One_Volume_Conf_Ex(spare_vol_no_list[i],
					&tmp_vol_data);
			if(ret<0) return ret; // get volume data fail
	
			if(tmp_vol_data.raid_level != -2 )
				return ERROR_FUNCTION_PARAMETER;
				//not signal volume

			if(tmp_vol_data.status != S_UNINITIALIZE &&
				tmp_vol_data.status != S_INVALID &&
				tmp_vol_data.status != S_READY &&
				tmp_vol_data.status != S_NOT_MOUNTED)
				return ERROR_FAIL;
			vol_data.spare_drive_list[i]=tmp_vol_data.drive_no_list[0];
		}
	}
	else vol_data.spare_list_cnt = 0;

	if(vol_data.list_cnt > 1 && vol_data.raid_level >-2) {//for raid volume
		switch(vol_data.raid_level) {
			case -1://JBOD
			case 0: //RAID 0
			case 1: //RAID 1
			case 5: //RAID 5
#ifdef DEBUG
				printf("RAID %d starts.....\n",
					vol_data.raid_level);
#endif
				backup_storage_conf();
#ifdef DEBUG
				printf("before Remove_Volume_Record\n");
#endif
				//update volume data
				for(i=0;i<vol_data.list_cnt;i++) { // Catherine 2002/02/25 modified
					VOLUME_CONF_EX tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[i]);
					Stop_User_Quota(vol_no);

					Get_One_Volume_Conf_Ex(vol_no, &tmpvol);
					clear_volume_related_messages_ex(&tmpvol);

					Remove_Volume_Record(vol_no);
				}
				for(i=0;i<vol_data.spare_list_cnt;i++) { // Catherine 2002/02/25 modified
					VOLUME_CONF_EX tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.spare_drive_list[i]);
					Stop_User_Quota(vol_no);	

					Get_One_Volume_Conf_Ex(vol_no, &tmpvol);
					clear_volume_related_messages_ex(&tmpvol);

					Remove_Volume_Record(vol_no);
				}

				// create raid table
				Get_First_Available_RaidName(rd_name,HD_DEVICE_NAME_LENGTH);

				ret = Create_NAS_RaidDev_Ex(rd_name,
					vol_data.drive_no_list,vol_data.list_cnt,
					vol_data.spare_drive_list,vol_data.spare_list_cnt,
					vol_data.raid_level);
				if(ret < 0) return ret;

				strcpy(vol_data.device_name,rd_name);
				vol_data.status = S_INITIALIZING;
				ret = Add_Volume_Record_Ex(&vol_data);

				if (ret<0) return ret;

				//Initialize single disk
				for(i=0;i<vol_data.list_cnt;i++) {
					set_exist_flag(vol_data.drive_no_list[i], "1"); // Catherine 2002/03/24
					if((ret=Init_NAS_Disk(vol_data.drive_no_list[i]))<0){
						Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[i]),
							S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
					if (vol_data.raid_level==5)
						sleep(1);
				}
				//Initialize spare disks
				for(i=0;i<vol_data.spare_list_cnt;i++) {
					set_exist_flag(vol_data.spare_drive_list[i], "1"); // Catherine 2002/03/24
					if((ret=Init_NAS_Disk(vol_data.spare_drive_list[i]))<0){
						Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.spare_drive_list[i]),
							S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
					if (vol_data.raid_level==5)
						sleep(1);
				}

				//get availabe raid device name
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_CREATING_RAID);

				// do make raid
				Make_Raid(rd_name);
				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_FORMATTING);
				Format_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]));
				if((ret=Mount_Volume(Get_Volume_No_For_Drive(vol_data.drive_no_list[0])))< 0 ) {
					Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),
						S_NOT_MOUNTED); // Catherine 2002/03/28
					return ret;
				}

				start_vol_quota(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),1); // Catherine 2002/02/25

				Set_Volume_Status(Get_Volume_No_For_Drive(vol_data.drive_no_list[0]),S_READY);

				break;
			case 4: //RAID 0+1
				backup_storage_conf();	

				//update volume data
				for(i=0;i<vol_data.list_cnt;i++) { // Catherine 2002/02/25 modified
					VOLUME_CONF_EX tmpvol;

					vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[i]);
					Stop_User_Quota(vol_no);

					Get_One_Volume_Conf_Ex(vol_no, &tmpvol);
					clear_volume_related_messages_ex(&tmpvol);

					Remove_Volume_Record(vol_no);
				}

				// create raid table
				strcpy(rd_name,"/dev/md2");
				ret = Create_NAS_RaidDev_Ex(rd_name,
					vol_data.drive_no_list,vol_data.list_cnt,
					vol_data.spare_drive_list,vol_data.spare_list_cnt,
					4);
				if(ret < 0) return ret;

				strcpy(vol_data.device_name,rd_name);
				vol_data.status = S_INITIALIZING;
				ret =Add_Volume_Record_Ex(&vol_data);
				if (ret<0) return ret;

				vol_no = Get_Volume_No_For_Drive(vol_data.drive_no_list[0]);

				//Initialize single disk
				for(i=0;i<vol_data.list_cnt;i++) {
					if((ret=Init_NAS_Disk(vol_data.drive_no_list[i]))<0){
						Set_Volume_Status(vol_no,S_UNINITIALIZE); // Catherine 2002/03/28
						return ret;	
					}
				}

				Set_Volume_Status(vol_no,S_CREATING_RAID);

				//Umount single volume partition to avoid os bug
				for(i=0;i<vol_data.list_cnt;i++) {
					Unmount_NAS_Partition(vol_data.drive_no_list[i],DATA_PART);
				}

				Unmount_Partition(rd_name);
				
				// do make raid
				if((ret = Make_Raid("/dev/md0"))<0) {
					Set_Volume_Status(vol_no,S_RAID_DIED); // Catherine 2002/03/28
					return ret;
				}
				if((ret = Make_Raid("/dev/md1"))<0) {
					Set_Volume_Status(vol_no,S_RAID_DIED); // Catherine 2002/03/28
					return ret;
				}
				if((ret = Make_Raid(rd_name))<0) {
					Set_Volume_Status(vol_no,S_RAID_DIED); // Catherine 2002/03/28
					return ret;
				}

				Set_Volume_Status(vol_no,S_FORMATTING);
				Format_Volume(vol_no);
				if((ret=Mount_Volume(vol_no))< 0 ) {
					Set_Volume_Status(vol_no,S_NOT_MOUNTED); // Catherine 2002/03/28
					return ret;
				}

				start_vol_quota(vol_no,1); // Catherine 2002/02/25

				Set_Volume_Status(vol_no,S_READY);
				break;
			default:
				return ERROR_FUNCTION_PARAMETER;
				break;
		}
	}
	else 
		return ERROR_FUNCTION_PARAMETER;
	return SUCCESS;
}

/**********************************************************************
** Delete_Raid_Volume_Ex
** Description:	Remove a RAID volume w/ spare disk support
** Output:	Error code
**		SUCCESS
**		ERROR_FAIL
***********************************************************************/
int Delete_Raid_Volume_Ex(int vol_no)
{
	VOLUME_CONF_EX raid_vol_data,single_vol_data;
	int ret,i,tmp_vol_no;

	ret = Get_One_Volume_Conf_Ex(vol_no,&raid_vol_data);
	if(ret<0) return ret;

	if(raid_vol_data.raid_level < -1) return ERROR_FAIL;

	//removing raid
	Set_Volume_Status(vol_no,S_REMOVING_RAID);

	Stop_User_Quota(vol_no); // Catherine 2002/02/25

	if(raid_vol_data.status != S_NOT_ACTIVE) {
		
		Umount_Volume(vol_no);
		
		if((ret=Stop_Raid(raid_vol_data.device_name))<0){
			Mount_Volume(vol_no);
			start_vol_quota(vol_no, 0); // Catherine shen
			Set_Volume_Status(vol_no,S_READY);
			return ret;
		}
	}

	if(raid_vol_data.raid_level == 4) {  // raid0+1
		if((ret=Stop_Raid("/dev/md0"))<0) return ret;
		if((ret=Stop_Raid("/dev/md1"))<0) return ret;
		if((ret=Delete_RaidDev("/dev/md0"))<0) return ret;
		if((ret=Delete_RaidDev("/dev/md1"))<0) return ret;
	}
	
	if((ret=Delete_RaidDev(raid_vol_data.device_name))<0) {
		return ret;
	}
	// clear old messages
	clear_volume_related_messages_ex(&raid_vol_data);

	//remove volume record from storage.conf
	Remove_Volume_Record(vol_no);

	//reinitialize single disk volume belong to this volume
	for(i=0;i<raid_vol_data.list_cnt;i++){
		Get_Partition_Name(raid_vol_data.drive_no_list[i],DATA_PART,single_vol_data.device_name);
		single_vol_data.list_cnt = 1;
		single_vol_data.drive_no_list[0] = raid_vol_data.drive_no_list[i];
		single_vol_data.spare_list_cnt = 0;
		single_vol_data.raid_level = -2;
		single_vol_data.status = S_INITIALIZING;
		ret = Add_Volume_Record_Ex(&single_vol_data);
		if(ret<0) {
			return ret;
		}
	}
	for(i=0;i<raid_vol_data.spare_list_cnt;i++){
		Get_Partition_Name(raid_vol_data.spare_drive_list[i],DATA_PART,single_vol_data.device_name);
		single_vol_data.list_cnt = 1;
		single_vol_data.drive_no_list[0] = raid_vol_data.spare_drive_list[i];
		single_vol_data.spare_list_cnt = 0;
		single_vol_data.raid_level = -2;
		single_vol_data.status = S_INITIALIZING;

		ret = Add_Volume_Record_Ex(&single_vol_data);
		if(ret<0) {
			return ret;
		}
	}

	for(i=0;i<raid_vol_data.list_cnt;i++){
		tmp_vol_no = Get_Volume_No_For_Drive(raid_vol_data.drive_no_list[i]);
		if(tmp_vol_no>0) {
			char device_name[HD_DEVICE_NAME_LENGTH];

			Get_HD_DevName(raid_vol_data.drive_no_list[i],device_name);
			if (Is_HD_Exist(device_name))
				Init_Single_Disk_Volume(tmp_vol_no);
			else
				Set_Volume_Status(tmp_vol_no, S_NOT_EXIST);
		}
		else {
			return ERROR_FAIL;
		}
		sleep(1);
	}
	//reinitialize single disk volume from spare_drive_list
	for(i=0;i<raid_vol_data.spare_list_cnt;i++){
		tmp_vol_no = Get_Volume_No_For_Drive(raid_vol_data.spare_drive_list[i]);
		if(tmp_vol_no>0) {
			char device_name[HD_DEVICE_NAME_LENGTH];

			Get_HD_DevName(raid_vol_data.spare_drive_list[i],device_name);
			if (Is_HD_Exist(device_name))
				Init_Single_Disk_Volume(tmp_vol_no);
			else
				Set_Volume_Status(tmp_vol_no, S_NOT_EXIST);
		}
		else {
			return ERROR_FAIL;
		}
		sleep(1);
	}

	return SUCCESS;
}

/**********************************************************************
** Description:	Add a RAID volumes w/ spare disk support if level=1 or 5
** Output:	the index of the new volume record or the error number
**		of adding the record
***********************************************************************/
int Add_Volume_Record_Ex(VOLUME_CONF_EX *vol_data)
{
	int vol_cnt=0,i=0,j=0;
	VOLUME_CONF_EX *vol_data_list=NULL;
	FILE *fp;
	time_t the_time;
#ifndef	STORAGE_FILE_LOCK
	int fd=-1;
#endif
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE], tmp_str[10];
	void *ptr=NULL;

	vol_cnt = Get_All_Volumes_Conf_Ex_2(&vol_data_list);
	
	if(vol_cnt >=MAX_VOLUME_NO) {
		free(vol_data_list);
		return ERROR_OUT_OF_RANGE;
	}
	if(vol_cnt >=0) {
		ptr = realloc((void*)vol_data_list,
			(vol_cnt+1)*sizeof(VOLUME_CONF_EX));
		if (ptr==NULL) {
			if (vol_data_list) free(vol_data_list);
			return ERROR_OUT_OF_MEMORY;
		}
		vol_data_list = (VOLUME_CONF_EX *)ptr;

		// Catherine 2002/04/22 ==>
		memcpy((void*)&vol_data_list[vol_cnt],
			(void*)vol_data,
			sizeof(VOLUME_CONF_EX));

		{
			int i,j,vol_no,flag;
			
			for (i=1; i<MAX_VOLUME_NO; i++) {
				flag=0;
				for (j=0; j<vol_cnt; j++) {
					if (i==vol_data_list[j].vol_no) {
						flag=1;
						break;
					}
				}
				if (flag==0) {
					vol_no=i;
					break;
				}
			}
			vol_data_list[vol_cnt].vol_no=vol_no;
			vol_data->vol_no=vol_no;
		}
		// <== Catherine 2002/04/22

		vol_cnt++;
		sort_volume_data_ex(vol_data_list,vol_cnt);

// Catherine 2002/03/28 ==>
#ifdef	STORAGE_FILE_LOCK
		if (NAS_File_Lock(STORAGE_CONF, 'w')==0) {
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}

		// open /etc/config/storage.conf for write
		if ((fp = fopen(STORAGE_CONF, "w+")) == NULL) {
			free(vol_data_list);
			NAS_File_Unlock(STORAGE_CONF);
			return ERROR_OPEN_FILE;
		}
#else
		if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
			fclose(fp);
		else {
			free(vol_data_list);
			return ERROR_FAIL;
		}
		// open /etc/config/storage.conf for write
		if ((fd=open(STORAGE_CONF, O_CREAT | O_RDWR))<0) {
			free(vol_data_list);
			return ERROR_OPEN_FILE;
		}
		// exclusively lock file
		if (flock(fd, LOCK_EX)<0) {
			close(fd);
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}
#endif
		// write the new content into the config file
		for(i=0;i<vol_cnt;i++){
			memset(buf,0,BUF_SIZE*2);
			// volume header
			sprintf(tmpbuf, "[VOLUME %d]\n", vol_data_list[i].vol_no); // Catherine 2002/04/22
			strcpy(buf, tmpbuf);
			// device name
			sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].device_name);
			strcat(buf, tmpbuf);
			// raid level
			sprintf(tmpbuf, "raid level = %d\n", vol_data_list[i].raid_level);
			strcat(buf, tmpbuf);
			// drive no
			strcpy(tmpbuf,"drive no = ");
			for(j=0;j<vol_data_list[i].list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].drive_no_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");

			strcat(buf, tmpbuf);
			// spare drive no
			strcpy(tmpbuf,"spare drive no = ");
			for(j=0;j<vol_data_list[i].spare_list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].spare_drive_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");

			strcat(buf, tmpbuf);
			// status
			sprintf(tmpbuf, "status = %d\n", vol_data_list[i].status);
			strcat(buf, tmpbuf);

			// record time
			(void) time(&the_time);
			sprintf(tmpbuf, "record_time = %s\n", ctime(&the_time));
			strcat(buf, tmpbuf);

#ifdef	STORAGE_FILE_LOCK
			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(vol_data_list);
				NAS_File_Unlock(STORAGE_CONF);
				return ERROR_WRITE_FILE;
			}
#else
			if (write(fd, buf, strlen(buf))<0) {
				flock(fd, LOCK_UN);
				close(fd);
				free(vol_data_list);
				return ERROR_WRITE_FILE;
			}
#endif
		}
		// unlock the file & close it
#ifdef	STORAGE_FILE_LOCK
		fclose(fp);
		NAS_File_Unlock(STORAGE_CONF);
#else
		flock(fd, LOCK_UN);
		close(fd);
#endif
// <== Catherine 2002/03/28
		free(vol_data_list);
	}
	return vol_cnt;
}

/**********************************************************************
** Get_All_Configured_Vol_For_Share_Ex_2
** Description:	Get all ready RAID volumes w/ spare disk support if level=1 or 5
** Output:	total volumes found
***********************************************************************/
int Get_All_Configured_Vol_For_Share_Ex_2(VOLUME_INFO_EX **mount_vol_ptr)
{
	VOLUME_INFO_EX *vol_info=NULL,*mount_vol=NULL;
	int vol_cnt,i,ret_cnt=0;
	void *ptr=NULL;

	vol_cnt = Get_All_Volumes_Info_Ex_2(&vol_info);

	if (vol_cnt<=0)
		return 0;

	for(i=0;i<vol_cnt;i++) {
		if(vol_info[i].vol_data.status == S_READY ||
			vol_info[i].vol_data.status == S_DEGRADED ||
			vol_info[i].vol_data.status == S_REBUILDING_RAID) {
			ret_cnt++;

			// Catherine 2002/04/09 ==>
			//mount_vol = (VOLUME_INFO_EX *) realloc(mount_vol,
			//		ret_cnt * sizeof(VOLUME_INFO_EX));
			ptr = realloc((void*)mount_vol,
				ret_cnt * sizeof(VOLUME_INFO_EX));
			if (ptr==NULL) {
#ifdef DEBUG
				printf("Fail to re-allocate buffer!\n");
#endif
				if (mount_vol) free(mount_vol);
				free(vol_info);
				return ERROR_OUT_OF_MEMORY;
			}
			mount_vol = (VOLUME_INFO_EX *)ptr;
			// <== Catherine 2002/04/09

			memcpy(&(mount_vol[ret_cnt-1]), &(vol_info[i]), sizeof(VOLUME_INFO_EX));

#ifdef DEBUG
			printf("After allocate mount_vol\n");
#endif	
		}
	}

	free(vol_info);
	*mount_vol_ptr = mount_vol;
	return ret_cnt;
}

/**********************************************************************
** Get_All_Volumes_Info_Ex_2
** Description:	Get RAID volumes status w/ spare disk support if level=1 or 5
** Output:	the count of all volumes
***********************************************************************/
int Get_All_Volumes_Info_Ex_2(VOLUME_INFO_EX **vol_info_list_ptr)
{
	VOLUME_CONF_EX *vol_data_list=NULL;
	PARTITION_INFO part_info;
	VOLUME_INFO_EX *vol_info_list=NULL;
	int vol_cnt,i,ret;

#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	ret = Get_All_Volumes_Conf_Ex_2(&vol_data_list);

	if(ret<=0) return ret;
	
	vol_cnt = ret;

	sort_volume_data_for_show_ex(vol_data_list, vol_cnt); // Catherine 2002/04/22

	vol_info_list = (VOLUME_INFO_EX*)calloc(vol_cnt, sizeof(VOLUME_INFO_EX));

	if (vol_info_list==NULL) { // Catherine 2002/04/09
		free(vol_data_list);
		return ERROR_OUT_OF_MEMORY;
	}

	for(i=0;i<vol_cnt;i++) {
		memcpy(&vol_info_list[i].vol_data, &vol_data_list[i], sizeof(VOLUME_CONF_EX));

		if(vol_info_list[i].vol_data.status > S_READY &&
			vol_info_list[i].vol_data.status < S_REBUILDING_RAID) {
			vol_info_list[i].total_size = -1;
			vol_info_list[i].free_size  = -1;
			vol_info_list[i].resync_per = -1;
			continue;
		}

		ret=Get_Partition_Info(vol_data_list[i].device_name,&part_info);
		if(ret != SUCCESS) {
			switch (ret) {
				case PARTITION_NOT_EXIST : 
					if(vol_info_list[i].vol_data.raid_level != -2)
						vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					else if(Is_Volume_HD_Exist(vol_info_list[i].vol_data.vol_no)) 
						vol_info_list[i].vol_data.status = S_UNINITIALIZE;
					else
						vol_info_list[i].vol_data.status = S_NOT_EXIST;
					break;
				case ERROR_NOT_MOUNTED :
					if(Verify_Hidden_Conf_File(vol_info_list[i].vol_data.drive_no_list[0])<0 )
						vol_info_list[i].vol_data.status = S_INVALID;
					else
					if(vol_info_list[i].vol_data.status != S_UNINITIALIZE)
						vol_info_list[i].vol_data.status = S_NOT_MOUNTED;
					break;
				default:
					vol_info_list[i].vol_data.status = S_UNKNOW_ERROR;
			}
			vol_info_list[i].total_size = -1;
			vol_info_list[i].free_size  = -1;
			vol_info_list[i].resync_per = -1;
		}
		else if(vol_info_list[i].vol_data.status <= S_READY || vol_info_list[i].vol_data.status == S_REBUILDING_RAID) 
		{
			vol_info_list[i].total_size = part_info.total_size;
			vol_info_list[i].free_size  = part_info.free_size;
			vol_info_list[i].resync_per = -1;

			if(vol_info_list[i].vol_data.raid_level == -2) { // Its single volume
				if(!Is_Volume_HD_Valid(vol_info_list[i].vol_data.vol_no)){
					if(Is_Volume_HD_Exist(vol_info_list[i].vol_data.vol_no)) {
						if(Verify_Hidden_Conf_File(vol_info_list[i].vol_data.drive_no_list[0])<= ERROR_HD_DEVICE_NAME )
							vol_info_list[i].vol_data.status = S_INVALID;
						else
							vol_info_list[i].vol_data.status = S_UNINITIALIZE;
					}
					else
						vol_info_list[i].vol_data.status = S_NOT_EXIST;
				}
				else
					vol_info_list[i].vol_data.status = S_READY;
			}
			else {// Its raid volume
				RAID_STATUS sts_rd;
				ret = Get_One_RaidDev_Status(vol_data_list[i].device_name,&sts_rd);
				if(ret == SUCCESS) {
					// Catherine 2002/05/20 ==>
					int intsize=sizeof(int);
					int m, n;

					if(sts_rd.active_drive_cnt<vol_info_list[i].vol_data.list_cnt) {
						// no disk is spare
						memset(vol_info_list[i].vol_data.drive_no_list,
							0, MAX_DRIVE_NO*intsize);
						memcpy(vol_info_list[i].vol_data.drive_no_list,
							sts_rd.active_drive_no,
							sts_rd.active_drive_cnt*intsize);
						sort_drive_no_for_show(vol_info_list[i].vol_data.drive_no_list,
							sts_rd.active_drive_cnt);
						memset(vol_info_list[i].vol_data.spare_drive_list,
							0, MAX_DRIVE_NO*intsize);
					}
					else {
						memset(vol_info_list[i].vol_data.drive_no_list,
							0, MAX_DRIVE_NO*intsize);
						memcpy(vol_info_list[i].vol_data.drive_no_list,
							sts_rd.active_drive_no,
							vol_info_list[i].vol_data.list_cnt*intsize);
						sort_drive_no_for_show(vol_info_list[i].vol_data.drive_no_list,
							vol_info_list[i].vol_data.list_cnt);
						memset(vol_info_list[i].vol_data.spare_drive_list,
							0, MAX_DRIVE_NO*intsize);
						if (sts_rd.active_drive_cnt>vol_info_list[i].vol_data.list_cnt) {
							for(m=vol_info_list[i].vol_data.list_cnt, n=0;
								m<sts_rd.active_drive_cnt; m++, n++)
							{
								vol_info_list[i].vol_data.spare_drive_list[n]=
									sts_rd.active_drive_no[m];
							}
						}
					}
					// <== Catherine 2002/05/20
					// Catherine 2002/06/11 for resync only ==>
					if(sts_rd.resync_per >= RAID_RESYNC_PER_BASE) {
						vol_info_list[i].resync_per = sts_rd.resync_per-RAID_RESYNC_PER_BASE;
						vol_info_list[i].vol_data.status = S_READY;
					} else // <== Catherine 2002/06/11 for resync only
					if(sts_rd.resync_per > -1){
						vol_info_list[i].resync_per = sts_rd.resync_per;
						vol_info_list[i].vol_data.status = S_REBUILDING_RAID;
					}
					else if(sts_rd.active_drive_cnt == vol_info_list[i].vol_data.list_cnt-1){
						if(vol_info_list[i].vol_data.raid_level == MIRROR || vol_info_list[i].vol_data.raid_level ==RAID5)
							vol_info_list[i].vol_data.status = S_DEGRADED;
						else if(vol_info_list[i].vol_data.raid_level == STRIPING)
							vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					}
					else if(sts_rd.active_drive_cnt < vol_info_list[i].vol_data.list_cnt-1)
						vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
					else
						vol_info_list[i].vol_data.status = S_READY;
				}
				else if (ret == ERROR_NOT_FOUND)
					vol_info_list[i].vol_data.status = S_NOT_ACTIVE;
				else
					vol_info_list[i].vol_data.status = S_UNKNOW_ERROR;
			}	
		}

		//recheck hd while status is ready
		Set_Volume_Status(vol_info_list[i].vol_data.vol_no,vol_info_list[i].vol_data.status);	// Catherine 2002/04/22
	}
	free(vol_data_list);
	*vol_info_list_ptr = vol_info_list;
	return vol_cnt;
}


/**********************************************************************
** Get_All_Volumes_Conf_Ex_2
** Description:	Get RAID volume configurations w/ spare disk support if level=1 or 5
** Output:	the count of all volumes
***********************************************************************/
int Get_All_Volumes_Conf_Ex_2(VOLUME_CONF_EX **vol_data_list_ptr)
{
	int i,cnt=0,ret;
	VOLUME_CONF_EX *vol_data_list=NULL;
	VOLUME_CONF_EX vol_data;
	void *ptr=NULL;

	for(i=1;i<MAX_VOLUME_NO+1;i++) {
		ret = Get_One_Volume_Conf_Ex(i,&vol_data);
		if (ret < 0)
			continue;
		else {
			ptr = realloc((void*)vol_data_list,
					(cnt+1)*sizeof(VOLUME_CONF_EX));
			if (ptr==NULL) {
				if (vol_data_list) free(vol_data_list);
				return ERROR_OUT_OF_MEMORY;
			}
			vol_data_list = (VOLUME_CONF_EX *)ptr;

			// Catherine 2002/04/22
			memcpy((void *) &vol_data_list[cnt],
				(void*) &vol_data,
				sizeof(VOLUME_CONF_EX));
			
			cnt++;
		}
	}
	*vol_data_list_ptr = vol_data_list;
	return cnt;
}

/**********************************************************************
** Get_One_Volume_Conf_Ex
** Description:	Get one RAID volume configurations w/ spare disk support if level=1 or 5
** Output:	Error Code --
**		SUCCESS
***********************************************************************/
int Get_One_Volume_Conf_Ex(int vol_no,VOLUME_CONF_EX *vol_data)
{
	char vol_section[20];
	char tmp_buf[BUF_SIZE],drv_no_str[4];
	int drive_no_cnt=1,ret;
	
	sprintf(vol_section,"VOLUME %d",vol_no);

	//set vol no
	vol_data->vol_no = vol_no;
	// get device name
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"device name",vol_data->device_name,HD_DEVICE_NAME_LENGTH);
	if(ret<0) return ret;
		
	// get raid level
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"raid level",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	vol_data->raid_level = atoi(tmp_buf);		
	//get drive no list & list count
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"drive no",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	while(Get_String_Field(tmp_buf,drive_no_cnt,',',drv_no_str,4)>=0) {
		if ((strlen(drv_no_str)==0) || (atoi(drv_no_str)<=0))
			break;

		vol_data->drive_no_list[drive_no_cnt-1] = atoi(drv_no_str);
		drive_no_cnt++;
	}
	vol_data->list_cnt = drive_no_cnt-1;

	//get spare drive no list & list count
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"spare drive no",tmp_buf,BUF_SIZE);
	if(ret==0) {
		drive_no_cnt=1;
		while(Get_String_Field(tmp_buf,drive_no_cnt,',',drv_no_str,4)>=0) {
			if ((strlen(drv_no_str)==0) || (atoi(drv_no_str)<=0))
				break;

			vol_data->spare_drive_list[drive_no_cnt-1] = atoi(drv_no_str);
			drive_no_cnt++;
		}
		vol_data->spare_list_cnt = drive_no_cnt-1;
	}
	else vol_data->spare_list_cnt = 0;

	//get volume status
	ret = Conf_Get_Field(STORAGE_CONF,vol_section,"status",tmp_buf,BUF_SIZE);
	if(ret<0) return ret;
	vol_data->status = atoi(tmp_buf);
	
	return SUCCESS;
}

/**********************************************************************
** Get_Raid_Disk_Volumes_Info_Ex_2
** Description:	Get one RAID volume configurations w/ spare disk support if level=1 or 5
** Output:	RAID volume count
***********************************************************************/
int Get_Raid_Disk_Volumes_Info_Ex_2(VOLUME_INFO_EX **vol_info_list_ptr)
{
	
	int i,cnt=0, vol_cnt;
	VOLUME_INFO_EX *vol_info_list=NULL;
	VOLUME_INFO_EX *all_vol_list=NULL;
	void *ptr=NULL;

	vol_cnt = Get_All_Volumes_Info_Ex_2(&all_vol_list);

	for(i=0; i<vol_cnt; i++) {
		if(all_vol_list[i].vol_data.raid_level > -2) {
			// Catherine 2002/04/09 ==>
			ptr = realloc((void*)vol_info_list,
					(cnt+1)*sizeof(VOLUME_INFO_EX));
			if (ptr==NULL) {
				if (vol_info_list) free(vol_info_list);
				Release_List(all_vol_list);
				return ERROR_OUT_OF_MEMORY;
			}
			vol_info_list = (VOLUME_INFO_EX *)ptr;

			//vol_info_list[cnt] = all_vol_list[i];
			memcpy((void*)&(vol_info_list[cnt]),
				(void*)&(all_vol_list[i]),
				sizeof(VOLUME_INFO_EX));
			// <== Catherine 2002/04/09
			/* Catherine 2002/04/23 remove dummy code
			if(all_vol_list[i].vol_data.status == S_READY) {
				if(!Is_Volume_HD_Valid(all_vol_list[i].vol_data.vol_no)){ // Catherine 2002/03/26
					if(Is_Volume_HD_Exist(all_vol_list[i].vol_data.vol_no))// Catherine 2002/03/26
						all_vol_list[i].vol_data.status = S_INVALID;
					else
						all_vol_list[i].vol_data.status = S_NOT_EXIST;

					// Catherine 2002/04/22
					Set_Volume_Status(all_vol_list[i].vol_data.vol_no,all_vol_list[i].vol_data.status);
				}
			}
			*/
			cnt++;
		}
	}
	if (vol_cnt>0) Release_List(all_vol_list);

	*vol_info_list_ptr = vol_info_list;
	return cnt;
}

/**********************************************************************
** Get_One_Volume_Info_Ex
** Description:	Get one RAID volume status w/ spare disk support if level=1 or 5
** Output:	Error Code --
**		SUCCESS
***********************************************************************/
int Get_One_Volume_Info_Ex(int vol_no,VOLUME_INFO_EX *vol_info)
{
	VOLUME_CONF_EX vol_data;
	PARTITION_INFO part_info;
	int ret;

#ifdef	STORAGE_FILE_LOCK
	repair_storage_conf_if_necessary();
#else
	FILE *fp;

	if((fp = fopen(STORAGE_CONF,"r")) == NULL) 
		Create_Storage_Conf();
	else {
		char tmp_line[80];
		strcpy(tmp_line,"");
		fgets(tmp_line,80,fp);
		fclose(fp);
		if(strlen(tmp_line)== 0) Restore_Storage_Conf(); 	
	}
#endif
	ret = Get_One_Volume_Conf_Ex(vol_no,&vol_data);

	if(ret<0) return ret;

	memcpy((void*)&(vol_info->vol_data), (void*)&vol_data,
		sizeof(VOLUME_CONF_EX));

	if(vol_info->vol_data.status > S_READY && vol_info->vol_data.status < S_REBUILDING_RAID) {
		vol_info->total_size = -1;
		vol_info->free_size  = -1;
		vol_info->resync_per = -1;
		return SUCCESS;	
	}

	ret=Get_Partition_Info(vol_data.device_name,&part_info);
	if(ret != SUCCESS) {
		switch (ret) {
			case PARTITION_NOT_EXIST : 
				if(vol_info->vol_data.raid_level != -2)
					vol_info->vol_data.status = S_NOT_ACTIVE;
				else
					vol_info->vol_data.status = S_UNINITIALIZE;
				break;
			case ERROR_NOT_MOUNTED :
				if (Verify_Hidden_Conf_File(vol_info->vol_data.drive_no_list[0])<0 )
				        vol_info->vol_data.status = S_INVALID;
				else
				if(vol_info->vol_data.status != S_UNINITIALIZE)
					vol_info->vol_data.status = S_NOT_MOUNTED;
				break;
			default:
				vol_info->vol_data.status = S_UNKNOW_ERROR;
		}
		vol_info->total_size = -1;
		vol_info->free_size  = -1;
		vol_info->resync_per = -1;
	}
	else if(vol_info->vol_data.status <= S_READY || vol_info->vol_data.status == S_REBUILDING_RAID) 
	{
		vol_info->total_size = part_info.total_size;
		vol_info->free_size  = part_info.free_size;
		vol_info->resync_per = -1;

		if(vol_info->vol_data.raid_level == -2) {// Its single volume
			if(!Is_Volume_HD_Valid(vol_no)){
				if(Is_Volume_HD_Exist(vol_no))
					vol_info->vol_data.status = S_INVALID;
				else
					vol_info->vol_data.status = S_NOT_EXIST;	
			}
			else
				vol_info->vol_data.status = S_READY;
		}
		else {// Its raid volume
			RAID_STATUS sts_rd;
			ret = Get_One_RaidDev_Status(vol_data.device_name,&sts_rd);
			if(ret == SUCCESS) {
				// Catherine 2002/05/20 ==>
				int intsize=sizeof(int);
				int i, j;

				if(sts_rd.active_drive_cnt<vol_info->vol_data.list_cnt) {
					// no disk is spare
					memset(vol_info->vol_data.drive_no_list,
						0, MAX_DRIVE_NO*intsize);
					memcpy(vol_info->vol_data.drive_no_list,
						sts_rd.active_drive_no,
						sts_rd.active_drive_cnt*intsize);
					sort_drive_no_for_show(vol_info->vol_data.drive_no_list,
						sts_rd.active_drive_cnt);
					memset(vol_info->vol_data.spare_drive_list,
						0, MAX_DRIVE_NO*intsize);
				}
				else {
					memset(vol_info->vol_data.drive_no_list,
						0, MAX_DRIVE_NO*intsize);
					memcpy(vol_info->vol_data.drive_no_list,
						sts_rd.active_drive_no,
						vol_info->vol_data.list_cnt*intsize);
					sort_drive_no_for_show(vol_info->vol_data.drive_no_list,
						vol_info->vol_data.list_cnt);
					memset(vol_info->vol_data.spare_drive_list,
						0, MAX_DRIVE_NO*intsize);
					if (sts_rd.active_drive_cnt>vol_info->vol_data.list_cnt) {
						for(i=vol_info->vol_data.list_cnt, j=0;
							i<sts_rd.active_drive_cnt; i++, j++)
						{
							vol_info->vol_data.spare_drive_list[j]=sts_rd.active_drive_no[i];
						}
					}
				}
				// <== Catherine 2002/05/20
				// Catherine 2002/06/11 for resync only ==>
				if(sts_rd.resync_per >= RAID_RESYNC_PER_BASE) {
					vol_info->resync_per = sts_rd.resync_per-RAID_RESYNC_PER_BASE;
					vol_info->vol_data.status = S_READY;
				} else // <== Catherine 2002/06/11 for resync only
				if(sts_rd.resync_per > -1){
					vol_info->resync_per = sts_rd.resync_per;
					vol_info->vol_data.status = S_REBUILDING_RAID;
				}
				else if(sts_rd.active_drive_cnt == vol_info->vol_data.list_cnt-1){
					if(vol_info->vol_data.raid_level == MIRROR || vol_info->vol_data.raid_level == RAID5)
						vol_info->vol_data.status = S_DEGRADED;
					else if(vol_info->vol_data.raid_level == STRIPING)
						vol_info->vol_data.status = S_NOT_ACTIVE;
				}
				else if(sts_rd.active_drive_cnt < vol_info->vol_data.list_cnt-1)
					vol_info->vol_data.status=S_NOT_ACTIVE;
				else
					vol_info->vol_data.status = S_READY;
			}
			else 
			if(ret == ERROR_NOT_FOUND)
				vol_info->vol_data.status = S_NOT_ACTIVE;
			else
				vol_info->vol_data.status = S_UNKNOW_ERROR;
		}	
	}

	Set_Volume_Status(vol_no,vol_info->vol_data.status);
	return SUCCESS;
}

/**********************************************************************
** Is_HD_Init
** Description:	tell if all the volumes(HDs) are uninitialized
** Output:	TRUE -- some of the HDs are initialized
**		FALSE -- all of the HDs are uninitialized
***********************************************************************/
BOOL Is_HD_Init()
{
	VOLUME_INFO_EX *vol_info_list=NULL;
	int i, ret;
	BOOL result=FALSE;

	ret = Get_All_Volumes_Info_Ex_2(&vol_info_list);
	if (ret<0) {
#ifdef DEBUG
		printf("Is_HD_Init : Get_All_Volumes_Info_Ex_2() : %d\n", ret);
#endif
		return FALSE;
	}

	for (i=0; i<ret; i++) {
#ifdef DEBUG
		printf("Is_HD_Init : volume %d status=%d\n",
			i+1, vol_info_list[i].vol_data.status);
#endif

		if ((vol_info_list[i].vol_data.status>=S_READY &&
			vol_info_list[i].vol_data.status<=S_REBUILDING_RAID) ||
			vol_info_list[i].vol_data.status==S_DEGRADED)
		{
			result = TRUE;
			break;
		}
	}
	if (vol_info_list) free(vol_info_list);
	return result;
}

// ReiserFS + LVM support
int Get_Volume_FileSystem_Type(int vol_no) 
{
	int ret;
	LVM_VOLUME_CONF volconf;

	ret = Get_One_LVM_Volume_Conf(vol_no, &volconf);
	if (ret<0) return ret;

	if (Is_LVM_Volume(vol_no))
		ret = Get_Device_FileSystem_Type(volconf.lv_name);
	else
		ret = Get_Device_FileSystem_Type(volconf.device_name);

	return ret;
}

/**********************************************************************
** Quick_Config_All_Disks
** Description:	re-config all volumes to the desired raid_level,
**		raid_level==-2 : all non-single volumes would be reinit;
**		raid_level==-1 : all disks would form a linear RAID;
**		raid_level== 0 : all disks would form a stripping volume;
**		raid_level== 1 : every 2 disks would form a RAID-1 volume;
**		raid_level== 5 : all disks would form a RAID-5 volume.
** Output:	error code --
**		SUCCESS
**		
***********************************************************************/
int Quick_Config_All_Disks(int raid_level)
{
	int ret=0, disk_no, vol_cnt, i, j;
	int flag, new_vol_cnt, list_disk_cnt;
	int real_drive_no, min_drive_no;
	int k, prev_vol_last_drive_no;
	char drvname[HD_DEVICE_NAME_LENGTH];
	VOLUME_INFO_EX *vol_list=NULL;
	VOLUME_CONF_EX *vol_data;
	VOLUME_CONF_EX new_vol_data;

#ifdef DEBUG
	printf("Quick_Config_All_Disks(%d) starts!\n", raid_level);
#endif
	// check if the operation is allowed?
	vol_cnt = Get_All_Volumes_Info_Ex_2(&vol_list);
	if (vol_cnt<0) return vol_cnt;

	disk_no = Get_Profile_Integer(
			NASCONF_STORAGE_SECTION,
			NASCONF_DRIVE_NO_FIELD,
			0);

#ifdef DEBUG
	printf("Quick_Config_All_Disks step 1:"
		" old vol cnt=%d, disk no=%d!\n",
		vol_cnt, disk_no);
#endif

	switch (raid_level) {
		case -2:
			min_drive_no = 1;

			flag = 0;
			for (i=0; i<vol_cnt; i++) {
				vol_data = &(vol_list[i].vol_data);
				if ((vol_data->raid_level>-2) ||
					(vol_data->status<S_READY))
				{
					flag = 1;
					break;
				}
			}

			if (flag==0) { // all volumes are single
#ifdef DEBUG
				printf("\tAll volumes are ready, stop!\n");
#endif
				ret = SUCCESS;
				goto quick_finish;			
			}
			break;
		case 1:
			min_drive_no = 2;

			for(i=1, real_drive_no=0; i<=disk_no; i++)
			{
				Get_HD_DevName(i, drvname);
				if (Is_HD_Exist(drvname))
					real_drive_no++;
			}

			if (real_drive_no<min_drive_no) {
				ret = ERROR_FAIL;
				goto quick_finish;
			}

			flag = 0;

			for (i=0; i<vol_cnt; i++) {
				vol_data = &(vol_list[i].vol_data);
				if ((vol_data->raid_level!=1) ||
					((vol_data->raid_level==1) &&
					(vol_data->status<S_READY)))
				{
					flag = 1;
					break;
				}
			}

			if (flag==0) { // all volumes are RAID-1
#ifdef DEBUG
				printf("\tAll volumes are ready, stop!\n");
#endif
				ret = SUCCESS;
				goto quick_finish;
			}
			break;
		case -1:
		case 0:
		case 5:
			vol_data = &(vol_list[0].vol_data);
#ifdef DEBUG
			printf("\tQuick_Config_All_Disks step 1.1\n");
#endif
			if ((vol_cnt==1) && (vol_data->raid_level==raid_level) &&
				(vol_data->status>=S_READY) &&
				(vol_data->status!=S_REMOVING_RAID))
			{
#ifdef DEBUG
				printf("\tAll volumes are ready, stop!\n");
#endif
				ret = SUCCESS;
				goto quick_finish;
			}
#ifdef DEBUG
			printf("\tQuick_Config_All_Disks step 1.2\n");
#endif
			for(i=1, real_drive_no=0; i<=disk_no; i++)
			{
				Get_HD_DevName(i, drvname);
				if (Is_HD_Exist(drvname))
					real_drive_no++;
			}
			if (raid_level==5)
				min_drive_no=3;
			else
				min_drive_no=2;
#ifdef DEBUG
			printf("\tQuick_Config_All_Disks step 1.3\n");
#endif
			if (min_drive_no>real_drive_no) {
#ifdef DEBUG
				printf("\tOnly %d disks exist, "
					"not enough disks to make RAID-%d\n",
					real_drive_no, raid_level);
#endif
				ret = ERROR_FAIL;
				goto quick_finish;
			}
			break;
		default: 
			;
	}

#ifdef DEBUG
	printf("Quick_Config_All_Disks step 2:"
		" before remove old RAID devices!\n");
#endif
	// remove all old RAID configs
	for(i=0; i<vol_cnt; i++) {
		vol_data = &(vol_list[i].vol_data);

#ifdef DEBUG
		printf("\t2.1 Volume # = %d, status=%d\n",
			vol_data->vol_no,
			vol_data->status);
#endif

		Set_Volume_Status(vol_data->vol_no, S_INITIALIZING);
		Stop_User_Quota(vol_data->vol_no);
		Umount_Volume(vol_data->vol_no);

#ifdef DEBUG
		printf("\t2.2 before Stop_Raid(%s)\n",
			vol_data->device_name);
#endif
		if ((vol_data->raid_level>-2) &&
			(vol_data->status!=S_NOT_ACTIVE) &&
			(vol_data->status!=S_INITIALIZING))
		{
			Set_Volume_Status(vol_data->vol_no,
				S_REMOVING_RAID);
			ret = Stop_Raid(vol_data->device_name);
			if (ret<0 && ret != ERROR_NO_RAIDDEV) {
#ifdef DEBUG
				printf("\tFails to Stop_Raid(%s)\n",
					vol_data->device_name);
#endif
				Set_Volume_Status(vol_data->vol_no,
					S_READY);
				Mount_Volume(vol_data->vol_no);
				goto quick_finish;
			}
		}

#ifdef DEBUG
		printf("\t2.3 before Delete_RaidDev(%s)\n",
			vol_data->device_name);
#endif
		if (vol_data->raid_level>-2) {
			VOLUME_CONF_EX thisvol;

			Get_One_Volume_Conf_Ex(vol_data->vol_no,
				&thisvol);

			Set_Volume_Status(thisvol.vol_no,
				S_REMOVING_RAID);

			if((ret=Delete_RaidDev(thisvol.device_name))<0 &&
				ret != RAID_DEVICE_NO)
			{
#ifdef DEBUG
				printf("\tFails to Delete_RaidDev(%s)\n",
					thisvol.device_name);
#endif
				Set_Volume_Status(vol_data->vol_no,
					S_RAID_DIED);
				goto quick_finish;
			}
#ifdef DEBUG
			printf("\t2.3.1 before Remove_Volume_Record(%d)\n",
				thisvol.vol_no);
#endif
			
			clear_volume_related_messages_ex(&thisvol);

			Remove_Volume_Record(thisvol.vol_no);

#ifdef DEBUG
			printf("\t2.3.2 before Add_Volume_Record_Ex for each drive\n");
#endif
			for(j=0; j<thisvol.list_cnt; j++){
				memset(&new_vol_data, 0, sizeof(VOLUME_CONF_EX));

				Get_Partition_Name(thisvol.drive_no_list[j],
					DATA_PART,
					new_vol_data.device_name);
				new_vol_data.list_cnt = 1;
				new_vol_data.drive_no_list[0] = thisvol.drive_no_list[j];
				new_vol_data.raid_level = -2;
				Get_HD_DevName(new_vol_data.drive_no_list[0], drvname);
#ifdef DEBUG
				printf("\t\t2.3.2.1 Is_HD_Exist(%s)?\n",
					drvname);
#endif
				if (Is_HD_Exist(drvname))
					new_vol_data.status = S_INITIALIZING;
				else
					new_vol_data.status = S_NOT_EXIST;
#ifdef DEBUG
				printf("\t\t2.3.2.2 Add_Volume_Record_Ex for drive %d?\n",
					new_vol_data.drive_no_list[0]);
#endif
				ret = Add_Volume_Record_Ex(&new_vol_data);
				if(ret<0) {
#ifdef DEBUG
					printf("\tFails to Add_Volume_Record_Ex"
						" for drive %d\n",
						thisvol.drive_no_list[j]);
#endif
					goto quick_finish;
				}
			}
#ifdef DEBUG
			printf("\t2.3.3 before Add_Volume_Record_Ex for each spare drive\n");
#endif
			for(j=0; j<thisvol.spare_list_cnt; j++){
				memset(&new_vol_data, 0, sizeof(VOLUME_CONF_EX));

				Get_Partition_Name(thisvol.spare_drive_list[j],
					DATA_PART,
					new_vol_data.device_name);
				new_vol_data.list_cnt = 1;
				new_vol_data.drive_no_list[0] = thisvol.spare_drive_list[j];
				new_vol_data.raid_level = -2;
				Get_HD_DevName(new_vol_data.drive_no_list[0], drvname);
#ifdef DEBUG
				printf("\t\t2.3.3.1 Is_HD_Exist(%s)?\n",
					drvname);
#endif
				if (Is_HD_Exist(drvname))
					new_vol_data.status = S_INITIALIZING;
				else
					new_vol_data.status = S_NOT_EXIST;
#ifdef DEBUG
				printf("\t\t2.3.3.2 Add_Volume_Record_Ex for drive %d?\n",
					new_vol_data.drive_no_list[0]);
#endif
				ret = Add_Volume_Record_Ex(&new_vol_data);
				if(ret<0) {
#ifdef DEBUG
					printf("\tFails to Add_Volume_Record_Ex"
						" for drive %d\n",
						thisvol.spare_drive_list[j]);
#endif
					goto quick_finish;
				}
			}
		}
	}

	Release_List(vol_list);
	vol_list = NULL;

#ifdef DEBUG
	printf("Quick_Config_All_Disks step 3:"
		" before init new volumes!\n");
#endif
	// create the new volume(s)
	vol_cnt = Get_All_Volumes_Info_Ex_2(&vol_list);
	if (vol_cnt<0) return vol_cnt;

	if (vol_cnt!=disk_no) {
#ifdef DEBUG
		printf("\t(new vol cnt=%d) != (disk no=%d) :"
			" Restore_Storage_Conf()\n",
			vol_cnt, disk_no);
#endif
		if (vol_cnt && vol_list) Release_List(vol_list);
		Restore_Storage_Conf();
		vol_cnt = Get_All_Volumes_Info_Ex_2(&vol_list);
	}

	for (i=0; i<vol_cnt; i++) {
		vol_data = &(vol_list[i].vol_data);
		Get_HD_DevName(vol_data->drive_no_list[0], drvname);
		if (!Is_HD_Exist(drvname)) {
			vol_data->status = S_NOT_EXIST;
			Set_Volume_Status(vol_data->vol_no,
				S_NOT_EXIST);
			continue;
		}

		ret=Verify_Hidden_Conf_File(vol_data->drive_no_list[0]);

		if (ret<0) {
#ifdef DEBUG
			printf("\tDrive %d is uninit!\n",
				vol_data->drive_no_list[0]);
#endif
			Set_Volume_Status(vol_data->vol_no,S_INITIALIZING);

			if ((ret=Init_NAS_Disk(vol_data->drive_no_list[0]))<0){
#ifdef DEBUG
				printf("\tInit_NAS_Disk(%d) fails : %d\n",
					vol_data->drive_no_list[0], ret);
#endif
				Set_Volume_Status(vol_data->vol_no,S_UNINITIALIZE);
				goto quick_finish;
			}
		}
	}
#ifdef DEBUG
	printf("Quick_Config_All_Disks step 4:"
		" before create new volumes!\n");
#endif
	switch (raid_level) {
		case -2:
			for (i=0; i<vol_cnt; i++) {
				vol_data = &(vol_list[i].vol_data);
#ifdef DEBUG
				printf("\tCreate volume(%d, %s) ....\n",
					vol_data->vol_no, vol_data->device_name);
#endif
				if (vol_data->status==S_READY || vol_data->status==S_NOT_EXIST)
					continue;

				Set_Volume_Status(vol_data->vol_no,S_FORMATTING);
				if((ret=Format_Volume(vol_data->vol_no))<0){
#ifdef DEBUG
					printf("\tFormat_Volume(%d) fails:%d\n",
						vol_data->vol_no, ret);
#endif
					Set_Volume_Status(vol_data->vol_no,S_NOT_MOUNTED);
					goto quick_finish;
				}

				if((ret=Mount_Volume(vol_data->vol_no))<0){
#ifdef DEBUG
					printf("\tMount_Volume(%d) fails:%d\n",
						vol_data->vol_no, ret);
#endif
					Set_Volume_Status(vol_data->vol_no,S_NOT_MOUNTED);
					goto quick_finish;
				}
				start_vol_quota(vol_data->vol_no, 1);
	
				Set_Volume_Status(vol_data->vol_no,S_READY);
			}
			break;
		case 1:
		case -1:
		case 0:
		case 5:
			if (raid_level==1) {
				new_vol_cnt = disk_no/2;
				list_disk_cnt = 2;
			}
			else {
				new_vol_cnt=1;
				list_disk_cnt = real_drive_no;
			}

			for (i=0, prev_vol_last_drive_no=0; i<new_vol_cnt; i++)
			{
#ifdef DEBUG
				printf("\tCreate Volume %d .....\n"
					"\t\tdisk_no=%d, prev_vol_last_drive=%d"
					", min_drive_no=%d, new_vol_cnt=%d\n",
					i,disk_no,prev_vol_last_drive_no,
					min_drive_no, new_vol_cnt);
#endif
				if (disk_no-prev_vol_last_drive_no<min_drive_no) {
#ifdef DEBUG
					printf("\tRemain disk no = %d, "
						"not enough for new volume\n",
						disk_no-prev_vol_last_drive_no);
#endif
					for (j=prev_vol_last_drive_no+1;j<=disk_no;j++) {
						int vol_no = Get_Volume_No_For_Drive(j);
#ifdef DEBUG
						printf("\tSet the single volume(%d),disk=%d to unit!\n",
							vol_no, j);
#endif
						if (vol_no<0) continue;

						if (S_NOT_EXIST!=Get_Volume_Status(vol_no))
							Set_Volume_Status(vol_no, S_UNINITIALIZE);
					}
					ret = ERROR_FAIL;
					break;
				}

				for (j=prev_vol_last_drive_no+1, k=0, real_drive_no=0;
					k<list_disk_cnt && j<=disk_no; j++, k++)
				{
					int vol_no;
					VOLUME_CONF_EX tmpvol;

					Get_HD_DevName(j, drvname);
					if (!Is_HD_Exist(drvname)) {
#ifdef DEBUG
						printf("\tDrive %d not exist!\n", j);
#endif
						k--;
						continue;
					}
					real_drive_no++;
					new_vol_data.drive_no_list[k]=j;

					vol_no = Get_Volume_No_For_Drive(j);
#ifdef DEBUG
					printf("\tComponent disk's old volume no=%d\n",
						vol_no);
#endif
					Get_One_Volume_Conf_Ex(vol_no, &tmpvol);
					clear_volume_related_messages_ex(&tmpvol);

					Remove_Volume_Record(vol_no);
				}

				if (real_drive_no<min_drive_no) {
#ifdef DEBUG
					printf("\tRemain disk no = %d, "
						"not enough for new volume\n",
						real_drive_no);
#endif
					for (k=prev_vol_last_drive_no+1;k<=disk_no;k++) {
						int vol_no = Get_Volume_No_For_Drive(k);
#ifdef DEBUG
						printf("\tSet the single volume(%d),disk=%d to unit!\n",
							vol_no, k);
#endif
						if (vol_no<0) { // the record has been removed
#ifdef DEBUG
							printf("\tRe-add a single volume for"
								" disk=%d to unit!\n", k);
#endif
							memset(&new_vol_data, 0, sizeof(VOLUME_CONF_EX));

							Get_Partition_Name(k,
								DATA_PART,
								new_vol_data.device_name);
							new_vol_data.list_cnt = 1;
							new_vol_data.drive_no_list[0] = k;
							new_vol_data.raid_level = -2;
							Get_HD_DevName(k, drvname);
							if (Is_HD_Exist(drvname))
								new_vol_data.status = S_UNINITIALIZE;
							else
								new_vol_data.status = S_NOT_EXIST;

							ret = Add_Volume_Record_Ex(&new_vol_data);
							if(ret<0) {
#ifdef DEBUG
								printf("\tFails to Add_Volume_Record_Ex"
								" for drive %d\n", k);
#endif
							}
							continue;
						}

						if (S_NOT_EXIST!=Get_Volume_Status(vol_no))
							Set_Volume_Status(vol_no, S_UNINITIALIZE);
					}

					ret = ERROR_FAIL;
					goto quick_finish;
				}
				prev_vol_last_drive_no = j-1;

				Get_First_Available_RaidName(new_vol_data.device_name,
					HD_DEVICE_NAME_LENGTH);

				new_vol_data.spare_list_cnt = 0;
				new_vol_data.raid_level = raid_level;
				new_vol_data.list_cnt = real_drive_no;

				ret = Create_NAS_RaidDev_Ex(
					new_vol_data.device_name,
					new_vol_data.drive_no_list,
					new_vol_data.list_cnt,
					new_vol_data.spare_drive_list,
					new_vol_data.spare_list_cnt,
					new_vol_data.raid_level);

				if(ret < 0) {
#ifdef DEBUG
					printf("\tCreate_NAS_RaidDev_Ex(%s,%d) fails:%d\n",
						new_vol_data.device_name,
						new_vol_data.raid_level, ret);
#endif
					Restore_Storage_Conf();
					goto quick_finish;
				}

				new_vol_data.status = S_INITIALIZING;
				ret = Add_Volume_Record_Ex(&new_vol_data);
				if (ret<0) {
#ifdef DEBUG
					printf("\tAdd_Volume_Record_Ex(%s,%d) fails:%d\n",
						new_vol_data.device_name,
						new_vol_data.raid_level, ret);
#endif
					goto quick_finish;
				}
#ifdef DEBUG
				printf("\tCreate volume(%d, %s) ....\n",
					new_vol_data.vol_no, new_vol_data.device_name);
#endif
				Set_Volume_Status(new_vol_data.vol_no,S_CREATING_RAID);

				Make_Raid(new_vol_data.device_name);
				Set_Volume_Status(new_vol_data.vol_no,S_FORMATTING);
				Format_Volume(new_vol_data.vol_no);
				if((ret=Mount_Volume(new_vol_data.vol_no))< 0 ) {
#ifdef DEBUG
					printf("\tMount_Volume(%d) fails:%d\n",
						new_vol_data.vol_no, ret);
#endif
					Set_Volume_Status(new_vol_data.vol_no,
						S_NOT_MOUNTED);
					goto quick_finish;
				}

				start_vol_quota(new_vol_data.vol_no,1);

				Set_Volume_Status(new_vol_data.vol_no,S_READY);
			}
			break;
		default: 
			;
	}
quick_finish:
	if (vol_cnt>0 && vol_list)
		Release_List(vol_list);
#ifdef DEBUG
	printf("Quick_Config_All_Disks step 5: OVER\n");
#endif
	return ret;
}

#ifdef	STORAGE_FILE_LOCK
int Add_Volume_Record_Ex_2(VOLUME_CONF_EX *vol_data, BOOL file_lock)
{
	int vol_cnt=0,i=0,j=0;
	VOLUME_CONF_EX *vol_data_list=NULL;
	FILE *fp;
	time_t the_time;
	char buf[2*BUF_SIZE], tmpbuf[BUF_SIZE], tmp_str[10];
	void *ptr=NULL;

	vol_cnt = Get_All_Volumes_Conf_Ex_2(&vol_data_list);

	if(vol_cnt >=MAX_VOLUME_NO) {
		free(vol_data_list);
		return ERROR_OUT_OF_RANGE;
	}

	if(vol_cnt >=0) {
		ptr = realloc((void*)vol_data_list,
			(vol_cnt+1)*sizeof(VOLUME_CONF_EX));
		if (ptr==NULL) {
			if (vol_data_list) free(vol_data_list);
			return ERROR_OUT_OF_MEMORY;
		}
		vol_data_list = (VOLUME_CONF_EX *)ptr;

		// Catherine 2002/04/22 ==>
		memcpy((void*)&vol_data_list[vol_cnt],
			(void*)vol_data,
			sizeof(VOLUME_CONF_EX));

		{
			int i,j,vol_no,flag;

			for (i=1; i<MAX_VOLUME_NO; i++) {
				flag=0;
				for (j=0; j<vol_cnt; j++) {
					if (i==vol_data_list[j].vol_no) {
						flag=1;
						break;
					}
				}
				if (flag==0) {
					vol_no=i;
					break;
				}
			}
			vol_data_list[vol_cnt].vol_no=vol_no;
			vol_data->vol_no = vol_no;
		}

		vol_cnt++;
		sort_volume_data_ex(vol_data_list,vol_cnt);

		if (file_lock && (NAS_File_Lock(STORAGE_CONF, 'w')==0)) {
			free(vol_data_list);
			return ERROR_LOCK_FILE;
		}

		// open /etc/config/storage.conf for write
		if ((fp = fopen(STORAGE_CONF, "w+")) == NULL) {
			free(vol_data_list);
			if (file_lock) NAS_File_Unlock(STORAGE_CONF);
			return ERROR_OPEN_FILE;
		}

		// write the new content into the config file
		for(i=0;i<vol_cnt;i++){
			memset(buf,0,BUF_SIZE*2);
			// volume header
			sprintf(tmpbuf, "[VOLUME %d]\n", vol_data_list[i].vol_no); // Catherine 2002/04/22
			strcpy(buf, tmpbuf);
			// device name
			sprintf(tmpbuf, "device name = %s\n", vol_data_list[i].device_name);
			strcat(buf, tmpbuf);
			// raid level
			sprintf(tmpbuf, "raid level = %d\n", vol_data_list[i].raid_level);
			strcat(buf, tmpbuf);
			// drive no
			strcpy(tmpbuf,"drive no = ");
			for(j=0;j<vol_data_list[i].list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].drive_no_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");

			strcat(buf, tmpbuf);
			// spare drive no
			strcpy(tmpbuf,"spare drive no = ");
			for(j=0;j<vol_data_list[i].spare_list_cnt;j++) {
				sprintf(tmp_str,"%d,",vol_data_list[i].spare_drive_list[j]);
				strcat(tmpbuf,tmp_str);
			}
			tmpbuf[strlen(tmpbuf)-1] = '\0';
			strcat(tmpbuf, "\n");

			strcat(buf, tmpbuf);
			// status
			sprintf(tmpbuf, "status = %d\n", vol_data_list[i].status);
			strcat(buf, tmpbuf);

			// record time
			(void) time(&the_time);
			sprintf(tmpbuf, "record_time = %s\n", ctime(&the_time));
			strcat(buf, tmpbuf);

			if (fputs(buf, fp)<0) {
				fclose(fp);
				free(vol_data_list);
				if (file_lock) NAS_File_Unlock(STORAGE_CONF);
				return ERROR_WRITE_FILE;
			}
		}
		// unlock the file & close it
		fclose(fp);
		free(vol_data_list);
		if (file_lock) NAS_File_Unlock(STORAGE_CONF);
	}
	return vol_cnt;
}
#endif
