//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		disk.c
//
//	Abstract: 
//		physical disk operation function
//
//	HISTORY:
//		2002/10/07	Catherine Shen -- modify "Is_HD_Exist()"
//		2002/09/25	Catherine Shen -- modify "Is_HD_Exist()" to
//					treat IDE HD's unknown status as not-connected
//		2002/09/24	Catherine Shen -- modify "Is_HD_Exist()"
//					so that it can detect the HD's power mode
//					as hdparm program does
//					modify "Is_Partition_Exist()" so that
//					RAID device can be named as "md00" instead of "md0"
//		2002/08/21	Catherine Shen -- modify "Scan_Partition" :
//					when CHECK_WITH_REPAIR, check to
//					see whether a fix-fixable or rebuild-tree
//					is needed
//		2002/08/15	Catherine Shen -- remove the code that assumes
//					drive number never exceeds 4
//		2002/08/06	Catherine Shen -- modify "Is_Partition_Exist()"
//		2002/04/02	Albert Shan adds a new function "Hot_Plug()"
//		2002/03/29	Catherine Shen add a new function "Hot_Unplug()"
//		2002/03/28	Catherine Shen add 3 functions:
//					"Is_HD_DMA_Enabled","Enable_HD_DMA","Disable_HD_DMA"
//		2002/03/12	Catherine Shen modified "Get_Partition_Info()"
//		2002/2/19	Catherine Shen modified "Mount_Partition()"
//					(remove code added on 2002/02/18)
//		2002/2/18	Catherine Shen modified "Mount_Partition()"
//		2002/1/30	Catherine Shen modified "Is_HD_Exist()"
//		2001/5/17	Louis Tsai created
//		2001/5/21	self test ok!
//
//**************************************************************************
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <sys/ioctl.h>
#include <dirent.h>
#include <errno.h>
#include <linux/hdreg.h>
#include <ctype.h>
#include <linux/ext3_fs.h>
#include "storage.h"
#include "storage_err.h"
#include "nas_quota.h"
#include "hwmon.h"
#include "reiserfs_fs_sb.h"	// ReiserFS support
#include "naslvm.h"		// LVM support


static DISK m_idedisk[]={	{"/dev/hda", 0}, {"/dev/hdb", 0}, {"/dev/hdc", 0},
				{"/dev/hdd", 0}, {"/dev/hde", 0}, {"/dev/hdf", 0},
				{"/dev/hdg", 0}, {"/dev/hdh", 0}};

static DISK m_scsidisk[]={	{"/dev/sda", 0}, {"/dev/sdb", 0}, {"/dev/sdc", 0},
				{"/dev/sdd", 0}, {"/dev/sde", 0}, {"/dev/sdf", 0},
				{"/dev/sdg", 0}, {"/dev/sdh", 0}, {"/dev/sdi", 0},
				{"/dev/sdj", 0}, {"/dev/sdk", 0}, {"/dev/sdl", 0},
				{"/dev/sdm", 0}, {"/dev/sdn", 0}, {"/dev/sdo", 0},
				{"/dev/sdp", 0}};

extern void get_drive_alphabetic_order(int hd_no, char start_char, char* out_buf, int buf_size);
extern int get_drive_no_by_alphabetic_order(char start_char, char* in_buf);

int Is_Dir_Exist(const char *path){
        DIR *dir;
        dir = opendir(path);
        if(dir == NULL){
                closedir(dir);
                return 0;
        }
        else{
                closedir(dir);
                return 1;
        }
}
//--------------------------------------//
// exp: /dev/hda5  ==> /dev/hda
//--------------------------------------//
void get_device_name(char *part_name,char *device_name) 
{
	strncpy(device_name,part_name,8);
	device_name[8] = '\0';
}

int hex_val(char c)
{
	if( isdigit(c))  
		return c - '0';
	else
		return tolower(c) + 10 - 'a';
}

int my_pow(int base,int power)
{
	int i,ret=1;
	for(i=0;i<power;i++)
		ret *=base;
	return ret;	
}

int atox(char *str) 
{
	int ret=0,len,i;
	char tmp_ch;
	len = strlen(str);
	for(i=0;i<len;i++) {
		tmp_ch = str[i];
		ret = hex_val(tmp_ch)*my_pow(16,len-i-1)+ret;
	}
	return ret;
}

int is_ide_cdrom(char *device)
{
        
    	FILE *procf;
    	char buf[100];
    	struct stat statbuf;

    	if (strncmp("/dev/hd", device, 7))
        	return 0;
    	snprintf(buf, sizeof(buf), "/proc/ide/%s/media", device+5);
    	procf = fopen(buf, "r");
    	if (procf != NULL && fgets(buf, sizeof(buf), procf))
	{
		fclose(procf);
        	return  !strncmp(buf, "cdrom", 5);
	}

	if (procf!=NULL)
		fclose(procf);
    	/* Now when this proc file does not exist, skip the
       	device when it is read-only. */
    	if (stat(device, &statbuf) == 0)
        	return (statbuf.st_mode & 0222) == 0;

    	return 0;
}

long get_dev_blksize(char *dev)
{
	int type_open=O_RDONLY;
	long size;
	int fd;

	if ((fd=open(dev, type_open))<0) {       // dev is not exist
		size=0;
	}
	else {
		if (!is_ide_cdrom(dev))
			ioctl(fd, MY_BLKGETSIZE, &size);
		else
			size=0;
		close(fd);
	}
	return size;
}

long Get_Dev_Blksize(char *dev)
{
        int type_open=O_RDONLY;
        long size;
        int fd;

        if ((fd=open(dev, type_open))<0) {       // dev is not exist
                size=0;
        }
        else {
                if (!is_ide_cdrom(dev))
                        ioctl(fd, MY_BLKGETSIZE, &size);
                else
                        size=0;
                close(fd);
        }
        return size;
}

int scan_active_disk(int type)
{
	DISK	*p;
	int	i, num;

	if (type==IDE) {
		num=sizeof(m_idedisk)/sizeof(DISK);
		p=m_idedisk;
	}
	else if (type==SCSI) {
		num=sizeof(m_scsidisk)/sizeof(DISK);
		p=m_scsidisk;
	}
	else
		return ERROR_FAIL;

	for (i=0;i<num;i++) {
		p->size=get_dev_blksize(p->dev);
		p++;
	}
	
	return SUCCESS;
}

// Catherine 2002/03/28
// remove the Cr or Lf
static int strip_EOL(char *str)
{
	if (str[strlen(str) - 1] == '\n') {
		str[strlen(str) - 1] = '\0';
		if (str[strlen(str) - 1] == '\r')
			str[strlen(str) - 1] = '\0';
		return 1;
	}
	return 0;
}

// Catherine 2002/03/28
// Get chipset name (only for pdc202xx)
int getchipname(char* chipname, int str_size)
{
	FILE *fp;
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	int ret=ERROR_NOT_FOUND; 

	if ((fp = fopen("/proc/ide/pdc202xx", "r"))==NULL)
		return ERROR_OPEN_FILE;

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		strip_EOL(buf);
		if (strlen(buf)==0)
			continue;
		if (Get_String_Field(buf, 2, ' ', tmp, BUF_SIZE)>=0) {
			if (!strncasecmp(tmp, "chipset", 7)) {
				Get_String_Field(buf, 1, ' ', chipname, str_size);
				ret = SUCCESS;
				break;
			}
		}
	}
	fclose(fp);
	return ret;
}

int Get_IDE_Chipset()
{
	char	buf[256];
	int	type=IDE_UNKNOWN;

	if (getchipname(buf, sizeof(buf))==SUCCESS)
	{
		if (!strcasecmp(buf, "pdc20262"))
			type=IDE_PDC20262;
		else
		if (!strcasecmp(buf, "pdc20265"))
			type=IDE_PDC20265;
	}
	return type;
}

// Catherine 2002/03/28
// Get chipset type to determine the ultra DMA transfer rate
// PDC20262 -> UDMA-4
// PDC20265 and else -> UDMA-5
int get_chip_udma_value()
{
	char chipname[BUF_SIZE];
	int ret;

	if ((ret=getchipname(chipname, BUF_SIZE))<0)
		return ret;

	if (!strcasecmp(chipname, "PDC20262"))
		return (int)XFER_UDMA_4;
	return (int)XFER_UDMA_5;
}

// Catherine 2002/03/28
// to get the "using_dma" flag
int chkdma(char *devname)
{
	int fd;	
	unsigned long set_dma;
	int retry=0;

  	fd = open (devname, O_RDONLY);
  	if(fd<0)
		return -1;
	while (retry<3)
	{
	    	if (ioctl(fd, HDIO_GET_DMA,&set_dma)==0)
			break;
		retry++;
		usleep(10000);
	}
       	close (fd);  
  	return set_dma;	
}

// Catherine 2002/03/28
// to set the "using_dma" flag
int setdma(char *devname,int value)
{
	int fd;	
	unsigned long set_dma ;
	int udma;
	unsigned char args[4] = {WIN_SETFEATURES,0,3,0};
	int	retry=0;
	
	set_dma = value;
	// get ultra DMA mode
	udma = get_chip_udma_value();
	if (udma<0)
		return udma;
	args[1] = (unsigned char)udma;
  	fd = open (devname, O_RDONLY);
  	if(fd<0)
		return -1;
	ioctl(fd, HDIO_DRIVE_CMD, &args); // set transfer mode
	while (retry<3)
	{
	    	if (ioctl(fd, HDIO_SET_DMA, set_dma)==0) // set DMA
			break;
		retry++;
		usleep(10000);
	}
       	close(fd);
  	return 0;
}

int Get_Disk_Sg_Map(int hdno, char* sg_name, int* host, int* channel, int* id, int* lun)
{
	FILE *fp = NULL;
	char buf[BUF_SIZE];
	int found = ERROR_NOT_FOUND, scsi_dev_order = 0, is_host_fixed = 0;

	// NAS-5168 can only detect one channel
	Get_Model_Name(buf, sizeof(buf));
	if (!strcasecmp(buf, "NAS-5168")) is_host_fixed++;

#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
	if ((fp = fopen("/proc/scsi/scsi", "r")) == NULL) {
#ifdef	DEBUG
		printf("%s -- line %d;/proc/scsi/scsi open fails?!<br>\n",
			__FUNCTION__, __LINE__);
#endif
		return ERROR_OPEN_FILE;
	}
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		char tmp1[BUF_SIZE], tmp2[BUF_SIZE];

#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
//printf("-----------------------------\n");
		if (Get_String_Field(buf, 1, ' ', tmp1, sizeof(tmp1)) != SUCCESS ||
			strcasecmp(tmp1, "Host:") != 0)
			continue;

		// this is a new device
		scsi_dev_order++;
//printf("scsi dev order : %d \n",scsi_dev_order);

		// get the host, channel, drive, lun number
		if (host!=NULL && Get_String_Field(buf, 2, ' ', tmp1, sizeof(tmp1)) == SUCCESS &&
			strlen(tmp1) > 4 && strncasecmp(tmp1, "scsi", 4) == 0)
		{
			*host = atoi((char*)(tmp1+4));
		}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
		if (channel!=NULL && Get_String_Field(buf, 4, ' ', tmp1, sizeof(tmp1)) == SUCCESS)
		{
			*channel = atoi(tmp1);
		}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
		if (id != NULL && Get_String_Field(buf, 6, ' ', tmp1, sizeof(tmp1)) == SUCCESS)
		{
			*id = atoi(tmp1);
		}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif
		if (lun != NULL && Get_String_Field(buf, 8, ' ', tmp1, sizeof(tmp1)) == SUCCESS)
		{
			*lun= atoi(tmp1);
		}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
printf("\thost = %d, channel = %d, id = %d, lun = %d\n",
	*host, *channel, *id, *lun);
#endif
		// read 2 lines
		if (fgets(buf, sizeof(buf), fp) == NULL ||
			fgets(buf, sizeof(buf), fp) == NULL)
			break;
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

		// check the device type
		if (Get_String_Field(buf, 1, ' ', tmp1, sizeof(tmp1)) != SUCCESS ||
			strcasecmp(tmp1, "Type:") != 0 ||
			Get_String_Field(buf, 2, ' ', tmp2, sizeof(tmp2)) != SUCCESS ||
			strcasecmp(tmp2, "Direct-Access") != 0)
			continue;
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

//		found++;
		if(Is_Sata_Support()){
			found = *host;
		}
		else{
			if (is_host_fixed)
				found = *id;
			else
				found = (*host)*16 + *id;
		}
#ifdef	DEBUG
printf("\tthe %d'th hd found!<br>\n", found);
#endif
		if (Is_Sata_Support()) {
			if(found < hdno) continue;
		}
		else
			if (found+1 < hdno) continue; // only the found'th disk wanted
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

		break;
	}
	fclose(fp);
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

	
	if (found+1 != hdno) {
		if(Is_Sata_Support() && found == hdno && sg_name != NULL){
			char tmp[BUF_SIZE];

			get_drive_alphabetic_order(scsi_dev_order, 'a', tmp, sizeof(tmp));
			sprintf(sg_name, "sg%s", tmp);
		}	
		else{
			found = ERROR_NOT_FOUND;
			*host = 0;
			*channel = 0;
			*id = 0;
			*lun = 0;
		}
	}
	else if(sg_name != NULL) {
		char tmp[BUF_SIZE];

		get_drive_alphabetic_order(scsi_dev_order, 'a', tmp, sizeof(tmp));
		sprintf(sg_name, "sg%s", tmp);
	}
#ifdef	DEBUG
printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
#endif

	return found;
}

// get the Pluggable? flag, Host/Channel/ID/Lun & sg name
// of a SCSI drive from temp status file /tmp/DiskX
int get_scsi_hd_status(int hdno, int* pflag, char* sgname, int* host, int* ch, int* id, int* lun)
{
	char tmp_file_name[BUF_SIZE], tmpstr[BUF_SIZE];
	int ret;

	if (pflag == NULL || sgname == NULL || host == NULL ||
		ch == NULL || id == NULL || lun == NULL)
		return ERROR_INVALID_PARAM;

	sprintf(tmp_file_name, "/tmp/Disk%d", hdno);
	if ((ret = Conf_Get_Field(tmp_file_name, "", "Pluggable", tmpstr, sizeof(tmpstr))) < 0) {
#ifdef	DEBUG
		printf("%s(%d) -- Conf_Get_Field(%s,,Pluggable) = %d!<br>\n",
			__FUNCTION__, hdno, tmp_file_name, ret);
#endif
		return ret;
	}
	*pflag = atoi(tmpstr);

	if ((ret = Conf_Get_Field(tmp_file_name, "", "Sg Map", sgname, sizeof(sgname))) < 0) {
#ifdef	DEBUG
		printf("%s(%d) -- Conf_Get_Field(%s,,sg map) = %d!<br>\n",
			__FUNCTION__, hdno, tmp_file_name, ret);
#endif
		return ret;
	}

	if ((ret = Conf_Get_Field(tmp_file_name, "", "Host", tmpstr, sizeof(tmpstr))) < 0) {
#ifdef	DEBUG
		printf("%s(%d) -- Conf_Get_Field(%s,,host) = %d!<br>\n",
			__FUNCTION__, hdno, tmp_file_name, ret);
#endif
		return ret;
	}
	*host = atoi(tmpstr);

	if ((ret = Conf_Get_Field(tmp_file_name, "", "Channel", tmpstr, sizeof(tmpstr))) < 0) {
#ifdef	DEBUG
		printf("%s(%d) -- Conf_Get_Field(%s,,channel) = %d!<br>\n",
			__FUNCTION__, hdno, tmp_file_name, ret);
#endif
		return ret;
	}
	*ch = atoi(tmpstr);

	if ((ret = Conf_Get_Field(tmp_file_name, "", "Id", tmpstr, sizeof(tmpstr))) < 0) {
#ifdef	DEBUG
		printf("%s(%d) -- Conf_Get_Field(%s,,id) = %d!<br>\n",
			__FUNCTION__, hdno, tmp_file_name, ret);
#endif
		return ret;
	}
	*id = atoi(tmpstr);

	if ((ret = Conf_Get_Field(tmp_file_name, "", "Lun", tmpstr, sizeof(tmpstr))) < 0) {
#ifdef	DEBUG
		printf("%s(%d) -- Conf_Get_Field(%s,,Lun) = %d!<br>\n",
			__FUNCTION__, hdno, tmp_file_name, ret);
#endif
		return ret;
	}
	*lun = atoi(tmpstr);
	return SUCCESS;
}

int get_scsi_disk_info(int hdno, DISK_INFO* disk_info)
{
	FILE* fp = NULL;
	char buf[BUF_SIZE], devstr[BUF_SIZE], sgname[BUF_SIZE];
	char tmpstr[255], tmp_file_name[255];
	int ret, tmpflag, hdsts = S_READY;

	if (hdno < 1 || disk_info == NULL)
		return ERROR_INVALID_PARAM;

	// step 1 : get model name from /proc/scsi/scsi
	strcpy(devstr, "scsi");

	if ((ret = get_scsi_hd_status(hdno, &tmpflag, sgname,
		&(disk_info->device_info.scsi_dev_info.host),
		&(disk_info->device_info.scsi_dev_info.channel),
		&(disk_info->device_info.scsi_dev_info.id),
		&(disk_info->device_info.scsi_dev_info.lun))) < 0)
	{
#ifdef	DEBUG
		printf("%s(%d) -- get_scsi_hd_status = %d!<br>\n",
			__FUNCTION__, hdno, ret);
#endif
		return ret;
	}
	if (!tmpflag) { // this HD doesn't exist when boot up
		disk_info->status = S_NOT_EXIST;
		strcpy(disk_info->model_name, "--");
		disk_info->capacity = 0;
		return SUCCESS;
	}

#ifdef	DEBUG
printf("get_scsi_disk_info(%d) step 4: get size!<br>\n", hdno);
#endif
	// step 2 : get vendor/capacity/size from chk_sg
	Get_HD_DevName(hdno, devstr);

	strcpy(tmpstr, "tmpXXXXXX");
	mktemp(tmpstr);
	sprintf(tmp_file_name,"/tmp/%s",tmpstr);
	sprintf(tmpstr, "/dev/%s", sgname);
	sprintf(buf, "/sbin/chk_sg %s > %s 1>> %s 2>> %s", tmpstr,
		tmp_file_name, tmp_file_name, tmp_file_name);
#ifdef DEBUG
	printf("\tcommand = %s<br>\n", buf);
#endif
	system(buf);

	if ((fp = fopen(tmp_file_name, "r")) == NULL) {
#ifdef	DEBUG
		printf("\tfails to open %s<br>\n", tmp_file_name);
#endif
		strcpy(disk_info->model_name, "--");
		disk_info->capacity = 0;
#ifndef	DEBUG
		unlink(tmp_file_name);
#endif
		return ERROR_OPEN_FILE;
	}
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		char tmp1[BUF_SIZE], tmp2[BUF_SIZE];

		if (strstr(buf, "command error") != NULL) {
			hdsts = S_RW_ERROR;
			break;
		}

		if (Get_String_Field(buf, 1, ':', tmp1, sizeof(tmp1)) == SUCCESS &&
			strncmp(tmp1, "Vendor name", 11) == 0 &&
			Get_String_Field(buf, 2, ':', tmp2, sizeof(tmp2)) == SUCCESS)
		{
			memset(disk_info->model_name, 0, sizeof(disk_info->model_name));
			strncpy(disk_info->model_name, tmp2, sizeof(disk_info->model_name)-1);
		}

		if (Get_String_Field(buf, 1, '=', tmp1, sizeof(tmp1)) == SUCCESS &&
			strncmp(tmp1, "HD size", 7) == 0 &&
			Get_String_Field(buf, 2, '=', tmp2, sizeof(tmp2)) == SUCCESS &&
			Get_String_Field(tmp2, 1, ' ', tmp1, sizeof(tmp1)) == SUCCESS)
		{
			disk_info->capacity = atof(tmp1) /1024;
			break;
		}
	}

	fclose(fp);
	disk_info->status = hdsts;
#ifndef	DEBUG
	unlink(tmp_file_name);
#else
printf("\ttemp file name = %s<br>\n", tmp_file_name);
printf("%s -- line %d; over with SUCCESS! (disk_info->status = %d)<br>\n",
__FUNCTION__, __LINE__, disk_info->status);
#endif
	return SUCCESS;
}
/*
int get_the_first_existing_scsi_hd_info(DISK_INFO* dinfo)
{
	int i, ret;
	int total_drv_cnt = Get_Profile_Integer(NASCONF_STORAGE_SECTION, NASCONF_DRIVE_NO_FIELD, 0);

	if (dinfo == NULL) return ERROR_INVALID_PARAM;

	for (i=1; i<=total_drv_cnt; i++) {
		if ((ret = get_scsi_disk_info(i, dinfo)) == SUCCESS &&
			dinfo->status == S_READY){
#ifdef DEBUG
			printf("%s -- line %d; the 1st hd is %d\n", __FUNCTION__, __LINE__, i);
#endif
			return i;
		}
#ifdef DEBUG
		printf("%s -- line %d; get_scsi_disk_info(%d) = %d, status = %d\n",
			__FUNCTION__, __LINE__, i, ret, dinfo->status);
#endif

	}
	return ERROR_NOT_FOUND;
}*/

BOOL is_scsi_disk_pwr_on(int drvno)
{
//albert add 2004/10/18 - start
	char tmpstr[HD_DEVICE_NAME_LENGTH];
	int found=0, host, ch, id, lun;
	
	found = Get_Disk_Sg_Map(drvno, tmpstr, &host, &ch, &id, &lun);
	return (found == ERROR_NOT_FOUND) ? FALSE : TRUE;
//albert add 2004/10/18 - end
/* albert mask 2004/10/18 - start
	char buf[BUF_SIZE];
	FILE* fp = NULL;
	char tmpstr[255]={"tmpXXXXXX"}, tmp_file_name[255];
	int not_found = 0, host, channel, id, lun, itmp, ret;

#ifdef	DEBUG
	printf("%s %s -- starts with drvno = %d<br>\n", __FILE__, __FUNCTION__, drvno);
#endif
	mktemp(tmpstr);
	sprintf(tmp_file_name,"/tmp/%s",tmpstr);

	if ((ret = get_scsi_hd_status(drvno, &itmp, tmpstr, &host, &channel, &id, &lun))<0)
	{
#ifdef	DEBUG
		printf("%s -- line %d, get_scsi_hd_status(%d) = %d<br>\n",
			__FUNCTION__, __LINE__, drvno, ret);
#endif
		return FALSE;
	}
	if (host < 0 || channel < 0 || id < 0 || lun < 0)
		return FALSE;

	sprintf(buf, "/bin/echo \"scsi scan-single-device %d %d %d %d\" > "
			"/proc/scsi/scsi; "
			"/bin/dmesg | /bin/awk -F ':' "
			"'$1 ~/^SCSI devices result/ {print}' | "
			"/usr/bin/tail -n 1 > %s",
			host, channel, id, lun, tmp_file_name);
	system(buf);
#ifdef	DEBUG
	printf("%s -- line %d, command = [%s]<br>\n", __FUNCTION__, __LINE__, buf);
#endif
	if ((fp = fopen(tmp_file_name, "r")) == NULL) {
#ifdef	DEBUG
		printf("\t open fails?!<br>\n");
#endif
		goto is_scsi_disk_pwr_on_out;
	}
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if (strstr(buf, "Not Found")) {
			not_found++;
			break;
		}
	}

is_scsi_disk_pwr_on_out:
	if (fp != NULL) fclose(fp);
#ifndef	DEBUG
	unlink(tmp_file_name);
#endif
	return (not_found > 0) ? FALSE : TRUE;
albert mask 2004/10/18 -end */
}

//--------------------- Export function --------------------------------//
int Get_Available_Disk_Partitions(PARTITION_ITEM *partition_item_list, int list_cnt,int system_id)
{
	int	i,j, num;
	int	current_cnt=0;
	PARTITION_TABLE tmp_table;

	scan_active_disk(IDE);
	num=sizeof(m_idedisk)/sizeof(DISK);
	for(i=0;i<num;i++) {
		if(m_idedisk[i].size != 0 ) {
			Get_Current_Partitions(m_idedisk[i].dev,&tmp_table);
			
			for(j=0;j<tmp_table.item_cnt;j++) {
				if(tmp_table.item_list[j].sys_id == system_id || system_id == 0){
					if(current_cnt+1 >list_cnt ) return ERROR_BUFFER_TOO_SMALL;
					strcpy(partition_item_list[current_cnt].part_name,tmp_table.item_list[j].part_name);
					partition_item_list[current_cnt].is_boot = tmp_table.item_list[j].is_boot;
					partition_item_list[current_cnt].start_cylinder = tmp_table.item_list[j].start_cylinder;
					partition_item_list[current_cnt].end_cylinder = tmp_table.item_list[j].end_cylinder;
					partition_item_list[current_cnt].blocks =  tmp_table.item_list[j].blocks;
					partition_item_list[current_cnt].sys_id = tmp_table.item_list[j].sys_id;
					current_cnt++;
				}
			}
		}		
	}
	
	scan_active_disk(SCSI);
	num=sizeof(m_scsidisk)/sizeof(DISK);
	for(i=0;i<num;i++) {
		if(m_scsidisk[i].size != 0 ) {
			Get_Current_Partitions(m_scsidisk[i].dev,&tmp_table);
			
			for(j=0;j<tmp_table.item_cnt;j++) {
				if(tmp_table.item_list[j].sys_id == system_id || system_id == 0){
					if(current_cnt+1 >list_cnt ) return ERROR_BUFFER_TOO_SMALL;
					strcpy(partition_item_list[current_cnt].part_name,tmp_table.item_list[j].part_name);
					partition_item_list[current_cnt].is_boot = tmp_table.item_list[j].is_boot;
					partition_item_list[current_cnt].start_cylinder = tmp_table.item_list[j].start_cylinder;
					partition_item_list[current_cnt].end_cylinder = tmp_table.item_list[j].end_cylinder;
					partition_item_list[current_cnt].blocks =  tmp_table.item_list[j].blocks;
					partition_item_list[current_cnt].sys_id = tmp_table.item_list[j].sys_id;
					current_cnt++;
				}
			}
		}
	}
	
	return current_cnt;
}

int Get_Available_LVM_Partitions(PARTITION_ITEM *partition_item_list, int list_cnt)
{	
	return Get_Available_Disk_Partitions(partition_item_list,list_cnt,0x8e);		
}

int Get_Current_Partitions(char *device_name,PARTITION_TABLE *table)
{
	char cmd_line_str[MAX_CMDLINE_LENGTH],*tmp_ptr,tmp_file_name[BUF_SIZE];
	char buffer[BUF_SIZE],tmp_str[BUF_SIZE];
	FILE *fp;
	int ret;
	
	
	if(!Is_HD_Exist(device_name))	return ERROR_NOT_EXISTED;		
	
	tmp_ptr = strrchr(device_name,'/')+1;
	sprintf(tmp_file_name,"/tmp/%s.fdisk",tmp_ptr);
	sprintf(cmd_line_str,"%s -m %s 2>/dev/null 1>%s",CMD_FDISK,device_name,tmp_file_name);
	ret = system(cmd_line_str);
	if (ret != 0) return ERROR_FAIL;

	if( (fp=fopen(tmp_file_name,"r")) != NULL ) {
		int m_item_cnt=0;
		
		//get total cylinders
		fgets(tmp_str,BUF_SIZE,fp);
		Get_String_Field(tmp_str,3,SPACE,buffer,BUF_SIZE);
		table->total_cylinders = atol(buffer);
	
		// get per_cylinder size
		fgets(tmp_str,BUF_SIZE,fp);
		Get_String_Field(tmp_str,1,SPACE,buffer,BUF_SIZE);
		table->per_cylinder_size = atol(buffer);
		Get_String_Field(tmp_str,2,SPACE,buffer,BUF_SIZE);
		table->per_cylinder_size *= atol(buffer);
		table->per_cylinder_size /= 1024;
	
		// start to get each item
		while(fgets(tmp_str,BUF_SIZE,fp)!=NULL) {
			
			Get_String_Field(tmp_str,1,SPACE,buffer,BUF_SIZE);
			strcpy(table->item_list[m_item_cnt].part_name,buffer);
			
			Get_String_Field(tmp_str,2,SPACE,buffer,BUF_SIZE);
			if(strcasecmp(buffer,"O") == 0) 
				table->item_list[m_item_cnt].is_boot = TRUE;
			else
				table->item_list[m_item_cnt].is_boot = FALSE;
				
			Get_String_Field(tmp_str,3,SPACE,buffer,BUF_SIZE);
			table->item_list[m_item_cnt].start_cylinder = atol(buffer);
			
			Get_String_Field(tmp_str,4,SPACE,buffer,BUF_SIZE);
			table->item_list[m_item_cnt].end_cylinder = atol(buffer);
			
			Get_String_Field(tmp_str,5,SPACE,buffer,BUF_SIZE);
			if(strchr(buffer,'+')!=NULL) buffer[strlen(buffer)-1] ='\0';
			table->item_list[m_item_cnt].blocks = atol(buffer);
			
			Get_String_Field(tmp_str,6,SPACE,buffer,BUF_SIZE);
			table->item_list[m_item_cnt].sys_id = atox(buffer);
			m_item_cnt++;
		}
		
		table->item_cnt = m_item_cnt;
		fclose(fp);
		unlink(tmp_file_name);
	}
	else 
		return ERROR_FAIL;
	return SUCCESS;
}


int Create_Partitions(char *device_name,PARTITION_DATA *part_data_list,int list_cnt)
{
	char para_partno[BUF_SIZE],para_capszie[BUF_SIZE],para_sysid[BUF_SIZE];
	char cmd_line_str[MAX_CMDLINE_LENGTH],tmp_str[64];
	int i=0,ret,boot_no;
	
	//initialize command line parameter!
	strcpy(para_partno,"");
	strcpy(para_capszie,"");
	strcpy(para_sysid,"");
	
//	if(!Is_HD_Exist(device_name))	return ERROR_NOT_EXISTED;
	if(list_cnt > 4 )		return PARTITION_NUM_OUT_OF_RANGE;
	for(i=0;i<list_cnt;i++) {
		if((part_data_list+i)->part_no > 4) return PARTITION_NUM_OUT_OF_RANGE;
		
		sprintf(tmp_str,"%s%d",device_name,(part_data_list+i)->part_no);
//		if(Is_Mounted(tmp_str))	return ERROR_ALREADY_MOUNTED;
//		if(Is_Partition_Exist(tmp_str)) return PARTITION_ALREADY_EXIST;
		
		
		sprintf(tmp_str,"%d,",(part_data_list+i)->part_no);
		strcat(para_partno,tmp_str);
		sprintf(tmp_str,"%x,",(part_data_list+i)->sys_id);
		strcat(para_sysid,tmp_str);
		if((part_data_list+i)->capacity_format == SIZE_M)
			sprintf(tmp_str,"%dM,",(part_data_list+i)->capacity_val);
		else
			sprintf(tmp_str,"%dM,",((part_data_list+i)->capacity_val)*1024);	
		
		strcat(para_capszie,tmp_str);
		if((part_data_list+i)->is_boot == TRUE )
			boot_no = (part_data_list+i)->part_no;		
	}
//	printf("para_partno: %s\n", para_partno);
//	printf("para_capszie: %s\n", para_capszie);
//	printf("para_sysid: %s\n", para_sysid);
	
	para_partno[strlen(para_partno)-1]='\0';
	para_capszie[strlen(para_capszie)-1]='\0';
	para_sysid[strlen(para_sysid)-1]='\0';	
	
	sprintf(cmd_line_str,"%s -n %s -q %s -z %s -a %d %s 2>/dev/null 1>/dev/null",CMD_FDISK,para_partno,para_capszie,para_sysid,boot_no,device_name);
	
//printf("%s\n",cmd_line_str);

	ret = system(cmd_line_str);
	if (ret != 0) return ERROR_FAIL;
	return SUCCESS;
	
	
}


int Delete_Partitions(char *device_name, int *part_no_list,int list_cnt)
{
	char cmd_line_str[MAX_CMDLINE_LENGTH];
	int i,ret;
	char m_para[BUF_SIZE],tmp_str[10];
	
	strcpy(m_para,"");
	strcpy(cmd_line_str,"");
	
	if(!Is_HD_Exist(device_name))	return ERROR_NOT_EXISTED;
	
	if(list_cnt == 0) 
		sprintf(cmd_line_str,"%s -d all %s 2>/dev/null 1>/dev/null",CMD_FDISK,device_name);
	else {
		for(i=0;i<list_cnt;i++) {
			sprintf(tmp_str,"%s%d",device_name,part_no_list[i]);
			if(Is_Mounted(tmp_str))	return ERROR_ALREADY_MOUNTED;
			if(!Is_Partition_Exist(tmp_str)) return PARTITION_NOT_EXIST;
			
			sprintf(tmp_str,"%d,",part_no_list[i]);
			strcat(m_para,tmp_str);
		}
		
		m_para[strlen(m_para)-1] = '\0';
		sprintf(cmd_line_str,"%s -d %s %s 2>/dev/null 1>/dev/null",CMD_FDISK,m_para,device_name);
	}
	ret = system(cmd_line_str);
//printf("%s\n",cmd_line_str);
	if (ret != 0) return ERROR_FAIL;
	return SUCCESS;
}

int Format_Partition(char *part_name, BOOL is_raid5,int fs_type)
{
	char cmd_line_str[MAX_CMDLINE_LENGTH];
	int ret = SUCCESS;

	switch(fs_type) {
		case EXT2_FS:
			sprintf(cmd_line_str,"%s -m0 -b %d -F %s 2>/dev/null 1>/dev/null",
				CMD_MKE2FS, FS_BLOCK_SIZE, part_name);
			ret = system(cmd_line_str);
			break;
		case EXT3_FS:
			if (!Is_LV_Exist(part_name) && is_raid5) {
				int sdno = 4096/FS_BLOCK_SIZE; // no of fs blocks per chunk

				sprintf(cmd_line_str,"%s -j -m0 -b %d -R stride=%d -F %s 2>/dev/null 1>/dev/null",
					CMD_MKE2FS, FS_BLOCK_SIZE, sdno, part_name);
				ret = system(cmd_line_str);
			}
			else {
				sprintf(cmd_line_str,"%s -j -m0 -b %d -F %s 2>/dev/null 1>/dev/null",
					CMD_MKE2FS, FS_BLOCK_SIZE, part_name);
				ret = system(cmd_line_str);
			}
			// 2002/12/23 Merry X'mas : disable checking by Andy Wu
			if (ret != 0) return ERROR_FAIL;

			sprintf(cmd_line_str,"%s -i 0 -c 0 %s 2>/dev/null 1>/dev/null",
				CMD_TUNE2FS,part_name);
			ret = system(cmd_line_str);
			// Merry X'mas
			break;
		case REISER_FS:
			sprintf(cmd_line_str,"%s -b %d -f %s 2>/dev/null 1>/dev/null",
				CMD_REISERMKFS, FS_BLOCK_SIZE, part_name);
			ret = system(cmd_line_str);
			break;
		default:
			return ERROR_UNKNOW_FS;
	}
	
// Albert mask on Sep 29 2004 for patch fail to create RAID0/1/5	if (ret != 0) return ERROR_FAIL;

	return SUCCESS;
}

int Scan_Partition(char *part_name, int action, int fs_type)
{
 	char  cmd_line[MAX_CMDLINE_LENGTH];
    	int   ret,out=0;

   	if(!Is_Partition_Exist(part_name))
		return PARTITION_NOT_EXIST;

   	if(Is_Mounted(part_name))
		return ERROR_ALREADY_MOUNTED;

   	switch(fs_type) {
   		case EXT2_FS:
   		case EXT3_FS:
    			if (action==CHECK_WITH_REPAIR)
      				sprintf(cmd_line,"%s -C 0 -pf %s 2>/dev/null 1>/dev/null",CMD_E2FSCK,part_name);
    			else
      				sprintf(cmd_line,"%s -C 0 -f %s 2>/dev/null 1>/dev/null",CMD_E2FSCK,part_name);
			break;
		default:
			return ERROR_UNKNOWN;
	}

    	ret = system(cmd_line);

//partition_check_over:
    	if(fs_type == EXT2_FS || fs_type == EXT3_FS) {
    		ret = ret / 256;
    		switch(ret) {
       			case FSCK_OK:
       			case FSCK_NONDESTRUCT:
       			case FSCK_UNCORRECTED:
       				out = SUCCESS;  			
       				break;
       			case FSCK_REBOOT:
       				out = SCAN_REBOOT;
       			case FSCK_ERROR:
       				out = SCAN_ERROR;
              			break;
       			case FSCK_USAGE:
       				out = SCAN_USAGE;
       				break;
       			case FSCK_LIBRARY:
       				out = SCAN_LIBRARY;
       				break;
      			default:
      				out = ERROR_UNKNOWN;
    		}
    	}

    	return out;
}

int Mount_Partition(char *part_name, char *mount_point,int fs_type)
{
	char  cmd_line[MAX_CMDLINE_LENGTH];
    	int   ret;
//	if(strstr(mount_point,"DATA")==NULL) 

	// In TS-101 model, only the first hard disk should mount with quota option
	if(strstr(mount_point,"HDA_DATA")==NULL)
    		sprintf(cmd_line,"%s -t ext3 %s %s 2>/dev/null 1>/dev/null",CMD_MOUNT,part_name,mount_point);
    	else {
		sprintf(cmd_line,"%s -t ext3 -o usrquota,grpquota,rw %s %s 2>/dev/null 1>/dev/null",CMD_MOUNT,part_name,mount_point);
//		sprintf(cmd_line,"%s -t ext3 -o usrquota=%s/.usrquota,grpquota=%s/.grpquota %s %s 2>/dev/null 1>/dev/null",CMD_MOUNT,mount_point,mount_point,part_name,mount_point);
   	}
	
	ret = system(cmd_line);
	if (ret != 0) return ERROR_FAIL;
	
	return SUCCESS;
	
}

int Unmount_Partition(char *part_name)
{
	char  cmd_line[MAX_CMDLINE_LENGTH];
    	int   ret;
    	
	// 2005.11.09, JohnsonCheng
	// We shouldn't check it here.
   	//if(!Is_Partition_Exist(part_name)) return PARTITION_NOT_EXIST;
   	if(!Is_Mounted(part_name)) return SUCCESS;
    	
    	sprintf(cmd_line,"%s %s 2>/dev/null 1>/dev/null",CMD_UMOUNT,part_name);
	
	ret = system(cmd_line);
	if (ret != 0) return ERROR_FAIL;
	return SUCCESS;
}
	
int Get_Disk_Info(char *device_name,DISK_INFO *disk_info)
{
  	FILE *fp;
	char buf[BUF_SIZE],*tmp_ptr;
	int hdno = Get_Drive_No(device_name);
	int disktype = Get_System_HD_Type();

  	if(!Is_HD_ReadWrite_Safe(hdno)){
  		strcpy(disk_info->model_name,"--");
  		disk_info->capacity = 0;
  		disk_info->status   = S_NOT_EXIST;
  		disk_info->type	= disktype;
  		memset(&(disk_info->device_info), 0, sizeof(disk_info->device_info));
  		return SUCCESS;
  	}
	// Catherine 2003/08/08
	if (disktype == IDE) {
		tmp_ptr = strrchr(device_name,'/')+1;
		sprintf(buf,"/proc/ide/%s/model",tmp_ptr);                                           

		if ((fp = fopen(buf, "r"))!=NULL)
		{
			if (fgets(buf, sizeof(buf), fp) != NULL)
				strcpy(disk_info->model_name,buf);
			fclose(fp);
		}
	
		sprintf(buf,"/proc/ide/%s/capacity",tmp_ptr);                                           

		if ((fp = fopen(buf, "r"))!=NULL)
		{
			if (fgets(buf, sizeof(buf), fp) != NULL)
				disk_info->capacity = (double) 0.5*atol(buf);
			fclose(fp);
		}
		disk_info->type = IDE;
		sprintf(buf,"/proc/ide/%s/geometry",tmp_ptr);                                           

		if ((fp = fopen(buf, "r"))!=NULL)
		{
			char tmp[BUF_SIZE];

			while (fgets(buf, sizeof(buf), fp) != NULL) {
				if (strstr(buf, "logical") != NULL &&
					Get_String_Field(buf, 2 , SPACE, tmp, sizeof(tmp)) == SUCCESS)
				{
					char fld[BUF_SIZE];

					if (Get_String_Field(tmp, 1 , '/', fld, sizeof(fld)) == SUCCESS)
						disk_info->device_info.ide_dev_info.cylinder = atoi(fld);
					if (Get_String_Field(tmp, 2 , '/', fld, sizeof(fld)) == SUCCESS)
						disk_info->device_info.ide_dev_info.head = atoi(fld);
					if (Get_String_Field(tmp, 3 , '/', fld, sizeof(fld)) == SUCCESS)
						disk_info->device_info.ide_dev_info.sector = atoi(fld);
					break;
				}
			}
			fclose(fp);
		}
	  	disk_info->status = S_READY;
	}
	else {	// SCSI
		int err;

		disk_info->type = SCSI;
		if ((err = get_scsi_disk_info(hdno, disk_info)) < 0)
			return err;
	}
	if(disk_info->status == S_READY &&
		Verify_Hidden_Conf_File(hdno) <= ERROR_HD_DEVICE_NAME )
	{
  		disk_info->status   = S_INVALID;
  		return SUCCESS;
  	}

  	return SUCCESS;
}

int Get_Partition_Info(char *part_name,PARTITION_INFO *part_info)
{
	struct statfs fsd;
  	int ret;
  	char mp_name[MP_LENGTH];
  	double tmp_blocks, tmp_bfree, tmp_bsize, tmp_bavail;

	if(!Is_Partition_Exist(part_name)) return PARTITION_NOT_EXIST;

	if((ret = Get_Mount_Point(part_name,mp_name))==SUCCESS) {

		if ( statfs(mp_name,&fsd) != 0) return ERROR_FAIL;

		tmp_blocks = fsd.f_blocks;
  		tmp_bfree  = fsd.f_bfree;     /** free blocks in fs **/
  		tmp_bavail = fsd.f_bavail;    /** free blocks for non-superuser **/
  		tmp_bsize  = fsd.f_bsize;

  		/** change free space fron b_free to b_bavail 1999/8/30 **/
 		/** Because the SMB use it to calcute free disk space   **/

  		part_info->total_size= (tmp_blocks * tmp_bsize) / 1024 ;
  		part_info->free_size = (tmp_bavail *  tmp_bsize) / 1024;
  		part_info->usage_size= part_info->total_size - part_info->free_size;
		 
	}
	else
		return ERROR_NOT_MOUNTED;
	
	return SUCCESS;
}

BOOL Is_Partition_Exist(char *part_name)
{
	char buf[BUF_SIZE], partname[HD_DEVICE_NAME_LENGTH];
	char newpartname[HD_DEVICE_NAME_LENGTH];
	char *p = strrchr(part_name, '/');
	FILE *fp;

	if (p == NULL) p = part_name;
	else p++;

	if (strncmp(p, "md", 2) == 0) {
		int raidno = atoi(p+2);
		sprintf(newpartname, "md%d", raidno);
		p = newpartname;
	}

	if ((fp = fopen(PARTITIONS_STAT_FILE, "r")) == NULL)
		return FALSE;

	while(fgets(buf,BUF_SIZE,fp)!=NULL) {
		Get_String_Field(buf,4,SPACE,partname,HD_DEVICE_NAME_LENGTH);
		if (!strcmp(partname, p)) {
			fclose(fp);
			return TRUE;
		}
	}

	fclose(fp);
	return FALSE;
}

BOOL Is_Mounted(char *part_name)
{
	int ret;
	ret = SYS_Get_One_String(MOUNT_TABLE_PATH,1,SPACE,part_name,(char*)NULL,0);
	if(ret<0) return FALSE;
	else return TRUE;
}

BOOL Is_HD_Exist(char *device_name)
{
	int ret = FALSE;
//	int type = Get_System_HD_Type();
	ret = Is_Sata_HD_Exist(device_name);
	return ret;
}

// Catherine 2003/04/14
// test if the drive exists and hotswap daemon has handled it
// s.t. it can be r/w safe (i.e. no "lost interrupt")
BOOL Is_HD_ReadWrite_Safe(int drvno)
{
	char key[BUF_SIZE];
	char devname[HD_DEVICE_NAME_LENGTH];

	if (drvno<1 || drvno>MAX_DRIVE_NO)
		return FALSE;

	if (Get_HD_DevName(drvno, devname)<=0)
		return FALSE;

	if (!Is_HD_Exist(devname))		// Strange place --- Ricky remark.
		return FALSE;

	sprintf(key, "DRIVE %d EXIST", drvno);
	if (Get_Profile_Integer("DISK", key, 0)==0)
		return FALSE;

	return TRUE;
}

int Get_Mount_Point(char *part_name,char *mp_name)
{
	
	int ret;
	char buffer[BUF_SIZE];
	ret = SYS_Get_One_String(MOUNT_TABLE_PATH,1,SPACE,part_name,(char*)NULL,0);
	if(ret>0) {
		char tmp_str[ret];
		SYS_Get_One_String(MOUNT_TABLE_PATH,1,SPACE,part_name,tmp_str,ret);
		Get_String_Field(tmp_str,2,SPACE,buffer,BUF_SIZE);
		strcpy(mp_name,buffer);
		return SUCCESS;
	}
	else
		return ERROR_FAIL;
}

int Get_Mount_DevName(char *mount_path,char *dev_name)
{
	
	int ret;
	char buffer[BUF_SIZE];
	ret = SYS_Get_One_String(MOUNT_TABLE_PATH,2,SPACE,mount_path,(char*)NULL,0);
	if(ret>0) {
		char tmp_str[ret];
		SYS_Get_One_String(MOUNT_TABLE_PATH,2,SPACE,mount_path,tmp_str,ret);
		Get_String_Field(tmp_str,1,SPACE,buffer,BUF_SIZE);
		strcpy(dev_name,buffer);
		return SUCCESS;
	}
	else
		return ERROR_FAIL;
}

BOOL Is_HD_DMA_Enabled(int drive_no)
{
	char devname[HD_DEVICE_NAME_LENGTH];

	Get_HD_DevName(drive_no, devname);

	if (chkdma(devname)>0)
		return TRUE;
	return FALSE;	
}

int Enable_HD_DMA(int drive_no)
{
	char devname[HD_DEVICE_NAME_LENGTH];
	int ret;

	if((ret=Get_HD_DevName(drive_no, devname))<0)
		return ret;
	if((ret=setdma(devname, 1))<0)
		return ERROR_FAIL;
	
	return SUCCESS;
}

int Disable_HD_DMA(int drive_no)
{
	char devname[HD_DEVICE_NAME_LENGTH];
	int ret;

	if((ret=Get_HD_DevName(drive_no, devname))<0)
		return ret;
	if((ret=setdma(devname, 0))<0)
		return ERROR_FAIL;
	
	return SUCCESS;
}

int Hot_Unplug(int drive_no)
{
	FILE *fp;
	int ret;
	char tmpfile[BUF_SIZE], tmp[BUF_SIZE];

	if (drive_no > MAX_DRIVE_NO)
		return ERROR_INVALID_PARAM;

	get_drive_alphabetic_order(drive_no, 'A', tmp, sizeof(tmp));
	sprintf(tmpfile, "/tmp/HD%s", tmp);
	fp = fopen(tmpfile, "w+");

	if (fp==NULL)
		return ERROR_OPEN_FILE;
	fclose(fp);

	ret = Disable_HD_DMA(drive_no);
	return ret;
}

int Hot_Plug(int drive_no)
{
	int ret;
	char tmpfile[BUF_SIZE], tmp[BUF_SIZE];

	if (drive_no > MAX_DRIVE_NO)
		return ERROR_INVALID_PARAM;

	get_drive_alphabetic_order(drive_no, 'A', tmp, sizeof(tmp));
	sprintf(tmpfile, "/tmp/HD%s", tmp);
	unlink(tmpfile);
	ret = Enable_HD_DMA(drive_no);
	return ret;
}

// ReiserFS support
#define EXT2_SUPER_MAGIC    0xEF53

#ifndef __EXT3_SNAP__
struct ext2_super_block {
        unsigned char   s_dummy1[56];
        unsigned char   s_magic[2];
        unsigned char   s_dummy2[46];
        unsigned char   s_uuid[16];
        unsigned char   s_volume_name[16];
};

#define ext2magic(s)    ((unsigned int) s.s_magic[0] + (((unsigned int) s.s_magic[1]) << 8))
#endif

int Get_Device_FileSystem_Type(char* device)
{
	int fd;
#ifndef __EXT3_SNAP__
	struct ext2_super_block e2sb;
#else
	struct ext3_super_block e3sb;
#endif
	int filetype = ERROR_NOT_FOUND;
	char buf[1024];

	fd = open(device, O_RDONLY);
	if (fd < 0)
		return ERROR_OPEN_FILE;

	// is ext2 / ext3 super block ?
#ifndef __EXT3_SNAP__
	if (lseek(fd, 1024, SEEK_SET) != 1024
	    || read(fd, (char *) &e2sb, sizeof(e2sb)) != sizeof(e2sb)) {
		close(fd);
		return ERROR_IO_ERROR;
	}
	if (ext2magic(e2sb) == EXT2_SUPER_MAGIC)
		filetype = EXT2_FS;
#else
	if (lseek(fd, 1024, SEEK_SET) != 1024
	    || read(fd, (char *) &e3sb, sizeof(e3sb)) != sizeof(e3sb)) {
		close(fd);
		return ERROR_IO_ERROR;
	}

	if (e3sb.s_magic == EXT2_SUPER_MAGIC) {
		if (e3sb.s_feature_compat & EXT3_FEATURE_COMPAT_HAS_JOURNAL)
			filetype = EXT3_FS;
		else
			filetype = EXT2_FS;
	}
#endif
	else {	// is reiser fs ?
		struct reiserfs_super_block *sb;
		int i;

		// reiserfs super block is either in 64-th or in 8-nd
		// 1KB block of the device
		for (i = 8; i < 65; i += 56) {
			if (lseek(fd, (off_t) i*1024, SEEK_SET)<0) {
				filetype = ERROR_IO_ERROR;
				break;
			}

			if (read(fd, (void*)buf, 1024)<1024) {
				filetype = ERROR_IO_ERROR;
				break;
			} else {
				// any magic string?
				sb = (struct reiserfs_super_block*) buf;
				if (!strcmp(sb->s_magic, "ReIsErFs") ||
					!strcmp(sb->s_magic, "ReIsEr2Fs")) {
					filetype=REISER_FS;
					break;
				}
			}
		}
	}
	close(fd);
	return filetype;
}

// Catherine 2003/08/08 ==>
int Get_System_HD_Type()
{
	char tmpbuf[BUF_SIZE] = {""};
	int type = IDE;

	Get_HD_DevName(1, tmpbuf);
	if (strncasecmp(tmpbuf, "/dev/sd", 7) == 0)
		type = SCSI;

	return type;
}

// <== Catherine 2003/08/08

float HD_Get_Check_Progress(char *dev){
	float ret=0;
	char item[64], buf[64];
	sprintf(item, "Check %s", dev);
	GetProfileString("Disk", item, "0", buf, sizeof(buf));
	ret=atof(buf);
        return ret;
}

void HD_Reset_Check_Progress(char *dev){
	char item[64];

	sprintf(item, "Check %s", dev);
	WriteProfileString("Disk", item, "0");
        return;
}

int HD_Get_Format_Progress(char *dev)
{
	int ret=0;
	char item[64], buf[64];

	sprintf(item, "Format %s", dev);
	GetProfileString("Disk", item, "-1", buf, sizeof(buf));
	ret=atoi(buf);
	return ret;
}

void HD_Reset_Format_Progress(char *dev)
{
	char item[64];

	sprintf(item, "Format %s", dev);
	WriteProfileString("Disk", item, "0");
	return;
}

long long Get_Free_Disk_Size(char *filesys_path){
        struct statfs fsd;
        long long usage_size=0;
	char cmd[128];
	int ret=0;

        if(!Is_Dir_Exist(filesys_path))
                return ERROR_PATH_NOT_EXIST;
	sprintf(cmd, "/bin/mount | grep %s 2>/dev/null 1>/dev/null", filesys_path);
	ret = system(cmd);
	if(ret != 0){
		return 0;
	}
        if (statfs(filesys_path,&fsd) == 0){
                double tmp_bsize, tmp_bavail;
                tmp_bavail = fsd.f_bavail;    /** free blocks for non-superuser **/
                tmp_bsize  = fsd.f_bsize;

                usage_size = (tmp_bavail * tmp_bsize)/1024/1024;

        }
        return usage_size;
}

