//**************************************************************************
//
//	Copyright (c) 2002  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		lvm.h
//
//	Abstract: 
//		Header for storage library LVM functions.
//
//	HISTORY:
//		2002/09/16	Catherine Shen -- add scheduling APIs
//		2002/04/15	Catherine Shen created for LVM functions
//
//**************************************************************************
#ifndef __NAS_LVM__
#define __NAS_LVM__

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include "Util.h"
#include "storage.h"
#include "storage_err.h"
#include "NAS.h"
#include "nas_quota.h"
#include "lvm-nas.h"
#include "file_lock.h"

#undef	__EXT3_SNAP__
#undef	__SUPPORT_SNAP_RW__	// Catherine 2003/03/28

#define MD_DEVICE_PREFIX	"/dev/md"
#define	VG_PATH_FORMAT		"/dev/VG-%s"
#define	LV_PATH_FORMAT		"/dev/VG-%s/LV-%s"
#define	LV_MD_NAME		"LV-MD"
#define VG_STATE_FILE_FORMAT	"/proc/lvm/VGs/%s/group"
#define PV_STATE_FILE_FORMAT	"/proc/lvm/VGs/%s/PVs/%s"
#define LV_STATE_FILE_FORMAT	"/proc/lvm/VGs/%s/LVs/%s"
#define LVM_STATE_PATH		"/proc/lvm/VGs/"

#define MP_SNAPSHOT_PATH_PREFIX "/share/SN"
#define SNAPSHOT_NAME_PREFIX	"_sn_"
#define BACKUP_SNAPSHOT_PREFIX	"_sn_bk_"
#define	SNAPSHOT_CONF		"/etc/config/snapshot.conf"
#define	SNAPD_CONFIG		"/etc/config/snapd.conf"

#define	VG_CREATE_CMD		"/sbin/vgcreate"
#define	PV_CREATE_CMD		"/sbin/pvcreate"
#define	LV_CREATE_CMD		"/sbin/lvcreate"
#define	VG_STATUS_CMD		"/sbin/vgdisplay"
#define	PV_STATUS_CMD		"/sbin/pvdisplay"
#define	LV_STATUS_CMD		"/sbin/lvdisplay"
#define	VG_EXTEND_CMD		"/sbin/vgextend"
#define	LV_EXTEND_CMD		"/sbin/lvextend"
#define	VG_REDUCE_CMD		"/sbin/vgreduce"
#define	LV_REDUCE_CMD		"/sbin/lvreduce"
#define VG_ACTIVATE_CMD		"/sbin/vgchange"
#define LV_ACTIVATE_CMD		"/sbin/lvchange"
#define	VG_REMOVE_CMD		"/sbin/vgremove"
#define	LV_REMOVE_CMD		"/sbin/lvremove"
#define	LVM_INIT_CMD		"/sbin/vgscan"
#define VG_CFGBACKUP_CMD	"/sbin/vgcfgbackup"
#define VG_CFGRESTORE_CMD	"/sbin/vgcfgrestore"
#define	EXT2_RESIZE_CMD		"/sbin/resize2fs"
#ifdef __EXT3_SNAP__
#define	EXT2_RESIZE_CMD2	"/sbin/ext2resize"
#endif
#define	REISERFS_RESIZE_CMD	"/sbin/resize_reiserfs"
#define	EXT2_ADM_CMD		"/sbin/e2fsadm"

#define WARN_THRESHOLD_FIELD	"snap warning threshold"
#define WARN_THRESHOLD_DEFAULT	90
#define DELETE_THRESHOLD_FIELD	"snap deletion threshold"
#define DELETE_THRESHOLD_DEFAULT	95

#define LV_PATH_MAX_LEN		HD_DEVICE_NAME_LENGTH
#define	MAX_LV_NO		MAX_VOLUME_NO
#define	MAX_SNAP_SIZE_ALLOWED	1.1
#define	PE_SIZE			65536	// unit: KB, default: 65536KB
#define MAX_SNAP_NO		256
#define	MAX_SNAP_NO_PER_VOLUME	5
#define TIMESTAMP_STRING_LENGTH	12	// '\0' NOT included
#define SNAP_NAME_LENGTH	(SHARE_NAME_LENGTH - TIMESTAMP_STRING_LENGTH)
				// to fit into share name length, '\0' included
#define MAX_SNAP_INSTANCE	MAX_SNAP_NO_PER_VOLUME	// the limit to the i_Max_Instance_Size field
//#define MAX_SNAP_INSTANCE_TRY	5
				// the max tries to create a snap instance
#define MAX_SNAP_INSTANCE_TIMEOUT	600
				// the timeout value inc which snap daemon
				// tries to create an instance
/* snapshot access/status masks */
#define LV_READABLE_MASK	0x01
#define LV_WRITABLE_MASK	0x02
#define	LV_SNAPSHOT_MASK	0x04
#define LV_SNAPSHOT_ORG_MASK	0x08

// Cat : for new snap repository size issue ==>
/* snapshot retrieve flag */
#define FLAG_IMPORTANCE_ASC	0x01	// importance (priority + age) lower first
#define FLAG_SPACE_USAGE_ASC	0x02	// more free space first

#define SNAP_DONATE_RATIO	50
#define SNAP_MIN_SIZE_RATIO	5	// the min snap size when ask for donations

/* the ratio of how each level can share the free snap space */
#define	PLOW_SPACE_RATIO	1
#define	PNORMAL_SPACE_RATIO	2
#define	PHIGH_SPACE_RATIO	4

/* new status defined here */
#define	S_ABORTED		-10	// snapshot is not consistent anymore
#define S_REMOVING_SNAPSHOT	9	// snapshot is removing
#define S_RESTORING_SNAPSHOT	10	// snapshot is restoring to its origin
#define	S_CREATING_SNAPSHOT	11	// snapshot is being created
#define S_VOLUME_BUSY		12	// destination volume being used
// <== Cat : for new snap repository size issue

/* system defined structure */
#ifndef _LVM_H_INCLUDE
#define LV_SNAPSHOT_USE_RATE    _IOWR ( 0xfe, 0x2c, 1)

/* Request structure LV_SNAPSHOT_USE_RATE */
typedef struct {
	int	block;
	int	rate;
} lv_snapshot_use_rate_req_t;
#endif

#define	LVM_VGDA_SIZE		4194304	// 4096 blocks

// Catherine 2003/05/15 for sending message to lvmd -->
#define	_CAT_20030515_TEST_
#ifdef	_CAT_20030515_TEST_

enum {	// event ID for lvmd
	SNAPSHOT_USE_RATE_TIMER = 20,
	SNAPD_TIMER,
	LVMD_MSG_RELOAD_CONFIG,
};

enum {	// command ID for LVMD_MSG_RELOAD_CONFIG
	CMD_SNAP_THRESHOLDS_RELOAD,
};

#define	LVM_DAEMON_PROG_NAME	"lvmd"
#endif
// <-- Catherine 2003/05/15

typedef struct struct_VG_INFO {	// from "/proc/lvm/VGs/VGx/group"
	char		c_VG_Name[LV_PATH_MAX_LEN]; // volume group name (field_1), ex. VG1
	long long	ll_VG_Size;		// VG Size in KB (field_2)
	int		i_VG_Access;		// access mode (field_3)
	int		i_VG_Status;		// status (field_4)
	int		i_VG_No;		// volume group number (field_5)
	int		i_Max_LV_Count;		// Max LV (field_6)
	int		i_LV_Count;		// Cur LV (field_7)
	int		i_Open_LV_Count;	// Open LV (field_8)
	int		i_Max_PV_Count;		// Max PV (field_9)
	int		i_PV_Count;		// Cur PV (field_10)
	int		i_Active_PV_Count;	// Act PV (field_11)
	long long	ll_PE_Size;		// PE Size in KB (field_12)
	int		i_PE_Count;		// Total PE (field_13)
	int		i_Allocated_PE_Count;	// Allocated PE (field_14)
	char		c_VG_UUID[BUF_SIZE];	// VG UUID (field_15)
} VG_INFO;

typedef struct struct_LV_INFO {	// from "/proc/lvm/VGs/MainVG/LVs/LVy"
	char		c_LV_Name[LV_PATH_MAX_LEN];	// logical volume name (field_1), ex. /dev/MainVG/LV1
	char		c_VG_Name[LV_PATH_MAX_LEN]; // volume group name (field_2)
	long long	ll_LV_Size;		// LV Size in KB(field_3/2)
	int		i_LV_Access;		// access mode (field_4)
	int		i_LV_Status;		// status (field_5)
	int		i_LV_No;		// logical volume number (field_6)
	int		i_Open_Count;		// # Open (field_7)
	int		i_Allocation;		// Allocation (field_8)
	char		c_Block_Device[LV_PATH_MAX_LEN];	// block device (field_9)
	long long	ll_SnapLV_Used_Size;	// used size
	long long	ll_SnapLV_Free_Size;	// free size
} LV_INFO;

typedef struct struct_LV_MOUNT_INFO {	// from statfs()
	char		c_LV_Mount_Point[LV_PATH_MAX_LEN];	// mount point, ex. /share/HDA_DATA
	long long	ll_LV_Total_Size;	// total size, in KB
	long long	ll_LV_Free_Size;		// free size, in KB
	long long	ll_LV_Used_Size;		// used size, in KB
} LV_MOUNT_INFO;

typedef struct struct_PV_INFO {	// from "/proc/lvm/MainVG/VGx/PVs/device"
	char		c_PV_Name[LV_PATH_MAX_LEN]; // pv name (field_1), ex. /dev/md0
	char		c_VG_Name[LV_PATH_MAX_LEN]; // volume group name (field_2)
	long long	ll_PV_Size;		// PV Size in KB (field_3/2)
	int		i_PV_Status;		// status (field_4)
	int		i_PV_No;		// physical volume number (field_5)
	int		i_Allocatable;		// allocatable (field_6)
	int		i_LV_Count;		// LV current (field_7)
	long long	ll_PE_Size;		// PE Size in KB (field_8)
	int		i_PE_Count;		// Total PE (field_9)
	int		i_Allocated_PE_Count;	// Allocated PE (field_10)
//	char		c_Block_Device[LV_PATH_MAX_LEN];	// device (field_11)
	char		c_PV_UUID[BUF_SIZE];	// PV UUID (field_12)
} PV_INFO;

typedef struct LVM_VOLUME_CONF{
	int		vol_no;
	char		lv_name[LV_PATH_MAX_LEN];		// LV path
	char		device_name[HD_DEVICE_NAME_LENGTH];
	int		raid_level;				// -2:single,-1:linear
	int		drive_no_list[MAX_DRIVE_NO];		// NAS drive number belong to this volume
	int		list_cnt;				// member hd cnt
	int		spare_drive_list[MAX_DRIVE_NO];		// spare drive numbers
	int		spare_list_cnt;				// spare disk count
	BOOL		snapshot_enable;			// free PEs for snapshots??
	int		snap_ratio;				// percentage of snapshot PEs
	long long	snap_repository_size;			// total snapshot repository size
	int             max_snap_count;
#ifdef	__EXT3_SNAP__
	int		max_snap_count;				// maximum number of snapshots
#endif
	int		status;
} LVM_VOLUME_CONF;

typedef enum {
	Create_Now=0,
	Repeat_Every,
	Daily,
	Weekly
} SNAP_SCHEDULE_METHOD;

typedef enum {
	Hour_1,
	Hour_2,
	Hour_3,
	Hour_4,
	Hour_6,
	Hour_8,
	Hour_12,
	Day_1,
	Day_2,
	Day_3,
	Week_1,
	Week_2,
} SNAP_SCHEDULE_CYCLE;

typedef enum {
	s_FAILURE = -1,
	s_NONE,
	s_SUCCESS,
	s_BUSY,
} SNAP_RESTORE_STATUS;

// Cat : for new snap repository size issue ==>
typedef enum {
	PRIORITY_LOW,
	PRIORITY_NORMAL,
	PRIORITY_HIGH,
} SNAP_PRIORITY;
// <== Cat : for new snap repository size issue

typedef struct {
	SNAP_SCHEDULE_CYCLE cycle;
	time_t	seconds;
} CYCLE_TIME_TABLE;

// Cat : for new snap repository size issue
typedef struct struct_SN_BASIC_INFO {
	int		i_Snapshot_No;			// the snapshot volume's number
	char		c_Path[LV_PATH_MAX_LEN];	// full path
	char		c_Src_LV[LV_PATH_MAX_LEN];	// source volume's path
	time_t		t_Build_Time;			// the time to create the snapshot
	SNAP_RESTORE_STATUS	Restore_Status;
	SNAP_PRIORITY	Priority;			// snapshot priority, used for auto-delete
	long long	ll_Snap_Size;
	long long	ll_Free_Size;
} SNAPSHOT_BASIC_INFO;

typedef struct struct_SN_CONF {
	int		i_Snapshot_No;		// the snapshot volume's number
	char		c_Snapshot_Path[LV_PATH_MAX_LEN];	// snapshot volume's path
	char		c_Src_LV[LV_PATH_MAX_LEN];	// source volume's path
	time_t		t_Build_Time;		// the time to create the snapshot
	int		i_Status;		// the status, including S_NOT_MOUNTED,
						// S_UNINITIALIZE,S_READY,S_INITIALIZING
	SNAP_RESTORE_STATUS	Restore_Status[MAX_SNAP_INSTANCE];
	int			i_Prev_Use_Rate[MAX_SNAP_INSTANCE];
	SNAP_PRIORITY	Priority;		// snapshot priority, used for auto-delete

	// scheduling parameters, useful only when b_Is_Scheduled_Snap==TRUE
	BOOL 		b_Is_Scheduled_Snap;	// is a scheduled snapshot
	time_t		t_Start_Time;
	SNAP_SCHEDULE_CYCLE	Cycle;
	int		i_Max_Instance_Size;	// max. number of instances of this snapshot
	char		c_Instances[MAX_SNAP_INSTANCE][SHARE_NAME_LENGTH];
//	BOOL 		b_Is_Snap_PV_Allocated;		// if the size for the snapshot volume is allocated
} SNAPSHOT_VOLUME_CONF;

typedef struct _SN_INSTANCE_INFO_ {
	char		c_Snap_Name[LV_PATH_MAX_LEN];	// snap lv name: /dev/MainVG/("LVname" + "timestamp")
							// ex. /dev/MainVG/MYSN200209171511
							// but sizeof("LVname" + "timestamp") < SHARE_NAME_LENGTH
	long long	ll_Snap_Size;			// current snapshot instance size
	int		i_Snap_Use_Rate;		// snapshot use rate
	time_t		t_Build_Time;			// the time to create the snapshot instance
	int		i_Status;			// status
} SNAPSHOT_VOLUME_INSTANCE_INFO;

typedef struct struct_SN_INFO {
	SNAPSHOT_VOLUME_CONF		snap_conf;
	long long			ll_Snapshot_Size;		// snapshot volume size in KB
	int				i_Current_Instance_Count;	// current number of instances
	int				i_Average_Snap_Use_Rate;
	SNAPSHOT_VOLUME_INSTANCE_INFO	*snap_instances;
	int				i_Restore_Percent;		// anyone instance restore progress
} SNAPSHOT_VOLUME_INFO;

typedef struct LVM_VOLUME_INFO{
	LVM_VOLUME_CONF		vol_data;
	int			snapshot_cnt;
	SNAPSHOT_VOLUME_CONF	*snap_list;
	long long		total_size;	// size in KB
	long long		free_size;	// size in KB
	int 			resync_per;
	int			snap_repository_use_rate;
} LVM_VOLUME_INFO;

typedef struct _SNAPD_CONF_ENTRY_ {
	int			snap_idx;	// snapshot volume index
	time_t			built_time;	// when this job was created
	time_t			start_time;	// when this job started
	SNAP_SCHEDULE_CYCLE	cycle;		// either hourly, daily....
	BOOL 			is_ok;		// is this cycle finished?
						// either fulfilled or time-out
} SNAPD_CONF_ENTRY;

//--------------------------- macro -------------------------------
#define MIN(a, b) (a>b) ? b : a
#define MAX(a, b) (a<b) ? b : a
#define ABS(a) (a>=0) ? a : (-1)*a

//--------------------------- lvm.c --------------------------------
int Init_LVM();
int Activate_LVM();
int Deactivate_LVM();
int Create_NAS_LVM_Partitions(int drive_no);
int Init_NAS_LVM_Disk(int drive_no);
int Clear_NAS_Disk_Content(int drive_no);	// Catherine 2003/03/25 -- for hotswap init disk
int Format_LVM_Volume(int vol_no);
int Scan_LVM_Volume(int vol_no,int action);
int Mount_LVM_Volume(int vol_no);
int Umount_LVM_Volume(int vol_no);
int Create_LVM_Volume(LVM_VOLUME_CONF *lv);
int Delete_LVM_Raid_Volume(int vol_no);
BOOL Is_LV_Exist(char* lv_path);
BOOL Is_PV_Exist(char* pv_path);
int LVM_Disk_Hot_Plug(int drive_no);
int LVM_Disk_Hot_Unplug(int drive_no);
int Activate_LV(char* lv_path, BOOL active);
int Unmount_LV(char *lv_name);
int LVM_Quick_Config_All_Disks(int raid_level);
int Reset_Disk();

int Enable_Snapshot(int volno, int ratio, int max_snap_count);
int Disable_Snapshot(int volno);

int Create_Snapshot_Volume(SNAPSHOT_VOLUME_CONF *sn_vol);
int Delete_Snapshot_Volume(int snapshot_vol_no);

int Mount_Snapshot_Volume(char* sn_lv);
int Umount_Snapshot_Volume(char* sn_lv);
int Get_Snapshot_Volume_Use_Rate(char* sn_lv);
int Restore_Snapshot_To_Org_Volume(char* sn_lv);
int Delete_Snapshot_Volume_Instance(char* sn_lv);
long long Get_Scheduled_Snapshot_Volume_Free_Space(int sn_no);
// Cat : for new snap repository size issue ==>
/* remove some APIs
int Resize_Snapshot_Volume_By_Rate(char* sn_lv, int rate);
int Resize_Snapshot_Volume_By_Increment(char* sn_lv, int more_size);
int Share_Space_Among_All_Snapshot_Instances(int snap_no, long long more_size);
int Prepare_Scheduled_Snapshot_Volume(int snap_no);
BOOL Is_A_Valid_Snapshot_Size(char* src_lv, long long snapsize, int max_inst_cnt);
*/
// <== Cat : for new snap repository size issue
BOOL Is_Allocatable_Snapshot_Volume(char* src_lv, int max_inst_cnt);
int Create_Snapshot_Instance(int snap_no);
int Clear_All_PV_Headers();

//--------------------- lvm_config.c --------------------------------
// interface functions for normal volumes
int Get_LV_Name_For_Device(char* device_name, char* lv_name, int buf_len);
int Get_LV_Volume_No(char* lv_name);
int Get_LV_Path(int volno, char* lv_path, int buf_size);
int Get_PV_Path(int volno, char* pv_path, int buf_size);
int Get_PV_Path_For_LV(char* lv_name, char* pv_name, int buf_len);
int Create_LVM_Storage_Conf();
int Restore_LVM_Storage_Conf();
int Restore_Snapshot_Conf();

BOOL Is_LVM_Volume(int vol_no);
int Get_VG_Name_By_Device(char* pv_path, char* vg_name, int vg_buf_size);
int Get_VG_Name_By_LV(char* lvname, char* vgname, int buf_size);
int Get_LV_Name_By_VG(char* vgname, char* lvname, int buf_size);

// interface functions for snapshot volumes
int Get_Snapshot_Volume_No(char* snapshot_volume_name);
int Get_Snapshot_Volume_Path(int sn_volno, char* sn_path, int buf_size);
int Get_Snapshot_PV_Path(int sn_vol_no, char* pv_path, int buf_size);
int Get_Snapshot_PV_From_LV(char* snlv, char* snpv, int buf_size);
int Get_Next_Snapshot_Volume_Instance_Path(int sn_volno, char* sn_path, int buf_size);
int Get_Snapshot_Name(char* snap_lv, char* sn_name, int buf_size);
int Get_Snapshot_Share_Name(char* snap_lv, char* share_name, int buf_size);
int Get_Snapshot_Build_Time(char* snap_lv, time_t *btime);
int Get_Snapshot_Mount_Point_String(char* snap_path, char *mp_str);

// normal volumes
int Get_All_LVM_Volumes_Conf(LVM_VOLUME_CONF **vol_data_list_ptr);
int Get_One_LVM_Volume_Conf(int vol_no, LVM_VOLUME_CONF *vol_data);
int Get_Uninit_Disk_LVM_Volumes_Conf(LVM_VOLUME_CONF **vol_data_list_ptr);
int Get_Single_Disk_LVM_Volumes_Conf(LVM_VOLUME_CONF **vol_data_list_ptr);
int Get_Raid_Disk_LVM_Volumes_Info(LVM_VOLUME_INFO **vol_info_list_ptr);
int Get_All_LVM_Configured_Vol_For_Share(LVM_VOLUME_INFO **mount_vol_ptr);
int Get_LVM_Configured_Vol_No_For_Share(char *share_name);
int Get_One_LVM_Volume_Info(int vol_no,LVM_VOLUME_INFO *vol_info);
int Get_All_LVM_Volumes_Info(LVM_VOLUME_INFO **vol_info_list_ptr);
int Add_LVM_Volume_Record(LVM_VOLUME_CONF *vol_conf);
int Remove_LVM_Volume_Record(int vol_no);
int Release_LVM_Info_List(LVM_VOLUME_INFO* vol_list, int vol_cnt);
BOOL Is_Snapshot_Taken_Volume(int vol_no);
int Get_LV_Mt_Info(LV_MOUNT_INFO *lv_minfo);
int Get_LVM_Volume_LV_From_Path(char* path, char* out_lv);

// snapshot volumes
int Get_One_Snapshot_Volume_Conf(int sn_vol_no, SNAPSHOT_VOLUME_CONF* sn_vol_conf_ptr);
int Get_All_Snapshot_Volumes_Conf(SNAPSHOT_VOLUME_CONF** sn_vol_confs_ptr);
int Get_One_Snapshot_Volume_Info(int sn_vol_no, SNAPSHOT_VOLUME_INFO* sn_vol_conf_ptr);
int Get_All_Snapshot_Volumes_Info(SNAPSHOT_VOLUME_INFO** sn_vol_confs_ptr);
int Add_Snapshot_Volume_Record(SNAPSHOT_VOLUME_CONF *sn_vol_conf);
int Get_Snapshot_Volumes_Conf_For_One_Volume(int vol_no, SNAPSHOT_VOLUME_CONF** sn_conf_list_ptr);
int Get_Snapshot_Volumes_Conf_On_One_LV(char *lv_name, SNAPSHOT_VOLUME_CONF** sn_conf_list_ptr);
int Remove_Snapshot_Volume_Record(int sn_vol_no);
int Set_Snapshot_Status(int sn_vol_no, int status);
int Get_Snapshot_Status(int sn_vol_no);
int Set_Snapshot_Volume_Size(int sn_vol_no, long long size);
int Modify_Snapshot_Scheduled_Volume_Conf(int sn_vol_no, SNAPSHOT_VOLUME_CONF* snconf);
int Set_Snapshot_Restore_Status(char* snlv, int status);
int Get_Snapshot_Restore_Status(char* snlv);

// Cat : for new snap repository size issue
int Get_Snapshots_On_One_LV_In_Order(char* srclv, uint order_flag,
	SNAPSHOT_BASIC_INFO **outlistptr);

// snapshot schedule-related
int Get_All_Snapshot_Schedules_Conf(SNAPSHOT_VOLUME_CONF** snap_vol_list);
int Get_All_Snapshot_Schedules_Info(SNAPSHOT_VOLUME_INFO** snap_info_list);
BOOL Is_A_Scheduled_Snapshot_Volume(int snap_vol_no);
BOOL Is_Instance_Name_Of_Scheduled_Snapshot(char* snap_conf_lv, char* snap_lv);

int Add_Snap_Daemon_Conf_Record(SNAPD_CONF_ENTRY* entry);
int Get_All_Snap_Daemon_Conf_Records(SNAPD_CONF_ENTRY** record_list);
//int Set_One_Snap_Daemon_Conf_Status(int snap_index, BOOL Is_OK, int try_count);
int Set_One_Snap_Daemon_Conf_Status(int snap_index, BOOL Is_OK);
int Get_One_Snap_Daemon_Conf_Record(int snap_index, SNAPD_CONF_ENTRY* record_ptr);
int Remove_One_Snap_Daemon_Conf_Record(int snap_index);
int Release_Snapshot_Info_List(SNAPSHOT_VOLUME_INFO* snap_list, int snap_cnt);
int Remove_Instance_From_Snapshot_Record(int snap_no, char* snap_instance_lv);
int Add_Instance_To_Snapshot_Record(int snap_no, char* snap_instance_lv);

int Restore_Snapshot_Conf();
int Modify_Snap_Conf_Time_Zone(int orgTZ, int newTZ);	// Catherine 2003/03/25

int Set_Snapshot_Prev_Use_Rate(char* snlv, int prv_use_rate);	// Catherine 2003/03/31
int Get_Snapshot_Prev_Use_Rate(char* snlv);			// Catherine 2003/03/31

#ifdef DEBUG /* for test only */
int get_lv_info(char* lv_name, LV_INFO* lv_info);
int get_pv_info(char* pv_name, PV_INFO* pv_info);
/*
int get_allowed_snap_vol_size(char *src_lv, int max_inst_cnt,
	long long *max_allowed_size, long long *min_allowed_size);
*/
long long get_avail_snap_vol_size(char *src_lv, BOOL create_flag);
#endif

#ifdef	STORAGE_FILE_LOCK
int Add_LVM_Volume_Record_Ex(LVM_VOLUME_CONF *vol_conf, BOOL lock_file);
#endif

//--------------------------- simple tools -------------------------------------
static inline void Make_Single_Volume_Conf(LVM_VOLUME_CONF* confptr, int driveno, int status)
{
	confptr->raid_level = SINGLE;
	confptr->drive_no_list[0] = driveno;
	confptr->list_cnt = 1;
	confptr->status = status;
	confptr->spare_list_cnt = 0; 
	confptr->snapshot_enable = FALSE;
	confptr->snap_ratio = 0;
	Get_Partition_Name(driveno, DATA_PART, confptr->device_name);
	Get_LV_Name_For_Device(confptr->device_name, confptr->lv_name, LV_PATH_MAX_LEN);
}

#endif
