/**************************************************************************
**	Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
**
**	FILE:
**		mangle.c
**
**	ABSTRACT: 
**
**	FUNCTIONS:
**
**	COMMENTS:
**
**	HISTORY:
**		2002/06/11	Created by Ellis
**
**************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <utime.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "mangle.h"

/*
** macros
*/
#ifndef __defined__bool_t
#define __defined__bool_t
typedef int	bool_t;
#endif

#ifndef B2S
#define B2S(b)		( (b) ? "TRUE" : "FALSE" )
#endif

#ifndef N2S
#define N2S(str)	( (str)? (str) : "NULL" )
#endif

#ifndef TRUE
#define TRUE		(1)
#endif

#ifndef FALSE
#define FALSE		(0)
#endif

#ifndef NULL
#define NULL		(0)
#endif

#ifndef IS_EMPTY_STRING
#define IS_EMPTY_STRING(str)	( '\0' == (str)[0] )
#endif

#ifndef SET_EMPTY_STRING
#define SET_EMPTY_STRING(str)	( (str)[0] = '\0' )
#endif

#ifndef IS_ROOT
#define IS_ROOT(path)		( '/' == (path)[0] && '\0' == (path)[1] )
#endif

//#define ENABLE_TRACE

#ifdef ENABLE_TRACE
	#define	TRACE	printf
#else
	#define	TRACE	(TRUE)? (void)0 : (void)printf
#endif

/*
** types
*/
typedef struct file_recorder_s {
	int	count;
	char	(* files)[13];
} FILE_RECORDER_S;

/*
** local function declarations
*/
static bool_t	does_file_exist		(const char *filename);

static int	copy_to_path		(const char *src_path, char *dest_path, int dest_len, int ret_err);
static int	copy_to_name		(const char *src_name, char *dest_name, int dest_len, int ret_err);

static void	init_recorder		(FILE_RECORDER_S *recorder);
static bool_t	record_file		(FILE_RECORDER_S *recorder, const char file[13]);
static bool_t	is_file_in		(FILE_RECORDER_S *recorder, const char file[13]);
static void	uninit_recorder		(FILE_RECORDER_S *recorder);

static void	make_path		(const char *startdir, const char *name, char path[PATH_MAX + 1]);

static bool_t	split_file		(const char *file, char main[NAME_MAX + 1], char ext[NAME_MAX + 1]);
static void	cat_main_ext		(char file[NAME_MAX + 1], const char *main, bool_t has_ext, const char *ext);
static int	mangle_main_ext		(const char *startdir, char main[NAME_MAX + 1], bool_t has_ext, 
					 char ext[NAME_MAX + 1], FILE_RECORDER_S *recorder);
static int	u2d_mangle_matched_file	(const char *startdir, const char *unixfile, char file[NAME_MAX + 1], 
					 int fs_len, int ext_len, FILE_RECORDER_S *recorder,
					 bool_t expected_has_ext, const char *ext_front);
static int	u2d_mangle_file		(const char *startdir, const char *unixfile, 
					 char file[NAME_MAX + 1], int fs_len, int ext_len);
					 
static bool_t	split_path		(const char *fullpath, int *idx, char startdir[PATH_MAX + 1], char path[NAME_MAX + 1]);

static int	u2d_path		(const char *unixpath, char *path, int len_path, 
					 int fs_len, int ext_len);
static int	u2d_name		(const char *startdir, const char *unixname, char *name, 
					 int len_name, int fs_len, int ext_len);

static bool_t	is_match		(const char *string, const char *front);

static int	d2u_retrieve_file	(const char *startdir, const char *file, 
					 char unixfile[NAME_MAX + 1], int fs_len, int ext_len);
	
static int	d2u_path		(const char *path, char *unixpath, int len_unixpath, 
					 int fs_len, int ext_len);	
static int	d2u_name		(const char *startdir, const char *name, char *unixname, 
					 int len_unixname, int fs_len, int ext_len);
/*
** global function definitions
*/
/** About the function's spec, please reference the document: "Unix_2_Dosname Spec.doc". */
int Unix_2_Dosname(const char *unixpath, const char *unixname, char *path, int len_path, 
		   char *name, int len_name, int fs_len, int ext_len, int type)
{
	int ret_val;

	assert(unixpath);
	assert(unixname);
	assert(path);
	assert(name);
	assert(0 < type && MANGLE_BOTH >= type);

	if (type & MANGLE_PATH)	{
		ret_val = u2d_path(unixpath, path, len_path, fs_len, ext_len);
		if (SUCCESS != ret_val) return ret_val;
	} else {
		ret_val = copy_to_path(unixpath, path, len_path, E_LEN_PATH);
		if (SUCCESS != ret_val) return ret_val;
	}

	if (type & MANGLE_FILE)	{
		ret_val = u2d_name(unixpath, unixname, name, len_name, fs_len, ext_len);
	} else {
		ret_val = copy_to_name(unixname, name, len_name, E_LEN_NAME);
	}
	
	return ret_val;
}

/** About the function's spec, please reference the document: "Dos_2_Unixname Spec.doc". */
int Dos_2_Unixname(const char *path, const char *name, char *unixpath, int len_unixpath, 
		   char *unixname, int len_unixname, int fs_len, int ext_len, int type)
{
	int ret_val;

	assert(path);
	assert(name);
	assert(unixpath);
	assert(unixname);

	if (type & MANGLE_PATH)	{
		ret_val = d2u_path(path, unixpath, len_unixpath, fs_len, ext_len);
		if (SUCCESS != ret_val) return ret_val;
	} else {
		ret_val = copy_to_path(path, unixpath, len_unixpath, E_LEN_UNIXPATH);
		if (SUCCESS != ret_val) return ret_val;
	}

	if (type & MANGLE_FILE)	{
		if (type & MANGLE_PATH)	{
			ret_val = d2u_name(unixpath, name, unixname, len_unixname, fs_len, ext_len);
		} else {
			char real_unixpath[PATH_MAX + 1];
			ret_val = d2u_path(path, real_unixpath, PATH_MAX + 1, fs_len, ext_len);
			if (SUCCESS != ret_val) return ret_val;
			
			ret_val = d2u_name(real_unixpath, name, unixname, len_unixname, fs_len, ext_len);
		}
	} else {
		ret_val = copy_to_name(name, unixname, len_unixname, E_LEN_UNIXNAME);
	}

	return ret_val;
}

/*
** local function definitions
*/

/*
**	Check if the file(normal file/dir/symbolic link/...) exists. 
**		It will follow the symbolic link.
**	RETURN:	TRUE means existance
**	    	FALSE means nonexistence
*/
static bool_t does_file_exist(const char *filename)
{
	assert(filename);
	
	return ( 0 == access(filename, F_OK) );
}

/*
**	Copy src path to dest path
**	RETURN:	SUCCESS
**	    	ret_err, If the dest_len is too small
*/
static int copy_to_path(const char *src_path, char *dest_path, int dest_len, int ret_err)
{	
	int src_strlen, i;
	
	assert(src_path);
	assert(dest_path);
	
	src_strlen = strlen(src_path);
	if ( '/' == src_path[src_strlen - 1] ) src_strlen--;
	
	if (dest_len <= src_strlen) return ret_err;
	
	for (i = 0; i < src_strlen; i++)
		dest_path[i] = src_path[i];
	dest_path[i] = '\0';
		
	return SUCCESS;
}

/*
**	Copy src name to dest name
**	RETURN:	SUCCESS
**	    	ret_err, If the dest_len is too small
*/
static int copy_to_name(const char *src_name, char *dest_name, int dest_len, int ret_err)
{
	assert(src_name);
	assert(dest_name);
	
	if ( dest_len <= strlen(src_name) ) return ret_err;

	strcpy(dest_name, src_name);

	return SUCCESS;
}

/*
**	Initialize file recoder
**	RETURN:	NONE
*/
static void init_recorder(FILE_RECORDER_S *recorder)
{
	assert(recorder);
	
	recorder->count	= 0;
	recorder->files	= NULL;	
}

/*
**	Add a file to recoder
**	RETURN:	TRUE	succeed
**		FALSE	malloc fail
*/
static bool_t record_file(FILE_RECORDER_S *recorder, const char file[13])
{
	char (* tmp)[13];
	
	assert(recorder);
	assert(file);
	assert( 13 > strlen(file) );
	
	tmp = ( char (*)[13] )realloc( recorder->files, 13 * (recorder->count + 1) );
	if (NULL == tmp) return FALSE;
	
	recorder->files = tmp;
	strcpy(recorder->files[recorder->count], file);
	recorder->count++;
	
	return TRUE;
}

/*
**	Check if the file is in the recoder
**	RETURN:	TRUE	OK
**		FALSE	not exist
*/
static bool_t is_file_in(FILE_RECORDER_S *recorder, const char file[13])
{
	int i;
	assert(recorder);
	assert(file);
	assert( 13 > strlen(file) );
	
	for (i = 0; i < recorder->count; i++)
		if ( 0 == strcmp(recorder->files[i], file) )
			return TRUE;
			
	return FALSE;
}

/*
**	Uninitialize file recoder
**	RETURN:	NONE
*/
static void uninit_recorder(FILE_RECORDER_S *recorder)
{
	assert(recorder);
	
	if (NULL != recorder->files) {
		assert(0 < recorder->count);
		free(recorder->files);
		recorder->files = NULL;
	} else {
		assert(0 == recorder->count);
	}
}

/*
**	Make a path by concatenating startdir and name.
**	IMPORTANT: startdir can be "" and the path is name.
**	RETURN: none
*/
static void make_path(const char *startdir, const char *name, char path[PATH_MAX + 1])
{
	assert(startdir);
	assert(name);
	assert(path);
	
	if ( IS_EMPTY_STRING(startdir) ) {
		SET_EMPTY_STRING(path);
	} else {
		strcpy(path, startdir);
		if ( '/' != path[strlen(startdir) - 1] ) strcat(path, "/");
	}
	
	strcat(path, name);
}

/*
**	Split file to main part and ext part.
**	RETURN: TRUE means file has the ext part;
**		FALSE means file hasn't the ext part, and set the ext as "".
*/
static bool_t split_file(const char *file, char main[NAME_MAX + 1], char ext[NAME_MAX + 1])
{
	int i, j;
	
	assert(file);
	assert(main);
	assert(ext);
	
	/** find '.' and split the file */
	for ( i = strlen(file); i > 0; i-- )
		if ('.' == file[i]) break;
		
	if ('.' != file[i]) {		
		strcpy(main,	file);
		strcpy(ext,	"");
		return FALSE;
	} else {
		for (j = 0; j < i; j++) main[j] = file[j];
		main[j] = '\0';
		strcpy (ext,	&file[i+1]);
		return TRUE;
	}
}

/*
**	Concatenate main name and ext name to make a complete filename.
**	RETURN: none
*/
static void cat_main_ext(char file[NAME_MAX + 1], const char *main, bool_t has_ext, const char *ext)
{
	assert(file);
	assert(main);
	
	strcpy(file, main);
	
	if (has_ext) {
		strcat(file, ".");
		strcat(file, ext);
	}
}

/*
**	Mangle main name and ext name to dos 8+3 format.
**		If succeed, add file to recorder
**	RETURN: SUCCESS
**		E_MALLOC
**		E_TOOMANYFILES
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int mangle_main_ext(const char *startdir, char main[NAME_MAX + 1], bool_t has_ext, 
			   char ext[NAME_MAX + 1], FILE_RECORDER_S *recorder)
{
	int i;
	
	assert(startdir);
	assert(main);
	assert(recorder);

	/** mangle ext */
	if ( has_ext && 3 < strlen(ext) ) ext[3] = '\0';
	
	/** mangle main */
	for (i = 1; TRUE; i++) {
		char postfix[10];
		int  len_postfix;
		char tmp_main[NAME_MAX + 1];
		char fullname[NAME_MAX + 1];
		char fullpath[PATH_MAX + 1];
		
		/** make postfix */
		sprintf(postfix, "~%d", i);
		len_postfix = strlen(postfix);
		assert(10 > len_postfix);
		if (8 < len_postfix) return E_TOOMANYFILES;
		
		/** mangle main */
		strcpy(tmp_main, main);
		if ( 7 >= strlen(tmp_main) )
			tmp_main[strlen(tmp_main) - 1] = '\0';
		tmp_main[8 - len_postfix] = '\0';
		strcat(tmp_main, postfix);
		
		/** make fullname */
		cat_main_ext(fullname, tmp_main, has_ext, ext);
		assert( 12 >= strlen(fullname) );
		
		/** does fullpath exist? */
		make_path(startdir, fullname, fullpath);
		if ( FALSE == does_file_exist(fullpath) &&
		     FALSE == is_file_in(recorder, fullname) ) {
			strcpy(main, tmp_main);
			if ( FALSE == record_file(recorder, fullname) ) return E_MALLOC;
			return SUCCESS;
		}
	}
}

/*
**	Mangle unix filename to dos 8+3 format if matched.
**		If not match or needn't match, return SUCCESS
**	RETURN: SUCCESS
**		E_MALLOC
**		E_TOOMANYFILES
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int u2d_mangle_matched_file(const char *startdir, const char *unixfile, char file[NAME_MAX + 1], 
				   int fs_len, int ext_len, FILE_RECORDER_S *recorder,
				   bool_t expected_has_ext, const char *ext_front)
{
	bool_t	has_ext;
	char	unixfile_main[NAME_MAX + 1];
	char	unixfile_ext[NAME_MAX + 1];
	int	ret_val;
	
	assert(startdir);
	assert(unixfile);
	assert(file);
	assert(recorder);

	has_ext = split_file(unixfile, unixfile_main, unixfile_ext);
	
	/** is ext match */
	if ( (has_ext != expected_has_ext) ||
	     ( has_ext && FALSE == is_match(unixfile_ext, ext_front) ) ) {
		strcpy(file, unixfile);
		return SUCCESS;
	}

	/** needn't mangling */	
	if ( ( !has_ext || 0 == ext_len || strlen(unixfile_ext) <= ext_len ) &&
	     ( strlen(unixfile_main) + strlen(unixfile_ext) <= fs_len ) ) {
	     strcpy(file, unixfile);
	     return SUCCESS;
	}
	
	/** mangle it */	
	ret_val = mangle_main_ext(startdir, unixfile_main, has_ext, unixfile_ext, recorder);
	if (SUCCESS != ret_val) return ret_val;

	cat_main_ext(file, unixfile_main, has_ext, unixfile_ext);
	return SUCCESS;
}

/*
**	Mangle unix filename to dos 8+3 format.
**		file can be "", it means root dir.
**	RETURN: SUCCESS
**		E_MALLOC
**		E_TOOMANYFILES
**		E_NOSUCHFILE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int u2d_mangle_file(const char *startdir, const char *unixfile, char file[NAME_MAX + 1],
			   int fs_len, int ext_len)
{
	char		unixfile_path[PATH_MAX + 1];
	bool_t		has_ext;
	char		unixfile_main[NAME_MAX + 1];
	char		unixfile_ext[NAME_MAX + 1];
	char		unixfile_ext_front[NAME_MAX + 1];	
	DIR		*dp;
	struct dirent	*dep;
	FILE_RECORDER_S recorder;
	
	assert(startdir);
	assert(unixfile);
	assert(file);

	/** is root? */
	if ( IS_EMPTY_STRING(unixfile) ) {
		strcpy(file, unixfile);
		return SUCCESS;
	}
		
	/** does the file exists? */
	make_path(startdir, unixfile, unixfile_path);
	if ( FALSE == does_file_exist(unixfile_path) ) return E_NOSUCHFILE;

	/** split file to main and ext */	
	has_ext = split_file(unixfile, unixfile_main, unixfile_ext);
	
	/** if needn't to mangle, then return it directly */
	if ( ( !has_ext || 0 == ext_len || strlen(unixfile_ext) <= ext_len ) &&
	     ( strlen(unixfile_main) + strlen(unixfile_ext) <= fs_len ) ) {
	     strcpy(file, unixfile);
	     return SUCCESS;
	}
	
	if (has_ext) {
		strcpy(unixfile_ext_front, unixfile_ext);
		unixfile_ext_front[3] = '\0';
	}
	
	/** mangle all of matched file */
	dp = opendir(startdir);
	if (NULL == dp) return E_NOSUCHFILE;
	
	init_recorder(&recorder);

	while ( NULL != ( dep = readdir(dp) ) ) {
		int ret_val;
		
		/** skip dot and dot-dot */
		if ( 0 == strcmp(".",  dep->d_name) )	continue;
		if ( 0 == strcmp("..", dep->d_name) )	continue;

		ret_val = u2d_mangle_matched_file(startdir, dep->d_name, file, fs_len, ext_len, 
						  &recorder, has_ext, unixfile_ext_front);
		if (SUCCESS != ret_val) {
			uninit_recorder(&recorder);	
			closedir(dp);
			return ret_val;
		}
		
		if ( 0 == strcmp(dep->d_name, unixfile) ) break;
	}
	
	uninit_recorder(&recorder);
	closedir(dp);
	
	return SUCCESS;
}

/*
**	Split path up.
**	IMPORTANT: if a path is "abc/123/def/456", will return:
**		first time:	startdir = "", 			path = "abc";
**		second time:	startdir = "abc/", 		path = "123";
**		third time:	startdir = "abc/123/", 		path = "def";
**		fourth time:	startdir = "abc/123/def/", 	path = "456";
**		fifth time:	return FALSE;
**	If fullpath is absolute path, then first path is ""!
**
**	RETURN: TRUE	split succeed.
**		FALSE	meet the path tail.
*/
static bool_t split_path(const char *fullpath, int *idx, char startdir[PATH_MAX + 1], char path[NAME_MAX + 1])
{
	int i;
	
	assert(fullpath);
	assert(idx);
	assert(startdir);
	assert(path);
	
	/** meet tail */
	if ('\0' == fullpath[*idx]) return FALSE;
	
	/** make startdir */
	for (i = 0; i < *idx; i++)
		startdir[i] = fullpath[i];
				
	startdir[i] = '\0';
	
	/** make single path */
	for ( i = 0; ('\0' != fullpath[*idx]) && ('/' != fullpath[*idx]); i++, (*idx)++ )
		path[i] = fullpath[*idx];
		
	path[i] = '\0';
	
	/** skip '/' */
	if ('/' == fullpath[*idx]) (*idx)++;
	
	return TRUE;
}

/*
**	Convert unixpath to dos 8+3 format.
**	RETURN: SUCCESS
**		E_MALLOC
**		E_TOOMANYFILES
**		E_NOSUCHFILE
**		E_LEN_PATH
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int u2d_path(const char *unixpath, char *path, int len_path, int fs_len, int ext_len)
{
	char	startdir[PATH_MAX + 1];
	char	individual_unixpath[NAME_MAX + 1];
	char	individual_path[NAME_MAX + 1];
	int	unixpath_idx;
	int	expected_len_path;
	bool_t	first_loop;
	
	assert(unixpath);
	assert(path);	
	
	/** initialize path */
	expected_len_path = 1;
	if (len_path < expected_len_path) return E_LEN_PATH;
	SET_EMPTY_STRING(path);
	
	/** make path */
	unixpath_idx = 0;
	first_loop = TRUE;
	while ( split_path(unixpath, &unixpath_idx, startdir, individual_unixpath) ) {
		/** mangle invdividual unixpath */
		int ret_val = u2d_mangle_file(startdir, individual_unixpath, individual_path, 
					      fs_len, ext_len);
		if (SUCCESS != ret_val) return ret_val;
								
		/** append individual_path to path */
		if (first_loop) {
			/** IMPORTANT: if unixpath is absolute, then the individual_unixpath will be "". */
			expected_len_path += strlen(individual_path);
			if (len_path < expected_len_path) return E_LEN_PATH;
			strcat(path, individual_path);
			first_loop = FALSE;
		} else {
			expected_len_path += 1 + strlen(individual_path);
			if (len_path < expected_len_path) return E_LEN_PATH;
			strcat(path, "/");
			strcat(path, individual_path);
		}
	}
	
	return SUCCESS;
}

/*
**	Convert unixname to dos 8+3 format.
**	RETURN: SUCCESS
**		E_MALLOC
**		E_TOOMANYFILES
**		E_NOSUCHFILE
**		E_LEN_NAME
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int u2d_name(const char *startdir, const char *unixname, 
		    char *name, int len_name, int fs_len, int ext_len)
{
	char tmp_name[NAME_MAX + 1];
	int  ret_val;
	
	assert(startdir);
	assert(unixname);
	assert(name);
	
	ret_val = u2d_mangle_file(startdir, unixname, tmp_name, fs_len, ext_len);
	if (SUCCESS != ret_val) return ret_val;
		
	if ( len_name < strlen(tmp_name) + 1 ) return E_LEN_NAME;
	strcpy(name, tmp_name);
	
	return SUCCESS;
}

/*
**	Check if the string has the front.
**	RETURN: TRUE	match
**		FALSE	unmatch
*/
static bool_t is_match(const char *string, const char *front)
{
	int i;
	
	assert(string);
	assert(front);
	
	for (i = 0; i < strlen(front); i++) {
		if ('\0' == string[i] || front[i] != string[i])
			return FALSE;
	}
	
	return TRUE;
}

/*
**	dos-to-unix try to retrieve the unixfile.
**		file can be "", it means root dir.
**	RETURN: SUCCESS
**		E_MALLOC
**		E_TOOMANYFILES
**		E_NOSUCHFILE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int d2u_retrieve_file(const char *startdir, const char *file, char unixfile[NAME_MAX + 1],
			     int fs_len, int ext_len)
{
	char		file_path[PATH_MAX + 1];
	bool_t		has_ext;
	char		file_main[NAME_MAX + 1];
	char		file_ext[NAME_MAX + 1];
	DIR		*dp;
	struct dirent	*dep;
	FILE_RECORDER_S recorder;
	char		tmp_file[NAME_MAX + 1];
	
	assert(startdir);
	assert(unixfile);
	assert(file);
	
	/** if the file exists, then return it */
	make_path(startdir, file, file_path);

	if ( IS_EMPTY_STRING(file) || does_file_exist(file_path) ) {
		strcpy(unixfile, file);
		return SUCCESS;
	}
		
	/** split file to main and ext */	
	has_ext = split_file(file, file_main, file_ext);
		
	if ( 8 < strlen(file_main) )		return E_NOSUCHFILE;
		
	if ( has_ext && 3 < strlen(file_ext) )	return E_NOSUCHFILE;
	
	/** mangle all of matched file */
	dp = opendir(startdir);
	if (NULL == dp) return E_NOSUCHFILE;
	
	init_recorder(&recorder);

	while ( NULL != ( dep = readdir(dp) ) ) {
		int ret_val;
		
		/** skip dot and dot-dot */
		if ( 0 == strcmp(".",  dep->d_name) )	continue;
		if ( 0 == strcmp("..", dep->d_name) )	continue;

		ret_val = u2d_mangle_matched_file(startdir, dep->d_name, tmp_file, fs_len, ext_len, 
						  &recorder, has_ext, file_ext);
		if (SUCCESS != ret_val) {
			uninit_recorder(&recorder);	
			closedir(dp);
			return ret_val;
		}
		
		if ( 0 == strcmp(file, tmp_file) ) {
			strcpy(unixfile, dep->d_name);
			uninit_recorder(&recorder);	
			closedir(dp);
			return SUCCESS;			
		}
	}
	
	uninit_recorder(&recorder);
	closedir(dp);
	
	return E_NOSUCHFILE;
}

/*
**	Convert the dos path to unix path.
**	RETURN: SUCCESS
**		E_LEN_UNIXPATH
**		E_NOSUCHFILE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int d2u_path(const char *path, char *unixpath, int len_unixpath, int fs_len, int ext_len)
{
	char	startdir[PATH_MAX + 1];
	char	individual_path[NAME_MAX + 1];
	char	individual_unixpath[NAME_MAX + 1];
	int	path_idx;
	int	expected_len_unixpath;
	bool_t	first_loop;
	
	assert(path);	
	assert(unixpath);
	
	/** initialize unixpath */
	expected_len_unixpath = 1;
	if (len_unixpath < expected_len_unixpath) return E_LEN_UNIXPATH;
	SET_EMPTY_STRING(unixpath);
	
	/** make unixpath */
	path_idx = 0;
	first_loop = TRUE;
	while ( split_path(path, &path_idx, startdir, individual_path) ) {
		int ret_val;
		
		/** mangle invdividual path */
		if ( IS_ROOT(startdir) )
			ret_val = d2u_retrieve_file("/",      individual_path, 
						    individual_unixpath, fs_len, ext_len);
		else
			ret_val = d2u_retrieve_file(unixpath, individual_path, 
						    individual_unixpath, fs_len, ext_len);
		if (SUCCESS != ret_val) return ret_val;

		/** append individual_unixpath to unixpath */
		if (first_loop) {
			/** IMPORTANT: if path is absolute, then the individual_unixpath will be "". */
			expected_len_unixpath += strlen(individual_unixpath);
			if (len_unixpath < expected_len_unixpath) return E_LEN_UNIXPATH;
			strcat(unixpath, individual_unixpath);
			first_loop = FALSE;
		} else {
			expected_len_unixpath += 1 + strlen(individual_unixpath);
			if (len_unixpath < expected_len_unixpath) return E_LEN_UNIXPATH;
			strcat(unixpath, "/");
			strcat(unixpath, individual_unixpath);
		}
	}
	
	return SUCCESS;
}

/*
**	Convert the dos name to unix name.
**	RETURN: SUCCESS
**		E_LEN_UNIXNAME
**		E_NOSUCHFILE
**	NOTE: 
**		About the meaning of return values, please reference the document: "Unix_2_Dosname Spec.doc".
*/
static int d2u_name(const char *startdir, const char *name, char *unixname, int len_unixname,
		    int fs_len, int ext_len)
{
	char tmp_unixname[NAME_MAX + 1];
	int  ret_val;
	
	assert(startdir);
	assert(name);
	assert(unixname);
	
	TRACE("d2u_name: startdir: %s\n", startdir);
	
	ret_val = d2u_retrieve_file(startdir, name, tmp_unixname, fs_len, ext_len);
	if (SUCCESS != ret_val) return ret_val;
		
	if ( len_unixname < strlen(tmp_unixname) + 1 ) return E_LEN_UNIXNAME;
	strcpy(unixname, tmp_unixname);
	
	return SUCCESS;
}


