//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		cfg_printer.c
//
//	Abstract: 
//		Printer config function
//
//	COMMENTS: 	N/A
//
//	HISTORY:
//      08/20/01	David Yang Created
//
//**************************************************************************
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <Util.h>
#include <config.h>
#include <storage.h>
#include "cfg_printer.h"

#define	SAMBA_CONF_PATH			"/etc/config/smb.conf"
#define	SZ_PRINTER_SPOOLPATH		"/mnt/HDA_ROOT/.spool"
#define SZ_PRINTER_PRINTERSH_PATH	"/etc/init.d/printer.sh %s > /dev/null"
#define SZ_PRINTER_PAPDSH_PATH      "/etc/init.d/papd.sh %s > /dev/null"

#define PRINTER_DEF_SPOOL_SIZE		0    // in Kbyte, 0 means 'no limit'
#define	SZ_PRINTCAP_TEMPLATE 		"%s|%s :\\\n\t:sd=%s:\\\n\t:mx#%d:\\\n\t:sh:\\\n\t:sf:\\\n\t:rw=true:\\\n\t:lp=/dev/lp0:\n\nprinters:\\\n\t:sd=%s:\\\n\t:mx#%d:\\\n\t:sh:\\\n\t:sf:\\\n\t:rw=true:\\\n\t:lp=/dev/lp0:\n"
//#define SZ_PRINTCAP_USB			"%s|%s :\\\n\t:sd=%s:\\\n\t:mx#%d:\\\n\t:sh:\\\n\t:sf:\\\n\t:rw=true:\\\n\t:lp=/dev/usb/lp0:\n\nprinters:\\\n\t:sd=%s:\\\n\t:mx#%d:\\\n\t:sh:\\\n\t:sf:\\\n\t:rw=true:\\\n\t:lp=/dev/usb/lp0:\n"
#define SZ_PRINTCAP_USB			"%s|%s :\\\n\t:sd=%s:\\\n\t:mx#%d:\\\n\t:sh:\\\n\t:sf:\\\n\t:rw=true:\\\n\t:lp=/dev/usb/lp0:\n\nprinters :\\\n\t:sd=%s:\\\n\t:mx#%d:\\\n\t:sh:\\\n\t:sf:\\\n\t:rw=true:\\\n\t:lp=/dev/usb/lp0:\n\n"
		
//#define	SZ_PAPDCONF_TEMPLATE 	    "%s:\\\n\t:pr=printers:op=administrator:\n"
#define       SZ_PAPDCONF_TEMPLATE	""

// These constants are used in /etc/config/uLinux.conf
#define	SZ_PRINTER_SECTION		"Printers"
#define	SZ_PRINTER_SUPPORT_FIELD	"Support"
#define SZ_PRINTER_USB_SUPPORT_FIELD	"Usb Support"
#define SZ_PRINTER_PARALLEL_SUPPORT_FIELD    "Parallel Support"
#define SZ_PRINTER_TYPE_FIELD		"Type"		// Parallel or USB
#define	SZ_PRINTER_ENABLE_FIELD		"Enable"
#define	SZ_PRINTER_NAME_FIELD		"Name"
#define	SZ_PRINTER_COMMENT_FIELD	"Comment"
#define	SZ_PRINTER_NAME_DEFAULT         "Printer"
#define SZ_EXIST_FIELD			"Phy exist"
#define SZ_MANU_FIELD			"Manufacture"
#define SZ_MODEL_FIELD			"Model"
#define	SZ_PRINTER_COMMENT_DEFAULT      "Printer port on NAS"
#define	SZ_PRINTER_CAPPATH_DEFAULT	"/etc/config/printcap"
#define	SZ_PRINTER_PAPDPATH_DEFAULT	"/etc/atalk/papd.conf"

//If the printcap file can be set in uLinux.conf,unmark this function
/*
#define	SZ_PRINTER_CAPPATH_FIELD	"Printcap Path"
int get_printcap_path(char *path, int buf_size)
{
	int val, ret = SUCCESS; 

	val = GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_CAPPATH_FIELD, SZ_PRINTER_CAPPATH_DEFAULT, printer_comment, buf_size);
	if(!val)
		ret = ERROR_READ_FILE;
	return ret;	
} 
*/

// declaration of some local functions
int set_printer_for_atalk(char *printer_name);
int set_load_printer(int enable_load);
int set_printer_cap_name(char *printer_name);
int set_printer_cap_comment(char *printer_comment);


/////////////////////////////////////////
// Main global functions
/////////////////////////////////////////
/*
 * Check if NAS support printer port
 * Return:
 *	true	Support		
 *	false	Not support
 */
BOOL Is_Support_Printer()
{
	char buf[BUF_SIZE];
	GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_SUPPORT_FIELD, "FALSE", buf, sizeof(buf));
	return NOT_FALSE(buf);	
}

BOOL Is_Support_Usb_Printer()
{
	char	buf[BUF_SIZE];

	GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_USB_SUPPORT_FIELD, "FALSE", buf, sizeof(buf));
	return NOT_FALSE(buf);
}

BOOL Is_Support_Parallel_Printer()
{
	char    buf[BUF_SIZE];

	GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_PARALLEL_SUPPORT_FIELD, "TRUE", buf, sizeof(buf));
	return NOT_FALSE(buf);
}

int Prn_Get_Type()
{
	char	buf[BUF_SIZE];
	int	ret=PRN_PARALLEL;

	GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_TYPE_FIELD, "USB", buf, sizeof(buf));
	if (!strcasecmp(buf, "Parallel"))
		ret=PRN_PARALLEL;
	else
	if (!strcasecmp(buf, "USB"))
		ret=PRN_USB;
	return ret;
}

int Prn_Set_Type(int type)
{
	int	val;

	if (type==PRN_PARALLEL)
	        val = WriteProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_TYPE_FIELD, "Parallel");
	else
		val = WriteProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_TYPE_FIELD, "USB");
        if(!val)
                return ERROR_WRITE_FILE;
	else
		return SUCCESS;
}

/*
 * Check if spool path is initialized
 * Return:
 *	true	Initialized			
 *	false	not instilized.
 */
BOOL Is_Spool_Initialized()
{
	BOOL ret = FALSE;
	ret = Is_Spool_Partition_Ready();
	return ret;	
}
 
/*
 * Check if printer enabled
 * Return:
 *	true	Enabled			
 *	false	Disabled
 */
BOOL Is_Printer_Enabled()
{
	char buf[BUF_SIZE];
	GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_ENABLE_FIELD, "FALSE", buf, sizeof(buf));
	return NOT_FALSE(buf);	
}

/*
 * Start/Stop lpd daemon
 * Return:
 * 	0		pass
 *  other		fail, according to system report
 */
int Install_Lpd(int bEnable)
{
	char buf[BUF_SIZE];
	sprintf(buf, SZ_PRINTER_PRINTERSH_PATH, (bEnable)?"start":"stop");
	if(bEnable){
		Add_Printer_Admin();
		system("/etc/init.d/smb restart > /dev/null");
	}
	else{
		Conf_Remove_Section(SAMBA_CONF_PATH,"Printers");
		system("/etc/init.d/smb restart > /dev/null");
	}
	
	return system(buf);
}

/*
 * Start/Stop PAP daemon
 * Return:
 * 	0		pass
 *  other		fail, according to system report
 */
 
int Install_PAPD(int bEnable)
{
	char buf[BUF_SIZE];
	sprintf(buf, SZ_PRINTER_PAPDSH_PATH, (bEnable)?"start":"stop");
	return system(buf);
}

/*
 * Enable Printer
 * Return:
 *	0		pass	
 *  ERROR_WRITE_FILE	can not write to config file
 *	others	fail, according to samba's report
 */
int Enable_Printer(int bEnable, BOOL refresh_now)
{
	int val, ret = SUCCESS;

	val = WriteProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_ENABLE_FIELD, BOOL2STR(bEnable));
	if(!val)
		return ERROR_WRITE_FILE; 
	
	// Set load printer	
	ret = set_load_printer(bEnable);

	// restart samba to make the change effective...
	if (refresh_now && Is_Samba_Enabled()) 
		Refresh_Samba();
	
	return ret;
}


/*
 * Read Printer Name from config file
 * Return:
 *	0			pass
 *	ERROR_READ_FILE	can not read config file
 */
int Get_Printer_Name(char *printer_name, int buf_size)
{
	int val, ret = SUCCESS; 

	val = GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_NAME_FIELD, SZ_PRINTER_NAME_DEFAULT, printer_name, buf_size);
	if(!val)
		ret = ERROR_READ_FILE;
	return ret;
}

/*
 * Write Printer Name to config file
 * Return:
 *	0			pass
 *	ERROR_WRITE_FILE	can not write config file
 */
int Set_Printer_Name(char *printer_name, BOOL refresh_now)
{
	int val, ret = SUCCESS;

	val = WriteProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_NAME_FIELD, printer_name);
	if(!val)
		return ERROR_WRITE_FILE;

	// activate printer name settings...	
	ret = set_printer_cap_name(printer_name);	

	// restart samba to make the change effective...
	if (refresh_now && Is_Samba_Enabled()) 
		Refresh_Samba();
	
	if ( ret == SUCCESS )
	{	
        	set_printer_for_atalk(printer_name);
    	}
	
	return ret;
}

/*
 * Read Printer Comment from config file
 * Return:
 *	0			pass
 *	ERROR_READ_FILE	can not read config file
 */
int Get_Printer_Comment(char *printer_comment, int buf_size)
{
	int val, ret = SUCCESS; 

	val = GetProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_COMMENT_FIELD, SZ_PRINTER_COMMENT_DEFAULT, printer_comment, buf_size);
	if(!val)
		ret = ERROR_READ_FILE;
	return ret;
}

/*
 * Write Printer Comment to config file
 * Return:
 *	0			pass
 *	ERROR_WRITE_FILE	can not write config file
 */
int Set_Printer_Comment(char *printer_comment, BOOL refresh_now)
{
	int val, ret = SUCCESS; 

	val = WriteProfileString(SZ_PRINTER_SECTION, SZ_PRINTER_COMMENT_FIELD, printer_comment);
	if(!val)
		return ERROR_WRITE_FILE;

	// activate printer comment settings...	
	ret= set_printer_cap_comment(printer_comment);

	// refresh samba to make the change effective...
	if (refresh_now && Is_Samba_Enabled()) 
		Refresh_Samba();
	
	return ret;
}

///////////////////////////////
// local functions
//////////////////////////////

int set_load_printer(int enable_load)
{
	char security[256];
	int ret = WritePrivateProfileString("global", "printcap name", SZ_PRINTER_CAPPATH_DEFAULT, SAMBA_CONF_PATH);	
	ret = WritePrivateProfileString("global", "show add printer wizard", "no", SAMBA_CONF_PATH);
			
	ret = WritePrivateProfileString("printers", "writable", "no", SAMBA_CONF_PATH);
	ret = WritePrivateProfileString("printers", "browsable", "no", SAMBA_CONF_PATH);
	ret = WritePrivateProfileString("printers", "printable", "yes", SAMBA_CONF_PATH);
	ret = WritePrivateProfileString("printers", "guest ok", "yes", SAMBA_CONF_PATH);
	ret = WritePrivateProfileString("global", "load printers", (enable_load?"yes":"no"), SAMBA_CONF_PATH);
	ret = WritePrivateProfileString("global", "print command", "/usr/bin/lpr -r -P%p %s", SAMBA_CONF_PATH);
	GetPrivateProfileString("global","security","user",security,sizeof(security), SAMBA_CONF_PATH);
    if ( ret ) 
    {
	    // this line is to insure the printer share folder is the same with the real spool path
	    ret = WritePrivateProfileString("printers", "path", SZ_PRINTER_SPOOLPATH, SAMBA_CONF_PATH);
	}

	return ret;
}

int set_printer_for_atalk(char *printer_name)
{
	int ret=0;
	FILE *fptr = fopen(SZ_PRINTER_PAPDPATH_DEFAULT, "wb");
    if ( fptr != NULL )
    {
//   	    ret = fprintf(fptr, SZ_PAPDCONF_TEMPLATE, printer_name);
	    ret = fprintf(fptr, " ");  // Closed temporarily to stop papd, not support macintosh yet.
        fclose(fptr);
    }
    
    return ((ret <= 0)?ERROR_WRITE_FILE:SUCCESS);
}

int set_printer_cap_name(char *printer_name)
{               
	FILE *fptr=NULL;
	int ret;
	char buf[BUF_SIZE] = "";
	int	type;
	
	// Get original comment first
	ret = Get_Printer_Comment(buf, BUF_SIZE - 1);
	
	if( ret != SUCCESS )
		return ret;
		
	fptr=fopen(SZ_PRINTER_CAPPATH_DEFAULT, "wb");
	if ( fptr == NULL )
		return ERROR_OPEN_FILE;

	type=Prn_Get_Type();
	if (type==PRN_PARALLEL)
	{
		ret = fprintf(fptr, SZ_PRINTCAP_TEMPLATE, printer_name, buf,
		                    SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE, 
	        	            SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE);
	}
	else
	{
		ret=fprintf(fptr, SZ_PRINTCAP_USB, printer_name, buf,
				SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE,
				SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE);
	}
	                    
        fclose(fptr);
	return ((ret <= 0)?ERROR_WRITE_FILE:SUCCESS);
}

int set_printer_cap_comment(char *printer_comment)
{
	FILE *fptr=NULL;
	int ret;
	char buf[BUF_SIZE] = "";
	int	type;
	
	// Get original name first
	ret = Get_Printer_Name(buf, BUF_SIZE - 1);
	
	if( ret != SUCCESS )
		return ret;
		
	fptr=fopen(SZ_PRINTER_CAPPATH_DEFAULT, "wb");
	if ( fptr == NULL )
		return ERROR_OPEN_FILE;
	
	//ret = fprintf(fptr, SZ_PRINTCAP_TEMPLATE, buf, printer_comment, SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE);
	type=Prn_Get_Type();
	if (type==PRN_PARALLEL)
	{
		ret = fprintf(fptr, SZ_PRINTCAP_TEMPLATE, buf, printer_comment,
		                    SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE, 
	        	            SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE);
	}
	else
	{
		int i=0;
		for(i=0;i<sizeof(buf);i++)
			buf[i]=tolower(buf[i]);
		ret=fprintf(fptr, SZ_PRINTCAP_USB, buf, printer_comment,
				SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE,
				SZ_PRINTER_SPOOLPATH, PRINTER_DEF_SPOOL_SIZE);
	}
	                    
       	fclose(fptr);                                                                                                                                                                     
       	
	return ((ret <= 0)?ERROR_WRITE_FILE:SUCCESS); 
}

BOOL Reset_Printer()
{
	if (Is_Support_Printer())
	{
		char	buf[256];

		Enable_Printer(0, FALSE);
		Set_Printer_Comment(SZ_PRINTER_COMMENT_DEFAULT, FALSE);
		Get_Server_Name(buf, 256);
		sprintf(buf, "%sPR", buf);
		Set_Printer_Name(buf, TRUE);
	}
        return 0;
}

int Add_Printer_Admin(){
	int ret;
	char cmd[CMD_LENGTH],per_user[CMD_LENGTH],users[CMD_LENGTH*2];
	FILE *fp;
	sprintf(cmd,"%s -g | grep \"Domain Users\" > %s", WBINFO_CMD,TMP_LOG);
	ret = system(cmd);
	fp = fopen(TMP_LOG,"r");
	while(fgets(per_user,sizeof(per_user),fp)!=NULL){
		strcat(users,"@\"");
		per_user[strcspn(per_user,"\n")] = '\0';
		strcat(users,per_user);
		strcat(users,"\",");
	}
	strcat(users,"@\"everyone\",guest");
	WritePrivateProfileString("Printers", "printer admin", users, SAMBA_CONF_PATH);
	fclose(fp);
	return ret;

}

int Get_Printer_Info(struct lp_info *lpinfo){
	FILE *fp=NULL;
	char cmd[CMD_LENGTH];
	char bus_buf[128], buf[128], bus[16], device[16];
	char vendor[128], product[128];
        char mfr[128], pro[128];
	char vendor_id[10], product_id[10];

	sprintf(cmd, "%s -vl", CMD_LSUSB);
	fp=popen(cmd, "r");
	
	while(fgets(bus_buf, sizeof(bus_buf), fp)){
		if(!strncmp(bus_buf, "Bus", 3)){
			strncpy(bus, bus_buf+4, 3);
			bus[3]='\0';
			strncpy(device, bus_buf+15, 3);
			while(fgets(buf, sizeof(buf), fp)){
				if(!strncmp(buf, "  idVendor", 10)){
					char *p = strstr(buf, "0x");
					strncpy(vendor_id, p+2, 4);
					vendor_id[4]='\0';
					strcpy(vendor, p+7);
				}
				else if(!strncmp(buf, "  idProduct", 11)){
					char *p = strstr(buf, "0x");
					strncpy(product_id, p+2, 4);
					product_id[4]='\0';
					strcpy(product, p+7);
				}
				else if(!strncmp(buf, "  iManufacturer", 14)){
					strcpy(mfr, buf+28);
				}
				else if(!strncmp(buf, "  iProduct", 9)){
					strcpy(pro, buf+28);
				}
				if(!strncmp(buf, "      bInterfaceClass", 20)){
					if(strstr(buf, "Printer")){
						lpinfo->bus = atoi(bus);
						lpinfo->device = atoi(device);			
						lpinfo->idvendor = atox(vendor_id);
						lpinfo->idproduct = atox(product_id);
						sprintf(lpinfo->idvendor_str, "%s", vendor);
						sprintf(lpinfo->idproduct_str, "%s", product);
						sprintf(lpinfo->manufacturer, "%s", mfr);
						sprintf(lpinfo->product, "%s", pro);
						sprintf(lpinfo->InterfaceClass, "Printer");
/*				printf("%3d\n", lpinfo->bus);
				printf("%3d\n", lpinfo->device);
				printf("%4x\n", lpinfo->idvendor);
				printf("%4x\n", lpinfo->idproduct);
				printf("%s\n", lpinfo->idvendor_str);
				printf("%s\n", lpinfo->idproduct_str);
				printf("%s\n", lpinfo->manufacturer);
				printf("%s\n", lpinfo->product);
				printf("%s\n", lpinfo->InterfaceClass);
*/						fclose(fp);
						return 1;
					}
				}
			}
			
			continue;
		}
	}
	fclose(fp);
	return 0;
}

int Get_Printer_Info_Ex(char *manufacturer, int manu_size, char *model, int model_size){
	Lp_info lpinfo;
	int ret=0;
	ret=Get_Printer_Info(&lpinfo);
	if(ret==0)
		return ret;
	if(ret>0){
		if(strlen(lpinfo.idvendor_str)<2){
			if(strlen(lpinfo.manufacturer)<2){
				sprintf(manufacturer, "Unkown (%5u)", lpinfo.idvendor);
			}
			else{
				if(strlen(lpinfo.manufacturer) > manu_size){
					strncpy(manufacturer, lpinfo.manufacturer, manu_size);
					manufacturer[manu_size]='\0';
				}
				else{
					strncpy(manufacturer, lpinfo.manufacturer, strlen(lpinfo.manufacturer));
					 manufacturer[strlen(lpinfo.manufacturer)-1]='\0';
				}
			}
		}
		else{
			if(strlen(lpinfo.idvendor_str) > manu_size){
				strncpy(manufacturer, lpinfo.idvendor_str, manu_size);
				manufacturer[manu_size]='\0';
			}
			else{
				strncpy(manufacturer, lpinfo.idvendor_str, strlen(lpinfo.idvendor_str));
				manufacturer[strlen(lpinfo.idvendor_str)-1]='\0';
			}
		}
		if(strlen(lpinfo.idproduct_str)<2){
			if(strlen(lpinfo.product)<2){
				sprintf(manufacturer, "Unkown (%5u)", lpinfo.idproduct);
			}
			else{
				if(strlen(lpinfo.product) > model_size){
					strncpy(model, lpinfo.product, model_size);
					model[model_size]='\0';
				}
				else{
					strncpy(model, lpinfo.product, strlen(lpinfo.product));
					model[strlen(lpinfo.product)-1]='\0';
				}
			}
		}
		else{
			if(strlen(lpinfo.idproduct_str) > model_size){
				strncpy(model, lpinfo.idproduct_str, model_size);
				model[model_size]='\0';
			}
			else{
				strncpy(model, lpinfo.idproduct_str, strlen(lpinfo.idproduct_str));
				model[strlen(lpinfo.idproduct_str)-1]='\0';
			}
		}
		return ret;
	}
	return ret;
}

void Write_Printer_Info_To_config(int plugged_type, char *manufacturer, char *model){
	if(plugged_type==PLUG_IN){
		WriteProfileString(SZ_PRINTER_SECTION, SZ_EXIST_FIELD, "1");
		WriteProfileString(SZ_PRINTER_SECTION, SZ_MANU_FIELD, manufacturer);
		WriteProfileString(SZ_PRINTER_SECTION, SZ_MODEL_FIELD, model);
	}
	else{
		WriteProfileString(SZ_PRINTER_SECTION, SZ_EXIST_FIELD, "0");
		WriteProfileString(SZ_PRINTER_SECTION, SZ_MANU_FIELD, " ");
		WriteProfileString(SZ_PRINTER_SECTION, SZ_MODEL_FIELD, " ");
	}
}

int Is_Printer_Exist(){
	char buf[8];
	GetProfileString(SZ_PRINTER_SECTION, SZ_EXIST_FIELD, "0", buf, sizeof(buf));
	return atoi(buf);
}

int Get_Printer_Status(){
	char buf[8];
	int ret=0;
	GetProfileString(SZ_PRINTER_SECTION, SZ_STATUS_FIELD, "-1", buf, sizeof(buf));
	ret=atoi(buf);
	if(ret==MSG_FINISH){
		sprintf(buf, "%d", MSG_READY);
		WriteProfileString(SZ_PRINTER_SECTION, SZ_STATUS_FIELD, buf);
		ret = MSG_READY;
	}
	return ret;
}

int Get_Printer_Info_From_Config(char *manufacturer, int manu_size, char *model, int model_size){
	char buf[32];
	int ret=0;
	GetProfileString(SZ_PRINTER_SECTION, SZ_EXIST_FIELD, "0", buf, sizeof(buf));
	ret = atoi(buf);
	if(ret==0){
		return ret;
	}
	else{
		GetProfileString(SZ_PRINTER_SECTION, SZ_MANU_FIELD, "", manufacturer, manu_size);
		GetProfileString(SZ_PRINTER_SECTION, SZ_MODEL_FIELD, "", model, model_size);
	}
	return ret;
}

void Refresh_Printer(int plugged_type, int refresh_now){
	char manufacture[128], model[128], buf[128];
	Get_Printer_Info_Ex(manufacture, sizeof(manufacture), model, sizeof(model));
	Write_Printer_Info_To_config(plugged_type, manufacture, model);
	if(plugged_type==PLUG_IN){
		set_load_printer(1);
		Set_Printer_Comment(SZ_PRINTER_COMMENT_DEFAULT, FALSE);
                Get_Server_Name(buf, sizeof(buf));
                sprintf(buf, "%sPR", buf);
                Set_Printer_Name(buf, FALSE);
		if(refresh_now)
			system("/bin/kill -HUP `/sbin/pidof smbd` 2>/dev/null 1>/dev/null");
		system("/sbin/daemon_mgr lpd start \"/usr/sbin/lpd\"");
		sprintf(buf, "%d", MSG_READY);
                WriteProfileString(SZ_PRINTER_SECTION, SZ_STATUS_FIELD, buf);
	}
	else{
		Conf_Remove_Section(SAMBA_CONF_PATH, "printers");
		Conf_Remove_Field(SAMBA_CONF_PATH,"global", "load printers");
		Conf_Remove_Field(SAMBA_CONF_PATH,"global", "printcap name"); 
		Conf_Remove_Field(SAMBA_CONF_PATH,"global", "show add printer wizard");
		Conf_Remove_Field(SAMBA_CONF_PATH,"global", "print command");
		system("/sbin/daemon_mgr lpd stop \"/usr/sbin/lpd\"");
		if(refresh_now)
			system("/etc/init.d/smb.sh restart 2>/dev/null 1>/dev/null");
	}
}

void CleanUpSpool(){
	char cmd[64];

	sprintf(cmd, "/bin/rm %s/cfA* 2>/dev/null 1>/dev/null", SZ_PRINTER_SPOOLPATH);
	system(cmd);
	sprintf(cmd, "/bin/rm %s/dfA* 2>/dev/null 1>/dev/null", SZ_PRINTER_SPOOLPATH);
	system(cmd);
}
