//**************************************************************************
//
//	Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		share.c
//
//	Abstract:
//		uLinux share management functions.
//
//	HISTORY:
//              03/5/14  Randy Chang   -- add a new function, "SMB_Reset_User_Group"            
//		02/10/15 Catherine Shen - add a new function, "Get_NAS_Shares",
//					to get NAS shares w/ or w/o snapshot shares
//		02/06/02 Catherine Shen - modify 'smb_get_userlist()',
//					'Create_NAS_Share()'
//		12/12/01 Catherine Shen - Replace some 'strcasecmp' by 'compare_case_string'
//		09/27/01 Louis Tsai  - fix Get_NAS_Share_List() bug
//		06/28/01 Louis Tsai  - free restriction of naming "global" service
//		04/02/01 Louis Tsai  - rewrite Create_NAS_Share() & Remove_NAS_Share()&
//                                     Get_NAS_Share_List()
//		12/18/00 Meiji Chang - splitted from usr_grp.c
//		09/01/00 Nick Ho - Created
//
//**************************************************************************
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include <dirent.h>
#include <sys/stat.h>

#include "NAS.h"
#include "uLinux.h"
#include "Util.h"
#include "cfg_ftp.h"
#include "config.h"
#include "naslvm.h"
#include "nas_lib_common.h"
#include "storage.h"

//#define DEBUG	1
// Strip off the white space characters appened on the end of a string.
// Return the original string ended with non-white-space characters
//extern char *stripe_white_space_tail(char *str);

//Catherine extern int compare_string(char* str1, char* str2);

//for Create_NAS_Share_From_Volume
int search_dir(char* dir, char* s_dir)
{
	DIR* dp;
	struct dirent *pdir;
	struct stat buf;
 	int ret = ERROR_NOT_FOUND;
	if((dp = opendir(dir)) == NULL )
		return ERROR_OPEN_FILE;
	chdir(dir);
	while((pdir = readdir(dp)) != NULL){
		lstat(pdir->d_name, &buf);
		if(S_ISDIR (buf.st_mode)){
			if(!strcmp(".", pdir->d_name) || !strcmp("..", pdir->d_name))
				continue;
			if(!strcmp(s_dir, pdir->d_name)){
				ret = ERROR_ALREADY_EXISTED;
				break;
			}
		}
	}
        //chdir("..");
	closedir(dp);
	return ret;
}

	
int make_dir(int index, char* s_dir)
{
	char buf[BUF_SIZE];
	struct stat dir;
	
	sprintf(buf,"%s%d", s_dir, index);		
	umask(0000);		
	while (!stat(buf, &dir)){
		index += 1;
		sprintf(buf,"%s%d", s_dir, index);
	}	
	mkdir(buf, S_IRWXU | S_IRWXG | S_IRWXO);
	return index;
}
// Skip the white space (SPACE & TAB) characters from the start of a string
// Return the string started from the first non-white-space character
void skip_white_space_start(char *str)
{
/* Kevin Liao 2001-07-11: Fix memory crash bug
	char *p = strdup(str);;
	while (*p == ' ' || *p == '\t') p++;
	strcpy(str,p);
	free(p);
Kevin Liao 2001-07-11: End of fix. */

	char *s = strdup(str), *p;


	p = s;
	while (*p == ' ' || *p == '\t')
		p++;

	strcpy(str, p);
	free(s);

}

void remove_successive_white_space(char *str)
{
	char *p ;
	int len,found_flag,i,j=0;
	len = strlen(str);
	p = (char*) malloc(len+1);
	for(i=0;i<len;i++) {
		if(*(str+i) == ' ' || *(str+i) == '\t') {
		 	if(found_flag == 0 ) {
				found_flag = 1;
				*(p+j) = ' ';
				j++;
			}
		}
		else {
			found_flag = 0;
			*(p+j) = *(str+i);
			j++;
		}
	}
	*(p+j) = '\0';
	strcpy(str,p);
	free(p);
}

void Remove_Blank_From_String(char* str)
{
	skip_white_space_start(str);
	stripe_white_space_tail(str);
	remove_successive_white_space(str);
}

BOOL Is_Reserved_Share(char *share_name)
{
	char *p;

	if(share_name == NULL)
		return FALSE;

	p = share_name;
	for(; *p && (*p == ' '); p++);
	// SHARE_USB_PREFIX = "USB Disk"
	if(!strncasecmp(p, "USB ", 4))
	{
		p = (char*)(p + 4);
		for(; *p && (*p == ' '); p++);
		if(*p == 0)	return FALSE;
		if(!strncasecmp(p, "Disk ", 5))
		{
			p = (char*)(p + 5);
			for(; *p && (*p == ' '); p++);
			if(*p == 0)	return FALSE;
			while(*p && (*p != ' '))
				if(!isdigit(*p++))
					return FALSE;
			for(; *p && (*p == ' '); p++);
			if(*p)
				return FALSE;
			else
				return TRUE;
		}
		else
			return FALSE;
	}
	else if(!strcasecmp(share_name, SHARE_ESATA))
		return TRUE;
	else
		return FALSE;
}

BOOL Check_Share_Name_Legal(char *share_name)
{
	int len;
	char *illegal_str = ILLEGAL_CHAR;
//	int share_name_len = strlen(share_name);
	// share name cannot longer than SHARE_NAME_LENGTH
	if (strlen(share_name) > SHARE_NAME_LENGTH-1) return FALSE;

	len = strlen(illegal_str);
	//check special case: share_name cannot only have '.'
	if((strlen(share_name) == 1)&&(share_name[0] =='.'))
		return FALSE;

	if(strcasecmp(share_name,"..") == 0)
		return FALSE;



	//before check illegal character,verify each word is chinese

	if(!Is_Legal_String(share_name))
		return FALSE;

	return TRUE;
}

//-------------------------------------------------------------------------------------
// Get All Share Name List, except the global session & do sorting action
// The line_post which in struct session_info doesn't correct value !                                         //
// Return: ERROR_OPEN_FILE     --> Open file fail
//         ERROR_OUT_OF_MEMORY --> You pass in smaller buffer
//         >0                  --> if list=NULL or list_cnt ,return required list count
//-------------------------------------------------------------------------------------


int Get_NAS_Share_List(SECTION_INFO *list, int list_cnt)
{
	int i,j,session_no;	//,curr_s=0;
    	char tmp_str[SHARE_NAME_LENGTH];
    	int ret;
	int browsable=0;

	// --  set all conf function case insensitive
	Conf_Case_Sensitive(0);
	if (Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)>=0)
		browsable=1;

    	if((list == NULL) || (list_cnt == 0)) {
    		i = 0;
		//free restriction of naming "global" service ,2001/6/28 Louis Tsai
    		if ( (ret = Conf_Enum_Section(SAMBA_CONF_PATH,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS) {
    			if( strcasecmp("global",tmp_str) != 0)
				i++;
			else {
//				if(Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)>=0)
				if (browsable)
					i++;
			}
    			while ( (ret = Conf_Enum_Section(NULL,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS ) {
    				if( strcasecmp("printers",tmp_str) == 0) continue;
    				if( strcasecmp("print$",tmp_str) == 0) continue;
    				if( strcasecmp("printer$",tmp_str) == 0) continue;
    				i++;
    			}
    			return i;
		}
    		else
    			return ret;
	}

	if ( (ret = Conf_Enum_Section(SAMBA_CONF_PATH,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS) {
    			i = 0;

    			if( strcasecmp("global",tmp_str) != 0) {
    				if ((i+1>list_cnt) || ((((list+i)->section_name))==NULL) ) {
    					//   fix bug that share cannot over 21
    					while ((ret = Conf_Enum_Section(NULL,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS );
    					return  ERROR_OUT_OF_MEMORY;
    				}
    				
    				strcpy((list+i)->section_name,tmp_str);
    				i++;
    			}
    			else {
//    				if(Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)>=0) {
				if (browsable)
				{
    					if ((i+1>list_cnt) || ((((list+i)->section_name))==NULL) ) {
    						while ((ret = Conf_Enum_Section(NULL,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS );
    						return  ERROR_OUT_OF_MEMORY;
    					}
    					strcpy((list+i)->section_name,tmp_str);
    					i++;
    				}
    			}
    			while ( (ret = Conf_Enum_Section(NULL,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS ) {
    				
    				if( strcasecmp("printers",tmp_str) == 0) continue;
    				if( strcasecmp("print$",tmp_str) == 0) continue;
    				if( strcasecmp("printer$",tmp_str) == 0) continue;
    				
    				if ((i+1>list_cnt) || (((list+i)->section_name)==NULL)) {
    					while ((ret = Conf_Enum_Section(NULL,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS );
    					return  ERROR_OUT_OF_MEMORY;
    				}
    				strcpy((list+i)->section_name,tmp_str);
    				i++;
    			}
	}
    	else
    		return ret;
    	session_no = i;
    	//- do sorting function -
    	for(i=0; i<session_no; i++) {
      		for(j=i+1;j<session_no;j++){
        		if ( strcmp( (list+i)->section_name,(list+j)->section_name ) >0){
          			strcpy(tmp_str,(list+i)->section_name );
          			strcpy( (list+i)->section_name,(list+j)->section_name );
          			strcpy((list+j)->section_name,tmp_str);
        		}
      		}
    	}

    return session_no;
}
//---------------------------------------------------------------------
// Create a new Share
// If not assign "read list" & "write list", it still write
// the char to file & it will add the "writable" & "browsable" two items                                      //
// Return:   ERROR_OUT_OF_MEMORY     --> Alloc memory fail
//	     ERROR_OPEN_FILE	     --> open smb.conf fail
//	     ERROR_LOCK_FILE	     --> smb.conf be locked
//           ERROR_ALREADY_EXISTED   --> The Share alreay exist
//           SUCCESS                 --> Successful
//--------------------------------------------------------------
int Create_NAS_Share(NAS_SHARE_INFO *share_ptr)
{
	char *buf_ptr, *valid_usr_list = NULL;
	char *read_list = NULL, *write_list = NULL;
	char tmp_name[100];
	char tmpbuf[USER_GROUP_NAME_LENGTH];
    	int  i,ret;
        FILE *fptr;
        char http_path[1024], cmd[1024], buf[1024];

      	// ignore space &tab at begin and tail
	skip_white_space_start(share_ptr->share_name);
	stripe_white_space_tail(share_ptr->share_name);
	remove_successive_white_space(share_ptr->share_name);

	if ( !Check_Share_Name_Legal(share_ptr->share_name))
		return ERROR_ILLEGAL_NAME;

	// --  set all conf function case insensitive
	Conf_Case_Sensitive(0);

	//special process for creating "global" share - Louis Tsai (2001/6/28)
	if(strcasecmp(share_ptr->share_name,"global") == 0) {
		//check global service exist or not
		if(Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)<0)
			goto ADD_SERVICE;
		else
			return ERROR_ALREADY_EXISTED;

	}
	// reject printer section name
	if( strcasecmp("printers",share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;
    	if( strcasecmp("print$"  ,share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;
    	if( strcasecmp("printer$",share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;

    	//--- Check the share name exist or not ? -------------
    	if ( (ret = Conf_Enum_Section(SAMBA_CONF_PATH,tmp_name,100)) == SUCCESS) {
    		if(compare_case_string(share_ptr->share_name,tmp_name) == 0)
    			return ERROR_ALREADY_EXISTED;
    		while ( (ret = Conf_Enum_Section(NULL,tmp_name,100)) == SUCCESS ) {
     			if(compare_case_string(share_ptr->share_name,tmp_name) == 0)
    				return ERROR_ALREADY_EXISTED;
    		}
	}
    	else
    		return ret;

ADD_SERVICE:
	Conf_Case_Sensitive(1);

    	//=============  Do append action  ===============
    	//----  append comment  ----
    	ret = Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"comment",share_ptr->comment);
    	if(ret != SUCCESS) return ret;

    	//----  append path  ----
      	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"path",share_ptr->path);
	if(ret != SUCCESS) return ret;

    	//----  append browsable ------------
    	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"browsable","yes");
	if(ret != SUCCESS) return ret;

    	//----  append public ------------
    	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"public","yes");
	if(ret != SUCCESS) return ret;

    	//----  append invalid users  ----
    	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"invalid users","");
	if(ret != SUCCESS) return ret;


    	//----  append read list  ----
    	if (share_ptr->read_list_cnt >0 ) {
    		if((buf_ptr = (char *) malloc((share_ptr->read_list_cnt)*USER_GROUP_NAME_LENGTH)) == NULL)
			return ERROR_OUT_OF_MEMORY;

       		strcpy(buf_ptr,"");
       		for(i=0; i<share_ptr->read_list_cnt; i++) {
			strcpy(tmpbuf, (char*)(share_ptr->read_list_ptr+i));
			insert_headntail_into_string(tmpbuf, USER_GROUP_NAME_LENGTH);
       			strcat(buf_ptr, tmpbuf);
          			strcat(buf_ptr,",");
		}
       		//--- overwrite the previously char ","  ---
       		*(buf_ptr+strlen(buf_ptr)-1) ='\0';
       		if((ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"read list",buf_ptr)) != SUCCESS)
		{
    			free(buf_ptr);
    			return ret;
    		}
    		// valid users list
		if((read_list = calloc(1, strlen(buf_ptr) + 1)) == NULL)
		{
    			free (buf_ptr);
			return ERROR_OUT_OF_MEMORY;
		}
		strcpy(read_list, buf_ptr);
    		free (buf_ptr);
    	}
    	else {
    		if((ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"read list","")) != SUCCESS)
			return ret;
    	}
    	//----  append write list  ----
    	if (share_ptr->write_list_cnt >0 ) {
    		if((buf_ptr = (char *) malloc((share_ptr->write_list_cnt)*USER_GROUP_NAME_LENGTH)) == NULL)
			return ERROR_OUT_OF_MEMORY;

       		strcpy(buf_ptr,"");
       		for(i=0; i<share_ptr->write_list_cnt; i++) {
			strcpy(tmpbuf, (char*)(share_ptr->write_list_ptr+i));
			insert_headntail_into_string(tmpbuf, USER_GROUP_NAME_LENGTH);
       			strcat(buf_ptr, tmpbuf);
         		strcat(buf_ptr,",");
       		}

       		//--- overwrite the previously char ","  ---
       		*(buf_ptr+strlen(buf_ptr)-1) = '\0';
    		if((ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"write list",buf_ptr)) != SUCCESS)
		{
    			free(buf_ptr);
    			return ret;
    		}

    		// valid users
		if((write_list =  calloc(1, strlen(buf_ptr) + 1)) == NULL)
		{
    			free (buf_ptr);
			return ERROR_OUT_OF_MEMORY;
		}
		strcpy(write_list, buf_ptr);
    		free (buf_ptr);
    	}
    	else {
    		if((ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"write list","")) != SUCCESS)
			return ret;
    	}

	//---	default valid users : root ---
	valid_usr_list = calloc(1, 5*sizeof(char));
	if (valid_usr_list == NULL)
		return ERROR_OUT_OF_MEMORY;
	strcpy(valid_usr_list, "root");

	// add "valid users" list as well
	if(read_list || write_list)
	{
		int len;
		char* ptr;
		len = (read_list ? strlen(read_list) : 0);
		len += (write_list ? strlen(write_list) : 0);
		if((ptr =  realloc(valid_usr_list, strlen(valid_usr_list) + len + 2)) == NULL)
		{
			free(valid_usr_list);
			return ERROR_OUT_OF_MEMORY;
		}
		valid_usr_list = ptr;
		if(read_list)
		{
			strcat(valid_usr_list, ",");
			strcat(valid_usr_list, read_list);
			free(read_list);
		}
		if(write_list)
		{
			strcat(valid_usr_list, ",");
			strcat(valid_usr_list, write_list);
			free(write_list);
		}
	}

	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"valid users",valid_usr_list);
	free(valid_usr_list);

        // add "inherit permissions" for acl, add by Nike 8/10/2004
	ret = Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name, "inherit permissions", "yes");

	//Update_Flash_Data(SAMBA_CONF_PATH);
        FTP_Create_Conf();
        // create http access link
        sprintf(http_path, "/home/httpd/%s", share_ptr->share_name);
        if ((fptr=fopen(http_path, "rb"))!=NULL)
        {
                ret=readlink(http_path, buf, 1024);
                if (ret==-1)
                {       // it is not a link
                }
                fclose(fptr);
                goto create_none_link;
        }
        sprintf(cmd, "/bin/ln -sf '%s' '%s'", share_ptr->path, http_path);
        system(cmd);
create_none_link:
    	return SUCCESS;
}

int Create_NAS_Share_Plus(NAS_SHARE_INFO *share_ptr)
{
//      char *buf_ptr, *valid_usr_list = NULL
        char str[128];
        char tmp_name[100];
        int  ret;
        FILE *fptr, *fp;
        char http_path[1024], cmd[1024], buf[1024];

  // ignore space &tab at begin and tail
        system(str);
        skip_white_space_start(share_ptr->share_name);
        stripe_white_space_tail(share_ptr->share_name);
        remove_successive_white_space(share_ptr->share_name);

        if ( !Check_Share_Name_Legal(share_ptr->share_name))
                return ERROR_ILLEGAL_NAME;

        // --  set all conf function case insensitive
        Conf_Case_Sensitive(0);

        //special process for creating "global" share - Louis Tsai (2001/6/28)
        if(strcasecmp(share_ptr->share_name,"global") == 0) {
                //check global service exist or not
                if(Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)<0)
                        goto ADD_SERVICE;
                else
                        return ERROR_ALREADY_EXISTED;

        }
        // reject printer section name
        if( strcasecmp("printers",share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;
        if( strcasecmp("print$"  ,share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;
        if( strcasecmp("printer$",share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;

        //--- Check the share name exist or not ? -------------
        if ( (ret = Conf_Enum_Section(SAMBA_CONF_PATH,tmp_name,100)) == SUCCESS) {
                if(compare_case_string(share_ptr->share_name,tmp_name) == 0)
                        return ERROR_ALREADY_EXISTED;
                while ( (ret = Conf_Enum_Section(NULL,tmp_name,100)) == SUCCESS ) {
                        if(compare_case_string(share_ptr->share_name,tmp_name) == 0)
                                return ERROR_ALREADY_EXISTED;
                }
        }
        else
                return ret;

ADD_SERVICE:
        fp = fopen(SAMBA_CONF_PATH, "a+");
        if(fp!= NULL)
        {
                sprintf(str, "[%s]\n", share_ptr->share_name);
                fputs(str, fp);
                sprintf(str, "comment = %s\n", share_ptr->comment);
                fputs(str, fp);
                sprintf(str, "path = %s\n", share_ptr->path);
                fputs(str, fp);
                sprintf(str, "browsable = yes\n");
                fputs(str, fp);
                sprintf(str, "invalid users =\n");
                fputs(str, fp);
                sprintf(str, "read list =\n");
                fputs(str, fp);
                sprintf(str, "write list = %s\n", share_ptr->share_name);
                fputs(str, fp);
                sprintf(str, "valid users = root,%s\n", share_ptr->share_name);
                fputs(str, fp);
                sprintf(str, "inherit permissions = yes\n");
                fputs(str, fp);
                fclose(fp);
        }
//        Update_Flash_Data(SAMBA_CONF_PATH);
        // create http access link
        sprintf(http_path, "/home/httpd/%s", share_ptr->share_name);
        if ((fptr=fopen(http_path, "rb"))!=NULL)
        {
                ret=readlink(http_path, buf, 1024);
                if (ret==-1)
                {       // it is not a link
                }
                fclose(fptr);
                goto create_none_link;
        }
        sprintf(cmd, "/bin/ln -sf '%s' '%s'", share_ptr->path, http_path);
        system(cmd);
create_none_link:
        return SUCCESS;
}

//------------------------------------------------------------
// Delete one share
//
// Return:
//	      ERROR_OPEN_FILE	   --> open smb.conf fail
//	      ERROR_LOCK_FILE	   --> smb.conf be locked
//            ERROR_NOT_FOUND      --> The Share not exist
//            SUCCESS              --> Successful
//------------------------------------------------------------

int Remove_NAS_Share(char *share_name)
{
    	int  ret;
	char cmd[1024];

	Conf_Case_Sensitive(0);

	//special process for creating "global" share - Louis Tsai (2001/6/28)
	if(strcasecmp(share_name,"global") == 0) {
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","comment"))<0) return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","path"))<0) return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","browsable"))<0) return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","invalid users"))<0) return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","read list"))<0) return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","write list"))<0) return ret;

		goto LEAVE;
	}
    	ret = Conf_Remove_Section(SAMBA_CONF_PATH,share_name);

LEAVE:
    	Remove_Connection_For_NAS_Share(share_name);
    	//Update_Flash_Data(SAMBA_CONF_PATH);
        FTP_Create_Conf();
        // remove http access link
        sprintf(cmd, "/bin/rm \"/home/httpd/%s\" 1>/dev/null 2>/dev/null", share_name);//Shone modified to solve name-with-space problem 2005,01,19
        system(cmd);
    	return ret;
}



/*
 * 2000/10/17: check the Directory is exist or not
 * return 0: exist, 1 not exist -xxx, error
 */
//int Dir_Exist(char *dir_name)
BOOL Is_Existing_Directory(char *dir_name)
{
	DIR *dp;

	dp = opendir(dir_name);
//	if (dp == NULL && errno == ENOENT) return FALSE;   // directory not exist
//	else
	if (dp != NULL) {
		closedir(dp);
		return TRUE;
	}
	return FALSE;
}


//------------------------------------------------------------------------//
//get active connection for the NAS share
//
//Reurn	:
//	NNN>=0 			, number of active user
//	ERROR_EXEC_FAIL		, excute system command error
//	ERROR_BUFFER_TOO_SMALL	, input buffer too small
//
//------------------------------------------------------------------------//

int Get_Active_Connection_For_Nas_Share(char *share_name, char (*user_list)[USER_NAME_LENGTH],int list_cnt)
{
	int ret;
	int tmp_cnt;
	if((ret = Get_Samba_Status(NULL,0))>0) {
		SAMBA_STATUS_INFO info_list[ret+20];// +20 to avoid active connection increase suddently
		int i=0,j=0,conn=0,found_flag = 0;
		if((conn = Get_Samba_Status(info_list,ret+20))>0) {
			tmp_cnt = 0;
			for(i=0;i<conn;i++) {
				found_flag = 0;
				if (list_cnt !=0 && user_list != NULL) {
//Catherine					if (strcasecmp(info_list[i].service,share_name) == 0 ) {
					if (compare_case_string(info_list[i].service,share_name) == 0 ) {
						for(j=0;j<tmp_cnt;j++)
//Catherine							if(strcasecmp(user_list[j],info_list[i].user_name) == 0 ) {
							if(compare_case_string(user_list[j],info_list[i].user_name) == 0 ) {
								found_flag = 1;
								break;
							}
						if ( 0 == found_flag ) {
							strcpy(user_list[tmp_cnt],info_list[i].user_name);
							tmp_cnt++;
							if(tmp_cnt>list_cnt)
								return ERROR_BUFFER_TOO_SMALL;
						}
					}
				}
				else {
//Catherine					if (strcasecmp(info_list[i].service,share_name) == 0 )
					if (compare_case_string(info_list[i].service,share_name) == 0 )
						tmp_cnt++;

				}
			}
		}
		else
			return conn;
	}
	else
		return ret;

	return tmp_cnt;
}

int Get_NAS_Code_Page()
{
	char code_str[5];
	int ret = Conf_Get_Field(SAMBA_CONF_PATH,"global","client code page",code_str,5);

	if(ret<0) return ret;
	return atoi(code_str);
}
#if 0
void erase_char(char *origin)
{
        char *start, *ptr;

        if (strlen(origin)==0)
                return;
        start=ptr=origin;
        ptr=origin+strlen(origin);
        while (*start==' ' || *start=='	' || *start==0x0d || *start==0x0a)
                start++;
        ptr--;
        while (*ptr==' ' || *ptr=='	' || *ptr==0x0d || *ptr==0x0a)
                ptr--;
        *(ptr+1)=0x0;
        ptr=origin;
        while (*start)
                *ptr++=*start++;
        *ptr=0x0;
}
#endif

char *get_token_value(char *inputstr, char *token)
{
	char *value;

	value=strtok(inputstr, token);
	if (value==NULL)
		return NULL;
	erase_char(value);
	value=strtok(NULL,"\t\n");
	if (value==NULL)
		return NULL;
	erase_char(value);
	return value;
}

USER_GROUP_LIST *smb_get_userlist(char *str)
{
	USER_GROUP_LIST	*start=NULL, *userlist=NULL;
	char 		*value;
	int 		count=0;

	value=strtok(str, ",\n");
	while (value!=NULL)
	{
		if (count==0)
		{
			if ((userlist=calloc(1, sizeof(USER_GROUP_LIST)))==NULL)
				return NULL;
			start=userlist;
			count++;
		}
		else
		{
			if ((userlist->next=calloc(1, sizeof(USER_GROUP_LIST)))==NULL)
				return start;
			userlist=(USER_GROUP_LIST *)userlist->next;
		}
		trim_chars_from_string(value, '\"'); // Catherine 2002/02/06
		strcpy(userlist->name, value);
		value=strtok(NULL, ",\n");
	}
	return start;
}

SMB_SHARE_INFO *Remove_User_From_Section(SMB_SHARE_INFO *smb_share_info, char *username){

        SMB_SHARE_INFO  *pshare_info;
//      USER_GROUP_LIST       *userlist;

        if (smb_share_info!=NULL)
        {
                pshare_info=smb_share_info;
                while (pshare_info!=NULL)
                {
                        pshare_info->invalid = Remove_User_From_List((USER_GROUP_LIST *)pshare_info->invalid, username);
                        pshare_info->read = Remove_User_From_List((USER_GROUP_LIST *)pshare_info->read, username);

                        pshare_info->write = Remove_User_From_List((USER_GROUP_LIST *)pshare_info->write, username);
                        pshare_info->valid = Remove_User_From_List((USER_GROUP_LIST *)pshare_info->valid, username);
                        pshare_info=(SMB_SHARE_INFO *)pshare_info->next;
                }
        }
        return smb_share_info;
}

SMB_SHARE_INFO *SMB_Get_Share_Info()
{
	FILE 		*fptr;
	char		line[LINE], *ptr, tmp[LINE], *value;
	SMB_SHARE_INFO	*start=NULL, *smb=NULL;
	int		count=0;
       
	if ((fptr=fopen("/etc/config/smb.conf", "rb"))==NULL)
		return NULL;
            		
	fgets(line, LINE, fptr);
	erase_char(line);

	while (!feof(fptr))
	{
		if (line[0]=='[' && line[strlen(line)-1]==']')	/* section start */
		{
			while (1)
			{
				if (line[0]=='[' && line[strlen(line)-1]==']')
				{
					ptr=line+1;
					line[strlen(line)-1]=0x0;
					strcpy(tmp, ptr);
				}
				else
                                if (!strncmp(line, "comment", 7)){
                                        if (count==0)
                                        {
                                                if ((smb=calloc(1, sizeof(SMB_SHARE_INFO)))==NULL) {
                                                        fclose(fptr);
                                                        return NULL;
                                                }
                                                start=smb;
                                                count++;
                                        }
                                        else
                                        {
                                                 if ((smb->next=calloc(1, sizeof(SMB_SHARE_INFO)))==NULL) {
                                                         fclose(fptr);
                                                         return start;
                                                 }
                                                 smb=(SMB_SHARE_INFO *)smb->next;
                                        }
                                        strcpy(smb->sharename, tmp);
                                        value=get_token_value(line, "=");
                                        if (value!=NULL){
                                                strcpy(smb->comment, value);
//                                              printf("%s\n", smb->comment);
                                        }
                                }
				else
				if (!strncmp(line, "path", 4))
				{
/*					if (count==0)
					{
						if ((smb=calloc(1, sizeof(SMB_SHARE_INFO)))==NULL) {
							fclose(fptr);
							return NULL;
						}
						start=smb;
						count++;
					}
					else
					{
						if ((smb->next=calloc(1, sizeof(SMB_SHARE_INFO)))==NULL) {
							fclose(fptr);
							return start;
						}
						smb=(SMB_SHARE_INFO *)smb->next;
					}
					strcpy(smb->sharename, tmp);
*/					value=get_token_value(line, "=");
					strcpy(smb->path, value);
				}
				else
				if (!strncmp(line, "invalid users", 13))
				{       
					value=get_token_value(line, "=");					
					if (value!=NULL)
						smb->invalid=smb_get_userlist(value);
				}
				else
				if (!strncmp(line, "read list", 9))
				{
					value=get_token_value(line, "=");
					if (value!=NULL)
						smb->read=smb_get_userlist(value);
				}
				else
				if (!strncmp(line, "write list", 10))
				{
					value=get_token_value(line, "=");
					if (value!=NULL)
						smb->write=smb_get_userlist(value);
				}
				else
                                if (!strncmp(line, "valid users", 11)){
                                        value=get_token_value(line, "=");
                                        if (value!=NULL)
                                                smb->valid=smb_get_userlist(value);
                                }
				fgets(line, LINE, fptr);
				erase_char(line);
				if ((line[0]=='[' && line[strlen(line)-1]==']') || feof(fptr))
				{
					break;
				}
			}
		}
		else{      
			fgets(line, LINE, fptr);
			erase_char(line);
		      }	
	}
	fclose(fptr);
	return start;
}

void SMB_Free_Share_Info(SMB_SHARE_INFO *smb_share_info)
{
        SMB_SHARE_INFO  *tmp_share_info;
        USER_GROUP_LIST       *userlist, *tmp_userlist;

        if (smb_share_info!=NULL)
        {
                while (smb_share_info!=NULL)
                {
			userlist=smb_share_info->invalid;
                        while (userlist!=NULL)
			{
				tmp_userlist=userlist;
                                userlist=(USER_GROUP_LIST *)userlist->next;
				free(tmp_userlist);
			}
                        userlist=smb_share_info->read;
                        while (userlist!=NULL)
			{
				tmp_userlist=userlist;
                                userlist=(USER_GROUP_LIST *)userlist->next;
				free(tmp_userlist);
			}
                        userlist=smb_share_info->write;
                        while (userlist!=NULL)
			{
				tmp_userlist=userlist;
                                userlist=(USER_GROUP_LIST *)userlist->next;
				free(tmp_userlist);
			}

			tmp_share_info=smb_share_info;
                        smb_share_info=(SMB_SHARE_INFO *)smb_share_info->next;
			free(tmp_share_info);
                }
        }
}

USER_GROUP_LIST* Remove_User_From_List(USER_GROUP_LIST *pnode, char *username){
        USER_GROUP_LIST *phead;
        USER_GROUP_LIST *plist;
        USER_GROUP_LIST *pprev;
        USER_GROUP_LIST *pfree;

        phead = pnode;
        plist = pnode;
        pprev = plist;

//      printf("Remove_User_From_List\n");
        while(plist){
                if(!strcmp(plist->name, username)){
                        if(plist==phead){
                                phead=(USER_GROUP_LIST *)plist->next;
                                pfree = plist;
                                plist=phead;
                                pprev=phead;
                                free(pfree);// I don't why I can't free it, debug later
                        }
                        else{
                                pprev->next = plist->next;
                                pfree = plist;
                                plist = (USER_GROUP_LIST *)pprev->next;
                                free(pfree);
                        }
//              printf("free: %p\n", pfree);
                }
                else{
//                      printf("%s ", plist->name);
                        pprev = plist;
                        plist = (USER_GROUP_LIST *)plist->next;
                }
        }

//      printf("\n");
        return phead;

}
//modify the smb.conf the follow field "invalid users" "read list" "write list"
//write the SMB_SHARE_INFO list to the smb.conf file
void smb_write_info(SMB_SHARE_INFO *pheadsmb)
{
        FILE 		*fptr , *fpdst;
	char		line[LINE], *ptr, tmp[LINE], ori_line[LINE] ;//, *value , *ptok
	SMB_SHARE_INFO	 *psmb=NULL ,*validsmb=NULL; //*start=NULL,
	int		linecount=0;
	char            SMB_CONF_NAME[] =  "/etc/config/smb.conf";
	char            TMP_SMB_CONF_NAME[] = "/etc/config/smb_tmp.conf";
     
	if ((fptr=fopen(SMB_CONF_NAME, "a+"))==NULL)
		return ;
		
	if ((fpdst=fopen(TMP_SMB_CONF_NAME, "w+"))==NULL)
		return ;
				
	fseek(fptr ,0 , SEEK_SET); 
	fseek(fpdst ,0 , SEEK_SET);
	
	fgets(ori_line, LINE, fptr);
	strcpy(line , ori_line);
	linecount++;
	erase_char(line);	
	while (!feof(fptr))
	{
		if (line[0]=='[' && line[strlen(line)-1]==']')	// section start 
		{
			while (1)
			{
				if (line[0]=='[' && line[strlen(line)-1]==']')
				{					
					ptr=line+1;
					line[strlen(line)-1]=0x0; //remove [ ]
					strcpy(tmp, ptr); //look for respond SMB_SHARE_INFO										
					validsmb = NULL;
					psmb = pheadsmb; 
					while (psmb){
					  if (!strncmp(psmb->sharename , tmp , strlen(tmp))) {
					      validsmb = psmb; break;					      
					  }					  
					  psmb=(SMB_SHARE_INFO*)psmb->next;
				        }
				        fprintf(fpdst,"%s" ,ori_line);				        
				}
				else if (validsmb) 
				{ 	
				 USER_GROUP_LIST* pUsr = 0;  			
				 int busers = 0;
				if (!strncmp(line, "invalid users", 13)){				 
				    pUsr = psmb->invalid; busers =1;
				    fprintf(fpdst,"%s" ,"invalid users = ");				      
				    if (!pUsr) fprintf(fpdst,"\n");
				}		 
				else if (!strncmp(line, "read list", 9)){				 
				    pUsr = psmb->read;	busers =1;		
				    fprintf(fpdst,"%s" ,"read list = ");				     
				    if (!pUsr) fprintf(fpdst,"\n");
				}		
				else if (!strncmp(line, "write list", 10)){
                                    pUsr = psmb->write; busers =1;
                                    fprintf(fpdst,"%s" ,"write list = ");                                                                          
                                    if (!pUsr) fprintf(fpdst,"\n");
                                 }
				else if (!strncmp(line, "valid users", 11)){
                                         pUsr = psmb->valid; busers =1;
                                         fprintf(fpdst,"%s" ,"valid users = ");

                                         if (!pUsr) fprintf(fpdst,"\n");
                                }
				if (pUsr) {					     
				    USER_GROUP_LIST* pIndex = pUsr;				 	
				    while (pIndex){
				     if ('@' == pIndex->name[0]) 	
				       fprintf(fpdst,"@\"%s\"",&(pIndex->name[1]));
				     else
				       fprintf(fpdst,"%s",pIndex->name); 				      
				     pIndex = (USER_GROUP_LIST*)pIndex->next;
				      if (pIndex) fprintf(fpdst,",");
				      //fwrite(psmb->invalid->name, sizeof(char),strlen(psmb->invalid->name),fptr);
				     }
				     fprintf(fpdst,"\n");
				 }				 				  				 
				 else if (!busers)
				   fprintf(fpdst,"%s" ,ori_line);
				}
				else 
				  fprintf(fpdst,"%s" ,ori_line);
				fgets(ori_line, LINE, fptr);
				linecount++;				
	                        strcpy(line , ori_line);
				erase_char(line);
				if ((line[0]=='[' && line[strlen(line)-1]==']') || feof(fptr))				
				  break;				
			}
		}
		else{
		 fprintf(fpdst,"%s" ,ori_line);
		// fwrite(ori_line, sizeof(char),strlen(ori_line)+1,fpdst);   
		 linecount++;
		 fgets(ori_line, LINE, fptr);
	         strcpy(line , ori_line);
		}
	}
	fclose(fptr);	
	fclose(fpdst);
	link(SMB_CONF_NAME , "/etc/config/smb_bak.conf" ); 
	unlink(SMB_CONF_NAME); 
	link(TMP_SMB_CONF_NAME , SMB_CONF_NAME); 	 		
	unlink(TMP_SMB_CONF_NAME); 
}

void Remove_User_From_Section_Ex(char *username){
        SMB_SHARE_INFO  *smb_share_info;
        smb_share_info=SMB_Get_Share_Info();

        smb_share_info = Remove_User_From_Section(smb_share_info, username);
        smb_write_info(smb_share_info);
        SMB_Free_Share_Info(smb_share_info);
        return;
}

int is_in_string(char* pszbuf , char* sstr)
{      
    char buffer[1024];
    char* tok = NULL;
    int valid = 0;    
     
    strcpy(buffer, sstr);
    tok = strtok(buffer, " ");	                                        
    while (tok)
    {     
       if (strstr(pszbuf ,tok)){
         valid = 1;
         break;
       }
       tok = strtok(NULL , " ");       
    }
    return valid;   
}

USER_GROUP_LIST* unlink_nodefault_node(USER_GROUP_LIST* pnode)
{
    USER_GROUP_LIST* plist; 
    USER_GROUP_LIST* pprev;      
    USER_GROUP_LIST* phead;       
    char szKeepWord[] = "guest everyone administrator administrators";
                        //the default user and group name for samba
    phead = pnode;
    plist = pnode;
    pprev = plist;    
    
    while(plist){       	
        if (!is_in_string(plist->name , szKeepWord)) {                                   
          if (pprev == plist) {
            pprev = (USER_GROUP_LIST*) pprev->next;
            free(plist);
            plist = pprev;
            phead = plist;
          }
          else {          
            pprev->next = plist->next;
            free(plist);          
            plist = (USER_GROUP_LIST*)pprev->next;          
          }          
        }
        else {
         pprev = plist;              
         plist = (USER_GROUP_LIST*)plist->next;         
        }
    }    	
    return phead;
}

//-------------------------------------------------------------------//
//abstraction: 5/13/03
// remove the users except the "administrator" and "guest"  
// remove the group except the "administrators" and "everyone"
void SMB_Reset_User_Group()
{           
     SMB_SHARE_INFO *psmb , *pheadsmb;    
     
     pheadsmb  = SMB_Get_Share_Info();
     psmb = pheadsmb;
     
     while (psmb)
     {        	           	       
       psmb->invalid = unlink_nodefault_node(psmb->invalid);
       psmb->read = unlink_nodefault_node(psmb->read);
       psmb->write = unlink_nodefault_node(psmb->write);       
       psmb=(SMB_SHARE_INFO*)psmb->next;
     }   
     
     if (pheadsmb)
     {
       smb_write_info(pheadsmb);  
       SMB_Free_Share_Info(pheadsmb);                               
     }
} 


//-------------------------------------------------------------------//
//create NAS share from volume
//Return:
//	SUCCESS                  ,success
//	ERROR_FAIL               ,
//      ERROR_ALREADY_EXISTED    ,this share or path is exited
//	Other                    ,according to Create_NAS_Share report 
//--------------------------------------------------------------------//
int Create_NAS_Share_From_Volume(char* share_name, int volume, char* path, char* comment)
{
	VOLUME_CONF vol;
	NAS_SHARE_INFO share;
//	SECTION_INFO list[MAX_SHARE_NUMBER];
	SECTION_INFO *list=NULL;
	char str[BUF_SIZE], buf[BUF_SIZE];
    	int ret = SUCCESS, exist = 0, i, num;

	memset(&vol, 0, sizeof(vol));
	memset(&share, 0, sizeof(share));
	memset(&list, 0, sizeof(list));

	if(!compare_string(path, ""))	return ERROR_FAIL;
/* Catherine
        num = Get_NAS_Share_List(NULL, 0);
	Get_NAS_Share_List(list, num);
*/
	num = Get_NAS_Share_List_Ex(&list);
	
	for(i = 0;i < num; i++){
		if(!strcmp(share_name, list[i].section_name)) {					
			free(list);			
			return ERROR_ALREADY_EXISTED;
		}
	}
	free(list);

	if((ret = Get_One_Volume_Conf(volume, &vol)) == SUCCESS){
		Get_MP_String(vol.drive_no_list[0], DATA_PART, str);
				
		if(path == NULL){					
			if((search_dir(str, share_name)) == ERROR_ALREADY_EXISTED){	
					
				i = make_dir(1, share_name);				
				exist = 1;
 				strcat(str,"/");
				strcat(str, share_name);
				sprintf(buf,"%s%d",str,i);								
				if (chmod(buf, S_IRWXU | S_IRWXG | S_IRWXO) != 0)					
					return -1;				
														
			}
			else{
				strcat(str,"/");
				strcat(str,share_name);
				umask(0000);				
				if (mkdir(str, S_IRWXU | S_IRWXG | S_IRWXO) != 0 )
					return -1;
			}
		}
		else{
			if((ret = search_dir(str, path)) == ERROR_NOT_FOUND){
				strcat(str, "/");
				strcat(str, path);
				umask(0000);
				if (mkdir(str, S_IRWXU | S_IRWXG | S_IRWXO) != 0 )
					return -1;
			}
			else if(ret == ERROR_ALREADY_EXISTED){
				strcat(str, "/");
				strcat(str, path);
			}
		}
		if(exist == 1)
			strcpy(share.path, buf);
		else
			strcpy(share.path, str);
		strcpy(share.share_name, share_name);
		strcpy(share.comment, comment);		
		if((ret = Create_NAS_Share(&share)) != SUCCESS)
			return ret;
			
	}
     	return ret;
}

//--------- dynamic allocation ----------------
//-------------------------------------------------------------------------------------
// Get All Share Name List, including the global session & do sorting action
// Return: ERROR_OPEN_FILE     --> Open file fail
//         >0                  --> if list=NULL or list_cnt ,return required list count
//-------------------------------------------------------------------------------------

int Get_NAS_Share_List_Ex(SECTION_INFO **list_ptr)
{
	int i,j,section_no;	//,curr_s=0;
    	char tmp_str[SHARE_NAME_LENGTH];
    	int ret;
    	SECTION_INFO *list=NULL;
	int	browsable=0;

	// --  set all conf function case insensitive
	Conf_Case_Sensitive(0);
	if (Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)>=0)
		browsable=1;
#ifdef DEBUG
	printf("Get_NAS_Share_List_Ex step 1\n");
#endif

	if ((ret = Conf_Enum_Section(SAMBA_CONF_PATH,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS) {
    			i = 0;

#ifdef DEBUG
			printf("Get_NAS_Share_List_Ex step 2: tmp_str=%s\n",tmp_str);
#endif
    			if( strcasecmp("global",tmp_str) != 0) {
    				list = (SECTION_INFO*)calloc(1, sizeof(SECTION_INFO));
#ifdef DEBUG
				printf("Get_NAS_Share_List_Ex step 2.1\n");
#endif
    				strcpy(list->section_name,tmp_str);
    				i++;
    			}
    			else {
//    				if(Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)>=0) {
				if (browsable)
				{
    					list = (SECTION_INFO*)calloc(1, sizeof(SECTION_INFO));
#ifdef DEBUG
					printf("Get_NAS_Share_List_Ex step 2.2\n");
#endif
    					strcpy(list->section_name,tmp_str);
    					i++;
    				}
    			}
#ifdef DEBUG
			printf("Get_NAS_Share_List_Ex step 3\n");
#endif
    			while ( (ret = Conf_Enum_Section(NULL,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS ) {
#ifdef DEBUG
				printf("Get_NAS_Share_List_Ex step 4\n");
#endif
    				
    				if( strcasecmp("printers",tmp_str) == 0) continue;
    				if( strcasecmp("print$",tmp_str) == 0) continue;
    				if( strcasecmp("printer$",tmp_str) == 0) continue;

#ifdef DEBUG
				printf("Get_NAS_Share_List_Ex step 5\n");
#endif
    				list = (SECTION_INFO*)realloc((void*)list,
					(i+1)*sizeof(SECTION_INFO));
    				strcpy(list[i].section_name,tmp_str);
    				i++;
    			}
	}
    	else {
#ifdef DEBUG
		printf("Get_NAS_Share_List_Ex step 4.5\n");
#endif
    		return ret;
	}

#ifdef DEBUG
	printf("Get_NAS_Share_List_Ex step 6: section no = %d\n", i);
#endif
    	section_no = i;
    	//- do sorting function -
    	for(i=0; i<section_no; i++) {
      		for(j=i+1;j<section_no;j++){
			if ( strcmp( (list+i)->section_name,(list+j)->section_name ) >0){
          				strcpy(tmp_str,(list+i)->section_name );
          				strcpy( (list+i)->section_name,(list+j)->section_name );
          				strcpy((list+j)->section_name,tmp_str);
        			}
      		}
    	}
	
#ifdef DEBUG
	printf("Get_NAS_Share_List_Ex step 7\n");
#endif
	*list_ptr = list;
	return section_no;
}

//-------------------------------------------------------------------------------------
// Get All Share Name List, including the global session & do sorting action
//	including/excluding snapshot shares by the input flag
// Return: ERROR_OPEN_FILE     --> Open file fail
//         >0                  --> if list=NULL or list_cnt ,return required list count
//-------------------------------------------------------------------------------------
int Get_NAS_Shares(SECTION_INFO **list_ptr, BOOL snapshot_included)
{
	int i,j,section_no;
    	char tmp_str[SHARE_NAME_LENGTH];
    	int ret;
    	SECTION_INFO *list=NULL;
	int	browsable=0;

	// --  set all conf function case insensitive
	Conf_Case_Sensitive(0);
	if (Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)>=0)
		browsable=1;

	if ((ret = Conf_Enum_Section(SAMBA_CONF_PATH,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS) {
    			i = 0;

    			if( strcasecmp("global",tmp_str) != 0) {
    				list = (SECTION_INFO*)calloc(1, sizeof(SECTION_INFO));
    				strcpy(list->section_name,tmp_str);
    				i++;
    			}
    			else {
				if (browsable)
				{
    					list = (SECTION_INFO*)calloc(1, sizeof(SECTION_INFO));
    					strcpy(list->section_name,tmp_str);
    					i++;
    				}
    			}
    			while ( (ret = Conf_Enum_Section(NULL,tmp_str,SHARE_NAME_LENGTH)) == SUCCESS ) {
    				if( strcasecmp("printers",tmp_str) == 0) continue;
    				if( strcasecmp("print$",tmp_str) == 0) continue;
    				if( strcasecmp("printer$",tmp_str) == 0) continue;

				if (!snapshot_included
					&& (strncmp(tmp_str, SNAPSHOT_NAME_PREFIX, strlen(SNAPSHOT_NAME_PREFIX))==0))
					continue;

    				list = (SECTION_INFO*)realloc((void*)list,
					(i+1)*sizeof(SECTION_INFO));
    				strcpy(list[i].section_name,tmp_str);
    				i++;
    			}
	}
    	else
    		return ret;

    	section_no = i;
    	//- do sorting function -
    	for(i=0; i<section_no; i++) {
      		for(j=i+1;j<section_no;j++){
			if ( strcmp( (list+i)->section_name,(list+j)->section_name ) >0){
          				strcpy(tmp_str,(list+i)->section_name );
          				strcpy( (list+i)->section_name,(list+j)->section_name );
          				strcpy((list+j)->section_name,tmp_str);
        			}
      		}
    	}
	
	*list_ptr = list;
	return section_no;
}

// Catherine 2003/01/25
//-------------------------------------------------------------------------------------
// Remove the invalid (route no longer exists) shares in smb.conf
// Return: ERROR_OPEN_FILE	--> Open file fail
//         SUCCESS		--> Success
//-------------------------------------------------------------------------------------
int Clear_Invalid_NAS_Shares(void)
{
	int cnt, ret, i;
	SECTION_INFO *listptr = NULL;

	cnt = Get_NAS_Shares(&listptr, TRUE);
	if (cnt<0) return cnt;

	for (i=0; i<cnt; i++) {
		NAS_SHARE_INFO shareinfo;

		memset(&shareinfo, 0, sizeof(NAS_SHARE_INFO));
		strcpy(shareinfo.share_name, listptr[i].section_name);

		ret = Get_NAS_Share_Info_Ex(&shareinfo); 
		if (ret<0) continue;

		if (!Is_Existing_Directory(shareinfo.path)) {
			ret = Remove_NAS_Share(shareinfo.share_name);
			if (ret<0) break;
		}
	}
	if (listptr!=NULL) free(listptr);
	FTP_Create_Conf();

	return ret;
}

// Catherine 2003/03/05
int mkdir_recursively(char* dirpath)
{
	char newpath[SHARE_PATH_LENGTH];
	char *ptr, *pp;
	int ret = SUCCESS;

	ptr = dirpath + strlen("/share/");
	pp = strchr(ptr, '/');
	ptr = pp+1;

	while ((pp=strchr(ptr, '/'))!=NULL) {
		strncpy(newpath, dirpath, (int)(pp-dirpath));
		newpath[(int)(pp-dirpath)]=0x0;

		umask(0000);
		if (mkdir(newpath, S_IRWXU | S_IRWXG | S_IRWXO)!=0)
		{
			ret = ERROR_EXEC_FAIL;
			break;
		}
		ptr = pp+1;
	}
	umask(0000);
	if (mkdir(dirpath, S_IRWXU | S_IRWXG | S_IRWXO)!=0)
	{
		ret = ERROR_EXEC_FAIL;
	}
	return ret;

}

//-------------------------------------------------------------------------------------
// Create corresponding directories for the invalid (route no longer exists) shares in smb.conf
// Return: ERROR_OPEN_FILE	--> Open file fail
//         SUCCESS		--> Success
//-------------------------------------------------------------------------------------
int Repair_Invalid_NAS_Shares(void)
{
	int cnt, ret, i;
	SECTION_INFO *listptr = NULL;

	cnt = Get_NAS_Shares(&listptr, TRUE);
	if (cnt<0) return cnt;

	for (i=0; i<cnt; i++) {
		NAS_SHARE_INFO shareinfo;

		memset(&shareinfo, 0, sizeof(NAS_SHARE_INFO));
		strcpy(shareinfo.share_name, listptr[i].section_name);

		ret = Get_NAS_Share_Info_Ex(&shareinfo); 
		if (ret<0) continue;

		if (!Is_Existing_Directory(shareinfo.path)) {
			char devpath[HD_DEVICE_NAME_LENGTH], mp[HD_DEVICE_NAME_LENGTH];
			char *ptr, *pp;

			ptr = shareinfo.path + strlen("/share/");
			if ((pp = strchr(ptr, '/')) == NULL) continue;

			strncpy(mp, shareinfo.path, (int)(pp - shareinfo.path));
			mp[(int)(pp - shareinfo.path)] = 0x0;

			if (Get_Mount_DevName(mp, devpath) != SUCCESS) // the device is not mounted
				continue;

			if ((ret = mkdir_recursively(shareinfo.path)) < 0)
				break;
		}
	}

	if (listptr!=NULL) free(listptr);
	FTP_Create_Conf();
	return ret;
}
int Create_Storage_Share(char *share_name, int type, char *dev_name){
	NAS_SHARE_INFO share_ptr;
        char read_list[1][USER_GROUP_NAME_LENGTH];
        char write_list[2][USER_GROUP_NAME_LENGTH];
        int ret=0;
	char mount_name[HD_DEVICE_NAME_LENGTH];

	strcpy(share_ptr.share_name,share_name);
	Get_QNAP_Mount_Name(dev_name, mount_name, ROOT_PART);

	strcpy(share_ptr.path, mount_name);

	mkdir(share_ptr.path, S_IRWXU | S_IRWXG | S_IRWXO);
        chmod(share_ptr.path, S_IRWXU | S_IRWXG | S_IRWXO);
	if(!strcmp(dev_name,"/dev/sdb")){
		strcpy(share_ptr.comment,"eSATA storage share");
	}
	else
        	strcpy(share_ptr.comment,"USB storage share");

	if(type==SHARE_ADMINISTRATOR)
        {
                share_ptr.write_list_cnt=1;
                share_ptr.read_list_cnt=1;
                strcpy(read_list[0], "@everyone");
                strcpy(write_list[0], "administrator");
        }
        else if(type==SHARE_EVERYONE)
        {
                share_ptr.write_list_cnt=2;
                share_ptr.read_list_cnt=0;
                strcpy(write_list[0], "administrator");
                strcpy(write_list[1], "@everyone");
        }
	share_ptr.read_list_ptr = read_list;
        share_ptr.write_list_ptr = write_list;
        ret=Create_NAS_Share(&share_ptr);
	if(type!=SHARE_EVERYONE_GUEST)
		Add_NAS_User_For_Share(share_name, 'I', "guest");

        return ret;
}

int Create_Share_SLink(int drive_no, char *share_name){//Created by KenChen 20060320
	
	char cmd[MAX_CMDLINE_LENGTH],path[512];
	int ret;
	
	Get_MP_String(drive_no, DATA_PART, path);
	sprintf(cmd,"/bin/rm /share/%s > /dev/null 2>&1", share_name);
	system(cmd);
	sprintf(cmd, "/bin/ln -sf '%s/%s' '/share/%s' > /dev/null 2>&1", path, share_name, share_name);
  ret=system(cmd);
  //printf("--- SLink of %s Created, ret=%d\n", share_name,ret);
  return ret;
}

int Create_Default_Share_SLink(int drive_no){//Created by KenChen 20060320
	
	int ret=0;
	
	//printf("Create_Default_Share_SLink on HD(%d)\n",drive_no);
	if(Check_Default_Share(drive_no)){  //KenChen 20060320
		//printf("Start to create...\n",drive_no);
		Create_Share_SLink(drive_no,"Qmultimedia");
    Create_Share_SLink(drive_no,"Qdownload");
    Create_Share_SLink(drive_no,"Qweb");
    Create_Share_SLink(drive_no,"Qusb");
    Create_Share_SLink(drive_no,"Public");
    ret=1;
  }
  return ret;
}

int Create_Default_Share_From_Drive(int drive_no, char *share_name,int type){
	NAS_SHARE_INFO share_ptr;
	char read_list[1][USER_GROUP_NAME_LENGTH];
	char write_list[3][USER_GROUP_NAME_LENGTH];
	char path[512];
	int ret=0;

	Get_MP_String(drive_no, DATA_PART, path);
	strcpy(share_ptr.share_name,share_name);
	strcpy(share_ptr.path,path);
	strcat(share_ptr.path,"/");
	strcat(share_ptr.path,share_name);

	mkdir(share_ptr.path, S_IRWXU | S_IRWXG | S_IRWXO);
        chmod(share_ptr.path, S_IRWXU | S_IRWXG | S_IRWXO);
	strcpy(share_ptr.comment,"System default share");
	if(type==SHARE_ADMINISTRATOR)
        {
                share_ptr.write_list_cnt=1;
                share_ptr.read_list_cnt=1;
                strcpy(read_list[0], "@everyone");
                strcpy(write_list[0], "administrator");
        }
        else if(type==SHARE_EVERYONE)
        {
                share_ptr.write_list_cnt=2;
                share_ptr.read_list_cnt=0;
                strcpy(write_list[0], "administrator");
                strcpy(write_list[1], "@everyone");
        }
        else if(type==SHARE_EVERYONE_GUEST)
        {
                share_ptr.write_list_cnt=3;
                share_ptr.read_list_cnt=0;
                strcpy(write_list[0], "administrator");
                strcpy(write_list[1], "@everyone");
                strcpy(write_list[2], "guest");
        }

        share_ptr.read_list_ptr = read_list;
        share_ptr.write_list_ptr = write_list;
        ret=Create_NAS_Share(&share_ptr);
        Create_Share_SLink(drive_no,share_name); //KenChen 20060320

        if(type!=SHARE_EVERYONE_GUEST)
                Add_NAS_User_For_Share(share_name, 'I', "guest");
        return ret;

}

int Check_Share_Folder(char *folder_path){ //Created by KenChen 20060322

	char real_folder_path[256];
	
	if (realpath(folder_path, real_folder_path)!=NULL)
		return SUCCESS;
	else
		return ERROR_FAIL;
}

int Create_Default_Share(char *share_name,int type)//Shone added 2005,09,28
{
	NAS_SHARE_INFO share_ptr;
	char read_list[1][USER_GROUP_NAME_LENGTH];
	char write_list[3][USER_GROUP_NAME_LENGTH];
	int ret=0;

	strcpy(share_ptr.share_name,share_name);
	strcpy(share_ptr.path,"/share/HDA_DATA/");
	strcat(share_ptr.path,share_name);
	mkdir(share_ptr.path, S_IRWXU | S_IRWXG | S_IRWXO);
	chmod(share_ptr.path, S_IRWXU | S_IRWXG | S_IRWXO);
	strcpy(share_ptr.comment,"System default share");
	if(type==SHARE_ADMINISTRATOR)
	{
		share_ptr.write_list_cnt=1;
		share_ptr.read_list_cnt=1;
		strcpy(read_list[0], "@everyone");
		strcpy(write_list[0], "administrator");
	}
	else if(type==SHARE_EVERYONE)
	{
		share_ptr.write_list_cnt=2;
		share_ptr.read_list_cnt=0;
		strcpy(write_list[0], "administrator");
		strcpy(write_list[1], "@everyone");
	}
	else if(type==SHARE_EVERYONE_GUEST)
	{
		share_ptr.write_list_cnt=3;
		share_ptr.read_list_cnt=0;
		strcpy(write_list[0], "administrator");
		strcpy(write_list[1], "@everyone");
		strcpy(write_list[2], "guest");
	}
	
	share_ptr.read_list_ptr = read_list;
	share_ptr.write_list_ptr = write_list;
	ret=Create_NAS_Share(&share_ptr);
	
	if(type!=SHARE_EVERYONE_GUEST)
		Add_NAS_User_For_Share(share_name, 'I', "guest");
	return ret;
}


#ifdef	LARGE_NO_OF_USERS_SUPPORT	// Catherine

int Create_NAS_Share_Ex(NAS_SHARE_INFO *share_ptr, BOOL update_flash)
{
	char *buf_ptr, *valid_usr_list = NULL, *tptr;
	char tmp_name[100];
	char tmpbuf[USER_GROUP_NAME_LENGTH];
    	int  i,ret;

      	// ignore space &tab at begin and tail
	skip_white_space_start(share_ptr->share_name);
	stripe_white_space_tail(share_ptr->share_name);
	remove_successive_white_space(share_ptr->share_name);

	if ( !Check_Share_Name_Legal(share_ptr->share_name))
		return ERROR_ILLEGAL_NAME;

	// --  set all conf function case insensitive
	Conf_Case_Sensitive(0);

	//special process for creating "global" share - Louis Tsai (2001/6/28)
	if(strcasecmp(share_ptr->share_name,"global") == 0) {
		//check global service exist or not
		if(Conf_Get_Field(SAMBA_CONF_PATH,"global","browsable",NULL,0)<0)
			goto ADD_SERVICE;
		else
			return ERROR_ALREADY_EXISTED;

	}
	// reject printer section name
	if( strcasecmp("printers",share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;
    	if( strcasecmp("print$"  ,share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;
    	if( strcasecmp("printer$",share_ptr->share_name) == 0) return ERROR_ALREADY_EXISTED;

    	//--- Check the share name exist or not ? -------------
    	if ( (ret = Conf_Enum_Section(SAMBA_CONF_PATH,tmp_name,100)) == SUCCESS) {
    		if(compare_case_string(share_ptr->share_name,tmp_name) == 0)
    			return ERROR_ALREADY_EXISTED;
    		while ( (ret = Conf_Enum_Section(NULL,tmp_name,100)) == SUCCESS ) {
     			if(compare_case_string(share_ptr->share_name,tmp_name) == 0)
    				return ERROR_ALREADY_EXISTED;
    		}
	}
    	else
    		return ret;

ADD_SERVICE:
	Conf_Case_Sensitive(1);

    	//=============  Do append action  ===============
    	//----  append comment  ----
    	ret = Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"comment",share_ptr->comment);
    	if(ret != SUCCESS) return ret;

    	//----  append path  ----
      	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"path",share_ptr->path);
	if(ret != SUCCESS) return ret;

    	//----  append browsable ------------
    	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"browsable","yes");
	if(ret != SUCCESS) return ret;

    	//----  append public ------------
    	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"public","yes");
	if(ret != SUCCESS) return ret;

    	//----  append invalid users  ----
    	ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"invalid users","");
	if(ret != SUCCESS) return ret;

    	//----  append read list  ----
    	if (share_ptr->read_list_cnt >0 ) {
    		buf_ptr = (char *) malloc((share_ptr->read_list_cnt)*USER_GROUP_NAME_LENGTH);
    		if (buf_ptr == NULL) return ERROR_OUT_OF_MEMORY;

       		strcpy(buf_ptr,"");
       		for(i=0; i<share_ptr->read_list_cnt; i++) {
			strcpy(tmpbuf, (char*)(share_ptr->read_list_ptr+i));
			insert_headntail_into_string(tmpbuf, USER_GROUP_NAME_LENGTH);
       			strcat(buf_ptr, tmpbuf);
			strcat(buf_ptr,",");
		}
       		//--- overwrite the previously char ","  ---
       		*(buf_ptr+strlen(buf_ptr)-1) ='\0';
       		ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"read list",buf_ptr);
		if(ret != SUCCESS) {
    			free(buf_ptr);
    			return ret;
    		}
    		// valid users list
		valid_usr_list = (char*)calloc(1, sizeof(char)*(strlen(buf_ptr)+1));
		if (valid_usr_list == NULL) {
			free(buf_ptr);
			return ERROR_OUT_OF_MEMORY;
		}
		strcpy(valid_usr_list, buf_ptr);

    		free (buf_ptr);
    	}
    	else {
    		ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"read list","");
		if(ret != SUCCESS) return ret;
    	}
    	//----  append write list  ----
    	if (share_ptr->write_list_cnt >0 ) {
    		buf_ptr = (char *) malloc((share_ptr->write_list_cnt)*USER_GROUP_NAME_LENGTH);
    		if (buf_ptr == NULL) return ERROR_OUT_OF_MEMORY;

       		strcpy(buf_ptr,"");
       		for(i=0; i<share_ptr->write_list_cnt; i++) {
			strcpy(tmpbuf, (char*)(share_ptr->write_list_ptr+i));
			insert_headntail_into_string(tmpbuf, USER_GROUP_NAME_LENGTH);
       			strcat(buf_ptr, tmpbuf);
          		strcat(buf_ptr,",");
       		}
       		//--- overwrite the previously char ","  ---
       		*(buf_ptr+strlen(buf_ptr)-1) = '\0';
    		ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"write list",buf_ptr);
		if(ret != SUCCESS) {
    			free(buf_ptr);
    			return ret;
    		}
    		// valid users
    		if (valid_usr_list == NULL) { // no content yet
			valid_usr_list = calloc(1, sizeof(char)*(strlen(buf_ptr)+1));
			if (valid_usr_list == NULL) {
				free(buf_ptr);
				return ERROR_OUT_OF_MEMORY;
			}
		}
		else {
    			tptr = realloc(valid_usr_list, strlen(valid_usr_list) + strlen(buf_ptr) + 2);
			if (tptr == NULL) {
				free(buf_ptr);
				if (valid_usr_list) free(valid_usr_list);
				return ERROR_OUT_OF_MEMORY;
			}
			valid_usr_list = tptr;
			strcat(valid_usr_list, ",");
		}
		strcat(valid_usr_list, buf_ptr);
    		free (buf_ptr);
    	}
    	else {
    		ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"write list","");
		if(ret != SUCCESS) return ret;
    	}
	// add "valid users" list as well
	if (valid_usr_list != NULL) {
		ret =Conf_Set_Field(SAMBA_CONF_PATH,share_ptr->share_name,"valid users",valid_usr_list);
		free(valid_usr_list);
	}

	//if (update_flash)
	//	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);
	FTP_Create_Conf();

    	return SUCCESS;
}

int Remove_NAS_Share_Ex(char *share_name, BOOL update_flash)
{
    	int  ret;

	Conf_Case_Sensitive(0);

	//special process for creating "global" share - Louis Tsai (2001/6/28)
	if(strcasecmp(share_name,"global") == 0) {
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","comment"))<0)
			return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","path"))<0)
			return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","browsable"))<0)
			return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","invalid users"))<0)
			return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","read list"))<0)
			return ret;
		if((ret=Conf_Remove_Field(SAMBA_CONF_PATH,"global","write list"))<0)
			return ret;

		goto LEAVE;
	}
    	ret = Conf_Remove_Section(SAMBA_CONF_PATH,share_name);

LEAVE:
    	Remove_Connection_For_NAS_Share(share_name);
    	//if (update_flash)
    	//	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);
	FTP_Create_Conf();

    	return ret;
}
int Create_NAS_Share_From_Volume_Ex(char* share_name, int volume, char* path,
	char* comment, BOOL update_flash)
{
	VOLUME_CONF vol;
	NAS_SHARE_INFO share;
	SECTION_INFO *list=NULL;
	char str[BUF_SIZE], buf[BUF_SIZE];
    	int ret = SUCCESS, exist = 0, i, num;

	memset(&vol, 0, sizeof(vol));
	memset(&share, 0, sizeof(share));
	memset(&list, 0, sizeof(list));

	if(!compare_string(path, ""))
		return ERROR_FAIL;

	num = Get_NAS_Share_List_Ex(&list);

	for(i = 0;i < num; i++){
		if(!strcmp(share_name, list[i].section_name)) {					
			free(list);			
			return ERROR_ALREADY_EXISTED;
		}
	}
	free(list);

	if((ret = Get_One_Volume_Conf(volume, &vol)) == SUCCESS){
		Get_MP_String(vol.drive_no_list[0], DATA_PART, str);

		if(path == NULL){					
			if((search_dir(str, share_name)) == ERROR_ALREADY_EXISTED){	
				i = make_dir(1, share_name);				
				exist = 1;
 				strcat(str,"/");
				strcat(str, share_name);
				sprintf(buf,"%s%d",str,i);								
				if (chmod(buf, S_IRWXU | S_IRWXG | S_IRWXO) != 0)					
					return -1;				
			}
			else{
				strcat(str,"/");
				strcat(str,share_name);
				umask(0000);				
				if (mkdir(str, S_IRWXU | S_IRWXG | S_IRWXO) != 0 )
					return -1;
			}
		}
		else{
			if((ret = search_dir(str, path)) == ERROR_NOT_FOUND){
				strcat(str, "/");
				strcat(str, path);
				umask(0000);
				if (mkdir(str, S_IRWXU | S_IRWXG | S_IRWXO) != 0 )
					return -1;
			}
			else if(ret == ERROR_ALREADY_EXISTED){
				strcat(str, "/");
				strcat(str, path);
			}
		}
		if(exist == 1)
			strcpy(share.path, buf);
		else
			strcpy(share.path, str);
		strcpy(share.share_name, share_name);
		strcpy(share.comment, comment);		
		if((ret = Create_NAS_Share_Ex(&share, update_flash)) != SUCCESS)
			return ret;
	}
     	return ret;
}

int Clear_Invalid_NAS_Shares_Ex(BOOL update_flash)
{
	int cnt, ret, i;
	SECTION_INFO *listptr = NULL;

	cnt = Get_NAS_Shares(&listptr, TRUE);
	if (cnt<0) return cnt;

	for (i=0; i<cnt; i++) {
		NAS_SHARE_INFO shareinfo;

		memset(&shareinfo, 0, sizeof(NAS_SHARE_INFO));
		strcpy(shareinfo.share_name, listptr[i].section_name);

		ret = Get_NAS_Share_Info_Ex(&shareinfo); 
		if (ret<0) continue;

		if (!Is_Existing_Directory(shareinfo.path)) {
			ret = Remove_NAS_Share_Ex(shareinfo.share_name, update_flash);
			if (ret<0) break;
		}
	}
	if (listptr!=NULL) free(listptr);
	FTP_Create_Conf();

	return ret;
}
#endif

void Killusers_Delshare_On_Volume(int vol_no,int del_share)
{
        SMB_SHARE_INFO *smb_info,*smb_start;
        //LVM_VOLUME_CONF vol_data;
        char mp_str[BUF_SIZE];//,buf[BUF_SIZE];

        //Get_One_LVM_Volume_Conf(vol_no,&vol_data);

        Get_MP_String(vol_no,DATA_PART,mp_str);

        smb_info = SMB_Get_Share_Info();
        smb_start = smb_info;
        while(smb_info != NULL)
        {
                if(strstr(smb_info->path,mp_str) !=NULL)
                {
                        //sprintf(buf, "/sbin/smbtools -k \"%s\"",smb_info->sharename);
                        //system(buf);

                        if(del_share) Remove_NAS_Share(smb_info->sharename);
                        //sprintf(buf,"/bin/kill -HUP `/sbin/pidof smbd` 2>/dev/null 1>/dev/null");
                        //system(buf);
                }
                smb_info = smb_info->next;
        }
        SMB_Free_Share_Info(smb_start);
}
