//**************************************************************************
//
//	Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		security.c
//
//	Abstract: 
//		uLinux user/share security management functions.
//
//	HISTORY:
//		02/08/02 Catherine Shen - modify 'insert_headntail_into_string()'
//		02/05/02 Caterine Shen - modify 'Get_NAS_User_List_For_Share()',
//					'Add_NAS_User_List_For_Share()',
//					'Remove_NAS_User_List_For_Share()',
//					'Add_NAS_User_For_Share()',
//					'Remove_NAS_User_For_Share()',
//					'Get_NAS_User_List_For_Share_Ex()'
//		01/08/02 Catherine Shen - modify 'Remove_NAS_User_For_Share()',
//					'Add_NAS_User_For_Share()'
//		01/02/02 Catherine Shen - add 'Get_NAS_Share_Info_Ex()',
//					'Get_NAS_User_List_For_Share_Ex()',
//					'Get_Nas_Share_List_For_Path_Ex()',
//					'Get_NAS_User_Accessible_Share_List_Ex()'
//				modify 'Get_NAS_User_Security_For_Share()'
//		12/12/01 Catherine Shen - replace string comparing routines by new 'compare_string'
//				or 'compare_case_string' from 'Util'
//		06/19/01 Louis Tsai - check invalid user before adding to list (Get_NAS_User_Accessible_Share_List())
//		06/04/01 Louis Tsai - add Update_Flash_Data(),Remove_Connection_For_NAS_Share() and
//				      Refresh_Connection_For_NAS_Share() to relative function
//		05/11/01 Louis Tsai - remove white space of share name
//		04/20/01 Louis Tsai - add int Remove_User_Share_Privilege(char *usr_name);
//					  int Remove_Group_Share_Privilege(char *grp_name);
//		04/09/01 Louis Tsai - add two new function Add_NAS_User_For_Share() &
//				      Remove_NAS_User_For_Share()
//		04/04/01 Ethan Wang - fix some return code problem in Get_NAS_User_Security_For_Share.
//				otherwise, if we don't specify "write list" member, it will skip
//				"read list" member.
//		04/02/01 Louis Tsai - use uLinux_Util to rewrite all function below
//		12/18/00 Meiji Chang - splitted from usr_grp.c
//		09/01/00 Nick Ho - Created
//
//**************************************************************************
#include <stdio.h>
#include <stdlib.h>       
#include <string.h>

#include <dirent.h>
#include <sys/stat.h>

#include "NAS.h"
#include "uLinux.h"
#include "Util.h"
#include "cfg_ftp.h"
#include "user_grp.h"
#include "nas_lib_common.h"

#ifdef	LARGE_NO_OF_USERS_SUPPORT
#include <fcntl.h>
#include <unistd.h>
#include "file_lock.h"
#endif

//extern char *stripe_white_space_tail(char *str);
extern void skip_white_space_start(char *str);
extern void remove_successive_white_space(char *str);

// remove a specific character from a string
int trim_chars_from_string(char* string, char c)
{
	char *new, *old;
	int j=0, len;

	if (string==NULL)
		return 0;
	len=strlen(string);
	if (len==0) return 0;
	new=old=string;
	while (*old)
	{
		if (*old!=c)
		{
			*new++=*old;
			j++;
		}
		old++;
	}
	*new=0x0;
	return j;
}

// insert "\"" in the front & rear of the string
int insert_headntail_into_string(char* string, int string_max_len)
{
	int len;
	char *buf;

	if (*string=='@') { // a group name
		len = strlen(string);
		if (len+3>string_max_len) return ERROR_BUFFER_TOO_SMALL;

		buf = (char*) calloc(1, (len+3)*sizeof(char));
		strcpy(buf, "@\"");
		strcat(buf, &string[1]);
		strcat(buf, "\"");
		strcpy(string, buf);
		free(buf);
		return SUCCESS;
	}
	else { // a user name --> do nothing
		// 2004.11.10, Johnson Cheng
		// Fixed samba3.0 access right bug for user name including 
		// space character. ex: Johnson Cheng --> "Johnson Cheng"
		if(strchr(string, ' ') != 0)
		{
			len = strlen(string);
			if (len+3>string_max_len) return ERROR_BUFFER_TOO_SMALL;
			
			buf = (char*) calloc(1, (len+3)*sizeof(char));
			strcpy(buf, "\"");
			strcat(buf, string);
			strcat(buf, "\"");
			strcpy(string, buf);
			free(buf);
		}
		
		return SUCCESS;
	}
}

//--------------------------------------------------------------------------
// Get Read/Write List members                                
// action :  'R'=Read List,  'W'=Write List  'I'=invald users   
// u_cnt  :  I/P I:indicate pass in u_list cnt             
// Return :  >0			    --> if user_list = NULL or list_cnt=0 , return required list count.
//	     ERROR_BUFFER_TOO_SMALL --> input buffer is less than required
//	     ERROR_OPEN_FILE	    --> error open smb.conf
//	 	
//---------------------------------------------------------------------------	     
int Get_NAS_User_List_For_Share(char *old_share_name, char security_id, char (*user_list)[USER_GROUP_NAME_LENGTH], int list_cnt) 
{
	char field_name[30];
	int  ret = SUCCESS,i=1;
    	char share_name[SHARE_NAME_LENGTH];
 
    	strcpy(share_name,old_share_name);

	Conf_Case_Sensitive(0);

	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);
	
	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");
	
	if((user_list == NULL) || (list_cnt == 0) ) { // return required list count
		if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0))>= 0 ) {
			char member[ret];
			Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
			while( Get_String_Field(member,i,',',NULL,0) >= 0 ) i++;
			return i-1;
		}	
		else 
			return 0;	// no field found, return 0 users in the list.
	}	
	
	i=1;
	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {
		char member[ret];

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);

		while( Get_String_Field(member,i,',',user_list[i-1],USER_GROUP_NAME_LENGTH) == SUCCESS ) {
			trim_chars_from_string(user_list[i-1], '\"'); // Catherine 2002/02/05
			i++;
			if(i>list_cnt) {
				ret = ERROR_BUFFER_TOO_SMALL;
				break;	
			}	
		}
		if (2 == i) { //prevent null list
			if(strcmp(user_list[0],"")==0) 
				i = 1;
		}
		ret = i-1;
	}

	return ret;
}


//-------------------------------------------------------------
// Add members To Read/Write List                              
// action : 'R'=Read List,  'W'=Write List  'I'=invalid users   
// u_cnt  :  I/P I:indicate pass in u_list cnt                    
// Return :  ERROR_OPEN_FILE  --> error open smb.conf
//	     ERROR_NOT_FOUND  --> no this share_name	
//	     SUCCESS	      --> function ok
// 
// comment:  04/02/2001 Louis Tsai  ;this function will ignore exist member	 	
//---------------------------------------------------------------------------
 
 
int Add_NAS_User_List_For_Share(char *old_share_name,char security_id, char (*u_list)[USER_GROUP_NAME_LENGTH], int u_cnt) 
{
	char field_name[30],tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,i=0,j=1;
	int  exist_flag = 0;
    	char share_name[SHARE_NAME_LENGTH];

    	strcpy(share_name,old_share_name);

    	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);

	Conf_Case_Sensitive(0);

	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");

	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret]; //oringinal member string
		char new_member[ret+u_cnt*(USER_GROUP_NAME_LENGTH)]; // new member string
		
		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		strcpy(new_member,member);
		for (i=0;i<u_cnt;i++) {
			j=1;
			exist_flag = 0;
			while( Get_String_Field(new_member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
				trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
				if(compare_case_string(tmp_usr_name,u_list[i]) == 0) {
					exist_flag = 1;
					break;
				}	
				j++;
			}
			
			if ( 0 == exist_flag ) {
				strcpy(tmp_usr_name, u_list[i]);
				insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);
				Insert_String_Field( new_member, 0, ',', tmp_usr_name,
					new_member, ret+u_cnt*(USER_GROUP_NAME_LENGTH));
			}
		}
		
		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,new_member);
	}
	// valid users
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret]; //oringinal member string
		char new_member[ret+u_cnt*(USER_GROUP_NAME_LENGTH)]; // new member string
		
		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		strcpy(new_member,member);
		for (i=0;i<u_cnt;i++) {
			strcpy(tmp_usr_name, u_list[i]);
			insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);
			Insert_String_Field( new_member, 0, ',', tmp_usr_name,
				new_member, ret+u_cnt*(USER_GROUP_NAME_LENGTH));
		}
		
		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",new_member);
	}

	Refresh_Connection_For_NAS_Share(share_name);
	
	//Update_Flash_Data(SAMBA_CONF_PATH);

//2001/7/23 add by louistsai
	FTP_Create_Conf();
//end
	return ret;
}

//--------------------------------------------------------------------------
// Delete members From Read/Write List                         
// action : 'R'=Read List,  'W'=Write List 'I'=invalid users    
// u_cnt  : I/P I:indicate pass in u_list cnt                    
// Return :  ERROR_OPEN_FILE  -->  error open smb.conf
//	     ERROR_NOT_FOUND  -->  no this share_name				
//	     SUCCESS          -->  function ok
// 
// comment:  04/02/2001 Louis Tsai  ;if remove member no found,ignore.	 	
//---------------------------------------------------------------------------	 
int Remove_NAS_User_List_For_Share(char *old_share_name,char security_id, char (*u_list)[USER_GROUP_NAME_LENGTH], int u_cnt) 
{
  	char field_name[30],tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,i=0,j=1;
	int  exist_flag = 0;
	char share_name[SHARE_NAME_LENGTH];

    	strcpy(share_name,old_share_name);

	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);
	Conf_Case_Sensitive(0);
	
	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");

	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret];
		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		for (i=0;i<u_cnt;i++) {
			j=1;
			exist_flag = 0;
			while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
				trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
				if(compare_case_string(tmp_usr_name,u_list[i]) == 0) {
					exist_flag = 1;
					break;
				}	
				j++;
			}
			
			if ( 1 == exist_flag ) 
				Remove_String_Field( member,j,',');
		}
		
		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,member);
	}
	// remove this user from "valid users" list as well
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret];

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		for (i=0;i<u_cnt;i++) {
			j=1;
			exist_flag = 0;
			while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
				trim_chars_from_string(tmp_usr_name, '\"');
				if(compare_case_string(tmp_usr_name,u_list[i]) == 0) {
					exist_flag = 1;
					break;
				}	
				j++;
			}
			
			if ( 1 == exist_flag ) 
				Remove_String_Field( member,j,',');
		}
		
		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",member);
	}

	Refresh_Connection_For_NAS_Share(share_name);
	//Update_Flash_Data(SAMBA_CONF_PATH);
	FTP_Create_Conf();
	return ret;
}
//-------------------------------------------------------------
// Add a member To Read/Write List                              
// action : 'R'=Read List,  'W'=Write List  'I'=invalid users   
// u_cnt  :  I/P I:indicate pass in u_list cnt                    
// Return :  ERROR_OPEN_FILE  --> error open smb.conf
//	     ERROR_NOT_FOUND  --> no this share_name	
//	     SUCCESS	      --> function ok
// 
// comment:  04/02/2001 Louis Tsai  ;this function will ignore exist member	 	
//---------------------------------------------------------------------------	   
int Add_NAS_User_For_Share(char *old_share_name,char security_id, char *user_name)
{
	char field_name[30],tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,j=1;
	int  exist_flag = 0;
    	char share_name[SHARE_NAME_LENGTH];

#ifdef DEBUG
	printf("Add_NAS_User_For_Share: %s, %c, %s\n",
		old_share_name, security_id, user_name);
#endif
    	strcpy(share_name,old_share_name);
	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);

	Conf_Case_Sensitive(0);
	
	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");	

#ifdef DEBUG
	printf("Add_NAS_User_For_Share: Step 1 -- %d\n",
		USER_GROUP_NAME_LENGTH);
#endif
	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret]; //oringinal member string
		char new_member[ret+USER_GROUP_NAME_LENGTH]; // new member string
#ifdef DEBUG
		printf("Add_NAS_User_For_Share: Step 2\n");
		printf("member size: %d, new_member size: %d\n",
			ret, ret+USER_GROUP_NAME_LENGTH);
#endif
		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		strcpy(new_member,member);
		
		j=1;
		exist_flag = 0;
#ifdef DEBUG
		printf("Add_NAS_User_For_Share: Step 3\n");
#endif
		while( Get_String_Field(new_member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
			trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
			if(compare_case_string(tmp_usr_name,user_name) == 0) {
#ifdef DEBUG
				printf("Add_NAS_User_For_Share: Step 4\n");
#endif				
				exist_flag = 1;
				break;
			}	
			j++;
		}
			
		if ( 0 == exist_flag ) {
#ifdef DEBUG
			printf("Add_NAS_User_For_Share: Step 5\n");
#endif
			strcpy(tmp_usr_name, user_name);
			insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);
			ret = Insert_String_Field( member, 0, ',', tmp_usr_name,
				new_member, ret+USER_GROUP_NAME_LENGTH);
#ifdef DEBUG
			printf("Add_NAS_User_For_Share: Step 6: %d\n", ret);
			printf("Old Member List: %s, len=%d\n",
				member, strlen(member));
			printf("New Member List: %s, len=%d\n",
				new_member, strlen(new_member));
#endif
			ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,new_member);
#ifdef DEBUG
			printf("Conf_Set_Field Success? %d\n", ret);
#endif
		}
		else
			return ERROR_ALREADY_EXISTED;
	}
	// valid users list
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret]; //oringinal member string
		char new_member[ret+USER_GROUP_NAME_LENGTH]; // new member string

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		strcpy(new_member,member);
		strcpy(tmp_usr_name, user_name);
		insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);
		ret = Insert_String_Field( member, 0, ',', tmp_usr_name,
			new_member, ret+USER_GROUP_NAME_LENGTH);

		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",new_member);
	}

#ifdef DEBUG
	printf("Add_NAS_User_For_Share: Step 7\n");
#endif	
	Refresh_Connection_For_NAS_Share(share_name);
#ifdef DEBUG
	printf("Add_NAS_User_For_Share: Step 8\n");
#endif
	//Update_Flash_Data(SAMBA_CONF_PATH);

#ifdef DEBUG
	printf("Add_NAS_User_For_Share: Step 9\n");
#endif
	FTP_Create_Conf();

	return ret;
}

//--------------------------------------------------------------------------
// Delete a members From Read/Write List                         
// action : 'R'=Read List,  'W'=Write List 'I'=invalid users    
// u_cnt  : I/P I:indicate pass in u_list cnt                    
// Return :  ERROR_OPEN_FILE  -->  error open smb.conf
//	     ERROR_NOT_FOUND  -->  no this share_name				
//	     SUCCESS          -->  function ok
// 
//---------------------------------------------------------------------------	 
int Remove_NAS_User_For_Share(char *old_share_name,char security_id, char *user_name) 
{
  	char field_name[30],tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,j=1;
	int  exist_flag = 0;
    	char share_name[SHARE_NAME_LENGTH];

    	strcpy(share_name,old_share_name);

	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);

	Conf_Case_Sensitive(0);

	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");

	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret];
		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		j=1;
		exist_flag = 0;
		while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
			trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
			if(compare_case_string(tmp_usr_name,user_name) == 0) {
				exist_flag = 1;
				break;
			}	
			j++;
		}
			
		if ( 1 == exist_flag ) { 
#ifdef DEBUG
			printf("Before Remove_String_Field, member=%s\n", member);
#endif
			Remove_String_Field( member,j,',');
#ifdef DEBUG
			printf("After Remove_String_Field, member=%s\n", member);
#endif
			ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,member);
#ifdef DEBUG
			printf("Conf_Set_Field Success? %d\n", ret);
#endif
		}
		else
			return ERROR_NOT_FOUND;
	}

	// remove this user from "valid users" list as well
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret];

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		j=1;
		exist_flag = 0;
		while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
			trim_chars_from_string(tmp_usr_name, '\"');
			if(compare_case_string(tmp_usr_name,user_name) == 0) {
				exist_flag = 1;
				break;
			}	
			j++;
		}
			
		if ( 1 == exist_flag ) { 
			Remove_String_Field( member,j,',');
			ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",member);
		}
	}

	Refresh_Connection_For_NAS_Share(share_name);
	//Update_Flash_Data(SAMBA_CONF_PATH);
//2001/7/23 add by louistsai
	FTP_Create_Conf();
//end	
	return ret;
}





//------------------------------------------------------------------
//- 2000/12/8: Add for other protocol security check requirement   -
//- return what permission of the user on the specificed share DIR -
//- It will check the user is belonged the group which has setted  -
//- permission on specficed share DIR                              -
//- If the user hasn't setted has any permission, it will regard as-
//- SHARE_NOACCESS, that is samba default setting, but I had       -
//- modified smbas source code to meet my requirement              -
//- 1/12 2000 Add First check invalid user=, return NOACCESS       -
//- Second Check Write List=, return READWRITE                     -
//- Last Check Read List=, return READONLY
//------------------------------------------------------------------
int Get_NAS_User_Security_For_Share(char *user_name, char *share_name)
{
  	char (*u_list)[USER_GROUP_NAME_LENGTH];
  	int  ret,i;
  	int group_right = SHARE_NOACCESS,user_right = SHARE_NOACCESS;
	int m_cnt,j;
	char (*m_list)[USER_NAME_LENGTH];
	char *group_name;
	
    	group_name = Get_Group_Name_By_User_Name(user_name);
    	
    	//-- First get he has NOACCESS permission or not ? -----
    	ret = Get_NAS_User_List_For_Share_Ex(share_name,'I', &u_list);
    	
    	if (ret >0) {  //- it has noaccess list member -
       		for(i=0;i<ret;i++){			
          			if ( *(*(u_list+i))=='@') { // a group 
	          				// 2004.12.09, Johnson Cheng
	          				// For multi-domain AD
										if(group_name && (compare_case_string(group_name, (*(u_list+i)+1)) == 0))
										{
											free(u_list);
		       						return SHARE_NOACCESS;
		                }	
             				m_cnt=Get_NAS_User_List_Of_Group_Ex( (*(u_list+i)+1), &m_list);

            				for(j=0; j<m_cnt; j++) {
               					if ( compare_case_string((char *) (m_list+j),user_name)==0 ) {
                 						free(u_list);
                 						free(m_list);
                 						return SHARE_NOACCESS;
               					}
             				}

             				if (m_cnt>0) {
						free(m_list);
						m_list = NULL;
					}
          			}
          			else { // a user -
            				if ( compare_case_string((char *)(u_list+i),user_name)==0 ) {
              					free(u_list);
              					return SHARE_NOACCESS;
            				}
          			}
       		} // end of for() 

       		free(u_list);
		u_list = NULL;
    	} // end of ret>0 
    	else if((ret != ERROR_NOT_FOUND) && (ret !=0)) {
      		return ret;
    	}

    	//-- Second get he has R/W permission or not ? -----
	ret = Get_NAS_User_List_For_Share_Ex(share_name,'W', &u_list);

	if (ret >0) {  //- it has write list member -
		for(i=0;i<ret;i++){
			if ( *(*(u_list+i))=='@') { // a group 
				// 2004.12.09, Johnson Cheng
				// For multi-domain AD
				if(group_name && (compare_case_string(group_name, (*(u_list+i)+1)) == 0))
					group_right = SHARE_READWRITE;						
				else
				{
				m_cnt=Get_NAS_User_List_Of_Group_Ex( (*(u_list+i)+1), &m_list);

	             			for(j=0; j<m_cnt; j++) {
	               				if ( compare_case_string((char *) (m_list+j),user_name)==0 ) {    
	                 					group_right = SHARE_READWRITE;//louis
	                 					break;     		
	               				}
	             			}
	             			
	             			if (m_cnt>0) {
						free(m_list);
						m_list = NULL;
					}
	          		}
	          	}	// 2004.12.09, Johnson Cheng Add End
	          		else { //- a user -
	            			if ( compare_case_string((char *)(u_list+i),user_name)==0 )
	              				user_right = SHARE_READWRITE;
	          		}
       		} // end of for()

       		free(u_list);
		u_list = NULL;
       	} // end of ret>0 
    	// BEGIN: Ethan modified
    	// if the return code is equal to 0, it means there're no member in the specified group.
	else if(ret != ERROR_NOT_FOUND && ret != 0) {
    		// END: Ethan modified
		return ret;
	}

   	//-- Third get he has Read Only permission or not ? -----
    	ret = Get_NAS_User_List_For_Share_Ex(share_name,'R', &u_list);
    	if (ret >0) {  //- it has write list member -
		for(i=0;i<ret;i++){
			if ( *(*(u_list+i))=='@') { // he is a group 
				// 2004.12.09, Johnson Cheng
				// For multi-domain AD
				if(group_name && (compare_case_string(group_name, (*(u_list+i)+1)) == 0))
					group_right = SHARE_READONLY;
				else
				{
           m_cnt=Get_NAS_User_List_Of_Group_Ex( (*(u_list+i)+1),&m_list);
             			
           for(j=0; j<m_cnt; j++) {
					if ( compare_case_string((char *)(m_list+j),user_name)==0 ) {
						group_right = SHARE_READONLY;
 						break;
               					}
             				}

             				if (m_cnt>0) {
						free(m_list);
						m_list = NULL;
					}
          			} // 2004.12.09, Johnson Cheng Add End
          		}	
          			else { //- he is a user -
            				if ( compare_case_string((char *)(u_list+i),user_name)==0 ) 
					user_right = SHARE_READONLY;
          			}
       		} // end of for() 

       		free(u_list);
    	} // end of ret>0 
    	// BEGIN: Ethan modified
    	else if(ret != ERROR_NOT_FOUND && ret != 0) {
    		// END: Ethan modified
		return ret;
    	}

    	if(user_right == SHARE_READONLY)
	{
		if (group_right==SHARE_READWRITE)
			return SHARE_READWRITE;
		else
	    		return SHARE_READONLY;
	}
    	else if(user_right == SHARE_READWRITE)
    		return SHARE_READWRITE;
    	else if(group_right == SHARE_READONLY)
    		return SHARE_READONLY;
    	else if(group_right == SHARE_READWRITE)
    		return SHARE_READWRITE;
    	//-- not R/W, not READ ONLY, so is NOACCESS -
 
   	 return SHARE_NOACCESS;
}

//--------------------------------------------------
// return: SUCCESS --> Successful
//       : Others  --> Fail
//--------------------------------------------------
int Get_NAS_Share_Info(NAS_SHARE_INFO *share_ptr) 
{
	int  ret;
	
	skip_white_space_start(share_ptr->share_name);
	stripe_white_space_tail(share_ptr->share_name);
	remove_successive_white_space(share_ptr->share_name);
  	
  	//---- Get comment -----
  	if ((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_ptr->share_name,"comment",share_ptr->comment,sizeof(share_ptr->comment))) !=SUCCESS ) {
     		strcpy(share_ptr->comment,"");
	}
  	//---- Get path -----
  	if ((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_ptr->share_name,"path",share_ptr->path,sizeof(share_ptr->path))) !=SUCCESS ) {
#ifdef DEBUG
       		printf("Get %s path Fail \n",share_ptr->path);
#endif
    		return ret;  
  	}

  	//---- Get Read List -----
  	if ( share_ptr->read_list_ptr != NULL || share_ptr->read_list_cnt==0) {
//  	if ( share_ptr->read_list_ptr != NULL && share_ptr->read_list_cnt != 0) {
     		ret = Get_NAS_User_List_For_Share(
     			share_ptr->share_name, 'R',
			share_ptr->read_list_ptr,
			share_ptr->read_list_cnt);

     		if ( ret < 0 ) {
#ifdef DEBUG
          		printf("Get_NAS_User_List_For_Share('R'), ret=%d \n",ret);
#endif
        		return ret;
     		}
     		else share_ptr->read_list_cnt = ret;
  	}

  //---- Get Write List -----
  if ( share_ptr->write_list_ptr != NULL || share_ptr->write_list_cnt==0) {
     ret=Get_NAS_User_List_For_Share(share_ptr->share_name,'W',
				     share_ptr->write_list_ptr,
				     share_ptr->write_list_cnt);

     if ( ret < 0 ) {
        #ifdef DEBUG
          printf("Get_NAS_User_List_For_Share('W'), ret=%d \n",ret);
        #endif
        return ret;
     }
     else share_ptr->write_list_cnt = ret;
  }

  return SUCCESS;
  
}


//---------------------------------------------------------
// return nnn: The Number of Share he has RO or RW right
//        other: Fail
//---------------------------------------------------------
int Get_NAS_User_Accessible_Share_List(char *user_name, SHARE_NAME_LIST *share_name_list, int list_cnt)
{
	int  i,j,m, ret, share_no, ret_no=0,invalid_cnt=0;
  	SECTION_INFO s_ptr[MAX_SECTION_NUMBER];
  	char member[MAX_USER_NUMBER][USER_GROUP_NAME_LENGTH];
  	char invalid_member[MAX_USER_NUMBER][USER_GROUP_NAME_LENGTH];
  	char *group_name;
    	
  	share_no = Get_NAS_Share_List(s_ptr,MAX_SECTION_NUMBER);
  	if ( share_no <= 0) return share_no;
  
  	group_name = Get_Group_Name_By_User_Name(user_name);
  	for (i=0; i<share_no; i++) {
    		int find_flag=0;
    		for (m=0; m<2; m++) {
    			//-- First Get Read Only List & Check user_name in them -------
    			//-- Or belong to one of their groups                   -------
    			//-- Second Get Write List                              -------

        		char access;  

			if ( find_flag ) break;
			if (m==0)  access='R';
			else       access='W';
			
			invalid_cnt = Get_NAS_User_List_For_Share(s_ptr[i].section_name,'I',invalid_member,MAX_USER_NUMBER);
			ret = Get_NAS_User_List_For_Share(s_ptr[i].section_name,access,member,MAX_USER_NUMBER);
			
#ifdef DEBUG
          		printf("Get_NAS_User_List_For_Share(%s), ret=%d \n",s_ptr[i].section_name,ret);
#endif

			for (j=0; j<ret; j++) {
	    			if ( find_flag ) break;
#ifdef DEBUG
	    			printf("member[%02d]='%s' \n",j,member[j]);
#endif

	    			if (member[j][0]=='@') { //-- he is a group
	       				//char t_mem[MAX_USER_NUMBER][USER_NAME_LENGTH];
	       				char (*t_mem)[USER_NAME_LENGTH] = NULL;
	       				int mem_cnt, k,pp,invalid_flag = 0;
	   
	       				for(pp=0;pp<invalid_cnt;pp++) {
	       					if ( invalid_member[pp][0] == '@') {
	       						if(User_Belongs_To_Group(user_name,&invalid_member[pp][1])) {
	       							invalid_flag = 1;
	       							break;
	       						}
	       					}
	       					else
	       					if (compare_case_string(user_name,invalid_member[pp])==0) {
	       						invalid_flag = 1;
	       						break;
	       					}
	       				}
	       				// 2004.12.09, Johnson Cheng
	       				// For multi-domain AD
	       				if(group_name && (compare_case_string(group_name, (char*)(member[j]+1)) == 0))
	       				{
	       					strcpy((share_name_list+ret_no)->share_name,s_ptr[i].section_name);
		     			 		ret_no++;
		      				find_flag = 1;
		      			}
		      			if(find_flag != 1)
		      			{	       				
	       				mem_cnt = Get_NAS_User_List_Of_Group_Ex(&member[j][1], &t_mem);
	       				for (k=0; k<mem_cnt; k++) {
		   				if ((compare_case_string(t_mem[k],user_name)==0) && (invalid_flag != 1)) {
		     	 				if ( ret_no >= list_cnt )  {
		     	 					if (mem_cnt>0 && t_mem)
		     	 						free(t_mem);
		     	 					return ret_no;
		     	 				}
		      					strcpy((share_name_list+ret_no)->share_name,s_ptr[i].section_name);
		     			 		ret_no++;
		      					find_flag = 1;
#ifdef DEBUG
							printf("'%s' is a member of group %s \n", user_name, member[j]);
#endif
		      					break;
		   				}
	       				}
	       			} // 2004.12.09, Johnson Cheng Add End
					if (mem_cnt>0 && t_mem)
						free(t_mem);
	    			}
	    			else { //-- he is user	

	       				if (compare_case_string(member[j],user_name)==0 ) {
#ifdef DEBUG
						printf("'%s' is a user member\n", user_name);
#endif

		  				if ( ret_no >= list_cnt )  return ret_no;
						if(Get_NAS_User_Security_For_Share(user_name,s_ptr[i].section_name) == SHARE_NOACCESS) continue;
		  				strcpy((share_name_list+ret_no)->share_name,s_ptr[i].section_name);
		 	 			ret_no++;
		  				find_flag = 1;
		  				break;
	       				}
	    			}
			}    
  		}
  	}
  return  ret_no;
}
//---------------------------------------------------------------//
// Remove all share privilege for specified user
// Return :	SUCCESS
//		ERROR_OUT_OF_MEMORY,  cannot allocate memory
//---------------------------------------------------------------//

int Remove_User_Share_Privilege(char *usr_name)
{
        int i,j, count, ret;
        char (*list_ptr)[USER_GROUP_NAME_LENGTH];
        char *s_ptr;
	SECTION_INFO *share_list;

	// get all shares....
	count = Get_NAS_Share_List((SECTION_INFO*)NULL, 0);
	if (count >0){
		share_list = (SECTION_INFO *) calloc ( count, sizeof(SECTION_INFO));
		if (share_list == NULL)
			return ERROR_OUT_OF_MEMORY;
		Get_NAS_Share_List(share_list,count);
	}
	else
		return count;

	for (i = 0; i < count; i++) {
		/*--- get read List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name,'R',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name,'R',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(usr_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share(share_list[i].section_name, 'R',usr_name);
					break;
				}
			}
			free(list_ptr);		
		}
		/*--- get write List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name,'W',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name,'W',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(usr_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share(share_list[i].section_name, 'W',usr_name);
					break;
				}
			}
			free(list_ptr);
		}
		/*--- get invalid user List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(usr_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share(share_list[i].section_name, 'I',usr_name);
					break;
				}
			}
			free(list_ptr);		
		}
		
        }
        
        free( share_list);
        
        //Update_Flash_Data(SAMBA_CONF_PATH);
//2001/7/23 add by louistsai
	FTP_Create_Conf();
//end        
	return SUCCESS;
}

int Remove_Multiple_User_Share_Privilege(char (*usr_list)[USER_NAME_LENGTH], int count)
{
	
        int i,j, cnt;
        char (*list_ptr)[USER_NAME_LENGTH];
	SECTION_INFO *share_list;
	
	list_ptr = usr_list;
	// get all shares....
	
	if (usr_list == NULL || count == 0)
		return	ERROR_FAIL;
	cnt = Get_NAS_Share_List((SECTION_INFO*)NULL, 0);
	if (cnt >0){
		share_list = (SECTION_INFO *) calloc ( cnt, sizeof(SECTION_INFO));
		if (share_list == NULL)
			return ERROR_OUT_OF_MEMORY;
		Get_NAS_Share_List(share_list,cnt);
	}
	else
		return cnt;
	
	for (i = 0; i < cnt; i++) {
		/*--- get read List ---*/
		for (j = 0; j < count; j++) {
			Remove_NAS_User_For_Share(share_list[i].section_name, 'R',*(list_ptr+j));
			Remove_NAS_User_For_Share(share_list[i].section_name, 'W',*(list_ptr+j));
			Remove_NAS_User_For_Share(share_list[i].section_name, 'I',*(list_ptr+j));
		}
	}
        free(share_list);        
        //Update_Flash_Data(SAMBA_CONF_PATH);
//2001/7/23 add by louistsai
	FTP_Create_Conf();
//end        
	return SUCCESS;
}

//---------------------------------------------------------------//
// Remove all share privilege for specified group
// Return :	SUCCESS
//		ERROR_OUT_OF_MEMORY,  cannot allocate memory
//---------------------------------------------------------------//

int Remove_Group_Share_Privilege(char *grp_name)
{
        int i,j, count, ret;
        char (*list_ptr)[USER_GROUP_NAME_LENGTH];
        char *s_ptr;
        char tmp_grp_name[strlen(grp_name)+2];
	SECTION_INFO *share_list;
	
	strcpy(tmp_grp_name,"@");
	strcat(tmp_grp_name,grp_name);
	
	// get all shares....
	count = Get_NAS_Share_List((SECTION_INFO *)NULL, 0);
	if (count >0){
		share_list = (SECTION_INFO *) calloc ( count, sizeof(SECTION_INFO));
		if (share_list == NULL)
			return ERROR_OUT_OF_MEMORY;
		Get_NAS_Share_List(share_list,count);
	}
	else
		return count;
	
	for (i = 0; i < count; i++) {
		/*--- get read List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'R',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'R',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
// Catherine				if (compare_string(tmp_grp_name, s_ptr) == 0) {
				if (compare_case_string(tmp_grp_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share(share_list[i].section_name, 'R',tmp_grp_name);
					break;
				}
			}
			free(list_ptr);		
		}
		/*--- get write List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'W',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'W',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
// Catherine				if (compare_string(tmp_grp_name, s_ptr) == 0) {
				if (compare_case_string(tmp_grp_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share(share_list[i].section_name, 'W',tmp_grp_name);
					break;
				}
			}
			free(list_ptr);		
		}
		/*--- get invalid user List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
// Catherine				if (compare_string(tmp_grp_name, s_ptr) == 0) {
				if (compare_case_string(tmp_grp_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share(share_list[i].section_name, 'I',tmp_grp_name);
					break;
				}
			}
			free(list_ptr);		
		}
		
        }
        
        free( share_list);
        
        //Update_Flash_Data(SAMBA_CONF_PATH);
//2001/7/23 add by louistsai
	FTP_Create_Conf();
//end
	return SUCCESS;
}

int Remove_Multiple_Group_Share_Privilege(char (*grp_list)[GROUP_NAME_LENGTH], int count)
{
        int i,j, cnt;
        char (*list_ptr)[GROUP_NAME_LENGTH+2];
	SECTION_INFO *share_list;
	
	if (grp_list == NULL || count == 0)
		return	ERROR_FAIL;
	// get all shares....
	cnt = Get_NAS_Share_List((SECTION_INFO *)NULL, 0);
	if (cnt >0){
		share_list = (SECTION_INFO *) calloc ( cnt, sizeof(SECTION_INFO));
		if (share_list == NULL)
			return ERROR_OUT_OF_MEMORY;
		Get_NAS_Share_List(share_list,cnt);
	}
	else
		return cnt;
	
	list_ptr = (char (*)[GROUP_NAME_LENGTH+2]) calloc (count,GROUP_NAME_LENGTH+2);
	if (list_ptr == NULL) {
		free( share_list);	
		return ERROR_OUT_OF_MEMORY;
	} 
	for (i = 0; i < count ; i++){
		strcpy(*(list_ptr+i),"@");
		strcat(*(list_ptr+i),*(grp_list+i));
	}
	
	for (i = 0; i < cnt; i++) {
		for (j = 0; j < count; j++) {
			Remove_NAS_User_For_Share(share_list[i].section_name, 'R',*(list_ptr+j));
			Remove_NAS_User_For_Share(share_list[i].section_name, 'W',*(list_ptr+j));
			Remove_NAS_User_For_Share(share_list[i].section_name, 'I',*(list_ptr+j));
		}
        }
        
        free( share_list);
        free(list_ptr);	
        //Update_Flash_Data(SAMBA_CONF_PATH);
//2001/7/23 add by louistsai
	FTP_Create_Conf();
//end
	return SUCCESS;
}

//----------------------------------------------------------//
//while removing samba share folder, must call this function
//at last to avoid illegal connection still keep connection
//---------------------------------------------------------//
int Remove_Connection_For_NAS_Share(char *share_name)
{
	char cmd_line[256];
	int ret;
	
	// Kevin Liao 2001-07-11: Fix share name with space bug
	//sprintf(cmd_line,"%s -k %s > /dev/null",SAMBA_TOOL_PATH,share_name);
	sprintf(cmd_line,"%s -k \"%s\" > /dev/null",SAMBA_TOOL_PATH,share_name);
//	printf("%s\n",cmd_line);
	ret = system(cmd_line);
	
	if(ret<0) return ERROR_FAIL;
	else	return SUCCESS;
}
//----------------------------------------------------------//
//while share property have been changed, program must call 
//this function to verify all connection is illeage, if not,
//kill this connection.
//----------------------------------------------------------//

int Refresh_Connection_For_NAS_Share(char *share_name)
{
	char cmd_line[256];
	int ret;
	
	sprintf(cmd_line,"%s -r \"%s\" > /dev/null",SAMBA_TOOL_PATH,share_name);
//	printf("%s\n",cmd_line);
	ret = system(cmd_line);
	
	if(ret<0) return ERROR_FAIL;
	else	return SUCCESS;
	
}

//---------------------------------------------------------------------------//
// Get all share list for some path, if pathname include filename, set file_flag
// to 1. If share_name_list == null or list_cnt =0 , just return required string
// buffer count.
//---------------------------------------------------------------------------//

int Get_Nas_Share_List_For_Path(char *path, SHARE_NAME_LIST *share_name_list,int list_cnt,int file_flag)
{
	
	SECTION_INFO s_ptr[MAX_SECTION_NUMBER];
	int i,sh_cnt = 0,share_no = 0;
  	char *ptr,*tmp_path;

  	tmp_path = strdup(path);

  	if(file_flag == 1) {
  		ptr = strrchr(tmp_path,'/');
  		*ptr = '\0';
  	}
  	
  	share_no = Get_NAS_Share_List(s_ptr,MAX_SECTION_NUMBER);
  	if ( share_no <= 0) return share_no;


	for(i=0;i<share_no;i++) {
		NAS_SHARE_INFO share_info;
		
		strcpy(share_info.share_name,s_ptr[i].section_name);

		share_info.read_list_cnt = 0;       
		share_info.read_list_ptr = NULL;       
		share_info.write_list_cnt = 0;     
		share_info.read_list_ptr = NULL;     

		Get_NAS_Share_Info(&share_info);

// Catherine		if(strcmp(share_info.path,tmp_path) == 0) {
		// 2005.04.27, Johnson Cheng
		// Fixed share name of Active User log bug while use browsing a file 
		//if(compare_string(share_info.path,tmp_path) == 0) {
		if(strncmp(share_info.path,tmp_path, strlen(share_info.path)) == 0) {
			if(list_cnt == 0 || share_name_list == NULL)
				sh_cnt++;
			else{

				strcpy(share_name_list[sh_cnt].share_name,share_info.share_name);
				sh_cnt++;
				if(sh_cnt>list_cnt) {
					sh_cnt = ERROR_BUFFER_TOO_SMALL;
					break;
				}
			}
		}
	}
	free(tmp_path);
	return sh_cnt;
	
}

//------------------ for dynamic allocation ----------------
//--------------------------------------------------
// return: SUCCESS --> Successful
//       : Others  --> Fail
//--------------------------------------------------
int Get_NAS_Share_Info_Ex(NAS_SHARE_INFO *share_ptr) 
{
	int  ret;

	skip_white_space_start(share_ptr->share_name);
	stripe_white_space_tail(share_ptr->share_name);
	remove_successive_white_space(share_ptr->share_name);
  	
  	//---- Get comment -----
  	if ((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_ptr->share_name,"comment",
  		share_ptr->comment,sizeof(share_ptr->comment))) !=SUCCESS ) {
     		strcpy(share_ptr->comment,"");
	}
  	//---- Get path -----
  	if ((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_ptr->share_name,"path",
  		share_ptr->path,sizeof(share_ptr->path))) !=SUCCESS ) {
#ifdef DEBUG
       		printf("Get %s path Fail \n",share_ptr->path);
#endif
    		return ret;  
  	}

  	//---- Get Read List -----
	ret = Get_NAS_User_List_For_Share_Ex(
		share_ptr->share_name, 'R',
		&(share_ptr->read_list_ptr));

	if ( ret < 0 ) {
#ifdef DEBUG
		printf("Get_NAS_User_List_For_Share('R'), ret=%d \n",ret);
#endif
		return ret;
	}
	else share_ptr->read_list_cnt = ret;

	//---- Get Write List -----
	ret=Get_NAS_User_List_For_Share_Ex(
		share_ptr->share_name,'W',
		&(share_ptr->write_list_ptr));

	if ( ret < 0 ) {
#ifdef DEBUG
		printf("Get_NAS_User_List_For_Share('W'), ret=%d \n",ret);
#endif
		 return ret;
	}
	else share_ptr->write_list_cnt = ret;

	return SUCCESS;
}

//--------------------------------------------------------------------------
// Get Read/Write List members                                
// action :  'R'=Read List,  'W'=Write List  'I'=invald users   
// Return :  >0			    --> if user_list = NULL or list_cnt=0 , return required list count.
//	     ERROR_OPEN_FILE	    --> error open smb.conf
//--------------------------------------------------------------------------

int Get_NAS_User_List_For_Share_Ex(char *old_share_name, char security_id,
		char (**user_list_ptr)[USER_GROUP_NAME_LENGTH]) 
{
	char field_name[30];
	int  ret = SUCCESS,i=1;
	char share_name[SHARE_NAME_LENGTH];
	char (*user_list)[USER_GROUP_NAME_LENGTH]=NULL;
	char *member=NULL;

#ifdef DEBUG
	printf("Get_NAS_User_List_For_Share_Ex(%s,%c)\n",
		old_share_name, security_id);
#endif
    	strcpy(share_name,old_share_name);

	Conf_Case_Sensitive(0);

	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);
	
	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");
#ifdef DEBUG
	printf("Get_NAS_User_List_For_Share_Ex Step 1\n");
	printf("Conf_Get_Field: path=%s, share=%s, field_name=%s\n",
		SAMBA_CONF_PATH, share_name, field_name);
#endif
	i=1;
	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0))>0 ) {
#ifdef DEBUG
		printf("Get_NAS_User_List_For_Share_Ex Step 2\n");
		printf("Conf_Get_Field: share_name = %s, field_name = %s, ret = %d\n", share_name, field_name, ret);
#endif
		member=(char*)calloc(1,ret*sizeof(char));
		user_list = (char (*)[USER_GROUP_NAME_LENGTH]) calloc(1,
				(USER_GROUP_NAME_LENGTH)*sizeof(char));

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
#ifdef DEBUG
		printf("Get_NAS_User_List_For_Share_Ex Step 3\n");
		printf("member=%s\n", member);
#endif		
		while( Get_String_Field(member,i,',',
			user_list[i-1],(int)USER_GROUP_NAME_LENGTH)==SUCCESS){
			trim_chars_from_string(user_list[i-1], '\"'); // Catherine 2002/02/05
#ifdef DEBUG
			printf("member[%d]=%s\n", i-1, user_list[i-1]);
#endif
			i++;
			user_list = (char (*)[USER_GROUP_NAME_LENGTH])
				realloc((void*)user_list,
				i*(USER_GROUP_NAME_LENGTH)*sizeof(char));
		}
#ifdef DEBUG
		printf("Get_NAS_User_List_For_Share_Ex Step 4\n");
#endif
		free(member);
		member = NULL;
		if (2 == i) { //prevent null list
#ifdef DEBUG
			printf("Get_NAS_User_List_For_Share_Ex Step 4.1\n");
			printf("First Item : %s\n", user_list[0]);
#endif
			if(strcmp(user_list[0],"")==0) {
				i = 1;
				free(user_list);
				user_list = NULL;
			}
		}
		else { // remove the last useless string
#ifdef DEBUG
			printf("Get_NAS_User_List_For_Share_Ex Step 4.2\n");
#endif
			user_list = (char (*)[USER_GROUP_NAME_LENGTH])
				realloc((void*)user_list,
				(i-1)*(USER_GROUP_NAME_LENGTH)*sizeof(char));
		}
#ifdef DEBUG
		printf("Get_NAS_User_List_For_Share_Ex Step 5\n");
#endif
	}
	ret = i-1;
	*user_list_ptr = user_list;
#ifdef DEBUG
	if (ret>0) {
		for (i=0; i<ret; i++) {
			printf("user_list(%d): %s\n", i, user_list[i]);
			printf("*user_list_ptr(%d): %s\n",i, (*user_list_ptr)[i]);
		}
	}
#endif
	return ret;
}

//---------------------------------------------------------------------------//
// Get all share list for some path, if pathname include filename, set file_flag
// to 1.
//---------------------------------------------------------------------------//

int Get_Nas_Share_List_For_Path_Ex(char *path,
		SHARE_NAME_LIST **share_name_list_ptr,int file_flag)
{
	
	SECTION_INFO *s_ptr;
	int i,sh_cnt = 0,share_no = 0;
  	char *ptr,*tmp_path;
	SHARE_NAME_LIST *share_name_list=NULL;
	NAS_SHARE_INFO share_info;

  	tmp_path = strdup(path);

  	if(file_flag == 1) {
  		ptr = strrchr(tmp_path,'/');
  		*ptr = '\0';
  	}

	share_no = Get_NAS_Share_List_Ex(&s_ptr);
  	if ( share_no <= 0) return share_no;

	for(i=0;i<share_no;i++) {
		
		strcpy(share_info.share_name,
			(s_ptr+i)->section_name);

		Get_NAS_Share_Info_Ex(&share_info);

		if(compare_string(share_info.path,tmp_path) == 0) {
			share_name_list = (SHARE_NAME_LIST *) realloc(
					(void*)share_name_list,
					(sh_cnt+1)*sizeof(SHARE_NAME_LIST));
			strcpy((share_name_list+sh_cnt)->share_name,
				share_info.share_name);
			sh_cnt++;
		}
	}
	free(tmp_path);
	free(s_ptr);
	if (share_info.read_list_cnt>0) free(share_info.read_list_ptr);
	if (share_info.write_list_cnt>0) free(share_info.write_list_ptr);
	*share_name_list_ptr = share_name_list;
	return sh_cnt;
	
}

//---------------------------------------------------------
// return nnn: The Number of Share he has RO or RW right
//        other: Fail
//---------------------------------------------------------
int Get_NAS_User_Accessible_Share_List_Ex(char *user_name, SHARE_NAME_LIST **share_name_list_ptr)
{
	int  i,j,m, ret, share_no, ret_no=0,invalid_cnt=0;
	SECTION_INFO *s_ptr;
	char (*member)[USER_GROUP_NAME_LENGTH]=NULL;
	char (*invalid_member)[USER_GROUP_NAME_LENGTH]=NULL;
  	SHARE_NAME_LIST *share_name_list=NULL;
	char (*t_mem)[USER_NAME_LENGTH]=NULL;

#ifdef DEBUG
	printf("Before Get_NAS_Share_List_Ex (%s)\n", user_name);
#endif
  	share_no = Get_NAS_Share_List_Ex(&s_ptr);
#ifdef DEBUG
	printf("After Get_NAS_Share_List_Ex : %d\n", share_no);
#endif
  	if ( share_no <= 0) return share_no;
  
  	for (i=0; i<share_no; i++) {
  		int find_flag=0;
 
		invalid_cnt = Get_NAS_User_List_For_Share_Ex(
			s_ptr[i].section_name,'I',
			&invalid_member);
#ifdef DEBUG
		printf("Get_NAS_User_List_For_Share_Ex: invalid_cnt=%d\n",
			invalid_cnt);
#endif
    		for (m=0; m<2; m++) {
  			char access;
			if ( find_flag ) break;

			if (m==0)  access='R';
			else access='W';
#ifdef DEBUG
       			printf("Get_NAS_User_List_For_Share_Ex(%s, mode=%c)\n",
				s_ptr[i].section_name, access);
#endif

			ret = Get_NAS_User_List_For_Share_Ex(
				s_ptr[i].section_name, access,
				&member);
#ifdef DEBUG
       			printf("List count=%d\n", ret);
#endif

			for (j=0; j<ret; j++) {
	    			if ( find_flag ) break;
#ifdef DEBUG
	    			printf("member[%02d]='%s' \n",j,member[j]);
#endif

	    			if (member[j][0]=='@') { //-- a group
					int mem_cnt, k,pp,invalid_flag = 0;
#ifdef DEBUG
					printf("Before Get_NAS_User_List_For_Group_Ex (%s)\n", &member[j][1]);
#endif
	       				mem_cnt = Get_NAS_User_List_Of_Group_Ex(
	       					&member[j][1],
	       					&t_mem);
#ifdef DEBUG
					printf("Get_NAS_User_List_For_Group_Ex: %d members\n", mem_cnt);
#endif
	       				
	       				for(pp=0;pp<invalid_cnt;pp++) {
#ifdef DEBUG
						printf("Is an invalid user? invalid_member[%d]=%s\n", pp, invalid_member[pp]);
#endif
	       					if (invalid_member[pp][0]=='@') {
#ifdef DEBUG
							printf("User Belongs To Group? user_name=%s, group=%s\n", user_name,&invalid_member[pp][1]);
#endif
	       						if(User_Belongs_To_Group(user_name, &invalid_member[pp][1])) {
#ifdef DEBUG
								printf("Yes\n");
#endif
	       							invalid_flag = 1;
	       							break;
	       						}
	       					}
	       					else if (compare_case_string(user_name,invalid_member[pp])==0) {
#ifdef DEBUG
						  printf("%s is invalid\n", user_name);
#endif
	       						invalid_flag = 1;
	       						break;
	       					}
	       				}	
	       			
#ifdef DEBUG
					printf("Check if is a member?\n");
#endif
	       				for (k=0; k<mem_cnt; k++) {
		   				if ((compare_case_string(t_mem[k],user_name)==0) && (invalid_flag != 1)) {
							share_name_list = (SHARE_NAME_LIST*) realloc((void*)share_name_list,
								(ret_no+1)*sizeof(SHARE_NAME_LIST));
		      					strcpy(share_name_list[ret_no].share_name,s_ptr[i].section_name);
		     			 		ret_no++;
		      					find_flag = 1;
#ifdef DEBUG
					                printf("Yeah..\n");
#endif
		      					break;
		   				}
	       				}
					if (mem_cnt>0)
	       					free(t_mem);
					t_mem=NULL;
	    			}
	    			else { //--a user	
#ifdef DEBUG
					printf("Check if the same person?\n");
#endif
	       				if (compare_case_string(member[j],user_name)==0 ) {
#ifdef DEBUG
						printf("What Right?\n");
#endif
						if(Get_NAS_User_Security_For_Share(user_name,s_ptr[i].section_name) == SHARE_NOACCESS)
							continue;
						share_name_list = (SHARE_NAME_LIST*) realloc((void*)share_name_list,
								(ret_no+1)*sizeof(SHARE_NAME_LIST));

		  				strcpy(share_name_list[ret_no].share_name,s_ptr[i].section_name);
		 	 			ret_no++;
		  				find_flag = 1;
		  				break;
	       				}
	    			}
			}    
#ifdef DEBUG
			printf("Before freeing member: count = %d....\n", ret);
#endif
			if (ret>0)
		    		free(member);
			member = NULL;
  		}
#ifdef DEBUG
		printf("Before freeing invalid_member: count=%d....\n", invalid_cnt);
#endif
		if (invalid_cnt>0)
		    	free(invalid_member);
		invalid_member = NULL;
  	}	
#ifdef DEBUG
	printf("Before freeing s_ptr....\n");
#endif
  	free(s_ptr);
  	*share_name_list_ptr = share_name_list;
	return  ret_no;
}

#ifdef	LARGE_NO_OF_USERS_SUPPORT

int Add_NAS_User_List_For_Share_Ex(char *old_share_name, char security_id,
	char (*u_list)[USER_GROUP_NAME_LENGTH], int u_cnt, BOOL update_flash)
{
	char field_name[30],tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,i=0,j=1;
	int  exist_flag = 0;
    	char share_name[SHARE_NAME_LENGTH];

    	strcpy(share_name,old_share_name);

    	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);

	Conf_Case_Sensitive(0);

	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");

	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret]; //oringinal member string
		char new_member[ret+u_cnt*(USER_GROUP_NAME_LENGTH)]; // new member string

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		strcpy(new_member,member);
		for (i=0;i<u_cnt;i++) {
			j=1;
			exist_flag = 0;
			while( Get_String_Field(new_member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
				trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
				if(compare_case_string(tmp_usr_name,u_list[i]) == 0) {
					exist_flag = 1;
					break;
				}	
				j++;
			}
			if ( 0 == exist_flag ) {
				strcpy(tmp_usr_name, u_list[i]);
				insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);
				Insert_String_Field( new_member, 0, ',', tmp_usr_name,
					new_member, ret+u_cnt*(USER_GROUP_NAME_LENGTH));
			}
		}

		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,new_member);
	}
	// valid users
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret]; //oringinal member string
		char new_member[ret+u_cnt*(USER_GROUP_NAME_LENGTH)]; // new member string
		
		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		strcpy(new_member,member);
		for (i=0;i<u_cnt;i++) {
			strcpy(tmp_usr_name, u_list[i]);
			insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);
			Insert_String_Field( new_member, 0, ',', tmp_usr_name,
				new_member, ret+u_cnt*(USER_GROUP_NAME_LENGTH));
		}
		
		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",new_member);
	}

	Refresh_Connection_For_NAS_Share(share_name);
	//if (update_flash)	
	//	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);

	FTP_Create_Conf();
	return ret;
}

int Remove_NAS_User_List_For_Share_Ex(char *old_share_name, char security_id,
	char (*u_list)[USER_GROUP_NAME_LENGTH], int u_cnt, BOOL update_flash)
{
  	char field_name[30], tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,i=0,j=1;
	int  exist_flag = 0;
	char share_name[SHARE_NAME_LENGTH];

    	strncpy(share_name,old_share_name, sizeof(share_name) - 1);
    	share_name[sizeof(share_name) - 1] = 0x0;

	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);

	Conf_Case_Sensitive(0);

	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");

	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret];

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		for (i=0;i<u_cnt;i++) {
			j=1;
			exist_flag = 0;
			while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
				trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
				if(compare_case_string(tmp_usr_name,u_list[i]) == 0) {
					exist_flag = 1;
					break;
				}	
				j++;
			}

			if ( 1 == exist_flag ) 
				Remove_String_Field( member,j,',');
		}

		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,member);
	}
	// remove this user from "valid users" list as well
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret];

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		for (i=0;i<u_cnt;i++) {
			j=1;
			exist_flag = 0;
			while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
				trim_chars_from_string(tmp_usr_name, '\"');
				if(compare_case_string(tmp_usr_name,u_list[i]) == 0) {
					exist_flag = 1;
					break;
				}	
				j++;
			}
			if ( 1 == exist_flag ) 
				Remove_String_Field( member,j,',');
		}
		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",member);
	}

	Refresh_Connection_For_NAS_Share(share_name);
	//if (update_flash)
	//	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);

	FTP_Create_Conf();
	return ret;
}

int Add_NAS_User_For_Share_Ex(char *old_share_name, char security_id,
	char *user_name, BOOL update_flash)
{
	char field_name[30], tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,j=1;
	int  exist_flag = 0;
    	char share_name[SHARE_NAME_LENGTH];

    	strncpy(share_name,old_share_name, sizeof(share_name) - 1);
    	share_name[sizeof(share_name) - 1] = 0x0;

	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);

	Conf_Case_Sensitive(0);

	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");	

	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret]; //oringinal member string
		char new_member[ret+USER_GROUP_NAME_LENGTH]; // new member string

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		strcpy(new_member,member);

		j=1;
		exist_flag = 0;

		while( Get_String_Field(new_member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
			trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
			if(compare_case_string(tmp_usr_name,user_name) == 0) {
				exist_flag = 1;
				break;
			}	
			j++;
		}
			
		if ( 0 == exist_flag ) {
			strcpy(tmp_usr_name, user_name);
			insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);

			ret = Insert_String_Field( member, 0, ',', tmp_usr_name,
				new_member, ret+USER_GROUP_NAME_LENGTH);
			ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,new_member);
		}
		else
			return ERROR_ALREADY_EXISTED;
	}
	// valid users list
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret]; //oringinal member string
		char new_member[ret+USER_GROUP_NAME_LENGTH]; // new member string

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		strcpy(new_member,member);

		strcpy(tmp_usr_name, user_name);
		insert_headntail_into_string(tmp_usr_name, USER_GROUP_NAME_LENGTH);

		ret = Insert_String_Field( member, 0, ',', tmp_usr_name,
			new_member, ret+USER_GROUP_NAME_LENGTH);

		ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",new_member);
	}

	Refresh_Connection_For_NAS_Share(share_name);
	//if (update_flash)
	//	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);

	FTP_Create_Conf();
	return ret;
}

int Remove_NAS_User_For_Share_Ex(char *old_share_name, char security_id,
	char *user_name, BOOL update_flash)
{
  	char field_name[30],tmp_usr_name[USER_GROUP_NAME_LENGTH];
	int  ret,j=1;
	int  exist_flag = 0;
    	char share_name[SHARE_NAME_LENGTH];

    	strncpy(share_name, old_share_name, sizeof(share_name) - 1);
	share_name[sizeof(share_name) - 1] = 0x0;

	skip_white_space_start(share_name);
	stripe_white_space_tail(share_name);
	remove_successive_white_space(share_name);

	Conf_Case_Sensitive(0);

	if ( security_id=='R' || security_id=='r')
		strcpy(field_name,"read list");
	else if ( security_id=='W' || security_id=='w')
		strcpy(field_name,"write list");
	else
		strcpy(field_name,"invalid users");

	if((ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,NULL,0)) >=0 ) {//found this field
		char member[ret];

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,field_name,member,ret);
		j=1;
		exist_flag = 0;
		while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
			trim_chars_from_string(tmp_usr_name, '\"'); // Catherine 2002/02/05
			if(compare_case_string(tmp_usr_name,user_name) == 0) {
				exist_flag = 1;
				break;
			}	
			j++;
		}

		if ( 1 == exist_flag ) { 
			Remove_String_Field( member,j,',');
			ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,field_name,member);
		}
		else
			return ERROR_NOT_FOUND;
	}
	// remove this user from "valid users" list as well
	if ((security_id == 'r' || security_id == 'R' || security_id == 'w' || security_id == 'W') &&
		(ret=Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",NULL,0)) >=0 )
	{
		char member[ret];

		Conf_Get_Field(SAMBA_CONF_PATH,share_name,"valid users",member,ret);
		j=1;
		exist_flag = 0;
		while( Get_String_Field(member,j,',',tmp_usr_name,USER_GROUP_NAME_LENGTH) == SUCCESS ) {
			trim_chars_from_string(tmp_usr_name, '\"');
			if(compare_case_string(tmp_usr_name,user_name) == 0) {
				exist_flag = 1;
				break;
			}	
			j++;
		}
			
		if ( 1 == exist_flag ) { 
			Remove_String_Field( member,j,',');
			ret = Conf_Set_Field(SAMBA_CONF_PATH,share_name,"valid users",member);
		}
	}

	Refresh_Connection_For_NAS_Share(share_name);
	//if (update_flash)
	//	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);
	FTP_Create_Conf();
	return ret;
}

int Remove_User_Share_Privilege_Ex(char *usr_name, BOOL update_flash)
{
        int i,j, count, ret;
        char (*list_ptr)[USER_GROUP_NAME_LENGTH];
        char *s_ptr;
	SECTION_INFO *share_list;

	// get all shares....
	count = Get_NAS_Share_List((SECTION_INFO*)NULL, 0);
	if (count >0){
		share_list = (SECTION_INFO *) calloc ( count, sizeof(SECTION_INFO));
		if (share_list == NULL)
			return ERROR_OUT_OF_MEMORY;
		Get_NAS_Share_List(share_list,count);
	}
	else
		return count;

	for (i = 0; i < count; i++) {
		/*--- get read List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name,'R',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name,'R',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(usr_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share_Ex(
						share_list[i].section_name, 'R',
						usr_name, update_flash);
					break;
				}
			}
			free(list_ptr);		
		}
		/*--- get write List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name,'W',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name,'W',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(usr_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share_Ex(
						share_list[i].section_name, 'W',
						usr_name, update_flash);
					break;
				}
			}
			free(list_ptr);
		}
		/*--- get invalid user List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH]) calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(usr_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share_Ex(
						share_list[i].section_name, 'I',
						usr_name, update_flash);
					break;
				}
			}
			free(list_ptr);	
		}
        }
        free( share_list);
        //if (update_flash)
        //	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);

	FTP_Create_Conf();
	return SUCCESS;
}

int Remove_Multiple_User_Share_Privilege_Ex(
	char (*usr_list)[USER_NAME_LENGTH], int count,
	BOOL update_flash, char* sts_file)
{
        int i, j, cnt, ret;
        char (*list_ptr)[USER_NAME_LENGTH];
	SECTION_INFO *share_list = NULL;
	PROGRESS_STATUS mystatus;
	PROGRESS_LIST*	pp;
	char tmp_smb_conf[BUF_SIZE], cmd[BUF_SIZE];
	char tmp_usr_name[USER_NAME_LENGTH];

	if (sts_file) {
		pp = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		mystatus.plist = pp;
		if (pp == NULL) {
			ret = ERROR_OUT_OF_MEMORY;
			mystatus.status = ret;
			Report_Progress_To_File(sts_file, &mystatus);
			return ret;
		}
		pp = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		if (pp == NULL) {
			ret = ERROR_OUT_OF_MEMORY;
			mystatus.status = ret;
			free(mystatus.plist);
			mystatus.plist = NULL;

			Report_Progress_To_File(sts_file, &mystatus);
			return ret;
		}

		pp->progress = 0;
		pp->next = NULL;
		mystatus.plist->progress = 0;
		mystatus.plist->next = pp;
		mystatus.status = S_PROCESSING;
		Report_Progress_To_File(sts_file, &mystatus);
	}
	if (usr_list == NULL || count == 0) {
		ret = ERROR_FAIL;
		goto remove_multiple_user_share_privilege_ex_out;
	}
	list_ptr = usr_list;

	// get all shares....
	cnt = Get_NAS_Share_List((SECTION_INFO*)NULL, 0);
	if (cnt >0){
		share_list = (SECTION_INFO *) calloc ( cnt, sizeof(SECTION_INFO));
		if (share_list == NULL) {
			ret = ERROR_OUT_OF_MEMORY;
			goto remove_multiple_user_share_privilege_ex_out;
		}
		Get_NAS_Share_List(share_list,cnt);
	}
	else {
		ret = cnt;
		goto remove_multiple_user_share_privilege_ex_out;
	}

	// lock SAMBA_CONF_PATH
	snprintf(tmp_smb_conf, sizeof(tmp_smb_conf) - 1, "%s.tmp", SAMBA_CONF_PATH);
	sprintf(cmd, "cp %s %s", SAMBA_CONF_PATH, tmp_smb_conf);
	system(cmd);

	NAS_File_Lock(SAMBA_CONF_PATH, O_RDONLY);
	ret = SUCCESS;
	for (i = 0; i < cnt; i++) {
		for (j = 0; j < count; j++) {
			int len, exist_flag, k;

			if (sts_file) {
				pp = mystatus.plist;
				pp->progress = i*100/cnt;
				pp = pp->next;
				pp->progress = j*100/count;
				Report_Progress_To_File(sts_file, &mystatus);
			}

			/*--- get read List ---*/
			if ((len = Conf_Get_Field(tmp_smb_conf,
				share_list[i].section_name, "read list", NULL, 0)) >=0 )
			{	//found this field
				char member[len];

				Conf_Get_Field(tmp_smb_conf, share_list[i].section_name,
					"read list", member,len);
				k=1;
				exist_flag = 0;
				while( Get_String_Field(member, k, ',',
					tmp_usr_name, USER_GROUP_NAME_LENGTH) == SUCCESS )
				{
					trim_chars_from_string(tmp_usr_name, '\"');
					if(compare_case_string(tmp_usr_name, list_ptr[j]) == 0) {
						exist_flag = 1;
						break;
					}	
					k++;
				}

				if ( 1 == exist_flag ) { 
					Remove_String_Field( member,k,',');
					ret = Conf_Set_Field(tmp_smb_conf,
						share_list[i].section_name,
						"read list", member);
				}
			}
			/*--- get write List ---*/
			if ((len = Conf_Get_Field(tmp_smb_conf,
				share_list[i].section_name, "write list", NULL, 0)) >=0 )
			{	//found this field
				char member[len];

				Conf_Get_Field(tmp_smb_conf, share_list[i].section_name,
					"write list", member,len);
				k=1;
				exist_flag = 0;
				while( Get_String_Field(member, k, ',',
					tmp_usr_name, USER_GROUP_NAME_LENGTH) == SUCCESS )
				{
					trim_chars_from_string(tmp_usr_name, '\"');
					if(compare_case_string(tmp_usr_name,list_ptr[j]) == 0) {
						exist_flag = 1;
						break;
					}	
					k++;
				}

				if ( 1 == exist_flag ) { 
					Remove_String_Field( member,k,',');
					ret = Conf_Set_Field(tmp_smb_conf,
						share_list[i].section_name,
						"write list", member);
				}
			}
			/*--- get invalid users ---*/
			if ((len = Conf_Get_Field(tmp_smb_conf,
				share_list[i].section_name, "invalid users", NULL, 0)) >=0 )
			{	//found this field
				char member[len];

				Conf_Get_Field(tmp_smb_conf, share_list[i].section_name,
					"invalid users", member,len);
				k=1;
				exist_flag = 0;
				while( Get_String_Field(member, k, ',',
					tmp_usr_name, USER_GROUP_NAME_LENGTH) == SUCCESS )
				{
					trim_chars_from_string(tmp_usr_name, '\"');
					if(compare_case_string(tmp_usr_name,list_ptr[j]) == 0) {
						exist_flag = 1;
						break;
					}	
					k++;
				}

				if ( 1 == exist_flag ) { 
					Remove_String_Field( member,k,',');
					ret = Conf_Set_Field(tmp_smb_conf,
						share_list[i].section_name,
						"invalid users", member);
				}
			}
		}
	}

	sprintf(cmd, "mv %s %s -f", tmp_smb_conf, SAMBA_CONF_PATH);
	NAS_File_Unlock(SAMBA_CONF_PATH);
	if (system(cmd) != 0) {
		ret = ERROR_FAIL;
		goto remove_multiple_user_share_privilege_ex_out;
	}
        //if (update_flash && ret == SUCCESS) 
        //	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);
	FTP_Create_Conf();

remove_multiple_user_share_privilege_ex_out:
	if (share_list != NULL) free(share_list);
	unlink(tmp_smb_conf);

	if (sts_file) {
		mystatus.status = ret;
		pp = mystatus.plist;
		pp->progress = 100;
		pp = pp->next;
		pp->progress = 100;

		Report_Progress_To_File(sts_file, &mystatus);

		pp = mystatus.plist;
		while (pp) {
			mystatus.plist = pp->next;
			free(pp);
			pp = mystatus.plist;
		}
	}

	return SUCCESS;
}

int Remove_Group_Share_Privilege_Ex(char *grp_name, BOOL update_flash)
{
        int i,j, count, ret;
        char (*list_ptr)[USER_GROUP_NAME_LENGTH];
        char *s_ptr;
        char tmp_grp_name[strlen(grp_name)+2];
	SECTION_INFO *share_list;

	strcpy(tmp_grp_name,"@");
	strcat(tmp_grp_name,grp_name);

	// get all shares....
	count = Get_NAS_Share_List((SECTION_INFO *)NULL, 0);
	if (count >0){
		share_list = (SECTION_INFO *) calloc ( count, sizeof(SECTION_INFO));
		if (share_list == NULL)
			return ERROR_OUT_OF_MEMORY;
		Get_NAS_Share_List(share_list,count);
	}
	else
		return count;

	for (i = 0; i < count; i++) {
		/*--- get read List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'R',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH])
					calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'R',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(tmp_grp_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share_Ex(
						share_list[i].section_name, 'R',
						tmp_grp_name, update_flash);
					break;
				}
			}
			free(list_ptr);
		}
		/*--- get write List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'W',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH])
					calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'W',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(tmp_grp_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share_Ex(
						share_list[i].section_name, 'W',
						tmp_grp_name, update_flash);
					break;
				}
			}
			free(list_ptr);
		}
		/*--- get invalid user List ---*/
		ret = Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',NULL, 0);
		if ( ret > 0 ) {
			list_ptr = (char (*)[USER_GROUP_NAME_LENGTH])
					calloc (ret,USER_GROUP_NAME_LENGTH);
			if (list_ptr == NULL) {
				free( share_list);	
				return ERROR_OUT_OF_MEMORY;
			}
			Get_NAS_User_List_For_Share(share_list[i].section_name, 'I',list_ptr, ret);
			for (j = 0; j < ret; j++) {
				s_ptr = list_ptr[j];
				if (compare_case_string(tmp_grp_name, s_ptr) == 0) {
					Remove_NAS_User_For_Share_Ex(
						share_list[i].section_name, 'I',
						tmp_grp_name, update_flash);
					break;
				}
			}
			free(list_ptr);
		}
        }
        free( share_list);
	//if (update_flash)
        //	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);
	FTP_Create_Conf();
	return SUCCESS;
}

int Remove_Multiple_Group_Share_Privilege_Ex(char (*grp_list)[GROUP_NAME_LENGTH],
	int count, BOOL update_flash, char* sts_file)
{
	int i,j, cnt, ret;
	char (*list_ptr)[GROUP_NAME_LENGTH+2] = NULL;
	SECTION_INFO *share_list = NULL;
	PROGRESS_STATUS mystatus;
	PROGRESS_LIST*	pp;
	char tmp_smb_conf[BUF_SIZE], cmd[BUF_SIZE];

	if (sts_file) {
		pp = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		mystatus.plist = pp;
		if (pp == NULL) {
			ret = ERROR_OUT_OF_MEMORY;
			mystatus.status = ret;
			Report_Progress_To_File(sts_file, &mystatus);
			return ret;
		}
		pp = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		if (pp == NULL) {
			ret = ERROR_OUT_OF_MEMORY;
			mystatus.status = ret;
			free(mystatus.plist);
			mystatus.plist = NULL;

			Report_Progress_To_File(sts_file, &mystatus);
			return ret;
		}

		pp->progress = 0;
		pp->next = NULL;
		mystatus.plist->progress = 0;
		mystatus.plist->next = pp;
		mystatus.status = S_PROCESSING;
		Report_Progress_To_File(sts_file, &mystatus);
	}

	if (grp_list == NULL || count == 0) {
		ret = ERROR_FAIL;
		goto remove_multiple_group_share_privilege_ex_out;
	}

	// get all shares....
	cnt = Get_NAS_Share_List((SECTION_INFO *)NULL, 0);
	if (cnt >0){
		share_list = (SECTION_INFO *) calloc ( cnt, sizeof(SECTION_INFO));
		if (share_list == NULL) {
			ret = ERROR_OUT_OF_MEMORY;
			goto remove_multiple_group_share_privilege_ex_out;
		}
		Get_NAS_Share_List(share_list,cnt);
	}
	else {
		ret = cnt;
		goto remove_multiple_group_share_privilege_ex_out;
	}

	list_ptr = (char (*)[GROUP_NAME_LENGTH+2]) calloc (count,GROUP_NAME_LENGTH+2);
	if (list_ptr == NULL) {
		ret = ERROR_OUT_OF_MEMORY;
		goto remove_multiple_group_share_privilege_ex_out;
	} 
	for (i = 0; i < count ; i++){
		sprintf(list_ptr[i], "@%s", grp_list[i]);
	}

	// lock SAMBA_CONF_PATH
	snprintf(tmp_smb_conf, sizeof(tmp_smb_conf) - 1, "%s.tmp", SAMBA_CONF_PATH);
	sprintf(cmd, "cp %s %s", SAMBA_CONF_PATH, tmp_smb_conf);
	system(cmd);

	NAS_File_Lock(SAMBA_CONF_PATH, O_RDWR);

	ret = SUCCESS;
	for (i = 0; i < cnt; i++) {
		for (j = 0; j < count; j++) {
			int k, len, exist_flag;
			char tmp_grp_name[USER_GROUP_NAME_LENGTH];

			if (sts_file) {
				pp = mystatus.plist;
				pp->progress = i*100/cnt;
				pp = pp->next;
				pp->progress = j*100/count;
				mystatus.status = S_PROCESSING;
				Report_Progress_To_File(sts_file, &mystatus);
			}

			/*--- get read List ---*/
			if ((len = Conf_Get_Field(tmp_smb_conf,
				share_list[i].section_name, "read list", NULL, 0)) >=0 )
			{	//found this field
				char member[len];

				Conf_Get_Field(tmp_smb_conf, share_list[i].section_name,
					"read list", member,len);
				k=1;
				exist_flag = 0;
				while( Get_String_Field(member, k, ',',
					tmp_grp_name, USER_GROUP_NAME_LENGTH) == SUCCESS )
				{
					trim_chars_from_string(tmp_grp_name, '\"');
					if(compare_case_string(tmp_grp_name, list_ptr[j]) == 0)
					{
						exist_flag = 1;
						break;
					}	
					k++;
				}

				if ( 1 == exist_flag ) { 
					Remove_String_Field( member,k,',');
					ret = Conf_Set_Field(tmp_smb_conf,
						share_list[i].section_name,
						"read list", member);
				}
			}
			/*--- get write List ---*/
			if ((len = Conf_Get_Field(tmp_smb_conf,
				share_list[i].section_name, "write list", NULL, 0)) >=0 )
			{	//found this field
				char member[len];

				Conf_Get_Field(tmp_smb_conf, share_list[i].section_name,
					"write list", member,len);
				k=1;
				exist_flag = 0;
				while( Get_String_Field(member, k, ',',
					tmp_grp_name, USER_GROUP_NAME_LENGTH) == SUCCESS )
				{
					trim_chars_from_string(tmp_grp_name, '\"');
					if(compare_case_string(tmp_grp_name,
						list_ptr[j]) == 0)
					{
						exist_flag = 1;
						break;
					}	
					k++;
				}

				if ( 1 == exist_flag ) { 
					Remove_String_Field( member,k,',');
					ret = Conf_Set_Field(tmp_smb_conf,
						share_list[i].section_name,
						"write list", member);
				}
			}
			/*--- get invalid users ---*/
			if ((len = Conf_Get_Field(tmp_smb_conf,
				share_list[i].section_name, "invalid users", NULL, 0)) >=0 )
			{	//found this field
				char member[len];

				Conf_Get_Field(tmp_smb_conf, share_list[i].section_name,
					"invalid users", member,len);
				k=1;
				exist_flag = 0;
				while( Get_String_Field(member, k, ',',
					tmp_grp_name, USER_GROUP_NAME_LENGTH) == SUCCESS )
				{
					trim_chars_from_string(tmp_grp_name, '\"');
					if(compare_case_string(tmp_grp_name,
						list_ptr[j]) == 0)
					{
						exist_flag = 1;
						break;
					}	
					k++;
				}

				if ( 1 == exist_flag ) { 
					Remove_String_Field( member,k,',');
					ret = Conf_Set_Field(tmp_smb_conf,
						share_list[i].section_name,
						"invalid users", member);
				}
			}
		}
        }

	sprintf(cmd, "mv %s %s -f", tmp_smb_conf, SAMBA_CONF_PATH);
	NAS_File_Unlock(SAMBA_CONF_PATH);
	if (system(cmd) != 0) {
		ret = ERROR_FAIL;
		goto remove_multiple_group_share_privilege_ex_out;
	}
        //if (update_flash && ret == SUCCESS)
        //	Update_Flash_Data_Ex(SAMBA_CONF_PATH, TRUE);
	FTP_Create_Conf();

remove_multiple_group_share_privilege_ex_out:        
        if (share_list) free(share_list);
        if (list_ptr) free(list_ptr);

	if (sts_file) {
		mystatus.status = ret;
		pp = mystatus.plist;
		pp->progress = 100;
		pp = pp->next;
		pp->progress = 100;

		Report_Progress_To_File(sts_file, &mystatus);

		pp = mystatus.plist;
		while (pp) {
			mystatus.plist = pp->next;
			free(pp);
			pp = mystatus.plist;
		}
	}
	return ret;
}

#endif	// LARGE_NO_OF_USERS_SUPPORT
