//****************************************************************************
//	Copyright (c) 2000	ICP Electronics Inc.	All right reserved.
//	FILE:
//			nassys.c
//
//	Abstract:
//			1. Provide called library for CGI
//
//	HISTORY:
//			03/19/2001		Ethan Wang -- Created
//*****************************************************************************

//#define DEBUG_TIMEZONE

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <sys/ioctl.h>
#include <string.h>
#include "hwmon.h"
#include "w83977fa.h"

#include <config.h>
#include "Util.h"
#include <nassys.h>

static int	DiffTime_Initialized = 0;
static int	DiffTime_Sign, DiffTime_Day, DiffTime_Hour, DiffTime_Minute;

// "Local Time" - "Greenwich Time"
void calc_diff_time()
{
	time_t		timeval;
	struct tm	*ptm;
	struct tm	gm, local;
	int		diff_minutes;

	if ( DiffTime_Initialized )
		return;

	DiffTime_Initialized = 1;

	time(&timeval);
	ptm = localtime(&timeval);
	local = *ptm;
	ptm = gmtime(&timeval);
	gm = *ptm;

#ifdef DEBUG_TIMEZONE
	printf("localtime = %02d-%02d:%02d, GMT = %02d-%02d:%02d\n",
		local.tm_wday, local.tm_hour, local.tm_min,
		gm.tm_wday, gm.tm_hour, gm.tm_min);
#endif

	diff_minutes = 0;
	diff_minutes += ((local.tm_wday + 1) % 7 == gm.tm_wday) ? -1440 : 0;
	diff_minutes += (local.tm_wday == (gm.tm_wday + 1) % 7) ? 1440 : 0;

	diff_minutes += ((local.tm_hour < gm.tm_hour) ?
		(((local.tm_hour + 24) - gm.tm_hour) * 60 - 1440) :
		((local.tm_hour - gm.tm_hour) * 60));
	
	diff_minutes += ((local.tm_min < gm.tm_min) ?
		(((local.tm_min + 60) - gm.tm_min) - 60) :
		(local.tm_min - gm.tm_min));

	if ( diff_minutes >= 0 )
	{
		DiffTime_Sign = 1;
	}
	else
	{
		DiffTime_Sign = -1;
		diff_minutes = -diff_minutes;
	}

	DiffTime_Day = diff_minutes / 1440;
	diff_minutes = diff_minutes % 1440;
	DiffTime_Hour = diff_minutes / 60;
	DiffTime_Minute = diff_minutes % 60;

#ifdef DEBUG_TIMEZONE
	printf("Time difference = %02d:%02d:%02d\n", DiffTime_Day, DiffTime_Hour, DiffTime_Minute);
#endif
}

void add_time(PPOWER_SCHEDULE pwrsch)
{
	int carry;

	pwrsch->minute = pwrsch->minute + DiffTime_Minute;
	carry = pwrsch->minute >= 60 ? 1 : 0;
	pwrsch->minute =pwrsch->minute % 60;

	pwrsch->hour = pwrsch->hour + DiffTime_Hour + carry;
	carry = pwrsch->hour >= 24 ? 1 : 0;
	pwrsch->hour = pwrsch->hour % 24;

	pwrsch->wday = pwrsch->wday + DiffTime_Day + carry;
	pwrsch->wday = pwrsch->wday % 7;
}

void sub_time(PPOWER_SCHEDULE pwrsch)
{
	int borrow;

	borrow = 0;
	if ( pwrsch->minute >= DiffTime_Minute )
	{
		pwrsch->minute -= DiffTime_Minute;
	}
	else
	{
		pwrsch->minute = pwrsch->minute + 60 - DiffTime_Minute;
		borrow = 1;
	}

	if ( (pwrsch->hour - borrow) >= DiffTime_Hour )
	{
		pwrsch->hour = pwrsch->hour - DiffTime_Hour - borrow;
		borrow = 0;
	}
	else
	{
		pwrsch->hour = pwrsch->hour + 24 - DiffTime_Hour - borrow;
		borrow = 1;
	}

	if ( (pwrsch->wday - borrow) >= DiffTime_Day )
	{
		pwrsch->wday = pwrsch->wday - DiffTime_Day - borrow;
	}
	else
	{
		pwrsch->wday = pwrsch->wday + 7 - DiffTime_Day - borrow;
	}
	
}

// The function is invoked by Set_Power_On_Schedule.
void localtime_to_greenwichtime(PPOWER_SCHEDULE pwrsch)
{
	calc_diff_time();
	if ( DiffTime_Sign == 1 )
	{
		sub_time(pwrsch);
	}
	else
	{
		add_time(pwrsch);
	}
}

// The function is invoked by Get_Power_On_Schedule.
void greenwichtime_to_localtime(PPOWER_SCHEDULE pwrsch)
{
	calc_diff_time();
	if ( DiffTime_Sign == 1 )
	{
		add_time(pwrsch);
	}
	else
	{
		sub_time(pwrsch);
	}
}

int Get_Wake_On_LAN(int *enable)
{
	int	fd;
	unsigned char wakeup_src;

	fd = open(HWMON_DEV, O_RDWR);
	if ( fd < 0 )
		return FALSE;

	ioctl(fd, IOCTL_HWMON_GET_WAKE_UP_EVENT, &wakeup_src);
	*enable = ( wakeup_src & PSON_PHRI ) ? TRUE : FALSE;
	close(fd);

	return TRUE;
}

int Set_Wake_On_LAN(int enable)
{
	int	fd;
	unsigned char wakeup_src;

	fd = open(HWMON_DEV, O_RDWR);
	if ( fd < 0 )
		return FALSE;

	ioctl(fd, IOCTL_HWMON_GET_WAKE_UP_EVENT, &wakeup_src);
	if ( enable )
	{
		wakeup_src |= PSON_PHRI;
	}
	else
	{
		wakeup_src &= ~PSON_PHRI;
	}
	
	ioctl(fd, IOCTL_HWMON_SET_WAKE_UP_EVENT, &wakeup_src);
	close(fd);

	return TRUE;

}

int Get_Power_On_Schedule(PPOWER_SCHEDULE pwrsch)
{
	int	fd;
	unsigned char wakeup_src;
	WAKE_UP_DATE wakeup;

	fd = open(HWMON_DEV, O_RDWR);
	if ( fd < 0 )
		return FALSE;

	ioctl(fd, IOCTL_HWMON_GET_WAKE_UP_EVENT, &wakeup_src);
	if ( !(wakeup_src & PSON_ALARMB) )
	{
		pwrsch->type = PWRSCH_NONE;
		close(fd);
		return TRUE;
	}

	ioctl(fd, IOCTL_HWMON_GET_WAKE_UP_DATE, &wakeup);
	pwrsch->wday = wakeup.wday;
	pwrsch->hour = wakeup.hour;
	pwrsch->minute = wakeup.minute;
	switch ( wakeup.mask )
	{
	case (ALARM_HOUR | ALARM_MINUTE):
		pwrsch->type = PWRSCH_DAILY;
		break;
	case ALARM_WDAY | ALARM_HOUR | ALARM_MINUTE:
		pwrsch->type = PWRSCH_WEEKLY;
		break;
	default:
		pwrsch->type = PWRSCH_NONE;
	}

	close(fd);

	// Translate Greenwich Time to Local Time.
#ifdef DEBUG_TIMEZONE
	printf("Get_Power_On_Schedule(before): dd:hh:mm %02d:%02d:%2d\n",
		pwrsch->wday, pwrsch->hour, pwrsch->minute);
#endif
	greenwichtime_to_localtime(pwrsch);
#ifdef DEBUG_TIMEZONE
	printf("Get_Power_On_Schedule(after): dd:hh:mm %02d:%02d:%2d\n",
		pwrsch->wday, pwrsch->hour, pwrsch->minute);
#endif

	return TRUE;	
}

int Set_Power_On_Schedule(PPOWER_SCHEDULE pwrsch)
{
	int	fd;
	unsigned char	wakeup_src;
	WAKE_UP_DATE wakeup;

	fd = open(HWMON_DEV, O_RDWR);
	if ( fd < 0 )
		return FALSE;

	ioctl(fd, IOCTL_HWMON_GET_WAKE_UP_EVENT, &wakeup_src);

	switch ( pwrsch->type )
	{
	case PWRSCH_DAILY:
	case PWRSCH_WEEKLY:
		wakeup_src |= PSON_ALARMB;
		ioctl(fd, IOCTL_HWMON_SET_WAKE_UP_EVENT, &wakeup_src);
		break;
	case PWRSCH_NONE:
		wakeup_src &= ~PSON_ALARMB;
		ioctl(fd, IOCTL_HWMON_SET_WAKE_UP_EVENT, &wakeup_src);
		close(fd);
		return TRUE;
	default:
		close(fd);
		return FALSE;
	}

	// Translate Local Time to Greenwich Time.
	localtime_to_greenwichtime(pwrsch);

	wakeup.hour = pwrsch->hour;
	wakeup.minute = pwrsch->minute;
	switch ( pwrsch->type )
	{
	case PWRSCH_DAILY:
		wakeup.mask = ALARM_HOUR | ALARM_MINUTE;
		break;
	case PWRSCH_WEEKLY:
		wakeup.wday = pwrsch->wday;
		wakeup.mask = ALARM_WDAY | ALARM_HOUR | ALARM_MINUTE;
		break;
	}

	ioctl(fd, IOCTL_HWMON_SET_WAKE_UP_DATE, &wakeup);
	close(fd);

	return TRUE;
}

#define SZ_POWER_SECTION		"Power"
#define SZ_POWEROFF_TYPE_FIELD		"PowerOff Type"
#define SZ_POWEROFF_DAILY_TYPE		"Daily"
#define SZ_POWEROFF_WEEKLY_TYPE		"Weekly"
#define SZ_POWEROFF_NONE_TYPE		"None"

#define SZ_POWEROFF_DAY_FIELD		"Day"
#define SZ_POWEROFF_HOUR_FIELD		"Hour"
#define SZ_POWEROFF_MINUTE_FIELD	"Minute"

int Get_Power_Off_Schedule(PPOWER_SCHEDULE pwrsch)
{
	char buf[BUF_SIZE];
	GetProfileString(SZ_POWER_SECTION, SZ_POWEROFF_TYPE_FIELD, SZ_POWEROFF_NONE_TYPE, buf, sizeof(buf));
	if ( strcmp(buf, SZ_POWEROFF_DAILY_TYPE) == 0 )
	{
		pwrsch->type = PWRSCH_DAILY;
	}
	else if ( strcmp(buf, SZ_POWEROFF_WEEKLY_TYPE) == 0 )
	{
		pwrsch->type = PWRSCH_WEEKLY;
	}
	else
	{
		pwrsch->type = PWRSCH_NONE;
		return TRUE;
	}

	switch ( pwrsch->type )
	{
	case PWRSCH_WEEKLY:
		GetProfileString(SZ_POWER_SECTION, SZ_POWEROFF_DAY_FIELD, "0", buf, sizeof(buf));
		pwrsch->wday = (unsigned char) strtoul(buf, NULL, 10);
	case PWRSCH_DAILY:
		GetProfileString(SZ_POWER_SECTION, SZ_POWEROFF_HOUR_FIELD, "0", buf, sizeof(buf));
		pwrsch->hour = (unsigned char) strtoul(buf, NULL, 10);
		GetProfileString(SZ_POWER_SECTION, SZ_POWEROFF_MINUTE_FIELD, "0", buf, sizeof(buf));
		pwrsch->minute = (unsigned char) strtoul(buf, NULL, 10);
		break;
	}

	return TRUE;
}

int Set_Power_Off_Schedule(PPOWER_SCHEDULE pwrsch)
{
	char	buf[BUF_SIZE];

	switch ( pwrsch->type )
	{
	case PWRSCH_DAILY:
		WriteProfileString(SZ_POWER_SECTION, SZ_POWEROFF_TYPE_FIELD, SZ_POWEROFF_DAILY_TYPE);
		break;
	case PWRSCH_WEEKLY:
		WriteProfileString(SZ_POWER_SECTION, SZ_POWEROFF_TYPE_FIELD, SZ_POWEROFF_WEEKLY_TYPE);
		break;
	case PWRSCH_NONE:
		WriteProfileString(SZ_POWER_SECTION, SZ_POWEROFF_TYPE_FIELD, SZ_POWEROFF_NONE_TYPE);
		return TRUE;
	default:
		return FALSE;
	}

	switch ( pwrsch->type )
	{
	case PWRSCH_WEEKLY:
		sprintf(buf, "%d", pwrsch->wday);
		WriteProfileString(SZ_POWER_SECTION, SZ_POWEROFF_DAY_FIELD, buf);
	case PWRSCH_DAILY:
		sprintf(buf, "%d", pwrsch->hour);
		WriteProfileString(SZ_POWER_SECTION, SZ_POWEROFF_HOUR_FIELD, buf);
		sprintf(buf, "%d", pwrsch->minute);
		WriteProfileString(SZ_POWER_SECTION, SZ_POWEROFF_MINUTE_FIELD, buf);
		break;
	}

	return TRUE;
}

void Reload_Power_Off_Schedule()
{
	system("/bin/kill -HUP `/sbin/pidof pwroffd`");
}
