#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>

//////////////////////////////////////////////////////////////////
// Update History
//    10/8/2001  Jeff Huang -- Double check for UID field in nfs.conf
//    11/26/2001 Jeff Huang -- Filter invalid NFS users 
//    11/27/2001 Jeff Huang -- Add an API for removing NFS users
//                             when PDC enabled
//    02/07/2002 kw lee -- Add API for removing user's entries
//
////////////////////////////////////////////////////////////////////

#include "NAS.h"
#include "Util.h"
//#include "config.h"	// for Update_Flash_Data

#define NFS_CONF	"/etc/nfs.conf"

#define KNFST_CMD_ZERO		0x00
#define KNFST_CMD_TEST		0x01
#define KNFST_CMD_ADD		0x02
#define KNFST_CMD_DEL		0x04
#define KNFST_CMD_FND		0x08
#define KNFST_CMD_UNKNOWN	0x10
#define KNFST_CMD_CLR		0x20
#define KNFST_CMD_LST		0x40
#define KNFST_CMD_ERR		0x80

int write_proc_knfst(const char *username, NFS_USER *nfsuser, int knfst_cmd);

static void get_nfs_field(FILE *fp, char *buf, int size)
{
	int pos;	// indicate the index of the next character will be filled in "buf"
	char ch;

	if ( size <= 0 )
		return;

	// Skip white space
	do {
		if ( feof(fp) )
		{
			buf[0] = 0;
			return;
		}
		ch = fgetc(fp);
	} while ( ch == ' ' || ch == '\t' );

	// read the next field with separator ':'
	for (pos = 0 ; pos < size ; pos++)
	{
		if ( feof(fp) || ch == ':' )	// EOF or separator
		{
			buf[pos] = 0;
			return;
		}
		else if ( ch == '\n' )
		{
			ungetc('\n', fp);
			buf[pos] = 0;
			return;
		}
		else
		{
			buf[pos] = ch;
		}
		
		ch = fgetc(fp);
	}

	if ( pos >= size )
	{
		buf[size-1] = 0;
	}
}


int Load_All_NFS_Users(NFS_USER_EX *pusers, int *size)
{
	FILE *fp;
	char ch;
	char name[BUF_SIZE], uid[BUF_SIZE], type[BUF_SIZE], ipaddr[BUF_SIZE], netmask[BUF_SIZE];
	int  count;	// how many records we had read
	struct in_addr net_fromaddr, net_toaddr;
	int  namelen;
	int  buf_enough;

	fp = fopen(NFSCONF_FILE, "r");
	if ( fp == NULL )
	{
		return ERROR_OPEN_FILE;
	}

	buf_enough = 1;
	if ( pusers == NULL || *size <= 0 )
	{
		buf_enough = 0;
	}

	count = 0;
	while ( !feof(fp) )
	{
                int is_numeric=1, k;
		get_nfs_field(fp, name, BUF_SIZE);
		get_nfs_field(fp, uid, BUF_SIZE);
		get_nfs_field(fp, type, BUF_SIZE);
		get_nfs_field(fp, ipaddr, BUF_SIZE);
		get_nfs_field(fp, netmask, BUF_SIZE);
                // added by Jeff for invalid user filtering
// comment by bird, this will get trouble on counting entry number
//                if( !getpwnam(name))
//                    continue;
                // end
#ifdef DEBUG
		printf("loading(%d) ... %s:%s:%s:%s:%s\n", count, name, uid, type, ipaddr, netmask);
#endif
		namelen = strlen(name);
                for( k=0; k<strlen(uid); k++)
                    if( !isdigit(uid[k] ) )
                        is_numeric=0;
		//if ( is_numeric && namelen > 0 && namelen <= USER_NAME_LENGTH && atol(uid) < 65535 )
		if ( is_numeric && namelen > 0 && namelen <= USER_NAME_LENGTH ) //By Andy Wu 03/14/2003
		{
			net_fromaddr.s_addr = 0;
			net_toaddr.s_addr = 0;
			switch ( atol(type) )
			{
			case NFSIP_RANGE:
				if ( !inet_aton(netmask, &net_toaddr) )	// nonzero if success
					break;
			case NFSIP_SINGLE:
				if ( !inet_aton(ipaddr, &net_fromaddr) )
					break;
			case NFSIP_WILDCARD:
				if ( count > *size - 1 )
				{
					buf_enough = 0;
				}

				if ( buf_enough )
				{
					pusers[count].uid = (unsigned int)atoll(uid);
					pusers[count].type = atol(type);
					strcpy(pusers[count].username, name);
					pusers[count].ipaddr.s_addr = ntohl(net_fromaddr.s_addr);
					pusers[count].netmask.s_addr = ntohl(net_toaddr.s_addr);
				}
				count++;
			}
		}

		// go to next line
		do {
			if ( feof(fp) )
				break;
			ch = fgetc(fp);
		} while ( ch != '\n' );
	}

	fclose(fp);

	*size = count;
	return ( buf_enough ) ? 0 : ERROR_OUT_OF_MEMORY;
}

static int write_nfs_config(NFS_USER_EX *pusers, int count)
{
	int  i;
	FILE *fp;
	struct in_addr fromaddr, toaddr;
	NFS_USER_EX *puser;

	fp = fopen(NFSCONF_FILE, "w");
	if ( fp == NULL )
	{
		return ERROR_OPEN_FILE;
	}

	for (i = 0 ; i < count ; i++)
	{
		puser = &pusers[i];
		if ( puser->type == NFSIP_END )
			continue;
		fromaddr = puser->ipaddr;
		toaddr = puser->netmask;
		fprintf(fp, "%s:%u:%d:",
			puser->username,
			puser->uid,
			puser->type);
		if ( puser->type == NFSIP_SINGLE || puser->type == NFSIP_RANGE )
		{
			fprintf(fp, "%d.%d.%d.%d",
				(unsigned char) (fromaddr.s_addr>>24),
				(unsigned char) (fromaddr.s_addr>>16),
				(unsigned char) (fromaddr.s_addr>>8),
				(unsigned char) fromaddr.s_addr);
		}
		fprintf(fp, ":");
		if ( puser->type == NFSIP_RANGE )
		{
			fprintf(fp, "%d.%d.%d.%d",
				(unsigned char) (toaddr.s_addr>>24),
				(unsigned char) (toaddr.s_addr>>16),
				(unsigned char) (toaddr.s_addr>>8),
				(unsigned char) toaddr.s_addr);
		}
		fprintf(fp, "\n");
	}

	fclose(fp);
	return 0;
}

int write_proc_knfst(const char *username, NFS_USER *nfsuser, int knfst_cmd)
{
	char	line[1024];
	char	cmd[10];
	char	s_ipaddr[16]="";
	char	s_netmask[16]="";
	struct 	in_addr in_ipaddr, in_netmask;
	struct passwd *usr_passwd;
	int	res = 0;

	switch (knfst_cmd){
		case KNFST_CMD_ADD:
			strcpy(cmd, "add");
			break;
		case KNFST_CMD_DEL:
			strcpy(cmd, "del");
			break;
		case KNFST_CMD_FND:
			strcpy(cmd, "fnd");
			break;
		case KNFST_CMD_CLR:
			strcpy(cmd, "clr");
			break;
		case KNFST_CMD_LST:
			strcpy(cmd, "lst");
			break;
		case KNFST_CMD_UNKNOWN:
		case KNFST_CMD_ZERO:
		case KNFST_CMD_TEST:
		case KNFST_CMD_ERR:
			res = -1;
	}
	
	usr_passwd = getpwnam(username);
	if ( nfsuser->type == NFSIP_SINGLE || nfsuser->type == NFSIP_RANGE ) {
		in_ipaddr = nfsuser->ipaddr;
		sprintf(s_ipaddr, "%d.%d.%d.%d",
			(unsigned char) (in_ipaddr.s_addr>>24),
			(unsigned char) (in_ipaddr.s_addr>>16),
			(unsigned char) (in_ipaddr.s_addr>>8),
			(unsigned char) in_ipaddr.s_addr);
	}
	if ( nfsuser->type == NFSIP_RANGE ) {
		in_netmask = nfsuser->netmask;
		sprintf(s_netmask, "%d.%d.%d.%d",
			(unsigned char) (in_netmask.s_addr>>24),
			(unsigned char) (in_netmask.s_addr>>16),
			(unsigned char) (in_netmask.s_addr>>8),
			(unsigned char) in_netmask.s_addr);
	}
	sprintf(line, "echo \"%s %s:%lu:%lu:%d:%s:%s:\" > /proc/knfst", 
			cmd, 
			username, 
			(unsigned long)usr_passwd->pw_uid, 
			(unsigned long)nfsuser->uid, 
			nfsuser->type, 
			s_ipaddr, 
			s_netmask);

	res = system(line);
	res = res >> 8;

	return res;
}

int Add_NFS_Config(const char *username, NFS_USER *nfs_user)
{
	int ret, count, size;
	NFS_USER_EX *pusers;

	// get all users
	count = 0;
	ret = Load_All_NFS_Users(NULL, &count);
	if ( ret < 0 && ret != ERROR_OUT_OF_MEMORY )
	{
		return 0;
	}

	size = count + 1;
	pusers = (NFS_USER_EX*) malloc(size * sizeof(NFS_USER_EX));
	if ( pusers == NULL )
	{
		return 0;
	}
	Load_All_NFS_Users(pusers, &count);

	if ( count != size - 1 )
	{
		free(pusers);
		return 0;
	}

	pusers[count].uid = nfs_user->uid;
	pusers[count].type = nfs_user->type;
	pusers[count].ipaddr = nfs_user->ipaddr;
	pusers[count].netmask = nfs_user->netmask;
	strncpy(pusers[count].username, username, USER_NAME_LENGTH);

	write_nfs_config(pusers, count+1);
	write_proc_knfst(username, nfs_user, KNFST_CMD_ADD);
	free(pusers);

	//Update_Flash_Data(NFS_CONF);

	return 1;
}

int Remove_NFS_Config(const char *username, NFS_USER *nfs_user)
{
	int i, ret, count;
	NFS_USER_EX *pusers, *puser;

	// get all users
	count = 0;
	ret = Load_All_NFS_Users(NULL, &count);
	if ( ret < 0 && ret != ERROR_OUT_OF_MEMORY )
	{
		return 0;
	}

	pusers = (NFS_USER_EX*) malloc(count * sizeof(NFS_USER_EX));
	if ( pusers == NULL )
	{
		return 0;
	}
	Load_All_NFS_Users(pusers, &count);

	// Find out the specified entry.
	for (i = 0 ; i < count ; i++)
	{
		puser = &pusers[i];
// Catherine		if ( strcmp(username, puser->username) == 0 &&
		if (compare_case_string((char*)username, puser->username) == 0 &&
			nfs_user->uid == puser->uid &&
			nfs_user->type == puser->type )
		{
			if ( nfs_user->type == NFSIP_WILDCARD )
				break;
			if ( nfs_user->type == NFSIP_RANGE && nfs_user->netmask.s_addr != puser->netmask.s_addr )
				continue;
			if ( (nfs_user->type == NFSIP_SINGLE || nfs_user->type == NFSIP_RANGE) &&
				nfs_user->ipaddr.s_addr == puser->ipaddr.s_addr )
				break;
		}
	}

	if ( i >= count )
		return 0;

	pusers[i] = pusers[count-1];
	ret = write_nfs_config(pusers, count-1);
	write_proc_knfst(username, nfs_user, KNFST_CMD_DEL);
	free(pusers);

	//Update_Flash_Data(NFS_CONF);

	return (ret == 0) ? 1 : 0;
}

int Remove_NFS_User_Config(const char *username)
{
	int i, ret, count;
	NFS_USER_EX *pusers, *puser;

	// get all users
	count = 0;
	ret = Load_All_NFS_Users(NULL, &count);
	if ( ret < 0 && ret != ERROR_OUT_OF_MEMORY )
	{
		return 0;
	}

	pusers = (NFS_USER_EX*) malloc(count * sizeof(NFS_USER_EX));
	if ( pusers == NULL )
	{
		return 0;
	}
	Load_All_NFS_Users(pusers, &count);

	// Find out the specified entry.
	for (i = 0 ; i < count ; i++)
	{
		puser = &pusers[i];
		if(!compare_case_string((char*)username, puser->username))
			puser->type = NFSIP_END;
	}

	ret = write_nfs_config(pusers, count);
	free(pusers);

	//Update_Flash_Data(NFS_CONF);

	return (ret == 0) ? 1 : 0;
}

int Get_NFS_Users(const char *username, NFS_USER *nfs_user, int *size)
{
	int i, ret, count;
	NFS_USER_EX *pusers;
	int buf_enough;
	int entry;	// specify the index of the next entry we will fill in "nfs_user".
#ifdef DEBUG
	NFS_USER_EX *puser;
	struct in_addr fromaddr, toaddr;
#endif

	// get all users
	count = 0;
	ret = Load_All_NFS_Users(NULL, &count);
	if ( ret < 0 && ret != ERROR_OUT_OF_MEMORY )
	{
		return 0;
	}

	if ( (pusers = (NFS_USER_EX*) malloc(count * sizeof(NFS_USER_EX))) == NULL )
	{
		return 0;
	}
	Load_All_NFS_Users(pusers, &count);

#ifdef DEBUG
	for (i = 0 ; i < count ; i++)
	{
		puser = &pusers[i];
		fromaddr = puser->ipaddr;
		toaddr = puser->netmask;
		printf("Get_NFS_Config(%d)... %s:%u:%d:", i,
			puser->username,
			puser->uid,
			puser->type);
		if ( puser->type == NFSIP_SINGLE || puser->type == NFSIP_RANGE )
		{
			printf("%d.%d.%d.%d",
				(unsigned char) (fromaddr.s_addr>>24),
				(unsigned char) (fromaddr.s_addr>>16),
				(unsigned char) (fromaddr.s_addr>>8),
				(unsigned char) fromaddr.s_addr);
		}
		printf(":");
		if ( puser->type == NFSIP_RANGE )
		{
			printf("%d.%d.%d.%d",
				(unsigned char) (toaddr.s_addr>>24),
				(unsigned char) (toaddr.s_addr>>16),
				(unsigned char) (toaddr.s_addr>>8),
				(unsigned char) toaddr.s_addr);
		}
		printf("\n");
	}
#endif

	buf_enough = 1;
	if ( nfs_user == NULL || *size < 0 )
	{
		buf_enough = 0;
	}

	// Retrieve all entries
	entry = 0;
	for (i = 0 ; i < count ; i++)
	{
// Catherine		if ( strcmp(username, pusers[i].username) == 0 )
		if ( compare_case_string((char*)username, pusers[i].username) == 0 )
		{
			if ( entry > *size - 1 )
			{
				buf_enough = 0;
			}

			if ( buf_enough )
			{
				nfs_user[entry].uid = pusers[i].uid;
				nfs_user[entry].type = pusers[i].type;
				nfs_user[entry].ipaddr = pusers[i].ipaddr;
				nfs_user[entry].netmask = pusers[i].netmask;
			}
			entry++;
		}
	}

	free(pusers);

	*size = entry;
	return ( buf_enough ) ? 0 : ERROR_OUT_OF_MEMORY;
}

int Remove_All_NFS_Users( void )
{
	FILE *fp;

	fp = fopen(NFSCONF_FILE, "w");
	if ( fp == NULL )
		return ERROR_OPEN_FILE;
	fclose(fp);
	return 0;
}

#ifdef	LARGE_NO_OF_USERS_SUPPORT

int Add_NFS_Config_Ex(const char *username, NFS_USER *nfs_user, BOOL update_flash)
{
	int ret, count, size;
	NFS_USER_EX *pusers;

	// get all users
	count = 0;
	ret = Load_All_NFS_Users(NULL, &count);
	if ( ret < 0 && ret != ERROR_OUT_OF_MEMORY )
	{
		return 0;
	}

	size = count + 1;
	pusers = (NFS_USER_EX*) malloc(size * sizeof(NFS_USER_EX));
	if ( pusers == NULL )
	{
		return 0;
	}
	Load_All_NFS_Users(pusers, &count);

	if ( count != size - 1 )
	{
		free(pusers);
		return 0;
	}

	pusers[count].uid = nfs_user->uid;
	pusers[count].type = nfs_user->type;
	pusers[count].ipaddr = nfs_user->ipaddr;
	pusers[count].netmask = nfs_user->netmask;
	strncpy(pusers[count].username, username, USER_NAME_LENGTH);

	write_nfs_config(pusers, count+1);
	write_proc_knfst(username, nfs_user, KNFST_CMD_ADD);
	free(pusers);

	//if (update_flash)
	//	Update_Flash_Data(NFS_CONF);

	return 1;
}

int Remove_NFS_Config_Ex(const char *username, NFS_USER *nfs_user, BOOL update_flash)
{
	int i, ret, count;
	NFS_USER_EX *pusers, *puser;

	// get all users
	count = 0;
	ret = Load_All_NFS_Users(NULL, &count);
	if ( ret < 0 && ret != ERROR_OUT_OF_MEMORY )
	{
		return 0;
	}

	pusers = (NFS_USER_EX*) malloc(count * sizeof(NFS_USER_EX));
	if ( pusers == NULL )
	{
		return 0;
	}
	Load_All_NFS_Users(pusers, &count);

	// Find out the specified entry.
	for (i = 0 ; i < count ; i++)
	{
		puser = &pusers[i];
		if (compare_case_string((char*)username, puser->username) == 0 &&
			nfs_user->uid == puser->uid &&
			nfs_user->type == puser->type )
		{
			if ( nfs_user->type == NFSIP_WILDCARD )
				break;
			if ( nfs_user->type == NFSIP_RANGE && nfs_user->netmask.s_addr != puser->netmask.s_addr )
				continue;
			if ( (nfs_user->type == NFSIP_SINGLE || nfs_user->type == NFSIP_RANGE) &&
				nfs_user->ipaddr.s_addr == puser->ipaddr.s_addr )
				break;
		}
	}

	if ( i >= count )
		return 0;

	pusers[i] = pusers[count-1];
	ret = write_nfs_config(pusers, count-1);
	write_proc_knfst(username, nfs_user, KNFST_CMD_DEL);
	free(pusers);

	//if (update_flash)
	//	Update_Flash_Data(NFS_CONF);

	return (ret == 0) ? 1 : 0;
}

int Remove_NFS_User_Config_Ex(const char *username, BOOL update_flash)
{
	int i, ret, count;
	NFS_USER_EX *pusers, *puser;
	NFS_USER *nfs_user;

	// get all users
	count = 0;
	ret = Load_All_NFS_Users(NULL, &count);
	if ( ret < 0 && ret != ERROR_OUT_OF_MEMORY )
	{
		return 0;
	}

	pusers = (NFS_USER_EX*) malloc(count * sizeof(NFS_USER_EX));
	if ( pusers == NULL )
	{
		return 0;
	}
	Load_All_NFS_Users(pusers, &count);

	// Find out the specified entry.
	for (i = 0 ; i < count ; i++)
	{
		puser = &pusers[i];
		if(!compare_case_string((char*)username, puser->username))
			puser->type = NFSIP_END;
	}

	ret = write_nfs_config(pusers, count);
	nfs_user = (NFS_USER *)malloc(sizeof(NFS_USER));
	nfs_user->uid = pusers->uid;
	nfs_user->type = pusers->type;
	nfs_user->ipaddr = pusers->ipaddr;
	nfs_user->netmask = pusers->netmask;
	write_proc_knfst(username, nfs_user, KNFST_CMD_DEL);
	free(nfs_user);
	free(pusers);

	//if (update_flash)
	//	Update_Flash_Data(NFS_CONF);

	return (ret == 0) ? 1 : 0;
}

#endif
