/*
   Unix SMB/Netbios implementation.
   Version 1.9.
   Samba utility functions
   Copyright (C) Andrew Tridgell 1992-1998

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#ifdef WITH_SSL
#include <openssl/ssl.h>
#undef Realloc  /* SSLeay defines this and samba has a function of this name */
extern SSL  *ssl;
extern int  sslFd;
#endif  /* WITH_SSL */

/* the last IP received from */
struct in_addr lastip;

/* the last port received from */
int lastport=0;

int smb_read_error = 0;

/****************************************************************************
 Determine if a file descriptor is in fact a socket.
****************************************************************************/

BOOL is_a_socket(int fd)
{
	int v;
	socklen_t l;
	l = sizeof(int);
	return(getsockopt(fd, SOL_SOCKET, SO_TYPE, (char *)&v, &l) == 0);
}

enum SOCK_OPT_TYPES {OPT_BOOL,OPT_INT,OPT_ON};

typedef struct smb_socket_option {
	const char *name;
	int level;
	int option;
	int value;
	int opttype;
} smb_socket_option;

smb_socket_option socket_options[] = {
  {"SO_KEEPALIVE",      SOL_SOCKET,    SO_KEEPALIVE,    0,                 OPT_BOOL},
  {"SO_REUSEADDR",      SOL_SOCKET,    SO_REUSEADDR,    0,                 OPT_BOOL},
  {"SO_BROADCAST",      SOL_SOCKET,    SO_BROADCAST,    0,                 OPT_BOOL},
#ifdef TCP_NODELAY
  {"TCP_NODELAY",       IPPROTO_TCP,   TCP_NODELAY,     0,                 OPT_BOOL},
#endif
#ifdef IPTOS_LOWDELAY
  {"IPTOS_LOWDELAY",    IPPROTO_IP,    IP_TOS,          IPTOS_LOWDELAY,    OPT_ON},
#endif
#ifdef IPTOS_THROUGHPUT
  {"IPTOS_THROUGHPUT",  IPPROTO_IP,    IP_TOS,          IPTOS_THROUGHPUT,  OPT_ON},
#endif
#ifdef SO_REUSEPORT
  {"SO_REUSEPORT",      SOL_SOCKET,    SO_REUSEPORT,    0,                 OPT_BOOL},
#endif
#ifdef SO_SNDBUF
  {"SO_SNDBUF",         SOL_SOCKET,    SO_SNDBUF,       0,                 OPT_INT},
#endif
#ifdef SO_RCVBUF
  {"SO_RCVBUF",         SOL_SOCKET,    SO_RCVBUF,       0,                 OPT_INT},
#endif
#ifdef SO_SNDLOWAT
  {"SO_SNDLOWAT",       SOL_SOCKET,    SO_SNDLOWAT,     0,                 OPT_INT},
#endif
#ifdef SO_RCVLOWAT
  {"SO_RCVLOWAT",       SOL_SOCKET,    SO_RCVLOWAT,     0,                 OPT_INT},
#endif
#ifdef SO_SNDTIMEO
  {"SO_SNDTIMEO",       SOL_SOCKET,    SO_SNDTIMEO,     0,                 OPT_INT},
#endif
#ifdef SO_RCVTIMEO
  {"SO_RCVTIMEO",       SOL_SOCKET,    SO_RCVTIMEO,     0,                 OPT_INT},
#endif
  {NULL,0,0,0,0}};

/****************************************************************************
 Print socket options.
****************************************************************************/

static void print_socket_options(int s)
{
	int value;
	socklen_t vlen = 4;
	smb_socket_option *p = &socket_options[0];

	for (; p->name != NULL; p++) {
		if (getsockopt(s, p->level, p->option, (void *)&value, &vlen) == -1) {
((void)0);
		} else {
((void)0);
		}
	}
 }

/****************************************************************************
 Set user socket options.
****************************************************************************/

void set_socket_options(int fd, const char *options)
{
	fstring tok;

	while (next_token(&options,tok," \t,", sizeof(tok))) {
		int ret=0,i;
		int value = 1;
		char *p;
		BOOL got_value = False;

		if ((p = strchr(tok,'='))) {
			*p = 0;
			value = atoi(p+1);
			got_value = True;
		}

		for (i=0;socket_options[i].name;i++)
			if (strequal(socket_options[i].name,tok))
				break;

		if (!socket_options[i].name) {
((void)0);
			continue;
		}

		switch (socket_options[i].opttype) {
		case OPT_BOOL:
		case OPT_INT:
			ret = setsockopt(fd,socket_options[i].level,
						socket_options[i].option,(char *)&value,sizeof(int));
			break;

		case OPT_ON:
			if (got_value)
((void)0);

			{
				int on = socket_options[i].value;
				ret = setsockopt(fd,socket_options[i].level,
							socket_options[i].option,(char *)&on,sizeof(int));
			}
			break;
		}

		if (ret != 0)
((void)0);
	}

	print_socket_options(fd);
}

/****************************************************************************
 Read from a socket.
****************************************************************************/

ssize_t read_udp_socket(int fd,char *buf,size_t len)
{
	ssize_t ret;
	struct sockaddr_in sock;
	socklen_t socklen = sizeof(sock);

	memset((char *)&sock,'\0',socklen);
	memset((char *)&lastip,'\0',sizeof(lastip));
	ret = (ssize_t)sys_recvfrom(fd,buf,len,0,(struct sockaddr *)&sock,&socklen);
	if (ret <= 0) {
((void)0);
		return(0);
	}

	lastip = sock.sin_addr;
	lastport = ntohs(sock.sin_port);

((void)0);

	return(ret);
}

/****************************************************************************
 Read data from a socket with a timout in msec.
 mincount = if timeout, minimum to read before returning
 maxcount = number to be read.
 time_out = timeout in milliseconds
****************************************************************************/

static ssize_t read_socket_with_timeout(int fd,char *buf,size_t mincnt,size_t maxcnt,unsigned int time_out)
{
  fd_set fds;
  int selrtn;
  ssize_t readret;
  size_t nread = 0;
  struct timeval timeout;

  /* just checking .... */
  if (maxcnt <= 0)
    return(0);

  smb_read_error = 0;

  /* Blocking read */
  if (time_out <= 0) {
    if (mincnt == 0) mincnt = maxcnt;

    while (nread < mincnt) {
#ifdef WITH_SSL
      if(fd == sslFd){
        readret = SSL_read(ssl, buf + nread, maxcnt - nread);
      }else{
        readret = sys_read(fd, buf + nread, maxcnt - nread);
      }
#else /* WITH_SSL */
      readret = sys_read(fd, buf + nread, maxcnt - nread);
#endif /* WITH_SSL */

      if (readret == 0) {
((void)0);
        smb_read_error = READ_EOF;
        return -1;
      }

      if (readret == -1) {
((void)0);
        smb_read_error = READ_ERROR;
        return -1;
      }
      nread += readret;
    }
    return((ssize_t)nread);
  }

  /* Most difficult - timeout read */
  /* If this is ever called on a disk file and
     mincnt is greater then the filesize then
     system performance will suffer severely as
     select always returns true on disk files */

  /* Set initial timeout */
  timeout.tv_sec = (time_t)(time_out / 1000);
  timeout.tv_usec = (long)(1000 * (time_out % 1000));

  for (nread=0; nread < mincnt; ) {
    FD_ZERO(&fds);
    FD_SET(fd,&fds);

    selrtn = sys_select_intr(fd+1,&fds,NULL,NULL,&timeout);

    /* Check if error */
    if(selrtn == -1) {
      /* something is wrong. Maybe the socket is dead? */
((void)0);
      smb_read_error = READ_ERROR;
      return -1;
    }

    /* Did we timeout ? */
    if (selrtn == 0) {
((void)0);
      smb_read_error = READ_TIMEOUT;
      return -1;
    }

#ifdef WITH_SSL
    if(fd == sslFd){
      readret = SSL_read(ssl, buf + nread, maxcnt - nread);
    }else{
      readret = sys_read(fd, buf + nread, maxcnt - nread);
    }
#else /* WITH_SSL */
    readret = sys_read(fd, buf+nread, maxcnt-nread);
#endif /* WITH_SSL */

    if (readret == 0) {
      /* we got EOF on the file descriptor */
((void)0);
      smb_read_error = READ_EOF;
      return -1;
    }

    if (readret == -1) {
      /* the descriptor is probably dead */
((void)0);
      smb_read_error = READ_ERROR;
      return -1;
    }

    nread += readret;
  }

  /* Return the number we got */
  return((ssize_t)nread);
}

/****************************************************************************
 Read data from a fd with a timout in msec.
 mincount = if timeout, minimum to read before returning
 maxcount = number to be read.
 time_out = timeout in milliseconds
****************************************************************************/

ssize_t read_with_timeout(int fd,char *buf,size_t mincnt,size_t maxcnt,unsigned int time_out)
{
  fd_set fds;
  int selrtn;
  ssize_t readret;
  size_t nread = 0;
  struct timeval timeout;

  /* just checking .... */
  if (maxcnt <= 0)
    return(0);

  /* Blocking read */
  if (time_out <= 0) {
    if (mincnt == 0) mincnt = maxcnt;

    while (nread < mincnt) {
#ifdef WITH_SSL
      if(fd == sslFd){
        readret = SSL_read(ssl, buf + nread, maxcnt - nread);
      }else{
        readret = sys_read(fd, buf + nread, maxcnt - nread);
      }
#else /* WITH_SSL */
      readret = sys_read(fd, buf + nread, maxcnt - nread);
#endif /* WITH_SSL */

      if (readret <= 0)
	return readret;

      nread += readret;
    }
    return((ssize_t)nread);
  }

  /* Most difficult - timeout read */
  /* If this is ever called on a disk file and
     mincnt is greater then the filesize then
     system performance will suffer severely as
     select always returns true on disk files */

  /* Set initial timeout */
  timeout.tv_sec = (time_t)(time_out / 1000);
  timeout.tv_usec = (long)(1000 * (time_out % 1000));

  for (nread=0; nread < mincnt; ) {
    FD_ZERO(&fds);
    FD_SET(fd,&fds);

    selrtn = sys_select_intr(fd+1,&fds,NULL,NULL,&timeout);

    if(selrtn <= 0)
      return selrtn;

#ifdef WITH_SSL
    if(fd == sslFd){
      readret = SSL_read(ssl, buf + nread, maxcnt - nread);
    }else{
      readret = sys_read(fd, buf + nread, maxcnt - nread);
    }
#else /* WITH_SSL */
    readret = sys_read(fd, buf+nread, maxcnt-nread);
#endif /* WITH_SSL */

    if (readret <= 0)
      return readret;

    nread += readret;
  }

  /* Return the number we got */
  return((ssize_t)nread);
}

/****************************************************************************
send a keepalive packet (rfc1002)
****************************************************************************/

BOOL send_keepalive(int client)
{
	unsigned char buf[4];

	buf[0] = 0x85;
	buf[1] = buf[2] = buf[3] = 0;

	return(write_socket_data(client,(char *)buf,4) == 4);
}

/****************************************************************************
  read data from the client, reading exactly N bytes.
****************************************************************************/

ssize_t read_data(int fd,char *buffer,size_t N)
{
	ssize_t  ret;
	size_t total=0;

	smb_read_error = 0;

	while (total < N) {
#ifdef WITH_SSL
		if(fd == sslFd){
			ret = SSL_read(ssl, buffer + total, N - total);
		}else{
			ret = sys_read(fd,buffer + total,N - total);
		}
#else /* WITH_SSL */
		ret = sys_read(fd,buffer + total,N - total);
#endif /* WITH_SSL */

		if (ret == 0) {
((void)0);
			smb_read_error = READ_EOF;
			return 0;
		}

		if (ret == -1) {
((void)0);
			smb_read_error = READ_ERROR;
			return -1;
		}
		total += ret;
	}
	return (ssize_t)total;
}

/****************************************************************************
 Write data to a fd.
****************************************************************************/

ssize_t write_data(int fd,char *buffer,size_t N)
{
	size_t total=0;
	ssize_t ret;

	while (total < N) {
#ifdef WITH_SSL
		if(fd == sslFd){
			ret = SSL_write(ssl,buffer + total,N - total);
		} else {
			ret = sys_write(fd,buffer + total,N - total);
		}
#else /* WITH_SSL */
		ret = sys_write(fd,buffer + total,N - total);
#endif /* WITH_SSL */

		if (ret == -1) {
((void)0);
			return -1;
		}
		if (ret == 0)
			return (ssize_t)total;

		total += (size_t)ret;
	}

	return (ssize_t)total;
}

/****************************************************************************
 Write data to a socket - use send rather than write.
****************************************************************************/

ssize_t write_socket_data(int fd,char *buffer,size_t N)
{
	size_t total=0;
	ssize_t ret;

	while (total < N) {
#ifdef WITH_SSL
		if(fd == sslFd){
			ret = SSL_write(ssl,buffer + total,N - total);
		}else{
			ret = sys_send(fd,buffer + total,N - total, 0);
		}
#else /* WITH_SSL */
		ret = sys_send(fd,buffer + total,N - total,0);
#endif /* WITH_SSL */

		if (ret == -1) {
((void)0);
			return -1;
		}
		if (ret == 0)
			return (ssize_t)total;

		total += ret;
	}
	return (ssize_t)total;
}

/****************************************************************************
write to a socket
****************************************************************************/

ssize_t write_socket(int fd,char *buf,size_t len)
{
	ssize_t ret=0;

((void)0);
	ret = write_socket_data(fd,buf,len);

((void)0);
	if(ret <= 0)
((void)0);

	return(ret);
}

/****************************************************************************
read 4 bytes of a smb packet and return the smb length of the packet
store the result in the buffer
This version of the function will return a length of zero on receiving
a keepalive packet.
timeout is in milliseconds.
****************************************************************************/

static ssize_t read_smb_length_return_keepalive(int fd,char *inbuf,unsigned int timeout)
{
	ssize_t len=0;
	int msg_type;
	BOOL ok = False;

	while (!ok) {
		if (timeout > 0)
			ok = (read_socket_with_timeout(fd,inbuf,4,4,timeout) == 4);
		else
			ok = (read_data(fd,inbuf,4) == 4);

		if (!ok)
			return(-1);

		len = smb_len(inbuf);
		msg_type = CVAL(inbuf,0);

		if (msg_type == 0x85)
((void)0);
	}

((void)0);

	return(len);
}

/****************************************************************************
read 4 bytes of a smb packet and return the smb length of the packet
store the result in the buffer. This version of the function will
never return a session keepalive (length of zero).
timeout is in milliseconds.
****************************************************************************/

ssize_t read_smb_length(int fd,char *inbuf,unsigned int timeout)
{
	ssize_t len;

	for(;;) {
		len = read_smb_length_return_keepalive(fd, inbuf, timeout);

		if(len < 0)
			return len;

		/* Ignore session keepalives. */
		if(CVAL(inbuf,0) != 0x85)
			break;
	}

((void)0);

	return len;
}

/****************************************************************************
  read an smb from a fd. Note that the buffer *MUST* be of size
  BUFFER_SIZE+SAFETY_MARGIN.
  The timeout is in milliseconds.
  This function will return on a
  receipt of a session keepalive packet.
****************************************************************************/

BOOL receive_smb(int fd,char *buffer, unsigned int timeout)
{
	ssize_t len,ret;

	smb_read_error = 0;

	memset(buffer,'\0',smb_size + 100);

	len = read_smb_length_return_keepalive(fd,buffer,timeout);
	if (len < 0) {
((void)0);

		/*
		 * Correct fix. smb_read_error may have already been
		 * set. Only set it here if not already set. Global
		 * variables still suck :-). JRA.
		 */

		if (smb_read_error == 0)
			smb_read_error = READ_ERROR;
		return(False);
	}

	/*
	 * A WRITEX with CAP_LARGE_WRITEX can be 64k worth of data plus 65 bytes
	 * of header. Don't print the error if this fits.... JRA.
	 */

	if (len > (BUFFER_SIZE + LARGE_WRITEX_HDR_SIZE)) {
((void)0);
		if (len > BUFFER_SIZE + (SAFETY_MARGIN/2)) {

			/*
			 * Correct fix. smb_read_error may have already been
			 * set. Only set it here if not already set. Global
			 * variables still suck :-). JRA.
			 */

			if (smb_read_error == 0)
				smb_read_error = READ_ERROR;
			return False;
		}
	}

	if(len > 0) {
		ret = read_data(fd,buffer+4,len);
		if (ret != len) {
			if (smb_read_error == 0)
				smb_read_error = READ_ERROR;
			return False;
		}
	}

	return(True);
}

/****************************************************************************
  read an smb from a fd ignoring all keepalive packets. Note that the buffer
  *MUST* be of size BUFFER_SIZE+SAFETY_MARGIN.
  The timeout is in milliseconds

  This is exactly the same as receive_smb except that it never returns
  a session keepalive packet (just as receive_smb used to do).
  receive_smb was changed to return keepalives as the oplock processing means this call
  should never go into a blocking read.
****************************************************************************/

BOOL client_receive_smb(int fd,char *buffer, unsigned int timeout)
{
  BOOL ret;

  for(;;)
  {
    ret = receive_smb(fd, buffer, timeout);

    if (!ret)
    {
((void)0);
      show_msg(buffer);
      return ret;
    }

    /* Ignore session keepalive packets. */
    if(CVAL(buffer,0) != 0x85)
      break;
  }
  show_msg(buffer);
  return ret;
}

/****************************************************************************
  send an smb to a fd
****************************************************************************/

BOOL send_smb(int fd,char *buffer)
{
	size_t len;
	size_t nwritten=0;
	ssize_t ret;
	len = smb_len(buffer) + 4;

	while (nwritten < len) {
		ret = write_socket(fd,buffer+nwritten,len - nwritten);
		if (ret <= 0) {
((void)0);
			return False;
		}
		nwritten += ret;
	}

	return True;
}

/****************************************************************************
send a single packet to a port on another machine
****************************************************************************/

BOOL send_one_packet(char *buf,int len,struct in_addr ip,int port,int type)
{
  BOOL ret;
  int out_fd;
  struct sockaddr_in sock_out;

  /* create a socket to write to */
  out_fd = socket(AF_INET, type, 0);
  if (out_fd == -1)
    {
((void)0);
      return False;
    }

  /* set the address and port */
  memset((char *)&sock_out,'\0',sizeof(sock_out));
  putip((char *)&sock_out.sin_addr,(char *)&ip);
  sock_out.sin_port = htons( port );
  sock_out.sin_family = AF_INET;

  if (DEBUGLEVEL > 0)
((void)0);

  /* send it */
  ret = (sys_sendto(out_fd,buf,len,0,(struct sockaddr *)&sock_out,sizeof(sock_out)) >= 0);

  if (!ret)
((void)0);

  close(out_fd);
  return(ret);
}

/****************************************************************************
 Open a socket of the specified type, port, and address for incoming data.
****************************************************************************/

int open_socket_in( int type, int port, int dlevel, uint32 socket_addr, BOOL rebind )
{
	struct sockaddr_in sock;
	int res;

	memset( (char *)&sock, '\0', sizeof(sock) );

#ifdef HAVE_SOCK_SIN_LEN
	sock.sin_len         = sizeof(sock);
#endif
	sock.sin_port        = htons( port );
	sock.sin_family      = AF_INET;
	sock.sin_addr.s_addr = socket_addr;

	res = socket( AF_INET, type, 0 );
	if( res == -1 ) {
		if( DEBUGLVL(0) ) {
((void)0);
((void)0);
		}
		return -1;
	}

	/* This block sets/clears the SO_REUSEADDR and possibly SO_REUSEPORT. */
	{
		int val = rebind ? 1 : 0;
		if( setsockopt(res,SOL_SOCKET,SO_REUSEADDR,(char *)&val,sizeof(val)) == -1 ) {
			if( DEBUGLVL( dlevel ) ) {
((void)0);
((void)0);
((void)0);
((void)0);
			}
		}
#ifdef SO_REUSEPORT
		if( setsockopt(res,SOL_SOCKET,SO_REUSEPORT,(char *)&val,sizeof(val)) == -1 ) {
			if( DEBUGLVL( dlevel ) ) {
((void)0);
((void)0);
((void)0);
((void)0);
			}
		}
#endif /* SO_REUSEPORT */
	}

	/* now we've got a socket - we need to bind it */
	if( bind( res, (struct sockaddr *)&sock, sizeof(sock) ) == -1 ) {
		if( DEBUGLVL(dlevel) && (port == SMB_PORT || port == NMB_PORT) ) {
((void)0);
((void)0);
((void)0);
		}
		close( res );
		return( -1 );
	}

((void)0);

	return( res );
 }

/****************************************************************************
  create an outgoing socket. timeout is in milliseconds.
  **************************************************************************/

int open_socket_out(int type, struct in_addr *addr, int port ,int timeout)
{
  struct sockaddr_in sock_out;
  int res,ret;
  int connect_loop = 250; /* 250 milliseconds */
  int loops = (timeout) / connect_loop;

  /* create a socket to write to */
  res = socket(PF_INET, type, 0);
  if (res == -1)
    { DEBUG(0,("socket error\n")); return -1; }

  if (type != SOCK_STREAM) return(res);

  memset((char *)&sock_out,'\0',sizeof(sock_out));
  putip((char *)&sock_out.sin_addr,(char *)addr);

  sock_out.sin_port = htons( port );
  sock_out.sin_family = PF_INET;

  /* set it non-blocking */
  set_blocking(res,False);

((void)0);

  /* and connect it to the destination */
connect_again:
  ret = connect(res,(struct sockaddr *)&sock_out,sizeof(sock_out));

  /* Some systems return EAGAIN when they mean EINPROGRESS */
  if (ret < 0 && (errno == EINPROGRESS || errno == EALREADY ||
        errno == EAGAIN) && loops--) {
    msleep(connect_loop);
    goto connect_again;
  }

  if (ret < 0 && (errno == EINPROGRESS || errno == EALREADY ||
         errno == EAGAIN)) {
((void)0);
      close(res);
      return -1;
  }

#ifdef EISCONN
  if (ret < 0 && errno == EISCONN) {
    errno = 0;
    ret = 0;
  }
#endif

  if (ret < 0) {
((void)0);
    close(res);
    return -1;
  }

  /* set it blocking again */
  set_blocking(res,True);

  return res;
}

/* the following 3 client_*() functions are nasty ways of allowing
   some generic functions to get info that really should be hidden in
   particular modules */
static int client_fd = -1;

void client_setfd(int fd)
{
	client_fd = fd;
}

char *client_name(void)
{
	return get_socket_name(client_fd);
}

char *client_addr(void)
{
	return get_socket_addr(client_fd);
}

/*******************************************************************
 matchname - determine if host name matches IP address. Used to
 confirm a hostname lookup to prevent spoof attacks
 ******************************************************************/
static BOOL matchname(char *remotehost,struct in_addr  addr)
{
	struct hostent *hp;
	int     i;

	if ((hp = sys_gethostbyname(remotehost)) == 0) {
((void)0);
		return False;
	}

	/*
	 * Make sure that gethostbyname() returns the "correct" host name.
	 * Unfortunately, gethostbyname("localhost") sometimes yields
	 * "localhost.domain". Since the latter host name comes from the
	 * local DNS, we just have to trust it (all bets are off if the local
	 * DNS is perverted). We always check the address list, though.
	 */

	if (strcasecmp(remotehost, hp->h_name)
	    && strcasecmp(remotehost, "localhost")) {
((void)0);
		return False;
	}

	/* Look up the host address in the address list we just got. */
	for (i = 0; hp->h_addr_list[i]; i++) {
		if (memcmp(hp->h_addr_list[i], (void *) & addr, sizeof(addr)) == 0)
			return True;
	}

	/*
	 * The host name does not map to the original host address. Perhaps
	 * someone has compromised a name server. More likely someone botched
	 * it, but that could be dangerous, too.
	 */

((void)0);
	return False;
}


/*******************************************************************
 return the DNS name of the remote end of a socket
 ******************************************************************/
char *get_socket_name(int fd)
{
	static pstring name_buf;
	static fstring addr_buf;
	struct hostent *hp;
	struct in_addr addr;
	char *p;

	p = get_socket_addr(fd);

	/* it might be the same as the last one - save some DNS work */
	if (strcmp(p, addr_buf) == 0) return name_buf;

	pstrcpy(name_buf,"UNKNOWN");
	if (fd == -1) return name_buf;

	fstrcpy(addr_buf, p);

	addr = *interpret_addr2(p);

	/* Look up the remote host name. */
	if ((hp = gethostbyaddr((char *)&addr.s_addr, sizeof(addr.s_addr), AF_INET)) == 0) {
((void)0);
		pstrcpy(name_buf, p);
	} else {
		pstrcpy(name_buf,(char *)hp->h_name);
		if (!matchname(name_buf, addr)) {
((void)0);
			pstrcpy(name_buf,"UNKNOWN");
		}
	}

// modify by Kent 2004/07/28
//      alpha_strcpy(name_buf, name_buf, "_-.", sizeof(name_buf));
	old_alpha_strcpy(name_buf, name_buf, "_-.", sizeof(name_buf));
// end
	if (strstr(name_buf,"..")) {
		pstrcpy(name_buf, "UNKNOWN");
	}

	return name_buf;
}

/*******************************************************************
 return the IP addr of the remote end of a socket as a string
 ******************************************************************/
char *get_socket_addr(int fd)
{
	struct sockaddr sa;
	struct sockaddr_in *sockin = (struct sockaddr_in *) (&sa);
	socklen_t length = sizeof(sa);
	static fstring addr_buf;

	fstrcpy(addr_buf,"0.0.0.0");

	if (fd == -1) {
		return addr_buf;
	}

	if (getpeername(fd, &sa, &length) < 0) {
((void)0);
		return addr_buf;
	}

	fstrcpy(addr_buf,(char *)inet_ntoa(sockin->sin_addr));

	return addr_buf;
}

/*******************************************************************
 Create protected unix domain socket.

 Some unixes cannot set permissions on a ux-dom-sock, so we
 have to make sure that the directory contains the protection
 permissions instead.
 ******************************************************************/

int create_pipe_sock(const char *socket_dir,
		     const char *socket_name,
		     mode_t dir_perms)
{
#ifdef HAVE_UNIXSOCKET
	struct sockaddr_un sunaddr;
	struct stat st;
	int sock;
	mode_t old_umask;
	pstring path;

	old_umask = umask(0);

	/* Create the socket directory or reuse the existing one */

	if (lstat(socket_dir, &st) == -1) {
		if (errno == ENOENT) {
			/* Create directory */
			if (mkdir(socket_dir, dir_perms) == -1) {
((void)0);
				goto out_umask;
			}
		} else {
((void)0);
			goto out_umask;
		}
	} else {
		/* Check ownership and permission on existing directory */
		if (!S_ISDIR(st.st_mode)) {
((void)0);
			goto out_umask;
		}
		if ((st.st_uid != sec_initial_uid()) ||
				((st.st_mode & 0777) != dir_perms)) {
((void)0);
			goto out_umask;
		}
	}

	/* Create the socket file */

	sock = socket(AF_UNIX, SOCK_STREAM, 0);

	if (sock == -1) {
		perror("socket");
                goto out_umask;
	}

	snprintf(path, sizeof(path), "%s/%s", socket_dir, socket_name);

	unlink(path);
	memset(&sunaddr, 0, sizeof(sunaddr));
	sunaddr.sun_family = AF_UNIX;
	safe_strcpy(sunaddr.sun_path, path, sizeof(sunaddr.sun_path)-1);

	if (bind(sock, (struct sockaddr *)&sunaddr, sizeof(sunaddr)) == -1) {
((void)0);
		goto out_close;
	}

	if (listen(sock, 5) == -1) {
((void)0);
		goto out_close;
	}

	umask(old_umask);
	return sock;

out_close:
	close(sock);

out_umask:
	umask(old_umask);
	return -1;

#else
((void)0);
        return -1;
#endif /* HAVE_UNIXSOCKET */
}

/*******************************************************************
this is like socketpair but uses tcp. It is used by the Samba
regression test code
The function guarantees that nobody else can attach to the socket,
or if they do that this function fails and the socket gets closed
returns 0 on success, -1 on failure
the resulting file descriptors are symmetrical
 ******************************************************************/
static int socketpair_tcp(int fd[2])
{
	int listener;
	struct sockaddr_in sock;
	struct sockaddr_in sock2;
	socklen_t socklen = sizeof(sock);
	int connect_done = 0;

	fd[0] = fd[1] = listener = -1;

	memset(&sock, 0, sizeof(sock));

	if ((listener = socket(PF_INET, SOCK_STREAM, 0)) == -1) goto failed;

        memset(&sock2, 0, sizeof(sock2));
#ifdef HAVE_SOCK_SIN_LEN
        sock2.sin_len = sizeof(sock2);
#endif
        sock2.sin_family = PF_INET;

        bind(listener, (struct sockaddr *)&sock2, sizeof(sock2));

	if (listen(listener, 1) != 0) goto failed;

	if (getsockname(listener, (struct sockaddr *)&sock, &socklen) != 0) goto failed;

	if ((fd[1] = socket(PF_INET, SOCK_STREAM, 0)) == -1) goto failed;

	set_blocking(fd[1], 0);

	sock.sin_addr.s_addr = htonl(INADDR_LOOPBACK);

	if (connect(fd[1],(struct sockaddr *)&sock,sizeof(sock)) == -1) {
		if (errno != EINPROGRESS) goto failed;
	} else {
		connect_done = 1;
	}

	if ((fd[0] = accept(listener, (struct sockaddr *)&sock, &socklen)) == -1) goto failed;

	close(listener);
	if (connect_done == 0) {
		if (connect(fd[1],(struct sockaddr *)&sock,sizeof(sock)) != 0
		    && errno != EISCONN) goto failed;
	}

	set_blocking(fd[1], 1);

	/* all OK! */
	return 0;

 failed:
	if (fd[0] != -1) close(fd[0]);
	if (fd[1] != -1) close(fd[1]);
	if (listener != -1) close(listener);
	return -1;
}


/*******************************************************************
run a program on a local tcp socket, this is used to launch smbd
when regression testing
the return value is a socket which is attached to a subprocess
running "prog". stdin and stdout are attached. stderr is left
attached to the original stderr
 ******************************************************************/
int sock_exec(const char *prog)
{
	int fd[2];
	if (socketpair_tcp(fd) != 0) {
((void)0);
		return -1;
	}
	if (fork() == 0) {
		close(fd[0]);
		close(0);
		close(1);
		dup(fd[1]);
		dup(fd[1]);
		exit(system(prog));
	}
	close(fd[1]);
	return fd[0];
}
