/*
   Unix SMB/Netbios implementation.
   Version 1.9.
   Groupname handling
   Copyright (C) Jeremy Allison 1998.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef USING_GROUPNAME_MAP

#include "includes.h"
extern DOM_SID global_sam_sid;


/**************************************************************************
 Groupname map functionality. The code loads a groupname map file and
 (currently) loads it into a linked list. This is slow and memory
 hungry, but can be changed into a more efficient storage format
 if the demands on it become excessive.
***************************************************************************/

typedef struct groupname_map {
   ubi_slNode next;

   char *windows_name;
   DOM_SID windows_sid;
   char *unix_name;
   gid_t unix_gid;
} groupname_map_entry;

static ubi_slList groupname_map_list;

/**************************************************************************
 Delete all the entries in the groupname map list.
***************************************************************************/

static void delete_groupname_map_list(void)
{
  groupname_map_entry *gmep;

  while((gmep = (groupname_map_entry *)ubi_slRemHead( &groupname_map_list )) != NULL) {
    SAFE_FREE(gmep->windows_name);
    SAFE_FREE(gmep->unix_name);
    SAFE_FREE(gmep);
  }
}

/**************************************************************************
 Load a groupname map file. Sets last accessed timestamp.
***************************************************************************/

void load_groupname_map(void)
{
  static time_t groupmap_file_last_modified = (time_t)0;
  static BOOL initialized = False;
  char *groupname_map_file = lp_groupname_map();
  SMB_STRUCT_STAT st;
  char **lines;
  int i;
  groupname_map_entry *new_ep;

  if(!initialized) {
    ubi_slInitList( &groupname_map_list );
    initialized = True;
  }

  if (!*groupname_map_file)
    return;

  if(sys_stat(groupname_map_file, &st) != 0) {
((void)0);
    return;
  }

  /*
   * Check if file has changed.
   */
  if( st.st_mtime <= groupmap_file_last_modified)
    return;

  groupmap_file_last_modified = st.st_mtime;

  /*
   * Load the file.
   */

  lines = file_lines_load(groupname_map_file,NULL,False);
  if (!lines) {
((void)0);
    return;
  }
  file_lines_slashcont(lines);

  /*
   * Throw away any previous list.
   */
  delete_groupname_map_list();

((void)0);

  for (i=0; lines[i]; i++) {
    pstring unixname;
    pstring windows_name;
    gid_t gid;
    DOM_SID tmp_sid;
    char *s = lines[i];

((void)0);

    if (!*s || strchr("#;",*s))
      continue;

    if(!next_token(&s,unixname, "\t\n\r=", sizeof(unixname)))
      continue;

    if(!next_token(&s,windows_name, "\t\n\r=", sizeof(windows_name)))
      continue;

    trim_string(unixname, " ", " ");
    trim_string(windows_name, " ", " ");

    if (!*windows_name)
      continue;

    if(!*unixname)
      continue;

((void)0);

    /*
     * Attempt to get the unix gid_t for this name.
     */

	if ((gid = nametogid(unixname)) == (gid_t)-1)
((void)0);\
      continue;
    }

    /*
     * Now map to an NT SID.
     */

    if(!lookup_wellknown_sid_from_name(windows_name, &tmp_sid)) {
      /*
       * It's not a well known name, convert the UNIX gid_t
       * to a rid within this domain SID.
       */
      tmp_sid = global_sam_sid;
      tmp_sid.sub_auths[tmp_sid.num_auths++] =
                    pdb_gid_to_group_rid(gid);
    }

    /*
     * Create the list entry and add it onto the list.
     */

    if((new_ep = (groupname_map_entry *)malloc( sizeof(groupname_map_entry) ))== NULL) {
((void)0);
      fclose(fp);
      return;
    }

    new_ep->unix_gid = gid;
    new_ep->windows_sid = tmp_sid;
    new_ep->windows_name = strdup( windows_name );
    new_ep->unix_name = strdup( unixname );

    if(new_ep->windows_name == NULL || new_ep->unix_name == NULL) {
((void)0);
      fclose(fp);
      SAFE_FREE(new_ep->windows_name);
      SAFE_FREE(new_ep->unix_name);
      SAFE_FREE(new_ep);
      file_lines_free(lines);
      return;
    }
    memset((char *)&new_ep->next, '\0', sizeof(new_ep->next) );

    ubi_slAddHead( &groupname_map_list, (ubi_slNode *)new_ep);
  }

((void)0);

  file_lines_free(lines);
}

/***********************************************************
 Lookup a SID entry by gid_t.
************************************************************/

void map_gid_to_sid( gid_t gid, DOM_SID *psid)
{
  groupname_map_entry *gmep;

  /*
   * Initialize and load if not already loaded.
   */
  load_groupname_map();

  for( gmep = (groupname_map_entry *)ubi_slFirst( &groupname_map_list);
       gmep; gmep = (groupname_map_entry *)ubi_slNext( gmep )) {

    if( gmep->unix_gid == gid) {
      *psid = gmep->windows_sid;
((void)0);
      return;
    }
  }

  /*
   * If there's no map, convert the UNIX gid_t
   * to a rid within this domain SID.
   */
  *psid = global_sam_sid;
  psid->sub_auths[psid->num_auths++] = pdb_gid_to_group_rid(gid);

  return;
}
#else /* USING_GROUPNAME_MAP */
 void load_groupname_map(void) {;}
#endif /* USING_GROUPNAME_MAP */
