/*
   Unix SMB/Netbios implementation.
   Version 3.0
   client security descriptor functions
   Copyright (C) Andrew Tridgell 2000

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

/****************************************************************************
  query the security descriptor for a open file
 ****************************************************************************/
SEC_DESC *cli_query_secdesc(struct cli_state *cli, int fnum,
			    TALLOC_CTX *mem_ctx)
{
	char param[8];
	char *rparam=NULL, *rdata=NULL;
	int rparam_count=0, rdata_count=0;
	prs_struct pd;
	SEC_DESC *psd = NULL;

	SIVAL(param, 0, fnum);
	SSVAL(param, 4, 0x7);

	if (!cli_send_nt_trans(cli,
			       NT_TRANSACT_QUERY_SECURITY_DESC,
			       0,
			       NULL, 0, 0,
			       param, 8, 4,
			       NULL, 0, 0x10000)) {
((void)0);
		goto cleanup;
	}


	if (!cli_receive_nt_trans(cli,
				  &rparam, &rparam_count,
				  &rdata, &rdata_count)) {
((void)0);
		goto cleanup;
	}

	prs_init(&pd, rdata_count, mem_ctx, UNMARSHALL);
	prs_append_data(&pd, rdata, rdata_count);
	pd.data_offset = 0;

	if (!sec_io_desc("sd data", &psd, &pd, 1)) {
((void)0);
		goto cleanup;
	}

 cleanup:

	SAFE_FREE(rparam);
	SAFE_FREE(rdata);

	prs_mem_free(&pd);
	return psd;
}

/****************************************************************************
  set the security descriptor for a open file
 ****************************************************************************/
BOOL cli_set_secdesc(struct cli_state *cli, int fnum, SEC_DESC *sd)
{
	char param[8];
	char *rparam=NULL, *rdata=NULL;
	int rparam_count=0, rdata_count=0;
	TALLOC_CTX *mem_ctx;
	prs_struct pd;
	BOOL ret = False;

	if ((mem_ctx = talloc_init()) == NULL) {
((void)0);
		goto cleanup;
	}

	prs_init(&pd, 0, mem_ctx, MARSHALL);
	prs_give_memory(&pd, NULL, 0, True);

	if (!sec_io_desc("sd data", &sd, &pd, 1)) {
((void)0);
		goto cleanup;
	}

	SIVAL(param, 0, fnum);
	SSVAL(param, 4, 0x7);

	if (!cli_send_nt_trans(cli,
			       NT_TRANSACT_SET_SECURITY_DESC,
			       0,
			       NULL, 0, 0,
			       param, 8, 0,
			       pd.data_p, pd.data_offset, 0)) {
((void)0);
		goto cleanup;
	}


	if (!cli_receive_nt_trans(cli,
				  &rparam, &rparam_count,
				  &rdata, &rdata_count)) {
((void)0);
		goto cleanup;
	}

	ret = True;

  cleanup:

	SAFE_FREE(rparam);
	SAFE_FREE(rdata);

	talloc_destroy(mem_ctx);

	prs_mem_free(&pd);
	return ret;
}
