/* vi:set sw=4 ts=4 autoindent: */
/* db.c

   Persistent database management routines for DHCPD... */

/*
 * Copyright (c) 1995, 1996 The Internet Software Consortium.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of The Internet Software Consortium nor the names
 *    of its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INTERNET SOFTWARE CONSORTIUM AND
 * CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE INTERNET SOFTWARE CONSORTIUM OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * This software has been written for the Internet Software Consortium
 * by Ted Lemon <mellon@fugue.com> in cooperation with Vixie
 * Enterprises.  To learn more about the Internet Software Consortium,
 * see ``http://www.vix.com/isc''.  To learn more about Vixie
 * Enterprises, see ``http://www.vix.com''.
 */

#ifndef lint
static char copyright[] =
"$Id: db.c,v 1.5 2001/10/19 11:49:10 davidm Exp $ Copyright (c) 1995, 1996 The Internet Software Consortium.  All rights reserved.\n";
#endif /* not lint */

#include "dhcpd.h"
#include <linux/autoconf.h>
#include <time.h>

#define	Lily_FILE	1

#if	Lily_FILE>0
#include "sysinfo.h"
#endif

FILE *db_file;

static int counting = 0;
static int count = 0;
TIME write_time;

#if	Lily_FILE>0
//struct	dhcp_client_elm{
//	unsigned long	client_ip;
//	unsigned long	start_time;
//	unsigned long	end_time;
//	char	client_host[20];
//	char	client_mac[6];
//	char	reserv[2];
//};
int
write_lease_lily(struct lease *lease, FILE *fp)
{
	struct	dhcp_client_elm	dce;

	if(lease->ends > time(0))
	{
		memset(&dce, 0x0, sizeof(dce));
		//IP
		if(lease->ip_addr.len==0)	//null addr
		{
			return 0;
		}
		else
		{
			memcpy(&(dce.client_ip),lease->ip_addr.iabuf, sizeof(dce.client_ip));
		}

		//time
		dce.start_time = lease->starts;
		dce.end_time = lease->ends;

		//hostname
		if(lease->client_hostname && strlen(lease->client_hostname))	//not NULL
		{
			strncpy(dce.client_host, lease->client_hostname, sizeof(dce.client_host));
		}
		else
		{
			strncpy(dce.client_host, "(null)", 6);
		}

		//mac addr
		if(lease->hardware_addr.hlen)	//not zero
		{
			memcpy(dce.client_mac, lease->hardware_addr.haddr, lease->hardware_addr.hlen);
		}
		else
		{
			return 0;
		}

		//write into file
		fwrite(&dce,sizeof(struct dhcp_client_elm), 1, fp);

		//update fix client lease
		update_fix_lease(dce.client_mac);
	}
	return 0;
}
#endif

/*
   add by aren for fix mapping dhcp client lease. 2007/06/06
   */
struct dhcp_client_record{
	struct dhcp_client_elm dce;
	struct dhcp_client_record *next;
};
typedef struct dhcp_client_record dhcp_client_node;

void update_fix_lease(char *mac)
{
	dhcp_client_node *list_head, *list_tail, *cur, *prev, *walk, *new, *del;
	FILE *fs;
	struct dhcp_client_elm dce_tmp;
	int flag=0;
	
	//step 1. read file.
	fs = fopen("/var/config/dhcpd.clients_fix","r");
	if(fs!=NULL)
	{
		while(!feof(fs))
		{
			if(fread(&dce_tmp,sizeof(struct	dhcp_client_elm),1,fs)>0)
			{
				new = (dhcp_client_node*)malloc(sizeof(dhcp_client_node));
				bzero(new,sizeof(dhcp_client_node));
				memcpy(&new->dce, &dce_tmp, sizeof(struct dhcp_client_elm));
				if(flag==0)
				{
					list_head = new;
					list_tail = new;
					flag = 1;
				}
				else
				{
					list_tail->next = new;
					list_tail = new;
				}
			}
		}
		fclose(fs);
	}

	if(flag==1)
	{
		//find if the same exist
		for(cur=list_head, prev=list_head; cur!=NULL; prev=cur, cur=cur->next)
		{
			if(!memcmp(cur->dce.client_mac, mac, 6))
			{
				//free cur
				if(cur==list_head)
				{
					list_head = cur->next;
					free(cur);
					break;
				}
				else if(cur==list_tail)
				{
					prev->next=NULL;
					free(cur);
					break;
				}
				else if((cur==list_head)&&(cur==list_tail))
				{
					free(cur);
					flag = 2;
					break;
				}
				else
				{
					prev->next = cur->next;
					free(cur);
					break;
				}
			}
		}
	}

	if(flag==2)
	{
		unlink("/var/config/dhcpd.clients_fix");
		return;
	}

	if(flag==1)
	{
		//update fix mapping
		fs = fopen("/var/config/dhcpd.clients_fix","w+");
		if(fs!=NULL)
		{
			//write
			for(walk=list_head; walk!=NULL; walk=walk->next)
			{
				fwrite(&walk->dce, sizeof(struct dhcp_client_elm), 1, fs);
			}
			//free
			for(walk=list_head; walk!=NULL; walk=walk->next,free(del))
			{
				del = walk;
			}
			fclose(fs);
		}
	}
	return;
	
}

int write_fix_lease(struct lease *lease)
{
	dhcp_client_node *list_head,*list_tail, *tmp, *new, *new2, *walk, *del;
	FILE *fs;
	int flag=0,flag2=0;
	struct	dhcp_client_elm	dce,dce_tmp;
	unsigned char *c;
	
	//step 0: read data.
	memset(&dce, 0x0, sizeof(dce));
	//IP
	if(lease->ip_addr.len==0)	//null addr
	{
		return 0;
	}
	else
	{
		memcpy(&(dce.client_ip),lease->ip_addr.iabuf, sizeof(dce.client_ip));
	}

	//time
	dce.start_time = lease->starts;
	dce.end_time = lease->ends;

	//hostname
	if(lease->client_hostname && strlen(lease->client_hostname))	//not NULL
	{
		strncpy(dce.client_host, lease->client_hostname, sizeof(dce.client_host));
	}
	else
	{
		strncpy(dce.client_host, "(null)", 6);
	}

	//mac addr
	if(lease->hardware_addr.hlen)	//not zero
	{
		memcpy(dce.client_mac, lease->hardware_addr.haddr, lease->hardware_addr.hlen);
	}
	else
	{
		return 0;
	}
				
	c = &(dce.client_ip);
#ifdef FOR_DEBUG
	printf("%s,%d dce->ip=%d,%d,%d,%d dce->hostname=%s, dce->MAC=%x:%x:%x:%x:%x:%x\n",
			__FUNCTION__,__LINE__,
			c[0],c[1],c[2],c[3],
			dce.client_host,
			dce.client_mac[0],dce.client_mac[1],dce.client_mac[2],
			dce.client_mac[3],dce.client_mac[4],dce.client_mac[5]);
#endif
	//step 1. read file.
	fs = fopen("/var/config/dhcpd.clients_fix","r");
	if(fs!=NULL)
	{
		while(!feof(fs))
		{
			if(fread(&dce_tmp,sizeof(struct	dhcp_client_elm),1,fs)>0)
			{
				new = (dhcp_client_node*)malloc(sizeof(dhcp_client_node));
				bzero(new,sizeof(dhcp_client_node));
				memcpy(&new->dce, &dce_tmp, sizeof(struct dhcp_client_elm));
				if(flag==0)
				{
					list_head = new;
					list_tail = new;
					flag = 1;
				}
				else
				{
					list_tail->next = new;
					list_tail = new;
				}
			}
		}
		fclose(fs);
	}
	
	//step 2. check if already exist.
	if(flag==1)	//list already has data.
	{
		for(walk=list_head; walk!=NULL; walk=walk->next)
		{	
#ifdef FOR_DEBUG
			printf("%s,%d walk->hostname=%s, walk->MAC=%x:%x:%x:%x:%x:%x\n",
				__FUNCTION__,__LINE__,
				walk->dce.client_host,
				walk->dce.client_mac[0],walk->dce.client_mac[1],walk->dce.client_mac[2],
				walk->dce.client_mac[3],walk->dce.client_mac[4],walk->dce.client_mac[5]);
			printf("%s,%d\n",__FILE__,__LINE__);
#endif
			if(!memcmp(walk->dce.client_mac, dce.client_mac, 6))
			{
				//find the same, just copy.
				memcpy(&walk->dce,&dce,sizeof(struct dhcp_client_elm));
				flag2=1;
			}
		}
		if(flag2==0)
		{
			//not found, add to the end of the list.
			new2 = (dhcp_client_node*)malloc(sizeof(dhcp_client_node));
			bzero(new2,sizeof(dhcp_client_node));
			memcpy(&new2->dce,&dce,sizeof(struct dhcp_client_elm));
			list_tail->next = new2;
			list_tail = new2;
		}
	}
	else	//the fist data of list. add to the front of the list.
	{
		new2 = (dhcp_client_node*)malloc(sizeof(dhcp_client_node));
		bzero(new2,sizeof(dhcp_client_node));
		memcpy(&new2->dce,&dce,sizeof(struct dhcp_client_elm));
		list_head = new2;
		list_tail = new2;
		flag = 1;
	}

	//step 3. write into file.
	if(flag==1)//list has data.
	{
		fs = fopen("/var/config/dhcpd.clients_fix","w+");
		if(fs!=NULL)
		{
			//write
			for(walk=list_head; walk!=NULL; walk=walk->next)
			{
				fwrite(&walk->dce, sizeof(struct dhcp_client_elm), 1, fs);
			}
			//free
			for(walk=list_head; walk!=NULL; walk=walk->next,free(del))
			{
				del = walk;
			}
			fclose(fs);
		}
	}
	return 0;
}

/* Write the specified lease to the current lease database file. */

int write_lease (lease)
	struct lease *lease;
{
	struct tm *t;
	char tbuf [64];
	int errors = 0;
	int i;

	if (counting)
		++count;
	errno = 0;
	fprintf (db_file, "lease %s {\n", piaddr (&lease -> ip_addr));
	if (errno) {
		++errors;
	}

	t = gmtime (&lease -> starts);
	sprintf (tbuf, "%d %d/%02d/%02d %02d:%02d:%02d;",
		 t -> tm_wday, t -> tm_year + 1900,
		 t -> tm_mon + 1, t -> tm_mday,
		 t -> tm_hour, t -> tm_min, t -> tm_sec);
	errno = 0;
	fprintf (db_file, "\tstarts %s\n", tbuf);
	if (errno) {
		++errors;
	}

	t = gmtime (&lease -> ends);
	sprintf (tbuf, "%d %d/%02d/%02d %02d:%02d:%02d;",
		 t -> tm_wday, t -> tm_year + 1900,
		 t -> tm_mon + 1, t -> tm_mday,
		 t -> tm_hour, t -> tm_min, t -> tm_sec);
	errno = 0;
	fprintf (db_file, "\tends %s", tbuf);
	if (errno) {
		++errors;
	}

	if (lease -> hardware_addr.hlen) {
		errno = 0;
		fprintf (db_file, "\n\thardware %s %s;",
			 hardware_types [lease -> hardware_addr.htype],
			 print_hw_addr (lease -> hardware_addr.htype,
					lease -> hardware_addr.hlen,
					lease -> hardware_addr.haddr));
		if (errno) {
			++errors;
		}

		/*TO DO - we might want to check that we don't overflow the hardware type*/
	}
	if (lease -> uid_len) {
		int j;
		errno = 0;
		fprintf (db_file, "\n\tuid %2.2x", lease -> uid [0]);
		if (errno) {
			++errors;
		}
		for (j = 1; j < lease -> uid_len; j++) {
			errno = 0;
			fprintf (db_file, ":%2.2x", lease -> uid [j]);
			if (errno) {
				++errors;
			}
		}
		putc (';', db_file);
	}
	if (lease -> flags & BOOTP_LEASE) {
		errno = 0;
		fprintf (db_file, "\n\tdynamic-bootp;");
		if (errno) {
			++errors;
		}
	}
	if (lease -> flags & ABANDONED_LEASE) {
		errno = 0;
		fprintf (db_file, "\n\tabandoned;");
		if (errno) {
			++errors;
		}
	}
	if (lease -> client_hostname) {
		for (i = 0; lease -> client_hostname [i]; i++)
			if (lease -> client_hostname [i] < 33 ||
			    lease -> client_hostname [i] > 126)
				goto bad_client_hostname;
		errno = 0;
		fprintf (db_file, "\n\tclient-hostname \"%s\";",
			 lease -> client_hostname);
		if (errno) {
			++errors;
		}
	}
       bad_client_hostname:
	if (lease -> hostname) {
		for (i = 0; lease -> hostname [i]; i++)
			if (lease -> hostname [i] < 33 ||
			    lease -> hostname [i] > 126)
				goto bad_hostname;
		errno = 0;
		errno = 0;
		fprintf (db_file, "\n\thostname \"%s\";",
			 lease -> hostname);
		if (errno) {
			++errors;
		}
	}
       bad_hostname:
	errno = 0;
	fputs ("\n}\n", db_file);
	if (errno) {
		++errors;
	}
	if (errors)
		note ("write_lease: unable to write lease %s",
		      piaddr (&lease -> ip_addr));
	return !errors;
}

/* Commit any leases that have been written out... */

int commit_leases ()
{
	/* Commit any outstanding writes to the lease database file.
	   We need to do this even if we're rewriting the file below,
	   just in case the rewrite fails. */
	if (fflush (db_file) == EOF) {
		note ("commit_leases: unable to commit: %m");
		return 0;
	}
	if (fsync (fileno (db_file)) < 0) {
		note ("commit_leases: unable to commit: %m");
		return 0;
	}

	/* If we've written more than fifty leases or if
	   we haven't rewritten the lease database in over an
	   hour, rewrite it now. */
	if (count > 50 || (count && cur_time - write_time > 3600)) {
		count = 0;
		write_time = cur_time;
		new_lease_file ();
	}
	return 1;
}

void db_startup ()
{
	/* Read in the existing lease file... */
	read_leases ();

	GET_TIME (&write_time);
	new_lease_file ();
}

void new_lease_file ()
{
	char newfname [512];
	char backfname [512];
	TIME t;
	int db_fd;

/* succ */
	path_dhcpd_db ="/var/config/dhcpd.leases";
/* succ */

	/* If we already have an open database, close it. */
	if (db_file) {
		fclose (db_file);
	}

	/* Make a temporary lease file... */
	GET_TIME (&t);
	sprintf (newfname, "%s.%d", path_dhcpd_db, (int)t);
	db_fd = open (newfname, O_WRONLY | O_TRUNC | O_CREAT, 0664);
	if (db_fd < 0) {
		error ("Can't create new lease file: %m");
	}
	if ((db_file = fdopen (db_fd, "w")) == NULL) {
		error ("Can't fdopen new lease file!");
	}

	/* Write an introduction so people don't complain about time
	   being off. */
/*
	fprintf (db_file, "# All times in this file are in UTC (GMT), not %s",
		 "your local timezone.   This is\n");
	fprintf (db_file, "# not a bug, so please don't ask about it.   %s",
		 "There is no portable way to\n");
	fprintf (db_file, "# store leases in the local timezone, so please %s",
		 "don't request this as a\n");
	fprintf (db_file, "# feature.   If this is inconvenient or %s",
		 "confusing to you, we sincerely\n");
	fprintf (db_file, "# apologize.   Seriously, though - don't ask.\n");
	fprintf (db_file, "# The format of this file is documented in the %s",
		 "dhcpd.leases(5) manual page.\n\n");
*///marked by Lily
	/* Write out all the leases that we know of... */
	counting = 0;
	write_leases ();

	/* Get the old database out of the way... */
	sprintf (backfname, "%s~", path_dhcpd_db);
	if (unlink (backfname) < 0 && errno != ENOENT)
		error ("Can't remove old lease database backup %s: %m",
		       backfname);
//	if (link (path_dhcpd_db, backfname) < 0)
//		error ("Can't backup lease database %s to %s: %m",
//		       path_dhcpd_db, backfname);
	
	/* Move in the new file... */
	if (rename (newfname, path_dhcpd_db) < 0)
		error ("Can't install new lease database %s to %s: %m",
		       newfname, path_dhcpd_db);

#if CONFIG_USER_FLATFSD_FLATFSD
	commitChanges();
#endif /* CONFIG_USER_FLATFSD_FLATFSD */
	counting = 1;
}
