/**
 * @file    IxQMgrQAccess.c
 *
 * @author Intel Corporation
 * @date    30-Oct-2001
 *
 * @brief   This file contains functions for putting entries on a queue and
 * removing entries from a queue.
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2001-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/

/*
 * Inlines are compiled as function when this is defined.
 * N.B. Must be placed before #include of "IxQMgr.h"
 */
#ifndef IXQMGR_H
#    define IXQMGRQACCESS_C
#else
#    error
#endif

/*
 * System defined include files.
 */

/*
 * User defined include files.
 */
#include "IxQMgr.h"
#include "IxQMgrAqmIf_p.h"
#include "IxQMgrQAccess_p.h"
#include "IxQMgrQCfg_p.h"
#include "IxQMgrDefines_p.h"

/*
 * Global variables and extern definitions
 */
extern IxQMgrQInlinedReadWriteInfo ixQMgrQInlinedReadWriteInfo[];

/*
 * Function definitions.
 */
void
ixQMgrQAccessInit (void)
{   
}

IX_STATUS
ixQMgrQReadWithChecks (IxQMgrQId qId,
                       UINT32 *entry)
{
    IxQMgrQEntrySizeInWords entrySizeInWords;
    IxQMgrQInlinedReadWriteInfo *infoPtr;

    if (NULL == entry)
    {
	return IX_QMGR_PARAMETER_ERROR;
    }

    /* Check QId */
    if (!ixQMgrQIsConfigured(qId))
    {
	return IX_QMGR_Q_NOT_CONFIGURED;
    }

    /* Get the q entry size in words */
    entrySizeInWords = ixQMgrQEntrySizeInWordsGet (qId);

    ixQMgrAqmIfQPop (qId, entrySizeInWords, entry);	    

    /* reset the current read count if the counter wrapped around 
    * (unsigned arithmetic)
    */
    infoPtr = &ixQMgrQInlinedReadWriteInfo[qId];
    if (infoPtr->qReadCount-- > infoPtr->qSizeInEntries)
    {
	infoPtr->qReadCount = 0;
    }

    /* Check if underflow occurred on the read */
    if (ixQMgrAqmIfUnderflowCheck (qId))
    {
	return IX_QMGR_Q_UNDERFLOW;
    }
    
    return IX_SUCCESS;
}

/* this function reads the remaining of the q entry
 * for queues configured with many words.
 * (the first word of the entry is already read 
 * in the inlined function and the entry pointer already
 * incremented
 */
IX_STATUS
ixQMgrQReadMWordsMinus1 (IxQMgrQId qId,
			 UINT32 *entry)
{
    IxQMgrQInlinedReadWriteInfo *infoPtr = &ixQMgrQInlinedReadWriteInfo[qId];
    UINT32 entrySize = infoPtr->qEntrySizeInWords;
    volatile UINT32 *qAccRegAddr = infoPtr->qAccRegAddr;
    
    while (--entrySize)
    {
	/* read the entry and accumulate the result */
	*(++entry) = IX_OSSERV_READ_LONG(++qAccRegAddr);
    }
    /* underflow is available for lower queues only */
    if (qId < IX_QMGR_MIN_QUEUPP_QID)
    {
	/* get the queue status */
	UINT32 status = IX_OSSERV_READ_LONG(infoPtr->qUOStatRegAddr);
	
	/* check the underflow status */
	if (status & infoPtr->qUflowStatBitMask)
	{
	    /* the queue is empty 
	     *  clear the underflow status bit if it was set 
	     */
	    IX_OSSERV_WRITE_LONG(infoPtr->qUOStatRegAddr,
				 status & ~infoPtr->qUflowStatBitMask);
	    return IX_QMGR_Q_UNDERFLOW;
	}
    }
    return IX_SUCCESS;
}

IX_STATUS
ixQMgrQWriteWithChecks (IxQMgrQId qId,
                        UINT32 *entry)
{
    IxQMgrQEntrySizeInWords entrySizeInWords;
    IxQMgrQInlinedReadWriteInfo *infoPtr;

    if (NULL == entry)
    {
	return IX_QMGR_PARAMETER_ERROR;
    }

    /* Check QId */
    if (!ixQMgrQIsConfigured(qId))
    {
	return IX_QMGR_Q_NOT_CONFIGURED;
    }

    /* Get the q entry size in words */
    entrySizeInWords = ixQMgrQEntrySizeInWordsGet (qId);
    
    ixQMgrAqmIfQPush (qId, entrySizeInWords, entry);

    /* reset the current read count if the counter wrapped around 
    * (unsigned arithmetic)
    */
    infoPtr = &ixQMgrQInlinedReadWriteInfo[qId];
    if (infoPtr->qWriteCount++ >= infoPtr->qSizeInEntries)
    {
	infoPtr->qWriteCount = infoPtr->qSizeInEntries;
    }

    /* Check if overflow occurred on the write*/
    if (ixQMgrAqmIfOverflowCheck (qId))
    {
	return IX_QMGR_Q_OVERFLOW;
    }
         
    return IX_SUCCESS;
}

IX_STATUS
ixQMgrQPeek (IxQMgrQId qId,
	     unsigned int entryIndex,
	     UINT32 *entry)
{
    unsigned int numEntries;

#ifndef NDEBUG
    if ((NULL == entry) || (entryIndex >= IX_QMGR_Q_SIZE_INVALID))
    {
	return IX_QMGR_PARAMETER_ERROR;
    }

    if (!ixQMgrQIsConfigured(qId))
    {
	return IX_QMGR_Q_NOT_CONFIGURED;
    }
#endif
    
    if (IX_SUCCESS != ixQMgrQNumEntriesGet (qId, &numEntries))
    {
	return IX_FAIL;
    }

    if (entryIndex >= numEntries) /* entryIndex starts at 0 */
    {
	return IX_QMGR_ENTRY_INDEX_OUT_OF_BOUNDS;
    }

    return ixQMgrAqmIfQPeek (qId, entryIndex, entry);
}

IX_STATUS
ixQMgrQPoke (IxQMgrQId qId,
	     unsigned entryIndex,
	     UINT32 *entry)
{
    unsigned int numEntries;

#ifndef NDEBUG
    if ((NULL == entry) || (entryIndex > 128))
    {
	return IX_QMGR_PARAMETER_ERROR;
    }

    if (!ixQMgrQIsConfigured(qId))
    {
	return IX_QMGR_Q_NOT_CONFIGURED;
    }
#endif
        
    if (IX_SUCCESS != ixQMgrQNumEntriesGet (qId, &numEntries))
    {
	return IX_FAIL;
    }

    if (numEntries < (entryIndex + 1)) /* entryIndex starts at 0 */
    {
	return IX_QMGR_ENTRY_INDEX_OUT_OF_BOUNDS;
    }

    return ixQMgrAqmIfQPoke (qId, entryIndex, entry);
}

IX_STATUS
ixQMgrQStatusGetWithChecks (IxQMgrQId qId,
                            IxQMgrQStatus *qStatus)
{
    if (NULL == qStatus)
    {
	return IX_QMGR_PARAMETER_ERROR;
    }
   
    if (!ixQMgrQIsConfigured (qId)) 
    {
        return IX_QMGR_Q_NOT_CONFIGURED;
    }

    ixQMgrAqmIfQueStatRead (qId, qStatus);

    return IX_SUCCESS;
}

IX_STATUS
ixQMgrQNumEntriesGet (IxQMgrQId qId,
		      unsigned *numEntriesPtr)
{
    UINT32 qPtrs;
    UINT32 qStatus;
    unsigned numEntries;
    IxQMgrQInlinedReadWriteInfo *infoPtr;


#ifndef NDEBUG
    if (NULL == numEntriesPtr)
    {
	return IX_QMGR_PARAMETER_ERROR;
    }

    /* Check QId */
    if (!ixQMgrQIsConfigured(qId))
    {
	return IX_QMGR_Q_NOT_CONFIGURED;
    }
#endif

    /* get fast access data */
    infoPtr = &ixQMgrQInlinedReadWriteInfo[qId];

    /* get snapshot */
    qPtrs = IX_OSSERV_READ_LONG(infoPtr->qConfigRegAddr);

    /* Mod subtraction of pointers to get number of words in Q. */
    numEntries = (qPtrs - (qPtrs >> 7)) & 0x7f;
  
    if (numEntries == 0)
    {
	/* 
	 * Could mean either full or empty queue
	 * so look at status
	 */
	ixQMgrAqmIfQueStatRead (qId, &qStatus);

	if (qId < IX_QMGR_MIN_QUEUPP_QID)
	{
	    if (qStatus & IX_QMGR_Q_STATUS_E_BIT_MASK)
	    {
		/* Empty */
		*numEntriesPtr = 0;
	    }
	    else if (qStatus & IX_QMGR_Q_STATUS_F_BIT_MASK)
	    {
		/* Full */
		*numEntriesPtr = infoPtr->qSizeInEntries;
	    }
	    else
	    {	    
		/* 
		 * Queue status and read/write pointers are volatile.
		 * The queue state has changed since we took the
		 * snapshot of the read and write pointers.
		 * Client can retry if they wish
		 */
		*numEntriesPtr = 0;
		return IX_QMGR_WARNING;
	    }
	}
	else /* It is an upper queue which does not have an empty status bit maintained */
	{
	    if (qStatus & IX_QMGR_Q_STATUS_F_BIT_MASK)
	    {
		/* The queue is Full at the time of snapshot. */
		*numEntriesPtr = infoPtr->qSizeInEntries;
	    }
	    else
	    {
  	       /* The queue is either empty, either moving,
	        * Client can retry if they wish
	        */
		*numEntriesPtr = 0;
	        return IX_QMGR_WARNING;
	    }
	}
    }
    else
    {
	*numEntriesPtr = (numEntries / infoPtr->qEntrySizeInWords) & (infoPtr->qSizeInEntries - 1);
    }
    
    return IX_SUCCESS;
}

