/** 
 * @file IxOsServicesMemAccess.h
 * 
 * @brief Header file for memory access
 * 
 * @par
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 1999-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#ifndef IxOsServicesMemAccess_H
#define IxOsServicesMemAccess_H

#include "IxTypes.h"
#include "IxOsServicesEndianess.h"

/**
 * @addtogroup IxOsServices
 * @par
 * <h2>OsServices I/O Memory Allocation and Access Routines</h2>
 * 
 * List of OSes and operating modes supported:
 * (Please update the list when adding a new OS or mode)
 *
 * - VxWorks BE
 * - VxWorks LE
 * - Linux BE
 *
 * 
 * <h3> Usage Information:</h3>
 *
 * @p IxOsServicesMemAccess.h defines OS/Endianess mode, memory mapped I/O access macros,
 * NPE-shared memory routines, SDRAM coherency mode, default component
 * coherency mode and static/dynamic memory mapping for every component.
 * The symbols defined below can be used and some overriden in the component-specific
 * section of IxOsServicesComponents.h. 
 *
 * @par OS/Endianess defines:
 * @par
 *   One of the following symbols will be defined:\n
 *   @b IX_OSSERV_VXWORKS_BE - component is compiled for VxWorks Big Endian \n
 *   @b IX_OSSERV_VXWORKS_LE - component is compiled for VxWorks Little Endian \n
 *   @b IX_OSSERV_LINUX_BE   - component is compiled for Linux Big Endian \n
 *
 * @par SDRAM coherency mode:
 *
 *   One of the following symbols will be defined:\n
 *   @b IX_SDRAM_BE                  - SDRAM is in Big Endian mode (default for Big Endian builds)\n
 *   @b IX_SDRAM_LE_ADDRESS_COHERENT - SDRAM is in Little Endian, Address Coherent Mode (not supported by current software)\n
 *   @b IX_SDRAM_LE_DATA_COHERENT    - SDRAM is in Little Endian, Data Coherent Mode (default for Little Endian builds)\n
 *
 * @par Static/dynamic memory mapping:
 *
 *   @b IX_STATIC_MEMORY_MAP  - component uses statically I/O mapped memory (default)\n
 *   @b IX_DYNAMIC_MEMORY_MAP - component uses OS-specific dynamically mapped I/O memory (define this in the component
 *                           specific section of IxOsServicesComponents.h to override the previous default value)\n
 *
 * @par Component coherency mode (define or override defaults in the component section of IxOsServicesComponents.h):
 *
 *   @b CSR_BE_MAPPING                  - component uses I/O memory in Big Endian mode (default in Big Endian Builds)\n
 *   @b CSR_LE_ADDRESS_COHERENT_MAPPING - component uses I/O memory in Little Endian, Address Coherent mode\n
 *   @b CSR_LE_DATA_COHERENT_MAPPING    - component uses I/O memory in Little Endian, Data Coherent mode\n
 *   @b CSR_NO_MAPPING                  - component does not use I/O memory (I/O read/write macros are not available)\n
 *
 * <h3>Macros for memory mapped I/O access:</h3>
 *
 *  Unless CSR_NO_MAPPING is defined, each component will have access to the following set of macros for reading and
 *  writing word (32 bit), short (16 bit) and byte (8 bits) data. The macros will perform all the necessary endianess conversions
 *  and use appropriate read and write functions in OSes where this is required.
 *
 *  The addresses (wAddr, sAddr, bAddr) should be volatile 32-bit pointers to UINT32, UINT16 and UINT8 respectively.
 *  The data (wData, sData, bData) should be UINT32, UINT16 and respectively UINT8 values.
 *
 *   @b IX_OSSERV_READ_LONG(wAddr)             - returns the 32-bit value at address wAddr\n
 *   @b IX_OSSERV_READ_SHORT(sAddr)            - returns the 16-bit value at address sAddr\n
 *   @b IX_OSSERV_READ_BYTE(bAddr)             - returns the 8-but value at address bAddr\n
 *   @b IX_OSSERV_WRITE_LONG(wAddr, wData)      - writes the 32-bit wData at address wAddr\n
 *   @b IX_OSSERV_WRITE_SHORT(sAddr, sData)    - writes the 16-bit sData at address sAddr\n
 *   @b IX_OSSERV_WRITE_BYTE(bAddr, bData)     - writes the 8-bit bData at address bAddr\n
 *
 * <h3> Macros for sharing data with the NPEs: </h3>
 *
 *  Each component will have access to the following set of macros for reading and writing word (32 bit) and short
 *  (16 bit) data, plus a macro for copying within SDRAM an array of words to be shared with an NPE. The macros will
 *  perform all the necessary endianess conversions depending on the SDRAM endianess and coherency mode to guarantee
 *  correct sharing of data between XScale components and NPEs.
 *
 *  The addresses (wAddr, sAddr, wSrcAddr and wDestAddr) should be 32-bit pointers to UINT32 and UINT16 respectively (volatility is NOT required).
 *  The data (wData and sData) should be UINT32 and respectively UINT16 values.
 *  The word count for the copy macro (wCount) should be a UINT32 value.
 *
 *  @b IX_OSSERV_READ_NPE_SHARED_LONG(wAddr)                              - returns the 32-bit value written by the NPE at address wAddr \n
 *  @b IX_OSSERV_READ_NPE_SHARED_SHORT(sAddr)                             - returns the 16-bit value written by the NPE at address sAddr \n
 *  @b IX_OSSERV_WRITE_NPE_SHARED_LONG(wAddr, wData)                      - writes 32-bit wData to be read by the NPE at address wAddr  \n
 *  @b IX_OSSERV_WRITE_NPE_SHARED_SHORT(sAddr, sData)                     - writes 16-bit sData to be read by the NPE at address sAddr \n
 *  @b IX_OSSERV_COPY_NPE_SHARED_LONG_ARRAY(wDestAddr, wSrcAddr, wCount)  - copies wCount 32-bit words shared with an NPE from wSrcAddr to wDestAddr \n
 *  @b IX_OSSERV_SWAP_NPE_SHARED_LONG(wData)                              - returns the correctly converted (if necessary) 32-bit value between NPE and SDRAM representation \n
 *
 */

/* Global BE switch
 * 
 *  Should be set only in BE mode and only if the component uses I/O memory.
 *  If I/O memory is not used the component-specific section in IxOsServicesComponents.h
 *  should #define CSR_NO_MAPPING instead */
#if defined (__BIG_ENDIAN)

#define CSR_BE_MAPPING

#endif /* Global BE switch */

/* By default only static memory maps in use;
   define IX_DYNAMIC_MEMORY_MAP per component if dynamic maps are
   used instead in that component */
#define IX_STATIC_MEMORY_MAP

/* SDRAM coherency mode

   Must be defined to BE, LE_DATA_COHERENT or LE_ADDRESS_COHERENT.
   The mode changes depending on OS */
#if defined (IX_OSSERV_LINUX_BE) || defined (IX_OSSERV_VXWORKS_BE)

#define IX_SDRAM_BE

#elif defined (IX_OSSERV_VXWORKS_LE)

#define IX_SDRAM_LE_DATA_COHERENT

#endif
  
/**************************************
 * Retrieve current component mapping *
 **************************************/

#include <IxOsServicesComponents.h>

/*******************************************************************
 * Turn off IX_STATIC_MEMORY map for components using dynamic maps *
 *******************************************************************/
#ifdef IX_DYNAMIC_MEMORY_MAP

#undef IX_STATIC_MEMORY_MAP

#endif

/************************************************************
 * Turn off BE access for components using LE or no mapping *
 ************************************************************/
#if (defined (CSR_LE_ADDRESS_COHERENT_MAPPING) || defined (CSR_LE_DATA_COHERENT_MAPPING) || defined (CSR_NO_MAPPING)) \
     && defined (CSR_BE_MAPPING)

#undef CSR_BE_MAPPING

#endif

/*****************
 * Safety checks *
 *****************/

/* At least one mapping must be defined - BE, LE_AC, LE_DC or NO_MAPPING */
#if !defined (CSR_BE_MAPPING) && !defined (CSR_LE_ADDRESS_COHERENT_MAPPING) && !defined (CSR_LE_DATA_COHERENT_MAPPING) && !defined (CSR_NO_MAPPING)

#error No I/O mapping is defined, unsupported component

#endif /* check at least one mapping */

/* No more than one mapping can be defined for a component */
#if (defined (CSR_BE_MAPPING) && (defined (CSR_LE_ADDRESS_COHERENT_MAPPING) || defined (CSR_LE_DATA_COHERENT_MAPPING) || defined (CSR_NO_MAPPING))) \
    || (defined (CSR_LE_ADDRESS_COHERENT_MAPPING) && (defined (CSR_BE_MAPPING) || defined (CSR_LE_DATA_COHERENT_MAPPING) || defined (CSR_NO_MAPPING))) \
    || (defined (CSR_LE_DATA_COHERENT_MAPPING) && (defined (CSR_BE_MAPPING) || defined (CSR_LE_ADDRESS_COHERENT_MAPPING) || defined (CSR_NO_MAPPING))) \
    || (defined (CSR_NO_MAPPING) && (defined (CSR_BE_MAPPING) || defined (CSR_LE_ADDRESS_COHERENT_MAPPING) || defined (CSR_LE_DATA_COHERENT_MAPPING)))
    
#ifdef CSR_BE_MAPPING
#warning CSR_BE_MAPPING is defined
#endif

#ifdef CSR_LE_ADDRESS_COHERENT_MAPPING
#warning CSR_LE_ADDRESS_COHERENT_MAPPING is defined
#endif

#ifdef CSR_LE_DATA_COHERENT_MAPPING
#warning CSR_LE_DATA_COHERENT_MAPPING is defined
#endif
    
#error More than one I/O mapping is defined, please check your component mapping

#endif /* check at most one mapping */

/* SDRAM coherency should be defined */
#if !defined (IX_SDRAM_BE) && !defined (IX_SDRAM_LE_DATA_COHERENT) && !defined (IX_SDRAM_LE_ADDRESS_COHERENT)

#error SDRAM coherency must be defined

#endif /* SDRAM coherency must be defined */

/* SDRAM coherency cannot be defined in several ways */
#if (defined (IX_SDRAM_BE) && (defined (IX_SDRAM_LE_DATA_COHERENT) || defined (IX_SDRAM_LE_ADDRESS_COHERENT))) \
    || (defined (IX_SDRAM_LE_DATA_COHERENT) && (defined (IX_SDRAM_BE) || defined (IX_SDRAM_LE_ADDRESS_COHERENT))) \
    || (defined (IX_SDRAM_LE_ADDRESS_COHERENT) && (defined (IX_SDRAM_BE) || defined (IX_SDRAM_LE_DATA_COHERENT)))

#error SDRAM coherency cannot be defined in more than one way

#endif /* SDRAM coherency must be defined exactly once */
  
/*********************
 * Read/write macros *
 *********************/

/* WARNING - except for addition of special cookie read/write macros (see below)
             these macros are NOT user serviceable. Please do not modify */

#define IX_OSSERV_READ_LONG_RAW(wAddr)          (*(wAddr))
#define IX_OSSERV_READ_SHORT_RAW(sAddr)         (*(sAddr))
#define IX_OSSERV_READ_BYTE_RAW(bAddr)          (*(bAddr))
#define IX_OSSERV_WRITE_LONG_RAW(wAddr, wData)  (*(wAddr) = (wData))
#define IX_OSSERV_WRITE_SHORT_RAW(sAddr,sData)  (*(sAddr) = (sData))
#define IX_OSSERV_WRITE_BYTE_RAW(bAddr, bData)  (*(bAddr) = (bData))

#ifdef __linux

/* Linux - specific cookie reads/writes. 
  Redefine per OS if dynamic memory maps are used
  and I/O memory is accessed via functions instead of raw pointer access. */
  
#define IX_OSSERV_READ_LONG_COOKIE(wCookie)           (readl((UINT32)wCookie))
#define IX_OSSERV_READ_SHORT_COOKIE(sCookie)          (readw((UINT32)sCookie))
#define IX_OSSERV_READ_BYTE_COOKIE(bCookie)           (readb((UINT32)bCookie))
#define IX_OSSERV_WRITE_LONG_COOKIE(wCookie, wData)   (writel(wData, (UINT32)wCookie))
#define IX_OSSERV_WRITE_SHORT_COOKIE(sCookie, sData)  (writew(sData, (UINT32)sCookie))
#define IX_OSSERV_WRITE_BYTE_COOKIE(bCookie, bData)   (writeb(bData, (UINT32)bCookie))

#endif /* linux */

#if defined (__vxworks) || (defined (__linux) && defined (IX_STATIC_MEMORY_MAP))

#define IX_OSSERV_READ_LONG_IO(wAddr)            IX_OSSERV_READ_LONG_RAW(wAddr)
#define IX_OSSERV_READ_SHORT_IO(sAddr)           IX_OSSERV_READ_SHORT_RAW(sAddr)
#define IX_OSSERV_READ_BYTE_IO(bAddr)            IX_OSSERV_READ_BYTE_RAW(bAddr)
#define IX_OSSERV_WRITE_LONG_IO(wAddr, wData)    IX_OSSERV_WRITE_LONG_RAW(wAddr, wData)
#define IX_OSSERV_WRITE_SHORT_IO(sAddr, sData)   IX_OSSERV_WRITE_SHORT_RAW(sAddr, sData)
#define IX_OSSERV_WRITE_BYTE_IO(bAddr, bData)    IX_OSSERV_WRITE_BYTE_RAW(bAddr, bData)

#elif defined (__linux) && !defined (IX_STATIC_MEMORY_MAP)

#include <asm/io.h>

#define IX_OSSERV_READ_LONG_IO(wAddr)            IX_OSSERV_READ_LONG_COOKIE(wAddr)
#define IX_OSSERV_READ_SHORT_IO(sAddr)           IX_OSSERV_READ_SHORT_COOKIE(sAddr)
#define IX_OSSERV_READ_BYTE_IO(bAddr)            IX_OSSERV_READ_BYTE_COOKIE(bAddr)
#define IX_OSSERV_WRITE_LONG_IO(wAddr, wData)    IX_OSSERV_WRITE_LONG_COOKIE(wAddr, wData)
#define IX_OSSERV_WRITE_SHORT_IO(sAddr, sData)   IX_OSSERV_WRITE_SHORT_COOKIE(sAddr, sData)
#define IX_OSSERV_WRITE_BYTE_IO(bAddr, bData)    IX_OSSERV_WRITE_BYTE_COOKIE(bAddr, bData)

#endif

#if defined (CSR_BE_MAPPING)

#define IX_OSSERV_READ_LONG(wAddr)          IX_OSSERV_BE_BUSTOXSL(IX_OSSERV_READ_LONG_IO((volatile UINT32 *) wAddr))
#define IX_OSSERV_READ_SHORT(sAddr)         IX_OSSERV_BE_BUSTOXSS(IX_OSSERV_READ_SHORT_IO((volatile UINT16 *) sAddr))
#define IX_OSSERV_READ_BYTE(bAddr)          IX_OSSERV_BE_BUSTOXSB(IX_OSSERV_READ_BYTE_IO((volatile UINT8 *) bAddr))
#define IX_OSSERV_WRITE_LONG(wAddr, wData)  IX_OSSERV_WRITE_LONG_IO((volatile UINT32 *) wAddr, IX_OSSERV_BE_XSTOBUSL((UINT32) wData))
#define IX_OSSERV_WRITE_SHORT(sAddr, sData) IX_OSSERV_WRITE_SHORT_IO((volatile UINT16 *) sAddr, IX_OSSERV_BE_XSTOBUSS((UINT16) sData))
#define IX_OSSERV_WRITE_BYTE(bAddr, bData)  IX_OSSERV_WRITE_BYTE_IO((volatile UINT8 *) bAddr, IX_OSSERV_BE_XSTOBUSB((UINT8) bData))

#elif defined (CSR_LE_ADDRESS_COHERENT_MAPPING)

#define IX_OSSERV_READ_LONG(wAddr)          IX_OSSERV_READ_LONG_IO((volatile UINT32 *) IX_OSSERV_LE_AC_BUSTOXSL((UINT32) wAddr))
#define IX_OSSERV_READ_SHORT(sAddr)         IX_OSSERV_READ_SHORT_IO((volatile UINT16 *) IX_OSSERV_LE_AC_BUSTOXSS((UINT32) sAddr))
#define IX_OSSERV_READ_BYTE(bAddr)          IX_OSSERV_READ_BYTE_IO((volatile UINT8 *) IX_OSSERV_LE_AC_BUSTOXSB((UINT32) bAddr))
#define IX_OSSERV_WRITE_LONG(wAddr, wData)  IX_OSSERV_WRITE_LONG_IO((volatile UINT32 *) IX_OSSERV_LE_AC_XSTOBUSL((UINT32) wAddr), (UINT32) wData)
#define IX_OSSERV_WRITE_SHORT(sAddr, sData) IX_OSSERV_WRITE_SHORT_IO((volatile UINT16 *) IX_OSSERV_LE_AC_XSTOBUSS((UINT32) sAddr), (UINT16) sData)
#define IX_OSSERV_WRITE_BYTE(bAddr, bData)  IX_OSSERV_WRITE_BYTE_IO((volatile UINT8 *) IX_OSSERV_LE_AC_XSTOBUSB((UINT32) bAddr), (UINT8) bData)

#elif defined (CSR_LE_DATA_COHERENT_MAPPING)

/* Inline functions are required here to avoid reading the same I/O location 2 or 4 times for the byte swap */
static __inline__ UINT32 ixOsServDataCoherentLongReadSwap(volatile UINT32 *wAddr)
{
  UINT32 wData = IX_OSSERV_READ_LONG_IO(wAddr);
  return IX_OSSERV_LE_DC_BUSTOXSL(wData);
}

static __inline__ UINT16 ixOsServDataCoherentShortReadSwap(volatile UINT16 *sAddr)
{
  UINT16 sData = IX_OSSERV_READ_SHORT_IO(sAddr);
  return IX_OSSERV_LE_DC_BUSTOXSS(sData);
}

static __inline__ void ixOsServDataCoherentLongWriteSwap(volatile UINT32 *wAddr, UINT32 wData)
{
  wData = IX_OSSERV_LE_DC_XSTOBUSL(wData);
  IX_OSSERV_WRITE_LONG_IO(wAddr, wData);
}

static __inline__ void ixOsServDataCoherentShortWriteSwap(volatile UINT16 *sAddr, UINT16 sData)
{
  sData = IX_OSSERV_LE_DC_XSTOBUSS(sData);
  IX_OSSERV_WRITE_LONG_IO(sAddr, sData);
}

#define IX_OSSERV_READ_LONG(wAddr)          ixOsServDataCoherentLongReadSwap((volatile UINT32 *) wAddr)
#define IX_OSSERV_READ_SHORT(sAddr)         ixOsServDataCoherentShortReadSwap((volatile UINT16 *) sAddr)
#define IX_OSSERV_READ_BYTE(bAddr)          IX_OSSERV_LE_DC_BUSTOXSB(IX_OSSERV_READ_BYTE_IO((volatile UINT8 *) bAddr))
#define IX_OSSERV_WRITE_LONG(wAddr, wData)  ixOsServDataCoherentLongWriteSwap((volatile UINT32 *) wAddr, (UINT32) wData)
#define IX_OSSERV_WRITE_SHORT(sAddr, sData) ixOsServDataCoherentShortWriteSwap((volatile UINT16 *) sAddr, (UINT16) sData)
#define IX_OSSERV_WRITE_BYTE(bAddr, bData)  IX_OSSERV_WRITE_BYTE_IO((volatile UINT8 *) bAddr, IX_OSSERV_LE_DC_XSTOBUSB((UINT8) bData))

#endif

/* Reads/writes to and from memory shared with NPEs - depends on the SDRAM coherency */
#if defined (IX_SDRAM_BE)

#define IX_OSSERV_READ_NPE_SHARED_LONG(wAddr)            IX_OSSERV_READ_LONG_RAW(wAddr)
#define IX_OSSERV_READ_NPE_SHARED_SHORT(sAddr)           IX_OSSERV_READ_SHORT_RAW(sAddr)
#define IX_OSSERV_WRITE_NPE_SHARED_LONG(wAddr, wData)    IX_OSSERV_WRITE_LONG_RAW(wAddr, wData)
#define IX_OSSERV_WRITE_NPE_SHARED_SHORT(sAddr, sData)   IX_OSSERV_WRITE_SHORT_RAW(sAddr, sData)

#define IX_OSSERV_SWAP_NPE_SHARED_LONG(wData)            (wData)
#define IX_OSSERV_SWAP_NPE_SHARED_SHORT(sData)           (sData)

#elif defined (IX_SDRAM_LE_ADDRESS_COHERENT)

#define IX_OSSERV_READ_NPE_SHARED_LONG(wAddr)            IX_OSSERV_READ_LONG_RAW(wAddr)
#define IX_OSSERV_READ_NPE_SHARED_SHORT(sAddr)           IX_OSSERV_READ_SHORT_RAW(IX_OSSERV_SWAP_SHORT_ADDRESS(sAddr))
#define IX_OSSERV_WRITE_NPE_SHARED_LONG(wAddr, wData)    IX_OSSERV_WRITE_LONG_RAW(wAddr, wData)
#define IX_OSSERV_WRITE_NPE_SHARED_SHORT(sAddr, sData)   IX_OSSERV_WRITE_SHORT_RAW(IX_OSSERV_SWAP_SHORT_ADDRESS(sAddr), sData)

#define IX_OSSERV_SWAP_NPE_SHARED_LONG(wData)            (wData)
#define IX_OSSERV_SWAP_NPE_SHARED_SHORT(sData)           (sData)

#elif defined (IX_SDRAM_LE_DATA_COHERENT)

#define IX_OSSERV_READ_NPE_SHARED_LONG(wAddr)            IX_OSSERV_SWAP_LONG(IX_OSSERV_READ_LONG_RAW(wAddr))
#define IX_OSSERV_READ_NPE_SHARED_SHORT(sAddr)           IX_OSSERV_SWAP_SHORT(IX_OSSERV_READ_SHORT_RAW(sAddr))
#define IX_OSSERV_WRITE_NPE_SHARED_LONG(wAddr, wData)    IX_OSSERV_WRITE_LONG_RAW(wAddr, IX_OSSERV_SWAP_LONG(wData))
#define IX_OSSERV_WRITE_NPE_SHARED_SHORT(sAddr, sData)   IX_OSSERV_WRITE_SHORT_RAW(sAddr, IX_OSSERV_SWAP_SHORT(sData))

#define IX_OSSERV_SWAP_NPE_SHARED_LONG(wData)            IX_OSSERV_SWAP_LONG(wData)
#define IX_OSSERV_SWAP_NPE_SHARED_SHORT(sData)           IX_OSSERV_SWAP_SHORT(sData)

#endif

#define IX_OSSERV_COPY_NPE_SHARED_LONG_ARRAY(wDestAddr, wSrcAddr, wCount) \
  { \
    UINT32 i; \
    \
    for ( i = 0 ; i < wCount ; i++ ) \
    { \
      * ((UINT32 *) wDestAddr) + i) = IX_OSSERV_READ_SHARED_NPE_LONG(((UINT32 *) wSrcAddr) + i); \
    }; \
  };

#endif /* IxOsServicesMemAccess_H */
