/*
 * FileName:    IxTimersCodelet.c
 * Author: Intel Corporation
 * Created:     15 May 2002
 * Description: IXP425 Hardware Timers access layer
 *
 * Design Notes:
 * A workaround is provided to recover from possible
 * status lost across many timers.
 *
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 */

#include <stdio.h>

#include "ixp425.h"

#include "IxOsServices.h"
#include "IxOsServicesMemMap.h"
#include "IxTimersCodelet_p.h"

/* Watchdog timer key protection */
#define IX_TIMER_WDOG_UNLOCK    (0x0000482e)
#define IX_TIMER_WDOG_LOCK      (0x000055aa)
/* Watchdog timer control */
#define IX_TIMER_WDOG_DISABLED  (0x0)
#define IX_TIMER_WDOG_ENABLE    (BIT(1) | BIT(2))

/* Offsets for timers */
#define IX_CODELET_TIMERS_OSTS_OFFSET  (0x00000000)
#define IX_CODELET_TIMERS_OST1_OFFSET  (0x00000004)
#define IX_CODELET_TIMERS_OSRT1_OFFSET (0x00000008)
#define IX_CODELET_TIMERS_OST2_OFFSET  (0x0000000C)
#define IX_CODELET_TIMERS_OSRT2_OFFSET (0x00000010)
#define IX_CODELET_TIMERS_OSWT_OFFSET  (0x00000014)
#define IX_CODELET_TIMERS_OSWE_OFFSET  (0x00000018)
#define IX_CODELET_TIMERS_OSWK_OFFSET  (0x0000001C)
#define IX_CODELET_TIMERS_OSST_OFFSET  (0x00000020)

/* Register access */
#define IX_REG_READ(reg, result) \
((result) = IX_OSSERV_READ_LONG((VUINT32 *)(reg)))

#define IX_REG_WRITE(reg, result) \
(IX_OSSERV_WRITE_LONG((VUINT32 *)(reg), (UINT32)(result)))

#define IX_REG_STALL(reg) \
	(ixTimerRegStall((UINT32)(reg)))

static __inline__ void ixTimerRegStall(UINT32 reg)
{
    /* read and stall, and bypass possible compiler optimisations */
    __asm__ volatile (" ldr r1,[%0]; mov r1,r1; "
		      : : "r" (reg) : "r1");
}

/*
 * Structure containing timers' registers addresses
 */
typedef struct
{
    UINT32 *ost1;     /* Timer 1 Timestamp register*/
    UINT32 *ost2;     /* Timer 2 Timestamp register*/
    UINT32 *oswt;     /* Watchdog Timer register */
    UINT32 *osts;     /* Continuous Timestamp register */
    UINT32 *osst;     /* Timer Status register */
    UINT32 *osrt1;    /* Timer 1 Reload register */
    UINT32 *osrt2;    /* Timer 2 Reload register */
    UINT32 *oswk;     /* Watchdog Key register */
    UINT32 *oswe;     /* Watchdog Enable register */
} IxCodeletTimersRegisters;

PRIVATE IxCodeletTimersRegisters ixCodeletTimersRegisters;

/* For each timer, store
 * - the ISR entry point,
 * - the ISR argument,
 * - the status mask
 * - the timer interrupt vector
 * - a count of lost status
*/
PRIVATE IxTimerIsr ixTimerIsr[IX_TIMER_MAX];
PRIVATE void *     ixTimerArg[IX_TIMER_MAX];
PRIVATE UINT32     ixTimerMask[IX_TIMER_MAX];
PRIVATE UINT32     ixTimerLvl[IX_TIMER_MAX];
PRIVATE UINT32     ixTimerLostIsrCount[IX_TIMER_MAX];
PRIVATE UINT32     ixTimerLostIsrEventCount     = 0;
PRIVATE UINT32     ixTimerEnabledStatusMask     = 0;
PRIVATE BOOL       ixTimerStatusRecoveryEnable = TRUE;
PRIVATE BOOL       ixTimerInitialised = FALSE;
PRIVATE BOOL       ixTimerPmuTimerEnabled = FALSE;

/*
 * Virtual Address corresponding to the timers' base address
 */
PRIVATE UINT32 ixTimerVirtualAddress = 0;

/* private function prototypes
 */
PRIVATE void ixTimerDummyRoutine(void *arg);
PRIVATE UINT32 ixTimerSnapshotGet(UINT32 statusMask);
PRIVATE void ixTimerLostStatusProcess(UINT32 statusMask);
PRIVATE void ixTimerSafeInt (void *arg);
PRIVATE void ixTimer1Int (void *arg);
PRIVATE void ixTimer2Int (void *arg);
PRIVATE void ixTimerWdogInt (void *arg);
PRIVATE void ixTimerTsInt (void *arg);

/* default isr : does nothing, not expected to be called,
 * Its implementation avoids the need to test function pointers
 * against NULL values.
 *
 */
PRIVATE void ixTimerDummyRoutine(void *arg)
{
}

PUBLIC IX_STATUS ixTimerMemMap(void)
{
    /* Timer registers are within the timer address space. So mapping the
     * entire timer address space */
    ixTimerVirtualAddress =
	(UINT32) (IX_OSSERV_MEM_MAP((UINT32)IX_OSSERV_OSTS_PHYS_BASE, (UINT32)IX_OSSERV_OSTS_MAP_SIZE));
    if (ixTimerVirtualAddress == 0)
    {
	return IX_FAIL;
    }

    /* Calculate the address of each necessary timers' register */
    ixCodeletTimersRegisters.ost1 =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OST1_OFFSET);
    ixCodeletTimersRegisters.ost2 =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OST2_OFFSET);
    ixCodeletTimersRegisters.oswt =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OSWT_OFFSET);
    ixCodeletTimersRegisters.osts =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OSTS_OFFSET);
    ixCodeletTimersRegisters.osst =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OSST_OFFSET);
    ixCodeletTimersRegisters.osrt1 =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OSRT1_OFFSET);
    ixCodeletTimersRegisters.osrt2 =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OSRT2_OFFSET);
    ixCodeletTimersRegisters.oswk =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OSWK_OFFSET);
    ixCodeletTimersRegisters.oswe =
	(UINT32 *)(ixTimerVirtualAddress + IX_CODELET_TIMERS_OSWE_OFFSET);

    return IX_SUCCESS;
}

PUBLIC void ixTimerMemUnmap(void)
{
    /* Set the address of registers to zero */
    memset (&ixCodeletTimersRegisters, 0, sizeof(ixCodeletTimersRegisters));

    /* Unmap the timer address space */
    IX_OSSERV_MEM_UNMAP(ixTimerVirtualAddress);

    /* Reset the virtual address to zero */
    ixTimerVirtualAddress = 0;
}

PUBLIC IX_STATUS ixTimerInit (BOOL recoverFromLostStatus)
{
    IxTimerId timerIndex;

    if (ixTimerInitialised)
    {
	return IX_FAIL;
    }
    if (ixTimerMemMap() != IX_SUCCESS)
    {
	return IX_FAIL;
    }
    ixTimerLostIsrEventCount = 0;
    ixTimerEnabledStatusMask = 0;
    for (timerIndex = 0; timerIndex < IX_TIMER_MAX; timerIndex++)
    {
	ixTimerIsr[timerIndex]          = ixTimerDummyRoutine;
	ixTimerArg[timerIndex]          = (void *)0;
	ixTimerLostIsrCount[timerIndex] = 0;
    }
    ixTimerMask[IX_TIMER_1]    = IXP425_OSST_TIMER_1_PEND;
    ixTimerMask[IX_TIMER_2]    = IXP425_OSST_TIMER_2_PEND;
    ixTimerMask[IX_TIMER_WDOG] = IXP425_OSST_TIMER_WDOG_PEND;
    ixTimerMask[IX_TIMER_TS]   = IXP425_OSST_TIMER_TS_PEND;
    ixTimerMask[IX_TIMER_PMU]  = 0;
    ixTimerLvl[IX_TIMER_1]     = IXP425_INT_LVL_TIMER1;
    ixTimerLvl[IX_TIMER_2]     = IXP425_INT_LVL_TIMER2;
    ixTimerLvl[IX_TIMER_WDOG]  = IXP425_INT_LVL_WDOG;
    ixTimerLvl[IX_TIMER_TS]    = IXP425_INT_LVL_TIMESTAMP;
    ixTimerLvl[IX_TIMER_PMU]   = IXP425_INT_LVL_XSCALE_PMU;

    ixTimerStatusRecoveryEnable = recoverFromLostStatus;
    ixTimerPmuTimerEnabled = FALSE;
    ixTimerInitialised = TRUE;
    return IX_SUCCESS;
}

/* Status recovery handling :
 *
 * Problem : When 2 timers expire at the exact same cycle, resetting 
 * the status may clear the other status.
 *
 * Workaround : When clearing the status, check if other
 * status are overwritten by checking the condition .
 * - get a snapshot
 * - clear the status for this interrupt (this may clear other status)
 * - get a second snapshot
 * - read the current status
 * - check if the current status is cleared when it should be set.
 *
*/
PRIVATE UINT32 ixTimerSnapshotGet(UINT32 statusMask)
{
    /* select the timers to be checked */
    UINT32 statusCheck = (ixTimerEnabledStatusMask & ~statusMask);
    /* snapshot data */
    static UINT32 timer1Before = 0;
    static UINT32 timer1After = 0;
    static UINT32 timer2Before = 0;
    static UINT32 timer2After = 0;
    static UINT32 timerWdogBefore = 0;
    static UINT32 timerWdogAfter = 0;
    static UINT32 timerTsBefore = 0;
    static UINT32 timerTsAfter = 0;
    UINT32 statusAfter;
    /* returnStatus will contain the lost interrupt mask */
    UINT32 returnStatus = 0;

#ifdef __linux
    int lockKey = ixOsServIntLock();
#endif

    /* get a snapshot */
    if ((statusCheck & IXP425_OSST_TIMER_1_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.ost1, timer1Before);
    }
    if ((statusCheck & IXP425_OSST_TIMER_2_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.ost2, timer2Before);
    }
    if ((statusCheck & IXP425_OSST_TIMER_WDOG_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.oswt, timerWdogBefore);
    }
    if ((statusCheck & IXP425_OSST_TIMER_TS_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.osts, timerTsBefore);
    }

    /* clear the status bit */
    IX_REG_WRITE(ixCodeletTimersRegisters.osst, statusMask);

    /* get a second snapshot */
    if ((statusCheck & IXP425_OSST_TIMER_1_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.ost1, timer1After);
    }
    if ((statusCheck & IXP425_OSST_TIMER_2_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.ost2, timer2After);
    }
    if ((statusCheck & IXP425_OSST_TIMER_WDOG_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.oswt, timerWdogAfter);
    }
    if ((statusCheck & IXP425_OSST_TIMER_TS_PEND) != 0)
    {
	IX_REG_READ(ixCodeletTimersRegisters.osts, timerTsAfter);
    }

    /* read the status bit */
    IX_REG_READ(ixCodeletTimersRegisters.osst, statusAfter);

    if ((statusCheck & IXP425_OSST_TIMER_1_PEND) != 0)
    {
	/* test the countDown counter wrapped around and the 
	   status is cleared
	 */
	if ((timer1After > timer1Before) && 
	    ((statusAfter & IXP425_OSST_TIMER_1_PEND) == 0))
	{
	    returnStatus |= IXP425_OSST_TIMER_1_PEND;
	}
    }

    if ((statusCheck & IXP425_OSST_TIMER_2_PEND) != 0)
    {
	/* test the countDown counter wrapped around and the 
	   status is cleared
	 */
	if ((timer2After > timer2Before) && 
	    ((statusAfter & IXP425_OSST_TIMER_2_PEND) == 0))
	{
	    returnStatus |= IXP425_OSST_TIMER_2_PEND;
	}
    }
    if ((statusCheck & IXP425_OSST_TIMER_WDOG_PEND) != 0)
    {
	/* test the countDown counter wrapped around and the 
	   status is cleared
	 */
	if ((timerWdogAfter > timerWdogBefore) && 
	    ((statusAfter & IXP425_OSST_TIMER_WDOG_PEND) == 0))
	{
	    returnStatus |= IXP425_OSST_TIMER_WDOG_PEND;
	}
    }
    if ((statusCheck & IXP425_OSST_TIMER_TS_PEND) != 0)
    {
	/* test the countUp counter wrapped around and the 
	   status is cleared
	 */
	if ((timerTsAfter < timerTsBefore) && 
	    ((statusAfter & IXP425_OSST_TIMER_TS_PEND) == 0))
	{
	    returnStatus |= IXP425_OSST_TIMER_TS_PEND;
	}
    }

#ifdef __linux
    ixOsServIntUnlock(lockKey);
#endif

    /* returnStatus contains now a mask of the Timers
     * whose status has ben overwritten.
     */
    return returnStatus;
}

/* This function processes the interrupts when a status loss
 * has been detected when writing the status register.
 *
 * param statusMask the mask of the ISRs to run
 * return : void
 *
*/
PRIVATE void ixTimerLostStatusProcess(UINT32 statusMask)
{
    IxTimerId timerIndex;

    ixTimerLostIsrEventCount++;

    for (timerIndex = 0; timerIndex < IX_TIMER_MAX; timerIndex++)
    {
	if ((statusMask & ixTimerMask[timerIndex]) != 0)
	{
	    (*ixTimerIsr[timerIndex])(ixTimerArg[timerIndex]);
	    ixTimerLostIsrCount[timerIndex]++;
	}
    }
}

/* If the lost status workaround is needed, then this interrupt
 * get a snapshot, processes the current interrupt and other
 * interrupts if their status has been lost
 *
 */
PRIVATE void ixTimerSafeInt (void *arg)
{
    IxTimerId timerIndex = (IxTimerId)arg;
    UINT32 lostStatusMask = ixTimerSnapshotGet(ixTimerMask[timerIndex]);

    (*ixTimerIsr[timerIndex])(ixTimerArg[timerIndex]);

    if (lostStatusMask != 0)
    {
	ixTimerLostStatusProcess(lostStatusMask);
    }
}

/* If the lost sttaus workaround is not needed, then these interrupts
 * will just clear the status and call the user ISR as registered
 * during the call to ixTimerBind().
 *
 * Note : because the Wdog timer reload value is not automatically
 * reloaded, clear the status has to be done after the watchdog
 * Isr reloads this timer (using ixTimerEnable()).
 */
PRIVATE void ixTimer1Int (void *arg)
{
    IX_REG_WRITE(ixCodeletTimersRegisters.osst, IXP425_OSST_TIMER_1_PEND);
    (*ixTimerIsr[IX_TIMER_1])(arg);
    IX_REG_STALL(ixCodeletTimersRegisters.osst);
}

PRIVATE void ixTimer2Int (void *arg)
{
    IX_REG_WRITE(ixCodeletTimersRegisters.osst, IXP425_OSST_TIMER_2_PEND);
    (*ixTimerIsr[IX_TIMER_2])(arg);
    IX_REG_STALL(ixCodeletTimersRegisters.osst);
}

PRIVATE void ixTimerWdogInt (void *arg)
{
    (*ixTimerIsr[IX_TIMER_WDOG])(arg);
    IX_REG_WRITE(ixCodeletTimersRegisters.osst, IXP425_OSST_TIMER_WDOG_PEND);
    IX_REG_STALL(ixCodeletTimersRegisters.osst);
}

PRIVATE void ixTimerTsInt (void *arg)
{
    IX_REG_WRITE(ixCodeletTimersRegisters.osst, IXP425_OSST_TIMER_TS_PEND);
    (*ixTimerIsr[IX_TIMER_TS])(arg);
    IX_REG_STALL(ixCodeletTimersRegisters.osst);
}

PUBLIC IX_STATUS ixTimerBind (IxTimerId timer,
			      IxTimerIsr isr, 
			      void *arg)
{
    IxTimerIsr isrToBind = (IxTimerIsr)NULL;
    void *argToBind = (void *)0;

    if ((isr == NULL) || 
	(timer >= IX_TIMER_MAX) || 
	!ixTimerInitialised ||
	(ixTimerIsr[timer] != ixTimerDummyRoutine))
    {
	return IX_FAIL;
    }
    
    /* sanity : disable the timer */
    ixTimerDisable(timer);

    /* Depending on the initialisation parameters, 
     * set the parameters of the bind function as follows
     * - if status recovery is enabled, the interrupt
     *   to be registered is a function which recover from
     *   lost status.
     * - if status recovery is not enabled, the interrupt
     *   to be registered is a function which just clear
     *   the ost status and invoke the user function
     * - The pmu Timer does not need a recovery. It does not
     *   need to ba acknowledged. Then the user Usr is bound
     *   to the interrupt vector.
     */
    if (ixTimerStatusRecoveryEnable && (timer != IX_TIMER_PMU))
    {
	/* interrupt will go thru the safe interrupt handler */
	isrToBind = ixTimerSafeInt;
	/* the registered parameter is the timer index */
	argToBind = (void *)timer;
    }
    else
    {
	/* interrupt will go directly to the user function */
	switch (timer)
	{
	case IX_TIMER_1:    
	    isrToBind = ixTimer1Int; 
	    break;
	case IX_TIMER_2:    
	    isrToBind = ixTimer2Int; 
	    break;
	case IX_TIMER_WDOG: 
	    isrToBind = ixTimerWdogInt; 
	    break;
	case IX_TIMER_TS:   
	    isrToBind = ixTimerTsInt; 
	    break;
	case IX_TIMER_PMU:   
	    isrToBind = isr; 
	    break;
	default:
	    break;
	}
	/* the registered parameter is the user parameter */
	argToBind = arg;
    }

    /* registers the user function and its parameter */
    ixTimerIsr[timer] = isr;
    ixTimerArg[timer] = arg;

    /* register the intterupt service routine */
    if (ixOsServIntBind(ixTimerLvl[timer], 
			isrToBind, 
			argToBind) != IX_SUCCESS)
    {
	/* sanity : register the dummy routine */
	ixTimerIsr[timer] = ixTimerDummyRoutine;
	return IX_FAIL;
    }

    if (timer == IX_TIMER_PMU)
    {
	UINT32 mask = 
	    BIT(0) | /* enable counters */
	    BIT(2);  /* reset clock counter; */
	    
	    __asm__(" mcr p14,0,%0,c0,c1,0; "  /* write control register */
		    : : "r" (mask));
    }

    return IX_SUCCESS;
}

PUBLIC IX_STATUS ixTimerEnable (IxTimerId timer,
				UINT32 downCounter,
				BOOL oneShot)
{
    UINT32 timerSetup;
    int lockKey;

    switch(timer)
    {
    case IX_TIMER_1:
	if ((downCounter & IXP425_OST_RELOAD_MASK) != 0)
	{
	    return IX_FAIL;
	}
	if (oneShot)
	{ 
	    timerSetup = downCounter | IXP425_OST_ONE_SHOT | IXP425_OST_ENABLE;
	}  
	else
        {
	    timerSetup = downCounter | IXP425_OST_ENABLE;
	}
	IX_REG_WRITE(ixCodeletTimersRegisters.osrt1, timerSetup);
	break;
    case IX_TIMER_2:
	if ((downCounter & IXP425_OST_RELOAD_MASK) != 0)
	{
	    return IX_FAIL;
	}
	if (oneShot)
	{ 
	    timerSetup = downCounter | IXP425_OST_ONE_SHOT | IXP425_OST_ENABLE;
	}  
	else
        {
	    timerSetup = downCounter | IXP425_OST_ENABLE;
	}
	IX_REG_WRITE(ixCodeletTimersRegisters.osrt2, timerSetup);
	break;
    case IX_TIMER_WDOG:
	if (oneShot == FALSE)
	{ 
	    return IX_FAIL;
	}
	lockKey = ixOsServIntLock();
	IX_REG_WRITE(ixCodeletTimersRegisters.oswk, IX_TIMER_WDOG_UNLOCK);
	IX_REG_WRITE(ixCodeletTimersRegisters.oswe, IX_TIMER_WDOG_DISABLED);
	IX_REG_WRITE(ixCodeletTimersRegisters.oswt, downCounter);
	IX_REG_WRITE(ixCodeletTimersRegisters.oswe, IX_TIMER_WDOG_ENABLE);
	IX_REG_WRITE(ixCodeletTimersRegisters.oswk, IX_TIMER_WDOG_LOCK);
	ixOsServIntUnlock(lockKey);
	break;
    case IX_TIMER_TS:
	if (oneShot == FALSE)
	{ 
	    return IX_FAIL;
	}
	if (downCounter != 0)
	{
	    IX_REG_WRITE(ixCodeletTimersRegisters.osts, 0 - downCounter);
	}
	break;
    case IX_TIMER_PMU:
	if (oneShot == FALSE)
	{ 
	    return IX_FAIL;
	}
	if (downCounter != 0)
	{
 	    /* set the pmu timer and enable it.
	     * (handle multiple overflow)
	     */
	    __asm__(" mrc p14,0,r1,c1,c1,0; "  /* get current counter */
		    " subs r1,r1,%0; "         /* sub downcounter */
		    " subcs pc,pc,#12; "       /* jump back until ok */
		    " mcr p14,0,r1,c1,c1,0; "  /* write current counter */ 
		    : : "r" (downCounter) : "r1");
	}

	__asm__(" mrc p14,0,r1,c4,c1,0; "  /* get int enable register */
		" orr r1,r1,#1; " 
		" mcr p14,0,r1,c5,c1,0; "  /* clear overflow */ 
		" mcr p14,0,r1,c4,c1,0; "  /* enable interrupts */ 
		: : : "r1");

	ixTimerPmuTimerEnabled = TRUE;
	return IX_SUCCESS;
	break;
    default:
	return IX_FAIL;
    }

    /* add this interrupt to the interrupt list */
    ixTimerEnabledStatusMask |= ixTimerMask[timer];

    return IX_SUCCESS;
}

PUBLIC IX_STATUS ixTimerDownCounterGet (IxTimerId timer,
					UINT32 *downCounter)
{
    UINT32 tempCounter;

    if (downCounter == NULL)
    {
	return IX_FAIL;
    }

    switch(timer)
    {
    case IX_TIMER_1:
	IX_REG_READ(ixCodeletTimersRegisters.ost1, *downCounter);
	break;
    case IX_TIMER_2:
	IX_REG_READ(ixCodeletTimersRegisters.ost2, *downCounter);
	break;
    case IX_TIMER_WDOG:
	IX_REG_READ(ixCodeletTimersRegisters.oswt, *downCounter);
	break;
    case IX_TIMER_TS:
	IX_REG_READ(ixCodeletTimersRegisters.osts, tempCounter);
	*downCounter = 0 - tempCounter;
	break;
    case IX_TIMER_PMU:
	__asm__(" mrc p14,0,%0,c1,c1,0; "  /* get current counter */
		: "=r" (tempCounter));
	*downCounter = 0 - tempCounter;
	break;
    default:
	return IX_FAIL;
    }
    return IX_SUCCESS;
}

PUBLIC IX_STATUS ixTimerDisable (IxTimerId timer)
{
    int lockKey;

    switch(timer)
    {
    case IX_TIMER_1:
	IX_REG_WRITE(ixCodeletTimersRegisters.osrt1, IXP425_OST_DISABLED);
	break;
    case IX_TIMER_2:
	IX_REG_WRITE(ixCodeletTimersRegisters.osrt2, IXP425_OST_DISABLED);
	break;
    case IX_TIMER_WDOG:
	lockKey = ixOsServIntLock();
	IX_REG_WRITE(ixCodeletTimersRegisters.oswk, IX_TIMER_WDOG_UNLOCK);
	IX_REG_WRITE(ixCodeletTimersRegisters.oswe, IX_TIMER_WDOG_DISABLED);
	IX_REG_WRITE(ixCodeletTimersRegisters.oswk, IX_TIMER_WDOG_LOCK);
	ixOsServIntUnlock(lockKey);
	break;
    case IX_TIMER_TS:
	break;
    case IX_TIMER_PMU:
	ixTimerPmuTimerEnabled = FALSE;
	__asm__(" mrc p14,0,r1,c4,c1,0; "  /* get int enable register */
		" and r1,r1,#0x1e; " 
		" mcr p14,0,r1,c4,c1,0; "  /* disable interrupts */ 
		: : : "r1");
	return IX_SUCCESS;
    default:
	return IX_FAIL;
    }

    /* remove this interrupt from the interrupt list */
    ixTimerEnabledStatusMask &= ~ixTimerMask[timer];

    /* clear pending interrupts */
    IX_REG_WRITE(ixCodeletTimersRegisters.osst, ixTimerMask[timer]);
    IX_REG_STALL(ixCodeletTimersRegisters.osst);

    return IX_SUCCESS;
}

PUBLIC IX_STATUS ixTimerUnbind (IxTimerId timer)
{
    /* parameter checks */
    if (timer >= IX_TIMER_MAX || !ixTimerInitialised)
    {
	return IX_FAIL;
    }
    
    /* sanity : disable the service */
    ixTimerDisable(timer);

    /* check isr is registered */
    if (ixTimerIsr[timer] != ixTimerDummyRoutine)
    {
	/* sanity : register the dummy routine */
	ixTimerIsr[timer] = ixTimerDummyRoutine;
	if (ixOsServIntUnbind(ixTimerLvl[timer]) != IX_SUCCESS)
	{
	    return IX_FAIL;
	}
    }
    return IX_SUCCESS;
}

PUBLIC void ixTimerShow(void)
{
    printf("Timer 1 .................... : %s\n",
	   ((ixTimerEnabledStatusMask & IXP425_OSST_TIMER_1_PEND) == 0) ?
	   "Disabled" : "Enabled");
    printf("Timer 2 .................... : %s\n",
	   ((ixTimerEnabledStatusMask & IXP425_OSST_TIMER_2_PEND) == 0) ?
	   "Disabled" : "Enabled");
    printf("Timestamp Timer ............ : %s\n",
	   ((ixTimerEnabledStatusMask & IXP425_OSST_TIMER_TS_PEND) == 0) ?
	   "Disabled" : "Enabled");
    printf("Watchdog Timer ............. : %s\n",
	   ((ixTimerEnabledStatusMask & IXP425_OSST_TIMER_WDOG_PEND) == 0) ?
	   "Disabled" : "Enabled");
    printf("Pmu Timer .................. : %s\n",
	   (ixTimerPmuTimerEnabled ? "Enabled" : "Disabled"));

    if (ixTimerStatusRecoveryEnable)
    {
	printf("Results from Lost Status recovery\n");
	printf("Status lost ................ : %10u\n", 
	       ixTimerLostIsrEventCount);
	printf("Timer 1   status recovered . : %10u\n", 
	       ixTimerLostIsrCount[IX_TIMER_1]);
	printf("Timer 2   status recovered . : %10u\n", 
	       ixTimerLostIsrCount[IX_TIMER_2]);
	printf("Timestamp status recovered . : %10u\n", 
	       ixTimerLostIsrCount[IX_TIMER_TS]);
	printf("Watchdog  status recovered . : %10u\n", 
	       ixTimerLostIsrCount[IX_TIMER_WDOG]);
    }
    else
    {
	printf("Lost Status recovery NOT enabled\n");
    }
}

