#include <stdio.h>
#include <sys/time.h>

#include "timerqueue.h"

TimerQueue::TimerQueue()
{
  events.next = 0;
  events.callback = 0;
  events.cookie = 0;
}

TimerQueue::~TimerQueue()
{
  while (events.next) {
    TimerEvent *e = events.next;
    
    events.next = e->next;

    delete e;
  }
}

/**
 * Compare two timevals.
 *
 * Returns 0 if equal, < 0 if tv1 < tv2, and > 0 if tv1 > tv2.
 */
static int compare_tv(const struct timeval& tv1, const struct timeval& tv2)
{
  if (tv1.tv_sec == tv2.tv_sec) {
    return(tv1.tv_usec - tv2.tv_usec);
  }
  return(tv1.tv_sec - tv2.tv_sec);
}

void TimerQueue::add(unsigned long ms, TimerQueueCallback *callback, void *cookie)
{
  TimerEvent *e = new TimerEvent;
  e->callback = callback;
  e->cookie = cookie;
  gettimeofday(&e->tv, 0);

  /* Add the offset and normalise */
  e->tv.tv_usec += ms * 1000L;
  e->tv.tv_sec += e->tv.tv_usec / 1000000L;
  e->tv.tv_usec = e->tv.tv_usec % 1000000L;

  /* Now insert it in the correct location */
  TimerEvent *ee;
  for (ee = &events; ee->next; ee = ee->next) {
    if (compare_tv(e->tv, ee->next->tv) < 0) {
      /* Need to insert it between ee and ee->next */
      break;
    }
  }
  e->next = ee->next;
  ee->next = e;
}

void TimerQueue::check_events()
{
  struct timeval now;
  gettimeofday(&now, 0);

  for (TimerEvent *ee = &events; ee->next; ) {
    if (compare_tv(ee->next->tv, now) <= 0) {
      /* Invoke the event callback */
      ee->next->callback(this, ee->next->cookie);

      /* And remove this entry */
      TimerEvent *e = ee->next;
      ee->next = ee->next->next;
      delete e;
    }
    else {
      ee = ee->next;
    }
  }
}

int TimerQueue::get_timeout(struct timeval& tv)
{
  if (!events.next) {
    return(1);
  }
  struct timeval now;
  gettimeofday(&now, 0);

  timerclear(&tv);

  if (compare_tv(events.next->tv, now) > 0) {
    tv.tv_sec = events.next->tv.tv_sec - now.tv_sec;
    if (events.next->tv.tv_usec < now.tv_usec) {
      tv.tv_usec = 1000000L + events.next->tv.tv_usec - now.tv_usec;
      tv.tv_sec--;
    }
    else {
      tv.tv_usec = events.next->tv.tv_usec - now.tv_usec;
    }
  }
  return(0);
}
