#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <assert.h>

#include "config.h"
#include "gets_alloc.h"

/*#define DEBUG*/

#define MAX_ERROR_MESSAGE 300

/* This routine is called when we've failed to parse the input file properly.
 * It sets conf->error to non-zero and creates an appropriate
 * error message in conf->error_string.
 */
static void fail(config_t *conf, const char *fmt, ...)
{
	va_list args;
	int pos = 0;

	va_start(args, fmt);

	if (!conf->error_string) {
		conf->error_string = malloc(MAX_ERROR_MESSAGE);
	}

	conf->error = -1;
	pos += snprintf(conf->error_string + pos, MAX_ERROR_MESSAGE - pos, "%s:%d:", conf->filename, conf->line);
	pos += vsnprintf(conf->error_string + pos, MAX_ERROR_MESSAGE - pos, fmt, args);
	/*pos += snprintf(conf->error_string + pos, MAX_ERROR_MESSAGE - pos, "\n");*/
	va_end(args);
}

/* Uses gets_alloc() to read lines one-at-a-time from the file.
 * 
 * Comments and trailing whitespace are removed and multiple spaces and
 * tabs are replaced with single spaces.
 *
 * This routine also silently skips over blank and comment lines.
 *
 * Call gets_alloc_done() when finished.
 *
 * WARNING: This function is not reentrant. It uses a static buffer.
 */
static char *config_gets(FILE *stream, int *lineno)
{
	/* Get the next line from the stream */
	for (;;) {
		char *s;
		char *p;

		if ((s = gets_alloc(stream)) == NULL) {
			return NULL;
		}
		(*lineno)++;
		/* Remove the comment if present */
		if ((p = strchr(s, '#')) != 0) {
			*p = '\0';
		}
		/* Now ditch trailing white space */
		if (!*s) {
			continue;
		}
		p = s + strlen(s) - 1;
		while (p >= s && isspace(*p)) {
			*p-- = 0;
		}

		if (*s) {
			return(s);
		}
		/* Reduced the string to nothing, try again */
	}
}

/* This routine loads up and parses the entire config structure into memory
 *
 * Note that if an error occurs during parsing, the config may be partially loaded.
 */
int config_read(const char *filename, config_t *conf)
{
	FILE	*confile;
	char	*buf = NULL;
	char	*p, *q;
	config_section_t *s = 0;

	/* Other initialisation */
	conf->filename = filename;
	conf->line = 0;
	conf->error = 0;
	if (conf->error_string) {
		free(conf->error_string);
		conf->error_string = 0;
	}

	/* Open our input file */
	confile = fopen(filename, "r");
	if (confile == NULL) {
		// This is not so bad
		//fprintf(stderr, "Failed to open configuration file %s\n", filename);
		return(0);
	}

	buf = config_gets(confile, &conf->line);

	/* Process the input section at a time */
	while (buf) {
		if (isspace(buf[0])) {
			fail(conf, "expected section header");
			return(-1);
		}

		/* Separate the two words */
		for (p=buf+1; *p && !isspace(*p); p++) {
		}

		if (*p == '\0') {
			fail(conf, "section header format error: no section name");
			return(-1);
		}
		*p++ = '\0';

		while(isspace(*p)) {
			p++;
		}

		s = config_section_add(conf, buf, p);
		if (!s) {
			fail(conf, "invalid section name: %s", p);
			return(-1);
		}
#ifdef DEBUG
		printf("After section_create():");
		{
			config_section_t *s = conf->sections;
			while (s) {
				printf("section %s\n", s->name);
				s = s->next;
			}
		}
#endif

		/* Now to process the contents of the section */
		while ((buf = config_gets(confile, &conf->line)) != NULL) {
			int c;
			if (!isspace(buf[0])) {
				/* Next section */
				break;
			}

			/* Strip leading space */
			for (p=buf; isspace(*p); p++) {
			}

			/* Separate two parameters "p=q" or "p q" */
			for (q=p; *q != '\0' && *q != '=' && !isspace(*q); q++) {
			}

			if (*q == '\0') {
				fail(conf, "syntax error in parameter setting: no value");
				return(-1);
			}

			if (*q != '=') {
				/* "p q", so eat anything in 'q' until we hit '=' */
				for (*q++ = '\0'; *q != '\0' && *q != '='; q++) {
				}

				if (*q == '\0') {
					fail(conf, "syntax error in parameter setting: no =");
					return(-1);
				}
			}

			/* Terminate 'p' and strip leading space from 'q' */
			for (*q++ = '\0'; *q != '\0' && isspace(*q); q++) {
			}

			/* de-quote strings if required */
			if (*q == '"') {
				c = strlen(++q)-1;
				if (q[c] != '"') {
					fail(conf, "unmatched quotes in parameter string");
					return(-1);
				}
				q[c] = '\0';
			}
			if (config_param_add_value(s, p, q) != 0) {
				fail(conf, "duplicated parameter %s", p);
				return(-1);
			}
		}
	}
	fclose(confile);

	gets_alloc_done();

	return(0);
}
