/**
 * @file IxOsBuffPoolMgt.c
 *
 * @date 9 Oct 2002
 *
 * @brief This file contains the mbuf pool management implementation.
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 *
 * @sa IxOsBuffPoolMgt.h
 * @sa IxOsBuffMgt.h
 */

/*
 * Put the system defined include files required.
 */

#ifdef __vxworks

#include <vxWorks.h>
#include <net/mbuf.h>
#include <config.h>
#include "IxOsBufLib.h"

#endif


/*
 * Put the user defined include files required.
 */

#include "IxTypes.h"
#include "IxAssert.h"
#include "IxOsCacheMMU.h"
#include "IxOsBuffPoolMgt.h"
#include "IxOsServices.h"


/*
 * #defines and macros used in this file.
 */
#ifndef __vxworks

/* IxMbufWrapper
 * 
 * This struct contains a regular mbuf and some extra "hidden"
 * fields which are needed by the IxOsBuffPoolMgt component.
 */
typedef struct
{
    IX_MBUF_POOL *parentPool;
    UINT8 *mdataOrig;
    int mlenOrig;
    UINT32 signature;
} IxMbufWrapperHiddenFields;

/* Calculate padding size so that the hidden portion of
 * the mbuf wrapper fills a single cache-line
 */
#define IX_MBUF_WRAPPER_PADDING_SIZE \
    (IX_XSCALE_CACHE_LINE_SIZE - (sizeof (IxMbufWrapperHiddenFields)))
#define IX_MBUF_WRAPPER_PADDING_WORDS \
    (IX_MBUF_WRAPPER_PADDING_SIZE / sizeof (unsigned))
#define IX_MBUF_WRAPPER_PADDING_PATTERN 0xF000BAAA
#define IX_MBUF_WRAPPER_SIGNATURE 0x8BADF00D

typedef struct
{
    /* Regular mbuf */
    IX_MBUF realMbuf;
    
/* PRIVATE FIELDS - Only to be used by this component */
    unsigned padding[IX_MBUF_WRAPPER_PADDING_WORDS];
    IxMbufWrapperHiddenFields hidden;
} IxMbufWrapper;

#endif  /* ndef __vxworks */

/*
 * Variable declarations global to this file only.  Externs are followed by
 * static variables.
 */

static IX_MBUF_POOL ixOsBuffPools[IX_MBUF_MAX_POOLS];
static int ixOsBuffPoolsInUse = 0;


/*
 * Extern function prototypes.
 */

#ifdef __vxworks
#ifdef INCLUDE_NET_SHOW
#include <netShow.h>
void netPoolShow(NET_POOL_ID pNetPool);
#endif

/* check consistency with the usage of cache */
#ifdef IX_ACC_CACHE_ENABLED
/* when cache is enabled, mbufs need to be aligned */
#define USE_ALIGNED_MBUFS
#else
/* undefine this macro to use the standard vxWorks pools of mbufs */
#define USE_ALIGNED_MBUFS 
#endif

#else /* __vxworks */
/* buffers are always cached and aligned on linux implementations */
#endif /* __vxworks */


/*
 * Function definition: ixOsBuffPoolAllocate
 */
IX_MBUF_POOL *
ixOsBuffPoolAllocate (void)
{
    IX_ASSERT (ixOsBuffPoolsInUse <= IX_MBUF_MAX_POOLS);
    return &ixOsBuffPools[ixOsBuffPoolsInUse++];
}


/*
 * Function definition: ixOsBuffPoolDataAreaSizeGet
 */
UINT32 
ixOsBuffPoolDataAreaSizeGet(int count, int size)
{
    UINT32 memorySize;
#ifndef __vxworks                                                   
    memorySize = count * IX_MBUF_POOL_SIZE_ALIGN(size);
#else
#ifndef USE_ALIGNED_MBUFS
    memorySize = count * (size + sizeof(void *));
#else
    memorySize = ixAlignedPoolDataSize(count, size);
#endif
#endif
    return memorySize;
}

/*
 * Function definition: ixOsBuffPoolMbufAreaSizeGet
 */
UINT32 
ixOsBuffPoolMbufAreaSizeGet(int count)
{
    UINT32 memorySize;
#ifndef __vxworks
    memorySize = count * IX_MBUF_POOL_SIZE_ALIGN(sizeof(IX_MBUF));
#else
#ifndef USE_ALIGNED_MBUFS
    memorySize =  
	(count * (M_BLK_SZ + sizeof (long))) + 
	(count * CL_BLK_SZ);
#else
    memorySize = ixAlignedPoolMbufSize(count);
#endif
#endif
    return memorySize;
}


#ifndef __vxworks

/*
 * Function definition: ixOsBuffPoolMbufInit
 */

/* Allocate memory for mbuf and data and initialise mbuf header fields.
 *
 * This function allocates memory for an individual mbuf contained in
 * an mbuf wrapper, defined above.  The purpose of the wrapper is to
 * add extra fields to the mbuf header which are hidden from the user.
 * These extra fields are intended for use only by IxOsBuffPoolMgt.
 */
IX_MBUF *
ixOsBuffPoolMbufInit (int mbufSizeAligned,
                      int dataSizeAligned,
                      IX_MBUF_POOL *poolPtr)
{
    UINT32 i;
    UINT8 *dataPtr;
    IX_MBUF *realMbufPtr;
    IxMbufWrapper *wrapperMbufPtr;
    
    /* Allocate cache-aligned memory for mbuf header, including wrapper */
    wrapperMbufPtr = (IxMbufWrapper *) IX_ACC_DRV_DMA_MALLOC (mbufSizeAligned);
    IX_ASSERT (wrapperMbufPtr != NULL);
    realMbufPtr = &wrapperMbufPtr->realMbuf;
    memset (wrapperMbufPtr, 0, mbufSizeAligned);

    /* Allocate cache-aligned memory for mbuf data */
    dataPtr = (UINT8 *) IX_ACC_DRV_DMA_MALLOC (dataSizeAligned);
    IX_ASSERT (dataPtr != NULL);
    memset (dataPtr, 0, dataSizeAligned);

    /* Fill in mbuf header fields */
    IX_MBUF_MDATA (realMbufPtr) = dataPtr;
    wrapperMbufPtr->hidden.mdataOrig  = dataPtr;
    IX_MBUF_MLEN (realMbufPtr) = dataSizeAligned;
    wrapperMbufPtr->hidden.mlenOrig  = dataSizeAligned;

    IX_MBUF_NET_POOL (realMbufPtr) = (unsigned int) poolPtr;
    wrapperMbufPtr->hidden.parentPool = poolPtr;

    /* Fill in padding field in mbuf wrapper */
    for (i = 0; i < IX_MBUF_WRAPPER_PADDING_WORDS; i++)
    {
	wrapperMbufPtr->padding[i] = IX_MBUF_WRAPPER_PADDING_PATTERN;
    }
    wrapperMbufPtr->hidden.signature = IX_MBUF_WRAPPER_SIGNATURE;

    /* update some statistical information */
    poolPtr->mbufMemSize += mbufSizeAligned;
    poolPtr->dataMemSize += dataSizeAligned;

    return realMbufPtr;
}
#endif /* ndef __vxworks */


/*
 * Function definition: ixOsBuffPoolInit
 */

#ifdef __vxworks

IX_STATUS
ixOsBuffPoolInit (IX_MBUF_POOL **poolPtrPtr,
		  int count,
		  int size,
		  char *name)
{
    void *poolBufPtr;
    void *poolDataPtr;
    int mbufMemSize;
    int dataMemSize;
    
    poolBufPtr = IX_MBUF_POOL_MBUF_AREA_ALLOC(count, mbufMemSize);
    IX_ASSERT(poolBufPtr != NULL);
    poolDataPtr = IX_MBUF_POOL_DATA_AREA_ALLOC(count, size, dataMemSize);
    IX_ASSERT(poolDataPtr != NULL);

    return ixOsBuffPoolInitNoAlloc (poolPtrPtr, 
				    poolBufPtr, 
				    poolDataPtr, 
				    count, 
				    size, 
				    name);
}

#else

IX_STATUS
ixOsBuffPoolInit (IX_MBUF_POOL **poolPtrPtr,
		  int count,
		  int size,
		  char *name)
{
    IX_MBUF *currentMbufPtr;
    int mbufSizeAligned;
    int dataSizeAligned;
    int i = 0;

    /* check parameters */
    if ((poolPtrPtr == NULL)  ||
        (count <= 0)          ||
        (name == NULL)        ||
        (strlen(name) > IX_MBUF_POOL_NAME_LEN))
    {
        ixOsServLog (LOG_ERROR, "ixOsBuffPoolInitNoAlloc(): "
                     "ERROR - Invalid Parameter (%u %u %u %u)\n",
                     (UINT32)poolPtrPtr,
                     (UINT32)count,
                     (UINT32)name,
                     (UINT32)strlen(name),
		     0, 0);
        IX_ASSERT(0);
    }
    
    /* Allocate a new pool */
    *poolPtrPtr = ixOsBuffPoolAllocate();

    /* calculate mbuf header and data sizes to ensure cache-alignment */
    mbufSizeAligned = IX_MBUF_POOL_SIZE_ALIGN (sizeof (IxMbufWrapper));
    dataSizeAligned = IX_MBUF_POOL_SIZE_ALIGN (size);

    /* Initialise pool parameters */
    (*poolPtrPtr)->nextFreeBuf = NULL;
    (*poolPtrPtr)->mbufMemPtr = NULL;
    (*poolPtrPtr)->dataMemPtr = NULL;
    (*poolPtrPtr)->bufDataSize = dataSizeAligned;
    (*poolPtrPtr)->totalBufsInPool = count;
    (*poolPtrPtr)->poolAllocType = IX_MBUF_POOL_TYPE_SYS_ALLOC;
    strcpy ((*poolPtrPtr)->name, name);

    for (i = 0; i < count; i++)
    {
	/* create an mbuf */
	currentMbufPtr = ixOsBuffPoolMbufInit (mbufSizeAligned,
					       dataSizeAligned,
					       *poolPtrPtr);
	/* Add it to the pool */
	ixOsBuffPoolBufFree (currentMbufPtr);

	/* flush the pool information to RAM */
	IX_ACC_DATA_CACHE_FLUSH (currentMbufPtr, mbufSizeAligned);
    }

    return IX_SUCCESS;
}

#endif


/*
 * Function definition: ixOsBuffPoolInitNoAlloc
 */

IX_STATUS
ixOsBuffPoolInitNoAlloc (IX_MBUF_POOL **poolPtrPtr, void *poolBufPtr, void *poolDataPtr, int count, int size, char *name)
{
#ifndef __vxworks
    int i, mbufSizeAligned, sizeAligned;
    IX_MBUF *currentMbufPtr = NULL;
    IX_MBUF *nextMbufPtr = NULL;
#else
    M_CL_CONFIG ixMbufClBlkCfg;
    CL_DESC ixMbufClDesc;
#endif

    /* check parameters */
    if ((poolPtrPtr == NULL)  ||
#ifdef __vxworks
        (poolDataPtr == NULL) ||
#endif
        (poolBufPtr == NULL)  ||
        (count <= 0)          ||
        (strlen(name) > IX_MBUF_POOL_NAME_LEN) ||
        /* checks for alignment of memory on cache-line boundaries */
        (((unsigned)poolBufPtr % IX_XSCALE_CACHE_LINE_SIZE) != 0) || 
        (((unsigned)poolDataPtr % IX_XSCALE_CACHE_LINE_SIZE) != 0))
    {
        ixOsServLog (LOG_ERROR, "ixOsBuffPoolInitNoAlloc(): "
        "ERROR - Invalid Parameter (%u %u %u %u %u %u)\n", 
        (UINT32)poolPtrPtr,
        (UINT32)poolDataPtr,
        (UINT32)poolBufPtr,
	(UINT32)count,
        (UINT32)name,
        (UINT32)strlen(name));
        IX_ASSERT(0);
    }

    *poolPtrPtr = ixOsBuffPoolAllocate();

#ifndef __vxworks
    /* Adjust sizes to ensure alignment on cache line boundaries */
    mbufSizeAligned = IX_MBUF_POOL_SIZE_ALIGN(sizeof(IX_MBUF));
    /* clear the mbuf memory area */
    memset (poolBufPtr, 0, mbufSizeAligned * count);
    
    if (poolDataPtr != NULL)
    {
        /* Adjust sizes to ensure alignment on cache line boundaries */
        sizeAligned = IX_MBUF_POOL_SIZE_ALIGN(size);
        /* clear the data memory area */
	memset (poolDataPtr, 0, sizeAligned * count);
    }
    else
    {
        sizeAligned = 0;
    }

    strcpy ((*poolPtrPtr)->name, name);

    (*poolPtrPtr)->mbufMemPtr = poolBufPtr;
    (*poolPtrPtr)->dataMemPtr = poolDataPtr;
    (*poolPtrPtr)->bufDataSize = sizeAligned;
    (*poolPtrPtr)->totalBufsInPool = count;
    (*poolPtrPtr)->mbufMemSize = mbufSizeAligned * count;
    (*poolPtrPtr)->dataMemSize = sizeAligned * count;
    
    currentMbufPtr = (IX_MBUF *)poolBufPtr;
    (*poolPtrPtr)->nextFreeBuf = currentMbufPtr;

    for (i = 0; i < count; i++)
    {
        if (i < (count - 1))
        {
            nextMbufPtr = (IX_MBUF *)((unsigned)currentMbufPtr + mbufSizeAligned);
        }
        else
        {   /* last mbuf in chain */
            nextMbufPtr = NULL;
        }
        IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(currentMbufPtr) = nextMbufPtr;
        IX_MBUF_NET_POOL(currentMbufPtr) = (unsigned int) *poolPtrPtr;

        if (poolDataPtr != NULL)
        {
            IX_MBUF_MDATA(currentMbufPtr) = poolDataPtr;
            IX_MBUF_MLEN(currentMbufPtr) = size;
            poolDataPtr = (void *)((unsigned)poolDataPtr + sizeAligned);
        }

        currentMbufPtr = nextMbufPtr;
    }

    /* update the number of free buffers in the pool */
    (*poolPtrPtr)->freeBufsInPool = count;

    (*poolPtrPtr)->poolAllocType = IX_MBUF_POOL_TYPE_USER_ALLOC;

#else  /* ndef __vxworks */

    ixMbufClBlkCfg.mBlkNum  = count;
    ixMbufClBlkCfg.clBlkNum = count;
    ixMbufClBlkCfg.memSize = IX_MBUF_POOL_MBUF_AREA_SIZE_ALIGNED(count);
    ixMbufClBlkCfg.memArea = poolBufPtr;

    ixMbufClDesc.clSize = size;
    ixMbufClDesc.clNum  = count;
    ixMbufClDesc.memSize = IX_MBUF_POOL_DATA_AREA_SIZE_ALIGNED(count, size);
    ixMbufClDesc.memArea = poolDataPtr;

#ifdef USE_ALIGNED_MBUFS
    if (netPoolInit (*poolPtrPtr, &ixMbufClBlkCfg, &ixMbufClDesc, 1, _pIxOsBufLibPoolFuncTbl) != OK)
    {
        /* pool was not initialised successfully */
        return IX_FAIL;
    }
#else
    if (netPoolInit (*poolPtrPtr, &ixMbufClBlkCfg, &ixMbufClDesc, 1, NULL) != OK)
    {
        /* pool was not initialised successfully */
        return IX_FAIL;
    }
#endif
#endif  /* __vxworks */

    /* flush the pool information to RAM */
    IX_ACC_DATA_CACHE_FLUSH(poolBufPtr,
			    IX_MBUF_POOL_MBUF_AREA_SIZE_ALIGNED(count));
    if (poolDataPtr != NULL)
    {
	IX_ACC_DATA_CACHE_FLUSH(poolDataPtr,
				IX_MBUF_POOL_DATA_AREA_SIZE_ALIGNED(count,
								    size));
    }

    return IX_SUCCESS;
}


/*
 * Function definition: ixOsBuffPoolUnchainedBufGet
 */

IX_STATUS
ixOsBuffPoolUnchainedBufGet (IX_MBUF_POOL *poolPtr, IX_MBUF **newBufPtrPtr)
{
#ifndef __vxworks
    int lock;
#else
    int bufSize;
#endif

    IX_STATUS status = IX_SUCCESS;

    /* check parameters */
    if ((poolPtr == NULL) || (newBufPtrPtr == NULL))
    {
        ixOsServLog (LOG_ERROR, "ixOsBuffPoolUnchainedBufGet(): "
		     "ERROR - Invalid Parameter\n", 0, 0, 0, 0, 0, 0);
        return IX_FAIL;
    }

#ifndef __vxworks
    lock = ixOsServIntLock();
    
    *newBufPtrPtr = poolPtr->nextFreeBuf;
    if (*newBufPtrPtr)
    {
        poolPtr->nextFreeBuf = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(*newBufPtrPtr);
        IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(*newBufPtrPtr) = NULL;

        /* update the number of free buffers in the pool */
        poolPtr->freeBufsInPool--;
    }
    else
    {
        /* Could not get a free buffer */ 
        status = IX_FAIL;
    }
  
    ixOsServIntUnlock(lock);

#else /* ndef __vxworks */

    bufSize = poolPtr->pClBlkHead->clSize;

    *newBufPtrPtr = netTupleGet(poolPtr, bufSize, M_DONTWAIT, MT_DATA, FALSE);
    if (*newBufPtrPtr == NULL)
    {
        status = IX_FAIL;
    }
#endif /* ndef __vxworks */

    return status;
}


/*
 * Function definition: ixOsBuffPoolBufFree
 */

IX_MBUF *
ixOsBuffPoolBufFree (IX_MBUF *bufPtr)
{
#ifndef __vxworks
    int lock;
    IX_MBUF_POOL *poolPtr;
    IX_MBUF *nextBufPtr;
#else
#endif

    /* check parameters */
    if (bufPtr == NULL)
    {
        ixOsServLog (LOG_ERROR, "ixOsBuffPoolChainedBufFree(): "
                    "ERROR - Invalid Parameter\n", 0, 0, 0, 0, 0, 0);
        return NULL;
    }

#ifndef __vxworks
    poolPtr = (IX_MBUF_POOL *) IX_MBUF_NET_POOL(bufPtr);

    lock = ixOsServIntLock();

    /* check the mbuf wrapper signature (if mbuf wrapper was used) */
    if (poolPtr->poolAllocType == IX_MBUF_POOL_TYPE_SYS_ALLOC)
    {
	IX_ENSURE (((IxMbufWrapper *)bufPtr)->hidden.signature ==
		   IX_MBUF_WRAPPER_SIGNATURE,
		   "ixOsBuffPoolBufFree: ERROR - Invalid mbuf signature.");
    }

    nextBufPtr = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(bufPtr);
    
    IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(bufPtr) = poolPtr->nextFreeBuf;
    poolPtr->nextFreeBuf = bufPtr;
    
    /* update the number of free buffers in the pool */
    poolPtr->freeBufsInPool++;

    ixOsServIntUnlock(lock);

    return nextBufPtr;

#else  /* ndef __vxworks */

    return netMblkClFree(bufPtr);
#endif /* ndef __vxworks */
}
    

/*
 * Function definition: ixOsBuffPoolBufChainFree
 */

void
ixOsBuffPoolBufChainFree (IX_MBUF *bufPtr)
{
#ifndef __vxworks
    while((bufPtr = ixOsBuffPoolBufFree(bufPtr)));
#else  /* ndef __vxworks */
    netMblkClChainFree(bufPtr);
#endif /* ndef __vxworks */
}


/*
 * Function definition: ixOsBuffPoolShow
 */

IX_STATUS
ixOsBuffPoolShow (IX_MBUF_POOL *poolPtr)
{
#ifndef __vxworks
    IX_MBUF *nextBufPtr;
    int count = 0;
    int lock;
#else
#endif

    /* check parameters */
    if (poolPtr == NULL)
    {
        ixOsServLog (LOG_ERROR, "ixOsBuffPoolShow(): "
                    "ERROR - Invalid Parameter", 0, 0, 0, 0, 0, 0);
        return IX_FAIL;
    }

#ifndef __vxworks
    lock = ixOsServIntLock();
        count = poolPtr->freeBufsInPool;
        nextBufPtr = poolPtr->nextFreeBuf;
    ixOsServIntUnlock(lock);

    printf ("=== POOL INFORMATION ===\n"); 
    printf ("Pool Name:                   %s\n", poolPtr->name);
    printf ("Pool Allocation Type:        %d\n", poolPtr->poolAllocType);
    printf ("Pool Mbuf Mem Usage (bytes): %d\n", poolPtr->mbufMemSize);
    printf ("Pool Data Mem Usage (bytes): %d\n", poolPtr->dataMemSize);
    printf ("Mbuf Data Capacity  (bytes): %d\n", poolPtr->bufDataSize);
    printf ("Total Mbufs in Pool:         %d\n", poolPtr->totalBufsInPool);
    printf ("Available Mbufs:             %d\n", count);
    printf ("Next Available Mbuf:         %p\n", nextBufPtr);

    if (poolPtr->poolAllocType == IX_MBUF_POOL_TYPE_USER_ALLOC)
    {
	printf ("Mbuf Mem Area Start address: %p\n", poolPtr->mbufMemPtr);
	printf ("Data Mem Area Start address: %p\n", poolPtr->dataMemPtr);
    }

#else  /* ndef __vxworks */

#ifdef INCLUDE_NET_SHOW
    netPoolShow (poolPtr);
#endif

#endif /* ndef __vxworks */

    return IX_SUCCESS;
}


/*
 * Function definition: ixOsBuffPoolBufDataPtrReset
 */

IX_STATUS
ixOsBuffPoolBufDataPtrReset (IX_MBUF *bufPtr)
{
#ifndef __vxworks
    IX_MBUF_POOL *poolPtr = (IX_MBUF_POOL *) IX_MBUF_NET_POOL(bufPtr);
    UINT8 *poolDataPtr = poolPtr->dataMemPtr;
#else
#endif

    /* check parameters */
    if (bufPtr == NULL)
    {
        ixOsServLog (LOG_ERROR, "ixOsBuffPoolBufDataPtrReset"
		     ": ERROR - Invalid Parameter\n", 0, 0, 0, 0, 0, 0);
        return IX_FAIL;
    }

#ifndef __vxworks
 
    if (poolPtr->poolAllocType == IX_MBUF_POOL_TYPE_SYS_ALLOC)
    {
	IxMbufWrapper *mbufWrapperPtr = (IxMbufWrapper *) bufPtr;
	if (mbufWrapperPtr->hidden.signature != IX_MBUF_WRAPPER_SIGNATURE)
	{
	    ixOsServLog (LOG_ERROR, "ixOsBuffPoolBufDataPtrReset"
			 ": invalid mbuf, cannot reset mData pointer\n",
			 0, 0, 0, 0, 0, 0);
	    return IX_FAIL;
	}
	IX_MBUF_MDATA(bufPtr) = mbufWrapperPtr->hidden.mdataOrig;
    }
    else
    {
	if (poolDataPtr)
	{
	    unsigned int bufSize = poolPtr->bufDataSize;
	    unsigned int bufDataAddr = (unsigned int)IX_MBUF_MDATA(bufPtr);
	    unsigned int poolDataAddr = (unsigned int)poolDataPtr;
	
	    /*
	     * the pointer is still pointing somewhere in the mbuf payload.
	     * This operation moves the pointer to the beginning of the 
	     * mbuf payload
	     */
	    bufDataAddr = ((bufDataAddr - poolDataAddr) / bufSize) * bufSize; 
	    IX_MBUF_MDATA(bufPtr) = &poolDataPtr[bufDataAddr];
	}
	else
	{
	    ixOsServLog(LOG_WARNING, "ixOsBuffPoolBufDataPtrReset"
			": cannot be used if user supplied NULL pointer for pool data area "
			"when pool was created\n", 0, 0, 0, 0, 0, 0);
	    return IX_FAIL;
	}
    }

#else  /* ndef __vxworks */

    IX_MBUF_MDATA(bufPtr) = (caddr_t) (bufPtr->pClBlk->clNode.pClBuf);
#endif /* ndef __vxworks */

    return IX_SUCCESS;
}


/*
 * Function definition: ixOsBuffPoolUninit
 */

IX_STATUS 
ixOsBuffPoolUninit(IX_MBUF_POOL *pool)
{
#ifndef __vxworks
  /*
   * TODO - implement this for non-VxWorks OS's (using IX_ACC_DRV_DMA_FREE ? )
   * Then add the function to the API
   */
    ixOsServLog(LOG_WARNING, "ixOsBuffPoolUninit" 
                ": is currently only implemented for the vxWorks platform\n",
                0, 0, 0, 0, 0, 0);
    return IX_FAIL;
#else
    return ((netPoolDelete(pool) == OK) ? IX_SUCCESS : IX_FAIL);
#endif
}

