 /**
 * @file IxEthAccCodelet_p.h
 *
 * @date 22 April 2002
 *
 * @brief This file contains some private data structures and
 * defines for the Ethernet Access Codelet.
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*
 */

#ifndef IXETHACCCODELET_P_H
#define IXETHACCCODELET_P_H

#include "IxTypes.h"
#include "IxEthAcc.h"

/**< Recommended priority of queue manager dispatch loop */
#define IX_ETHACC_CODELET_QMR_PRIORITY 150 

/**< Recommended priority of Ethernet DB Maintenance task */
#define IX_ETHACC_CODELET_DB_PRIORITY 91 


#ifdef __linux
#define IX_ETHACC_CODELET_USER_INSTRUCTION \
    "Unload the codelet module to stop the operation ... \n"
#else
#define IX_ETHACC_CODELET_USER_INSTRUCTION \
    "Press ESC at any time to stop the operation ... \n"
#endif


/* Buffer management utilities */


/**< Macro to check if the free buffer queue is empty */
#define IX_ETHACC_CODELET_IS_Q_EMPTY(mbuf_list) (mbuf_list == NULL)


/**< Macro to add an MBUF to the queue. The queue uses the m_nextpkt pointer 
 * which is not used by the Ethernet Access component */ 
#define IX_ETHACC_CODELET_ADD_MBUF_TO_Q_TAIL(mbuf_list,mbuf_to_add)   	\
  {									\
    int lockVal = ixOsServIntLock();					\
    IX_MBUF* mBufptr;							\
    mBufptr = mbuf_list;						\
    if ((mbuf_list) != NULL) 						\
    {									\
        while(IX_MBUF_NEXT_PKT_IN_CHAIN_PTR((mBufptr)) != NULL)		\
        {                                                               \
            mBufptr = (IX_MBUF_NEXT_PKT_IN_CHAIN_PTR((mBufptr)));	\
        }                                                               \
        IX_MBUF_NEXT_PKT_IN_CHAIN_PTR((mBufptr)) = (mbuf_to_add);	\
        IX_MBUF_NEXT_PKT_IN_CHAIN_PTR((mbuf_to_add)) = NULL;		\
    } 									\
    else								\
    {									\
        (mbuf_list) = (mbuf_to_add);					\
        IX_MBUF_NEXT_PKT_IN_CHAIN_PTR((mbuf_to_add)) = NULL;		\
    } 									\
    ixOsServIntUnlock(lockVal);						\
  }


/**< Macro to remove and MBUF from the queue */
#define IX_ETHACC_CODELET_REMOVE_MBUF_FROM_Q_HEAD(mbuf_list,mbuf_to_rem)\
  {									\
    int lockVal = ixOsServIntLock();					\
    if ((mbuf_list) != NULL ) 						\
    {									\
        (mbuf_to_rem) = (mbuf_list);					\
        (mbuf_list) = (IX_MBUF_NEXT_PKT_IN_CHAIN_PTR((mbuf_to_rem)));	\
    } 									\
    else								\
    {									\
        (mbuf_to_rem) = NULL;						\
    } 									\
    ixOsServIntUnlock(lockVal);						\
  }


/**< Macro to check if the codelet has been initialised */
#define IX_ETHACC_IS_CODELET_INITIALISED()				\
  {									\
    if(!ixEthAccCodeletInitialised)					\
    {									\
	printf("Ethernet Access Codelet not yet initialised!\n");	\
	return (IX_FAIL);						\
    }									\
  }


/**< Macro to generate random payload data */
#define IX_ETHACC_CODELET_DATAGEN(compPtr)				\
  {									\
	int i = 0;							\
	srand(ixOsServTimestampGet());					\
	for(i=0; i<IX_ETHACC_CODELET_TXGEN_PCK_LEN; i++)		\
	 {								\
	     compPtr[i] = rand();					\
	 }								\
  }


/**< Macro to verify payload */
#define IX_ETHACC_CODELET_DATA_VERIFY(m_data, compPtr)			\
  {									\
      if(memcmp(&compPtr[0], m_data, IX_ETHACC_CODELET_TXGEN_PCK_LEN) != 0)  \
	{								\
	    ixOsServLog(LOG_ERROR, "Mbuf data verification failed\n",   \
                        0, 0, 0, 0, 0, 0);                              \
	}								\
  }


/**< Type of operations of the Ethernet Access Codelet. */
typedef enum
{
    IX_ETHACC_CODELET_RX_SINK = 1,    /**< All frames received (from external device)
                                        *  will be sinked for available ports. */
    IX_ETHACC_CODELET_SW_LOOPBACK,    /**< All frames received are software loopbacked 
                                        *  to the same port for available ports */
    IX_ETHACC_CODELET_TXGEN_RXSINK,   /**< Frames generated and transmitted from port 1, 
                                        *  remote loopbacked to port 2 by using cross cable
					*  and received on port 2. */
    IX_ETHACC_CODELET_PHY_LOOPBACK,   /**< Frames generated and PHY loopbacked on the 
                                        *  same port for available ports. */
    IX_ETHACC_CODELET_BRIDGE,         /**< Frames received on one port will be transmitted
                                        *  through the other port. */
    IX_ETHACC_CODELET_ETH_LEARNING    /**< Ethernet Learning Facility where it adds some static
                                        *  and dynamic entries. Dynamic entries are then aged and 
					*  verified that they no longer appear in the database. */
} IxEthAccCodeletOperation;  


/**< The Eth PHY Configuration (100/10 Mbps , Full/Half Duplex, with/without autonegotiation).*/
typedef struct
{
    BOOL speed100;	/**< 100 Mbits */
    BOOL fullDuplex;	/**< Full Duplex */
    BOOL autonegotiate;	/**< Autonegotiation */
} IxEthAccCodeletPhyConf;

/**
 *
 * This function is the entry point to the Ethernet Access codelet.
 * It must be called before any of the operations as it initialises the neccessary 
 * components.
 * 
 * The initialisation sequence is as follows:
 * <UL>
 *	<LI> Initialise Queue manager
 *	<LI> Start Queue manager dispatcher loop
 *	<LI> Download NPE microcode
 *	<LI> Start NPE message handler
 *	<LI> Start NPEs
 *	<LI> Initialise Ethernet Access component
 *	<LI> Initialise Ethernet ports
 *	<LI> Initialise Ethernet PHYs
 *	<LI> Program MAC addresses
 *	<LI> Set ports to promiscuous mode
 *	<LI> Initialise MBUF pool
 *	<LI> Set the Tx scheduling priority to FIFO_NO_PRIORITY 
 * </UL>  
 *
 * At the end of the initialisation sequence ports are not enabled.
 * They are enabled as appropriate for each operation that's run.
 *
 * return IX_SUCCESS - Codelet successfully initialised 
 * return IX_FAIL - Error initialising codelet
 */
IX_STATUS
ixEthAccCodeletInit(BOOL phyLoopback);

/**
 *
 * This function unintialize the Ethernet Access codelet.
 * It must be called after any Operation is complete to uninitialise the codelet.
 * 
 * return IX_SUCCESS - Codelet successfully uninitialised 
 * return IX_FAIL - Error uninitialising codelet
 */
IX_STATUS
ixEthAccCodeletUninit(void);

/**
 *
 * Receive Sink operation.
 *
 * This function sinks received packets as fast as possible on both ports.
 * An external source must be connected to both ports transmitting
 * Ethernet packets.  
 *
 * return IX_SUCCESS - Rx Sink Operation successfully started
 * return IX_FAIL - Error starting Rx Sink Operation
 */ 
IX_STATUS
ixEthAccCodeletRxSink(void);

/**
 * Software loopback operation.
 *
 * This function sets up some callbacks so that any frames received are
 * looped back and retransmitted unmodified on the same port.
 * An external Ethernet source must be connected to both ports transmitting
 * packets.
 * 
 * return IX_SUCCESS - Software Loopback Operation successfully started
 * return IX_FAIL - Software started external Loopback Operation
 */ 
IX_STATUS
ixEthAccCodeletSwLoopback(void);


/**
 *
 * Port-to-Port (Tx to Rx Sink) loopback operation.
 *
 * This function performs a software loopback on the two ethernet ports.
 * Frames are generated on the XScale and sent from port 1 and received on port 
 * 2 via a cross-over cable only. Received frames on port 2 are put back on the free 
 * queue which is shared between the two ports. A verify function ensures that the
 * frames transmitted are the same as those received. A cross-over cable connecting
 * port 1 and port 2 is needed to perform this operation.
 *
 * return IX_SUCCESS - SW Loopback successfully started
 * return IX_FAIL - Error starting SW loopback Operation
 */
IX_STATUS
ixEthAccCodeletTxGenRxSinkLoopback(void);

/**
 *
 * @brief PHY loopback operation.
 *
 * This function performs a MII-PHY level loopback on each ethernet port.
 * Frames are generated on the XScale and sent from each port.At PHY level,
 * frames will be loopbacked to the same port and then received on port.  
 *
 * @return IX_SUCCESS - PHY Loopback successfully started
 * @return IX_FAIL - Error starting PHY loopback Operation
 */
IX_STATUS
ixEthAccCodeletPhyLoopback(void);

/**
 *
 * Bridge Operation.
 *
 * This Operation transmits any frames received on one port through the other one,
 * as a bridge would do. An external Ethernet source must be connected to
 * both ports transmitting frames.
 * 
 * return IX_SUCCESS - Bridge Operation successfully started.
 * return IX_FAIL - Error starting bridge operation.
 */
IX_STATUS
ixEthAccCodeletBridge(void);


/**
 *
 * Ethernet MAC address learning Operation.
 *
 * This function demonstrates the use of the Ethernet MAC learning facility.
 * It adds some static and dynamic entries. Dynamic entries are then aged and
 * verified that they no longer appear in the database.
 *
 * return IX_SUCCESS - DB Learning operation successfully started
 * return IX_FAIL - Error starting DB Learning operation.
 */ 
IX_STATUS
ixEthAccCodeletDBLearning(void);


/**
 *
 * This function is called at the end of each ethernet operation.
 * It displays the MIB statistics for each port.
 *
 * return void
 */
void
ixEthAccCodeletShow(void);

#endif /* IXETHACCCODELET_P_H */
