/*
 * @file IxAdsl.c
 *
 * @date 2-Jun-2003
 *
 * @brief  ADSL Driver source file for Public API (Component Interface module)
 *
 *
 * Design Notes:  this Adsl Driver only Supports Utopia Level 1
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2000-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#include <ix_ossl.h>
#include <IxOsServices.h>

#include <string.h>

#include "IxAdsl.h"
#include "IxAdslCtrleLib.h"
#include "IxAdslCtrleConstants.h"
#include "IxAdsl_p.h"
#include "IxTypes.h"
#include "IxAssert.h"
#include "IxAdslUtil.h"


/**
 * @fn ixAdslLineOpenInternal
 *
 * @brief Open Adsl Line
 *
 * @param UINT32 lineNum
 *
 * @return IxAdslStatus
 *             IX_ADSL_STATUS_SUCCESS
 *             IX_ADSL_STATUS_FAIL
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE IxAdslStatus ixAdslLineOpenInternal( UINT32 lineNum );
/**
 * @fn ixTrainingStatusShow
 *
 * @brief   Display the number of times the Adsl line is retrained.
 *          This function is called by ixAdslShow.
 *
 * @param UINT32 lineNum
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixTrainingStatusShow(UINT32 lineNum);
/**
 * @fn ixAdslLineStateMachine
 *
 * @brief   Tracks and Triggers changes in Operational State
 *          based on lineEnable State  and Adsl Phy state.
 *          This function is called by ixAdslLineSupervisoryTask.
 *
 * @param UINT32 lineNum
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixAdslLineStateMachine(UINT32 lineNum);
/**
 * @fn ixAdslLineSupervisoryTask
 *
 * @brief  This Task runs in the background to periodically activate ixAdslLineStateMachine
 *
 * @param void *p
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixAdslLineSupervisoryTask(void *p);
/**
 * @fn ixAdslStateChangeCallback
 *
 * @brief  Callback function that is invoked when there is a state change
 *         Display message to announce a state change
 *
 * @param UINT32          lineNum  : Adsl line number
 * @param IxAdslLineState lineState : New state of Adsl line
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixAdslStateChangeCallback(UINT32 lineNum, IxAdslLineState lineNewState);
/**
 * @fn ixAdslDriverInit
 *
 * @brief  Initialize Adsl Driver
 *
 * @param none
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixAdslDriverInit(void);
/**
 * @fn ixAdslLineCloseInternal
 *
 * @brief  Shutdown adsl line
 *
 * @param UINT32 lineNum
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixAdslLineCloseInternal( UINT32 lineNum );
/**
 * @fn ixAdslLineStateGetInternal
 *
 * @brief Get the state of the Adsl Line; the state machine periodically
 *        queries the adsl phy for the line state; this information
 *        is updated to adsl line data structure by state machine
 *
 * @param UINT32 lineNum
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE IxAdslLineState ixAdslLineStateGetInternal(UINT32 lineNum);
/**
 * @fn ixAdslStateChanged
 *
 * @brief  Calls multiple callbacks registered to an adsl line
 *         during a state change and updates the change to adsl line state
 *
 * @param UINT32 lineNum
 * @param IxAdslLineState newState
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixAdslStateChanged(UINT32 lineNum, IxAdslLineState newState  );
/**
 * @fn ixAdslAtmCellCountersShow
 *
 * @brief  Show Atm counter values
 *
 * @param UINT32 lineNum
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE IxAdslStatus ixAdslAtmCellCountersShow(UINT32 lineNum);
/**
 * @fn ixAdslCpeTrainingStatusUpdate
 *
 * @brief  Update number of times the Adsl line is retrained.
 *          This function is triggered by state change callback.
 *
 * @param UINT32 lineNum
 * @param UINT32 upstream  : upstream line rate
 * @param UINT32 downstream : downstream line rate
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
PRIVATE void ixAdslCpeTrainingStatusUpdate ( UINT32 lineNum,
                                      UINT32 upstream,
                                      UINT32 downstream );
/**
 * @fn ixAdslTrainingLogShow
 *
 * @brief  Show Training Log
 *
 * @param UINT32 lineNum
 *
 * @return none
 *
 * Reentrant     : yes
 * ISR Callable  : no
 */
#ifndef NDEBUG
PRIVATE IxAdslStatus ixAdslTrainingLogShow(UINT32 lineNum);
#endif
/**
 *
 * Adsl taskId
 *
 */

static ix_ossl_thread_t ixADSL_taskId;
/**
 *
 * osError
 *
 */
static ix_error osError;
/**
 *
 * Number of lines initialised
 *
 */
static UINT32 linesInitialized = 0;

/**
 *
 * Structure to hold adsl line control parameters
 *
 */
static struct
{
    struct
    {
        UINT32 total;
        UINT32 GE8000;     /* Number of retrains for > 8Mbps Downstream*/
        UINT32 B7000_7999; /* Number of retrains for 7Mbps to 7.999Mbps Downstream*/
        UINT32 L7000;      /* Number of retrains for < 7Mbps Downstream*/
        UINT32 GE768;      /* Number of retrains for > 768kbps Upstream*/
        UINT32 L768;       /* Number of retrains for < 768kbps Upstream*/
    } training_counters;

    BOOL ixAdslCOInterleavedMode;              /* Flag to indicate COInterleaved mode */
    BOOL ixAdslEnableGasp;                     /* Flag to indicate Dying Gasp Enabled/Disabled */
    IxAdslPhyType phyType;                     /* Phy Type CPE, CO */

    IxAdsllineEnableState lineEnableState;     /* lineEnable State  */
    IxAdslLineType lineType;                   /* Autoselect, gdmt, ansi, glite */
    IxAdslLineState lineState;                 /* State of the Adsl Phy */

    BOOL fwInitComplete;                       /* Flag to indicate firmware download is completed */
    volatile IxAdslSMState state;              /* Current Operational State of State Machine */
    UINT32 downPollCount;                      /* Counter for line down timeout in POLLING state*/
    UINT32 openPollCount;                      /* Counter for line up timeout in LINE_OPEN state*/
    UINT32 rc;                                 /* Return code for status of ixAdslLineCloseInternal */
    UINT32 trainingLogBufferIndex;                     /* Training log buffer counter */
    UINT8  trainingLogBuffer[IX_ADSL_TRAININGLOGSIZE]; /* Training log buffer */
} ixAdslLineData[IX_ADSL_NUM_LINES];
/**
 *
 * Structure to hold adsl line control parameters
 *
 */
static struct
{
    IxAdslStateChangeCallback lineStateCallbacks[IX_ADSL_SIZEOF_CALLBACK_LIST]; /* Callback pointer List */
    UINT32 ixAdslLineStateCallbackIndex;         /* Maximum index for the above Array */
} ixAdslLineCallbackData[IX_ADSL_NUM_LINES];

/**
 *
 * Vendor code data structure
 *
 */
IxAdslVendorInfo ixAdslVendorInfo[IX_ADSL_NUM_LINES];       /* Vendor Id Info */
/**
 *
 * Debug strings for dump line state.
 *
 */
static const INT8 *listStateStrs[]=
{
    "Adsl Line UP Fast and Interleave Channels",
    "Adsl Line Waiting for Activating",
    "Adsl Line Activating",
    "Adsl Line DOWN",
    "Adsl Line UP Fast Channel",
    "Adsl Line UP Interleaved Channel",
    "Unknown Line State"
};
#define IX_ADSL_STATE_DEBUG_STR_COUNT 5
/**
 *
 * Flag to indicate if driver is initialized
 *
 */
static BOOL isInitialised = FALSE;

/**
 *
 * Flag to indicate if callback list is initialized
 *
 */
static BOOL isCallbackListInitialized = FALSE;

/**
 * @fn ixAdslLineOpen
 * @brief Open the Adsl Line
 *
 */
PUBLIC IxAdslStatus
ixAdslLineOpen( UINT32 lineNum,
                IxAdslLineType lineType,
                IxAdslPhyType phyType
            )
{
    UINT8 *initFirmware;
    UINT32 initFwLen;
    UINT8 *phyFirmware;
    UINT32 phyFwLen;
    /* ----------------------------------------------------------------------------
    * Check Argument : phyType
    */
    if (IX_ADSL_PHY_INVALID <= phyType)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_OPEN : Invalid Phy Type Specified\n"));
        return(IX_ADSL_STATUS_UNSUPPORTED_MODE);
    }
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_OPEN Failed: Invalid Adsl line number specified\n"));
        return IX_ADSL_STATUS_FAIL;
    }
    /* ----------------------------------------------------------------------------
    * Check Argument : lineType
    */
    if ( lineType >= IX_ADSL_INVALID_MODE )
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_OPEN Failed: Invalid Adsl line mode specified\n"));
        return(IX_ADSL_STATUS_UNSUPPORTED_MODE);
    }
    /* ----------------------------------------------------------------------------
    * Check lineEnable State  : line cannot be opened if already up.
    */
    if(IX_ADSL_LINE_ENABLE==ixAdslLineData[lineNum].lineEnableState)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_OPEN Failed: Adsl line already up\n"));
        return(IX_ADSL_STATUS_FAIL);
    }
    /* ----------------------------------------------------------------------------
    * Initialize firmware parameters for selected phyType
    */
    if (IX_ADSL_PHY_CPE==phyType)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_OPEN : Initializing CPE Firmware download\n"));
        initFirmware=IX_ASW_CPE_INIT;
        initFwLen=IX_ASW_CPE_INIT_LEN;
        phyFirmware=IX_ASW_CPE_LOAD;
        phyFwLen=IX_ASW_CPE_LOAD_LEN;
    }
#ifdef IX_USE_ADSL_CO
    else if ((IX_ADSL_PHY_CO_FAST==ixAdslLineData[lineNum].phyType)||
             (IX_ADSL_PHY_CO_INTERLEAVED==ixAdslLineData[lineNum].phyType))
    {
#error  /* CO is not an official feature in this software release */
        /* CO is not supported in by 20170 chipset */
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_OPEN : Initializing CO Firmware download\n"));
        initFirmware=IX_ASW_CO_INIT;
        initFwLen=IX_ASW_CO_INIT_LEN;
        phyFirmware=IX_ASW_CO_LOAD;
        phyFwLen=IX_ASW_CO_LOAD_LEN;
    }
#endif

    /* ----------------------------------------------------------------------------
    * Initialize adsl line parameters and start ixAdslLineSupervisoryTask
    * this function only executes one time and does nothing in subsequent calls
    */
    ixAdslDriverInit();
    /* ----------------------------------------------------------------------------
    * Download Adsl modem firmware only if firmware not yet downloaded
    */
    if ( ! ixAdslLineData[lineNum].fwInitComplete )
    {
        if (  IX_ADSL_STATUS_FAIL == ixAdslUtilDeviceDownload( lineNum,
                                                               initFirmware,
                                                               initFwLen,
                                                               phyFirmware,
                                                               phyFwLen) )
        {
            /* Download failed */
            DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_OPEN: Device Download failed\n"));
            return(IX_ADSL_STATUS_FAIL);
        }
    }
    /* Update adsl line parameters to reflect the modem parameters that was
    succesfully downloaded and set State Machine to activate modem */
    ixAdslLineData[lineNum].phyType = phyType;
    ixAdslLineData[lineNum].lineType = lineType;
    ixAdslLineData[lineNum].state = IX_ADSL_LINE_INIT;
    ixAdslLineData[lineNum].lineEnableState = IX_ADSL_LINE_ENABLE;
    ixAdslLineData[lineNum].fwInitComplete = TRUE;
    linesInitialized = IX_ADSL_NUM_LINES;

    if (isCallbackListInitialized == FALSE)
    {
    	/* register the line state display callback as the first Callback case when no
	   callbacks are registered by an external application before line open */
	ixAdslLineCallbackData[lineNum].lineStateCallbacks[0] = ixAdslStateChangeCallback;
	ixAdslLineCallbackData[lineNum].ixAdslLineStateCallbackIndex = 1;
	isCallbackListInitialized = TRUE;
    }
    return(IX_ADSL_STATUS_SUCCESS);
}

/**
 * @fn ixAdslLineClose
 * @brief Close the Adsl Line
 *
 */
PUBLIC IxAdslStatus
ixAdslLineClose(UINT32 lineNum)
{
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_CLOSE Failed: Invalid Adsl line number specified\n"));
        return IX_ADSL_STATUS_FAIL;
    }
    /* ----------------------------------------------------------------------------
    * Check if line already initialised
    */
    if ( FALSE==isInitialised )
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_CLOSE Failed: Adsl driver not initialised\n"));
        return IX_ADSL_STATUS_FAIL;
    };
    /* ----------------------------------------------------------------------------
    * Check if line already down
    */
    if (IX_ADSL_LINE_DISABLE==ixAdslLineData[lineNum].lineEnableState)
    {
    	DEBUG_CMD(DEBUG_LOG("Line %d is already down.\n",lineNum));
        return IX_ADSL_STATUS_ALREADY_DOWN;
    };

    /* Set the lineEnable State  to DOWN */
    ixAdslLineData[lineNum].lineEnableState = IX_ADSL_LINE_DISABLE;
    /* Set the state machine Operational State to CLOSE
    to force modem shutdown */
    ixAdslLineData[lineNum].state = IX_ADSL_LINE_CLOSE;
    if (IX_ADSL_LOOPBACK==ixAdslLineData[lineNum].lineType)
    {
    	DEBUG_CMD(DEBUG_LOG("Loopback Mode disabled in line %d.\n",lineNum));
        isInitialised = FALSE; /* Force driver to reinitialize */
        ix_ossl_thread_kill(ixADSL_taskId);
    	return IX_ADSL_STATUS_ALREADY_DOWN;
    }

    while (ixAdslLineData[lineNum].state != IX_ADSL_LINE_WAIT)
    {
        ixOsServTaskSleep(IX_ADSL_POLL_DELAY);
    }
    /* the return code of the close command is returned */
    return(ixAdslLineData[lineNum].rc);
}
/*
 * @fn ixAdslLineStateGet
 * @brief Get the state of the Adsl Line
 *
 */
PUBLIC IxAdslLineState
ixAdslLineStateGet(UINT32 lineNum)
{
    IxAdslLineState returnLineState = IX_ADSL_LINE_STATE_INVALID;
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("ixAdslLineStateGet Failed: Invalid Adsl line number specified\n"));
        return IX_ADSL_LINE_STATE_INVALID;
    }

    /* ----------------------------------------------------------------------------
    * Check if driver is initialised
    */
    if (FALSE==isInitialised)
    {
        DEBUG_CMD(DEBUG_LOG("ixAdslLineStateGet Failed: Driver not initialised\n"));
        return IX_ADSL_LINE_STATE_INVALID;
    }
    returnLineState = ixAdslLineData[lineNum].lineState;
    return returnLineState;
}
/*
 * @fn ixAdslLineStateGetInternal
 * @brief Get the state of the Adsl Line; the state machine periodically
 *        queries the adsl phy for the line state; this information
 *        is updated to adsl line data structure by state machine
 *
 */
PRIVATE IxAdslLineState
ixAdslLineStateGetInternal(UINT32 lineNum)
{
    UINT8 ctrleGetBuffer[1];
    UINT8 ixStatus;
    IxAdslLineState returnLineState = IX_ADSL_LINE_STATE_INVALID;
    UINT32 rc;
    BOOL  asyncResponse;
    T_LTadslNearEndLineOperData nearOperData;
    T_LTadslFarEndLineOperData farOperData;
    T_adsl_def_bitmap_set ctrleDefectBitmap;

    /* -----------------------------------------------------------------------
    * First read command reg, to see if Loss of Signal
    */
    asyncResponse =  ixAdslUtilCtrleRead(lineNum, &ctrleGetBuffer[0],RXCOMMANDREG,1);
    if (asyncResponse)
    {
        /* There was a command available */
        if (ADSL_SIGNAL_LOST==ctrleGetBuffer[0])
        {
            DEBUG_CMD(DEBUG_LOG("!! ADSL SIGNAL LOST !! Initiating modem shutdown \n"));
            /* Need to close the line from this side ,
            * so it can be reopened later */
	        ADSL_disableModemLine(lineNum);
            return IX_ADSL_LINE_STATE_DOWN;
        }
        else if (ADSL_SUICIDE_REQUEST== ctrleGetBuffer[0])
        {
            DEBUG_CMD(DEBUG_LOG("!! ADSL SUICIDE REQUEST !! Initiating modem shutdown \n"));
            /* Need to close the line from this side ,
            * so it can be reopened later */
	        ADSL_disableModemLine(lineNum);
            return IX_ADSL_LINE_STATE_DOWN;
        }
    }

    /* -----------------------------------------------------------------------
    * Get modem state if no loss of signal
    */
    asyncResponse = ixAdslUtilCtrleRead(lineNum,&ctrleGetBuffer[0],CTEAD_MODEM_STATUS,1);
    if ( !asyncResponse )
    {
        DEBUG_CMD(DEBUG_LOG("!! FAILED TO READ MODEM STATUS \n"));
        returnLineState = IX_ADSL_LINE_STATE_INVALID;
    } else
    {
        /* -----------------------------------------------------------------------
        * Return modem state that is retrieved from Phy
        */
        switch (ctrleGetBuffer[0])
        {
            case ( MODEM_STATUS_DOWN ):
                returnLineState = IX_ADSL_LINE_STATE_DOWN;
                break;
            case ( MODEM_STATUS_WAIT_FOR_INIT ):
                returnLineState = IX_ADSL_LINE_STATE_WAIT_FOR_ACTIVATING;
                break;
            case ( MODEM_STATUS_INITIALIZING ):
                returnLineState = IX_ADSL_LINE_STATE_ACTIVATING;
                break;
            case ( MODEM_STATUS_UP ):
            {
                rc = ADSL_def_bitmap_poll(lineNum, &ctrleDefectBitmap);
                /**
                 *   Although the Modem returns line up; it could possibly be down in cases below.
                 *   There is a subtle difference between 'Loss of' seconds and failures.
                 *   A 'Loss of' second is a defect which has occured in the last second,
                 *   whereas a failure is declared when a defect has been active for the
                 *   last 2.5 seconds +/- 0.5 seconds. A failure is a more terminal event
                 *   and would normally cause a modem resync event. - DH 04/04/01
                 */
    	       if ( C_CONF_DATA_ACK != rc )
    		   {
    		       returnLineState =  ( IX_ADSL_LINE_STATE_INVALID) ;
                   break;
               }
    	       ixStatus = ctrleDefectBitmap.near_end.status;
    	       if ( (ixStatus & CURR_STATUS_LOSS_FRAME) ||
    	            (ixStatus & CURR_STATUS_LOSS_CD_INTERLEAVED) ||
    	            (ixStatus & CURR_STATUS_LOSS_CD_NONINTERLEAVED) ||
    	            (ixStatus & CURR_STATUS_LOSS_SIGNAL) ||
    	            (ixStatus & CURR_STATUS_LOSS_POWER) )
               {
                    returnLineState = IX_ADSL_LINE_STATE_DOWN;
                    break;
    	       }
    	       else
    	       {
                    rc = LT_ADSL_get_FarEndLineOperData(lineNum, & farOperData);
                    if ( C_CONF_DATA_ACK != rc )
                    {
                        returnLineState = IX_ADSL_LINE_STATE_INVALID;
                        break;
                    }
                    rc = LT_ADSL_get_NearEndLineOperData(lineNum, & nearOperData);
                    if ( C_CONF_DATA_ACK != rc )
                    {
                        returnLineState =  IX_ADSL_LINE_STATE_INVALID;
                        break;
                    }
                    if ( ((0!=nearOperData.fastBitRate) || (0!=farOperData.fastBitRate)) &&
                         ((0==nearOperData.interleavedBitRate) && (0==farOperData.interleavedBitRate)) )
                    {
                        returnLineState = IX_ADSL_LINE_STATE_UP_FASTCHANNEL;
                        break;
                    }
                    else if ( ((0==nearOperData.fastBitRate) && (0==farOperData.fastBitRate)) &&
                              ((0!=nearOperData.interleavedBitRate) || (0!=farOperData.interleavedBitRate)) )
                    {
                    	returnLineState = IX_ADSL_LINE_STATE_UP_INTERLEAVECHANNEL;
                    	break;
                    }
                    else
                    {
                        returnLineState = IX_ADSL_LINE_STATE_UP_DUAL_LATENCY;
                        break;
                    }
    	        }
    	    }
        }
    }
    return returnLineState;
}

/**
 * @fn ixAdslVendorCodeSet
 * @brief Set vendor code
 *
 */
PUBLIC IxAdslStatus ixAdslVendorCodeSet (
    UINT32 lineNum,
    UINT8 ixAdslItuVendoridCountrycode,
    UINT8 ixAdslItuVendoridVendorcode1,
    UINT8 ixAdslItuVendoridVendorcode2,
    UINT8 ixAdslItuVendoridVendorcode3,
    UINT8 ixAdslItuVendoridVendorcode4,
    UINT8 ixAdslItuVendoridVendorspecific1,
    UINT8 ixAdslItuVendoridVendorspecific2)
{
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_VENDOR_CODE_SET Failed: Invalid Adsl line number specified\n"));
        return IX_ADSL_STATUS_FAIL;
    }
    /* ----------------------------------------------------------------------------
    * Check lineEnable State  : These values can only be updated when Adsl line is down.
    */
    if (IX_ADSL_LINE_DISABLE == ixAdslLineData[lineNum].lineEnableState)
    {
        /* ----------------------------------------------------------------------------
        * Update VendorId information since Adsl line is down.
        */
        ixAdslVendorInfo[lineNum].itu_vendorid_countrycode = ixAdslItuVendoridCountrycode;
        ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_1 = ixAdslItuVendoridVendorcode1;
        ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_2 = ixAdslItuVendoridVendorcode2;
        ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_3 = ixAdslItuVendoridVendorcode3;
        ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_4 = ixAdslItuVendoridVendorcode4;
        ixAdslVendorInfo[lineNum].itu_vendorid_vendorspecific_1 = ixAdslItuVendoridVendorspecific1;
        ixAdslVendorInfo[lineNum].itu_vendorid_vendorspecific_2 = ixAdslItuVendoridVendorspecific2;
        return IX_ADSL_STATUS_SUCCESS;
    }
    else
    {
        DEBUG_CMD(DEBUG_LOG("\nVendorCodeSet Operation Failed because ADSL Line is up\n"));
        return IX_ADSL_STATUS_FAIL;
    }
}

/**
 * @fn ixAdslLineRateUpstreamGet
 * @brief Get upstream line rate
 *
 */
PUBLIC UINT32
ixAdslLineRateUpstreamGet (UINT32 lineNum)
{
    UINT32 rc;
    T_LTadslFarEndLineOperData farOperData;
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_RATE_UPSTREAM_GET Failed: Invalid Adsl line number specified\n"));
        return(0);
    }

    if (IX_ADSL_LOOPBACK==ixAdslLineData[lineNum].lineType)
    {
        return(0);
    }
    rc = LT_ADSL_get_FarEndLineOperData(lineNum, & farOperData);
    /* Check for positive acknowledge from CtrlE function */
    if ( C_CONF_DATA_ACK==rc )
    {
        /* Return upstream linerate  */
        return(farOperData.interleavedBitRate + farOperData.fastBitRate);
    }
    else
    {
        return(0);
    }
}

/**
 * @fn ixAdslLineRateDownstreamGet
 * @brief Get downstream line rate
 *
 */
PUBLIC UINT32
ixAdslLineRateDownstreamGet (UINT32 lineNum)
{
    UINT32 rc;
    T_LTadslNearEndLineOperData nearOperData;
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("IX_ADSL_LINE_RATE_DOWNSTREAM Failed: Invalid Adsl line number specified\n"));
        return(0);
    }

    if (IX_ADSL_LOOPBACK==ixAdslLineData[lineNum].lineType)
    {
        return(0);
    }
    rc = LT_ADSL_get_NearEndLineOperData(lineNum, & nearOperData);
    /* Check for positive acknowledge from CtrlE function */
    if ( C_CONF_DATA_ACK == rc )
    {
        /* Return downstream linerate  */
        return(nearOperData.interleavedBitRate + nearOperData.fastBitRate);
    }
    else
    {
        return(0);
    }
}

/**
 * @fn ixAdslShow
 * @brief Show Adsl Phy statistics
 *
 */
PUBLIC void
ixAdslShow (UINT32 lineNum)
{
    struct T_Controller_SW_version controllerSwVersion;
    UINT8 lineState;
    UINT32 rc;
    T_LTadslNearEndLineOperData nearOperData;
    T_LTadslFarEndLineOperData farOperData;
    T_adsl_def_bitmap_set ixDefectBitmap;
    /* ----------------------------------------------------------------------------
    * Range check for input parameter : lineNum
    */
    if(lineNum > (IX_ADSL_NUM_LINES-1))
    {
        printf("Invalid line number specified\n\n");
        return;
    }
    if (isInitialised != TRUE)
    {
        /* Cannot run if driver not initialized */
        printf("Please initialize driver before calling this function\n\n");
        return;
    }

    printf("Intel Communications - ADSL Driver\n\n");
    /* Get Controller Software version */
    if ( ADSL_get_controller_SWVersion(lineNum, &controllerSwVersion) != C_CONF_DATA_ACK )
    {
        printf("Get Controller SW Version Failed");
        return;
    }
    else
    {
        /* Display Controller Software version */
        printf("Controller SW Version: %d.%d.%d\n",
                controllerSwVersion.number1,
                controllerSwVersion.number2,
                controllerSwVersion.number3);
    }
    /* Get line state from Phy  */
    lineState = ixAdslLineStateGet(lineNum);
    /* Display line state from Phy  */
    printf("ADSL Line State : %-32.32s\n",listStateStrs[lineState]);
    printf("ADSL line type  : %d\n",ixAdslLineData[lineNum].lineType);
    /* Show vendor info  */
    printf("\nCountry code: ");
    printf("0x%02x ",ixAdslVendorInfo[lineNum].itu_vendorid_countrycode);
    printf("\nVendor code : ");
    printf("0x%02x ",ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_1);
    printf("0x%02x ",ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_2);
    printf("0x%02x ",ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_3);
    printf("0x%02x ",ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_4);
    printf("\nVendor product code: ");
    printf("0x%02x ",ixAdslVendorInfo[lineNum].itu_vendorid_vendorspecific_1);
    printf("0x%02x ",ixAdslVendorInfo[lineNum].itu_vendorid_vendorspecific_2);
    printf("\n\n");

    /* Get & display dying gasp state */
    if( TRUE == ixAdslLineData[lineNum].ixAdslEnableGasp )
    {
        printf("ADSL Dying Gasp is Enabled.\n");
    }

    /* ----------------------------------------------------------------------------
    * GetLine Statistics.
    */
    /* Get & display dying gasp state */
    rc = LT_ADSL_get_NearEndLineOperData(lineNum, & nearOperData);
    if ( C_CONF_DATA_COMMPROBLEM == rc )
    {
        printf("ADSL Get near line stats failed");
    }
    else if ( C_CONF_DATA_REJ == rc )
    {
        printf("ADSL No Near end stats available: Modem not in showtime\n");
    }
    else if ( C_CONF_DATA_ACK == rc )
    {
#ifdef IX_USE_ADSL_CO
        if(ixAdslLineData[lineNum].phyType==IX_ADSL_CO_FAST ||
            ixAdslLineData[lineNum].phyType==IX_ADSL_CO_INTERLEAVED)
        {
            printf("Adsl CO side modem\n");
        }
        else
        {
#endif
            printf("Adsl CPE side modem\n");
#ifdef IX_USE_ADSL_CO
        }
#endif
        /* Show training statistics */
        ixTrainingStatusShow(lineNum);
        /* Show adsl line statistics */
        printf("\nADSL Near End Operational Data\n");
        printf("  Upstream Relative Capacity Occupancy : %d \n",
                                                nearOperData.relCapacityOccupationUpstr);
        printf("  Attainable Bitrate                   : %d kbits/sec\n",
                                                        nearOperData.attainableBitrate);
        printf("  Noise Margin Upstream                : %d dB \n",
                                                        nearOperData.noiseMarginUpstr);
        printf("  Output Pwr Downstream                : %d dBm\n",
                                                        nearOperData.outputPowerDnstr);
        printf("  Attenuation Upstream                 : %d dB\n",
                                                        nearOperData.attenuationUpstr);
        printf("  Downstream Fast Bitrate              : %d kbits/sec\n",
                                                        nearOperData.fastBitRate);
        printf("  Downstream Interleaved Bitrate       : %d kbits/sec\n",
                                                        nearOperData.interleavedBitRate);
    }
    rc = ADSL_def_bitmap_poll(lineNum, &ixDefectBitmap);
    if( C_CONF_DATA_ACK == rc )
    {
        printf("\n  Near-end defect bitmap \t\t: 0x%x\n", ixDefectBitmap.near_end.status);
    }

    rc = LT_ADSL_get_FarEndLineOperData(lineNum, & farOperData);
    if ( C_CONF_DATA_COMMPROBLEM == rc )
    {
        printf("\nADSL Get Far line stats failed\n");
    }
    else if ( C_CONF_DATA_REJ == rc )
    {
        printf("\nADSL No Far end stats available: Modem not in showtime\n");
    }
    else if ( C_CONF_DATA_ACK == rc )
    {
        /* The data is good */
        printf("\nADSL Far End Operational Data\n");
        printf("  Downstream Relative Capacity Occupancy : %d %%\n",
                                                        farOperData.relCapacityOccupationDnstr);
        printf("  Attainable Bitrate                     : %d kbits/sec\n",
                                                        farOperData.attainableBitrate);
        printf("  Noise Margin Downstream                : %d dB \n",
                                                        farOperData.noiseMarginDnstr);
        printf("  Output Pwr Upstream                    : %d dBm\n",
                                                        farOperData.outputPowerUpstr);
        printf("  Attenuation Downstream                 : %d dB\n",
                                                        farOperData.attenuationDnstr);
        printf("  Upstream Fast Bitrate                  : %d kbits/sec\n",
                                                        farOperData.fastBitRate);
        printf("  Upstream Interleaved Bitrate           : %d kbits/sec\n",
                                                        farOperData.interleavedBitRate);
    }

    rc = ADSL_def_bitmap_poll(lineNum, &ixDefectBitmap);
    if( rc == C_CONF_DATA_ACK )
    {
        printf("\n  Far-end defect bitmap  \t\t: 0x%x\n", ixDefectBitmap.far_end.status);
    }
    printf("\n\n\n");
    if (IX_ADSL_STATUS_FAIL==ixAdslAtmCellCountersShow(lineNum))
    {
        printf("\n  Atm cell counter show failed \n");
    };
    printf("\n\n\n");
#ifndef NDEBUG
 {
     IxAdslSMState holdState;
     holdState = ixAdslLineData[lineNum].state;
     ixAdslLineData[lineNum].state = IX_ADSL_LINE_WAIT;
     if (IX_ADSL_STATUS_FAIL== ixAdslTrainingLogShow(lineNum))
     {
	 printf("\n  Training Log show failed \n");
     };
     ixAdslLineData[lineNum].state = holdState;
 }
#endif
    return;
}

/**
 * @fn ixAdslLineStateChangeCallbackRegister
 * @brief Register line state change triggered callback
 *        Multiple callbacks can be registered per line
 */
IxAdslStatus
ixAdslLineStateChangeCallbackRegister(UINT32 lineNum, IxAdslStateChangeCallback callback_fn)
{
    UINT32 indexL = ixAdslLineCallbackData[lineNum].ixAdslLineStateCallbackIndex;
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("Invalid Adsl line number specified\n"));
        return IX_ADSL_STATUS_FAIL;
    }

    /* ----------------------------------------------------------------------------
    * Block callback register if this happens after line is Enabled
    */

    if(IX_ADSL_LINE_ENABLE==ixAdslLineData[lineNum].lineEnableState)
    {

        DEBUG_CMD(DEBUG_LOG("Callback registration rejected: please disable line and try again\n"));
        return IX_ADSL_STATUS_FAIL;
    }
    /* check parameters */
    if(NULL == callback_fn)
    {
        DEBUG_CMD(DEBUG_LOG("Callback registration rejected: callback function is a NULL pointer\n"));
        return IX_ADSL_STATUS_FAIL;
    }
    if(IX_ADSL_SIZEOF_CALLBACK_LIST <= indexL)
    {
        DEBUG_CMD(DEBUG_LOG("Callback registration rejected: maximum number exceeded\n"));
        return IX_ADSL_STATUS_FAIL;
    }
    /* If callback list not initialized then make sure callback index is zero
     * and register the first callback : ixAdslStateChangeCallback
     */
    if (isCallbackListInitialized == FALSE)
    {
	/* Initialise indexL for the first time */
	indexL = 0;
        ixAdslLineCallbackData[lineNum].ixAdslLineStateCallbackIndex = 0;
        isCallbackListInitialized = TRUE;
        /* register the line state display callback as the first Callback*/
        ixAdslLineCallbackData[lineNum].lineStateCallbacks[indexL++] = ixAdslStateChangeCallback;
        ixAdslLineCallbackData[lineNum].ixAdslLineStateCallbackIndex = indexL;
        /* check indexL parameter */
        if(IX_ADSL_SIZEOF_CALLBACK_LIST < indexL)
        {
            return IX_ADSL_STATUS_FAIL;
        }
    }
    /* register callback */
    ixAdslLineCallbackData[lineNum].lineStateCallbacks[indexL++] = callback_fn;
    ixAdslLineCallbackData[lineNum].ixAdslLineStateCallbackIndex = indexL;
    return IX_ADSL_STATUS_SUCCESS;
}

/**
 * @fn ixAdslDyingGaspEnable
 * @brief Enable dying gasp feature. Off by default.
 *
 */
PUBLIC IxAdslStatus
ixAdslDyingGaspEnable (UINT32 lineNum)
{
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("DyingGaspEnable Failed: Invalid Adsl line number specified\n"));
        return IX_ADSL_STATUS_FAIL;
    }
    /* ----------------------------------------------------------------------------
    * Check lineEnable State  : These values can only be updated when Adsl line is down.
    */
    if (IX_ADSL_LINE_DISABLE == ixAdslLineData[lineNum].lineEnableState)
    {
        /* ----------------------------------------------------------------------------
        * Update enable dying gasp
        */
        ixAdslLineData[lineNum].ixAdslEnableGasp=TRUE;
        return IX_ADSL_STATUS_SUCCESS;
    }
    else
    {
        DEBUG_CMD(DEBUG_LOG("\nEnable dying gasp operation Failed because ADSL Line is already up\n"));
        return IX_ADSL_STATUS_FAIL;
    }
}

/**
 * @fn ixAdslCOInterleavedModeSet
 * @brief Set CO interleaved mode
 *        This feature is disabled for release 1.3
 */
#ifdef IX_USE_ADSL_CO
void
ixAdslCOInterleavedModeSet(UINT32 lineNum, int on)
{
    /* ----------------------------------------------------------------------------
    * Check Argument : lineNum
    */
    if (lineNum >= IX_ADSL_NUM_LINES)
    {
        DEBUG_CMD(DEBUG_LOG("CO_InterleavedModeSet Failed: Invalid Adsl line number specified\n"));
        return IX_ADSL_STATUS_FAIL;
    }

    if(on)
    {
        ixAdslLineData[lineNum].ixAdslCOInterleavedMode=TRUE;
    }
    else
    {
        ixAdslLineData[lineNum].ixAdslCOInterleavedMode=FALSE;
    }
}
#endif
/**
 * @fn ixAdslDriverInit
 * @brief Initialise Adsl driver
 *
 */
void
ixAdslDriverInit(void)
{
    UINT32 lineNum;
    if (isInitialised != FALSE)
    {
        /* already initialized */
        return;
    }
    /* Initialize fast mutex locks */
    ixOsServFastMutexInit(&ixAdslCtrleAccessLock);
    /* Set flag to block subsequent calls to this function */
    isInitialised = TRUE;
    /* Reset Adsl parameters */
    memset(&ixAdslLineData, 0, sizeof(ixAdslLineData));
    /* Initialise Adsl parameters */
    for (lineNum = 0 ; lineNum < IX_ADSL_NUM_LINES; lineNum++)
    {
        ixAdslLineData[lineNum].fwInitComplete = FALSE;
        ixAdslLineData[lineNum].lineEnableState = IX_ADSL_LINE_DISABLE;
        ixAdslLineData[lineNum].lineType = IX_ADSL_AUTOSELECT ;
        ixAdslLineData[lineNum].ixAdslCOInterleavedMode = FALSE;
        ixAdslLineData[lineNum].ixAdslEnableGasp = FALSE;
        ixAdslLineData[lineNum].state = IX_ADSL_LINE_WAIT;
        ixAdslLineData[lineNum].lineState = IX_ADSL_LINE_STATE_DOWN;
        ixAdslLineData[lineNum].rc = IX_ADSL_STATUS_SUCCESS;
    }
    /* Adsl module softreset */
    ixAdslSoftReset();
    /* Chipselect enable */
    IX_ADSL_CS_ENABLE;
    /* start the task */
    osError = ix_ossl_thread_create(
                (ix_ossl_thread_entry_point_t)ixAdslLineSupervisoryTask,
                (void *)0,
                &ixADSL_taskId);

    if (osError != IX_OSSL_ERROR_SUCCESS)
    {
        DEBUG_CMD(DEBUG_LOG("Error creating ADSL Supervisory task\n"));
    }
    return;
}

/**
 * @fn ixAdslStateChanged
 * @brief  Calls multiple callbacks registered to an adsl line
 *         during a state change and updates the change to adsl line state
 */
void
ixAdslStateChanged(UINT32 lineNum, IxAdslLineState newState  )
{
    if (IX_ADSL_STATE_DEBUG_STR_COUNT<newState)
    {
        DEBUG_CMD(DEBUG_LOG("Adsl Callback(s): invalid value for lineNewState %d\n",newState));
        return;
    }
    if ( ixAdslLineData[lineNum].lineState != newState )
    {
        UINT32 indexL;
        /* update the current state */
        ixAdslLineData[lineNum].lineState = newState;
        /* invoke callbacks (first registered, first served first) */
        for( indexL = 0;
            indexL < ixAdslLineCallbackData[lineNum].ixAdslLineStateCallbackIndex;
            indexL++)
        {
            (ixAdslLineCallbackData[lineNum].lineStateCallbacks[indexL])(lineNum, newState);
        }
    }
}

/**
 * @fn ixAdslLineOpenInternal
 * @brief  Opens Adsl Line. Called by State Machine.
 *
 */
IxAdslStatus
ixAdslLineOpenInternal(UINT32 lineNum)
{
    UINT32 ctrleReturnCode = 0;
    T_adsl_ConfigurationData  adslLineConfig;
    IxAdslPhyType phyType = ixAdslLineData[lineNum].phyType;
    IxAdslLineType lineType = ixAdslLineData[lineNum].lineType;
    /* ----------------------------------------------------------------------------
    * An array is defined to hold the pointers to various CtrlE library
    * modem activation functions.
    */
    IxAdslEnableModemLine_fp enableModemFns[IX_ADSL_INVALID_MODE] =
    {
        LT_ADSL_enableModemLine,
        LT_ADSL_enableModemLine_glite,
        LT_ADSL_enableModemLine_gdmt,
        LT_ADSL_enableModemLine_ansi,
        LT_ADSL_enableModemLine_loopback
    };
    DEBUG_CMD(DEBUG_LOG("ADSL Line Open\n"));
    /* ----------------------------------------------------------------------------
    * Enable dying gasp function if required
    */
    if(TRUE==ixAdslLineData[lineNum].ixAdslEnableGasp)
    {
        UINT8 enable=ADSL_DYING_GASP_ENABLE;
        ixAdslUtilCtrleWrite(lineNum,&enable, ADSL_DYING_GASP_REG, 1);
    }
    /* ----------------------------------------------------------------------------
    * Configure parameter values for Adsl line
    */
    memset( &adslLineConfig,0, sizeof(adslLineConfig) );
    adslLineConfig.UtopiaLevel=IX_ADSL_UTOPIALEVEL;
    adslLineConfig.RAModeDnstr=AUTOMATIC;
    adslLineConfig.RAModeUpstr=AUTOMATIC;
    if (IX_ADSL_PHY_CPE==phyType)
    {
        adslLineConfig.FunctionCode1FD = IX_ASW_SUBFUNCTION_CODE_CPE;
    }
#ifdef IX_USE_ADSL_CO
    else if (phyType == IX_ADSL_CO_FAST || phyType == IX_ADSL_CO_INTERLEAVED )
    {
        adslLineConfig.MaxAddNoiseMarginDnstr=IX_ASW_CO_MaxAddNoiseMarginDnstr;
        adslLineConfig.MaxAddNoiseMarginUpstr=IX_ASW_CO_MaxAddNoiseMarginUpstr;
        adslLineConfig.TargetNoiseMarginDnstr=IX_ASW_CO_TargetNoiseMarginDnstr;
        adslLineConfig.TargetNoiseMarginUpstr=IX_ASW_CO_TargetNoiseMarginUpstr;
        adslLineConfig.MaxPSDDnstr=IX_ASW_CO_MaxPSDDnstr;
        adslLineConfig.MaxAggrPwrLvlDnstr=IX_ASW_CO_MaxAggrPwrLvlDnstr;
        adslLineConfig.MaxAggrPwrLvlUpstr=IX_ASW_CO_MaxAggrPwrLvlUpstr;
        adslLineConfig.RARatioDnstr=IX_ASW_CO_RARatioDnstr;
        adslLineConfig.RARatioUpstr=IX_ASW_CO_RARatioUpstr;
        adslLineConfig.InterlMaxDelayDnstr=IX_ASW_CO_InterlMaxDelayDnstr;
        adslLineConfig.InterlMaxDelayUpstr=IX_ASW_CO_InterlMaxDelayUpstr;
        adslLineConfig.FunctionCode1FD = IX_ASW_SUBFUNCTION_CODE_CO;
        if (phyType == IX_ADSL_CO_FAST)
        {
            adslLineConfig.FastMinBitrateDnstr=IX_ASW_CO_FastMinBitrateDnstr;
            adslLineConfig.FastMinBitrateUpstr=IX_ASW_CO_FastMinBitrateUpstr;
            adslLineConfig.FastMaxBitrateDnstr=IX_ASW_CO_FastMaxBitrateDnstr;
            adslLineConfig.FastMaxBitrateUpstr=IX_ASW_CO_FastMaxBitrateUpstr;
        } else if (phyType == IX_ADSL_CO_INTERLEAVED)
        {
            adslLineConfig.InterlMinBitrateDnstr=IX_ASW_CO_InterlMinBitrateDnstr;
            adslLineConfig.InterlMinBitrateUpstr=IX_ASW_CO_InterlMinBitrateUpstr;
            adslLineConfig.InterlMaxBitrateDnstr=IX_ASW_CO_InterlMaxBitrateDnstr;
            adslLineConfig.InterlMaxBitrateUpstr=IX_ASW_CO_InterlMaxBitrateUpstr;
        }
    }
#endif
    /* ----------------------------------------------------------------------------
    * Initialize vendor id related Adsl line configuration parameters
    */
    adslLineConfig.itu_vendorid_countrycode  =
                   ixAdslVendorInfo[lineNum].itu_vendorid_countrycode;
    adslLineConfig.itu_vendorid_vendorcode_1 =
                   ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_1;
    adslLineConfig.itu_vendorid_vendorcode_2 =
                   ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_2;
    adslLineConfig.itu_vendorid_vendorcode_3 =
                   ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_3;
    adslLineConfig.itu_vendorid_vendorcode_4 =
                   ixAdslVendorInfo[lineNum].itu_vendorid_vendorcode_4;
    adslLineConfig.itu_vendorid_vendorspecific_1 =
                   ixAdslVendorInfo[lineNum].itu_vendorid_vendorspecific_1;
    adslLineConfig.itu_vendorid_vendorspecific_2 =
                   ixAdslVendorInfo[lineNum].itu_vendorid_vendorspecific_2;
    /* ----------------------------------------------------------------------------
    * Initialize Utopia Addresses for Adsl line configuration
    *            Line number Address  Channel type
    *            ---------------------------------
    *                 0         0        Fast
    *                 0         1        Interleave
    *                 1         2        Fast
    *                 1         3        Interleave
    *                 2         4        Fast
    *                 2         5        Interleave
    *                 3         6        Fast
    *                 3         7        Interleave
    *                 4         8        Fast
    *                 4         9        Interleave
    *                 5         10       Fast
    *                 5         11       Interleave
    *                 6         12       Fast
    *                 6         13       Interleave
    *                 7         14       Fast
    *                 7         15       Interleave
    *            ---------------------------------
    */

    adslLineConfig.UtopiaAddrFast = lineNum*2;
    adslLineConfig.UtopiaAddrInterl = lineNum*2+1;
#ifdef IX_USE_ADSL_20170
    adslLineConfig.UtopiaAddrFast |= 0x1c;     /* The Utopia Address for 20170 must be formatted to bin xxx1 11xx  */
    adslLineConfig.UtopiaAddrInterl |= 0x1c;   /* The Utopia Address for 20170 must be formatted to bin xxx1 11xx  */
#endif /* IX_USE_ADSL_20170 */
#ifndef NDEBUG
    ixAdslLineData[lineNum].trainingLogBufferIndex=0;
#endif
    /* Call the appropriate modem line function */
    ctrleReturnCode = (enableModemFns[lineType])( lineNum, &adslLineConfig );
    if( ctrleReturnCode )
    {
        /* Error Failed to open line.*/
        DEBUG_CMD(DEBUG_LOG("ADSL: Line open failed\n"));
        return (IX_ADSL_STATUS_FAIL);
    }
    /* Return status for Line Open */
    return(IX_ADSL_STATUS_SUCCESS);
}

/**
 * @fn ixAdslCpeTrainingStatusUpdate
 * @brief   Update number of times the Adsl line is retrained.
 *          This function is triggered by state change callback.
 *
 */
void
ixAdslCpeTrainingStatusUpdate ( UINT32 lineNum,
                             UINT32 upstream,
                             UINT32 downstream)
{
    ixAdslLineData[lineNum].training_counters.total++;
    if(downstream >= 8000)
    {
        ixAdslLineData[lineNum].training_counters.GE8000++;
    }
    else if(downstream>= 7000 && downstream <=7999)
    {
        ixAdslLineData[lineNum].training_counters.B7000_7999++;
    }
    else if(downstream <7000)
    {
        ixAdslLineData[lineNum].training_counters.L7000++;
    }
    if(upstream>=768)
    {
        ixAdslLineData[lineNum].training_counters.GE768++;
    }
    else
    {
        ixAdslLineData[lineNum].training_counters.L768++;
    }
}

/**
 * @fn ixTrainingStatusShow
 * @brief   Display the number of times the Adsl line is retrained.
 *          This function is called by ixAdslShow.
 *
 */
void
ixTrainingStatusShow(UINT32 lineNum)
{
    UINT32 lineRateUpstream,lineRateDownstream;
    lineRateUpstream = ixAdslLineRateUpstreamGet(lineNum);
    lineRateDownstream = ixAdslLineRateDownstreamGet(lineNum);
    printf("ADSL Line %u Upstream: %u kbits/s; Downstream: %u kbits/s \n",
        lineNum, lineRateUpstream, lineRateDownstream);
    printf("Training rate statistics:\n");
    printf("   Total no of training: %u\n",
        ixAdslLineData[lineNum].training_counters.total);
    printf("   Downstream rates:\n");
    printf("      Above and equal to 8000kbps: %u\n",
        ixAdslLineData[lineNum].training_counters.GE8000);
    printf("      Between 7000 and 7999 kbps: %u\n",
        ixAdslLineData[lineNum].training_counters.B7000_7999);
    printf("      Less than and equal 6999 : %u\n",
        ixAdslLineData[lineNum].training_counters.L7000);
    printf("   Upstream  rates:\n");
    printf("      Above and equal 768 kbps: %u\n",
        ixAdslLineData[lineNum].training_counters.GE768);
    printf("      Less than and equal  767 kbps: %u\n",
        ixAdslLineData[lineNum].training_counters.L768);
}

/**
 * @fn ixAdslLineSupervisoryTask
 * @brief   This Task runs in the background to periodically activate the ixAdslLineStateMachine
 *
 */
void
ixAdslLineSupervisoryTask(void *p)
{
    UINT32 lineNum;
#ifndef __linux
    osError = ix_ossl_thread_set_priority(ixADSL_taskId, IX_ADSL_TASK_PRIORITY);
    if (IX_OSSL_ERROR_SUCCESS != osError)
    {
    	DEBUG_CMD(DEBUG_LOG("ossl: thread set priority failed!!!\n" ));
	IX_ASSERT(0); /* fatal problem with ossl */
    }
#endif
    DEBUG_CMD(DEBUG_LOG("ADSL task started\n"));
    while (1)
    {
        if (linesInitialized < 1)
        {
            /* no DSL started */
            ixOsServTaskSleep(IX_ADSL_POLL_DELAY);
        }
        else
        {
            for (lineNum = 0; lineNum < linesInitialized; lineNum++)
            {
                /* Activate the state machine at the period of IX_ADSL_POLL_DELAY */
                if (TRUE == ixAdslLineData[lineNum].fwInitComplete)
                {
                    ixAdslLineStateMachine(lineNum);
                }
                ixOsServTaskSleep(IX_ADSL_POLL_DELAY / linesInitialized);
            }
        }
    }
}


/**
 * @fn ixAdslLineStateMachine
 * @brief   Tracks changes and Triggers changes in Operational State
 *          based on lineEnable State  and Adsl Phy state.
 *          This function is called by ixAdslLineSupervisoryTask.
 */
void ixAdslLineStateMachine(UINT32 lineNum)
{

    switch (ixAdslLineData[lineNum].state)
    {
        /* --------------------------------------------------------------------
        * LINE_INIT Operational State
        */
        case IX_ADSL_LINE_INIT:
        {
            UINT32 retval = 0;
            IxAdslLineState lineState;
            DEBUG_CMD(DEBUG_LOG("ADSL State Init\n"));
            /* ----------------------------------------------------------------
            * Notify state change
            */
            lineState = ixAdslLineStateGetInternal(lineNum);
            /* ----------------------------------------------------------------
            * To check Enable state if request is made to activate modem
            */
            if (IX_ADSL_LINE_ENABLE==ixAdslLineData[lineNum].lineEnableState)
            {
                if((IX_ADSL_LINE_STATE_UP_FASTCHANNEL==lineState)||
                   (IX_ADSL_LINE_STATE_UP_INTERLEAVECHANNEL==lineState)||
                   (IX_ADSL_LINE_STATE_UP_DUAL_LATENCY==lineState))
                {
                    /* We need to close the line down to make it reflect the
                       lineEnableState = DOWN
                       This scenario occurs in case when there is no softreset
                       facility for a given platform */
                    ixAdslLineCloseInternal(lineNum);
                    break;
                }
               /* ----------------------------------------------------------------
                * Notify state change
                */
                ixAdslStateChanged(lineNum, lineState);
                /* ------------------------------------------------------------
                * To activate modem
                */
                ixAdslLineData[lineNum].downPollCount=0;
                retval = ixAdslLineOpenInternal(lineNum);
                if ( IX_ADSL_STATUS_FAIL == retval )
                {
                    DEBUG_CMD(DEBUG_LOG("ADSL Line State: Init Failed\n"));
                    /* Close adsl line. Since modem failed to open   */
                    ixAdslLineCloseInternal(lineNum);
                }
                else
                {
                    /* --------------------------------------------------------
                    * Modem activation succesful
                    */
                    if (IX_ADSL_LOOPBACK==ixAdslLineData[lineNum].lineType)
                    {
                        /* Loopback mode is treated as WAIT Operational State */
                        DEBUG_CMD(DEBUG_LOG("ADSL: Loopback - Success\n"));
                        ixAdslLineData[lineNum].state = IX_ADSL_LINE_WAIT;
                    }
                    else
                    {
                        /* Non-Loopback mode is treated as OPEN Operational State */
                        ixAdslLineData[lineNum].state = IX_ADSL_LINE_OPEN;
                        ixAdslLineData[lineNum].openPollCount=0;
                    }
                }
            }
            break;
        }
        /* --------------------------------------------------------------------
        * LINE_OPEN Operational State
        */
         case IX_ADSL_LINE_OPEN:
         {
            IxAdslLineState lineState;
            DEBUG_CMD(DEBUG_LOG("."));
            /* --------------------------------------------------------------------
            * Check if we have timed out trying to open the adsl line
            */
            if (ixAdslLineData[lineNum].openPollCount++ > IX_ADSL_WAIT_FOR_OPEN_CNT)
            {
                /* Attempt a close... if we have timed out  */
                DEBUG_CMD(DEBUG_LOG("ADSL: Line open - TIMEOUT\n"));
                ixAdslLineCloseInternal(lineNum);
                ixAdslLineData[lineNum].state = IX_ADSL_LINE_INIT;
                break;
            }
            /* ----------------------------------------------------------------
             * Notify state change
             */
    	    lineState = ixAdslLineStateGetInternal(lineNum);
            ixAdslStateChanged(lineNum, lineState);
            /* ----------------------------------------------------------------
            * Check if we have succesfully opened the adsl line
            */
            if((IX_ADSL_LINE_STATE_UP_FASTCHANNEL==lineState)||
               (IX_ADSL_LINE_STATE_UP_INTERLEAVECHANNEL==lineState)||
               (IX_ADSL_LINE_STATE_UP_DUAL_LATENCY==lineState))
            {
                /* --------------------------------------------------------
                * We have succesfully opened the adsl line
                * Change Operational State to POLLING
                */
                DEBUG_CMD(DEBUG_LOG("ADSL: Line open - SUCCESS\n"));
                ixAdslLineData[lineNum].state = IX_ADSL_LINE_POLLING;
                break;
            } else
            {
                /* --------------------------------------------------------
                * We are still trying to open the Adsl line
                */
                DEBUG_CMD(DEBUG_LOG("ADSL: Line open - IN PROGRESS\n"));
                break;
            }
        }
        /* --------------------------------------------------------------------
        * LINE_POLLING Operational State
        */
        case IX_ADSL_LINE_POLLING:
        {
            /* ----------------------------------------------------------------
            * Check lineEnable State  if modem is still supposed to be activated
            */
            if (IX_ADSL_LINE_ENABLE==ixAdslLineData[lineNum].lineEnableState)
            {
                /* ------------------------------------------------------------
                * Modem is still supposed to be activated
                */
                IxAdslLineState lineState;
                /* ----------------------------------------------------------------
                 * Notify state change
                 */
                lineState = ixAdslLineStateGetInternal(lineNum);
                ixAdslStateChanged(lineNum, lineState);
                /* Check State */
                if ( (IX_ADSL_LINE_STATE_UP_FASTCHANNEL==lineState)  ||
                     (IX_ADSL_LINE_STATE_UP_INTERLEAVECHANNEL==lineState) ||
                     (IX_ADSL_LINE_STATE_UP_DUAL_LATENCY==lineState) ||
                     (IX_ADSL_LINE_STATE_WAIT_FOR_ACTIVATING==lineState)  ||
                     (IX_ADSL_LINE_STATE_ACTIVATING==lineState)  )
                {
                    /* --------------------------------------------------------
                    * If Operational State is UP or trying to get UP we reset
                    * the Line Down debounce counter
                    */
                    ixAdslLineData[lineNum].downPollCount = 0;
                }
                else
                {
                    /* --------------------------------------------------------
                    * If Operational State is Down.
                    * We check debounce counter to see if it has gone down
                    * too many times.
                    */
                    if ( ixAdslLineData[lineNum].downPollCount >= IX_ADSL_DOWN_DEBOUNCE_CNT)
                    {
                        /* Too many times gone down confirms that Line is down
                        * we initiate modem shutdown */
                        DEBUG_CMD(DEBUG_LOG("ADSL Line state: Down timeout\n"));
                        /* Attempt a close...  */
                        ixAdslLineCloseInternal(lineNum);
                        ixAdslLineData[lineNum].state = IX_ADSL_LINE_INIT;
                    }
                    else
                    {
                        /* We ignore line down in this case... i.e.
                        this condition does not trigger a modem shutdown */
                        DEBUG_CMD(DEBUG_LOG("ADSL Line state: Down detected\n"));
                        ixAdslLineData[lineNum].downPollCount++;
                    }
                }
            }
            break;
        }
        /* --------------------------------------------------------------------
        * LINE_CLOSE Operational State
        */
        case IX_ADSL_LINE_CLOSE:
        {
            /* ----------------------------------------------------------------
            * This state is triggered by ixAdslLineClose PUBLIC API
            * This state forces a modem shutdown.
            * And sets the State Machine to WAIT Operational State
            */
            DEBUG_CMD(DEBUG_LOG("ADSL Line state: Close state\n" ));
            if (IX_ADSL_LOOPBACK != ixAdslLineData[lineNum].lineType)
            {
                ixAdslLineCloseInternal(lineNum);
            }
            ixAdslStateChanged(lineNum, IX_ADSL_LINE_STATE_DOWN);
            ixAdslLineData[lineNum].state = IX_ADSL_LINE_WAIT;
            break;
        }
        /* --------------------------------------------------------------------
        * LINE_WAIT Operational State
        */
        case IX_ADSL_LINE_WAIT:
            /* Standby state */
            break;
        /* --------------------------------------------------------------------
        * Invalid Operational State
        */
        default:
            /* Report the invalid state via ixAdslStateChanged & triggering the calback*/
            DEBUG_CMD(DEBUG_LOG("ADSL Line state: invalid state\n" ));
            IX_ASSERT(0);
            break;
    } /* of switch */
}

/**
 * @fn ixAdslLineCloseInternal
 * @brief  This function closes the Adsl Line and is triggered by the State Machine.
 *
 */
void
ixAdslLineCloseInternal(UINT32 lineNum)
{
    UINT32 ctrleReturnCode = 0;
    /* Send CTRLE command to shutdown modem */
    ctrleReturnCode = ADSL_disableModemLine(lineNum);
    if (C_CONF_DATA_ACK==ctrleReturnCode)
    {
        /* Line closed success */
        DEBUG_CMD(DEBUG_LOG("ADSL: Line closed - success\n"));
        ixAdslLineData[lineNum].rc = IX_ADSL_STATUS_SUCCESS;
    }
    else if (C_CONF_DATA_REJ==ctrleReturnCode)
    {
        /* Line close rejected */
        DEBUG_CMD(DEBUG_LOG("ADSL: Line close rejected\n"));
        ixAdslLineData[lineNum].rc = IX_ADSL_STATUS_FAIL;
    }
    else if (C_CONF_DATA_COMMPROBLEM==ctrleReturnCode)
    {
        /* Line close failed */
        DEBUG_CMD(DEBUG_LOG("ADSL: Line close failed\n"));
        ixAdslLineData[lineNum].rc = IX_ADSL_STATUS_FAIL;
    }
    else if (C_CONF_DATA_ALREADY_DOWN==ctrleReturnCode)
    {
        /* Line already closed */
        DEBUG_CMD(DEBUG_LOG("ADSL: Line closed - already down\n"));
        ixAdslLineData[lineNum].rc = IX_ADSL_STATUS_ALREADY_DOWN;
    }
    else
    {
        /* Line closed - unknown return code */
        DEBUG_CMD(DEBUG_LOG("ADSL: unknown Ctrl-E return code for Line Close request\n"));
        ixAdslLineData[lineNum].rc=IX_ADSL_STATUS_SUCCESS;
    }
}


/**
 * @fn ixAdslStateChangeCallback
 * @brief  This Callback function is invoked when there is a state change
 *
 */
void
ixAdslStateChangeCallback(UINT32 lineNum, IxAdslLineState lineNewState)
{
    /* Display message to indicate callback is triggered */
    printf("\nixAdslStateChangeCallback: ADSL line #%u status = %s\n", lineNum, listStateStrs[lineNewState]);
    /* If Line Operational State is Up we update training statistics */
    if((IX_ADSL_LINE_STATE_UP_FASTCHANNEL==lineNewState)||
      (IX_ADSL_LINE_STATE_UP_INTERLEAVECHANNEL==lineNewState)||
      (IX_ADSL_LINE_STATE_UP_DUAL_LATENCY==lineNewState))
    {
        UINT32 upStreamRate, downStreamRate;
        upStreamRate = ixAdslLineRateUpstreamGet(lineNum);
        downStreamRate = ixAdslLineRateDownstreamGet(lineNum);
       /* Update training statistics */
       ixAdslCpeTrainingStatusUpdate(lineNum, upStreamRate, downStreamRate);
    }
}

/**
 * @fn ixAdslTrainingLogShow
 * @brief  ixAdslTrainingLogShow displays the contents of the CtrlE training log buffer
 *         This function is available only in debug mode.
 */
#ifndef NDEBUG
IxAdslStatus ixAdslTrainingLogShow(UINT32 lineNum)
{
    UINT8 putdata[CTRLE_LOCALWRITEBUFFERSIZE];
    UINT8 getdata[CTRLE_LOCALREADBUFFERSIZE];
    UINT32 trainingLogAddress = 0x0;
    UINT32 j;
    if(0<ixAdslLineData[lineNum].trainingLogBufferIndex)
    {
        printf("\n\n\nThe training log :\n");
        for(j=0;j<IX_ADSL_TRAININGLOGSIZE;j++)
        {
            /* Condition below filters out continuous blank spaces */
            if( !( (32==ixAdslLineData[lineNum].trainingLogBuffer[j-1]) 
                 &&(32==ixAdslLineData[lineNum].trainingLogBuffer[j]) ) ) 
            {
                printf("%c", ixAdslLineData[lineNum].trainingLogBuffer[j]);
            }
        }
        return(IX_ADSL_STATUS_SUCCESS);
    }

    /* ----------------------------------------------------------------------------
    *  Query Phy for training log pointer
    */
    /* write the training log index */
    putdata[IX_ADSL_INDEX0] = IX_ADSL_TRAININGLOGINDEX;
    if (!ixAdslUtilCtrleWrite(lineNum,putdata,BLCKDATAREG,IX_ADSL_SIZE1BYTE))
        return(IX_ADSL_STATUS_FAIL);

    /* Write the command for Modem_Get_Address */
    /* This command momentarily brings the line down */
    putdata[IX_ADSL_INDEX0] = IX_ASW_MODEMGETADDRESS_CMD;
    if (!ixAdslUtilCtrleWrite(lineNum,putdata,TXCOMMANDREG,IX_ADSL_SIZE1BYTE))
        return(IX_ADSL_STATUS_FAIL);

    /* Check for response */
    memset(&getdata, 0, sizeof(getdata));
    if (!ixAdslUtilCtrleRead(lineNum,getdata,RXCOMMANDREG,IX_ADSL_SIZE1BYTE))
        return(IX_ADSL_STATUS_FAIL);

    /* Check for acknowledge code */
    if(getdata[IX_ADSL_INDEX0] != IX_ASW_MODEMGETADDRESS_ACK)
    {
        printf("Failed to get training log buffer pointer! return code : %02x\n",
                getdata[IX_ADSL_INDEX0]);
        return(IX_ADSL_STATUS_FAIL);
    }
    /* Read the address  */
    memset(&getdata, 0, sizeof(getdata));
    if (!ixAdslUtilCtrleRead(lineNum,getdata,BLCKDATAREG,IX_ADSL_SIZE4BYTE))
        return(IX_ADSL_STATUS_FAIL);

    trainingLogAddress = (  getdata[IX_ADSL_INDEX3]<<IX_ADSL_SHIFTBYTES_3 |
                            getdata[IX_ADSL_INDEX2]<<IX_ADSL_SHIFTBYTES_2 |
                            getdata[IX_ADSL_INDEX1]<<IX_ADSL_SHIFTBYTES_1 |
                            getdata[IX_ADSL_INDEX0]<<IX_ADSL_SHIFTBYTES_0 );

    /* ----------------------------------------------------------------------------
    * Retrieve training log from phy and display on terminal
    */
    {
        UINT32 i;
        UINT32 indexL=0;
        UINT32 counter;                            /* Counter for Training Log */
        UINT32 blockSize= IX_ADSL_TRAININGBLOCKSIZE;  /* Chunk size during data retrieval */
        UINT32 addrsTemp;
    	UINT32 sizeTemp;
        printf("\n\n\nThe training log :\n");
	for (counter=0; counter<(IX_ADSL_TRAININGLOGSIZE/IX_ADSL_TRAININGBLOCKSIZE);
             counter++)
        {
    	    addrsTemp=trainingLogAddress;
            /* Load Block Address Register with Training Log Address */
            for(i=0;i<IX_ADSL_SIZE4BYTE;i++)
            {
            	putdata[IX_ADSL_INDEX0] = (UINT8) (addrsTemp % IX_ADSL_0x100);
            	if (!ixAdslUtilCtrleWrite(lineNum,putdata,BLCKBADDRREG+i,IX_ADSL_SIZE1BYTE))
                {                
                    DEBUG_CMD(DEBUG_LOG("ADSL: Training log AddressParameter write failed\n"));
                    return(IX_ADSL_STATUS_FAIL);
                }
            	addrsTemp /= IX_ADSL_0x100;
            }
            sizeTemp=blockSize;
            /* Load Block Length Register with blockSize */
            for(i=0;i<IX_ADSL_SIZE2BYTE;i++)
            {
            	putdata[IX_ADSL_INDEX0] = (UINT8) (sizeTemp % IX_ADSL_0x100);
            	if (!ixAdslUtilCtrleWrite(lineNum,putdata,BLCKLENREG+i,IX_ADSL_SIZE1BYTE))
                {                
                    DEBUG_CMD(DEBUG_LOG("ADSL: Training log SizeParameter write failed\n"));
                    return(IX_ADSL_STATUS_FAIL);
                }
            	sizeTemp /= IX_ADSL_0x100;
            }

            putdata[IX_ADSL_INDEX0] = MODEM_LOAD_BLOCK;
            /* Read the LOAD BLOCK ACK */
            memset(&getdata, 0, sizeof(getdata));
            /* Write the LOAD Command until a Positive Ack is received */
            while (getdata[IX_ADSL_INDEX0]!= ADSL_Modem_Block_Available)
            {
                /* Write the LOAD BLOCK command */
                if (!ixAdslUtilCtrleWrite(lineNum,putdata,TXCOMMANDREG,IX_ADSL_SIZE1BYTE))
                {                
                    DEBUG_CMD(DEBUG_LOG("ADSL: Training log data get ModemLoadBlock Command write failed\n"));
                    return(IX_ADSL_STATUS_FAIL);
                }
                if (!ixAdslUtilCtrleRead(lineNum,getdata,RXCOMMANDREG,IX_ADSL_SIZE1BYTE))
                {                
                    DEBUG_CMD(DEBUG_LOG("ADSL: Training log Command Acknowledge read failed\n"));
                    return(IX_ADSL_STATUS_FAIL);
                }
            }
            /* Get and Display <blockSize> number of bytes from training log buffer */
            memset(&getdata, 0, sizeof(getdata));
            ixAdslUtilCtrleRead(lineNum,getdata,BLCKDATAREG,blockSize);
            for(i=0;i<blockSize;i++)
            {
                /* Condition below filters out meaningless ASCII symbols */
                if((getdata[i]<128 && getdata[i]>31) || (getdata[i]==10 || getdata[i]==13)) 
                { 
                    /* Condition below filters out continuous blank spaces */
                    if(!((getdata[i]==getdata[i-1]) && (getdata[i]==32)))
                    {
                        printf("%c", getdata[i]);
                    }
                    ixAdslLineData[lineNum].trainingLogBufferIndex = indexL;
                    ixAdslLineData[lineNum].trainingLogBuffer[indexL++] = getdata[i];
                }
            }
            trainingLogAddress += blockSize;
        }
    }
    return(IX_ADSL_STATUS_SUCCESS);
}
#endif

/**
 * @fn ixAdslAtmCellCountersShow
 * @brief  ixAdslAtmCellCountersShow displays the Atm cell counter values in the adsl Phy
 *
 */
IxAdslStatus
ixAdslAtmCellCountersShow(UINT32 lineNum)
{
    UINT8 getdata[CTRLE_LOCALREADBUFFERSIZE];
    UINT32 fastRx       = 0x0;
    UINT32 interleaveRx = 0x0;
    UINT32 fastTx       = 0x0;
    UINT32 interleaveTx = 0x0;
    printf("\nAtm Cell Counter Statistics:\n");
    /* Get and Show the Atm Cells Received at Fast Channel */
    if (!ixAdslUtilCtrleRead(lineNum,getdata,R_ACTIVERXCELLSFAST,IX_ADSL_SIZE4BYTE) )
        return(IX_ADSL_STATUS_FAIL);
    fastRx =       ( getdata[IX_ADSL_INDEX3]<<IX_ADSL_SHIFTBYTES_3 |
                     getdata[IX_ADSL_INDEX2]<<IX_ADSL_SHIFTBYTES_2 |
                     getdata[IX_ADSL_INDEX1]<<IX_ADSL_SHIFTBYTES_1 |
                     getdata[IX_ADSL_INDEX0]<<IX_ADSL_SHIFTBYTES_0 );
    printf("\nAtm cells received in fast channel          : %10u", fastRx);

    /* Get and Show the Atm Cells Received at Interleaved Channel */
    if (!ixAdslUtilCtrleRead(lineNum,getdata,R_ACTIVERXCELLSINTERLEAVED,IX_ADSL_SIZE4BYTE) )
        return(IX_ADSL_STATUS_FAIL);
    interleaveRx = ( getdata[IX_ADSL_INDEX3]<<IX_ADSL_SHIFTBYTES_3 |
                     getdata[IX_ADSL_INDEX2]<<IX_ADSL_SHIFTBYTES_2 |
                     getdata[IX_ADSL_INDEX1]<<IX_ADSL_SHIFTBYTES_1 |
                     getdata[IX_ADSL_INDEX0]<<IX_ADSL_SHIFTBYTES_0 );
    printf("\nAtm cells received in interleave channel    : %10u",interleaveRx);

    /* Get and Show Atm Cells Transmitted at Fast channel*/
    if (!ixAdslUtilCtrleRead(lineNum,getdata,R_ACTIVETXCELLSFAST,IX_ADSL_SIZE4BYTE) )
        return(IX_ADSL_STATUS_FAIL);
    fastTx =       ( getdata[IX_ADSL_INDEX3]<<IX_ADSL_SHIFTBYTES_3 |
                     getdata[IX_ADSL_INDEX2]<<IX_ADSL_SHIFTBYTES_2 |
                     getdata[IX_ADSL_INDEX1]<<IX_ADSL_SHIFTBYTES_1 |
                     getdata[IX_ADSL_INDEX0]<<IX_ADSL_SHIFTBYTES_0 );
    printf("\nAtm cells transmitted in fast channel       : %10u", fastTx);

    /* Get and Show Atm Cells Transmitted at Interleaved channel */
    if (!ixAdslUtilCtrleRead(lineNum,getdata,R_ACTIVETXCELLSINTERLEAVED,IX_ADSL_SIZE4BYTE) )
        return(IX_ADSL_STATUS_FAIL);
    interleaveTx = ( getdata[IX_ADSL_INDEX3]<<IX_ADSL_SHIFTBYTES_3 |
                     getdata[IX_ADSL_INDEX2]<<IX_ADSL_SHIFTBYTES_2 |
                     getdata[IX_ADSL_INDEX1]<<IX_ADSL_SHIFTBYTES_1 |
                     getdata[IX_ADSL_INDEX0]<<IX_ADSL_SHIFTBYTES_0 );
    printf("\nAtm cells transmitted in interleave channel : %10u", interleaveTx);
    return(IX_ADSL_STATUS_SUCCESS);
}
/*****************************************************************************
* ixAdslMemoryUnmap - Release Dynamically Allocated Memory
*
* RETURNS: N/A
*/
void ixAdslMemoryUnmap(void)
{
    UINT32 lineNum=IX_ADSL_NUM_LINES;
    if(TRUE==isInitialised)
    {
        while (lineNum-- > 0)
        { 
            if(IX_ADSL_LINE_DISABLE!=ixAdslLineData[lineNum].lineEnableState)
            {
                printf("Line %u is not disabled: Adsl Memory Unmap Failed\n",
	            lineNum);
                return; 
            }
        }
        ixAdslUtilMemoryUnmap();
        isInitialised = FALSE;
        isCallbackListInitialized = FALSE;
        printf("Adsl Driver succesfully unmapped\n");
    } else
    {
        printf("Adsl Driver not initialised: no memory to unmap\n");
        return;
    }
}
