/*
 *   stunnel       Universal SSL tunnel
 *   Copyright (c) 1998-2001 Michal Trojnara <Michal.Trojnara@mirt.net>
 *                 All Rights Reserved
 *
 *   Version:      3.26             (stunnel.c)
 *   Date:         2003.08.29
 *
 *   Author:       Michal Trojnara  <Michal.Trojnara@mirt.net>
 *   Maintainer:   Brian Hatch <bri@stunnel.org>
 *                 See CREDITS for list of folks who have supplied patches.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   In addition, as a special exception, Michal Trojnara gives
 *   permission to link the code of this program with the OpenSSL
 *   library (or with modified versions of OpenSSL that use the same
 *   license as OpenSSL), and distribute linked combinations including
 *   the two.  You must obey the GNU General Public License in all
 *   respects for all of the code used other than OpenSSL.  If you modify
 *   this file, you may extend this exception to your version of the
 *   file, but you are not obligated to do so.  If you do not wish to
 *   do so, delete this exception statement from your version.
 */

#include "common.h"
#include "prototypes.h"
#include "client.h"

#ifdef HAVE_OPENSSL
#include <openssl/crypto.h> /* for SSLeay_version */
#else
#include <crypto.h>
#endif

#ifdef USE_WIN32
static struct WSAData wsa_state;
#endif

    /* Prototypes */
static void daemon_loop(void);
#ifndef USE_WIN32
static void daemonize(void);
static void create_pid(void);
static void delete_pid(void);
#endif

    /* Socket functions */
static int listen_local(void);

    /* Error/exceptions handling functions */
void ioerror(char *);
void sockerror(char *);
void log_error(int, int, char *);
#ifdef DEBUG_LOG
static char *my_strerror(int);
#endif
#ifdef USE_PTHREAD
static void exec_status(void);
#endif
#ifdef USE_FORK
static void client_status(void);
#endif
#ifndef USE_WIN32
static void sigchld_handler(int);
static void signal_handler(int);
static int signal_pipe[2];
static char signal_buffer[16];
#else
static BOOL CtrlHandler(DWORD);
#endif

server_options options;

#ifdef USE_WIN32
/*
int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
    PSTR szCmdLine, int iCmdShow) {
    return 0;
}
*/
#endif

    /* Functions */
int main(int argc, char* argv[]) { /* execution begins here 8-) */
    struct stat st; /* buffer for stat */

#ifdef USE_WIN32
    if(WSAStartup(0x0101, &wsa_state)) {
        sockerror("WSAStartup");
        exit(1);
    }
    if(!SetConsoleCtrlHandler((PHANDLER_ROUTINE)CtrlHandler, TRUE)) {
        ioerror("SetConsoleCtrlHandler");
        exit(1);
    }
#else
    signal(SIGPIPE, SIG_IGN); /* avoid 'broken pipe' signal */
    signal(SIGTERM, signal_handler);
    signal(SIGQUIT, signal_handler);
    signal(SIGINT, signal_handler);
    signal(SIGHUP, signal_handler);
    /* signal(SIGSEGV, signal_handler); */
#endif

    /* process options */
    options.foreground=1;
    options.cert_defaults=CERT_DEFAULTS;

    safecopy(options.pem, PEM_DIR);
    if(options.pem[0]) /* not an empty string */
        safeconcat(options.pem, "/");
    safeconcat(options.pem, "stunnel.pem");

    parse_options(argc, argv);
    if(!(options.option&OPT_FOREGROUND))
        options.foreground=0;
    log_open();
    log(LOG_NOTICE, "Using '%s' as tcpwrapper service name", options.servname);

    /* check if certificate exists */
    if(options.option&OPT_CERT) {
        if(stat(options.pem, &st)) {
            ioerror(options.pem);
            exit(1);
        }
#ifndef USE_WIN32
        if(st.st_mode & 7)
            log(LOG_WARNING, "Wrong permissions on %s", options.pem);
#endif /* defined USE_WIN32 */
    }

    /* check if started from inetd */
    context_init(); /* initialize global SSL context */
    sthreads_init(); /* initialize threads */
    log(LOG_NOTICE, "%s", stunnel_info());
    if(options.option & OPT_DAEMON) { /* daemon mode */
#ifndef USE_WIN32
        if(pipe(signal_pipe)) {
            ioerror("pipe");
            exit(1);
        }
#ifdef FD_CLOEXEC
	fcntl(signal_pipe[0], F_SETFD, FD_CLOEXEC);
	fcntl(signal_pipe[1], F_SETFD, FD_CLOEXEC);
#endif
        if(!(options.option & OPT_FOREGROUND))
            daemonize();
        create_pid();
#endif
        daemon_loop();
    } else { /* inetd mode */
        max_fds=16; /* Just in case */
// Argon
//        max_fds=10; /* Just in case */
        d=calloc(max_fds, sizeof(FD));
        if(!d) {
            log(LOG_ERR, "Memory allocation failed");
            exit(1);
        }
        options.clients = 1;
        client((void *)STDIO_FILENO); /* rd fd=0, wr fd=1 */
    }
    /* close SSL */
    context_free(); /* free global SSL context */
    log_close();
    return 0; /* success */
}

static void daemon_loop(void) {
    int ls, s;
    struct sockaddr_in addr;
    int addrlen;
    int max_clients, fds_ulimit=-1;
    int ready;
    int old_val;
    fd_set read_fds;

#if defined HAVE_SYSCONF
    fds_ulimit=sysconf(_SC_OPEN_MAX);
    if(fds_ulimit<0)
        ioerror("sysconf");
#elif defined HAVE_GETRLIMIT
    struct rlimit rlim;
    if(getrlimit(RLIMIT_NOFILE, &rlim)<0)
        ioerror("getrlimit");
    else
        fds_ulimit=rlim.rlim_cur;
    if(fds_ulimit==RLIM_INFINITY)
        fds_ulimit=-1;
#endif
    if(fds_ulimit>=16 && fds_ulimit<FD_SETSIZE)
        max_fds=fds_ulimit;
    else
        max_fds=FD_SETSIZE;
    //max_fds=20;    
    d=calloc(max_fds, sizeof(FD));
    if(!d) {
        log(LOG_ERR, "Memory allocation failed");
        exit(1);
    }
    max_clients=max_fds>=256 ? max_fds*125/256 : (max_fds-8)/2;
    log(LOG_NOTICE, "FD_SETSIZE=%d, file ulimit=%d%s -> %d clients allowed",
        FD_SETSIZE, fds_ulimit, fds_ulimit<0?" (unlimited)":"", max_clients);
    ls=listen_local();
    options.clients=0;
#ifndef USE_WIN32
    /* Handle signals about dead children */
    signal(SIGCHLD, sigchld_handler);
#endif /* defined USE_FORK */
    while(1) {
        addrlen=sizeof(addr);
        do {
            FD_ZERO(&read_fds);
            FD_SET(ls, &read_fds);
#ifndef USE_WIN32
            FD_SET(signal_pipe[0], &read_fds);
            ready=select((ls > signal_pipe[0]) ? ls + 1 : signal_pipe[0] + 1,
                         &read_fds, NULL, NULL, NULL);
#else
            ready=select(ls + 1, &read_fds, NULL, NULL, NULL);
#endif
        } while(ready<=0 && get_last_error()==EINTR);
#ifndef USE_WIN32
        if(FD_ISSET(signal_pipe[0], &read_fds)) {
            read(signal_pipe[0], signal_buffer, sizeof(signal_buffer));
#ifdef	USE_PTHREAD 
            exec_status(); /* Report status of 'exec' process */
#else
            client_status(); /* Report the status of the child process */
#endif
        }
#endif
        /* if we didn't also have a new connection, go back to waiting */
        if(!FD_ISSET(ls, &read_fds)) {
            continue;
        }
        /* if we also have a new connection, process it. First, put
	 * the socket into nonblocking mode so that network errors
	 * won't hang the next call "forever". */
#ifndef USE_WIN32
	old_val = fcntl(ls, F_GETFL, 0);
	if (old_val >= 0)
	    fcntl(ls, F_SETFL, old_val | O_NONBLOCK);
#endif
        s=accept(ls, (struct sockaddr *)&addr, &addrlen);
#ifndef USE_WIN32
	if (old_val >= 0)
	    fcntl(ls, F_SETFL, old_val);
#endif

	if(s<0) {
            sockerror("accept");
            continue;
        }
        enter_critical_section(CRIT_NTOA); /* inet_ntoa is not mt-safe */
        log(LOG_DEBUG, "%s accepted FD=%d from %s:%d", options.servname, s,
            inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
        leave_critical_section(CRIT_NTOA);
        if(options.clients>=max_clients) {
            log(LOG_WARNING, "Connection rejected: too many clients (>=%d)",
                max_clients);
            closesocket(s);
            continue;
        }
        if(s>=max_fds) {
            log(LOG_ERR,
                "Connection rejected: file descriptor out of range (>=%d)",
                max_fds);
            closesocket(s);
            continue;
        }
#ifdef FD_CLOEXEC
        fcntl(s, F_SETFD, FD_CLOEXEC); /* close socket in child execvp */
#endif
        if(create_client(ls, s, client)) {
            log(LOG_ERR, "Connection rejected: create_client failed");
            closesocket(s);
            continue;
        }
        enter_critical_section(CRIT_CLIENTS); /* for multi-cpu machines */
        options.clients++;
        leave_critical_section(CRIT_CLIENTS);
    }
    log(LOG_ERR, "INTERNAL ERROR: End of infinite loop");
}

#ifndef USE_WIN32
static void daemonize() { /* go to background */
#ifdef HAVE_DAEMON
    if(daemon(0,0)==-1){
        ioerror("daemon");
        exit(1);
    }
#else
    chdir("/");
    switch(fork()) {
    case -1:    /* fork failed */
        ioerror("fork");
        exit(1);
    case 0:     /* child */
        break;
    default:    /* parent */
        exit(0);
    }
    if (setsid() == -1) {
        ioerror("setsid");
        exit(1);
    }
    close(0);
    close(1);
    close(2);
#endif
}

static void create_pid(void) {
    int pf;
    char pid[STRLEN];
    struct stat sb;
    int force_dir;
    char tmpdir[STRLEN];

    safecopy(tmpdir, options.pid_dir);

    if(strcmp(tmpdir, "none") == 0) {
        log(LOG_DEBUG, "No pid file being created");
        options.pidfile[0]='\0';
        return;
    }
    if(!strchr(tmpdir, '/')) {
        log(LOG_ERR, "Argument to -P (%s) must be full path name",
            tmpdir);
        /* Why?  Because we don't want to confuse by
           allowing '.', which would be '/' after
           daemonizing) */
        exit(1);
    }
    options.dpid=(unsigned long)getpid();

    /* determine if they specified a pid dir or pid file,
       and set our options.pidfile appropriately */
    if(tmpdir[strlen(tmpdir)-1] == '/' ) {
        force_dir=1; /* user requested -P argument to be a directory */
        tmpdir[strlen(tmpdir)-1] = '\0';
    } else {
        force_dir=0; /* this can be either a file or a directory */
    }
    if(!stat(tmpdir, &sb) && S_ISDIR(sb.st_mode)) { /* directory */
#ifdef HAVE_SNPRINTF
        snprintf(options.pidfile, STRLEN,
            "%s/stunnel.%s.pid", tmpdir, options.servname);
#else /* No data from network here.  Am I paranoid? */
        safecopy(options.pidfile, tmpdir);
        safeconcat(options.pidfile, "/stunnel.");
        safeconcat(options.pidfile, options.servname);
        safeconcat(options.pidfile, ".pid");
#endif
    } else { /* file */
        if(force_dir) {
            log(LOG_ERR, "Argument to -P (%s/) is not valid a directory name",
                tmpdir);
            exit(1);
        }
        safecopy(options.pidfile, tmpdir);
    }

    /* silently remove old pid file */
    unlink(options.pidfile);
    if (-1==(pf=open(options.pidfile, O_WRONLY|O_CREAT|O_TRUNC|O_EXCL,0644))) {
        log(LOG_ERR, "Cannot create pid file %s", options.pidfile);
        ioerror("create");
        exit(1);
    }
    sprintf(pid, "%lu", options.dpid);
    write( pf, pid, strlen(pid) );
    close(pf);
    log(LOG_DEBUG, "Created pid file %s", options.pidfile);
    atexit(delete_pid);
}

static void delete_pid(void) {
    log(LOG_DEBUG, "removing pid file %s", options.pidfile);
    if((unsigned long)getpid()!=options.dpid)
        return; /* Current process is not main daemon process */
    if(unlink(options.pidfile)<0)
        ioerror(options.pidfile); /* not critical */
}
#endif /* defined USE_WIN32 */

static int listen_local(void) { /* bind and listen on local interface */
    struct sockaddr_in addr;
    int ls;

    if((ls=socket(AF_INET, SOCK_STREAM, 0))<0) {
        sockerror("local socket");
        exit(1);
    }
    if(set_socket_options(ls, 0)<0)
        exit(1);
    memset(&addr, 0, sizeof(addr));
    addr.sin_family=AF_INET;
    addr.sin_addr.s_addr=*options.localnames;
    addr.sin_port=options.localport;
    if(bind(ls, (struct sockaddr *)&addr, sizeof(addr))) {
        sockerror("bind");
        exit(1);
    }
    enter_critical_section(CRIT_NTOA); /* inet_ntoa is not mt-safe */
    log(LOG_DEBUG, "%s bound to %s:%d", options.servname,
        inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
    leave_critical_section(CRIT_NTOA);
    if(listen(ls, 5)) {
        sockerror("listen");
        exit(1);
    }
#ifdef FD_CLOEXEC
    fcntl(ls, F_SETFD, FD_CLOEXEC); /* close socket in child execvp */
#endif
	    
#ifndef USE_WIN32
    if(options.setgid_group) {
        struct group *gr;
        gid_t gr_list[1];

        gr=getgrnam(options.setgid_group);
        if(!gr) {
            log(LOG_ERR, "Failed to get GID for group %s",
                options.setgid_group);
            exit(1);
        }
        if(setgid(gr->gr_gid)) {
            sockerror("setgid");
            exit(1);
        }
        gr_list[0]=gr->gr_gid;
        if(setgroups(1, gr_list)) {
            sockerror("setgroups");
            exit(1);
        }
    }

    if(options.setuid_user) {
        struct passwd *pw;

        pw=getpwnam(options.setuid_user);
        if(!pw) {
            log(LOG_ERR, "Failed to get UID for user %s",
                options.setuid_user);
            exit(1);
        }
#ifndef USE_WIN32
        /* gotta chown that pid file, or we can't remove it. */
        if ( options.pidfile[0] && chown( options.pidfile, pw->pw_uid, -1) ) {
            log(LOG_ERR, "Failed to chown pidfile %s", options.pidfile);
        }
#endif
        if(setuid(pw->pw_uid)) {
            sockerror("setuid");
            exit(1);
        }
    }
#endif /* USE_WIN32 */

    return ls;
}

int set_socket_options(int s, int type) {
    sock_opt *ptr;
    extern sock_opt sock_opts[];
#ifdef DEBUG_LOG    
    static char *type_str[3]={"accept", "local", "remote"};
#endif    
    int opt_size;

    for(ptr=sock_opts;ptr->opt_str;ptr++) {
        if(!ptr->opt_val[type])
            continue; /* default */
        switch(ptr->opt_type) {
        case TYPE_LINGER:
            opt_size=sizeof(struct linger); break;
        case TYPE_TIMEVAL:
            opt_size=sizeof(struct timeval); break;
        case TYPE_STRING:
            opt_size=strlen(ptr->opt_val[type]->c_val)+1; break;
        default:
            opt_size=sizeof(int); break;
        }
        if(setsockopt(s, ptr->opt_level, ptr->opt_name,
                (void *)ptr->opt_val[type], opt_size)) {
            sockerror(ptr->opt_str);
            return -1; /* FAILED */
        } else {
            log(LOG_DEBUG, "%s option set on %s socket",
                ptr->opt_str, type_str[type]);
        }
    }
    return 0; /* OK */
}

void ioerror(char *txt) { /* Input/Output error handler */
    log_error(LOG_ERR, get_last_error(), txt);
}

void sockerror(char *txt) { /* Socket error handler */
    log_error(LOG_ERR, get_last_socket_error(), txt);
}

void log_error(int level, int error, char *txt) { /* Generic error logger */
    log(level, "%s: %s (%d)", txt, my_strerror(error), error);
}

#ifdef DEBUG_LOG
static char *my_strerror(int errnum) {
    switch(errnum) {
#ifdef USE_WIN32
    case 10004:
        return "Interrupted system call (WSAEINTR)";
    case 10009:
        return "Bad file number (WSAEBADF)";
    case 10013:
        return "Permission denied (WSAEACCES)";
    case 10014:
        return "Bad address (WSAEFAULT)";
    case 10022:
        return "Invalid argument (WSAEINVAL)";
    case 10024:
        return "Too many open files (WSAEMFILE)";
    case 10035:
        return "Operation would block (WSAEWOULDBLOCK)";
    case 10036:
        return "Operation now in progress (WSAEINPROGRESS)";
    case 10037:
        return "Operation already in progress (WSAEALREADY)";
    case 10038:
        return "Socket operation on non-socket (WSAENOTSOCK)";
    case 10039:
        return "Destination address required (WSAEDESTADDRREQ)";
    case 10040:
        return "Message too long (WSAEMSGSIZE)";
    case 10041:
        return "Protocol wrong type for socket (WSAEPROTOTYPE)";
    case 10042:
        return "Bad protocol option (WSAENOPROTOOPT)";
    case 10043:
        return "Protocol not supported (WSAEPROTONOSUPPORT)";
    case 10044:
        return "Socket type not supported (WSAESOCKTNOSUPPORT)";
    case 10045:
        return "Operation not supported on socket (WSAEOPNOTSUPP)";
    case 10046:
        return "Protocol family not supported (WSAEPFNOSUPPORT)";
    case 10047:
        return "Address family not supported by protocol family (WSAEAFNOSUPPORT)";
    case 10048:
        return "Address already in use (WSAEADDRINUSE)";
    case 10049:
        return "Can't assign requested address (WSAEADDRNOTAVAIL)";
    case 10050:
        return "Network is down (WSAENETDOWN)";
    case 10051:
        return "Network is unreachable (WSAENETUNREACH)";
    case 10052:
        return "Net dropped connection or reset (WSAENETRESET)";
    case 10053:
        return "Software caused connection abort (WSAECONNABORTED)";
    case 10054:
        return "Connection reset by peer (WSAECONNRESET)";
    case 10055:
        return "No buffer space available (WSAENOBUFS)";
    case 10056:
        return "Socket is already connected (WSAEISCONN)";
    case 10057:
        return "Socket is not connected (WSAENOTCONN)";
    case 10058:
        return "Can't send after socket shutdown (WSAESHUTDOWN)";
    case 10059:
        return "Too many references, can't splice (WSAETOOMANYREFS)";
    case 10060:
        return "Connection timed out (WSAETIMEDOUT)";
    case 10061:
        return "Connection refused (WSAECONNREFUSED)";
    case 10062:
        return "Too many levels of symbolic links (WSAELOOP)";
    case 10063:
        return "File name too long (WSAENAMETOOLONG)";
    case 10064:
        return "Host is down (WSAEHOSTDOWN)";
    case 10065:
        return "No Route to Host (WSAEHOSTUNREACH)";
    case 10066:
        return "Directory not empty (WSAENOTEMPTY)";
    case 10067:
        return "Too many processes (WSAEPROCLIM)";
    case 10068:
        return "Too many users (WSAEUSERS)";
    case 10069:
        return "Disc Quota Exceeded (WSAEDQUOT)";
    case 10070:
        return "Stale NFS file handle (WSAESTALE)";
    case 10091:
        return "Network SubSystem is unavailable (WSASYSNOTREADY)";
    case 10092:
        return "WINSOCK DLL Version out of range (WSAVERNOTSUPPORTED)";
    case 10093:
        return "Successful WSASTARTUP not yet performed (WSANOTINITIALISED)";
    case 10071:
        return "Too many levels of remote in path (WSAEREMOTE)";
    case 11001:
        return "Host not found (WSAHOST_NOT_FOUND)";
    case 11002:
        return "Non-Authoritative Host not found (WSATRY_AGAIN)";
    case 11003:
        return "Non-Recoverable errors: FORMERR, REFUSED, NOTIMP (WSANO_RECOVERY)";
    case 11004:
        return "Valid name, no data record of requested type (WSANO_DATA)";
#if 0
    case 11004: /* typically, only WSANO_DATA is reported */
        return "No address, look for MX record (WSANO_ADDRESS)";
#endif
#endif /* defined USE_WIN32 */
    default:
        return strerror(errnum);
    }
}
#endif

#ifdef USE_FORK
static void client_status(void) { /* dead children detected */
    int pid, status;

#ifdef HAVE_WAIT_FOR_PID
    while((pid=wait_for_pid(-1, &status, WNOHANG))>0) {
        option.clients--; /* one client less */
#else
        if((pid=wait(&status))>0) {
            option.clients--; /* one client less */
#endif
#ifdef WIFSIGNALED
        if(WIFSIGNALED(status)) {
            log(LOG_DEBUG, "Process %d terminated on signal %d (%d left)",
                pid, WTERMSIG(status), num_clients);
        } else {
            log(LOG_DEBUG, "Process %d finished with code %d (%d left)",
                pid, WEXITSTATUS(status), num_clients);
        }
    }
#else
        log(LOG_DEBUG, "Process %d finished with code %d (%d left)",
            pid, status, num_clients);
    }
#endif
}
#endif


#ifdef USE_PTHREAD
static void exec_status(void) { /* dead local ('exec') process detected */
    int pid, status;

#ifdef HAVE_WAIT_FOR_PID
    while((pid=wait_for_pid(-1, &status, WNOHANG))>0) {
#else
        if((pid=wait(&status))>0) {
#endif
#ifdef WIFSIGNALED
             if(WIFSIGNALED(status)) {
                 log(LOG_INFO, "Local process %d terminated on signal %d",
                     pid, WTERMSIG(status));
             } else {
                 log(LOG_INFO, "Local process %d finished with code %d",
                     pid, WEXITSTATUS(status));
             }
#else
        log(LOG_INFO, "Local process %d finished with status %d",
            pid, status);
#endif
        }
}
#endif /* !defined USE_WIN32 */


#ifndef USE_WIN32
static void sigchld_handler(int sig) { /* Death of child process detected */
    int save_errno=errno;
    
    write(signal_pipe[1], signal_buffer, 1);
    signal(SIGCHLD, sigchld_handler);
    errno=save_errno;
}

static void signal_handler(int sig) { /* Signal handler */
    log(LOG_ERR, "Received signal %d; terminating", sig);
    exit(3);
}

#else /* !defined USE_WIN32 */

static BOOL CtrlHandler(DWORD fdwCtrlType) {
    switch(fdwCtrlType) {
    case CTRL_C_EVENT: /* Handle the CTRL+C signal */
        log(LOG_NOTICE, "Process exit: CTRL+C");
        break;
    case CTRL_CLOSE_EVENT: /* CTRL+CLOSE: confirm that the user wants to exit */
        log(LOG_NOTICE, "Process exit: window closed");
        break;
    case CTRL_BREAK_EVENT:
        log(LOG_NOTICE, "Process exit: CTRL+BREAK");
        break;
    case CTRL_LOGOFF_EVENT:
        log(LOG_NOTICE, "Process exit: user logoff");
        break;
    case CTRL_SHUTDOWN_EVENT:
        log(LOG_NOTICE, "Process exit: system shutdown");
        break;
    default:
        return FALSE;
    }
    ExitProcess(0);
}

#endif /* !defined USE_WIN32 */

char *stunnel_info() {
    static char retval[STRLEN];

    safecopy(retval, "stunnel " VERSION " on " HOST);
#ifdef USE_PTHREAD
    safeconcat(retval, " PTHREAD");
#endif
#ifdef USE_WIN32
    safeconcat(retval, " WIN32");
#endif
#ifdef USE_FORK
    safeconcat(retval, " FORK");
#endif
#ifdef USE_LIBWRAP
    safeconcat(retval, "+LIBWRAP");
#endif
    safeconcat(retval, " with ");
    safeconcat(retval, SSLeay_version(SSLEAY_VERSION));
    return retval;
}

/* End of stunnel.c */
