/** @file    IxEthAcc.h
 *
 * @brief this file contains the public API of @ref IxEthAcc component
 *
 * Design notes:
 * The mbuf address is to be specified in the least significant 28-bit
 * location [0:27]. The most significant 4-bit [31:29] are for interal 
 * use only.
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 *
 */

#ifndef IxEthAcc_H
#define IxEthAcc_H

#include <IxOsBuffMgt.h>
#include <IxTypes.h>

/**
 * @defgroup IxEthAcc IXP425 Ethernet Access (IxEthAcc) API
 *
 * @brief ethAcc is a library that does provides access to the internal IXP425 10/100Bt Ethernet MACs.
 *
 *@{
 */

/**
 * @ingroup IxEthAcc
 * @enum IxEthAccStatus
 * @brief This is an enum to define the Ethernet Access status
 */
typedef enum /* IxEthAccStatus */
{
    IX_ETH_ACC_SUCCESS = IX_SUCCESS, /**<return success*/
    IX_ETH_ACC_FAIL = IX_FAIL, /**<return fail*/
    IX_ETH_ACC_INVALID_PORT, /**<return invalid port*/
    IX_ETH_ACC_PORT_UNINITIALIZED, /**<return uninitialized*/
    IX_ETH_ACC_MAC_UNINITIALIZED, /**<return MAC uninitialized*/
    IX_ETH_ACC_INVALID_ARG, /**<return invalid arg*/
    IX_ETH_TX_Q_FULL, /**<return tx queue is full*/
    IX_ETH_ACC_NO_SUCH_ADDR /**<return no such address*/
} IxEthAccStatus;

/**
 * @ingroup IxEthAcc
 * @enum IxEthAccPortId
 * @brief This is an emum to define the IXP425 Mac Ethernet device.
 */
typedef enum  
{
	IX_ETH_PORT_1 = 0, /**< Ethernet Port 1 */
	IX_ETH_PORT_2 = 1  /**< Ethernet port 2 */
} IxEthAccPortId;

/**
 * @ingroup IxEthAcc
 *
 * @def IX_ETH_ACC_NUMBER_OF_PORTS
 *
 * @brief  Defines related to the number of NPE's and mapping between PortId and NPE.
 *
 */
#define IX_ETH_ACC_NUMBER_OF_PORTS (2)

/**
 * @ingroup IxEthAcc
 *
 * @def IX_IEEE803_MAC_ADDRESS_SIZE
 *
 * @brief  Defines the size of the MAC address NPE.
 *
 */
#define IX_IEEE803_MAC_ADDRESS_SIZE (6)

/**
 *
 * @brief The IEEE 802.3 Ethernet MAC address structure.
 *
 * The data should be packed with bytes xx:xx:xx:xx:xx:xx 
 * @note
 * The data must be packed in network byte order.
 */
typedef struct  
{
    UINT8 macAddress[IX_IEEE803_MAC_ADDRESS_SIZE]; /**< MAC address */
} IxEthAccMacAddr;

/**
 * @ingroup IxEthAcc
 * @def IX_ETH_ACC_NUM_TX_PRIORITIES
 * @brief The number of transmit priorities
 * 
 */
#define IX_ETH_ACC_NUM_TX_PRIORITIES (8)

/**
 * @ingroup IxEthAcc
 * @enum IxEthAccTxPriority
 * @brief enum to submit a frame with relative priority.
 * 
 */
typedef enum  
{
	IX_ETH_ACC_TX_PRIORITY_0 = 0, /**<Lowest Priorty submission */
	IX_ETH_ACC_TX_PRIORITY_1 = 1, /**<submission prority of 1 (0 is lowest)*/
	IX_ETH_ACC_TX_PRIORITY_2 = 2, /**<submission prority of 2 (0 is lowest)*/
	IX_ETH_ACC_TX_PRIORITY_3 = 3, /**<submission prority of 3 (0 is lowest)*/
	IX_ETH_ACC_TX_PRIORITY_4 = 4, /**<submission prority of 4 (0 is lowest)*/
	IX_ETH_ACC_TX_PRIORITY_5 = 5, /**<submission prority of 5 (0 is lowest)*/
	IX_ETH_ACC_TX_PRIORITY_6 = 6, /**<submission prority of 6 (0 is lowest)*/
	IX_ETH_ACC_TX_PRIORITY_7 = 7, /**<Highest priority submission */

	IX_ETH_ACC_TX_DEFAULT_PRIORITY = IX_ETH_ACC_TX_PRIORITY_0 /**< By default send all 
								 pkts with lowest priority */
} IxEthAccTxPriority;

/**
 * @ingroup IxEthAcc
 * @enum IxEthAccDuplexMode
 * @brief Definition to provision the duplex mode of the MAC. 
 * 
 */
typedef enum
{
    IX_ETH_ACC_FULL_DUPLEX, /**< Full duplex operation of the MAC */
    IX_ETH_ACC_HALF_DUPLEX  /**< Half duplex operation of the MAC */
} IxEthAccDuplexMode;


/**
 * @ingroup IxEthAcc
 *
 * @brief This defines the recommanded minimum size of MBUF's submitted
 * to the frame receive service.
 *
 */
#define IX_ETHACC_RX_MBUF_MIN_SIZE (2048)

/**
 * @ingroup IxEthAcc
 *
 * @brief This defines the highest MII address of any attached PHYs 
 * 
 * Max number for PHY address is 31, add on for range checking.
 *
 */
#define IXP425_ETH_ACC_MII_MAX_ADDR   32

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccInit(void)
 * 
 * @brief Initialize the Ethernet Access Service.
 * 
 * Initialize the IXP425 Ethernet Access Service.
 * 
 * - Reentrant    - no
 * - ISR Callable - no
 * 
 * This should be called once per module initialization.
 * @pre
 *   The NPE must first be downloaded with the required microcode which supports all
 *   required features.
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL	:  Service has failed to initialize.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccInit(void);


/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccUnload(void)
 * 
 * @brief Unload the Ethernet Access Service.
 * 
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @return void 
 *
 * <hr>
 */
void ixEthAccUnload(void);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortInit( IxEthAccPortId portId)
 *
 * @brief Initialize an Ethernet MAC Port.
 *
 * Initialize the NPE/Ethernet MAC hardware.
 * Verify NPE downloaded and operational.
 * The NPE shall be available for usage once this API returns.
 * Verify that the Ethernet port is present before initializing
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * This should be called once per mac device.
 * The NPE/MAC shall be in disabled state after init.
 *
 * @pre
 *   The component must be initialized via @a ixEthAccInit
 *   The NPE must first be downloaded with the required microcode which supports all
 *   required features.
 *
 * Dependant on Services: (Must be initialized before using this service may be initialized)
 * 	ixNPEmh - NPE Message handling service.
 * 	ixQmgr	- Queue Manager component.
 *
 * @param portId :  @a IxEthAccPortId
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS: if the ethernet port is not present, a warning will be issued.
 * - IX_ETH_ACC_FAIL : The NPE processor has failed to initialize.
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortInit(IxEthAccPortId portId);


/*************************************************************************

 #####     ##     #####    ##            #####     ##     #####  #    #
 #    #   #  #      #     #  #           #    #   #  #      #    #    #
 #    #  #    #     #    #    #          #    #  #    #     #    ######
 #    #  ######     #    ######          #####   ######     #    #    #
 #    #  #    #     #    #    #          #       #    #     #    #    #
 #####   #    #     #    #    #          #       #    #     #    #    #

*************************************************************************/


/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortTxFrameSubmit( 
    IxEthAccPortId portId,
    IX_MBUF *buffer, 
    IxEthAccTxPriority priority)
 * 
 * @brief This function shall be used to submit MBUFs buffers for transmission on a particular MAC device. 
 *
 * This function shall be used to submit MBUFs buffers for transmission on a particular MAC device.
 *
 * When the frame is transmitted, the buffer shall be returned thru the 
 * callback @a IxEthAccPortTxDoneCallback.
 *
 * The only alterations made to the buffer are assocuated with the next packet chain pointer. 
 * This is used to internally  queue frames in the service when submitting faster than
 * the network line rate. In case of over=Submitting, the order of the frames on the 
 * network may be modified.
 *
 * Buffers shall be not queued for transmission if the port is disabled.
 * The port can be enabled using @a ixEthAccPortEnable
 *
 * 
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 *
 * @pre 
 *  @a ixEthAccPortTxDoneCallbackRegister must be called to register a function to allow this service to
 *   return the buffer to the calling service. 
 * 
 * @note 
 *  If the buffer submit fails for any reason the user has retained ownership of the buffer.
 *
 * @param portId - MAC port ID to transmit Ethernet frame on.
 * @param buffer - pointer to an MBUF formatted buffer. Chained buffers are supported for transmission.
 *             Chained packets are not supported and the field IX_MBUF_NEXT_PKT_IN_CHAIN_PTR must be NULL. 
 * @param priority - @a IxEthAccTxPriority
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL  : Failed to queue frame for transmission. 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */

IxEthAccStatus ixEthAccPortTxFrameSubmit( 
    IxEthAccPortId portId,
    IX_MBUF *buffer, 
    IxEthAccTxPriority priority
);

/**
 * @ingroup IxEthAcc
 *
 * @brief Function prototype for Ethernet Tx Buffer Done callback. Registered 
 *  via @a ixEthAccTxBufferDoneCallbackRegister 
 * 
 * This function is called once the previously submitted buffer is no longer required by this service.
 * It may be returned upon successful transmission of the frame or shutdown of port prior to submission.
 * The calling of this registered function is not a guarantee of successful transmission of the buffer.
 *
*  
 * - Reentrant    - yes , The user provided function should be reentrant.
 * - ISR Callable - yes , The user provided function must be callable from an ISR.
 *
 *
 * <b>Calling Context </b>: 
 * @par
 *   This callback is called in the context of the queue manager dispatch loop @a ixQmgrgrDispatcherLoopRun
 *   within the @ref IxQMgrAPI component. The calling context may be from interrupt or high priority thread. 
 *   The decision is system specific.
 *
 * @param callbackTag - This tag is that provided when the callback was registered for a particular MAC 
 * via @a ixEthAccPortRxCallbackRegister. It allows the same callback to be used for multiple MACs.
 * @param mbuf - Pointer to the Tx mbuf descriptor. 
 * 
 * @return void
 *
 * @note
 * The field IX_MBUF_NEXT_PKT_IN_CHAIN_PTR is modified by the access layer and reset to NULL.
 *
 * <hr>
 */
typedef void (*IxEthAccPortTxDoneCallback) ( UINT32 callbackTag, IX_MBUF *buffer );



/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortTxDoneCallbackRegister( IxEthAccPortId portId, 
					   IxEthAccPortTxDoneCallback txCallbackFn, 
 					   UINT32 callbackTag)
 *
 * @brief This function registers a callback function to facilitate the return of transmit buffers to the  
 * user.
 * 
 * This function registers the transmit buffer done function callback for a particular port.
 *
 * The registered callback function is called once the previously submitted buffer is no longer required by this service.
 * It may be returned upon successful transmission of the frame or  shutdown of port prior to submission.
 * The calling of this registered function is not a guarantee of successful transmission of the buffer.
 *
 * If called several times the latest callback shall be registered for a particular port.
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 * @pre
 *	The port must be initialized via @a ixEthAccPortInit
 *
 *
 * @param portId - Register callback for a particular MAC device.
 * @param txCallbackFn - @a IxEthAccTxBufferDoneCallbackFn - Function to be called to return transmit buffers to the user.
 * @param callbackTag -  This tag shall be provided to the callback function.
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 * - IX_ETH_ACC_INVALID_ARG : An argument other than portId is invalid.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortTxDoneCallbackRegister( IxEthAccPortId portId, 
					   IxEthAccPortTxDoneCallback txCallbackFn, 
 					   UINT32 callbackTag);



/**
 * @ingroup IxEthAcc
 *
 * @brief Function prototype for Ethernet Frame Rx callback. Registered via @a ixEthAccPortRxCallbackRegister 
 * 
 * It is the responsibility of the user function to free any MBUF's which it receives.
 *  
 * - Reentrant    - yes , The user provided function should be reentrant.
 * - ISR Callable - yes , The user provided function must be callable from an ISR.
 * @par
 *
 * <b>Calling Context </b>: 
 * @par
 *   This callback is called in the context of the queue manager dispatch loop @a ixQmgrgrDispatcherLoopRun
 *   within the @ref IxQMgrAPI component. The calling context may be from interrupt or high priority thread. 
 *   The decision is system specific.
 *
 *
 * @param callbackTag - This tag is that provided when the callback was registered for a particular MAC 
 * via @a ixEthAccPortRxCallbackRegister. It allows the same callback to be used for multiple MACs.
 * @param mbuf - Pointer to the Rx mbuf header. Mbufs may be chained if 
 *               the frame length is greater than the supplied mbuf length.
 * @param portId - ID of the port which match the destination MAC address
 *      for this frame (The value is greater or equal to 
 *      IX_ETH_DB_NUMBER_OF_PORTS if the MAC address is not found
 *      in the copy of the database shared with the NPE)
 * 
 * @return void
 *
 * @note
 * Buffers may not be filled up to the length supplied in 
 * @a ixEthAccPortRxFreeReplenish(). The firmware fills
 * them to the previous 64 bytes boundary. The user has to be aware 
 * that the length of the received mbufs may be smaller than the length
 * of the supplied mbufs. 
 * The mbuf header contains the following modified field
 * @li IX_MBUF_PKT_LEN is set in the header of the first mbuf and indicates
 *  the total frame size
 * @li IX_MBUF_LEN is set each mbuf header and indicates the payload length
 * @li IX_MBUF_NEXT_BUFFER_IN_PKT_PTR contains a pointer to the next 
 *     mbuf, or NULL at the end of a chain.
 * @li IX_MBUF_NEXT_PKT_IN_CHAIN_PTR is modified. Its value is reset to NULL
 * @li IX_MBUF_FLAGS contains the bit 4 set for a broadcast packet and the bit 5
 *     set for a multicast packet. Other bits are unmodified.
 *
 * @warning Any portID returned via the received callback is invalid if
 * greater than 5. If your system defines more than 6 ports (0 to 5) in
 * IxEthDBPortDefs.h you MUST check this value before using it
 * in conjunction with the Ethernet Learning/Filtering Database.
 * This is a constraint of an NPE Rx descriptor format
 * limitation, which can report valid ports only between 0 and 5, 
 * and will use 6 for "reserved" and "7" for "not found", while
 * the XScale Ethernet Database can use these numbers for legitimate
 * ports.
 *
 * <hr>
 */

typedef void (*IxEthAccPortRxCallback) (UINT32 callbackTag, IX_MBUF *buffer, IxEthAccPortId portId);




/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortRxCallbackRegister( IxEthAccPortId portId, IxEthAccPortRxCallback rxCallbackFn, UINT32 callbackTag)
 *
 * @brief The function registered through this function shall be called once per received Ethernet frame.
 *
 *
 * This function will dispatch a predefined number of frames to the user level
 * via the provided function. The invocation shall be made for each
 * frame dequeued from the Ethernet QM queue. The user is required to free any MBUF's 
 * supplied via this callback. In addition the registered callback must free up MBUF's
 * from the recieve free queue when the port is disabled 
 * 
 * If called several times the latest callback shall be registered for a particular port.
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 *
 * @param portId - Register callback for a particular MAC device.
 * @param rxCallbackFn - @a IxEthAccRxCallbackFn - Function to be called when Ethernet frames are availble.
 * @param callbackTag -  This tag shall be provided to the callback function.
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 * - IX_ETH_ACC_INVALID_ARG : An argument other than portId is invalid.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortRxCallbackRegister( IxEthAccPortId portId, IxEthAccPortRxCallback rxCallbackFn, UINT32 callbackTag);


/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortRxFreeReplenish( IxEthAccPortId portId, IX_MBUF *buffer)
 *
 * @brief This function provides buffers for the Ethernet receive path. 
 *
 * This component does not have a buffer management mechanisms built in. All Rx buffers must be supplied to it
 * via this interface. 
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 * @param portId - Provide buffers only to specific Rx MAC. 
 * @param buffer - Provide an MBUF to the Ethernet receive mechanism. 
 *                 Buffers size smaller than IX_ETHACC_RX_MBUF_MIN_SIZE may result in poor
 *                 performances and excessive buffer chaining. Buffers
 *                 larger than this size may be suitable for jumbo frames.
 *                 Chained packets are not supported and the field IX_MBUF_NEXT_PKT_IN_CHAIN_PTR must be NULL. 
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : Buffer has was not able to queue the 
 *                     buffer in the receive service.
 * - IX_ETH_ACC_FAIL : Buffer size is less than IX_ETHACC_RX_MBUF_MIN_SIZE
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * @note
 * If the buffer replenish operation fails it is the responsibility 
 * of the user to free the buffer.
 *
 * @note
 * Sufficient buffers must be supplied to the component to maintain
 * receive throughput and avoid rx buffer underflow conditions.
 * To meet this goal, It is expected that the user preload the 
 * component with a sufficent number of buffers prior to enabling the
 * NPE Ethernet receive path. The recommended minimum number of 
 * buffers is 8.
 *
 * @note
 * For maximum performances, the mbuf size should be greater 
 * than the maximum frame size (Ethernet header, payload and FCS) + 64. 
 * Supplying smaller mbufs to the service results in mbuf
 * chaining and degraded performances. The recommended size
 * is @a IX_ETHACC_RX_MBUF_MIN_SIZE, which is
 * enough to take care of 802.3 frames and "baby jumbo" frames without
 * chaining, and "jumbo" frame within chaining.
 *
 * @note
 * Buffers may not be filled up to their length. The firware will fill
 * them up to the previous 64 bytes boundary. The user has to be aware 
 * that the length of the received mbufs may be smaller than the length
 * of the supplied mbufs.
 *
 * @warning This function will only check the parameters if the NDEBUG 
 * flag is not defined. Turning on the argument checking (disabled by 
 * default) will result in a lower EthAcc performance as this function
 * is part of the data path.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortRxFreeReplenish( IxEthAccPortId portId, IX_MBUF *buffer);



/***************************************************************

  ####    ####   #    #   #####  #####    ####   #
 #    #  #    #  ##   #     #    #    #  #    #  #
 #       #    #  # #  #     #    #    #  #    #  #
 #       #    #  #  # #     #    #####   #    #  #
 #    #  #    #  #   ##     #    #   #   #    #  #
  ####    ####   #    #     #    #    #   ####   ######


         #####   #         ##    #    #  ######
         #    #  #        #  #   ##   #  #
         #    #  #       #    #  # #  #  #####
         #####   #       ######  #  # #  #
         #       #       #    #  #   ##  #
         #       ######  #    #  #    #  ######

***************************************************************/

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortEnable(IxEthAccPortId portId)
 *
 * @brief Enable a port
 * 
 * This enables an Ethernet port for both Tx and Rx. 
 *
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 * @pre The port must first be initialized via @a ixEthAccPortInit and the MAC address 
 * must be set using @a ixEthAccUnicastMacAddressSet before enabling it
 * The rx and Tx Done callbacks registration via @a
 * ixEthAccPortTxDoneCallbackRegister amd @a  ixEthAccPortRxCallbackRegister
 * has to be done before enabling the traffic.
 * 
 * @param  portId : @a IxEthAccPortId : Port id to act upon.
 * 
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is not initialized
 * - IX_ETH_ACC_MAC_UNINITIALIZED : port MAC address is not initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortEnable(IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortDisable(IxEthAccPortId portId)
 *
 * @brief Disable a port
 * 
 * This disables an Ethernet port for both Tx and Rx. 
 * Free MBufs are returned to the user via the registered callback when the port is disabled 
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 * @pre The port must be enabled with @a ixEthAccPortEnable, otherwise this
 * function has no effect
 *
 * @param  portId : @a IxEthAccPortId : Port id to act upon.
 * 
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is not initialized
 * - IX_ETH_ACC_MAC_UNINITIALIZED : port MAC address is not initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortDisable(IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortEnabledQuery(IxEthAccPortId portId, BOOL *enabled)
 *
 * @brief Get the enabled state of a port.
 *
 * Return the enabled state of the port.
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 * @pre The port must first be initialized via @a ixEthAccPortInit
 *
 * @param  portId : @a IxEthAccPortId : Port id to act upon.
 * @param  enabled : BOOL : location to store the state of the port
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortEnabledQuery(IxEthAccPortId portId, BOOL *enabled);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortPromiscuousModeClear(IxEthAccPortId portId)
 *
 * @brief Put the Ethernet MAC device in non-promiscuous mode.
 * 
 * In non-promiscuous mode the MAC will filter all frames other than 
 * destination MAC address which matches the following criteria:
 * - Unicast address provisioned via @a ixEthAccUnicastMacAddressSet
 * - All broadcast frames.
 * - Multicast addresses provisioned via @a ixEthAccMulticastAddressJoin
 *
 * See also: @a ixEthAccPortPromiscuousModeSet
 * 
 * Other functions modify the MAC filtering
 *
 * @li @a ixEthAccPortMulticastAddressJoinAll() - all multicast
 *     frames are forwarded to the application
 * @li @a ixEthAccPortMulticastAddressLeaveAll() - rollback the
 *     effects of @a ixEthAccPortMulticastAddressJoinAll()
 * @li @a ixEthAccPortMulticastAddressLeave() - unprovision a new 
 *     filtering address
 * @li @a ixEthAccPortMulticastAddressJoin() - provision a new 
 *     filtering address
 * @li @a ixEthAccPortPromiscuousModeSet() - all frames are 
 *     forwarded to the application regardless of the multicast 
 *     address provisioned
 * @li @a ixEthAccPortPromiscuousModeClear() - frames are forwarded 
 *     to the application following the multicast address provisioned
 *
 * In all cases, unicast and broadcast addresses are forwarded to 
 * the application.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 * 
 * @param portId - Ethernet port id.
 * 
 * @return IxEthAccStatus 
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortPromiscuousModeClear(IxEthAccPortId portId);


/**
 * @ingroup IxEthAcc
 *
 * @fn  ixEthAccPortPromiscuousModeSet(IxEthAccPortId portId)
 *
 * @brief Put the MAC device in promiscuous mode.
 * 
 * If the device is in promiscuous mode then all all received frames shall be forwared
 * to the NPE for processing.
 *
 * See also: @a ixEthAccPortPromiscuousModeClear
 *
 * Other functions modify the MAC filtering
 *
 * @li @a ixEthAccPortMulticastAddressJoinAll() - all multicast
 *     frames are forwarded to the application
 * @li @a ixEthAccPortMulticastAddressLeaveAll() - rollback the
 *     effects of @a ixEthAccPortMulticastAddressJoinAll()
 * @li @a ixEthAccPortMulticastAddressLeave() - unprovision a new 
 *     filtering address
 * @li @a ixEthAccPortMulticastAddressJoin() - provision a new 
 *     filtering address
 * @li @a ixEthAccPortPromiscuousModeSet() - all frames are 
 *     forwarded to the application regardless of the multicast 
 *     address provisioned
 * @li @a ixEthAccPortPromiscuousModeClear() - frames are forwarded 
 *     to the application following the multicast address provisioned
 *
 * In all cases, unicast and broadcast addresses are forwarded to 
 * the application.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 * 
 * @param portId - Ethernet port id.
 * 
 * @return IxEthAccStatus 
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus  ixEthAccPortPromiscuousModeSet(IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortUnicastMacAddressSet(      IxEthAccPortId portId,
                                                  IxEthAccMacAddr *macAddr)
 *
 * @brief Configure unicast MAC address for a particular port
 *
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId - Ethernet port id.
 * @param *macAddr - Ethernet Mac address.
 *
 * @return IxEthAccStatus 
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortUnicastMacAddressSet(      IxEthAccPortId portId,
                                                  IxEthAccMacAddr *macAddr);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortUnicastMacAddressGet(	IxEthAccPortId portId, 
					IxEthAccMacAddr *macAddr)
 *
 * @brief Get unicast MAC address for a particular MAC port 
 *
 * @pre
 * The MAC address must first be set via @a ixEthAccMacPromiscuousModeSet
 * If the MAC address has not been set, the function will return a 
 * IX_ETH_ACC_MAC_UNINITIALIZED status
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId - Ethernet port id.
 * @param *macAddr - Ethernet MAC address.
 *
 * @return  IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_MAC_UNINITIALIZED : port MAC address is not initialized.
 * - IX_ETH_ACC_FAIL : macAddr is invalid.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortUnicastMacAddressGet(	IxEthAccPortId portId, 
					IxEthAccMacAddr *macAddr);




/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortMulticastAddressJoin(      IxEthAccPortId portId,
                                             IxEthAccMacAddr *macAddr)
 *
 * @brief ADD a multicast address to the MAC address table.
 *
 *  ADD a multicast address to the MAC address table.
 *  @note
 *  Due to the operation of the Ethernet MAC multicast filtering mechanism, frames which do not 
 *  have a multicast destination address which were provisioned via this API may be forwarded 
 *  to the NPE's. This is a result of the hardware comparison  algorithm used in the destination mac address logic
 *  within the Ethernet MAC. 
 *
 *  See Also: IXP425 hardware development manual.
 * 
 * Other functions modify the MAC filtering
 *
 * @li @a ixEthAccPortMulticastAddressJoinAll() - all multicast
 *     frames are forwarded to the application
 * @li @a ixEthAccPortMulticastAddressLeaveAll() - rollback the
 *     effects of @a ixEthAccPortMulticastAddressJoinAll()
 * @li @a ixEthAccPortMulticastAddressLeave() - unprovision a new 
 *     filtering address
 * @li @a ixEthAccPortMulticastAddressJoin() - provision a new 
 *     filtering address
 * @li @a ixEthAccPortPromiscuousModeSet() - all frames are 
 *     forwarded to the application regardless of the multicast 
 *     address provisioned
 * @li @a ixEthAccPortPromiscuousModeClear() - frames are forwarded 
 *     to the application following the multicast address provisioned
 *
 * In all cases, unicast and broadcast addresses are forwarded to 
 * the application.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @param portId - Ethernet port id.
 * @param *macAddr - Ethernet Mac address.
 *
 * @return IxEthAccStatus 
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : Error writing to the MAC registers
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortMulticastAddressJoin(      IxEthAccPortId portId,
                                             IxEthAccMacAddr *macAddr);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortMulticastAddressJoinAll(  IxEthAccPortId portId)
 *
 * @brief Filter all frames with multicast dest.
 *
 * This function clears the MAC address table, and then sets
 * the MAC to forward ALL multicast frames to the NPE.
 * Specifically, it forwards all frames whose destination address
 * has the LSB of the highest byte set  (01:00:00:00:00:00).  This
 * bit is commonly referred to as the "multicast bit".
 * Broadcast frames will still be forwarded. 
 *
 * Other functions modify the MAC filtering
 *
 * @li @a ixEthAccPortMulticastAddressJoinAll() - all multicast
 *     frames are forwarded to the application
 * @li @a ixEthAccPortMulticastAddressLeaveAll() - rollback the
 *     effects of @a ixEthAccPortMulticastAddressJoinAll()
 * @li @a ixEthAccPortMulticastAddressLeave() - unprovision a new 
 *     filtering address
 * @li @a ixEthAccPortMulticastAddressJoin() - provision a new 
 *     filtering address
 * @li @a ixEthAccPortPromiscuousModeSet() - all frames are 
 *     forwarded to the application regardless of the multicast 
 *     address provisioned
 * @li @a ixEthAccPortPromiscuousModeClear() - frames are forwarded 
 *     to the application following the multicast address provisioned
 *
 * In all cases, unicast and broadcast addresses are forwarded to 
 * the application.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @param portId - Ethernet port id.
 *
 * @return IxEthAccStatus 
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortMulticastAddressJoinAll( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortMulticastAddressLeave( IxEthAccPortId portId,
                                         IxEthAccMacAddr *macAddr)
 *
 * @brief Remove a multicast address from the MAC address table.
 *
 *  Remove a multicast address from the MAC address table.
 *
 * Other functions modify the MAC filtering
 *
 * @li @a ixEthAccPortMulticastAddressJoinAll() - all multicast
 *     frames are forwarded to the application
 * @li @a ixEthAccPortMulticastAddressLeaveAll() - rollback the
 *     effects of @a ixEthAccPortMulticastAddressJoinAll()
 * @li @a ixEthAccPortMulticastAddressLeave() - unprovision a new 
 *     filtering address
 * @li @a ixEthAccPortMulticastAddressJoin() - provision a new 
 *     filtering address
 * @li @a ixEthAccPortPromiscuousModeSet() - all frames are 
 *     forwarded to the application regardless of the multicast 
 *     address provisioned
 * @li @a ixEthAccPortPromiscuousModeClear() - frames are forwarded 
 *     to the application following the multicast address provisioned
 *
 * In all cases, unicast and broadcast addresses are forwarded to 
 * the application.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @param portId - Ethernet port id.
 * @param *macAddr - Ethernet Mac address.
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_NO_SUCH_ADDR :  Failed if MAC address was not in the table.
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortMulticastAddressLeave( IxEthAccPortId portId,
                                         IxEthAccMacAddr *macAddr);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortMulticastAddressLeaveAll( IxEthAccPortId portId)
 *
 * @brief Clear the MAC address table.
 *
 * This function clears the MAC address table, and then sets
 * the MAC as configured by the promiscuous mode current settings.
 *
 * Other functions modify the MAC filtering
 *
 * @li @a ixEthAccPortMulticastAddressJoinAll() - all multicast
 *     frames are forwarded to the application
 * @li @a ixEthAccPortMulticastAddressLeaveAll() - rollback the
 *     effects of @a ixEthAccPortMulticastAddressJoinAll()
 * @li @a ixEthAccPortMulticastAddressLeave() - unprovision a new 
 *     filtering address
 * @li @a ixEthAccPortMulticastAddressJoin() - provision a new 
 *     filtering address
 * @li @a ixEthAccPortPromiscuousModeSet() - all frames are 
 *     forwarded to the application regardless of the multicast 
 *     address provisioned
 * @li @a ixEthAccPortPromiscuousModeClear() - frames are forwarded 
 *     to the application following the multicast address provisioned
 *
 * In all cases, unicast and broadcast addresses are forwarded to 
 * the application.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @param portId - Ethernet port id.
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortMulticastAddressLeaveAll( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortUnicastAddressShow(IxEthAccPortId portId)
 *
 * @brief Display unicast address has  been configured using 
 * @a ixEthAccUnicastMacAddressSet
 *
 * Display unicast address has  been configured using 
 * @a ixEthAccUnicastMacAddressSet. Display also the MAC filter used
 *
 * Other functions modify the MAC filtering
 *
 * @li @a ixEthAccPortMulticastAddressJoinAll() - all multicast
 *     frames are forwarded to the application
 * @li @a ixEthAccPortMulticastAddressLeaveAll() - rollback the
 *     effects of @a ixEthAccPortMulticastAddressJoinAll()
 * @li @a ixEthAccPortMulticastAddressLeave() - unprovision a new 
 *     filtering address
 * @li @a ixEthAccPortMulticastAddressJoin() - provision a new 
 *     filtering address
 * @li @a ixEthAccPortPromiscuousModeSet() - all frames are 
 *     forwarded to the application regardless of the multicast 
 *     address provisioned
 * @li @a ixEthAccPortPromiscuousModeClear() - frames are forwarded 
 *     to the application following the multicast address provisioned
 *
 * In all cases, unicast and broadcast addresses are forwarded to 
 * the application.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId - Ethernet port id.
 *
 * @return void
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortUnicastAddressShow(IxEthAccPortId portId);


/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortMulticastAddressShow( IxEthAccPortId portId)
 *
 * @brief Display multicast address which have been configured using @a ixEthAccMulticastAddressJoin
 * Display multicast address which have been configured using @a ixEthAccMulticastAddressJoin
 * 
 * - Reentrant    - yes
 * - ISR Callable - no
 * 
 * @param portId - Ethernet port id.
 *
 * @return void
 *
 * <hr>
 */
void ixEthAccPortMulticastAddressShow( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortDuplexModeSet( IxEthAccPortId portId, IxEthAccDuplexMode mode )
 *
 * @brief  Set the duplex mode for the MAC.
 *
 * Configure the IXP425 MAC to either full or half duplex. 
 *
 * @note 
 * The configuration should match that provisioned on the PHY.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId
 * @param mode : @a IxEthAccDuplexMode
 *
 * @return IxEthAccStatus 
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortDuplexModeSet( IxEthAccPortId portId, IxEthAccDuplexMode mode );

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortDuplexModeGet( IxEthAccPortId portId, IxEthAccDuplexMode *mode )
 *
 * @brief  Get the duplex mode for the MAC.
 *
 * return the duplex configuration of the IXP425 MAC.
 *
 * @note
 * The configuration should match that provisioned on the PHY.
 * See @a ixEthAccDuplexModeSet
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId
 * @param *mode : @a IxEthAccDuplexMode 
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 *
 */
IxEthAccStatus ixEthAccPortDuplexModeGet( IxEthAccPortId portId, IxEthAccDuplexMode *mode );





/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortTxFrameAppendPaddingEnable( IxEthAccPortId portId)
 *
 * @brief  Enable the appending of padding bytes to runt frames submitted to
 * this port
 * 
 * Enable the appending of up to 60 null-bytes to runt frames submitted 
 * to this port. This is the default behavior of the access component.
 *
 * @warning Do not change this behaviour while the port is enabled.
 *
 * @note When Tx padding is enabled, Tx FCS generation is turned on
 *
 * @sa ixEthAccPortTxFrameAppendFCSDusable
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId 
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortTxFrameAppendPaddingEnable( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortTxFrameAppendPaddingDisable( IxEthAccPortId portId)
 *
 * @brief  Disable the appending of padding bytes to the runt
 *         frames submitted to this port.
 * 
 * Disable the appending of padding bytes to runt frames submitted to 
 * this port. This is not the default behavior of the access component.
 *
 * @warning Do not change this behaviour while the port is enabled.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId 
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortTxFrameAppendPaddingDisable( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortTxFrameAppendFCSEnable( IxEthAccPortId portId)
 *
 * @brief  Enable the appending of Ethernet FCS to all frames submitted to this port
 * 
 * Enable the appending of Ethernet FCS to all frames submitted to this port. This is the default 
 * behavior of the access component.
 * Do not change this behaviour while the port is enabled.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId 
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortTxFrameAppendFCSEnable( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortTxFrameAppendFCSDisable( IxEthAccPortId portId)
 *
 * @brief  Disable the appending of Ethernet FCS to all frames submitted to this port.
 * 
 * Disable the appending of Ethernet FCS to all frames submitted to this port. This is not the default
 * behavior of the access component.
 * Note: Since the FCS is not appended to the frame it is expected that the frame submitted to the 
 * component includes a valid FCS at the end of the data, although this will not be validated.
 * The component shall forward the frame to the Ethernet MAC WITHOUT modification.
 * Do not change this behaviour while the port is enabled.
 *
 * @note Tx FCS append is not disabled while Tx padding is enabled.
 *
 * @sa ixEthAccPortTxFrameAppendPaddingEnable
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId 
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortTxFrameAppendFCSDisable( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortRxFrameAppendFCSEnable( IxEthAccPortId portId)
 *
 * @brief Forward frames with FCS included in the receive buffer to the user.
 *
 * Enable the appending of Ethernet FCS to all frames submitted to this port. This is the default
 * behavior of the access component.
 * The Frame length recieved will include the FCS. ie. A minimum sized ethernet frame shall have a 
 * length of 64bytes.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortRxFrameAppendFCSEnable( IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccPortRxFrameAppendFCSDisable( IxEthAccPortId portId)
 *
 * @brief  Disable the appending of Ethernet FCS to all frames submitted to this port.
 *
 *
 * Do not forward the FCS portion of the received Ethernet frame to the user. The FCS is striped
 * from the receive buffer. Frame FCS validity checks are still carried out on all received frames.
 * This is the default behavior of the component.
 * Do not change this behaviour while the port is enabled.
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @param portId : @a IxEthAccPortId
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS 
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccPortRxFrameAppendFCSDisable( IxEthAccPortId portId);




/**
 * @ingroup IxEthAcc
 *
 * @enum IxEthAccTxSchedulerDiscipline
 *
 * @brief  Definition for the port transmit scheduling discipline
 * Definition for the port transmit scheduling discipline
 * 
 *  Select the port transmit scheduling discipline
 * - FIFO : No Priority : In this configuration all frames submitted to the access component 
 *                        shall be sub-mitted to the MAC hardware in the strict order in which 
 *                        it was received.
 * - FIFO : Priority : This shall be a very simple priority mechanism all submitted frames at a 
 *                     higher prior-ity shall be forwarded to Ethernet MAC for transmission before 
 *                     lower priorities. There shall be no fairness mechanisms applied across 
 *                     different priorities. Higher priority frames could starve lower priority 
 *                     frames indefinitely.
 */
typedef  enum 
{
    FIFO_NO_PRIORITY, /**<frames submitted with no priority*/
    FIFO_PRIORITY /**<higher prority frames submitted before lower priority*/
}IxEthAccTxSchedulerDiscipline;


/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccTxSchedulingDisciplineSet( IxEthAccPortId portId, IxEthAccTxSchedulerDiscipline sched)
 *
 * @brief Set the port scheduling to one of @a IxEthAccTxSchedulerDiscipline
 * Set the port scheduling to one of @a IxEthAccTxSchedulerDiscipline
 *
 * The default behavior of the component is @a FIFO_NO_PRIORITY.
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 * @pre
 *
 *
 * @param portId : @a IxEthAccPortId 
 * @param sched : @a IxEthAccTxSchedulerDiscipline 
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS : Set appropriate discipline.
 * - IX_ETH_ACC_FAIL :  Invalid/unsupported discipline.
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccTxSchedulingDisciplineSet( IxEthAccPortId portId, IxEthAccTxSchedulerDiscipline sched);


  
/*********************************************************************************
  ####    #####    ##     #####     #     ####    #####     #     ####    ####
 #          #     #  #      #       #    #          #       #    #    #  #
  ####      #    #    #     #       #     ####      #       #    #        ####
      #     #    ######     #       #         #     #       #    #            #
 #    #     #    #    #     #       #    #    #     #       #    #    #  #    #
  ####      #    #    #     #       #     ####      #       #     ####    ####
**********************************************************************************/


/**
 *
 * @brief This struct defines the statistics returned by this component.
 * The component returns MIB2 EthObj variables which should are obtained from the 
 * hardware or maintained by this component. 
 *
 *
 */
typedef struct   
{
    UINT32 dot3StatsAlignmentErrors;            /**< link error count */
    UINT32 dot3StatsFCSErrors;                  /**< link error count */
    UINT32 dot3StatsFrameTooLongs;              /**< link error count */
    UINT32 dot3StatsInternalMacReceiveErrors;   /**< link error count */
    UINT32 LearnedEntryDiscards;                /**< NPE  error count */
    UINT32 UnderflowEntryDiscards;              /**< NPE  error count */
    UINT32 dot3StatsSingleCollisionFrames;      /**< link error count */
    UINT32 dot3StatsMultipleCollisionFrames;    /**< link error count */
    UINT32 dot3StatsDeferredTransmissions;      /**< link error count */
    UINT32 dot3StatsLateCollisions;             /**< link error count */
    UINT32 dot3StatsExcessiveCollsions;         /**< link error count */
    UINT32 dot3StatsInternalMacTransmitErrors;  /**< link error count */
    UINT32 dot3StatsCarrierSenseErrors;         /**< link error count */
}IxEthEthObjStats;

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccMibIIStatsGet(IxEthAccPortId portId ,IxEthEthObjStats *retStats )
 *
 * @brief  Return the statistics maintained for a port.
 *  Return the statistics maintained for a port.
 *
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @pre
 *
 *
 * @param portId @a IxEthAccPortId 
 * @param retStats @a IxEthEthObjStats
 * @note Please note the user is responsible for cache coheriency of the retStat
 * buffer. The data is actually populated via the NPE's. As such cache safe
 * memory should be used in the retStats argument.
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : Invalid arguments.
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccMibIIStatsGet(IxEthAccPortId portId ,IxEthEthObjStats *retStats );

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccMibIIStatsGetClear(IxEthAccPortId portId, IxEthEthObjStats *retStats)
 * 
 * @brief  Return and clear the statistics maintained for a port. 
 *  Return and clear the statistics maintained for a port. 
 *
 *
 * - Reentrant    - yes
 * - ISR Callable - yes
 *
 * @pre
 *
 * @param portId @a IxEthAccPortId  
 * @param retStats @a IxEthEthObjStats
 * @note Please note the user is responsible for cache coheriency of the retStats
 * buffer. The data is actually populated via the NPE's. As such cache safe
 * memory should be used in the retStats argument.
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : invalid arguments.
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccMibIIStatsGetClear(IxEthAccPortId portId, IxEthEthObjStats *retStats);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccMibIIStatsClear(IxEthAccPortId portId)
 *
 *
 * @brief   Clear the statistics maintained for a port.
 * Clear the statistics maintained for a port.
 *
 *
 * - Reentrant    - yes
 * - ISR Callable - no
 *
 * @pre
 *
 * @param portId @a IxEthAccPortId
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : Invalid arguments.
 * - IX_ETH_ACC_INVALID_PORT : portId is invalid.
 * - IX_ETH_ACC_PORT_UNINITIALIZED : portId is un-initialized
 *
 * <hr>
 */
IxEthAccStatus ixEthAccMibIIStatsClear(IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccMacInit(IxEthAccPortId portId)
 * 
 * @brief Initialize the ethernet MAC settings 
 * 
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @param portId @a IxEthAccPortId
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_INVALID_PORT	:  portId is invalid.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccMacInit(IxEthAccPortId portId);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccStatsShow(IxEthAccPortId portId)
 *
 *
 * @brief Display a ports statistics on the standard io console using printf.
 * Display a ports statistics on the standard io console using printf.
 *
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @pre
 *
 * @param portId @a IxEthAccPortId
 *
 * @return void
 *
 * <hr>
 */
void ixEthAccStatsShow(IxEthAccPortId portId);

/*************************************************************************

 #    #     #       #            #    #  #####      #     ####
 ##  ##     #       #            ##  ##  #    #     #    #    #
 # ## #     #       #            # ## #  #    #     #    #    #
 #    #     #       #            #    #  #    #     #    #    #
 #    #     #       #            #    #  #    #     #    #    #
 #    #     #       #            #    #  #####      #     ####

*************************************************************************/


/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccMiiReadRtn (UINT8 phyAddr, 
                           UINT8 phyReg, 
	                   UINT16 *value)
 *
 *
 * @brief Read a 16 bit value from a PHY
 *
 * Read a 16-bit word from a register of a MII-compliant PHY. Reading
 * is performed through the MII management interface.  This function returns
 * when the read has successfully completed, or when a timeout has elapsed.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @pre The MAC on Ethernet Port 2 (NPE C) must be initialised, and generating the MDIO clock.
 *   
 * @param phyAddr: the address of the Ethernet PHY (0-31)
 * @param phyReg:  the number of the MII register to read (0-31)
 * @param value:   the value read from the register
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : failed to read register.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccMiiReadRtn (UINT8 phyAddr, 
				   UINT8 phyReg, 
				   UINT16 *value);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccMiiWriteRtn (UINT8 phyAddr, 
                            UINT8 phyReg, 
	                    UINT16 value)
 *
 *
 * @brief Write a 16 bit value to a PHY
 *
 * Write a 16-bit word from a register of a MII-compliant PHY. Writing
 * is performed through the MII management interface.  This function returns
 * when the write has successfully completed, or when a timeout has elapsed.
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @pre The MAC on Ethernet Port 2 (NPE C) must be initialised, and generating the MDIO clock.
 *   
 * @param phyAddr: the address of the Ethernet PHY (0-31)
 * @param phyReg:  the number of the MII register to write (0-31)
 * @param value:   the value to write to the register
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : failed to write register.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccMiiWriteRtn (UINT8 phyAddr, 
				    UINT8 phyReg, 
				    UINT16 value);

/**
 * @ingroup IxEthAcc
 *
 * @fn ixEthAccMiiStatsShow (UINT32 phyAddr)
 *
 *
 * @brief Display detailed information on a specified PHY
 *
 * Display the current values of the first eigth MII registers for a PHY, 
 *
 * - Reentrant    - no
 * - ISR Callable - no
 *
 * @pre The MAC on Ethernet Port 2 (NPE C) must be initialised, and 
 *      generating the MDIO clock.
 *   
 * @param phyAddr: the address of the Ethernet PHY (0-31)
 *
 * @return IxEthAccStatus
 * - IX_ETH_ACC_SUCCESS
 * - IX_ETH_ACC_FAIL : invalid arguments.
 *
 * <hr>
 */
IxEthAccStatus ixEthAccMiiStatsShow (UINT32 phyAddr);



/******* BOARD SPECIFIC DEPRECATED API *********/

/* The following functions are high level functions which rely
 * on the properties and interface of some Ethernet PHYs. The
 * implementation is hardware specific and has been moved to 
 * the hardware-specific component IxEthMii.
 */

 #include "IxEthMii.h"

/**
 * @ingroup IxEthAcc
 *
 * @def  ixEthAccMiiPhyScan
 *
 * @brief : deprecated API entry point. This definition 
 * ensures backward compatibility
 *
 * See @ref ixEthMiiPhyScan
 *
 * @note this feature is board specific
 *
 */
#define ixEthAccMiiPhyScan(phyPresent) ixEthMiiPhyScan(phyPresent,IXP425_ETH_ACC_MII_MAX_ADDR)

/**
 * @ingroup IxEthAcc
 *
 * @def ixEthAccMiiPhyConfig
 *
 * @brief : deprecated API entry point. This definition 
 * ensures backward compatibility
 *
 * See @ref ixEthMiiPhyConfig
 *
 * @note this feature is board specific
 */
#define ixEthAccMiiPhyConfig(phyAddr,speed100,fullDuplex,autonegotiate) \
           ixEthMiiPhyConfig(phyAddr,speed100,fullDuplex,autonegotiate)

/**
 * @ingroup IxEthAcc
 *
 * @def ixEthAccMiiPhyReset
 *
 * @brief : deprecated API entry point. This definition 
 * ensures backward compatibility
 *
 * See @ref ixEthMiiPhyReset
 *
 * @note this feature is board specific
 */
#define ixEthAccMiiPhyReset(phyAddr) \
           ixEthMiiPhyReset(phyAddr)

/**
 * @ingroup IxEthAcc
 *
 * @def ixEthAccMiiLinkStatus
 *
 * @brief : deprecated API entry point. This definition 
 * ensures backward compatibility
 *
 * See @ref ixEthMiiLinkStatus
 *
 * @note this feature is board specific
 */
#define ixEthAccMiiLinkStatus(phyAddr,linkUp,speed100,fullDuplex,autoneg) \
           ixEthMiiLinkStatus(phyAddr,linkUp,speed100,fullDuplex,autoneg)



/**
 * @ingroup IxEthAcc
 *
 * @def ixEthAccMiiShow  
 *
 * @brief : deprecated API entry point. This definition 
 * ensures backward compatibility
 *
 * See @ref ixEthMiiPhyShow
 *
 * @note this feature is board specific
 */
#define ixEthAccMiiShow(phyAddr) \
        ixEthMiiPhyShow(phyAddr)

#endif /* ndef IxEthAcc_H */
/**
 *@}
 */
