/*
 *  Boa, an http server, transfer.c implements messaging to daemons.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "boa.h"
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fdipc.h>

/*
 * Since the iov buffers sent are received as one buffer historically it is
 * necessary to include the terminating null character for each iov buffer.
 * Three iov buffers are required for MIME type transfers, two for URI type
 * transfers: <URI><\0>[<method:><int><\0>]<query><\0>
 */
#define INCLUDE_NULL_CHARACTER 1
#define N_IOV 3

/*
 * Send attempts are due to the possibility that the receiving end restarted,
 * thus the cached file descriptor may be invalid and a new open is attempted.
 * The size of the query buffer is historic and kept just in case. The size of
 * the POST_BUFFER_SIZE could be increased since the files sent are on the
 * order of MByte.
 */
enum {
	SEND_ATTEMPTS = 1,
	QUERY_BUFFER_SIZE = 20480,
	POST_BUFFER_SIZE = 16384
};

/* Error messages. */
static char out_of_memory[] = "Out of memory.";
static char qbuf_overflow[] = "Too much POST data: query buffer overflow.";

/*
 * The transfer list stores each registered:
 *   TransferCGI <path>
 *   Transfer <URI> <socket path>
 *   TransferMime <MIME type> <socket path>
 *   TransferPost <POST_URI> <socket path>
 *
 * key_name: <path>
 * type:     TYPE_CGI
 * Executable with un-buffered stdin and stdout which accepts options
 * and arguments by e.g. ?-i+17+-s+string+argument
 *
 * key_name: <URI> or <MIME type>
 * path:     <socket path>
 * fd:       cached open file descriptor connected to <socket path>
 * type:     TYPE_URI or TYPE_MIME
 *
 * key_name: <POST_URI>
 * path:     <socket path>
 * fd:       cached open file descriptor connected to <socket path>
 *           transferred before the body of a POST request is read
 * type:     TYPE_URI or TYPE_MIME
*/
static const unsigned char TYPE_POST = '3';
static const unsigned char TYPE_CGI = '2';
static const unsigned char TYPE_URI = '1';
static const unsigned char TYPE_MIME = '0';

typedef struct _transfer_key transfer_key;

struct _transfer_key {
	char *key_name;
	char *path;
	int fd;
	unsigned char type;
	transfer_key *next;
};

static transfer_key *transfer_list = NULL;

/*
 * Environment variables in the query buffer.
 */
static const char q_user_agent[] = "&http_user_agent=";
static const char q_user[] = "&http_user=";
static const char q_remote_addr[] = "&http_remote_addr=";
static const char q_remote_port[] = "&http_remote_port=";
static const char q_referer[] = "&http_referer=";
static const char q_cookie[] = "&http_cookie=";
static const char q_realm[] = "&http_realm=";
static const char q_access_realms[] = "&http_user_realms=";
static const char q_content_type[] = "&content_type=";
static const char q_x_sessioncookie[] = "&http_x_sessioncookie=";
static const char q_method[] = "&method=";

void add_transfer_uri(char *uri_path, char *socket_path);
void add_transfer_post_uri(char *uri_path, char *socket_path);
void add_transfer_mime(char *mime_type, char *socket_path);
void add_transfer_cgi(char *path);
void transfer_closedown(void);
int is_transfer_uri(const request* req);
int is_transfer_post_uri(const request* req);
int is_transfer_mime(const request* req);
int is_transfer_cgi(const request* req);
int init_transfer(request* req);
int init_transfer_cgi(request* req);
static int add_transfer_type(char *key_name, char *path, unsigned char type);
static transfer_key* list_find(const char *key_name, unsigned char type);
static int is_transfer_type(const char *key_name, unsigned char type);
static char *add_str(char *query, const char *env_var, const char *value);

/* Extern hook in parser which adds Transfer <URI> <socket path> to the
   global transfer_list. */
void
add_transfer_uri(char *uri_path, char *socket_path)
{
	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "%c: [%s] to [%s]\n", TYPE_URI, uri_path,
			socket_path);
	}
	if (add_transfer_type(uri_path, socket_path, TYPE_URI) < 0) {
		_exit(EXIT_FAILURE);
	}
}

/* Extern hook in parser which adds TransferPost <URI> <socket path> to the
   global transfer_list. */
void
add_transfer_post_uri(char *uri_path, char *socket_path)
{
	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "%c: [%s] to [%s]\n", TYPE_POST, uri_path,
			socket_path);
	}
	if (add_transfer_type(uri_path, socket_path, TYPE_POST) < 0) {
		_exit(EXIT_FAILURE);
	}
}

/* Extern hook in parser which adds TransferMime <MIME type> <socket path>
   to the global transfer_list. */
void
add_transfer_mime(char *mime_type, char *socket_path)
{
	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "%c: [%s] to [%s]\n", TYPE_MIME, mime_type,
			socket_path);
	}
	if (add_transfer_type(mime_type, socket_path, TYPE_MIME) < 0) {
		_exit(EXIT_FAILURE);
	}
}

/* Extern hook in parser which adds TransferCgi <CGI path> to the
   global transfer_list. */
void
add_transfer_cgi(char *path)
{
	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "%c: [%s]\n", TYPE_CGI, path);
	}
	if (add_transfer_type(path, path, TYPE_CGI) < 0) {
		_exit(EXIT_FAILURE);
	}
}

/* Removes all items in transfer_list and returns allocated memory. */
void
transfer_closedown(void)
{
	transfer_key *item;
	transfer_key *next_item;

	item = transfer_list;
	while (item != NULL) {
		free(item->path);
		free(item->key_name);
		next_item = item->next;
		free(item);
		item = next_item;
	}
	transfer_list = NULL;
}

/* Return 1 if the URI path is in transfer_list, 0 if not. */
int
is_transfer_uri(const request* req)
{
	return is_transfer_type(req->request_uri, TYPE_URI);
}

/* Return 1 if the POST_URI path is in transfer_list, 0 if not. */
int
is_transfer_post_uri(const request* req)
{
	return is_transfer_type(req->request_uri, TYPE_POST);
}

/* Return 1 if the MIME type is in transfer_list, 0 if not. */
int
is_transfer_mime(const request* req)
{
	return is_transfer_type(get_mime_type(req->request_uri), TYPE_MIME);
}

/* Return 1 if the CGI path is in transfer_list, 0 if not. */
int
is_transfer_cgi(const request* req)
{
	return is_transfer_type(req->request_uri, TYPE_CGI);
}

/* Propagate transfer request req to daemon defined. Return 1 on success
   which is interpreted as success, continue, -1 on failure. */
int
init_transfer(request* req)
{
	struct iovec iov[N_IOV];
	int n_iov;
	int content_length;
	char *buf_start;
	int len;
	char query[QUERY_BUFFER_SIZE];
	char *qptr;
	int querysize;
	transfer_key *item;
	char *mime_type = NULL;
	char mimebuf[32];
	int send_attempt;
	char *int_repr;
	int post_without_body = 0;
	char* ref_uri = NULL;
	// added by rey
	int iGotDecsribeInfo = 0;

	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "%s: %s\n", __FUNCTION__, req->request_uri);
	}

	SQUASH_KA(req);

	/* Search the list of configured transfers. */
	if ((item = list_find(req->request_uri, TYPE_URI)) == NULL) {
		if ((item = list_find(req->request_uri, TYPE_POST)) == NULL) {
			mime_type = get_mime_type(req->request_uri);
			if ((item = list_find(mime_type, TYPE_MIME)) == NULL) {
				send_r_not_found(req);
				return -1;
			}
		} else {
			post_without_body = 1;
		}
	}
log_error_time();
fprintf(stderr, "item: key_name = %s, path = %s, fd = %d, type = %c\n", 
    item->key_name, item->path, item->fd, item->type);

	/* Create a protocol specific item to send on. Each iov contains a
	   string _including_ the terminating null character. Thus, the
	   protocol requires the receiving daemon to step through a multi
	   null character buffer: <URI><\0>[<method:><int><\0>]<query><\0> */

	/* Prepare and store the <URI> string. */
	n_iov = 0;
	iov[n_iov].iov_base = req->request_uri;
	iov[n_iov].iov_len = strlen(req->request_uri) + INCLUDE_NULL_CHARACTER;
	n_iov++;
	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "URI: [%d] [%s]\n", 
		    strlen(req->request_uri) + INCLUDE_NULL_CHARACTER, 
		    req->request_uri);
	}

	/* Prepare the <query> string including a few environment variables. */
	/* FIXME Store all environment variables. */
	query[0] = '\0';
	qptr = &query[0];

	if (req->method == M_GET && req->query_string != NULL) {
		if (strlen(req->query_string) > QUERY_BUFFER_SIZE) {
			WARN(qbuf_overflow);
			return -1;
		}
		strcat(query, req->query_string);
		qptr += strlen(req->query_string);
		strcat(query, "&");
		qptr++;
	}
	if (req->method == M_POST) {
		/* Must have content length. */
		/* comment by acer 2011/03/23
		   "content_length shouldn't be zero" 
		   because at read.c, the content will be read.*/
		if (/*post_without_body ||*/ 
		    (mime_type &&
		     !strcmp(mime_type, "application/x-rtsp-tunnelled"))) {
			/* Fix to enable the possibility to transfer the
			 * file descriptor before we read the body on a
			 * post request (TransferPost).
			 */
			content_length = 0;
                        /* Prevent free_request() from consuming any
                         * outstanding information which is supposed to be
                         * read by the transferee.
                         */
                        req->no_post_data_consume = 1;
		} else if (req->content_length) {
			content_length = boa_atoi(req->content_length);
		} else {
			send_r_bad_request(req);
			return -1;
		}

		/* Flush the cached data from excessive reading in boa. */
		buf_start = req->client_stream + req->parse_pos;
		len = req->client_stream_pos - req->parse_pos;

		DEBUG(DEBUG_TRANSFER) {
			fprintf(stderr, "POST buffered length: [%d]\n", len);
			fprintf(stderr, "POST content_length: [%d]\n",
				content_length);
        }
		if (len > 0) {
			strncat(query, buf_start, len);
			qptr += len;
		}
		/* 8 kByte is cached and the header normally uses < 1 kByte. */
		if (len < content_length) {
			ssize_t byte_read;
			size_t buf_byte;
			do {
				// added by rey : set timeout to prevent the waiting
				fd_set	readfds;
				struct timeval timeout;
				timeout.tv_sec = 3;
				timeout.tv_usec = 0;
				FD_ZERO(&readfds);
				FD_SET(req->fd, &readfds);
				if (select(req->fd + 1, &readfds, NULL, NULL, &timeout) < 0)
				{
					fprintf(stderr, "init_transfer : select error\n");
					break;
				}
				if (!FD_ISSET(req->fd, &readfds))
				{
					/* socket is blocked */
					break;
				}
				
				buf_byte = &query[QUERY_BUFFER_SIZE] - qptr;
				byte_read = read(req->fd, qptr, buf_byte);
				if (byte_read < 0) {
					if (errno == EAGAIN
					    || errno == EINTR) {
						continue;
					}
					fprintf(stderr, "read errno: %d\n", errno);
					send_r_bad_request(req);
					WARN("read");
					return -1;
				}
				else if (byte_read == 0)
					break;

				len += byte_read;
				if (len > QUERY_BUFFER_SIZE) {
					send_r_bad_request(req);
					WARN(qbuf_overflow);
					return -1;
				}
				qptr += byte_read;
				// added by rey : wait for quicktime send base64 describe info, get and break
				iGotDecsribeInfo = 1;
				break;
			} while (len < content_length);
			*qptr = '\0';
		}
		strcat(query, "&");
		qptr++;
	} // end: if (req->method == M_POST)
	if (post_without_body) {
		/* Forward Content-Type on TransferPost. */
		if (add_str(query, q_content_type, req->content_type) == NULL) {
			return -1;
		}
	}
	if (add_str(query, q_user_agent, req->header_user_agent) == NULL) {
		return -1;
	}
	if (add_str(query, q_x_sessioncookie, req->x_sessioncookie) == NULL) {
		return -1;
	}
#ifdef BASIC_AUTH
	if (add_str(query, q_user, req->user) == NULL) {
		return -1;
	}
#endif
	if (strlen(req->remote_ip_addr) > 0) {
		if (add_str(query, q_remote_addr, req->remote_ip_addr) == NULL) {
			return -1;
		}
	}
	if (req->remote_port > 0) {
		int_repr = simple_itoa((unsigned int)req->remote_port);
		if (add_str(query, q_remote_port, int_repr) == NULL) {
			return -1;
		}
	}
	/* only set uri of referer, skip query. */
	if (req->header_referer && (ref_uri = strchr(req->header_referer, '?'))) {
		*ref_uri = '\0';
	}
	if (add_str(query, q_referer, req->header_referer) == NULL) {
		return -1;
	}
	if (ref_uri) {
		*ref_uri = '?';
	}
	if (add_str(query, q_cookie, req->http_cookie) == NULL) {
		return -1;
	}
#ifdef BASIC_AUTH
	if (add_str(query, q_realm, req->realm) == NULL ) {
		return -1;
	}
	if (add_str(query, q_access_realms, req->user_realms) == NULL ) {
		return -1;
	}
#endif
	if (req->real_method == M_POST) {
		if (add_str(query, q_method, "POST") == NULL) {
			return -1;
		}
	} else if (req->real_method == M_GET) {
		if (add_str(query, q_method, "GET") == NULL) {
			return -1;
		}
	} else if (req->real_method == M_PUT) {
		if (add_str(query, q_method, "PUT") == NULL) {
			return -1;
		}
	}else if (req->real_method == M_DELETE) {
		if (add_str(query, q_method, "DELETE") == NULL) {
			return -1;
		}
	}
	//Add the HTTP query string to the query string of fidipc.  
	if(req->query_string!=NULL)
	{
		if (add_str(query, "&query_string=",req->query_string ) == NULL) {
			return -1;
		}
	}
	/* The size is sent (FIXME and ignored?) in MIME type transfers. */
	querysize = (int)strlen(query);

	/* Prepare and store a protocol specific string, <method:> and content
	   length <int> _only_ for TransferMime, not Transfer. */
	if (item->type == TYPE_MIME) {
		if (req->method == M_POST) {
			strcpy(mimebuf, "POST:");
		}
		else {
			strcpy(mimebuf, "GET:");
		}
		int_repr = simple_itoa((unsigned int)querysize);
		strcat(mimebuf, int_repr);
		/* Include the null character in the buffer. */
		iov[n_iov].iov_base = mimebuf;
		iov[n_iov].iov_len = strlen(mimebuf) + INCLUDE_NULL_CHARACTER;
		n_iov++;
		DEBUG(DEBUG_TRANSFER) {
			fprintf(stderr, "MIME: [%d] [%s]\n",
				(int)strlen(mimebuf), mimebuf);
		}
	}

	/* Store the prepared <query> string. */
	iov[n_iov].iov_base = query;
	iov[n_iov].iov_len = (size_t)querysize + INCLUDE_NULL_CHARACTER;
	n_iov++;

    /*
	 * added by rey : reply to Quicktime client "200 OK" (when rtp over http)
	 *                if rtp over http : x_sessioncookie will not be empty
	 * modiy by jesi: if rtp over http : !strcmp(mime_type, "application/x-rtsp-tunnelled")
	 */
	mime_type = get_mime_type(req->request_uri);
	if (mime_type && !strcmp(mime_type, "application/x-rtsp-tunnelled"))
	{
    	if (req->method == M_GET)
    	{
    		req_write(req, http_ver_string(req->http_version));
    		req_write(req, " 200 OK" CRLF);
    		print_content_type(req);
    		print_http_headers(req);
    		req_write(req, CRLF);
    		req_flush(req);

			// added by djhow: 2009/04/13 give RTSP server "GET" hint
			iov[n_iov].iov_base = "GET";
			iov[n_iov].iov_len = sizeof("GET");
			n_iov++;
    	}
   	    // added by rey : give RTSP server "POST" "GET" hint
    	// modiy by jesi: just recognize POST
    	else 
    	{
    		if (iGotDecsribeInfo == 1)
    		{
    			iov[n_iov].iov_base = "POST1";
    		}
    		else
    		{
    			iov[n_iov].iov_base = "POST0";
    		}
    		//iov[n_iov].iov_len = strlen("POST") + 1;
			// modified by djhow: 2009/04/13 to cooperate with new rtsp server
			iov[n_iov].iov_len = sizeof("POST") + 1;
    		n_iov++;
    	}
    }
	// added by jesi : reply to Browser "200 OK" (when server push)
    static char date_header[] = "Date: "
        "                             " CRLF;
    rfc822_time_buf(date_header + 6, 0);
    //if (mime_type && !strcmp(mime_type, "multipart/x-mixed-replace;boundary=myboundary"))
    if (mime_type && strstr(mime_type, "multipart/x-mixed-replace;boundary=")!=NULL)
	{
		req_write(req, http_ver_string(req->http_version));
		req_write(req, " 200 OK" CRLF);
		print_content_type(req);
		req_write(req, "Server: Vivotek Video Server" CRLF);
		req_write(req, date_header);
		req_write(req, "Cache-Control: no-store" CRLF);
    	req_write(req, "Pragma: no-cache" CRLF);
    	print_ka_phrase(req);
    	req_write(req, CRLF);
		req_flush(req);
		
/*
		// add boundary message
		iov[n_iov].iov_base = mime_type;
		iov[n_iov].iov_len = strlen(mime_type) + INCLUDE_NULL_CHARACTER;
        n_iov++;
*/
	}

	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "Query: [%d] [%s]\n", querysize, query);
	}


	/* Try to write and assume that an error means the read side closed,
	   re-attempt to open once. */
	for (send_attempt = 0; send_attempt <= SEND_ATTEMPTS; send_attempt++) {
		/* Open the connection if it is not cached. */
		if (item->fd < 0) {
			item->fd = fdipc_client_socket(item->path);
			if (item->fd < 0) {
				send_r_error(req);
				return -1;
			}
		}
		/* Send the protocol defined packet or retry. */
		if (fdipc_send(item->fd, iov, n_iov, req->fd) < 0) {
			if (close(item->fd) < 0) {
				return -1;
			} else {
				item->fd = -1;
			}
		} else {
			return 1;
		}
	}
	send_r_error(req);
	return -1;
}

/* Propagate CGI request req to executable defined. Return 1 on success
   which is interpreted as success, continue, -1 on failure. */
int
init_transfer_cgi(request* req)
{
	int pid;
	int exec_status;
	char *aargv[CGI_ARGC_MAX + 1];
	int pipe_stdin[2];
	int content_length;
	char *buf_start;
	char buffer[POST_BUFFER_SIZE];
	int buf_byte;
	int byte_written;
	int byte_read;

	DEBUG(DEBUG_TRANSFER) {
		fprintf(stderr, "%s: %s\n", __FUNCTION__, req->pathname);
	}

	SQUASH_KA(req);

	/* Only unbuffered uploads make sense. */
	if (req->method != M_POST) {
		send_r_bad_request(req);
		return -1;
	}

	/* Set the environment variables for the executable. */
	// added by rey
	/*
	if (cgi_complete_env(req) == 0) {
		return -1;
	}
	*/

	/* Connect the executable's STDIN. */
	if (pipe(pipe_stdin) == -1) {
		log_error_time();
		perror("pipe");
		return -1;
	}

	/* Must have content length. */
	if (req->content_length) {
		content_length = boa_atoi(req->content_length);
		req->filesize = (unsigned int)content_length;
		req->filepos = 0;

		if (req_write(req, "HTTP/1.0 200 OK\r\n") < 0) {
			return -1;
		}
		if (req_flush(req) < 0) {
			return -1;
		}
	} else {
		send_r_bad_request(req);
		return -1;
	}

	if ((pid = fork()) < 0) {
		WARN(req->pathname);
		(void)close(req->fd);
		return -1;
	} else if (pid > 0) {
		/* in parent */

		DEBUG(DEBUG_TRANSFER) {
			fprintf(stderr, "Forked child \"%s\" pid %d\n",
				req->pathname, pid);
		}

		/* Flush the cached data from excessive reading in boa. */
		buf_start = req->client_stream + req->parse_pos;
		buf_byte = req->client_stream_pos - req->parse_pos;
		byte_read = buf_byte;

		do {
			byte_written = (int)write(pipe_stdin[1], buf_start, (size_t)byte_read);
			DEBUG(DEBUG_TRANSFER) {
				fprintf(stderr, "byte_written [%d]\n",
					byte_written);
			}
			if (byte_written < 0) {
				if (errno == EAGAIN || errno == EINTR) {
					continue;
				}
				WARN(req->pathname);
				(void)close(req->fd);
				return -1;
			} else {
				byte_read -= byte_written;
				req->filepos += byte_written;
				DEBUG(DEBUG_TRANSFER) {
					fprintf(stderr, "req->filepos [%d]\n",
						(int)req->filepos);
				}
			}
		} while (byte_read > 0);

		/* Do not use the boa buffer since it is a complicated beast
		   and the risk to overwrite is great. */
		buf_start = buffer;
		buf_byte = POST_BUFFER_SIZE;
		while (req->filepos < req->filesize) {
			if ((int)(req->filesize - req->filepos) < buf_byte) {
				buf_byte = (int)(req->filesize - req->filepos);
			}
			byte_read = (int)read(req->fd, buf_start, (size_t)buf_byte);
			DEBUG(DEBUG_TRANSFER) {
				fprintf(stderr, "byte_read [%d]\n",
					byte_read);
			}
			if (byte_read < 0) {
				if (byte_read < 0) {
					if (errno == EAGAIN
					    || errno == EINTR) {
						continue;
					}
					WARN(req->pathname);
					(void)close(req->fd);
					return -1;
				}
			}

			do {
				byte_written = (int)write(pipe_stdin[1], buf_start, (size_t)byte_read);
				DEBUG(DEBUG_TRANSFER) {
					fprintf(stderr, "byte_written [%d]\n",
						byte_written);
				}
				if (byte_written < 0) {
					if (errno == EAGAIN
					    || errno == EINTR) {
						continue;
					}
					WARN(req->pathname);
					(void)close(req->fd);
					return -1;
				} else {
					byte_read -= byte_written;
					req->filepos += byte_written;
					DEBUG(DEBUG_TRANSFER) {
						fprintf(stderr, "req->filepos [%d]\n",
							(int)req->filepos);
					}
				}
			} while (byte_read > 0);
		}
		if (close(pipe_stdin[1]) < 0) {
			perror("close - pipe_stdin");
			(void)close(req->fd);
			return -1;
		}
		if (close(req->fd) < 0) {
			perror("close - req->fd");
			WARN(req->pathname);
			return -1;
		}

		/* Wait for script to finish. */
		if (waitpid(pid, &exec_status, 0) < 0) {
			perror("waitpid");
			return -1;
		}
	} else {
		/* in child */

		/* Tie stdin to read end of pipe. */
		if (dup2(pipe_stdin[0], STDIN_FILENO) == -1) {
			log_error_time();
			perror("dup2 - pipe_stdin");
			(void)close(pipe_stdin[0]);
			_exit(1);
		}
		if (close(pipe_stdin[0]) < 0) {
			log_error_time();
			perror("close - pipe_stdin[0]");
			_exit(1);
		}
		if (close(pipe_stdin[1]) < 0) {
			log_error_time();
			perror("close - pipe_stdin[1]");
			_exit(1);
		}

		/* Tie stdout to socket. */
		if (dup2(req->fd, STDOUT_FILENO) == -1) {
			log_error_time();
			perror("dup2 - fd");
			_exit(1);
		}

		/* Switch socket flags back to blocking. */
		if (set_block_fd(req->fd) == -1) {
			log_error_time();
			perror("cgi-fcntl");
			_exit(1);
		}

		/* Creates options from e.g. -m+flash+-t+http_post+-s+2. */
		// added by rey
		/*
		cgi_create_argv(req, aargv);
		*/

		/* Execute with proper options and environment. */
		if (execve(req->pathname, aargv, req->cgi_env) < 0) {
			/* execve failed */
			WARN(req->pathname);
			return -1;
		}
	}

	return 1;
}

/* Inserts an item of type and transfer socket path with key_name in the
   global transfer_list, returns 0 on success, -1 on malloc failure. */
static int
add_transfer_type(char *key_name, char *path, unsigned char type)
{
	transfer_key *new_item;
	transfer_key *item;

	/* New item. */
	new_item = (transfer_key *)malloc(sizeof(transfer_key));
	if (new_item == NULL) {
		perror(out_of_memory);
		WARN(out_of_memory);
		return -1;
	}
	new_item->key_name = strdup(key_name);
	if (new_item->key_name == NULL) {
		perror(out_of_memory);
		WARN(out_of_memory);
		free(new_item);
		return -1;
	}
	new_item->path = strdup(path);
	if (new_item->path == NULL) {
		perror(out_of_memory);
		WARN(out_of_memory);
		free(new_item->key_name);
		free(new_item);
		return -1;
	}
	new_item->fd = -1;
	new_item->type = type;
	new_item->next = NULL;

	/* Insert item into transfer list. */
	if (transfer_list == NULL) {
		transfer_list = new_item;
	} else {
		item = transfer_list;
		while (item->next != NULL) {
			item = item->next;
		}
		item->next = new_item;
	}

	return 0;
}

/* Return the item with key_name of type in the transfer_list or NULL. */
static transfer_key*
list_find(const char *key_name, unsigned char type)
{
	transfer_key* item;

	item = transfer_list;
	while (item != NULL) {
		if (item->type == type) {
			if (strcmp(item->key_name, key_name) == 0) {
				return item;
			}
		}
		item = item->next;
	}

	return NULL;
}

/* Return 1 if key_name of type exists in the transfer list, 0 if not. */
static int
is_transfer_type(const char *key_name, unsigned char type)
{
	if (list_find(key_name, type) == NULL) {
		return 0;
	}

	return 1;
}

/* Add the value of the environment variable env_var to the query string
   if it is defined (required) and return query or return NULL. */
static char *
add_str(char *query, const char *env_var, const char *value)
{
	size_t add_len;

	/* Not required variable. */
	if (value == NULL) {
		return query;
	}
	/* Verbose queries are possible. */
	add_len = strlen(env_var) + strlen(value);
	if (strlen(query) + add_len > QUERY_BUFFER_SIZE) {
		WARN(qbuf_overflow);
		return NULL;
	}
	/* Add the variable and the value. */
	strcat(query, env_var);
	strcat(query, value);
	return query;
}
